// (C) 2001-2024 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.





//------------------------------------------------------------------------------
// Description:
// This is a synthesizable pipeline register slice for the ready latency adapter.
// It comes in two flavors, one more behavioral with multi-demensional arrays
// and for loops, and another more structure implementation.


module ready_latency_regslice #(parameter DATA_WIDTH = 32, 
                                parameter ID_WIDTH   = 4 ) ( 
   input                       reset_n,
   input                       clk,

   input  [DATA_WIDTH-1:0]     source_data,
   input  [  ID_WIDTH-1:0]     source_id,
   input                       source_valid,
   
   input                       sink_ready,

   input [2:0]                 ready_delay,
   input [2:0]                 valid_delay,

   output reg [DATA_WIDTH-1:0] sink_data,
   output reg [  ID_WIDTH-1:0] sink_id,
   output reg                  sink_valid,
   
   output reg                  source_ready );

   localparam MAX_DEPTH = 4;
   integer i;

   reg [DATA_WIDTH-1:0] source_data_h  [MAX_DEPTH]; 
   reg [  ID_WIDTH-1:0] source_id_h    [MAX_DEPTH]; 
   reg                  source_valid_h [MAX_DEPTH]; 
   reg                  sink_ready_h   [MAX_DEPTH];


  always @(posedge clk or negedge reset_n) begin
       if (~reset_n) begin 
         for (i=0; i<MAX_DEPTH; i=i+1) begin
             source_data_h[i]  <= {DATA_WIDTH{1'b0}};
             source_id_h[i]    <= {ID_WIDTH{1'b0}};
             source_valid_h[i] <= 1'b0;
             sink_ready_h[i]  <= 1'b0;
         end 
       end else begin
         for (i=0; i<MAX_DEPTH; i=i+1) begin
             if (i != 0) begin
               source_data_h[i]  <= source_data_h[i-1];
               source_id_h[i]    <= source_id_h[i-1]; 
               source_valid_h[i] <= source_valid_h[i-1];
               sink_ready_h[i]  <= sink_ready_h[i-1];
             end else begin
               source_data_h[0]  <= source_data;
               source_id_h[0]    <= source_id; 
               source_valid_h[0] <= source_valid;
               sink_ready_h[0]  <= sink_ready;
             end
         end
       end 
   end 
  
   // ready mux
   always @(*) begin 
      source_ready = 1'bx;   
      if (ready_delay == 0) 
         source_ready = sink_ready;
      else begin
         for (i=1; i<5; i=i+1) begin
            if (ready_delay == i)
               source_ready = sink_ready_h[i-1];
         end
      end
   end

   // valid mux
   always @(*) begin 
      sink_valid = 1'bx;   
      sink_data  = {DATA_WIDTH{1'bx}};
      sink_id    = {ID_WIDTH{1'bx}};
      if (valid_delay == 0) begin
         sink_valid = source_valid;
         sink_data  = source_data; 
         sink_id    = source_id;
      end else begin
         for (i=1; i<5; i=i+1) begin
            if (valid_delay == i) begin
               sink_valid = source_valid_h[i-1];
               sink_data  = source_data_h[i-1];
               sink_id    = source_id_h[i-1];
            end
         end
      end
   end
endmodule

// A more structural coding without for loops, same interface & functionality:
module ready_latency_regslice_2 #(parameter DATA_WIDTH = 32, 
                                  parameter ID_WIDTH   = 4 ) ( 
   input                       reset_n,
   input                       clk,

   input  [DATA_WIDTH-1:0]     source_data,
   input  [  ID_WIDTH-1:0]     source_id,
   input                       source_valid,
   
   input                       sink_ready,

   input [2:0]                 ready_delay,
   input [2:0]                 valid_delay,

   output reg [DATA_WIDTH-1:0] sink_data,
   output reg [  ID_WIDTH-1:0] sink_id,
   output reg                  sink_valid,
   
   output reg                  source_ready );

   reg [DATA_WIDTH-1:0] source_data_1h;
   reg [DATA_WIDTH-1:0] source_data_2h;
   reg [DATA_WIDTH-1:0] source_data_3h;
   reg [DATA_WIDTH-1:0] source_data_4h;

   reg [  ID_WIDTH-1:0] source_id_1h;
   reg [  ID_WIDTH-1:0] source_id_2h;
   reg [  ID_WIDTH-1:0] source_id_3h;
   reg [  ID_WIDTH-1:0] source_id_4h;

   reg                  source_valid_1h;
   reg                  source_valid_2h;
   reg                  source_valid_3h;
   reg                  source_valid_4h;
   
   reg                  sink_ready_1h;
   reg                  sink_ready_2h;
   reg                  sink_ready_3h;
   reg                  sink_ready_4h;

   always @(posedge clk or negedge reset_n) begin
      if (~reset_n) begin
         source_data_1h <= {DATA_WIDTH{1'b0}};
         source_data_2h <= {DATA_WIDTH{1'b0}};
         source_data_3h <= {DATA_WIDTH{1'b0}};
         source_data_4h <= {DATA_WIDTH{1'b0}};

         source_id_1h <= {ID_WIDTH{1'b0}};
         source_id_2h <= {ID_WIDTH{1'b0}};
         source_id_3h <= {ID_WIDTH{1'b0}};
         source_id_4h <= {ID_WIDTH{1'b0}};

         source_valid_1h <= 1'b0;
         source_valid_2h <= 1'b0;
         source_valid_3h <= 1'b0;
         source_valid_4h <= 1'b0;
      
         sink_ready_1h <= 1'b0;
         sink_ready_2h <= 1'b0;
         sink_ready_3h <= 1'b0;
         sink_ready_4h <= 1'b0;
     end else begin
         source_data_1h <= source_data;
         source_data_2h <= source_data_1h;
         source_data_3h <= source_data_2h;
         source_data_4h <= source_data_3h;

         source_id_1h <= source_id;
         source_id_2h <= source_id_1h;
         source_id_3h <= source_id_2h;
         source_id_4h <= source_id_3h;

         source_valid_1h <= source_valid;
         source_valid_2h <= source_valid_1h;
         source_valid_3h <= source_valid_2h;
         source_valid_4h <= source_valid_3h;
      
         sink_ready_1h <= sink_ready;
         sink_ready_2h <= sink_ready_1h;
         sink_ready_3h <= sink_ready_2h;
         sink_ready_4h <= sink_ready_3h;
      end
   end 

   //always @(*) begin
   always_comb begin
      case (1'b1)
         ready_delay == 3'b000: source_ready = sink_ready;
         ready_delay == 3'b001: source_ready = sink_ready_1h;
         ready_delay == 3'b010: source_ready = sink_ready_2h;
         ready_delay == 3'b011: source_ready = sink_ready_3h;
         ready_delay == 3'b100: source_ready = sink_ready_4h;
         default:               source_ready = 1'bx;
      endcase
   end // always

   //always @(*) begin
   always_comb begin
      case (1'b1)
         valid_delay == 3'b000: 
            begin
               sink_valid = source_valid;
               sink_data  = source_data;
               sink_id    = source_id;
            end 
         valid_delay == 3'b001:
            begin
               sink_valid = source_valid_1h;
               sink_data  = source_data_1h;
               sink_id    = source_id_1h;
            end 
         valid_delay == 3'b010:
            begin
               sink_valid = source_valid_2h;
               sink_data  = source_data_2h;
               sink_id    = source_id_2h;
            end 
         valid_delay == 3'b011: 
            begin
               sink_valid = source_valid_3h;
               sink_data  = source_data_3h;
               sink_id    = source_id_3h;
            end 
         valid_delay == 3'b100:
            begin
               sink_valid = source_valid_4h;
               sink_data  = source_data_4h;
               sink_id    = source_id_4h;
            end 
         default:
            begin
               sink_valid = 1'bx;
               sink_data  = {DATA_WIDTH{1'bx}};
               sink_id    = {ID_WIDTH{1'bx}};
            end 
      endcase
   end // always
    
endmodule  

// Full Register Slice logic
// This is nothing but a 2-deep FIFO
// Combined synchronous FIFO with 1 read port and 1 write port

// Option 2: Moves some logic over to before flop stage - will harm valid_init input path, but to the benefit of ready_init output path

(* altera_attribute = "-name AUTO_RAM_RECOGNITION OFF" *)
module rla_full_reg_slice #(
  parameter WIDTH = 'd8,       // Width of Register Slice
  parameter PTR_WIDTH  = 'd1    // Width of pointers - include rollover bit
) (
   input  clk, rst_n,           // Clock and Reset

   input  valid_init,           // Initiator Side: Valid Input
   output ready_init,           // Initiator Side: Ready Output
   input  [WIDTH-1:0] data_init,// Initiator Side: Data Input
   input  readylatency_init,    // Initiator Side: ReadyLatency Input

   output valid_targ,           // Target Side: Valid Output
   input  ready_targ,           // Target Side: Ready Input
   output [WIDTH-1:0] data_targ,// Target Side: Data Output
   input  readylatency_targ     // Target Side: ReadyLatency Input
);

localparam DEPTH      = 1 << PTR_WIDTH;

wire put, get, avail;
reg  free;
wire [PTR_WIDTH:0]    putptr_nxt, getptr_nxt;
reg  [PTR_WIDTH:0]    putptr, getptr;
reg  [PTR_WIDTH+1:0]  putptr_temp, getptr_temp;
reg  [WIDTH-1:0] mem [0:DEPTH-1];

// put & putptr logic
assign put = valid_init & (readylatency_init | free );  // ReadyLatency is handled here. When ReadyLatency != 0, put when Valid is high
assign putptr_temp = putptr + 'd1;
assign putptr_nxt = put ? putptr_temp[PTR_WIDTH:0] : putptr;

always @(posedge clk or negedge rst_n) begin
  if (!rst_n) putptr <= {PTR_WIDTH+1{1'b0}};
  else        putptr <= putptr_nxt;
end

// get & getptr logic
assign get = avail & ready_targ;
assign getptr_temp= getptr + 'd1;
assign getptr_nxt = get ? getptr_temp[PTR_WIDTH:0] : getptr;

always @(posedge clk or negedge rst_n) begin
  if (!rst_n) getptr <= {PTR_WIDTH+1{1'b0}};
  else        getptr <= getptr_nxt;
end

// free/full logic (free = ~full)
// FB: 182359 - generate all free signals directly from flop - so that it can be reset to Zero
//assign free = ~((putptr[PTR_WIDTH] != getptr[PTR_WIDTH]) && (putptr[PTR_WIDTH-1:0] == getptr[PTR_WIDTH-1:0]));
always @(posedge clk or negedge rst_n) begin
  if (!rst_n) free <= 0;
  else        free <= ~((putptr_nxt[PTR_WIDTH] != getptr_nxt[PTR_WIDTH]) && (putptr_nxt[PTR_WIDTH-1:0] == getptr_nxt[PTR_WIDTH-1:0]));
end

// avail/empty logic (avail = ~empty)
assign avail = ~(putptr[PTR_WIDTH:0] == getptr[PTR_WIDTH:0]);

// ready_init logic
// if ReadyLatency is enabled Ready deasserts at 2 entries from full
// else Ready deasserts at full
generate
if (DEPTH > 2) begin : DEPTH_MORE_THAN_2

  wire                free_gt_2;
  reg  [PTR_WIDTH:0]  nfree;

  always @(posedge clk or negedge rst_n) begin
    if (!rst_n) nfree <= {PTR_WIDTH+1{1'b0}};
    else        nfree <= DEPTH[PTR_WIDTH:0] - (putptr_nxt - getptr_nxt);
  end

  assign free_gt_2 = nfree > 'd4;

  assign ready_init = readylatency_init ? free_gt_2 : free;
end
else begin : DEPTH_LESS_OR_EQUAL_2
  assign ready_init = free;
end
endgenerate

// valid_targ logic
assign valid_targ = get | ( avail & ~readylatency_targ );   // ReadyLatency is handled here. When ReadyLatency is !=0, qualify Valid with Ready

// Data Storage Portion - data_init enters here while data_targ exits from here
assign data_targ = mem[getptr[PTR_WIDTH-1:0]];

always @(posedge clk) begin
   if (put) mem[putptr[PTR_WIDTH-1:0]] <= data_init;
end

endmodule


// Description:
// This is a synthesizable ready-latency adapter.  It comprises a
// ready-latency pipeline of selectable depth and a ready-latency fifo. 
// Instantiate it between the Clarke HPS LWBridge and an AXI-compliant slave.


module f2s_ready_latency_adapter (
  input           ar_clk,
  input           ar_rst_n,
  input           aw_clk,
  input           aw_rst_n,
  input           b_clk,
  input           b_rst_n,
  input           r_clk,
  input           r_rst_n,
  input           w_clk,
  input           w_rst_n,
  input           ac_clk,
  input           ac_rst_n,
  input           cr_clk,
  input           cr_rst_n,
  input           cd_clk,
  input           cd_rst_n,

  input [2:0]     ar_ready_latency,
  input [2:0]     ar_valid_latency,
  input [2:0]     aw_ready_latency,
  input [2:0]     aw_valid_latency,
  input [2:0]     w_ready_latency,
  input [2:0]     w_valid_latency,
  input [2:0]     r_ready_latency,
  input [2:0]     r_valid_latency,
  input [2:0]     b_ready_latency,
  input [2:0]     b_valid_latency,
  input [2:0]     ac_valid_latency,
  input [2:0]     ac_ready_latency,
  input [2:0]     cr_valid_latency,
  input [2:0]     cr_ready_latency,
  input [2:0]     cd_valid_latency,
  input [2:0]     cd_ready_latency,
  input [2:0]     ack_latency,

  input           ar_ready_s1,
  input    [43:0] ar_addr_m1,
  input     [1:0] ar_burst_m1,
  input     [3:0] ar_cache_m1,
  input     [3:0] ar_id_m1,
  input     [7:0] ar_len_m1,
  input           ar_lock_m1,
  input     [2:0] ar_prot_m1,
  input     [3:0] ar_qos_m1,
  input     [2:0] ar_size_m1,
  input           ar_valid_m1,
  input     [1:0] ar_domain_m1,
  input     [3:0] ar_snoop_m1,
  input     [1:0] ar_bar_m1,

  input           aw_ready_s1,
  input    [43:0] aw_addr_m1,
  input     [1:0] aw_burst_m1,
  input     [3:0] aw_cache_m1,
  input     [3:0] aw_id_m1,
  input     [7:0] aw_len_m1,
  input           aw_lock_m1,
  input     [2:0] aw_prot_m1,
  input     [3:0] aw_qos_m1,
  input     [2:0] aw_size_m1,
  input           aw_valid_m1,
  input     [1:0] aw_domain_m1,
  input     [3:0] aw_snoop_m1,
  input     [1:0] aw_bar_m1,

  input   [127:0] w_data_m1,
  input           w_last_m1,
  input    [15:0] w_strb_m1,
  input           w_valid_m1,
  input           w_ready_s1,
  input           w_ack_m1,

  input     [3:0] b_id_s1,
  input     [1:0] b_resp_s1,
  input           b_valid_s1,
  input           b_ready_m1,

  input   [127:0] r_data_s1,
  input     [3:0] r_id_s1,
  input           r_last_s1,
  input     [1:0] r_resp_s1,
  input           r_valid_s1,
  input           r_ready_m1, 
  input           r_ack_m1,

  input           ac_ready_m1,
  input [2:0]     ac_prot_s1 ,
  input           ac_valid_s1,
  input [3:0]     ac_snoop_s1,
  input [43:0]    ac_addr_s1 ,

  input           cr_ready_s1,
  input           cr_valid_m1,
  input  [4:0]    cr_resp_m1 ,
                    
  input           cd_ready_s1,
  input           cd_valid_m1,
  input  [127:0]  cd_data_m1 ,
  input           cd_last_m1 , 

  output   [43:0] ar_addr_s1,
  output    [1:0] ar_burst_s1,
  output    [3:0] ar_cache_s1,
  output    [3:0] ar_id_s1,
  output    [7:0] ar_len_s1,
  output          ar_lock_s1,
  output    [2:0] ar_prot_s1,
  output    [3:0] ar_qos_s1,
  output    [2:0] ar_size_s1,
  output          ar_valid_s1,
  output          ar_ready_m1,
  output    [1:0] ar_domain_s1,
  output    [3:0] ar_snoop_s1,
  output    [1:0] ar_bar_s1,

  output   [43:0] aw_addr_s1,
  output    [1:0] aw_burst_s1,
  output    [3:0] aw_cache_s1,
  output    [3:0] aw_id_s1,
  output    [7:0] aw_len_s1,
  output          aw_lock_s1,
  output    [2:0] aw_prot_s1,
  output    [3:0] aw_qos_s1,
  output    [2:0] aw_size_s1,
  output          aw_valid_s1, 
  output          aw_ready_m1,
  output    [1:0] aw_domain_s1,
  output    [3:0] aw_snoop_s1,
  output    [1:0] aw_bar_s1,

  output    [3:0] b_id_m1,
  output    [1:0] b_resp_m1,
  output          b_valid_m1,
  output          b_ready_s1,

  output  [127:0] r_data_m1,
  output    [3:0] r_id_m1,
  output          r_last_m1,
  output    [1:0] r_resp_m1,
  output          r_valid_m1,
  output          r_ready_s1, 
  output          r_ack_s1,

  output  [127:0] w_data_s1,
  output          w_last_s1,
  output   [15:0] w_strb_s1,
  output          w_valid_s1,
  output          w_ready_m1,
  output          w_ack_s1,

  output          ac_ready_s1 ,
  output [2:0]    ac_prot_m1  ,
  output          ac_valid_m1 ,
  output [3:0]    ac_snoop_m1 ,
  output [43:0]   ac_addr_m1  ,

  output          cr_ready_m1 ,
  output          cr_valid_s1 ,
  output  [4:0]   cr_resp_s1  ,
                  
  output          cd_ready_m1 ,
  output          cd_valid_s1 ,
  output  [127:0] cd_data_s1  ,
  output          cd_last_s1  

);            

   wire        rlp_rlf_arready_h; 
   wire        rlp_rlf_awready_h;
   wire        rlp_rlf_wready_h;

   wire [3:0]  rlp_rlf_bid_h;
   wire [1:0]  rlp_rlf_bresp_h;
   wire        rlp_rlf_bvalid_h;

   wire rlf_rlp_bready_h;

   wire [127:0] rlp_rlf_rdata_h;
   wire [3:0]   rlp_rlf_rid_h;
   wire         rlp_rlf_rlast_h;
   wire [1:0]   rlp_rlf_rresp_h;
   wire         rlp_rlf_rvalid_h;
  
   wire rlf_rlp_rready_h;

   wire [43:0] rlf_rlp_araddr_h;
   wire [1:0]  rlf_rlp_arburst_h;
   wire [3:0]  rlf_rlp_arcache_h;
   wire [3:0]  rlf_rlp_arid_h;
   wire [7:0]  rlf_rlp_arlen_h;
   wire        rlf_rlp_arlock_h;
   wire [2:0]  rlf_rlp_arprot_h;
   wire [3:0]  rlf_rlp_arqos_h;
   wire [2:0]  rlf_rlp_arsize_h;
   wire        rlf_rlp_arvalid_h;
   wire [1:0]  rlf_rlp_ardomain;
   wire [3:0]  rlf_rlp_arsnoop;
   wire [1:0]  rlf_rlp_arbar;

   wire [43:0] rlf_rlp_awaddr_h;
   wire [1:0]  rlf_rlp_awburst_h;
   wire [3:0]  rlf_rlp_awcache_h;
   wire [3:0]  rlf_rlp_awid_h;
   wire [7:0]  rlf_rlp_awlen_h;
   wire        rlf_rlp_awlock_h;
   wire [2:0]  rlf_rlp_awprot_h;
   wire [3:0]  rlf_rlp_awqos_h;
   wire [2:0]  rlf_rlp_awsize_h;
   wire        rlf_rlp_awvalid_h;
   wire [1:0]  rlf_rlp_awdomain;
   wire [3:0]  rlf_rlp_awsnoop;
   wire [1:0]  rlf_rlp_awbar;

   wire [127:0]  rlf_rlp_wdata_h;
   wire          rlf_rlp_wlast_h;
   wire [15:0]   rlf_rlp_wstrb_h;
   wire          rlf_rlp_wvalid_h;

  
   wire [2:0]    rlp_rlf_acprot  ;
   wire          rlp_rlf_acvalid ;
   wire [3:0]    rlp_rlf_acsnoop ;
   wire [43:0]   rlp_rlf_acaddr  ;
   wire          rlf_rlp_acready ;

   wire          rlp_rlf_crready ;
   wire          rlf_rlp_crvalid ;
   wire [4:0]    rlf_rlp_crresp  ;

   wire          rlp_rlf_cdready ;
   wire          rlf_rlp_cdvalid ;
   wire [127:0]  rlf_rlp_cddata  ;
   wire          rlf_rlp_cdlast  ;

   wire        ar_latency_en;
   wire        aw_latency_en;
   wire         w_latency_en;
   wire         r_latency_en;
   wire         b_latency_en;
   wire        ac_latency_en;
   wire        cr_latency_en;
   wire        cd_latency_en;

   //---------------------------------------------------------------- 

   assign ar_latency_en = |{ar_ready_latency, ar_valid_latency};
   assign aw_latency_en = |{aw_ready_latency, aw_valid_latency};
   assign  w_latency_en = |{ w_ready_latency,  w_valid_latency};
   assign  r_latency_en = |{ r_ready_latency,  r_valid_latency};
   assign  b_latency_en = |{ b_ready_latency,  b_valid_latency};
   assign ac_latency_en = |{ac_ready_latency, ac_valid_latency};
   assign cr_latency_en = |{cr_ready_latency, cr_valid_latency};
   assign cd_latency_en = |{cd_ready_latency, cd_valid_latency};

   //---------------------------------------------------------------- 

   f2s_ready_latency_pipe f2s_rlp (
      .ar_clk  ( ar_clk  ),
      .ar_rst_n( ar_rst_n),
      .aw_clk  ( aw_clk  ),
      .aw_rst_n( aw_rst_n),
      .b_clk   ( b_clk   ),
      .b_rst_n ( b_rst_n ),
      .r_clk   ( r_clk   ),
      .r_rst_n ( r_rst_n ),
      .w_clk   ( w_clk   ),
      .w_rst_n ( w_rst_n ),
      .ac_clk  ( ac_clk   ),
      .ac_rst_n( ac_rst_n ),
      .cr_clk  ( cr_clk   ),
      .cr_rst_n( cr_rst_n ),
      .cd_clk  ( cd_clk   ),
      .cd_rst_n( cd_rst_n ),

      .ar_ready_latency ( ar_ready_latency ),
      .ar_valid_latency ( ar_valid_latency ),
      .aw_ready_latency ( aw_ready_latency ),
      .aw_valid_latency ( aw_valid_latency ),
      .w_ready_latency  ( w_ready_latency  ),
      .w_valid_latency  ( w_valid_latency  ),
      .r_ready_latency  ( r_ready_latency  ),
      .r_valid_latency  ( r_valid_latency  ),
      .b_ready_latency  ( b_ready_latency  ),
      .b_valid_latency  ( b_valid_latency  ),
      .ac_ready_latency ( ac_ready_latency  ),
      .ac_valid_latency ( ac_valid_latency  ),
      .cr_ready_latency ( cr_ready_latency  ),
      .cr_valid_latency ( cr_valid_latency  ),
      .cd_ready_latency ( cd_ready_latency  ),
      .cd_valid_latency ( cd_valid_latency  ),
      .ack_latency      ( ack_latency       ),

      .ar_ready_s1( ar_ready_s1 ),
      .ar_addr_m1 ( rlf_rlp_araddr_h ),
      .ar_burst_m1( rlf_rlp_arburst_h),
      .ar_cache_m1( rlf_rlp_arcache_h),
      .ar_id_m1   ( rlf_rlp_arid_h   ),
      .ar_len_m1  ( rlf_rlp_arlen_h  ),
      .ar_lock_m1 ( rlf_rlp_arlock_h ),
      .ar_prot_m1 ( rlf_rlp_arprot_h ),
      .ar_qos_m1  ( rlf_rlp_arqos_h  ),
      .ar_size_m1 ( rlf_rlp_arsize_h ),
      .ar_valid_m1( rlf_rlp_arvalid_h),
      .ar_domain_m1 (rlf_rlp_ardomain),
      .ar_snoop_m1  (rlf_rlp_arsnoop ),
      .ar_bar_m1    (rlf_rlp_arbar   ),

      .aw_ready_s1( aw_ready_s1 ),
      .aw_addr_m1 ( rlf_rlp_awaddr_h ),
      .aw_burst_m1( rlf_rlp_awburst_h),
      .aw_cache_m1( rlf_rlp_awcache_h),
      .aw_id_m1   ( rlf_rlp_awid_h   ),
      .aw_len_m1  ( rlf_rlp_awlen_h  ),
      .aw_lock_m1 ( rlf_rlp_awlock_h ),
      .aw_prot_m1 ( rlf_rlp_awprot_h ),
      .aw_qos_m1  ( rlf_rlp_awqos_h  ),
      .aw_size_m1 ( rlf_rlp_awsize_h ),
      .aw_valid_m1( rlf_rlp_awvalid_h),
      .aw_domain_m1(  rlf_rlp_awdomain),
      .aw_snoop_m1 (  rlf_rlp_awsnoop ),
      .aw_bar_m1   (  rlf_rlp_awbar   ),

      .w_data_m1  (rlf_rlp_wdata_h ),  
      .w_last_m1  (rlf_rlp_wlast_h ), 
      .w_strb_m1  (rlf_rlp_wstrb_h ),
      .w_valid_m1 (rlf_rlp_wvalid_h),
      .w_ready_s1 ( w_ready_s1 ),
      .w_ack_m1   ( w_ack_m1   ),

      .ac_ready_m1 ( rlf_rlp_acready ),
      .ac_prot_s1  ( ac_prot_s1      ),
      .ac_valid_s1 ( ac_valid_s1     ),
      .ac_snoop_s1 ( ac_snoop_s1     ),
      .ac_addr_s1  ( ac_addr_s1      ),

      .cr_ready_s1 ( cr_ready_s1     ),
      .cr_valid_m1 ( rlf_rlp_crvalid ),
      .cr_resp_m1  ( rlf_rlp_crresp  ),

      .cd_ready_s1 ( cd_ready_s1     ),
      .cd_valid_m1 ( rlf_rlp_cdvalid ),
      .cd_data_m1  ( rlf_rlp_cddata  ),
      .cd_last_m1  ( rlf_rlp_cdlast  ),
      
      .b_id_s1    (b_id_s1     ),
      .b_resp_s1  (b_resp_s1   ),
      .b_valid_s1 (b_valid_s1  ),
      .b_ready_m1 (rlf_rlp_bready_h ),

      .r_data_s1  ( r_data_s1 ),
      .r_id_s1    ( r_id_s1   ),
      .r_last_s1  ( r_last_s1 ),
      .r_resp_s1  ( r_resp_s1 ),
      .r_valid_s1 ( r_valid_s1),
      .r_ready_m1 ( rlf_rlp_rready_h ),
      .r_ack_m1   ( r_ack_m1  ),

      .ar_addr_s1 ( ar_addr_s1  ),
      .ar_burst_s1( ar_burst_s1 ),
      .ar_cache_s1( ar_cache_s1 ),
      .ar_id_s1   ( ar_id_s1    ),
      .ar_len_s1  ( ar_len_s1   ),
      .ar_lock_s1 ( ar_lock_s1  ),
      .ar_prot_s1 ( ar_prot_s1  ),
      .ar_qos_s1  ( ar_qos_s1   ),
      .ar_size_s1 ( ar_size_s1  ),
      .ar_valid_s1( ar_valid_s1 ),
      .ar_ready_m1( rlp_rlf_arready_h ),
      .ar_domain_s1( ar_domain_s1 ),
      .ar_snoop_s1 ( ar_snoop_s1  ),
      .ar_bar_s1   ( ar_bar_s1    ),

      .aw_addr_s1 ( aw_addr_s1  ),
      .aw_burst_s1( aw_burst_s1 ),
      .aw_cache_s1( aw_cache_s1 ),
      .aw_id_s1   ( aw_id_s1    ),
      .aw_len_s1  ( aw_len_s1   ),
      .aw_lock_s1 ( aw_lock_s1  ),
      .aw_prot_s1 ( aw_prot_s1  ),
      .aw_qos_s1  ( aw_qos_s1   ),
      .aw_size_s1 ( aw_size_s1  ),
      .aw_valid_s1( aw_valid_s1 ), 
      .aw_ready_m1( rlp_rlf_awready_h ),
      .aw_domain_s1 ( aw_domain_s1 ),
      .aw_snoop_s1  ( aw_snoop_s1  ),
      .aw_bar_s1    ( aw_bar_s1    ),

      .b_id_m1    ( rlp_rlf_bid_h    ),
      .b_resp_m1  ( rlp_rlf_bresp_h  ),
      .b_valid_m1 ( rlp_rlf_bvalid_h ),
      .b_ready_s1 ( b_ready_s1 ),

      .r_data_m1  ( rlp_rlf_rdata_h ),
      .r_id_m1    ( rlp_rlf_rid_h   ),
      .r_last_m1  ( rlp_rlf_rlast_h ),
      .r_resp_m1  ( rlp_rlf_rresp_h ),
      .r_valid_m1 ( rlp_rlf_rvalid_h),
      .r_ready_s1 ( r_ready_s1 ), 
      .r_ack_s1   ( r_ack_s1   ),

      .w_data_s1  ( w_data_s1  ),
      .w_last_s1  ( w_last_s1  ),
      .w_strb_s1  ( w_strb_s1  ),
      .w_valid_s1 ( w_valid_s1 ),
      .w_ready_m1 ( rlp_rlf_wready_h ),
      .w_ack_s1   ( w_ack_s1   ),

      .ac_ready_s1 ( ac_ready_s1     ),
      .ac_prot_m1  ( rlp_rlf_acprot  ),
      .ac_valid_m1 ( rlp_rlf_acvalid ),
      .ac_snoop_m1 ( rlp_rlf_acsnoop ),
      .ac_addr_m1  ( rlp_rlf_acaddr  ),

      .cr_ready_m1 ( rlp_rlf_crready ),
      .cr_valid_s1 ( cr_valid_s1     ),
      .cr_resp_s1  ( cr_resp_s1      ),
                                    
      .cd_ready_m1 ( rlp_rlf_cdready ),
      .cd_valid_s1 ( cd_valid_s1     ),
      .cd_data_s1  ( cd_data_s1      ),
      .cd_last_s1  ( cd_last_s1      )
      
   );

   //---------------------------------------------------------------- 

   f2s_ready_latency_fifo f2s_rlf (
      .ar_clk  ( ar_clk  ),
      .ar_rst_n( ar_rst_n),
      .aw_clk  ( aw_clk  ),
      .aw_rst_n( aw_rst_n),
      .b_clk   ( b_clk   ),
      .b_rst_n ( b_rst_n ),
      .r_clk   ( r_clk   ),
      .r_rst_n ( r_rst_n ),
      .w_clk   ( w_clk   ),
      .w_rst_n ( w_rst_n ),
      .ac_clk   ( ac_clk   ),
      .ac_rst_n ( ac_rst_n ),
      .cr_clk   ( cr_clk   ),
      .cr_rst_n ( cr_rst_n ),
      .cd_clk   ( cd_clk   ),
      .cd_rst_n ( cd_rst_n ),

      .ar_latency_en ( ar_latency_en  ),
      .aw_latency_en ( aw_latency_en  ),
      .w_latency_en  ( w_latency_en   ),
      .r_latency_en  ( r_latency_en   ),
      .b_latency_en  ( b_latency_en   ),
      .ac_latency_en ( ac_latency_en  ),
      .cr_latency_en ( cr_latency_en  ),
      .cd_latency_en ( cd_latency_en  ),

      .ar_ready_s1( rlp_rlf_arready_h ),
      .ar_addr_m1 ( ar_addr_m1  ),
      .ar_burst_m1( ar_burst_m1 ),
      .ar_cache_m1( ar_cache_m1 ),
      .ar_id_m1   ( ar_id_m1    ),
      .ar_len_m1  ( ar_len_m1   ),
      .ar_lock_m1 ( ar_lock_m1  ),
      .ar_prot_m1 ( ar_prot_m1  ),
      .ar_qos_m1  ( ar_qos_m1   ),
      .ar_size_m1 ( ar_size_m1  ),
      .ar_valid_m1( ar_valid_m1 ),
      .ar_domain_m1 (ar_domain_m1 ),
      .ar_snoop_m1  (ar_snoop_m1  ),
      .ar_bar_m1    (ar_bar_m1    ),

      .aw_ready_s1( rlp_rlf_awready_h ),
      .aw_addr_m1 ( aw_addr_m1  ),
      .aw_burst_m1( aw_burst_m1 ),
      .aw_cache_m1( aw_cache_m1 ),
      .aw_id_m1   ( aw_id_m1    ),
      .aw_len_m1  ( aw_len_m1   ),
      .aw_lock_m1 ( aw_lock_m1  ),
      .aw_prot_m1 ( aw_prot_m1  ),
      .aw_qos_m1  ( aw_qos_m1   ),
      .aw_size_m1 ( aw_size_m1  ),
      .aw_valid_m1( aw_valid_m1 ),
      .aw_domain_m1( aw_domain_m1 ),
      .aw_snoop_m1 ( aw_snoop_m1  ),
      .aw_bar_m1   ( aw_bar_m1    ),

      .w_data_m1  ( w_data_m1  ),  
      .w_last_m1  ( w_last_m1  ), 
      .w_strb_m1  ( w_strb_m1  ),
      .w_valid_m1 ( w_valid_m1 ),
      .w_ready_s1 ( rlp_rlf_wready_h ), 

      .b_id_s1    ( rlp_rlf_bid_h    ),
      .b_resp_s1  ( rlp_rlf_bresp_h  ),
      .b_valid_s1 ( rlp_rlf_bvalid_h ),
      .b_ready_m1 ( b_ready_m1 ),

      .r_data_s1  ( rlp_rlf_rdata_h  ),
      .r_id_s1    ( rlp_rlf_rid_h    ),
      .r_last_s1  ( rlp_rlf_rlast_h  ),
      .r_resp_s1  ( rlp_rlf_rresp_h  ),
      .r_valid_s1 ( rlp_rlf_rvalid_h ),
      .r_ready_m1 ( r_ready_m1 ),

      .ac_ready_m1 ( ac_ready_m1     ),
      .ac_prot_s1  ( rlp_rlf_acprot  ),
      .ac_valid_s1 ( rlp_rlf_acvalid ),
      .ac_snoop_s1 ( rlp_rlf_acsnoop ),
      .ac_addr_s1  ( rlp_rlf_acaddr  ),
      
      .cr_ready_s1 ( rlp_rlf_crready ),
      .cr_valid_m1 ( cr_valid_m1     ),
      .cr_resp_m1  ( cr_resp_m1      ),
      
      .cd_ready_s1 ( rlp_rlf_cdready ),
      .cd_valid_m1 ( cd_valid_m1     ),
      .cd_data_m1  ( cd_data_m1      ),
      .cd_last_m1  ( cd_last_m1      ),

      .ar_addr_s1 ( rlf_rlp_araddr_h  ),
      .ar_burst_s1( rlf_rlp_arburst_h ),
      .ar_cache_s1( rlf_rlp_arcache_h ),
      .ar_id_s1   ( rlf_rlp_arid_h    ),
      .ar_len_s1  ( rlf_rlp_arlen_h   ),
      .ar_lock_s1 ( rlf_rlp_arlock_h  ),
      .ar_prot_s1 ( rlf_rlp_arprot_h  ),
      .ar_qos_s1  ( rlf_rlp_arqos_h   ),
      .ar_size_s1 ( rlf_rlp_arsize_h  ),
      .ar_valid_s1( rlf_rlp_arvalid_h ),
      .ar_ready_m1( ar_ready_m1 ),
      .ar_domain_s1( rlf_rlp_ardomain ),
      .ar_snoop_s1 ( rlf_rlp_arsnoop  ),
      .ar_bar_s1   ( rlf_rlp_arbar    ),

      .aw_addr_s1 ( rlf_rlp_awaddr_h  ),
      .aw_burst_s1( rlf_rlp_awburst_h ),
      .aw_cache_s1( rlf_rlp_awcache_h ),
      .aw_id_s1   ( rlf_rlp_awid_h    ),
      .aw_len_s1  ( rlf_rlp_awlen_h   ),
      .aw_lock_s1 ( rlf_rlp_awlock_h  ),
      .aw_prot_s1 ( rlf_rlp_awprot_h  ),
      .aw_qos_s1  ( rlf_rlp_awqos_h   ),
      .aw_size_s1 ( rlf_rlp_awsize_h  ),
      .aw_valid_s1( rlf_rlp_awvalid_h ), 
      .aw_ready_m1( aw_ready_m1 ),
      .aw_domain_s1 ( rlf_rlp_awdomain ),
      .aw_snoop_s1  ( rlf_rlp_awsnoop  ),
      .aw_bar_s1    ( rlf_rlp_awbar    ),

      .b_id_m1    ( b_id_m1    ),
      .b_resp_m1  ( b_resp_m1  ),
      .b_valid_m1 ( b_valid_m1 ),
      .b_ready_s1 ( rlf_rlp_bready_h ),

      .r_data_m1  ( r_data_m1  ),
      .r_id_m1    ( r_id_m1    ),
      .r_last_m1  ( r_last_m1  ),
      .r_resp_m1  ( r_resp_m1  ),
      .r_valid_m1 ( r_valid_m1 ),
      .r_ready_s1 ( rlf_rlp_rready_h ), 

      .w_data_s1  ( rlf_rlp_wdata_h  ),
      .w_last_s1  ( rlf_rlp_wlast_h  ),
      .w_strb_s1  ( rlf_rlp_wstrb_h  ),
      .w_valid_s1 ( rlf_rlp_wvalid_h ),
      .w_ready_m1 ( w_ready_m1 ),

      .ac_ready_s1 ( rlf_rlp_acready ),
      .ac_prot_m1  ( ac_prot_m1  ),
      .ac_valid_m1 ( ac_valid_m1 ),
      .ac_snoop_m1 ( ac_snoop_m1 ),
      .ac_addr_m1  ( ac_addr_m1  ),

      .cr_ready_m1 ( cr_ready_m1     ),
      .cr_valid_s1 ( rlf_rlp_crvalid ),
      .cr_resp_s1  ( rlf_rlp_crresp  ),
                     
      .cd_ready_m1 ( cd_ready_m1     ),
      .cd_valid_s1 ( rlf_rlp_cdvalid ),
      .cd_data_s1  ( rlf_rlp_cddata  ),
      .cd_last_s1  ( rlf_rlp_cdlast  )

   ); 
endmodule
//------------------------------------------------------------------------------
//
// Description:
// This is a synthesizable FIFO for buffering lwbridge AXI signals from the
// ready-latency pipeline


module f2s_ready_latency_fifo (
   input           ar_clk,
   input           ar_rst_n,
   input           aw_clk,
   input           aw_rst_n,
   input           b_clk,
   input           b_rst_n,
   input           r_clk,
   input           r_rst_n,
   input           w_clk,
   input           w_rst_n,
   input           ac_clk,
   input           ac_rst_n,
   input           cr_clk,
   input           cr_rst_n,
   input           cd_clk,
   input           cd_rst_n,

   input           ar_latency_en,
   input           aw_latency_en,
   input           w_latency_en,
   input           r_latency_en,
   input           b_latency_en,
   input           ac_latency_en,
   input           cd_latency_en,
   input           cr_latency_en,
 
   input           ar_ready_s1,
   input    [43:0] ar_addr_m1,
   input     [1:0] ar_burst_m1,
   input     [3:0] ar_cache_m1,
   input     [3:0] ar_id_m1,
   input     [7:0] ar_len_m1,
   input           ar_lock_m1,
   input     [2:0] ar_prot_m1,
   input     [3:0] ar_qos_m1,
   input     [2:0] ar_size_m1,
   input           ar_valid_m1,
   input     [1:0] ar_domain_m1,
   input     [3:0] ar_snoop_m1,
   input     [1:0] ar_bar_m1,
 
   input           aw_ready_s1,
   input    [43:0] aw_addr_m1,
   input     [1:0] aw_burst_m1,
   input     [3:0] aw_cache_m1,
   input     [3:0] aw_id_m1,
   input     [7:0] aw_len_m1,
   input           aw_lock_m1,
   input     [2:0] aw_prot_m1,
   input     [3:0] aw_qos_m1,
   input     [2:0] aw_size_m1,
   input           aw_valid_m1,
   input     [1:0] aw_domain_m1,
   input     [3:0] aw_snoop_m1,
   input     [1:0] aw_bar_m1,
 
   input   [127:0] w_data_m1,
   input           w_last_m1,
   input    [15:0] w_strb_m1,
   input           w_valid_m1,
   input           w_ready_s1,
 
   input     [3:0] b_id_s1,
   input     [1:0] b_resp_s1,
   input           b_valid_s1,
   input           b_ready_m1,
 
   input   [127:0] r_data_s1,
   input     [3:0] r_id_s1,
   input           r_last_s1,
   input     [1:0] r_resp_s1,
   input           r_valid_s1,
   input           r_ready_m1,

   input           ac_ready_m1 ,
   input [2:0]     ac_prot_s1  ,
   input           ac_valid_s1 ,
   input [3:0]     ac_snoop_s1 ,
   input [43:0]    ac_addr_s1  ,

   input           cr_ready_s1 ,
   input           cr_valid_m1 ,
   input  [4:0]    cr_resp_m1  ,
                     
   input           cd_ready_s1 ,
   input           cd_valid_m1 ,
   input  [127:0]  cd_data_m1  ,
   input           cd_last_m1  ,
 
   output   [43:0] ar_addr_s1,
   output    [1:0] ar_burst_s1,
   output    [3:0] ar_cache_s1,
   output    [3:0] ar_id_s1,
   output    [7:0] ar_len_s1,
   output          ar_lock_s1,
   output    [2:0] ar_prot_s1,
   output    [3:0] ar_qos_s1,
   output    [2:0] ar_size_s1,
   output          ar_valid_s1,
   output          ar_ready_m1,
   output    [1:0] ar_domain_s1,
   output    [3:0] ar_snoop_s1,
   output    [1:0] ar_bar_s1,
 
   output   [43:0] aw_addr_s1,
   output    [1:0] aw_burst_s1,
   output    [3:0] aw_cache_s1,
   output    [3:0] aw_id_s1,
   output    [7:0] aw_len_s1,
   output          aw_lock_s1,
   output    [2:0] aw_prot_s1,
   output    [3:0] aw_qos_s1,
   output    [2:0] aw_size_s1,
   output          aw_valid_s1, 
   output          aw_ready_m1,
   output    [1:0] aw_domain_s1,
   output    [3:0] aw_snoop_s1,
   output    [1:0] aw_bar_s1,
 
   output    [3:0] b_id_m1,
   output    [1:0] b_resp_m1,
   output          b_valid_m1,
   output          b_ready_s1,
 
   output  [127:0] r_data_m1,
   output    [3:0] r_id_m1,
   output          r_last_m1,
   output    [1:0] r_resp_m1,
   output          r_valid_m1,
   output          r_ready_s1, 
 
   output  [127:0] w_data_s1,
   output          w_last_s1,
   output   [15:0] w_strb_s1,
   output          w_valid_s1,
   output          w_ready_m1,

   output          ac_ready_s1 ,
   output [2:0]    ac_prot_m1  ,
   output          ac_valid_m1 ,
   output [3:0]    ac_snoop_m1 ,
   output [43:0]   ac_addr_m1  ,

   output          cr_ready_m1 ,
   output          cr_valid_s1 ,
   output  [4:0]   cr_resp_s1  ,
                   
   output          cd_ready_m1 ,
   output          cd_valid_s1 ,
   output  [127:0] cd_data_s1  ,
   output          cd_last_s1  
);

   //----------------------------------------------------------------

   wire   [80:0] ar_data_init;
   wire   [80:0] ar_data_targ;
   wire   [80:0] aw_data_init;
   wire   [80:0] aw_data_targ;
   wire    [5:0] b_data_init;
   wire    [5:0] b_data_targ;
   wire   [134:0] r_data_init;
   wire   [134:0] r_data_targ;
   wire   [144:0] w_data_init;
   wire   [144:0] w_data_targ;
   wire   [50:0] ac_data_init;
   wire   [50:0] ac_data_targ;

   //----------------------------------------------------------------

   // lwsoc2fpga_ar_data_init assignment
   assign ar_data_init = {ar_domain_m1,
                          ar_snoop_m1,
                          ar_bar_m1,
                          ar_size_m1, 
                          ar_qos_m1, 
		          ar_prot_m1, 
			  ar_lock_m1, 
			  ar_len_m1, 
			  ar_id_m1, 
			  ar_cache_m1, 
			  ar_burst_m1, 
			  ar_addr_m1};

   // Register Slice instantiation for lwsoc2fpga ar channel
   rla_full_reg_slice #(
     .WIDTH(81),
     .PTR_WIDTH(1)
   ) archan_fifo (
     .clk(ar_clk),
     .rst_n(ar_rst_n),
     .valid_init(ar_valid_m1),
     .ready_init(ar_ready_m1),
     .data_init(ar_data_init),
     .readylatency_init(1'b0),
     .valid_targ(ar_valid_s1),
     .ready_targ(ar_ready_s1),
     .data_targ(ar_data_targ),
     .readylatency_targ(ar_latency_en)
   );

   // lwsoc2fpga_ar_data_targ assignment
   assign ar_addr_s1[43:0]       = ar_data_targ[43:0];
   assign ar_burst_s1[1:0]       = ar_data_targ[45:44];
   assign ar_cache_s1[3:0]       = ar_data_targ[49:46];
   assign ar_id_s1[3:0]          = ar_data_targ[53:50];
   assign ar_len_s1[7:0]         = ar_data_targ[61:54];
   assign ar_lock_s1             = ar_data_targ[62];
   assign ar_prot_s1[2:0]        = ar_data_targ[65:63];
   assign ar_qos_s1[3:0]         = ar_data_targ[69:66];
   assign ar_size_s1[2:0]        = ar_data_targ[72:70];
   assign ar_bar_s1[1:0]         = ar_data_targ[74:73];
   assign ar_snoop_s1[3:0]       = ar_data_targ[78:75];
   assign ar_domain_s1[1:0]      = ar_data_targ[80:79];

   //----------------------------------------------------------------

   // lwsoc2fpga_aw_data_init assignment
   assign aw_data_init        = { aw_domain_m1,
                                  aw_snoop_m1,
                                  aw_bar_m1,  
                                  aw_size_m1, 
                                  aw_qos_m1, 
				  aw_prot_m1, 
				  aw_lock_m1, 
				  aw_len_m1, 
				  aw_id_m1, 
				  aw_cache_m1, 
				  aw_burst_m1, 
				  aw_addr_m1 }; 

   // Register Slice instantiation for lwsoc2fpga aw channel
   rla_full_reg_slice #(
     .WIDTH(81),
     .PTR_WIDTH(1)
   ) awchan_fifo (
     .clk(aw_clk),
     .rst_n(aw_rst_n),
     .valid_init(aw_valid_m1),
     .ready_init(aw_ready_m1),
     .data_init(aw_data_init),
     .readylatency_init(1'b0),
     .valid_targ(aw_valid_s1),
     .ready_targ(aw_ready_s1),
     .data_targ(aw_data_targ),
     .readylatency_targ(aw_latency_en)
   );

   // lwsoc2fpga_aw_data_targ assignment
   assign aw_addr_s1[43:0]       = aw_data_targ[43:0];
   assign aw_burst_s1[1:0]       = aw_data_targ[45:44];
   assign aw_cache_s1[3:0]       = aw_data_targ[49:46];
   assign aw_id_s1[3:0]          = aw_data_targ[53:50];
   assign aw_len_s1[7:0]         = aw_data_targ[61:54];
   assign aw_lock_s1             = aw_data_targ[62];
   assign aw_prot_s1[2:0]        = aw_data_targ[65:63];
   assign aw_qos_s1[3:0]         = aw_data_targ[69:66];
   assign aw_size_s1[2:0]        = aw_data_targ[72:70];
   assign aw_bar_s1[1:0]         = aw_data_targ[74:73];
   assign aw_snoop_s1[3:0]       = aw_data_targ[78:75];
   assign aw_domain_s1[1:0]      = aw_data_targ[80:79];

   //----------------------------------------------------------------
   
   // lwsoc2fpga_w_data_init assignment
   assign w_data_init  = {w_strb_m1, 
                          w_last_m1, 
		          w_data_m1};

   // Register Slice instantiation for lwsoc2fpga w channel
   rla_full_reg_slice #(
     .WIDTH(145),
     .PTR_WIDTH(1)
   ) wchan_fifo (
     .clk(w_clk),
     .rst_n(w_rst_n),
     .valid_init(w_valid_m1),
     .ready_init(w_ready_m1),
     .data_init(w_data_init),
     .readylatency_init(1'b0),
     .valid_targ(w_valid_s1),
     .ready_targ(w_ready_s1),
     .data_targ(w_data_targ),
     .readylatency_targ(w_latency_en)
   );

   // lwsoc2fpga_w_data_targ assignment
   assign w_data_s1[127:0]        = w_data_targ[127:0];
   assign w_last_s1              = w_data_targ[128];
   assign w_strb_s1[15:0]         = w_data_targ[144:129];

   //----------------------------------------------------------------
   
   // lwsoc2fpga_r_data_init assignment
   assign r_data_init = { r_resp_s1, 
                          r_last_s1, 
	                  r_id_s1, 
			  r_data_s1 };
   
   // Register Slice instantiation for lwsoc2fpga r channel
   rla_full_reg_slice #(
     .WIDTH(135),
     .PTR_WIDTH(3)
   ) rchan_fifo (
     .clk(r_clk),
     .rst_n(r_rst_n),
     .valid_init(r_valid_s1),
     .ready_init(r_ready_s1),
     .data_init(r_data_init),
     .readylatency_init(r_latency_en),
     .valid_targ(r_valid_m1),
     .ready_targ(r_ready_m1),
     .data_targ(r_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_r_data_targ assignment
   assign r_data_m1[127:0]   = r_data_targ[127:0];
   assign r_id_m1[3:0]      = r_data_targ[131:128];
   assign r_last_m1         = r_data_targ[132];
   assign r_resp_m1[1:0]    = r_data_targ[134:133];

   //----------------------------------------------------------------

   // lwsoc2fpga_b_data_init assignment
   assign b_data_init = { b_resp_s1, 
                          b_id_s1 }; 
   
   // Register Slice instantiation for lwsoc2fpga b channel
   rla_full_reg_slice #(
     .WIDTH(6),
     .PTR_WIDTH(3)
   ) bchan_fifo (
     .clk(b_clk),
     .rst_n(b_rst_n),
     .valid_init(b_valid_s1),
     .ready_init(b_ready_s1),
     .data_init(b_data_init),
     .readylatency_init(b_latency_en),
     .valid_targ(b_valid_m1),
     .ready_targ(b_ready_m1),
     .data_targ(b_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_b_data_targ assignment
   assign b_id_m1[3:0]      = b_data_targ[3:0];
   assign b_resp_m1[1:0]    = b_data_targ[5:4];

   //---------------------------------------------------------------- 

   // lwsoc2fpga_b_data_init assignment
   assign ac_data_init = { ac_prot_s1, 
                           ac_snoop_s1,
                           ac_addr_s1  }; 
   
   // Register Slice instantiation for lwsoc2fpga b channel
   rla_full_reg_slice #(
     .WIDTH(51),
     .PTR_WIDTH(3)
   ) ac_chan_fifo (
     .clk(ac_clk),
     .rst_n(ac_rst_n),
     .valid_init(ac_valid_s1),
     .ready_init(ac_ready_s1),
     .data_init(ac_data_init),
     .readylatency_init(ac_latency_en),
     .valid_targ(ac_valid_m1),
     .ready_targ(ac_ready_m1),
     .data_targ(ac_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_b_data_targ assignment
   assign ac_prot_m1      = ac_data_targ[50:48];
   assign ac_snoop_m1     = ac_data_targ[47:44];
   assign ac_addr_m1      = ac_data_targ[43:0]; 

   //----------------------------------------------------------------

   // Register Slice instantiation for lwsoc2fpga w channel
   rla_full_reg_slice #(
     .WIDTH(5),
     .PTR_WIDTH(1)
   ) cr_chan_fifo (
     .clk(cr_clk),
     .rst_n(cr_rst_n),
     .valid_init(cr_valid_m1),
     .ready_init(cr_ready_m1),
     .data_init(cr_resp_m1),
     .readylatency_init(1'b0),
     .valid_targ(cr_valid_s1),
     .ready_targ(cr_ready_s1),
     .data_targ(cr_resp_s1),
     .readylatency_targ(cr_latency_en)
   ); 

   //----------------------------------------------------------------

   rla_full_reg_slice #(
     .WIDTH(129),
     .PTR_WIDTH(1)
   ) cd_chan_fifo (
     .clk(cd_clk),
     .rst_n(cd_rst_n),
     .valid_init(cd_valid_m1),
     .ready_init(cd_ready_m1),
     .data_init({cd_last_m1, cd_data_m1}),
     .readylatency_init(1'b0),
     .valid_targ(cd_valid_s1),
     .ready_targ(cd_ready_s1),
     .data_targ({cd_last_s1, cd_data_s1}),
     .readylatency_targ(cd_latency_en)
   );

   //----------------------------------------------------------------

endmodule
//------------------------------------------------------------------------------

//
// Description:
// This is a synthesizable ready latency pipeline of selectable depth, for use with the Clarke HPS LW bridge.


module f2s_ready_latency_pipe (
  input           ar_clk,
  input           ar_rst_n,
  input           aw_clk,
  input           aw_rst_n,
  input           b_clk,
  input           b_rst_n,
  input           r_clk,
  input           r_rst_n,
  input           w_clk,
  input           w_rst_n,
  input           ac_clk,
  input           ac_rst_n,
  input           cr_clk,
  input           cr_rst_n,
  input           cd_clk,
  input           cd_rst_n,

  input [2:0]     ar_ready_latency,
  input [2:0]     ar_valid_latency,
  input [2:0]     aw_ready_latency,
  input [2:0]     aw_valid_latency,
  input [2:0]     w_ready_latency,
  input [2:0]     w_valid_latency,
  input [2:0]     r_ready_latency,
  input [2:0]     r_valid_latency,
  input [2:0]     b_ready_latency,
  input [2:0]     b_valid_latency,
  input [2:0]     ac_valid_latency,
  input [2:0]     ac_ready_latency,
  input [2:0]     cr_valid_latency,
  input [2:0]     cr_ready_latency,
  input [2:0]     cd_valid_latency,
  input [2:0]     cd_ready_latency,
  input [2:0]     ack_latency,

  input           ar_ready_s1,
  input    [43:0] ar_addr_m1,
  input     [1:0] ar_burst_m1,
  input     [3:0] ar_cache_m1,
  input     [3:0] ar_id_m1,
  input     [7:0] ar_len_m1,
  input           ar_lock_m1,
  input     [2:0] ar_prot_m1,
  input     [3:0] ar_qos_m1,
  input     [2:0] ar_size_m1,
  input           ar_valid_m1,
  input     [1:0] ar_domain_m1,
  input     [3:0] ar_snoop_m1,
  input     [1:0] ar_bar_m1,

  input           aw_ready_s1,
  input    [43:0] aw_addr_m1,
  input     [1:0] aw_burst_m1,
  input     [3:0] aw_cache_m1,
  input     [3:0] aw_id_m1,
  input     [7:0] aw_len_m1,
  input           aw_lock_m1,
  input     [2:0] aw_prot_m1,
  input     [3:0] aw_qos_m1,
  input     [2:0] aw_size_m1,
  input           aw_valid_m1,
  input     [1:0] aw_domain_m1,
  input     [3:0] aw_snoop_m1,
  input     [1:0] aw_bar_m1,

  input   [127:0] w_data_m1,
  input           w_last_m1,
  input    [15:0] w_strb_m1,
  input           w_valid_m1,
  input           w_ready_s1,
  input           w_ack_m1,

  input     [3:0] b_id_s1,
  input     [1:0] b_resp_s1,
  input           b_valid_s1,
  input           b_ready_m1,

  input    [127:0] r_data_s1,
  input     [3:0] r_id_s1,
  input           r_last_s1,
  input     [1:0] r_resp_s1,
  input           r_valid_s1,
  input           r_ready_m1,
  input           r_ack_m1,

  input          ac_ready_m1 ,
  input [2:0]    ac_prot_s1  ,
  input          ac_valid_s1 ,
  input [3:0]    ac_snoop_s1 ,
  input [43:0]   ac_addr_s1  ,

  input          cr_ready_s1 ,
  input          cr_valid_m1 ,
  input  [4:0]   cr_resp_m1  ,
                   
  input          cd_ready_s1 ,
  input          cd_valid_m1 ,
  input  [127:0] cd_data_m1  ,
  input          cd_last_m1  ,

  output   [43:0] ar_addr_s1,
  output    [1:0] ar_burst_s1,
  output    [3:0] ar_cache_s1,
  output    [3:0] ar_id_s1,
  output    [7:0] ar_len_s1,
  output          ar_lock_s1,
  output    [2:0] ar_prot_s1,
  output    [3:0] ar_qos_s1,
  output    [2:0] ar_size_s1,
  output          ar_valid_s1,
  output          ar_ready_m1,
  output    [1:0] ar_domain_s1,
  output    [3:0] ar_snoop_s1,
  output    [1:0] ar_bar_s1,
 

  output   [43:0] aw_addr_s1,
  output    [1:0] aw_burst_s1,
  output    [3:0] aw_cache_s1,
  output    [3:0] aw_id_s1,
  output    [7:0] aw_len_s1,
  output          aw_lock_s1,
  output    [2:0] aw_prot_s1,
  output    [3:0] aw_qos_s1,
  output    [2:0] aw_size_s1,
  output          aw_valid_s1, 
  output          aw_ready_m1,
  output    [1:0] aw_domain_s1,
  output    [3:0] aw_snoop_s1,
  output    [1:0] aw_bar_s1,

  output    [3:0] b_id_m1,
  output    [1:0] b_resp_m1,
  output          b_valid_m1,
  output          b_ready_s1,

  output  [127:0] r_data_m1,
  output    [3:0] r_id_m1,
  output          r_last_m1,
  output    [1:0] r_resp_m1,
  output          r_valid_m1,
  output          r_ready_s1, 
  output          r_ack_s1,

  output  [127:0] w_data_s1,
  output          w_last_s1,
  output   [15:0] w_strb_s1,
  output          w_valid_s1,
  output          w_ready_m1,
  output          w_ack_s1,

  output          ac_ready_s1 ,
  output [2:0]    ac_prot_m1  ,
  output          ac_valid_m1 ,
  output [3:0]    ac_snoop_m1 ,
  output [43:0]   ac_addr_m1  ,

  output          cr_ready_m1 ,
  output          cr_valid_s1 ,
  output  [4:0]   cr_resp_s1  ,
                  
  output          cd_ready_m1 ,
  output          cd_valid_s1 ,
  output  [127:0] cd_data_s1  ,
  output          cd_last_s1  

);

   //-------------------------------------------------------------

   wire [76:0] aw_source_payload;
   wire [76:0] aw_sink_payload;

   wire [76:0] ar_source_payload;
   wire [76:0] ar_sink_payload;

   wire [50:0] ac_source_payload;
   wire [50:0] ac_sink_payload;

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(2)) bchannel_rla (
      .reset_n(ar_rst_n),
      .clk(ar_clk),
   
      .source_data (b_resp_s1),
      .source_id   (b_id_s1),
      //.source_valid(b_valid_s1 & b_ready_s1),
      .source_valid(b_valid_s1),
   
      .sink_ready(b_ready_m1),
   
      .ready_delay(b_ready_latency),
      .valid_delay(b_valid_latency),
   
      .sink_data (b_resp_m1),
      .sink_id   (b_id_m1),
      .sink_valid(b_valid_m1),
      
      .source_ready(b_ready_s1) 
   );

   //-------------------------------------------------------------
   
   ready_latency_regslice #(.DATA_WIDTH(131)) rchannel_rla (
      .reset_n(aw_rst_n),
      .clk(aw_clk),
   
      .source_data ({r_last_s1, r_resp_s1, r_data_s1}),
      .source_id   (r_id_s1),
      //.source_valid(r_valid_s1 & r_ready_s1),
      .source_valid(r_valid_s1),
   
      .sink_ready(r_ready_m1),
   
      .ready_delay(r_ready_latency),
      .valid_delay(r_valid_latency),
   
      .sink_data ({r_last_m1, r_resp_m1, r_data_m1}),
      .sink_id   (r_id_m1),
      .sink_valid(r_valid_m1),
      
      .source_ready(r_ready_s1) 
   );

   //-------------------------------------------------------------

   assign aw_source_payload = { aw_domain_m1,
                                aw_snoop_m1,
				aw_bar_m1,
				aw_size_m1, // 3 
                                aw_qos_m1,  // 4
                                aw_prot_m1, // 3
				aw_lock_m1, // 1
				aw_len_m1,  // 8
				aw_cache_m1,// 4
				aw_burst_m1,// 2
				aw_addr_m1  // 44
		              };            // 69 bits total

   ready_latency_regslice #(.DATA_WIDTH(77)) awchannel_rla (
      .reset_n(aw_rst_n),
      .clk(aw_clk),
   
      .source_data (aw_source_payload),
      .source_id   (aw_id_m1),
      .source_valid(aw_valid_m1),
   
      .sink_ready(aw_ready_s1),
   
      .ready_delay(aw_ready_latency),
      .valid_delay(aw_valid_latency),
   
      .sink_data (aw_sink_payload),
      .sink_id   (aw_id_s1),
      .sink_valid(aw_valid_s1),
     
      .source_ready(aw_ready_m1) 
   );

   assign  { aw_domain_s1, // 2
             aw_snoop_s1,  // 4
	     aw_bar_s1,    // 2
             aw_size_s1,   // 3 
             aw_qos_s1,    // 4
             aw_prot_s1,   // 3
             aw_lock_s1,   // 1
	     aw_len_s1,    // 8
	     aw_cache_s1,  // 4
	     aw_burst_s1,  // 2
	     aw_addr_s1    // 44
           }               = aw_sink_payload;

   //-------------------------------------------------------------

   assign ar_source_payload = { ar_domain_m1,
                                ar_snoop_m1,
				ar_bar_m1,
				ar_size_m1, //  
                                ar_qos_m1,  //  
                                ar_prot_m1, //  
				ar_lock_m1, //  
				ar_len_m1,  //  
				ar_cache_m1,//  
				ar_burst_m1,//  
				ar_addr_m1  //  
		              };            //   bits total

   ready_latency_regslice #(.DATA_WIDTH(77)) archannel_rla (
      .reset_n(ar_rst_n),
      .clk(ar_clk),
   
      .source_data (ar_source_payload),
      .source_id   (ar_id_m1),
      .source_valid(ar_valid_m1),
   
      .sink_ready(ar_ready_s1),
   
      .ready_delay(ar_ready_latency),
      .valid_delay(ar_valid_latency),
   
      .sink_data (ar_sink_payload),
      .sink_id   (ar_id_s1),
      .sink_valid(ar_valid_s1),
      
      .source_ready(ar_ready_m1) 
   );

   assign  { ar_domain_s1,
             ar_snoop_s1,
	     ar_bar_s1,  //
	     ar_size_s1, // 3 
             ar_qos_s1,  // 4
             ar_prot_s1, // 3
             ar_lock_s1, // 1
	     ar_len_s1,  // 8
	     ar_cache_s1,// 4
	     ar_burst_s1,// 2
	     ar_addr_s1  // 44
           }                   = ar_sink_payload;

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(145)) wchannel_rla (
      .reset_n(w_rst_n),
      .clk(w_clk),
   
      .source_data ({w_last_m1, w_strb_m1, w_data_m1}),
      .source_id   ({4{1'b0}}),
      .source_valid(w_valid_m1),
   
      .sink_ready(w_ready_s1),
   
      .ready_delay(w_ready_latency),
      .valid_delay(w_valid_latency),
   
      .sink_data ({w_last_s1, w_strb_s1, w_data_s1}),
      .sink_id   (),
      .sink_valid(w_valid_s1),
      
      .source_ready(w_ready_m1) 
   );

   //-------------------------------------------------------------

   assign ac_source_payload = {   ac_prot_s1 ,   // 3
                                  ac_snoop_s1,   // 4
                                  ac_addr_s1     // 44
		              };                 // 51 bits total

   ready_latency_regslice #(.DATA_WIDTH(51)) ac_channel_rla (
      .reset_n(ar_rst_n),
      .clk(ar_clk),
   
      .source_data (ac_source_payload),
      .source_id   ({4{1'b0}}),
      .source_valid(ac_valid_s1),
   
      .sink_ready(ac_ready_m1),
   
      .ready_delay(ac_ready_latency),
      .valid_delay(ac_valid_latency),
   
      .sink_data (ac_sink_payload),
      .sink_id   (),
      .sink_valid(ac_valid_m1),
      
      .source_ready(ac_ready_s1) 
   );

   assign  { ac_prot_m1, 
	     ac_snoop_m1,
	     ac_addr_m1 
           }                   = ac_sink_payload;

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(5)) cr_channel_rla (
      .reset_n(w_rst_n),
      .clk(w_clk),
   
      .source_data (cr_resp_m1),
      .source_id   ({4{1'b0}}),
      .source_valid(cr_valid_m1),
   
      .sink_ready(cr_ready_s1),
   
      .ready_delay(cr_ready_latency),
      .valid_delay(cr_valid_latency),
   
      .sink_data (cr_resp_s1),
      .sink_id   (),
      .sink_valid(cr_valid_s1),
      
      .source_ready(cr_ready_m1) 
   );

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(129)) cd_channel_rla (
      .reset_n(w_rst_n),
      .clk(w_clk),
   
      .source_data ({cd_last_m1, cd_data_m1}),
      .source_id   ({4{1'b0}}),
      .source_valid( cd_valid_m1 ),
   
      .sink_ready(cd_ready_s1),
   
      .ready_delay(cd_ready_latency),
      .valid_delay(cd_valid_latency),
   
      .sink_data ({cd_last_s1, cd_data_s1}),
      .sink_id   (),
      .sink_valid(cd_valid_s1),
      
      .source_ready(cd_ready_m1) 
   );

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(2)) ack_channel_rla (
      .reset_n(w_rst_n),
      .clk(w_clk),
   
      .source_data ({w_ack_m1, r_ack_m1}),
      .source_id   ({4{1'b0}}),
      .source_valid( 1'b1 ),
   
      .sink_ready( 1'b1 ),
   
      .ready_delay(ack_latency),
      .valid_delay(ack_latency),
   
      .sink_data ({w_ack_s1, r_ack_s1}),
      .sink_id   (),
      .sink_valid(),
      
      .source_ready() 
   );

   //-------------------------------------------------------------
	      
endmodule
//------------------------------------------------------------------------------

// Description:
// This is a synthesizable ready-latency adapter.  It comprises a
// ready-latency pipeline of selectable depth and a ready-latency fifo. 
// Instantiate it between the Clarke HPS LWBridge and an AXI-compliant slave.


`define SVT_AXI_MAX_DATA_WIDTH 128
module f2sdram_ready_latency_adapter (
  input           ar_clk,
  input           ar_rst_n,
  input           aw_clk,
  input           aw_rst_n,
  input           b_clk,
  input           b_rst_n,
  input           r_clk,
  input           r_rst_n,
  input           w_clk,
  input           w_rst_n,

  input [2:0]     ar_ready_latency,
  input [2:0]     ar_valid_latency,
  input [2:0]     aw_ready_latency,
  input [2:0]     aw_valid_latency,
  input [2:0]     w_ready_latency,
  input [2:0]     w_valid_latency,
  input [2:0]     r_ready_latency,
  input [2:0]     r_valid_latency,
  input [2:0]     b_ready_latency,
  input [2:0]     b_valid_latency,

  input           ar_ready_s1,
  input    [36:0] ar_addr_m1,
  input     [1:0] ar_burst_m1,
  input     [3:0] ar_cache_m1,
  input     [3:0] ar_id_m1,
  input     [7:0] ar_len_m1,
  input           ar_lock_m1,
  input     [2:0] ar_prot_m1,
  input     [3:0] ar_qos_m1,
  input     [2:0] ar_size_m1,
  input           ar_valid_m1,

  input           aw_ready_s1,
  input    [36:0] aw_addr_m1,
  input     [1:0] aw_burst_m1,
  input     [3:0] aw_cache_m1,
  input     [3:0] aw_id_m1,
  input     [7:0] aw_len_m1,
  input           aw_lock_m1,
  input     [2:0] aw_prot_m1,
  input     [3:0] aw_qos_m1,
  input     [2:0] aw_size_m1,
  input           aw_valid_m1,

  input  [`SVT_AXI_MAX_DATA_WIDTH-1:0]   w_data_m1,
  input           w_last_m1,
  input     [15:0] w_strb_m1,
  input           w_valid_m1,
  input           w_ready_s1,

  input     [3:0] b_id_s1,
  input     [1:0] b_resp_s1,
  input           b_valid_s1,
  input           b_ready_m1,

  input  [`SVT_AXI_MAX_DATA_WIDTH-1:0] r_data_s1,
  input     [3:0] r_id_s1,
  input           r_last_s1,
  input     [1:0] r_resp_s1,
  input           r_valid_s1,
  input           r_ready_m1,

  output   [36:0] ar_addr_s1,
  output    [1:0] ar_burst_s1,
  output    [3:0] ar_cache_s1,
  output    [3:0] ar_id_s1,
  output    [7:0] ar_len_s1,
  output          ar_lock_s1,
  output    [2:0] ar_prot_s1,
  output    [3:0] ar_qos_s1,
  output    [2:0] ar_size_s1,
  output          ar_valid_s1,
  output          ar_ready_m1,

  output   [36:0] aw_addr_s1,
  output    [1:0] aw_burst_s1,
  output    [3:0] aw_cache_s1,
  output    [3:0] aw_id_s1,
  output    [7:0] aw_len_s1,
  output          aw_lock_s1,
  output    [2:0] aw_prot_s1,
  output    [3:0] aw_qos_s1,
  output    [2:0] aw_size_s1,
  output          aw_valid_s1, 
  output          aw_ready_m1,

  output    [3:0] b_id_m1,
  output    [1:0] b_resp_m1,
  output          b_valid_m1,
  output          b_ready_s1,

  output   [`SVT_AXI_MAX_DATA_WIDTH-1:0] r_data_m1,
  output    [3:0] r_id_m1,
  output          r_last_m1,
  output    [1:0] r_resp_m1,
  output          r_valid_m1,
  output          r_ready_s1, 

  output   [`SVT_AXI_MAX_DATA_WIDTH-1:0] w_data_s1,
  output          w_last_s1,
  output    [15:0] w_strb_s1,
  output          w_valid_s1,
  output          w_ready_m1
);            

   wire        rlp_rlf_arready_h; 
   wire        rlp_rlf_awready_h;
   wire        rlp_rlf_wready_h;

   wire [3:0]  rlp_rlf_bid_h;
   wire [1:0]  rlp_rlf_bresp_h;
   wire        rlp_rlf_bvalid_h;

   wire rlf_rlp_bready_h;

   wire [`SVT_AXI_MAX_DATA_WIDTH-1:0] rlp_rlf_rdata_h;
   wire [3:0]  rlp_rlf_rid_h;
   wire        rlp_rlf_rlast_h;
   wire [1:0]  rlp_rlf_rresp_h;
   wire        rlp_rlf_rvalid_h;
  
   wire rlf_rlp_rready_h;

   wire [36:0] rlf_rlp_araddr_h;
   wire [1:0]  rlf_rlp_arburst_h;
   wire [3:0]  rlf_rlp_arcache_h;
   wire [3:0]  rlf_rlp_arid_h;
   wire [7:0]  rlf_rlp_arlen_h;
   wire        rlf_rlp_arlock_h;
   wire [2:0]  rlf_rlp_arprot_h;
   wire [3:0]  rlf_rlp_arqos_h;
   wire [2:0]  rlf_rlp_arsize_h;
   wire        rlf_rlp_arvalid_h;

   wire [36:0] rlf_rlp_awaddr_h;
   wire [1:0]  rlf_rlp_awburst_h;
   wire [3:0]  rlf_rlp_awcache_h;
   wire [3:0]  rlf_rlp_awid_h;
   wire [7:0]  rlf_rlp_awlen_h;
   wire        rlf_rlp_awlock_h;
   wire [2:0]  rlf_rlp_awprot_h;
   wire [3:0]  rlf_rlp_awqos_h;
   wire [2:0]  rlf_rlp_awsize_h;
   wire        rlf_rlp_awvalid_h;

   wire [127:0] rlf_rlp_wdata_h;
   wire         rlf_rlp_wlast_h;
   wire [15:0]   rlf_rlp_wstrb_h;
   wire         rlf_rlp_wvalid_h;

   wire        ar_latency_en;
   wire        aw_latency_en;
   wire         w_latency_en;
   wire         r_latency_en;
   wire         b_latency_en;

   //---------------------------------------------------------------- 

   assign ar_latency_en = |{ar_ready_latency, ar_valid_latency};
   assign aw_latency_en = |{aw_ready_latency, aw_valid_latency};
   assign  w_latency_en = |{ w_ready_latency,  w_valid_latency};
   assign  r_latency_en = |{ r_ready_latency,  r_valid_latency};
   assign  b_latency_en = |{ b_ready_latency,  b_valid_latency};

   //---------------------------------------------------------------- 

   f2sdram_ready_latency_pipe f2sdram_rlp (
      .ar_clk  ( ar_clk  ),
      .ar_rst_n( ar_rst_n),
      .aw_clk  ( aw_clk  ),
      .aw_rst_n( aw_rst_n),
      .b_clk   ( b_clk   ),
      .b_rst_n ( b_rst_n ),
      .r_clk   ( r_clk   ),
      .r_rst_n ( r_rst_n ),
      .w_clk   ( w_clk   ),
      .w_rst_n ( w_rst_n ),

      .ar_ready_latency ( ar_ready_latency ),
      .ar_valid_latency ( ar_valid_latency ),
      .aw_ready_latency ( aw_ready_latency ),
      .aw_valid_latency ( aw_valid_latency ),
      .w_ready_latency  ( w_ready_latency  ),
      .w_valid_latency  ( w_valid_latency  ),
      .r_ready_latency  ( r_ready_latency  ),
      .r_valid_latency  ( r_valid_latency  ),
      .b_ready_latency  ( b_ready_latency  ),
      .b_valid_latency  ( b_valid_latency  ),

      .ar_ready_s1( ar_ready_s1 ),
      .ar_addr_m1 ( rlf_rlp_araddr_h ),
      .ar_burst_m1( rlf_rlp_arburst_h),
      .ar_cache_m1( rlf_rlp_arcache_h),
      .ar_id_m1   ( rlf_rlp_arid_h   ),
      .ar_len_m1  ( rlf_rlp_arlen_h  ),
      .ar_lock_m1 ( rlf_rlp_arlock_h ),
      .ar_prot_m1 ( rlf_rlp_arprot_h ),
      .ar_qos_m1  ( rlf_rlp_arqos_h  ),
      .ar_size_m1 ( rlf_rlp_arsize_h ),
      .ar_valid_m1( rlf_rlp_arvalid_h),

      .aw_ready_s1( aw_ready_s1 ),
      .aw_addr_m1 ( rlf_rlp_awaddr_h ),
      .aw_burst_m1( rlf_rlp_awburst_h),
      .aw_cache_m1( rlf_rlp_awcache_h),
      .aw_id_m1   ( rlf_rlp_awid_h   ),
      .aw_len_m1  ( rlf_rlp_awlen_h  ),
      .aw_lock_m1 ( rlf_rlp_awlock_h ),
      .aw_prot_m1 ( rlf_rlp_awprot_h ),
      .aw_qos_m1  ( rlf_rlp_awqos_h  ),
      .aw_size_m1 ( rlf_rlp_awsize_h ),
      .aw_valid_m1( rlf_rlp_awvalid_h),

      .w_data_m1  (rlf_rlp_wdata_h ),  
      .w_last_m1  (rlf_rlp_wlast_h ), 
      .w_strb_m1  (rlf_rlp_wstrb_h ),
      .w_valid_m1 (rlf_rlp_wvalid_h),
      .w_ready_s1 ( w_ready_s1 ),

      .b_id_s1    (b_id_s1     ),
      .b_resp_s1  (b_resp_s1   ),
      .b_valid_s1 (b_valid_s1  ),
      .b_ready_m1 (rlf_rlp_bready_h ),

      .r_data_s1  ( r_data_s1 ),
      .r_id_s1    ( r_id_s1   ),
      .r_last_s1  ( r_last_s1 ),
      .r_resp_s1  ( r_resp_s1 ),
      .r_valid_s1 ( r_valid_s1),
      .r_ready_m1 ( rlf_rlp_rready_h ),

      .ar_addr_s1 ( ar_addr_s1  ),
      .ar_burst_s1( ar_burst_s1 ),
      .ar_cache_s1( ar_cache_s1 ),
      .ar_id_s1   ( ar_id_s1    ),
      .ar_len_s1  ( ar_len_s1   ),
      .ar_lock_s1 ( ar_lock_s1  ),
      .ar_prot_s1 ( ar_prot_s1  ),
      .ar_qos_s1  ( ar_qos_s1   ),
      .ar_size_s1 ( ar_size_s1  ),
      .ar_valid_s1( ar_valid_s1 ),
      .ar_ready_m1( rlp_rlf_arready_h ),

      .aw_addr_s1 ( aw_addr_s1  ),
      .aw_burst_s1( aw_burst_s1 ),
      .aw_cache_s1( aw_cache_s1 ),
      .aw_id_s1   ( aw_id_s1    ),
      .aw_len_s1  ( aw_len_s1   ),
      .aw_lock_s1 ( aw_lock_s1  ),
      .aw_prot_s1 ( aw_prot_s1  ),
      .aw_qos_s1  ( aw_qos_s1   ),
      .aw_size_s1 ( aw_size_s1  ),
      .aw_valid_s1( aw_valid_s1 ), 
      .aw_ready_m1( rlp_rlf_awready_h ),

      .b_id_m1    ( rlp_rlf_bid_h    ),
      .b_resp_m1  ( rlp_rlf_bresp_h  ),
      .b_valid_m1 ( rlp_rlf_bvalid_h ),
      .b_ready_s1 ( b_ready_s1 ),

      .r_data_m1  ( rlp_rlf_rdata_h ),
      .r_id_m1    ( rlp_rlf_rid_h   ),
      .r_last_m1  ( rlp_rlf_rlast_h ),
      .r_resp_m1  ( rlp_rlf_rresp_h ),
      .r_valid_m1 ( rlp_rlf_rvalid_h),
      .r_ready_s1 ( r_ready_s1 ), 

      .w_data_s1  ( w_data_s1  ),
      .w_last_s1  ( w_last_s1  ),
      .w_strb_s1  ( w_strb_s1  ),
      .w_valid_s1 ( w_valid_s1 ),
      .w_ready_m1 ( rlp_rlf_wready_h ) 
   );

   //---------------------------------------------------------------- 

   f2sdram_ready_latency_fifo f2sdram_rlf (
      .ar_clk  ( ar_clk  ),
      .ar_rst_n( ar_rst_n),
      .aw_clk  ( aw_clk  ),
      .aw_rst_n( aw_rst_n),
      .b_clk   ( b_clk   ),
      .b_rst_n ( b_rst_n ),
      .r_clk   ( r_clk   ),
      .r_rst_n ( r_rst_n ),
      .w_clk   ( w_clk   ),
      .w_rst_n ( w_rst_n ),

      .ar_latency_en ( ar_latency_en  ),
      .aw_latency_en ( aw_latency_en  ),
      .w_latency_en  ( w_latency_en   ),
      .r_latency_en  ( r_latency_en   ),
      .b_latency_en  ( b_latency_en   ),

      .ar_ready_s1( rlp_rlf_arready_h ),
      .ar_addr_m1 ( ar_addr_m1  ),
      .ar_burst_m1( ar_burst_m1 ),
      .ar_cache_m1( ar_cache_m1 ),
      .ar_id_m1   ( ar_id_m1    ),
      .ar_len_m1  ( ar_len_m1   ),
      .ar_lock_m1 ( ar_lock_m1  ),
      .ar_prot_m1 ( ar_prot_m1  ),
      .ar_qos_m1  ( ar_qos_m1   ),
      .ar_size_m1 ( ar_size_m1  ),
      .ar_valid_m1( ar_valid_m1 ),

      .aw_ready_s1( rlp_rlf_awready_h ),
      .aw_addr_m1 ( aw_addr_m1  ),
      .aw_burst_m1( aw_burst_m1 ),
      .aw_cache_m1( aw_cache_m1 ),
      .aw_id_m1   ( aw_id_m1    ),
      .aw_len_m1  ( aw_len_m1   ),
      .aw_lock_m1 ( aw_lock_m1  ),
      .aw_prot_m1 ( aw_prot_m1  ),
      .aw_qos_m1  ( aw_qos_m1   ),
      .aw_size_m1 ( aw_size_m1  ),
      .aw_valid_m1( aw_valid_m1 ),

      .w_data_m1  ( w_data_m1  ),  
      .w_last_m1  ( w_last_m1  ), 
      .w_strb_m1  ( w_strb_m1  ),
      .w_valid_m1 ( w_valid_m1 ),
      .w_ready_s1 ( rlp_rlf_wready_h ),

      .b_id_s1    ( rlp_rlf_bid_h    ),
      .b_resp_s1  ( rlp_rlf_bresp_h  ),
      .b_valid_s1 ( rlp_rlf_bvalid_h ),
      .b_ready_m1 ( b_ready_m1 ),

      .r_data_s1  ( rlp_rlf_rdata_h  ),
      .r_id_s1    ( rlp_rlf_rid_h    ),
      .r_last_s1  ( rlp_rlf_rlast_h  ),
      .r_resp_s1  ( rlp_rlf_rresp_h  ),
      .r_valid_s1 ( rlp_rlf_rvalid_h ),
      .r_ready_m1 ( r_ready_m1 ),

      .ar_addr_s1 ( rlf_rlp_araddr_h  ),
      .ar_burst_s1( rlf_rlp_arburst_h ),
      .ar_cache_s1( rlf_rlp_arcache_h ),
      .ar_id_s1   ( rlf_rlp_arid_h    ),
      .ar_len_s1  ( rlf_rlp_arlen_h   ),
      .ar_lock_s1 ( rlf_rlp_arlock_h  ),
      .ar_prot_s1 ( rlf_rlp_arprot_h  ),
      .ar_qos_s1  ( rlf_rlp_arqos_h   ),
      .ar_size_s1 ( rlf_rlp_arsize_h  ),
      .ar_valid_s1( rlf_rlp_arvalid_h ),
      .ar_ready_m1( ar_ready_m1 ),

      .aw_addr_s1 ( rlf_rlp_awaddr_h  ),
      .aw_burst_s1( rlf_rlp_awburst_h ),
      .aw_cache_s1( rlf_rlp_awcache_h ),
      .aw_id_s1   ( rlf_rlp_awid_h    ),
      .aw_len_s1  ( rlf_rlp_awlen_h   ),
      .aw_lock_s1 ( rlf_rlp_awlock_h  ),
      .aw_prot_s1 ( rlf_rlp_awprot_h  ),
      .aw_qos_s1  ( rlf_rlp_awqos_h   ),
      .aw_size_s1 ( rlf_rlp_awsize_h  ),
      .aw_valid_s1( rlf_rlp_awvalid_h ), 
      .aw_ready_m1( aw_ready_m1 ),

      .b_id_m1    ( b_id_m1    ),
      .b_resp_m1  ( b_resp_m1  ),
      .b_valid_m1 ( b_valid_m1 ),
      .b_ready_s1 ( rlf_rlp_bready_h ),

      .r_data_m1  ( r_data_m1  ),
      .r_id_m1    ( r_id_m1    ),
      .r_last_m1  ( r_last_m1  ),
      .r_resp_m1  ( r_resp_m1  ),
      .r_valid_m1 ( r_valid_m1 ),
      .r_ready_s1 ( rlf_rlp_rready_h ), 

      .w_data_s1  ( rlf_rlp_wdata_h  ),
      .w_last_s1  ( rlf_rlp_wlast_h  ),
      .w_strb_s1  ( rlf_rlp_wstrb_h  ),
      .w_valid_s1 ( rlf_rlp_wvalid_h ),
      .w_ready_m1 ( w_ready_m1 ) 
   ); 

endmodule
//------------------------------------------------------------------------------

// Description:
// This is a synthesizable FIFO for buffering lwbridge AXI signals from the
// ready-latency pipeline


module f2sdram_ready_latency_fifo (
   input           ar_clk,
   input           ar_rst_n,
   input           aw_clk,
   input           aw_rst_n,
   input           b_clk,
   input           b_rst_n,
   input           r_clk,
   input           r_rst_n,
   input           w_clk,
   input           w_rst_n,

   input           ar_latency_en,
   input           aw_latency_en,
   input           w_latency_en,
   input           r_latency_en,
   input           b_latency_en,
 
   input           ar_ready_s1,
   input    [36:0] ar_addr_m1,
   input     [1:0] ar_burst_m1,
   input     [3:0] ar_cache_m1,
   input     [3:0] ar_id_m1,
   input     [7:0] ar_len_m1,
   input           ar_lock_m1,
   input     [2:0] ar_prot_m1,
   input     [3:0] ar_qos_m1,
   input     [2:0] ar_size_m1,
   input           ar_valid_m1,
 
   input           aw_ready_s1,
   input    [36:0] aw_addr_m1,
   input     [1:0] aw_burst_m1,
   input     [3:0] aw_cache_m1,
   input     [3:0] aw_id_m1,
   input     [7:0] aw_len_m1,
   input           aw_lock_m1,
   input     [2:0] aw_prot_m1,
   input     [3:0] aw_qos_m1,
   input     [2:0] aw_size_m1,
   input           aw_valid_m1,
 
   input    [127:0] w_data_m1,
   input           w_last_m1,
   input     [15:0] w_strb_m1,
   input           w_valid_m1,
   input           w_ready_s1,
 
   input     [3:0] b_id_s1,
   input     [1:0] b_resp_s1,
   input           b_valid_s1,
   input           b_ready_m1,
 
   input    [127:0] r_data_s1,
   input     [3:0] r_id_s1,
   input           r_last_s1,
   input     [1:0] r_resp_s1,
   input           r_valid_s1,
   input           r_ready_m1,
 
   output   [36:0] ar_addr_s1,
   output    [1:0] ar_burst_s1,
   output    [3:0] ar_cache_s1,
   output    [3:0] ar_id_s1,
   output    [7:0] ar_len_s1,
   output          ar_lock_s1,
   output    [2:0] ar_prot_s1,
   output    [3:0] ar_qos_s1,
   output    [2:0] ar_size_s1,
   output          ar_valid_s1,
   output          ar_ready_m1,
 
   output   [36:0] aw_addr_s1,
   output    [1:0] aw_burst_s1,
   output    [3:0] aw_cache_s1,
   output    [3:0] aw_id_s1,
   output    [7:0] aw_len_s1,
   output          aw_lock_s1,
   output    [2:0] aw_prot_s1,
   output    [3:0] aw_qos_s1,
   output    [2:0] aw_size_s1,
   output          aw_valid_s1, 
   output          aw_ready_m1,
 
   output    [3:0] b_id_m1,
   output    [1:0] b_resp_m1,
   output          b_valid_m1,
   output          b_ready_s1,
 
   output   [127:0] r_data_m1,
   output    [3:0] r_id_m1,
   output          r_last_m1,
   output    [1:0] r_resp_m1,
   output          r_valid_m1,
   output          r_ready_s1, 
 
   output   [127:0] w_data_s1,
   output          w_last_s1,
   output    [15:0] w_strb_s1,
   output          w_valid_s1,
   output          w_ready_m1
);

   //----------------------------------------------------------------

   wire   [65:0] ar_data_init;
   wire   [65:0] ar_data_targ;
   wire   [65:0] aw_data_init;
   wire   [65:0] aw_data_targ;
   wire    [5:0] b_data_init;
   wire    [5:0] b_data_targ;
   wire   [134:0] r_data_init;
   wire   [134:0] r_data_targ;
   wire   [144:0] w_data_init;
   wire   [144:0] w_data_targ;

   //----------------------------------------------------------------

   // lwsoc2fpga_ar_data_init assignment
   assign ar_data_init = {ar_size_m1, 
                          ar_qos_m1, 
		          ar_prot_m1, 
			  ar_lock_m1, 
			  ar_len_m1, 
			  ar_id_m1, 
			  ar_cache_m1, 
			  ar_burst_m1, 
			  ar_addr_m1};

   // Register Slice instantiation for lwsoc2fpga ar channel
   rla_full_reg_slice #(
     .WIDTH(66),
     .PTR_WIDTH(1)
   ) archan_fifo (
     .clk(ar_clk),
     .rst_n(ar_rst_n),
     .valid_init(ar_valid_m1),
     .ready_init(ar_ready_m1),
     .data_init(ar_data_init),
     .readylatency_init(1'b0),
     .valid_targ(ar_valid_s1),
     .ready_targ(ar_ready_s1),
     .data_targ(ar_data_targ),
     .readylatency_targ(ar_latency_en)
   );

   // lwsoc2fpga_ar_data_targ assignment
   assign ar_addr_s1[36:0]       = ar_data_targ[36:0];
   assign ar_burst_s1[1:0]       = ar_data_targ[38:37];
   assign ar_cache_s1[3:0]       = ar_data_targ[42:39];
   assign ar_id_s1[3:0]          = ar_data_targ[46:43];
   assign ar_len_s1[7:0]         = ar_data_targ[54:47];
   assign ar_lock_s1             = ar_data_targ[55];
   assign ar_prot_s1[2:0]        = ar_data_targ[58:56];
   assign ar_qos_s1[3:0]         = ar_data_targ[62:59];
   assign ar_size_s1[2:0]        = ar_data_targ[65:63];

   //----------------------------------------------------------------

   // lwsoc2fpga_aw_data_init assignment
   assign aw_data_init        = { aw_size_m1, 
                                  aw_qos_m1, 
				  aw_prot_m1, 
				  aw_lock_m1, 
				  aw_len_m1, 
				  aw_id_m1, 
				  aw_cache_m1, 
				  aw_burst_m1, 
				  aw_addr_m1 }; 

   // Register Slice instantiation for lwsoc2fpga aw channel
   rla_full_reg_slice #(
     .WIDTH(66),
     .PTR_WIDTH(1)
   ) awchan_fifo (
     .clk(aw_clk),
     .rst_n(aw_rst_n),
     .valid_init(aw_valid_m1),
     .ready_init(aw_ready_m1),
     .data_init(aw_data_init),
     .readylatency_init(1'b0),
     .valid_targ(aw_valid_s1),
     .ready_targ(aw_ready_s1),
     .data_targ(aw_data_targ),
     .readylatency_targ(aw_latency_en)
   );

   // lwsoc2fpga_aw_data_targ assignment
   assign aw_addr_s1[36:0]       = aw_data_targ[36:0];
   assign aw_burst_s1[1:0]       = aw_data_targ[38:37];
   assign aw_cache_s1[3:0]       = aw_data_targ[42:39];
   assign aw_id_s1[3:0]          = aw_data_targ[46:43];
   assign aw_len_s1[7:0]         = aw_data_targ[54:47];
   assign aw_lock_s1             = aw_data_targ[55];
   assign aw_prot_s1[2:0]        = aw_data_targ[58:56];
   assign aw_qos_s1[3:0]         = aw_data_targ[62:59];
   assign aw_size_s1[2:0]        = aw_data_targ[65:63];

   //----------------------------------------------------------------
   
   // lwsoc2fpga_w_data_init assignment
   assign w_data_init  = {w_strb_m1, 
                          w_last_m1, 
		          w_data_m1};

   // Register Slice instantiation for lwsoc2fpga w channel
   rla_full_reg_slice #(
     .WIDTH(145),
     .PTR_WIDTH(1)
   ) wchan_fifo (
     .clk(w_clk),
     .rst_n(w_rst_n),
     .valid_init(w_valid_m1),
     .ready_init(w_ready_m1),
     .data_init(w_data_init),
     .readylatency_init(1'b0),
     .valid_targ(w_valid_s1),
     .ready_targ(w_ready_s1),
     .data_targ(w_data_targ),
     .readylatency_targ(w_latency_en)
   );

   // lwsoc2fpga_w_data_targ assignment
   assign w_data_s1[127:0]        = w_data_targ[127:0];
   assign w_last_s1              = w_data_targ[128];
   assign w_strb_s1[15:0]         = w_data_targ[144:129];

   //----------------------------------------------------------------
   
   // lwsoc2fpga_r_data_init assignment
   assign r_data_init = { r_resp_s1, 
                          r_last_s1, 
	                  r_id_s1, 
			  r_data_s1 };
   
   // Register Slice instantiation for lwsoc2fpga r channel
   rla_full_reg_slice #(
     .WIDTH(135),
     .PTR_WIDTH(3)
   ) rchan_fifo (
     .clk(r_clk),
     .rst_n(r_rst_n),
     .valid_init(r_valid_s1),
     .ready_init(r_ready_s1),
     .data_init(r_data_init),
     .readylatency_init(r_latency_en),
     .valid_targ(r_valid_m1),
     .ready_targ(r_ready_m1),
     .data_targ(r_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_r_data_targ assignment
   assign r_data_m1[127:0]   = r_data_targ[127:0];
   assign r_id_m1[3:0]      = r_data_targ[131:128];
   assign r_last_m1         = r_data_targ[132];
   assign r_resp_m1[1:0]    = r_data_targ[134:133];

   //----------------------------------------------------------------

   // lwsoc2fpga_b_data_init assignment
   assign b_data_init = { b_resp_s1, 
                          b_id_s1 }; 
   
   // Register Slice instantiation for lwsoc2fpga b channel
   rla_full_reg_slice #(
     .WIDTH(6),
     .PTR_WIDTH(3)
   ) bchan_fifo (
     .clk(b_clk),
     .rst_n(b_rst_n),
     .valid_init(b_valid_s1),
     .ready_init(b_ready_s1),
     .data_init(b_data_init),
     .readylatency_init(b_latency_en),
     .valid_targ(b_valid_m1),
     .ready_targ(b_ready_m1),
     .data_targ(b_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_b_data_targ assignment
   assign b_id_m1[3:0]      = b_data_targ[3:0];
   assign b_resp_m1[1:0]    = b_data_targ[5:4];

   //---------------------------------------------------------------- 
endmodule
//------------------------------------------------------------------------------
//
// Description:
// This is a synthesizable ready latency pipeline of selectable depth, for use with the Clarke HPS LW bridge.


module f2sdram_ready_latency_pipe (
  input           ar_clk,
  input           ar_rst_n,
  input           aw_clk,
  input           aw_rst_n,
  input           b_clk,
  input           b_rst_n,
  input           r_clk,
  input           r_rst_n,
  input           w_clk,
  input           w_rst_n,

  input [2:0]     ar_ready_latency,
  input [2:0]     ar_valid_latency,
  input [2:0]     aw_ready_latency,
  input [2:0]     aw_valid_latency,
  input [2:0]     w_ready_latency,
  input [2:0]     w_valid_latency,
  input [2:0]     r_ready_latency,
  input [2:0]     r_valid_latency,
  input [2:0]     b_ready_latency,
  input [2:0]     b_valid_latency,

  input           ar_ready_s1,
  input    [36:0] ar_addr_m1,
  input     [1:0] ar_burst_m1,
  input     [3:0] ar_cache_m1,
  input     [3:0] ar_id_m1,
  input     [7:0] ar_len_m1,
  input           ar_lock_m1,
  input     [2:0] ar_prot_m1,
  input     [3:0] ar_qos_m1,
  input     [2:0] ar_size_m1,
  input           ar_valid_m1,

  input           aw_ready_s1,
  input    [36:0] aw_addr_m1,
  input     [1:0] aw_burst_m1,
  input     [3:0] aw_cache_m1,
  input     [3:0] aw_id_m1,
  input     [7:0] aw_len_m1,
  input           aw_lock_m1,
  input     [2:0] aw_prot_m1,
  input     [3:0] aw_qos_m1,
  input     [2:0] aw_size_m1,
  input           aw_valid_m1,

  input    [127:0] w_data_m1,
  input           w_last_m1,
  input     [15:0] w_strb_m1,
  input           w_valid_m1,
  input           w_ready_s1,

  input     [3:0] b_id_s1,
  input     [1:0] b_resp_s1,
  input           b_valid_s1,
  input           b_ready_m1,

  input    [127:0] r_data_s1,
  input     [3:0] r_id_s1,
  input           r_last_s1,
  input     [1:0] r_resp_s1,
  input           r_valid_s1,
  input           r_ready_m1,

  output   [36:0] ar_addr_s1,
  output    [1:0] ar_burst_s1,
  output    [3:0] ar_cache_s1,
  output    [3:0] ar_id_s1,
  output    [7:0] ar_len_s1,
  output          ar_lock_s1,
  output    [2:0] ar_prot_s1,
  output    [3:0] ar_qos_s1,
  output    [2:0] ar_size_s1,
  output          ar_valid_s1,
  output          ar_ready_m1,

  output   [36:0] aw_addr_s1,
  output    [1:0] aw_burst_s1,
  output    [3:0] aw_cache_s1,
  output    [3:0] aw_id_s1,
  output    [7:0] aw_len_s1,
  output          aw_lock_s1,
  output    [2:0] aw_prot_s1,
  output    [3:0] aw_qos_s1,
  output    [2:0] aw_size_s1,
  output          aw_valid_s1, 
  output          aw_ready_m1,

  output    [3:0] b_id_m1,
  output    [1:0] b_resp_m1,
  output          b_valid_m1,
  output          b_ready_s1,

  output   [127:0] r_data_m1,
  output    [3:0] r_id_m1,
  output          r_last_m1,
  output    [1:0] r_resp_m1,
  output          r_valid_m1,
  output          r_ready_s1, 

  output   [127:0] w_data_s1,
  output          w_last_s1,
  output    [15:0] w_strb_s1,
  output          w_valid_s1,
  output          w_ready_m1
);

   //-------------------------------------------------------------

   wire [61:0] aw_source_payload;
   wire [61:0] aw_sink_payload;

   wire [61:0] ar_source_payload;
   wire [61:0] ar_sink_payload;

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(2)) bchannel_rla (
      .reset_n(ar_rst_n),
      .clk(ar_clk),
   
      .source_data (b_resp_s1),
      .source_id   (b_id_s1),
      //.source_valid(b_valid_s1 & b_ready_s1),
      .source_valid(b_valid_s1),
   
      .sink_ready(b_ready_m1),
   
      .ready_delay(b_ready_latency),
      .valid_delay(b_valid_latency),
   
      .sink_data (b_resp_m1),
      .sink_id   (b_id_m1),
      .sink_valid(b_valid_m1),
      
      .source_ready(b_ready_s1) 
   );

   //-------------------------------------------------------------
   
   ready_latency_regslice #(.DATA_WIDTH(131)) rchannel_rla (
      .reset_n(aw_rst_n),
      .clk(aw_clk),
   
      .source_data ({r_last_s1, r_resp_s1, r_data_s1}),
      .source_id   (r_id_s1),
      //.source_valid(r_valid_s1 & r_ready_s1),
      .source_valid(r_valid_s1),
   
      .sink_ready(r_ready_m1),
   
      .ready_delay(r_ready_latency),
      .valid_delay(r_valid_latency),
   
      .sink_data ({r_last_m1, r_resp_m1, r_data_m1}),
      .sink_id   (r_id_m1),
      .sink_valid(r_valid_m1),
      
      .source_ready(r_ready_s1) 
   );

   //-------------------------------------------------------------

   assign aw_source_payload = { aw_size_m1, // 3 
                                aw_qos_m1,  // 4
                                aw_prot_m1, // 3
				aw_lock_m1, // 1
				aw_len_m1,  // 8
				aw_cache_m1,// 4
				aw_burst_m1,// 2
				aw_addr_m1  // 37
		              };            // 62 bits total

   ready_latency_regslice #(.DATA_WIDTH(62)) awchannel_rla (
      .reset_n(aw_rst_n),
      .clk(aw_clk),
   
      .source_data (aw_source_payload),
      .source_id   (aw_id_m1),
      .source_valid(aw_valid_m1),
   
      .sink_ready(aw_ready_s1),
   
      .ready_delay(aw_ready_latency),
      .valid_delay(aw_valid_latency),
   
      .sink_data (aw_sink_payload),
      .sink_id   (aw_id_s1),
      .sink_valid(aw_valid_s1),
     
      .source_ready(aw_ready_m1) 
   );

   assign  { aw_size_s1, // 3 
             aw_qos_s1,  // 4
             aw_prot_s1, // 3
             aw_lock_s1, // 1
	     aw_len_s1,  // 8
	     aw_cache_s1,// 4
	     aw_burst_s1,// 2
	     aw_addr_s1  // 37
           }                   = aw_sink_payload;

   //-------------------------------------------------------------

   assign ar_source_payload = { ar_size_m1, // 3 
                                ar_qos_m1,  // 4
                                ar_prot_m1, // 3
				ar_lock_m1, // 1
				ar_len_m1,  // 8
				ar_cache_m1,// 4
				ar_burst_m1,// 2
				ar_addr_m1  // 37
		              };            // 62 bits total

   ready_latency_regslice #(.DATA_WIDTH(62)) archannel_rla (
      .reset_n(ar_rst_n),
      .clk(ar_clk),
   
      .source_data (ar_source_payload),
      .source_id   (ar_id_m1),
      .source_valid(ar_valid_m1),
   
      .sink_ready(ar_ready_s1),
   
      .ready_delay(ar_ready_latency),
      .valid_delay(ar_valid_latency),
   
      .sink_data (ar_sink_payload),
      .sink_id   (ar_id_s1),
      .sink_valid(ar_valid_s1),
      
      .source_ready(ar_ready_m1) 
   );

   assign  { ar_size_s1, // 3 
             ar_qos_s1,  // 4
             ar_prot_s1, // 3
             ar_lock_s1, // 1
	     ar_len_s1,  // 8
	     ar_cache_s1,// 4
	     ar_burst_s1,// 2
	     ar_addr_s1  // 32
           }                   = ar_sink_payload;

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(145)) wchannel_rla (
      .reset_n(w_rst_n),
      .clk(w_clk),
   
      .source_data ({w_last_m1, w_strb_m1, w_data_m1}),
      .source_id   ({4{1'b0}}),
      .source_valid(w_valid_m1),
   
      .sink_ready(w_ready_s1),
   
      .ready_delay(w_ready_latency),
      .valid_delay(w_valid_latency),
   
      .sink_data ({w_last_s1, w_strb_s1, w_data_s1}),
      .sink_id   (),
      .sink_valid(w_valid_s1),
      
      .source_ready(w_ready_m1) 
   );

   //-------------------------------------------------------------
	      
endmodule
//------------------------------------------------------------------------------

//
// Description:
// This is a synthesizable ready-latency adapter.  It comprises a
// ready-latency pipeline of selectable depth and a ready-latency fifo. 
// Instantiate it between the Clarke HPS LWBridge and an AXI-compliant slave.

`define SVT_AXI_MAX_DATA_WIDTH 128
module h2f_ready_latency_adapter (
  input           ar_clk,
  input           ar_rst_n,
  input           aw_clk,
  input           aw_rst_n,
  input           b_clk,
  input           b_rst_n,
  input           r_clk,
  input           r_rst_n,
  input           w_clk,
  input           w_rst_n,

  input [2:0]     ar_ready_latency,
  input [2:0]     ar_valid_latency,
  input [2:0]     aw_ready_latency,
  input [2:0]     aw_valid_latency,
  input [2:0]     w_ready_latency,
  input [2:0]     w_valid_latency,
  input [2:0]     r_ready_latency,
  input [2:0]     r_valid_latency,
  input [2:0]     b_ready_latency,
  input [2:0]     b_valid_latency,

  input           ar_ready_s1,
  input    [31:0] ar_addr_m1,
  input     [1:0] ar_burst_m1,
  input     [3:0] ar_cache_m1,
  input     [3:0] ar_id_m1,
  input     [7:0] ar_len_m1,
  input           ar_lock_m1,
  input     [2:0] ar_prot_m1,
  input     [3:0] ar_qos_m1,
  input     [2:0] ar_size_m1,
  input           ar_valid_m1,

  input           aw_ready_s1,
  input    [31:0] aw_addr_m1,
  input     [1:0] aw_burst_m1,
  input     [3:0] aw_cache_m1,
  input     [3:0] aw_id_m1,
  input     [7:0] aw_len_m1,
  input           aw_lock_m1,
  input     [2:0] aw_prot_m1,
  input     [3:0] aw_qos_m1,
  input     [2:0] aw_size_m1,
  input           aw_valid_m1,

  input  [`SVT_AXI_MAX_DATA_WIDTH-1:0]   w_data_m1,
  input           w_last_m1,
  input     [15:0] w_strb_m1,
  input           w_valid_m1,
  input           w_ready_s1,

  input     [3:0] b_id_s1,
  input     [1:0] b_resp_s1,
  input           b_valid_s1,
  input           b_ready_m1,

  input  [`SVT_AXI_MAX_DATA_WIDTH-1:0] r_data_s1,
  input     [3:0] r_id_s1,
  input           r_last_s1,
  input     [1:0] r_resp_s1,
  input           r_valid_s1,
  input           r_ready_m1,

  output   [31:0] ar_addr_s1,
  output    [1:0] ar_burst_s1,
  output    [3:0] ar_cache_s1,
  output    [3:0] ar_id_s1,
  output    [7:0] ar_len_s1,
  output          ar_lock_s1,
  output    [2:0] ar_prot_s1,
  output    [3:0] ar_qos_s1,
  output    [2:0] ar_size_s1,
  output          ar_valid_s1,
  output          ar_ready_m1,

  output   [31:0] aw_addr_s1,
  output    [1:0] aw_burst_s1,
  output    [3:0] aw_cache_s1,
  output    [3:0] aw_id_s1,
  output    [7:0] aw_len_s1,
  output          aw_lock_s1,
  output    [2:0] aw_prot_s1,
  output    [3:0] aw_qos_s1,
  output    [2:0] aw_size_s1,
  output          aw_valid_s1, 
  output          aw_ready_m1,

  output    [3:0] b_id_m1,
  output    [1:0] b_resp_m1,
  output          b_valid_m1,
  output          b_ready_s1,

  output   [`SVT_AXI_MAX_DATA_WIDTH-1:0]  r_data_m1,
  output    [3:0] r_id_m1,
  output          r_last_m1,
  output    [1:0] r_resp_m1,
  output          r_valid_m1,
  output          r_ready_s1, 

  output   [`SVT_AXI_MAX_DATA_WIDTH-1:0] w_data_s1,
  output          w_last_s1,
  output    [15:0] w_strb_s1,
  output          w_valid_s1,
  output          w_ready_m1
);            

   wire        rlf_rlp_arready_h;
   wire        rlf_rlp_awready_h;
   wire        rlf_rlp_wready_h;
   wire [3:0]  rlf_rlp_bid_h;
   wire [1:0]  rlf_rlp_bresp_h;
   wire        rlf_rlp_bvalid_h;
   wire [`SVT_AXI_MAX_DATA_WIDTH-1:0] rlf_rlp_rdata_h;
   wire [3:0]  rlf_rlp_rid_h;
   wire        rlf_rlp_rlast_h;
   wire [1:0]  rlf_rlp_rresp_h;
   wire        rlf_rlp_rvalid_h;

   wire [31:0] rlp_rlf_araddr_h;
   wire [1:0]  rlp_rlf_arburst_h;
   wire [3:0]  rlp_rlf_arcache_h;
   wire [3:0]  rlp_rlf_arid_h;
   wire [7:0]  rlp_rlf_arlen_h;
   wire        rlp_rlf_arlock_h;
   wire [2:0]  rlp_rlf_arprot_h;
   wire [3:0]  rlp_rlf_arqos_h;
   wire [2:0]  rlp_rlf_arsize_h;
   wire        rlp_rlf_arvalid_h;

   wire [31:0] rlp_rlf_awaddr_h;
   wire [1:0]  rlp_rlf_awburst_h;
   wire [3:0]  rlp_rlf_awcache_h;
   wire [3:0]  rlp_rlf_awid_h;
   wire [7:0]  rlp_rlf_awlen_h;
   wire        rlp_rlf_awlock_h;
   wire [2:0]  rlp_rlf_awprot_h;
   wire [3:0]  rlp_rlf_awqos_h;
   wire [2:0]  rlp_rlf_awsize_h;
   wire        rlp_rlf_awvalid_h;

   wire [127:0] rlp_rlf_wdata_h;
   wire        rlp_rlf_wlast_h;
   wire [15:0]  rlp_rlf_wstrb_h;
   wire        rlp_rlf_wvalid_h;
   
   wire        ar_latency_en;
   wire        aw_latency_en;
   wire         w_latency_en;
   wire         r_latency_en;
   wire         b_latency_en;

   //---------------------------------------------------------------- 

   assign ar_latency_en = |{ar_ready_latency, ar_valid_latency};
   assign aw_latency_en = |{aw_ready_latency, aw_valid_latency};
   assign  w_latency_en = |{ w_ready_latency,  w_valid_latency};
   assign  r_latency_en = |{ r_ready_latency,  r_valid_latency};
   assign  b_latency_en = |{ b_ready_latency,  b_valid_latency};

   //---------------------------------------------------------------- 

   h2f_ready_latency_pipe h2f_rlp (
      .ar_clk  ( ar_clk  ),
      .ar_rst_n( ar_rst_n),
      .aw_clk  ( aw_clk  ),
      .aw_rst_n( aw_rst_n),
      .b_clk   ( b_clk   ),
      .b_rst_n ( b_rst_n ),
      .r_clk   ( r_clk   ),
      .r_rst_n ( r_rst_n ),
      .w_clk   ( w_clk   ),
      .w_rst_n ( w_rst_n ),

      .ar_ready_latency ( ar_ready_latency ),
      .ar_valid_latency ( ar_valid_latency ),
      .aw_ready_latency ( aw_ready_latency ),
      .aw_valid_latency ( aw_valid_latency ),
      .w_ready_latency  ( w_ready_latency  ),
      .w_valid_latency  ( w_valid_latency  ),
      .r_ready_latency  ( r_ready_latency  ),
      .r_valid_latency  ( r_valid_latency  ),
      .b_ready_latency  ( b_ready_latency  ),
      .b_valid_latency  ( b_valid_latency  ),

      .ar_ready_s1( rlf_rlp_arready_h ),
      .ar_addr_m1 ( ar_addr_m1  ),
      .ar_burst_m1( ar_burst_m1 ),
      .ar_cache_m1( ar_cache_m1 ),
      .ar_id_m1   ( ar_id_m1    ),
      .ar_len_m1  ( ar_len_m1   ),
      .ar_lock_m1 ( ar_lock_m1  ),
      .ar_prot_m1 ( ar_prot_m1  ),
      .ar_qos_m1  ( ar_qos_m1   ),
      .ar_size_m1 ( ar_size_m1  ),
      .ar_valid_m1( ar_valid_m1 ),

      .aw_ready_s1( rlf_rlp_awready_h ),
      .aw_addr_m1 ( aw_addr_m1  ),
      .aw_burst_m1( aw_burst_m1 ),
      .aw_cache_m1( aw_cache_m1 ),
      .aw_id_m1   ( aw_id_m1    ),
      .aw_len_m1  ( aw_len_m1   ),
      .aw_lock_m1 ( aw_lock_m1  ),
      .aw_prot_m1 ( aw_prot_m1  ),
      .aw_qos_m1  ( aw_qos_m1   ),
      .aw_size_m1 ( aw_size_m1  ),
      .aw_valid_m1( aw_valid_m1 ),

      .w_data_m1  ( w_data_m1  ),  
      .w_last_m1  ( w_last_m1  ), 
      .w_strb_m1  ( w_strb_m1  ),
      .w_valid_m1 ( w_valid_m1 ),
      .w_ready_s1 ( rlf_rlp_wready_h ),

      .b_id_s1    ( rlf_rlp_bid_h ),
      .b_resp_s1  ( rlf_rlp_bresp_h ),
      .b_valid_s1 ( rlf_rlp_bvalid_h ),
      .b_ready_m1 ( b_ready_m1 ),

      .r_data_s1  ( rlf_rlp_rdata_h ),
      .r_id_s1    ( rlf_rlp_rid_h ),
      .r_last_s1  ( rlf_rlp_rlast_h ),
      .r_resp_s1  ( rlf_rlp_rresp_h ),
      .r_valid_s1 ( rlf_rlp_rvalid_h ),
      .r_ready_m1 ( r_ready_m1 ),

      .ar_addr_s1 ( rlp_rlf_araddr_h  ),
      .ar_burst_s1( rlp_rlf_arburst_h ),
      .ar_cache_s1( rlp_rlf_arcache_h ),
      .ar_id_s1   ( rlp_rlf_arid_h    ),
      .ar_len_s1  ( rlp_rlf_arlen_h   ),
      .ar_lock_s1 ( rlp_rlf_arlock_h  ),
      .ar_prot_s1 ( rlp_rlf_arprot_h  ),
      .ar_qos_s1  ( rlp_rlf_arqos_h   ),
      .ar_size_s1 ( rlp_rlf_arsize_h  ),
      .ar_valid_s1( rlp_rlf_arvalid_h ),
      .ar_ready_m1( ar_ready_m1 ),

      .aw_addr_s1 ( rlp_rlf_awaddr_h  ),
      .aw_burst_s1( rlp_rlf_awburst_h ),
      .aw_cache_s1( rlp_rlf_awcache_h ),
      .aw_id_s1   ( rlp_rlf_awid_h    ),
      .aw_len_s1  ( rlp_rlf_awlen_h   ),
      .aw_lock_s1 ( rlp_rlf_awlock_h  ),
      .aw_prot_s1 ( rlp_rlf_awprot_h  ),
      .aw_qos_s1  ( rlp_rlf_awqos_h   ),
      .aw_size_s1 ( rlp_rlf_awsize_h  ),
      .aw_valid_s1( rlp_rlf_awvalid_h ), 
      .aw_ready_m1( aw_ready_m1 ),

      .b_id_m1    ( b_id_m1    ),
      .b_resp_m1  ( b_resp_m1  ),
      .b_valid_m1 ( b_valid_m1 ),
      .b_ready_s1 ( rlp_rlf_bready_h ),

      .r_data_m1  ( r_data_m1  ),
      .r_id_m1    ( r_id_m1    ),
      .r_last_m1  ( r_last_m1  ),
      .r_resp_m1  ( r_resp_m1  ),
      .r_valid_m1 ( r_valid_m1 ),
      .r_ready_s1 ( rlp_rlf_rready_h ), 

      .w_data_s1  ( rlp_rlf_wdata_h  ),
      .w_last_s1  ( rlp_rlf_wlast_h  ),
      .w_strb_s1  ( rlp_rlf_wstrb_h  ),
      .w_valid_s1 ( rlp_rlf_wvalid_h ),
      .w_ready_m1 ( w_ready_m1 ) 
   );

   //---------------------------------------------------------------- 

   h2f_ready_latency_fifo h2f_rlf (
      .ar_clk  ( ar_clk  ),
      .ar_rst_n( ar_rst_n),
      .aw_clk  ( aw_clk  ),
      .aw_rst_n( aw_rst_n),
      .b_clk   ( b_clk   ),
      .b_rst_n ( b_rst_n ),
      .r_clk   ( r_clk   ),
      .r_rst_n ( r_rst_n ),
      .w_clk   ( w_clk   ),
      .w_rst_n ( w_rst_n ),

      .ar_latency_en ( ar_latency_en  ),
      .aw_latency_en ( aw_latency_en  ),
      .w_latency_en  ( w_latency_en   ),
      .r_latency_en  ( r_latency_en   ),
      .b_latency_en  ( b_latency_en   ),

      .ar_ready_s1( ar_ready_s1 ),
      .ar_addr_m1 ( rlp_rlf_araddr_h  ),
      .ar_burst_m1( rlp_rlf_arburst_h ),
      .ar_cache_m1( rlp_rlf_arcache_h ),
      .ar_id_m1   ( rlp_rlf_arid_h    ),
      .ar_len_m1  ( rlp_rlf_arlen_h   ),
      .ar_lock_m1 ( rlp_rlf_arlock_h  ),
      .ar_prot_m1 ( rlp_rlf_arprot_h  ),
      .ar_qos_m1  ( rlp_rlf_arqos_h   ),
      .ar_size_m1 ( rlp_rlf_arsize_h  ),
      .ar_valid_m1( rlp_rlf_arvalid_h ),

      .aw_ready_s1( aw_ready_s1 ),
      .aw_addr_m1 ( rlp_rlf_awaddr_h  ),
      .aw_burst_m1( rlp_rlf_awburst_h ),
      .aw_cache_m1( rlp_rlf_awcache_h ),
      .aw_id_m1   ( rlp_rlf_awid_h    ),
      .aw_len_m1  ( rlp_rlf_awlen_h   ),
      .aw_lock_m1 ( rlp_rlf_awlock_h  ),
      .aw_prot_m1 ( rlp_rlf_awprot_h  ),
      .aw_qos_m1  ( rlp_rlf_awqos_h   ),
      .aw_size_m1 ( rlp_rlf_awsize_h  ),
      .aw_valid_m1( rlp_rlf_awvalid_h ),

      .w_data_m1  ( rlp_rlf_wdata_h  ),  
      .w_last_m1  ( rlp_rlf_wlast_h  ), 
      .w_strb_m1  ( rlp_rlf_wstrb_h  ),
      .w_valid_m1 ( rlp_rlf_wvalid_h ),
      .w_ready_s1 ( w_ready_s1 ),

      .b_id_s1    ( b_id_s1    ),
      .b_resp_s1  ( b_resp_s1  ),
      .b_valid_s1 ( b_valid_s1 ),
      .b_ready_m1 ( rlp_rlf_bready_h  ),

      .r_data_s1  ( r_data_s1  ),
      .r_id_s1    ( r_id_s1    ),
      .r_last_s1  ( r_last_s1  ),
      .r_resp_s1  ( r_resp_s1  ),
      .r_valid_s1 ( r_valid_s1 ),
      .r_ready_m1 ( rlp_rlf_rready_h ),

      .ar_addr_s1 ( ar_addr_s1  ),
      .ar_burst_s1( ar_burst_s1 ),
      .ar_cache_s1( ar_cache_s1 ),
      .ar_id_s1   ( ar_id_s1    ),
      .ar_len_s1  ( ar_len_s1   ),
      .ar_lock_s1 ( ar_lock_s1  ),
      .ar_prot_s1 ( ar_prot_s1  ),
      .ar_qos_s1  ( ar_qos_s1   ),
      .ar_size_s1 ( ar_size_s1  ),
      .ar_valid_s1( ar_valid_s1 ),
      .ar_ready_m1( rlf_rlp_arready_h ),

      .aw_addr_s1 ( aw_addr_s1  ),
      .aw_burst_s1( aw_burst_s1 ),
      .aw_cache_s1( aw_cache_s1 ),
      .aw_id_s1   ( aw_id_s1    ),
      .aw_len_s1  ( aw_len_s1   ),
      .aw_lock_s1 ( aw_lock_s1  ),
      .aw_prot_s1 ( aw_prot_s1  ),
      .aw_qos_s1  ( aw_qos_s1   ),
      .aw_size_s1 ( aw_size_s1  ),
      .aw_valid_s1( aw_valid_s1 ), 
      .aw_ready_m1( rlf_rlp_awready_h ),

      .b_id_m1    ( rlf_rlp_bid_h    ),
      .b_resp_m1  ( rlf_rlp_bresp_h  ),
      .b_valid_m1 ( rlf_rlp_bvalid_h ),
      .b_ready_s1 ( b_ready_s1 ),

      .r_data_m1  ( rlf_rlp_rdata_h  ),
      .r_id_m1    ( rlf_rlp_rid_h    ),
      .r_last_m1  ( rlf_rlp_rlast_h  ),
      .r_resp_m1  ( rlf_rlp_rresp_h  ),
      .r_valid_m1 ( rlf_rlp_rvalid_h ),
      .r_ready_s1 ( r_ready_s1 ), 

      .w_data_s1  ( w_data_s1   ),
      .w_last_s1  ( w_last_s1   ),
      .w_strb_s1  ( w_strb_s1   ),
      .w_valid_s1 ( w_valid_s1  ),
      .w_ready_m1 ( rlf_rlp_wready_h ) 
   ); 

endmodule
//------------------------------------------------------------------------------

// Description:
// This is a synthesizable FIFO for buffering lwbridge AXI signals from the
// ready-latency pipeline


module h2f_ready_latency_fifo (
   input           ar_clk,
   input           ar_rst_n,
   input           aw_clk,
   input           aw_rst_n,
   input           b_clk,
   input           b_rst_n,
   input           r_clk,
   input           r_rst_n,
   input           w_clk,
   input           w_rst_n,

   input           ar_latency_en,
   input           aw_latency_en,
   input           w_latency_en,
   input           r_latency_en,
   input           b_latency_en,
 
   input           ar_ready_s1,
   input    [31:0] ar_addr_m1,
   input     [1:0] ar_burst_m1,
   input     [3:0] ar_cache_m1,
   input     [3:0] ar_id_m1,
   input     [7:0] ar_len_m1,
   input           ar_lock_m1,
   input     [2:0] ar_prot_m1,
   input     [3:0] ar_qos_m1,
   input     [2:0] ar_size_m1,
   input           ar_valid_m1,
 
   input           aw_ready_s1,
   input    [31:0] aw_addr_m1,
   input     [1:0] aw_burst_m1,
   input     [3:0] aw_cache_m1,
   input     [3:0] aw_id_m1,
   input     [7:0] aw_len_m1,
   input           aw_lock_m1,
   input     [2:0] aw_prot_m1,
   input     [3:0] aw_qos_m1,
   input     [2:0] aw_size_m1,
   input           aw_valid_m1,
 
   input    [127:0] w_data_m1,
   input           w_last_m1,
   input     [15:0] w_strb_m1,
   input           w_valid_m1,
   input           w_ready_s1,
 
   input     [3:0] b_id_s1,
   input     [1:0] b_resp_s1,
   input           b_valid_s1,
   input           b_ready_m1,
 
   input    [127:0] r_data_s1,
   input     [3:0] r_id_s1,
   input           r_last_s1,
   input     [1:0] r_resp_s1,
   input           r_valid_s1,
   input           r_ready_m1,
 
   output   [31:0] ar_addr_s1,
   output    [1:0] ar_burst_s1,
   output    [3:0] ar_cache_s1,
   output    [3:0] ar_id_s1,
   output    [7:0] ar_len_s1,
   output          ar_lock_s1,
   output    [2:0] ar_prot_s1,
   output    [3:0] ar_qos_s1,
   output    [2:0] ar_size_s1,
   output          ar_valid_s1,
   output          ar_ready_m1,
 
   output   [31:0] aw_addr_s1,
   output    [1:0] aw_burst_s1,
   output    [3:0] aw_cache_s1,
   output    [3:0] aw_id_s1,
   output    [7:0] aw_len_s1,
   output          aw_lock_s1,
   output    [2:0] aw_prot_s1,
   output    [3:0] aw_qos_s1,
   output    [2:0] aw_size_s1,
   output          aw_valid_s1, 
   output          aw_ready_m1,
 
   output    [3:0] b_id_m1,
   output    [1:0] b_resp_m1,
   output          b_valid_m1,
   output          b_ready_s1,
 
   output   [127:0] r_data_m1,
   output    [3:0] r_id_m1,
   output          r_last_m1,
   output    [1:0] r_resp_m1,
   output          r_valid_m1,
   output          r_ready_s1, 
 
   output   [127:0] w_data_s1,
   output          w_last_s1,
   output    [15:0] w_strb_s1,
   output          w_valid_s1,
   output          w_ready_m1
);

   //----------------------------------------------------------------

   wire   [60:0] ar_data_init;
   wire   [60:0] ar_data_targ;
   wire   [60:0] aw_data_init;
   wire   [60:0] aw_data_targ;
   wire    [5:0] b_data_init;
   wire    [5:0] b_data_targ;
   wire   [134:0] r_data_init;
   wire   [134:0] r_data_targ;
   wire   [144:0] w_data_init;
   wire   [144:0] w_data_targ;

   //----------------------------------------------------------------

   // lwsoc2fpga_ar_data_init assignment
   assign ar_data_init = {ar_size_m1, 
                          ar_qos_m1, 
		          ar_prot_m1, 
			  ar_lock_m1, 
			  ar_len_m1, 
			  ar_id_m1, 
			  ar_cache_m1, 
			  ar_burst_m1, 
			  ar_addr_m1};

   // Register Slice instantiation for lwsoc2fpga ar channel
   rla_full_reg_slice #(
     .WIDTH(61),
     .PTR_WIDTH(3)
   ) archan_fifo (
     .clk(ar_clk),
     .rst_n(ar_rst_n),
     .valid_init(ar_valid_m1),
     .ready_init(ar_ready_m1),
     .data_init(ar_data_init),
     .readylatency_init(ar_latency_en),
     .valid_targ(ar_valid_s1),
     .ready_targ(ar_ready_s1),
     .data_targ(ar_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_ar_data_targ assignment
   assign ar_addr_s1[31:0]       = ar_data_targ[31:0];
   assign ar_burst_s1[1:0]       = ar_data_targ[33:32];
   assign ar_cache_s1[3:0]       = ar_data_targ[37:34];
   assign ar_id_s1[3:0]          = ar_data_targ[41:38];
   assign ar_len_s1[7:0]         = ar_data_targ[49:42];
   assign ar_lock_s1             = ar_data_targ[50];
   assign ar_prot_s1[2:0]        = ar_data_targ[53:51];
   assign ar_qos_s1[3:0]         = ar_data_targ[57:54];
   assign ar_size_s1[2:0]        = ar_data_targ[60:58];

   //----------------------------------------------------------------

   // lwsoc2fpga_aw_data_init assignment
   assign aw_data_init        = { aw_size_m1, 
                                  aw_qos_m1, 
				  aw_prot_m1, 
				  aw_lock_m1, 
				  aw_len_m1, 
				  aw_id_m1, 
				  aw_cache_m1, 
				  aw_burst_m1, 
				  aw_addr_m1 }; 

   // Register Slice instantiation for lwsoc2fpga aw channel
   rla_full_reg_slice #(
     .WIDTH(61),
     .PTR_WIDTH(3)
   ) awchan_fifo (
     .clk(aw_clk),
     .rst_n(aw_rst_n),
     .valid_init(aw_valid_m1),
     .ready_init(aw_ready_m1),
     .data_init(aw_data_init),
     .readylatency_init(aw_latency_en),
     .valid_targ(aw_valid_s1),
     .ready_targ(aw_ready_s1),
     .data_targ(aw_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_aw_data_targ assignment
   assign aw_addr_s1[31:0]       = aw_data_targ[31:0];
   assign aw_burst_s1[1:0]       = aw_data_targ[33:32];
   assign aw_cache_s1[3:0]       = aw_data_targ[37:34];
   assign aw_id_s1[3:0]          = aw_data_targ[41:38];
   assign aw_len_s1[7:0]         = aw_data_targ[49:42];
   assign aw_lock_s1             = aw_data_targ[50];
   assign aw_prot_s1[2:0]        = aw_data_targ[53:51];
   assign aw_qos_s1[3:0]         = aw_data_targ[57:54];
   assign aw_size_s1[2:0]        = aw_data_targ[60:58];

   //----------------------------------------------------------------
   
   // lwsoc2fpga_w_data_init assignment
   assign w_data_init  = {w_strb_m1, 
                          w_last_m1, 
		          w_data_m1};

   // Register Slice instantiation for lwsoc2fpga w channel
   rla_full_reg_slice #(
     .WIDTH(145),
     .PTR_WIDTH(3)
   ) wchan_fifo (
     .clk(w_clk),
     .rst_n(w_rst_n),
     .valid_init(w_valid_m1),
     .ready_init(w_ready_m1),
     .data_init(w_data_init),
     .readylatency_init(w_latency_en),
     .valid_targ(w_valid_s1),
     .ready_targ(w_ready_s1),
     .data_targ(w_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_w_data_targ assignment
   assign w_data_s1[127:0]        = w_data_targ[127:0];
   assign w_last_s1              = w_data_targ[128];
   assign w_strb_s1[15:0]         = w_data_targ[144:129];

   //----------------------------------------------------------------
   
   // lwsoc2fpga_r_data_init assignment
   assign r_data_init = { r_resp_s1, 
                          r_last_s1, 
	                  r_id_s1, 
			  r_data_s1 };
   
   // Register Slice instantiation for lwsoc2fpga r channel
   rla_full_reg_slice #(
     .WIDTH(135),
     .PTR_WIDTH(1)
   ) rchan_fifo (
     .clk(r_clk),
     .rst_n(r_rst_n),
     .valid_init(r_valid_s1),
     .ready_init(r_ready_s1),
     .data_init(r_data_init),
     .readylatency_init(1'b0),
     .valid_targ(r_valid_m1),
     .ready_targ(r_ready_m1),
     .data_targ(r_data_targ),
     .readylatency_targ(r_latency_en)
   );

   // lwsoc2fpga_r_data_targ assignment
   assign r_data_m1[127:0]   = r_data_targ[127:0];
   assign r_id_m1[3:0]      = r_data_targ[131:128];
   assign r_last_m1         = r_data_targ[132];
   assign r_resp_m1[1:0]    = r_data_targ[134:133];

   //----------------------------------------------------------------

   // lwsoc2fpga_b_data_init assignment
   assign b_data_init = { b_resp_s1, 
                          b_id_s1 }; 
   
   // Register Slice instantiation for lwsoc2fpga b channel
   rla_full_reg_slice #(
     .WIDTH(6),
     .PTR_WIDTH(1)
   ) bchan_fifo (
     .clk(b_clk),
     .rst_n(b_rst_n),
     .valid_init(b_valid_s1),
     .ready_init(b_ready_s1),
     .data_init(b_data_init),
     .readylatency_init(1'b0),
     .valid_targ(b_valid_m1),
     .ready_targ(b_ready_m1),
     .data_targ(b_data_targ),
     .readylatency_targ(b_latency_en)
   );

   // lwsoc2fpga_b_data_targ assignment
   assign b_id_m1[3:0]      = b_data_targ[3:0];
   assign b_resp_m1[1:0]    = b_data_targ[5:4];

   //---------------------------------------------------------------- 
endmodule
//------------------------------------------------------------------------------

//
// Description:
// This is a synthesizable ready latency pipeline of selectable depth, for use with the Clarke HPS LW bridge.



module h2f_ready_latency_pipe (
  input           ar_clk,
  input           ar_rst_n,
  input           aw_clk,
  input           aw_rst_n,
  input           b_clk,
  input           b_rst_n,
  input           r_clk,
  input           r_rst_n,
  input           w_clk,
  input           w_rst_n,

  input [2:0]     ar_ready_latency,
  input [2:0]     ar_valid_latency,
  input [2:0]     aw_ready_latency,
  input [2:0]     aw_valid_latency,
  input [2:0]     w_ready_latency,
  input [2:0]     w_valid_latency,
  input [2:0]     r_ready_latency,
  input [2:0]     r_valid_latency,
  input [2:0]     b_ready_latency,
  input [2:0]     b_valid_latency,

  input           ar_ready_s1,
  input    [31:0] ar_addr_m1,
  input     [1:0] ar_burst_m1,
  input     [3:0] ar_cache_m1,
  input     [3:0] ar_id_m1,
  input     [7:0] ar_len_m1,
  input           ar_lock_m1,
  input     [2:0] ar_prot_m1,
  input     [3:0] ar_qos_m1,
  input     [2:0] ar_size_m1,
  input           ar_valid_m1,

  input           aw_ready_s1,
  input    [31:0] aw_addr_m1,
  input     [1:0] aw_burst_m1,
  input     [3:0] aw_cache_m1,
  input     [3:0] aw_id_m1,
  input     [7:0] aw_len_m1,
  input           aw_lock_m1,
  input     [2:0] aw_prot_m1,
  input     [3:0] aw_qos_m1,
  input     [2:0] aw_size_m1,
  input           aw_valid_m1,

  input    [127:0] w_data_m1,
  input           w_last_m1,
  input     [15:0] w_strb_m1,
  input           w_valid_m1,
  input           w_ready_s1,

  input     [3:0] b_id_s1,
  input     [1:0] b_resp_s1,
  input           b_valid_s1,
  input           b_ready_m1,

  input    [127:0] r_data_s1,
  input     [3:0] r_id_s1,
  input           r_last_s1,
  input     [1:0] r_resp_s1,
  input           r_valid_s1,
  input           r_ready_m1,

  output   [31:0] ar_addr_s1,
  output    [1:0] ar_burst_s1,
  output    [3:0] ar_cache_s1,
  output    [3:0] ar_id_s1,
  output    [7:0] ar_len_s1,
  output          ar_lock_s1,
  output    [2:0] ar_prot_s1,
  output    [3:0] ar_qos_s1,
  output    [2:0] ar_size_s1,
  output          ar_valid_s1,
  output          ar_ready_m1,

  output   [31:0] aw_addr_s1,
  output    [1:0] aw_burst_s1,
  output    [3:0] aw_cache_s1,
  output    [3:0] aw_id_s1,
  output    [7:0] aw_len_s1,
  output          aw_lock_s1,
  output    [2:0] aw_prot_s1,
  output    [3:0] aw_qos_s1,
  output    [2:0] aw_size_s1,
  output          aw_valid_s1, 
  output          aw_ready_m1,

  output    [3:0] b_id_m1,
  output    [1:0] b_resp_m1,
  output          b_valid_m1,
  output          b_ready_s1,

  output   [127:0] r_data_m1,
  output    [3:0] r_id_m1,
  output          r_last_m1,
  output    [1:0] r_resp_m1,
  output          r_valid_m1,
  output          r_ready_s1, 

  output   [127:0] w_data_s1,
  output          w_last_s1,
  output    [15:0] w_strb_s1,
  output          w_valid_s1,
  output          w_ready_m1
);

   //-------------------------------------------------------------

   wire [56:0] aw_source_payload;
   wire [56:0] aw_sink_payload;

   wire [56:0] ar_source_payload;
   wire [56:0] ar_sink_payload;

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(2)) bchannel_rla (
      .reset_n(ar_rst_n),
      .clk(ar_clk),
   
      .source_data (b_resp_s1),
      .source_id   (b_id_s1),
      //.source_valid(b_valid_s1 & b_ready_s1),
      .source_valid(b_valid_s1),
   
      .sink_ready(b_ready_m1),
   
      .ready_delay(b_ready_latency),
      .valid_delay(b_valid_latency),
   
      .sink_data (b_resp_m1),
      .sink_id   (b_id_m1),
      .sink_valid(b_valid_m1),
      
      .source_ready(b_ready_s1) 
   );

   //-------------------------------------------------------------
   
   ready_latency_regslice #(.DATA_WIDTH(131)) rchannel_rla (
      .reset_n(aw_rst_n),
      .clk(aw_clk),
   
      .source_data ({r_last_s1, r_resp_s1, r_data_s1}),
      .source_id   (r_id_s1),
      //.source_valid(r_valid_s1 & r_ready_s1),
      .source_valid(r_valid_s1),
   
      .sink_ready(r_ready_m1),
   
      .ready_delay(r_ready_latency),
      .valid_delay(r_valid_latency),
   
      .sink_data ({r_last_m1, r_resp_m1, r_data_m1}),
      .sink_id   (r_id_m1),
      .sink_valid(r_valid_m1),
      
      .source_ready(r_ready_s1) 
   );

   //-------------------------------------------------------------

   assign aw_source_payload = { aw_size_m1, // 3 
                                aw_qos_m1,  // 4
                                aw_prot_m1, // 3
				aw_lock_m1, // 1
				aw_len_m1,  // 8
				aw_cache_m1,// 4
				aw_burst_m1,// 2
				aw_addr_m1  // 32
		              };            // 57 bits total

   ready_latency_regslice #(.DATA_WIDTH(57)) awchannel_rla (
      .reset_n(aw_rst_n),
      .clk(aw_clk),
   
      .source_data (aw_source_payload),
      .source_id   (aw_id_m1),
      .source_valid(aw_valid_m1),
   
      .sink_ready(aw_ready_s1),
   
      .ready_delay(aw_ready_latency),
      .valid_delay(aw_valid_latency),
   
      .sink_data (aw_sink_payload),
      .sink_id   (aw_id_s1),
      .sink_valid(aw_valid_s1),
     
      .source_ready(aw_ready_m1) 
   );

   assign  { aw_size_s1, // 3 
             aw_qos_s1,  // 4
             aw_prot_s1, // 3
             aw_lock_s1, // 1
	     aw_len_s1,  // 8
	     aw_cache_s1,// 4
	     aw_burst_s1,// 2
	     aw_addr_s1  // 32
           }                   = aw_sink_payload;

   //-------------------------------------------------------------

   assign ar_source_payload = { ar_size_m1, // 3 
                                ar_qos_m1,  // 4
                                ar_prot_m1, // 3
				ar_lock_m1, // 1
				ar_len_m1,  // 8
				ar_cache_m1,// 4
				ar_burst_m1,// 2
				ar_addr_m1  // 32
		              };            // 57 bits total

   ready_latency_regslice #(.DATA_WIDTH(57)) archannel_rla (
      .reset_n(ar_rst_n),
      .clk(ar_clk),
   
      .source_data (ar_source_payload),
      .source_id   (ar_id_m1),
      .source_valid(ar_valid_m1),
   
      .sink_ready(ar_ready_s1),
   
      .ready_delay(ar_ready_latency),
      .valid_delay(ar_valid_latency),
   
      .sink_data (ar_sink_payload),
      .sink_id   (ar_id_s1),
      .sink_valid(ar_valid_s1),
      
      .source_ready(ar_ready_m1) 
   );

   assign  { ar_size_s1, // 3 
             ar_qos_s1,  // 4
             ar_prot_s1, // 3
             ar_lock_s1, // 1
	     ar_len_s1,  // 8
	     ar_cache_s1,// 4
	     ar_burst_s1,// 2
	     ar_addr_s1  // 32
           }                   = ar_sink_payload;

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(145)) wchannel_rla (
      .reset_n(w_rst_n),
      .clk(w_clk),
   
      .source_data ({w_last_m1, w_strb_m1, w_data_m1}),
      .source_id   ({4{1'b0}}),
      .source_valid(w_valid_m1),
   
      .sink_ready(w_ready_s1),
   
      .ready_delay(w_ready_latency),
      .valid_delay(w_valid_latency),
   
      .sink_data ({w_last_s1, w_strb_s1, w_data_s1}),
      .sink_id   (),
      .sink_valid(w_valid_s1),
      
      .source_ready(w_ready_m1) 
   );

   //-------------------------------------------------------------
	      
endmodule
//------------------------------------------------------------------------------
//
// Description:
// This is a synthesizable ready-latency adapter.  It comprises a
// ready-latency pipeline of selectable depth and a ready-latency fifo. 
// Instantiate it between the Clarke HPS LWBridge and an AXI-compliant slave.


module lwbridge_ready_latency_adapter (
  input           ar_clk,
  input           ar_rst_n,
  input           aw_clk,
  input           aw_rst_n,
  input           b_clk,
  input           b_rst_n,
  input           r_clk,
  input           r_rst_n,
  input           w_clk,
  input           w_rst_n,

  input [2:0]     ar_ready_latency,
  input [2:0]     ar_valid_latency,
  input [2:0]     aw_ready_latency,
  input [2:0]     aw_valid_latency,
  input [2:0]     w_ready_latency,
  input [2:0]     w_valid_latency,
  input [2:0]     r_ready_latency,
  input [2:0]     r_valid_latency,
  input [2:0]     b_ready_latency,
  input [2:0]     b_valid_latency,

  input           ar_ready_s1,
  input    [31:0] ar_addr_m1,
  input     [1:0] ar_burst_m1,
  input     [3:0] ar_cache_m1,
  input     [3:0] ar_id_m1,
  input     [7:0] ar_len_m1,
  input           ar_lock_m1,
  input     [2:0] ar_prot_m1,
  input     [3:0] ar_qos_m1,
  input     [2:0] ar_size_m1,
  input           ar_valid_m1,

  input           aw_ready_s1,
  input    [31:0] aw_addr_m1,
  input     [1:0] aw_burst_m1,
  input     [3:0] aw_cache_m1,
  input     [3:0] aw_id_m1,
  input     [7:0] aw_len_m1,
  input           aw_lock_m1,
  input     [2:0] aw_prot_m1,
  input     [3:0] aw_qos_m1,
  input     [2:0] aw_size_m1,
  input           aw_valid_m1,

  input    [31:0] w_data_m1,
  input           w_last_m1,
  input     [3:0] w_strb_m1,
  input           w_valid_m1,
  input           w_ready_s1,

  input     [3:0] b_id_s1,
  input     [1:0] b_resp_s1,
  input           b_valid_s1,
  input           b_ready_m1,

  input    [31:0] r_data_s1,
  input     [3:0] r_id_s1,
  input           r_last_s1,
  input     [1:0] r_resp_s1,
  input           r_valid_s1,
  input           r_ready_m1,

  output   [31:0] ar_addr_s1,
  output    [1:0] ar_burst_s1,
  output    [3:0] ar_cache_s1,
  output    [3:0] ar_id_s1,
  output    [7:0] ar_len_s1,
  output          ar_lock_s1,
  output    [2:0] ar_prot_s1,
  output    [3:0] ar_qos_s1,
  output    [2:0] ar_size_s1,
  output          ar_valid_s1,
  output          ar_ready_m1,

  output   [31:0] aw_addr_s1,
  output    [1:0] aw_burst_s1,
  output    [3:0] aw_cache_s1,
  output    [3:0] aw_id_s1,
  output    [7:0] aw_len_s1,
  output          aw_lock_s1,
  output    [2:0] aw_prot_s1,
  output    [3:0] aw_qos_s1,
  output    [2:0] aw_size_s1,
  output          aw_valid_s1, 
  output          aw_ready_m1,

  output    [3:0] b_id_m1,
  output    [1:0] b_resp_m1,
  output          b_valid_m1,
  output          b_ready_s1,

  output   [31:0] r_data_m1,
  output    [3:0] r_id_m1,
  output          r_last_m1,
  output    [1:0] r_resp_m1,
  output          r_valid_m1,
  output          r_ready_s1, 

  output   [31:0] w_data_s1,
  output          w_last_s1,
  output    [3:0] w_strb_s1,
  output          w_valid_s1,
  output          w_ready_m1
);            

   wire        rlf_rlp_arready_h;
   wire        rlf_rlp_awready_h;
   wire        rlf_rlp_wready_h;
   wire [3:0]  rlf_rlp_bid_h;
   wire [1:0]  rlf_rlp_bresp_h;
   wire        rlf_rlp_bvalid_h;
   wire [31:0] rlf_rlp_rdata_h;
   wire [3:0]  rlf_rlp_rid_h;
   wire        rlf_rlp_rlast_h;
   wire [1:0]  rlf_rlp_rresp_h;
   wire        rlf_rlp_rvalid_h;

   wire [31:0] rlp_rlf_araddr_h;
   wire [1:0]  rlp_rlf_arburst_h;
   wire [3:0]  rlp_rlf_arcache_h;
   wire [3:0]  rlp_rlf_arid_h;
   wire [7:0]  rlp_rlf_arlen_h;
   wire        rlp_rlf_arlock_h;
   wire [2:0]  rlp_rlf_arprot_h;
   wire [3:0]  rlp_rlf_arqos_h;
   wire [2:0]  rlp_rlf_arsize_h;
   wire        rlp_rlf_arvalid_h;

   wire [31:0] rlp_rlf_awaddr_h;
   wire [1:0]  rlp_rlf_awburst_h;
   wire [3:0]  rlp_rlf_awcache_h;
   wire [3:0]  rlp_rlf_awid_h;
   wire [7:0]  rlp_rlf_awlen_h;
   wire        rlp_rlf_awlock_h;
   wire [2:0]  rlp_rlf_awprot_h;
   wire [3:0]  rlp_rlf_awqos_h;
   wire [2:0]  rlp_rlf_awsize_h;
   wire        rlp_rlf_awvalid_h;

   wire [31:0] rlp_rlf_wdata_h;
   wire        rlp_rlf_wlast_h;
   wire [3:0]  rlp_rlf_wstrb_h;
   wire        rlp_rlf_wvalid_h;

   wire        ar_latency_en;
   wire        aw_latency_en;
   wire         w_latency_en;
   wire         r_latency_en;
   wire         b_latency_en;

   //---------------------------------------------------------------- 

   assign ar_latency_en = |{ar_ready_latency, ar_valid_latency};
   assign aw_latency_en = |{aw_ready_latency, aw_valid_latency};
   assign  w_latency_en = |{ w_ready_latency,  w_valid_latency};
   assign  r_latency_en = |{ r_ready_latency,  r_valid_latency};
   assign  b_latency_en = |{ b_ready_latency,  b_valid_latency};

   //---------------------------------------------------------------- 

   lwbridge_ready_latency_pipe lwbridge_rlp (
      .ar_clk  ( ar_clk  ),
      .ar_rst_n( ar_rst_n),
      .aw_clk  ( aw_clk  ),
      .aw_rst_n( aw_rst_n),
      .b_clk   ( b_clk   ),
      .b_rst_n ( b_rst_n ),
      .r_clk   ( r_clk   ),
      .r_rst_n ( r_rst_n ),
      .w_clk   ( w_clk   ),
      .w_rst_n ( w_rst_n ),

      .ar_ready_latency ( ar_ready_latency ),
      .ar_valid_latency ( ar_valid_latency ),
      .aw_ready_latency ( aw_ready_latency ),
      .aw_valid_latency ( aw_valid_latency ),
      .w_ready_latency  ( w_ready_latency  ),
      .w_valid_latency  ( w_valid_latency  ),
      .r_ready_latency  ( r_ready_latency  ),
      .r_valid_latency  ( r_valid_latency  ),
      .b_ready_latency  ( b_ready_latency  ),
      .b_valid_latency  ( b_valid_latency  ),

      .ar_ready_s1( rlf_rlp_arready_h ),
      .ar_addr_m1 ( ar_addr_m1  ),
      .ar_burst_m1( ar_burst_m1 ),
      .ar_cache_m1( ar_cache_m1 ),
      .ar_id_m1   ( ar_id_m1    ),
      .ar_len_m1  ( ar_len_m1   ),
      .ar_lock_m1 ( ar_lock_m1  ),
      .ar_prot_m1 ( ar_prot_m1  ),
      .ar_qos_m1  ( ar_qos_m1   ),
      .ar_size_m1 ( ar_size_m1  ),
      .ar_valid_m1( ar_valid_m1 ),

      .aw_ready_s1( rlf_rlp_awready_h ),
      .aw_addr_m1 ( aw_addr_m1  ),
      .aw_burst_m1( aw_burst_m1 ),
      .aw_cache_m1( aw_cache_m1 ),
      .aw_id_m1   ( aw_id_m1    ),
      .aw_len_m1  ( aw_len_m1   ),
      .aw_lock_m1 ( aw_lock_m1  ),
      .aw_prot_m1 ( aw_prot_m1  ),
      .aw_qos_m1  ( aw_qos_m1   ),
      .aw_size_m1 ( aw_size_m1  ),
      .aw_valid_m1( aw_valid_m1 ),

      .w_data_m1  ( w_data_m1  ),  
      .w_last_m1  ( w_last_m1  ), 
      .w_strb_m1  ( w_strb_m1  ),
      .w_valid_m1 ( w_valid_m1 ),
      .w_ready_s1 ( rlf_rlp_wready_h ),

      .b_id_s1    ( rlf_rlp_bid_h ),
      .b_resp_s1  ( rlf_rlp_bresp_h ),
      .b_valid_s1 ( rlf_rlp_bvalid_h ),
      .b_ready_m1 ( b_ready_m1 ),

      .r_data_s1  ( rlf_rlp_rdata_h ),
      .r_id_s1    ( rlf_rlp_rid_h ),
      .r_last_s1  ( rlf_rlp_rlast_h ),
      .r_resp_s1  ( rlf_rlp_rresp_h ),
      .r_valid_s1 ( rlf_rlp_rvalid_h ),
      .r_ready_m1 ( r_ready_m1 ),

      .ar_addr_s1 ( rlp_rlf_araddr_h  ),
      .ar_burst_s1( rlp_rlf_arburst_h ),
      .ar_cache_s1( rlp_rlf_arcache_h ),
      .ar_id_s1   ( rlp_rlf_arid_h    ),
      .ar_len_s1  ( rlp_rlf_arlen_h   ),
      .ar_lock_s1 ( rlp_rlf_arlock_h  ),
      .ar_prot_s1 ( rlp_rlf_arprot_h  ),
      .ar_qos_s1  ( rlp_rlf_arqos_h   ),
      .ar_size_s1 ( rlp_rlf_arsize_h  ),
      .ar_valid_s1( rlp_rlf_arvalid_h ),
      .ar_ready_m1( ar_ready_m1 ),

      .aw_addr_s1 ( rlp_rlf_awaddr_h  ),
      .aw_burst_s1( rlp_rlf_awburst_h ),
      .aw_cache_s1( rlp_rlf_awcache_h ),
      .aw_id_s1   ( rlp_rlf_awid_h    ),
      .aw_len_s1  ( rlp_rlf_awlen_h   ),
      .aw_lock_s1 ( rlp_rlf_awlock_h  ),
      .aw_prot_s1 ( rlp_rlf_awprot_h  ),
      .aw_qos_s1  ( rlp_rlf_awqos_h   ),
      .aw_size_s1 ( rlp_rlf_awsize_h  ),
      .aw_valid_s1( rlp_rlf_awvalid_h ), 
      .aw_ready_m1( aw_ready_m1 ),

      .b_id_m1    ( b_id_m1    ),
      .b_resp_m1  ( b_resp_m1  ),
      .b_valid_m1 ( b_valid_m1 ),
      .b_ready_s1 ( rlp_rlf_bready_h ),

      .r_data_m1  ( r_data_m1  ),
      .r_id_m1    ( r_id_m1    ),
      .r_last_m1  ( r_last_m1  ),
      .r_resp_m1  ( r_resp_m1  ),
      .r_valid_m1 ( r_valid_m1 ),
      .r_ready_s1 ( rlp_rlf_rready_h ), 

      .w_data_s1  ( rlp_rlf_wdata_h  ),
      .w_last_s1  ( rlp_rlf_wlast_h  ),
      .w_strb_s1  ( rlp_rlf_wstrb_h  ),
      .w_valid_s1 ( rlp_rlf_wvalid_h ),
      .w_ready_m1 ( w_ready_m1 ) 
   );

   //---------------------------------------------------------------- 

   lwbridge_ready_latency_fifo lwbridge_rlf (
      .ar_clk  ( ar_clk  ),
      .ar_rst_n( ar_rst_n),
      .aw_clk  ( aw_clk  ),
      .aw_rst_n( aw_rst_n),
      .b_clk   ( b_clk   ),
      .b_rst_n ( b_rst_n ),
      .r_clk   ( r_clk   ),
      .r_rst_n ( r_rst_n ),
      .w_clk   ( w_clk   ),
      .w_rst_n ( w_rst_n ),

      .ar_latency_en ( ar_latency_en  ),
      .aw_latency_en ( aw_latency_en  ),
      .w_latency_en  ( w_latency_en   ),
      .r_latency_en  ( r_latency_en   ),
      .b_latency_en  ( b_latency_en   ),

      .ar_ready_s1( ar_ready_s1 ),
      .ar_addr_m1 ( rlp_rlf_araddr_h  ),
      .ar_burst_m1( rlp_rlf_arburst_h ),
      .ar_cache_m1( rlp_rlf_arcache_h ),
      .ar_id_m1   ( rlp_rlf_arid_h    ),
      .ar_len_m1  ( rlp_rlf_arlen_h   ),
      .ar_lock_m1 ( rlp_rlf_arlock_h  ),
      .ar_prot_m1 ( rlp_rlf_arprot_h  ),
      .ar_qos_m1  ( rlp_rlf_arqos_h   ),
      .ar_size_m1 ( rlp_rlf_arsize_h  ),
      .ar_valid_m1( rlp_rlf_arvalid_h ),

      .aw_ready_s1( aw_ready_s1 ),
      .aw_addr_m1 ( rlp_rlf_awaddr_h  ),
      .aw_burst_m1( rlp_rlf_awburst_h ),
      .aw_cache_m1( rlp_rlf_awcache_h ),
      .aw_id_m1   ( rlp_rlf_awid_h    ),
      .aw_len_m1  ( rlp_rlf_awlen_h   ),
      .aw_lock_m1 ( rlp_rlf_awlock_h  ),
      .aw_prot_m1 ( rlp_rlf_awprot_h  ),
      .aw_qos_m1  ( rlp_rlf_awqos_h   ),
      .aw_size_m1 ( rlp_rlf_awsize_h  ),
      .aw_valid_m1( rlp_rlf_awvalid_h ),

      .w_data_m1  ( rlp_rlf_wdata_h  ),  
      .w_last_m1  ( rlp_rlf_wlast_h  ), 
      .w_strb_m1  ( rlp_rlf_wstrb_h  ),
      .w_valid_m1 ( rlp_rlf_wvalid_h ),
      .w_ready_s1 ( w_ready_s1 ),

      .b_id_s1    ( b_id_s1    ),
      .b_resp_s1  ( b_resp_s1  ),
      .b_valid_s1 ( b_valid_s1 ),
      .b_ready_m1 ( rlp_rlf_bready_h  ),

      .r_data_s1  ( r_data_s1  ),
      .r_id_s1    ( r_id_s1    ),
      .r_last_s1  ( r_last_s1  ),
      .r_resp_s1  ( r_resp_s1  ),
      .r_valid_s1 ( r_valid_s1 ),
      .r_ready_m1 ( rlp_rlf_rready_h ),

      .ar_addr_s1 ( ar_addr_s1  ),
      .ar_burst_s1( ar_burst_s1 ),
      .ar_cache_s1( ar_cache_s1 ),
      .ar_id_s1   ( ar_id_s1    ),
      .ar_len_s1  ( ar_len_s1   ),
      .ar_lock_s1 ( ar_lock_s1  ),
      .ar_prot_s1 ( ar_prot_s1  ),
      .ar_qos_s1  ( ar_qos_s1   ),
      .ar_size_s1 ( ar_size_s1  ),
      .ar_valid_s1( ar_valid_s1 ),
      .ar_ready_m1( rlf_rlp_arready_h ),

      .aw_addr_s1 ( aw_addr_s1  ),
      .aw_burst_s1( aw_burst_s1 ),
      .aw_cache_s1( aw_cache_s1 ),
      .aw_id_s1   ( aw_id_s1    ),
      .aw_len_s1  ( aw_len_s1   ),
      .aw_lock_s1 ( aw_lock_s1  ),
      .aw_prot_s1 ( aw_prot_s1  ),
      .aw_qos_s1  ( aw_qos_s1   ),
      .aw_size_s1 ( aw_size_s1  ),
      .aw_valid_s1( aw_valid_s1 ), 
      .aw_ready_m1( rlf_rlp_awready_h ),

      .b_id_m1    ( rlf_rlp_bid_h    ),
      .b_resp_m1  ( rlf_rlp_bresp_h  ),
      .b_valid_m1 ( rlf_rlp_bvalid_h ),
      .b_ready_s1 ( b_ready_s1 ),

      .r_data_m1  ( rlf_rlp_rdata_h  ),
      .r_id_m1    ( rlf_rlp_rid_h    ),
      .r_last_m1  ( rlf_rlp_rlast_h  ),
      .r_resp_m1  ( rlf_rlp_rresp_h  ),
      .r_valid_m1 ( rlf_rlp_rvalid_h ),
      .r_ready_s1 ( r_ready_s1 ), 

      .w_data_s1  ( w_data_s1   ),
      .w_last_s1  ( w_last_s1   ),
      .w_strb_s1  ( w_strb_s1   ),
      .w_valid_s1 ( w_valid_s1  ),
      .w_ready_m1 ( rlf_rlp_wready_h ) 
   ); 

endmodule
//------------------------------------------------------------------------------
//
// Description:
// This is a synthesizable FIFO for buffering lwbridge AXI signals from the
// ready-latency pipeline


module lwbridge_ready_latency_fifo (
   input           ar_clk,
   input           ar_rst_n,
   input           aw_clk,
   input           aw_rst_n,
   input           b_clk,
   input           b_rst_n,
   input           r_clk,
   input           r_rst_n,
   input           w_clk,
   input           w_rst_n,
 
   input           ar_latency_en,
   input           aw_latency_en,
   input           w_latency_en,
   input           r_latency_en,
   input           b_latency_en,

   input           ar_ready_s1, 
   input    [31:0] ar_addr_m1,
   input     [1:0] ar_burst_m1,
   input     [3:0] ar_cache_m1,
   input     [3:0] ar_id_m1,
   input     [7:0] ar_len_m1,
   input           ar_lock_m1,
   input     [2:0] ar_prot_m1,
   input     [3:0] ar_qos_m1,
   input     [2:0] ar_size_m1,
   input           ar_valid_m1,
 
   input           aw_ready_s1,
   input    [31:0] aw_addr_m1,
   input     [1:0] aw_burst_m1,
   input     [3:0] aw_cache_m1,
   input     [3:0] aw_id_m1,
   input     [7:0] aw_len_m1,
   input           aw_lock_m1,
   input     [2:0] aw_prot_m1,
   input     [3:0] aw_qos_m1,
   input     [2:0] aw_size_m1,
   input           aw_valid_m1,
 
   input    [31:0] w_data_m1,
   input           w_last_m1,
   input     [3:0] w_strb_m1,
   input           w_valid_m1,
   input           w_ready_s1,
 
   input     [3:0] b_id_s1,
   input     [1:0] b_resp_s1,
   input           b_valid_s1,
   input           b_ready_m1,
 
   input    [31:0] r_data_s1,
   input     [3:0] r_id_s1,
   input           r_last_s1,
   input     [1:0] r_resp_s1,
   input           r_valid_s1,
   input           r_ready_m1,
 
   output   [31:0] ar_addr_s1,
   output    [1:0] ar_burst_s1,
   output    [3:0] ar_cache_s1,
   output    [3:0] ar_id_s1,
   output    [7:0] ar_len_s1,
   output          ar_lock_s1,
   output    [2:0] ar_prot_s1,
   output    [3:0] ar_qos_s1,
   output    [2:0] ar_size_s1,
   output          ar_valid_s1,
   output          ar_ready_m1,
 
   output   [31:0] aw_addr_s1,
   output    [1:0] aw_burst_s1,
   output    [3:0] aw_cache_s1,
   output    [3:0] aw_id_s1,
   output    [7:0] aw_len_s1,
   output          aw_lock_s1,
   output    [2:0] aw_prot_s1,
   output    [3:0] aw_qos_s1,
   output    [2:0] aw_size_s1,
   output          aw_valid_s1, 
   output          aw_ready_m1,
 
   output    [3:0] b_id_m1,
   output    [1:0] b_resp_m1,
   output          b_valid_m1,
   output          b_ready_s1,
 
   output   [31:0] r_data_m1,
   output    [3:0] r_id_m1,
   output          r_last_m1,
   output    [1:0] r_resp_m1,
   output          r_valid_m1,
   output          r_ready_s1, 
 
   output   [31:0] w_data_s1,
   output          w_last_s1,
   output    [3:0] w_strb_s1,
   output          w_valid_s1,
   output          w_ready_m1
);

   //----------------------------------------------------------------

   wire   [60:0] ar_data_init;
   wire   [60:0] ar_data_targ;
   wire   [60:0] aw_data_init;
   wire   [60:0] aw_data_targ;
   wire    [5:0] b_data_init;
   wire    [5:0] b_data_targ;
   wire   [38:0] r_data_init;
   wire   [38:0] r_data_targ;
   wire   [36:0] w_data_init;
   wire   [36:0] w_data_targ;

   //----------------------------------------------------------------

   // lwsoc2fpga_ar_data_init assignment
   assign ar_data_init = {ar_size_m1, 
                          ar_qos_m1, 
		          ar_prot_m1, 
			  ar_lock_m1, 
			  ar_len_m1, 
			  ar_id_m1, 
			  ar_cache_m1, 
			  ar_burst_m1, 
			  ar_addr_m1};

   // Register Slice instantiation for lwsoc2fpga ar channel
   rla_full_reg_slice #(
     .WIDTH(61),
     .PTR_WIDTH(3)
   ) archan_fifo (
     .clk(ar_clk),
     .rst_n(ar_rst_n),
     .valid_init(ar_valid_m1),
     .ready_init(ar_ready_m1),
     .data_init(ar_data_init),
     .readylatency_init(ar_latency_en),
     .valid_targ(ar_valid_s1),
     .ready_targ(ar_ready_s1),
     .data_targ(ar_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_ar_data_targ assignment
   assign ar_addr_s1[31:0]       = ar_data_targ[31:0];
   assign ar_burst_s1[1:0]       = ar_data_targ[33:32];
   assign ar_cache_s1[3:0]       = ar_data_targ[37:34];
   assign ar_id_s1[3:0]          = ar_data_targ[41:38];
   assign ar_len_s1[7:0]         = ar_data_targ[49:42];
   assign ar_lock_s1             = ar_data_targ[50];
   assign ar_prot_s1[2:0]        = ar_data_targ[53:51];
   assign ar_qos_s1[3:0]         = ar_data_targ[57:54];
   assign ar_size_s1[2:0]        = ar_data_targ[60:58];

   //----------------------------------------------------------------

   // lwsoc2fpga_aw_data_init assignment
   assign aw_data_init        = { aw_size_m1, 
                                  aw_qos_m1, 
				  aw_prot_m1, 
				  aw_lock_m1, 
				  aw_len_m1, 
				  aw_id_m1, 
				  aw_cache_m1, 
				  aw_burst_m1, 
				  aw_addr_m1 }; 

   // Register Slice instantiation for lwsoc2fpga aw channel
   rla_full_reg_slice #(
     .WIDTH(61),
     .PTR_WIDTH(3)
   ) awchan_fifo (
     .clk(aw_clk),
     .rst_n(aw_rst_n),
     .valid_init(aw_valid_m1),
     .ready_init(aw_ready_m1),
     .data_init(aw_data_init),
     .readylatency_init(aw_latency_en),
     .valid_targ(aw_valid_s1),
     .ready_targ(aw_ready_s1),
     .data_targ(aw_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_aw_data_targ assignment
   assign aw_addr_s1[31:0]       = aw_data_targ[31:0];
   assign aw_burst_s1[1:0]       = aw_data_targ[33:32];
   assign aw_cache_s1[3:0]       = aw_data_targ[37:34];
   assign aw_id_s1[3:0]          = aw_data_targ[41:38];
   assign aw_len_s1[7:0]         = aw_data_targ[49:42];
   assign aw_lock_s1             = aw_data_targ[50];
   assign aw_prot_s1[2:0]        = aw_data_targ[53:51];
   assign aw_qos_s1[3:0]         = aw_data_targ[57:54];
   assign aw_size_s1[2:0]        = aw_data_targ[60:58];

   //----------------------------------------------------------------
   
   // lwsoc2fpga_w_data_init assignment
   assign w_data_init  = {w_strb_m1, 
                          w_last_m1, 
		          w_data_m1};

   // Register Slice instantiation for lwsoc2fpga w channel
   rla_full_reg_slice #(
     .WIDTH(37),
     .PTR_WIDTH(3)
   ) wchan_fifo (
     .clk(w_clk),
     .rst_n(w_rst_n),
     .valid_init(w_valid_m1),
     .ready_init(w_ready_m1),
     .data_init(w_data_init),
     .readylatency_init(w_latency_en),
     .valid_targ(w_valid_s1),
     .ready_targ(w_ready_s1),
     .data_targ(w_data_targ),
     .readylatency_targ(1'b0)
   );

   // lwsoc2fpga_w_data_targ assignment
   assign w_data_s1[31:0]        = w_data_targ[31:0];
   assign w_last_s1              = w_data_targ[32];
   assign w_strb_s1[3:0]         = w_data_targ[36:33];

   //----------------------------------------------------------------
   
   // lwsoc2fpga_r_data_init assignment
   assign r_data_init = { r_resp_s1, 
                          r_last_s1, 
	                  r_id_s1, 
			  r_data_s1 };
   
   // Register Slice instantiation for lwsoc2fpga r channel
   rla_full_reg_slice #(
     .WIDTH(39),
     .PTR_WIDTH(1)
   ) rchan_fifo (
     .clk(r_clk),
     .rst_n(r_rst_n),
     .valid_init(r_valid_s1),
     .ready_init(r_ready_s1),
     .data_init(r_data_init),
     .readylatency_init(1'b0),
     .valid_targ(r_valid_m1),
     .ready_targ(r_ready_m1),
     .data_targ(r_data_targ),
     .readylatency_targ(r_latency_en)
   );

   // lwsoc2fpga_r_data_targ assignment
   assign r_data_m1[31:0]   = r_data_targ[31:0];
   assign r_id_m1[3:0]      = r_data_targ[35:32];
   assign r_last_m1         = r_data_targ[36];
   assign r_resp_m1[1:0]    = r_data_targ[38:37];

   //----------------------------------------------------------------

   // lwsoc2fpga_b_data_init assignment
   assign b_data_init = { b_resp_s1, 
                          b_id_s1 }; 
   
   // Register Slice instantiation for lwsoc2fpga b channel
   rla_full_reg_slice #(
     .WIDTH(6),
     .PTR_WIDTH(1)
   ) bchan_fifo (
     .clk(b_clk),
     .rst_n(b_rst_n),
     .valid_init(b_valid_s1),
     .ready_init(b_ready_s1),
     .data_init(b_data_init),
     .readylatency_init(1'b0),
     .valid_targ(b_valid_m1),
     .ready_targ(b_ready_m1),
     .data_targ(b_data_targ),
     .readylatency_targ(b_latency_en)
   );

   // lwsoc2fpga_b_data_targ assignment
   assign b_id_m1[3:0]      = b_data_targ[3:0];
   assign b_resp_m1[1:0]    = b_data_targ[5:4];

   //---------------------------------------------------------------- 
endmodule
//------------------------------------------------------------------------------
// Description:
// This is a synthesizable ready latency pipeline of selectable depth, for use with the Clarke HPS LW bridge.


module lwbridge_ready_latency_pipe (
  input           ar_clk,
  input           ar_rst_n,
  input           aw_clk,
  input           aw_rst_n,
  input           b_clk,
  input           b_rst_n,
  input           r_clk,
  input           r_rst_n,
  input           w_clk,
  input           w_rst_n,

  input [2:0]     ar_ready_latency,
  input [2:0]     ar_valid_latency,
  input [2:0]     aw_ready_latency,
  input [2:0]     aw_valid_latency,
  input [2:0]     w_ready_latency,
  input [2:0]     w_valid_latency,
  input [2:0]     r_ready_latency,
  input [2:0]     r_valid_latency,
  input [2:0]     b_ready_latency,
  input [2:0]     b_valid_latency,

  input           ar_ready_s1,
  input    [31:0] ar_addr_m1,
  input     [1:0] ar_burst_m1,
  input     [3:0] ar_cache_m1,
  input     [3:0] ar_id_m1,
  input     [7:0] ar_len_m1,
  input           ar_lock_m1,
  input     [2:0] ar_prot_m1,
  input     [3:0] ar_qos_m1,
  input     [2:0] ar_size_m1,
  input           ar_valid_m1,

  input           aw_ready_s1,
  input    [31:0] aw_addr_m1,
  input     [1:0] aw_burst_m1,
  input     [3:0] aw_cache_m1,
  input     [3:0] aw_id_m1,
  input     [7:0] aw_len_m1,
  input           aw_lock_m1,
  input     [2:0] aw_prot_m1,
  input     [3:0] aw_qos_m1,
  input     [2:0] aw_size_m1,
  input           aw_valid_m1,

  input    [31:0] w_data_m1,
  input           w_last_m1,
  input     [3:0] w_strb_m1,
  input           w_valid_m1,
  input           w_ready_s1,

  input     [3:0] b_id_s1,
  input     [1:0] b_resp_s1,
  input           b_valid_s1,
  input           b_ready_m1,

  input    [31:0] r_data_s1,
  input     [3:0] r_id_s1,
  input           r_last_s1,
  input     [1:0] r_resp_s1,
  input           r_valid_s1,
  input           r_ready_m1,

  output   [31:0] ar_addr_s1,
  output    [1:0] ar_burst_s1,
  output    [3:0] ar_cache_s1,
  output    [3:0] ar_id_s1,
  output    [7:0] ar_len_s1,
  output          ar_lock_s1,
  output    [2:0] ar_prot_s1,
  output    [3:0] ar_qos_s1,
  output    [2:0] ar_size_s1,
  output          ar_valid_s1,
  output          ar_ready_m1,

  output   [31:0] aw_addr_s1,
  output    [1:0] aw_burst_s1,
  output    [3:0] aw_cache_s1,
  output    [3:0] aw_id_s1,
  output    [7:0] aw_len_s1,
  output          aw_lock_s1,
  output    [2:0] aw_prot_s1,
  output    [3:0] aw_qos_s1,
  output    [2:0] aw_size_s1,
  output          aw_valid_s1, 
  output          aw_ready_m1,

  output    [3:0] b_id_m1,
  output    [1:0] b_resp_m1,
  output          b_valid_m1,
  output          b_ready_s1,

  output   [31:0] r_data_m1,
  output    [3:0] r_id_m1,
  output          r_last_m1,
  output    [1:0] r_resp_m1,
  output          r_valid_m1,
  output          r_ready_s1, 

  output   [31:0] w_data_s1,
  output          w_last_s1,
  output    [3:0] w_strb_s1,
  output          w_valid_s1,
  output          w_ready_m1
);

   //-------------------------------------------------------------

   wire [56:0] aw_source_payload;
   wire [56:0] aw_sink_payload;

   wire [56:0] ar_source_payload;
   wire [56:0] ar_sink_payload;

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(2)) bchannel_rla (
      .reset_n(ar_rst_n),
      .clk(ar_clk),
   
      .source_data (b_resp_s1),
      .source_id   (b_id_s1),
      //.source_valid(b_valid_s1 & b_ready_s1),
      .source_valid(b_valid_s1),
   
      .sink_ready(b_ready_m1),
   
      .ready_delay(b_ready_latency),
      .valid_delay(b_valid_latency),
   
      .sink_data (b_resp_m1),
      .sink_id   (b_id_m1),
      .sink_valid(b_valid_m1),
      
      .source_ready(b_ready_s1) 
   );

   //-------------------------------------------------------------
   
   ready_latency_regslice #(.DATA_WIDTH(35)) rchannel_rla (
      .reset_n(aw_rst_n),
      .clk(aw_clk),
   
      .source_data ({r_last_s1, r_resp_s1, r_data_s1}),
      .source_id   (r_id_s1),
      //.source_valid(r_valid_s1 & r_ready_s1),
      .source_valid(r_valid_s1),
   
      .sink_ready(r_ready_m1),
   
      .ready_delay(r_ready_latency),
      .valid_delay(r_valid_latency),
   
      .sink_data ({r_last_m1, r_resp_m1, r_data_m1}),
      .sink_id   (r_id_m1),
      .sink_valid(r_valid_m1),
      
      .source_ready(r_ready_s1) 
   );

   //-------------------------------------------------------------

   assign aw_source_payload = { aw_size_m1, // 3 
                                aw_qos_m1,  // 4
                                aw_prot_m1, // 3
				aw_lock_m1, // 1
				aw_len_m1,  // 8
				aw_cache_m1,// 4
				aw_burst_m1,// 2
				aw_addr_m1  // 32
		              };            // 57 bits total

   ready_latency_regslice #(.DATA_WIDTH(57)) awchannel_rla (
      .reset_n(aw_rst_n),
      .clk(aw_clk),
   
      .source_data (aw_source_payload),
      .source_id   (aw_id_m1),
      .source_valid(aw_valid_m1),
   
      .sink_ready(aw_ready_s1),
   
      .ready_delay(aw_ready_latency),
      .valid_delay(aw_valid_latency),
   
      .sink_data (aw_sink_payload),
      .sink_id   (aw_id_s1),
      .sink_valid(aw_valid_s1),
     
      .source_ready(aw_ready_m1) 
   );

   assign  { aw_size_s1, // 3 
             aw_qos_s1,  // 4
             aw_prot_s1, // 3
             aw_lock_s1, // 1
	     aw_len_s1,  // 8
	     aw_cache_s1,// 4
	     aw_burst_s1,// 2
	     aw_addr_s1  // 32
           }                   = aw_sink_payload;

   //-------------------------------------------------------------

   assign ar_source_payload = { ar_size_m1, // 3 
                                ar_qos_m1,  // 4
                                ar_prot_m1, // 3
				ar_lock_m1, // 1
				ar_len_m1,  // 8
				ar_cache_m1,// 4
				ar_burst_m1,// 2
				ar_addr_m1  // 32
		              };            // 57 bits total

   ready_latency_regslice #(.DATA_WIDTH(57)) archannel_rla (
      .reset_n(ar_rst_n),
      .clk(ar_clk),
   
      .source_data (ar_source_payload),
      .source_id   (ar_id_m1),
      .source_valid(ar_valid_m1),
   
      .sink_ready(ar_ready_s1),
   
      .ready_delay(ar_ready_latency),
      .valid_delay(ar_valid_latency),
   
      .sink_data (ar_sink_payload),
      .sink_id   (ar_id_s1),
      .sink_valid(ar_valid_s1),
      
      .source_ready(ar_ready_m1) 
   );

   assign  { ar_size_s1, // 3 
             ar_qos_s1,  // 4
             ar_prot_s1, // 3
             ar_lock_s1, // 1
	     ar_len_s1,  // 8
	     ar_cache_s1,// 4
	     ar_burst_s1,// 2
	     ar_addr_s1  // 32
           }                   = ar_sink_payload;

   //-------------------------------------------------------------

   ready_latency_regslice #(.DATA_WIDTH(37)) wchannel_rla (
      .reset_n(w_rst_n),
      .clk(w_clk),
   
      .source_data ({w_last_m1, w_strb_m1, w_data_m1}),
      .source_id   ({4{1'b0}}),
      .source_valid(w_valid_m1),
   
      .sink_ready(w_ready_s1),
   
      .ready_delay(w_ready_latency),
      .valid_delay(w_valid_latency),
   
      .sink_data ({w_last_s1, w_strb_s1, w_data_s1}),
      .sink_id   (),
      .sink_valid(w_valid_s1),
      
      .source_ready(w_ready_m1) 
   );

   //-------------------------------------------------------------
	      
endmodule
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "oDDtlXUtB+UysQe7FAoTjxcfb3pKKZZLrS5gWp3sw6TuLTYbGGPlXrXcZ7+F6YBIPhJ5h3QXHIHqEBi+fZrdQvw8PmpPHeXxhJPA5NrJCYxLypp3uyWIvknCrKDV5VajPhu/6EDgrGyy1jEOeS9sSoWBjvWqcpNUXK9nvlOUVIV0xIHfWjY+W8oSdfc6y5vPRe0I38rOdGy+yVCaj9K659/1ZRR/+WygUMQoj4RbOhFWolwDaND0xtnLe7hc5itwys5+I8lQQTgOKJ4NA5E7WFTyxtgeGsmybHj3/jOjRgSwyTwAdXJG1cCfL4x7LvGhbihVrckQtsfCsZm7yfhmZB2imimjLv2bQ7+loHkurrlHHkAVVGUJPv8iZ0aSFpN/fV5rRUo94nKdXYOPobf7JF9z0iGSarUN10fpcVZg3i3t7PCcL6zUixXOqE+PWuAkn+Yoai0JUegsIbr8qr3I+AOFwWwMJ/zla6ZAiWr5dYHu0vMj2XlKfRCO31XA1UsHkN32m4fkS3lOIY0D3y6g/XtysOWNyDtDe/xvgcIKu/vVdxq0neH3wXkQTGgY4cPzryez+9KWXzSJa3koof+Hw6s4mhouC+50Rxb7xF3ED+DEcNgR40sxI3kA+LcLsLPueHhoMHs1lyt1100zNvzWyqohOvz/qqd3AtlxPn7vk2o4R4chs/xcAsQOUeyxVKVqD7p2ySWbeqgSAS4qfcXIJaymarc/QoFq8hsYMRcKpqAZmsjXXhkQO9qVGXXePEZjVfs6+RviGoWYaTIJ9kwcu5JQe+jQEeduVhhFjaD36Fntn223TkS0QV+o8ePkHYH6vbYybkSSlHhkm7WNI0nv2+rsxL2n6rJ8uvUtF8AkCKsRlnyOBAeHJ72C/sSFi1s/Dk5QhoLL8kOrh+j1tR5OO5iFTUNbO6xr7dEhu8M4rRqDvM1ckXIRSCUJ8cdvMb+vyqhNdLBWbXcLJ2+SNu5n34XhjQn0Fz2Ap+0evlC9GQ+hmp1/oTrISQIfSRnCrbh7"
`endif