// (C) 2001-2024 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


// Module: alt_xcvr_resync_std
//
// Description:
//  A general purpose resynchronization module that uses the recommended altera_std_synchronizer 
//  and altera_std_synchronizer_nocut synchronizer
//  
//  Parameters:
//    SYNC_CHAIN_LENGTH
//      - Specifies the length of the synchronizer chain for metastability
//        retiming.
//    WIDTH
//      - Specifies the number of bits you want to synchronize. Controls the width of the
//        d and q ports.
//    SLOW_CLOCK - USE WITH CAUTION. 
//      - Leaving this setting at its default will create a standard resynch circuit that
//        merely passes the input data through a chain of flip-flops. This setting assumes
//        that the input data has a pulse width longer than one clock cycle sufficient to
//        satisfy setup and hold requirements on at least one clock edge.
//      - By setting this to 1 (USE CAUTION) you are creating an asynchronous
//        circuit that will capture the input data regardless of the pulse width and 
//        its relationship to the clock. However it is more difficult to apply static
//        timing constraints as it ties the data input to the clock input of the flop.
//        This implementation assumes the data rate is slow enough
//    INIT_VALUE
//      - Specifies the initial values of the synchronization registers.
//	  NO_CUT
//		- Specifies whether to apply embedded false path timing constraint. 
//		  0: Apply the constraint 1: Not applying the constraint
//

`timescale 1ps/1ps 

module alt_xcvr_resync_std #(
    parameter SYNC_CHAIN_LENGTH = 2,  // Number of flip-flops for retiming. Must be >1
    parameter WIDTH             = 1,  // Number of bits to resync
    parameter SLOW_CLOCK        = 0,  // See description above
    parameter INIT_VALUE        = 0,
    parameter NO_CUT		= 1	  // See description above
  ) (
  input   wire              clk,
  input   wire              reset,
  input   wire  [WIDTH-1:0] d,
  output  wire  [WIDTH-1:0] q
  );

localparam  INT_LEN       = (SYNC_CHAIN_LENGTH > 1) ? SYNC_CHAIN_LENGTH : 2;
localparam  L_INIT_VALUE  = (INIT_VALUE == 1) ? 1'b1 : 1'b0;

genvar ig;

// Generate a synchronizer chain for each bit
generate for(ig=0;ig<WIDTH;ig=ig+1) begin : resync_chains
	wire                d_in;   // Input to sychronization chain.
	wire				sync_d_in;
	wire		        sync_q_out;
	
	// Adding inverter to the input of first sync register and output of the last sync register to implement power-up high for INIT_VALUE=1
	assign sync_d_in = (INIT_VALUE == 1) ? ~d_in : d_in;
	assign q[ig] = (INIT_VALUE == 1)  ? ~sync_q_out : sync_q_out;
	
	if (NO_CUT == 0) begin		
		altera_std_synchronizer #(
			.depth(INT_LEN)				
		) synchronizer (
			.clk		(clk),
			.reset_n	(~reset),
			.din		(sync_d_in),
			.dout		(sync_q_out)
		);
		
		//synthesis translate_off			
		initial begin
			synchronizer.dreg = {(INT_LEN-1){1'b0}};
			synchronizer.din_s1 = 1'b0;
		end
		//synthesis translate_on
				
	end else begin
		altera_std_synchronizer_nocut #(
			.depth(INT_LEN)				
		) synchronizer_nocut (
			.clk		(clk),
			.reset_n	(~reset),
			.din		(sync_d_in),
			.dout		(sync_q_out)
		);
				
		//synthesis translate_off
		initial begin
			synchronizer_nocut.dreg = {(INT_LEN-1){1'b0}};
			synchronizer_nocut.din_s1 = 1'b0;
		end
		//synthesis translate_on	
	end
	
    // Generate asynchronous capture circuit if specified.
    if(SLOW_CLOCK == 0) begin
      assign  d_in = d[ig];
    end else begin
      wire  d_clk;
      reg   d_r = L_INIT_VALUE;
      wire  clr_n;

      assign  d_clk = d[ig];
      assign  d_in  = d_r;
      assign  clr_n = ~q[ig] | d_clk; // Clear when output is logic 1 and input is logic 0

      // Asynchronously latch the input signal.
      always @(posedge d_clk or negedge clr_n)
        if(!clr_n)      d_r <= 1'b0;
        else if(d_clk)  d_r <= 1'b1;
    end // SLOW_CLOCK
  end // for loop
endgenerate

endmodule
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "BIb9V0ft+jvpIcuGRwhKGsoL3AOEopke8JThp4T5Jg2Uy+sc2BLI392h7+4lI6dlKWBxCa/jOYDjFEU1VRns2w+k+iRAlY1oAtWuE5Wne03DTSHFphY5z2Cswro2U3k3Si2j3CMIfnhXcQ+XrTv5SPj14iU87vDEr1aXWhhkFlwCMPdTBSB75EZt5TeyFXLNq5b5aGmTMzild5RgW4o9n0l+n7WydxSXLTpGh+1zTYeuEYE/W1wXNrL5RGyi8VH4vqIp44tKOkyZng35nyDV4xAEQoBHgiuGgBN/DuUyhavS29yjcPfiE2TIIRhB/QoB0htrKhUjiqic1nq51YnNR7s2KyKbWGfE70d9xc7PM3Z1VQ9Eemk4v74kA/FDzwLJRirSyZOj4qegDGiX8GgVFKfuFogsLo5DyeC/ykYBgpSgoKbV03oPLZhE4eJ/PjBUZl94Zw8C1vYktLwlEo6P6hQrGvoLaVYzXTtbGBPfyInPfPp6KpZ3ewvylmmPtMq4FLa58SLROYN5YFKz6xkAg6s9pmsKLDfxZ+siqQ2+9Oo0eGWw6J1oySUeCy6jVzwzEpqjo0/OcAX0TMh0tBJ4raB7sGzBALDMcO3Cda8LM2+X1R7PpVAvFMFmnSqo92HnWxvRZ8RObuFlK18fV0LjNLjuBpvpaxwRte1L22hi1T8iHK3oeYgKnoWSNbJzipKg6Hlaz0nELWPuCS5horJcUGmw7WcUFpU1Yt3wlSXT7nwJ4dQtlkSFIUCfN3QnMkbKsEP0p01ZrANzwbTpx1kMcEaU6wm81rjfxQG1PfigOkqrkpB08btg9VHJNZYAXIPJ/iJyKqR0LhGlTfLwC5NIXi/jOKKQhmReH9UXWZB7umyy+RN4+EF/SFf6brMS8EzbDg7uu8VZzkaPLQ6crbh10Mqy3JcwwIFD0ckoJOfpxRMUcwYhmYf2FSPviA/ZhV2mrVMB/nkx8jqjvbSbskz0LAH0kqhaUKASyO2txwkp2oTd4Jrvi9p9iaBaDXPELp8v"
`endif