// SPDX-License-Identifier: GPL-2.0
/*
 * A V4L2 driver for Sony IMX708 cameras.
 * Copyright (C) 2022, Raspberry Pi Ltd
 *
 * Based on Sony imx477 camera driver
 * Copyright (C) 2020 Raspberry Pi Ltd
 */


// porting from: https://github.com/ArduCAM/IMX519_AK7375

/*
 * IMX5198.cpp
 *
 *  Created on: June 21, 2024
 *      Author: User
 */

#include <stdio.h>
#include <string.h>
#include "IMX519.h"
#include "IMX519_REG.h"

#define ARRAY_SIZE(x) (sizeof(x)/sizeof(x[0]))
#define max(x, y) (x) > (y) ? (x) : (y)

enum{
	MODE_16MP_10FPS = 0, //mode_4656x3496_regs
	MODE_4K_21FPS, // mode_3840x2160_regs
	MODE_2328x1748_30FPS, // mode_2328x1748_regs
	MODE_1080P_60FPS, // mode_1920x1080_regs
	MODE_720P_120FPS // mode_1280x720_regs
};


/* Default PDAF pixel correction gains */
static const alt_u8 pdaf_gains[2][9] = {
	{ 0x4c, 0x4c, 0x4c, 0x46, 0x3e, 0x38, 0x35, 0x35, 0x35 },
	{ 0x35, 0x35, 0x35, 0x38, 0x3e, 0x46, 0x4c, 0x4c, 0x4c }
};

/* Link frequency setup */
enum {
	imx519_LINK_FREQ_450MHZ,
	imx519_LINK_FREQ_447MHZ,
	imx519_LINK_FREQ_453MHZ,
};

static const alt_64 link_freqs[] = {
	[imx519_LINK_FREQ_450MHZ] = 450000000,
	[imx519_LINK_FREQ_447MHZ] = 447000000,
	[imx519_LINK_FREQ_453MHZ] = 453000000,
};

/* 450MHz is the nominal "default" link frequency */
static const struct imx519_reg link_450Mhz_regs[] = {
	{0x030E, 0x01},
	{0x030F, 0x2c},
};

static const struct imx519_reg link_447Mhz_regs[] = {
	{0x030E, 0x01},
	{0x030F, 0x2a},
};

static const struct imx519_reg link_453Mhz_regs[] = {
	{0x030E, 0x01},
	{0x030F, 0x2e},
};

// change order
static const struct imx519_reg_list link_freq_regs[] = {
	[imx519_LINK_FREQ_450MHZ] = {
		//.regs = link_450Mhz_regs,
		.num_of_regs = ARRAY_SIZE(link_450Mhz_regs),
		.regs = link_450Mhz_regs
	},
	[imx519_LINK_FREQ_447MHZ] = {
		//.regs = link_447Mhz_regs,
		.num_of_regs = ARRAY_SIZE(link_447Mhz_regs),
		.regs = link_447Mhz_regs
	},
	[imx519_LINK_FREQ_453MHZ] = {
		//.regs = link_453Mhz_regs,
		.num_of_regs = ARRAY_SIZE(link_453Mhz_regs),
		.regs = link_453Mhz_regs
	}
};

static const struct imx519_reg mode_common_regs[] = {
		{0x0136, 0x18},
		{0x0137, 0x00},
		{0x3c7e, 0x01},
		{0x3c7f, 0x07},
		{0x3020, 0x00},
		{0x3e35, 0x01},
		{0x3f7f, 0x01},
		{0x5609, 0x57},
		{0x5613, 0x51},
		{0x561f, 0x5e},
		{0x5623, 0xd2},
		{0x5637, 0x11},
		{0x5657, 0x11},
		{0x5659, 0x12},
		{0x5733, 0x60},
		{0x5905, 0x57},
		{0x590f, 0x51},
		{0x591b, 0x5e},
		{0x591f, 0xd2},
		{0x5933, 0x11},
		{0x5953, 0x11},
		{0x5955, 0x12},
		{0x5a2f, 0x60},
		{0x5a85, 0x57},
		{0x5a8f, 0x51},
		{0x5a9b, 0x5e},
		{0x5a9f, 0xd2},
		{0x5ab3, 0x11},
		{0x5ad3, 0x11},
		{0x5ad5, 0x12},
		{0x5baf, 0x60},
		{0x5c15, 0x2a},
		{0x5c17, 0x80},
		{0x5c19, 0x31},
		{0x5c1b, 0x87},
		{0x5c25, 0x25},
		{0x5c27, 0x7b},
		{0x5c29, 0x2a},
		{0x5c2b, 0x80},
		{0x5c2d, 0x31},
		{0x5c2f, 0x87},
		{0x5c35, 0x2b},
		{0x5c37, 0x81},
		{0x5c39, 0x31},
		{0x5c3b, 0x87},
		{0x5c45, 0x25},
		{0x5c47, 0x7b},
		{0x5c49, 0x2a},
		{0x5c4b, 0x80},
		{0x5c4d, 0x31},
		{0x5c4f, 0x87},
		{0x5c55, 0x2d},
		{0x5c57, 0x83},
		{0x5c59, 0x32},
		{0x5c5b, 0x88},
		{0x5c65, 0x29},
		{0x5c67, 0x7f},
		{0x5c69, 0x2e},
		{0x5c6b, 0x84},
		{0x5c6d, 0x32},
		{0x5c6f, 0x88},
		{0x5e69, 0x04},
		{0x5e9d, 0x00},
		{0x5f18, 0x10},
		{0x5f1a, 0x0e},
		{0x5f20, 0x12},
		{0x5f22, 0x10},
		{0x5f24, 0x0e},
		{0x5f28, 0x10},
		{0x5f2a, 0x0e},
		{0x5f30, 0x12},
		{0x5f32, 0x10},
		{0x5f34, 0x0e},
		{0x5f38, 0x0f},
		{0x5f39, 0x0d},
		{0x5f3c, 0x11},
		{0x5f3d, 0x0f},
		{0x5f3e, 0x0d},
		{0x5f61, 0x07},
		{0x5f64, 0x05},
		{0x5f67, 0x03},
		{0x5f6a, 0x03},
		{0x5f6d, 0x07},
		{0x5f70, 0x07},
		{0x5f73, 0x05},
		{0x5f76, 0x02},
		{0x5f79, 0x07},
		{0x5f7c, 0x07},
		{0x5f7f, 0x07},
		{0x5f82, 0x07},
		{0x5f85, 0x03},
		{0x5f88, 0x02},
		{0x5f8b, 0x01},
		{0x5f8e, 0x01},
		{0x5f91, 0x04},
		{0x5f94, 0x05},
		{0x5f97, 0x02},
		{0x5f9d, 0x07},
		{0x5fa0, 0x07},
		{0x5fa3, 0x07},
		{0x5fa6, 0x07},
		{0x5fa9, 0x03},
		{0x5fac, 0x01},
		{0x5faf, 0x01},
		{0x5fb5, 0x03},
		{0x5fb8, 0x02},
		{0x5fbb, 0x01},
		{0x5fc1, 0x07},
		{0x5fc4, 0x07},
		{0x5fc7, 0x07},
		{0x5fd1, 0x00},
		{0x6302, 0x79},
		{0x6305, 0x78},
		{0x6306, 0xa5},
		{0x6308, 0x03},
		{0x6309, 0x20},
		{0x630b, 0x0a},
		{0x630d, 0x48},
		{0x630f, 0x06},
		{0x6311, 0xa4},
		{0x6313, 0x03},
		{0x6314, 0x20},
		{0x6316, 0x0a},
		{0x6317, 0x31},
		{0x6318, 0x4a},
		{0x631a, 0x06},
		{0x631b, 0x40},
		{0x631c, 0xa4},
		{0x631e, 0x03},
		{0x631f, 0x20},
		{0x6321, 0x0a},
		{0x6323, 0x4a},
		{0x6328, 0x80},
		{0x6329, 0x01},
		{0x632a, 0x30},
		{0x632b, 0x02},
		{0x632c, 0x20},
		{0x632d, 0x02},
		{0x632e, 0x30},
		{0x6330, 0x60},
		{0x6332, 0x90},
		{0x6333, 0x01},
		{0x6334, 0x30},
		{0x6335, 0x02},
		{0x6336, 0x20},
		{0x6338, 0x80},
		{0x633a, 0xa0},
		{0x633b, 0x01},
		{0x633c, 0x60},
		{0x633d, 0x02},
		{0x633e, 0x60},
		{0x633f, 0x01},
		{0x6340, 0x30},
		{0x6341, 0x02},
		{0x6342, 0x20},
		{0x6343, 0x03},
		{0x6344, 0x80},
		{0x6345, 0x03},
		{0x6346, 0x90},
		{0x6348, 0xf0},
		{0x6349, 0x01},
		{0x634a, 0x20},
		{0x634b, 0x02},
		{0x634c, 0x10},
		{0x634d, 0x03},
		{0x634e, 0x60},
		{0x6350, 0xa0},
		{0x6351, 0x01},
		{0x6352, 0x60},
		{0x6353, 0x02},
		{0x6354, 0x50},
		{0x6355, 0x02},
		{0x6356, 0x60},
		{0x6357, 0x01},
		{0x6358, 0x30},
		{0x6359, 0x02},
		{0x635a, 0x30},
		{0x635b, 0x03},
		{0x635c, 0x90},
		{0x635f, 0x01},
		{0x6360, 0x10},
		{0x6361, 0x01},
		{0x6362, 0x40},
		{0x6363, 0x02},
		{0x6364, 0x50},
		{0x6368, 0x70},
		{0x636a, 0xa0},
		{0x636b, 0x01},
		{0x636c, 0x50},
		{0x637d, 0xe4},
		{0x637e, 0xb4},
		{0x638c, 0x8e},
		{0x638d, 0x38},
		{0x638e, 0xe3},
		{0x638f, 0x4c},
		{0x6390, 0x30},
		{0x6391, 0xc3},
		{0x6392, 0xae},
		{0x6393, 0xba},
		{0x6394, 0xeb},
		{0x6395, 0x6e},
		{0x6396, 0x34},
		{0x6397, 0xe3},
		{0x6398, 0xcf},
		{0x6399, 0x3c},
		{0x639a, 0xf3},
		{0x639b, 0x0c},
		{0x639c, 0x30},
		{0x639d, 0xc1},
		{0x63b9, 0xa3},
		{0x63ba, 0xfe},
		{0x7600, 0x01},
		{0x79a0, 0x01},
		{0x79a1, 0x01},
		{0x79a2, 0x01},
		{0x79a3, 0x01},
		{0x79a4, 0x01},
		{0x79a5, 0x20},
		{0x79a9, 0x00},
		{0x79aa, 0x01},
		{0x79ad, 0x00},
		{0x79af, 0x00},
		{0x8173, 0x01},
		{0x835c, 0x01},
		{0x8a74, 0x01},
		{0x8c1f, 0x00},
		{0x8c27, 0x00},
		{0x8c3b, 0x03},
		{0x9004, 0x0b},
		{0x920c, 0x6a},
		{0x920d, 0x22},
		{0x920e, 0x6a},
		{0x920f, 0x23},
		{0x9214, 0x6a},
		{0x9215, 0x20},
		{0x9216, 0x6a},
		{0x9217, 0x21},
		{0x9385, 0x3e},
		{0x9387, 0x1b},
		{0x938d, 0x4d},
		{0x938f, 0x43},
		{0x9391, 0x1b},
		{0x9395, 0x4d},
		{0x9397, 0x43},
		{0x9399, 0x1b},
		{0x939d, 0x3e},
		{0x939f, 0x2f},
		{0x93a5, 0x43},
		{0x93a7, 0x2f},
		{0x93a9, 0x2f},
		{0x93ad, 0x34},
		{0x93af, 0x2f},
		{0x93b5, 0x3e},
		{0x93b7, 0x2f},
		{0x93bd, 0x4d},
		{0x93bf, 0x43},
		{0x93c1, 0x2f},
		{0x93c5, 0x4d},
		{0x93c7, 0x43},
		{0x93c9, 0x2f},
		{0x974b, 0x02},
		{0x995c, 0x8c},
		{0x995d, 0x00},
		{0x995e, 0x00},
		{0x9963, 0x64},
		{0x9964, 0x50},
		{0xaa0a, 0x26},
		{0xae03, 0x04},
		{0xae04, 0x03},
		{0xae05, 0x03},
		{0xbc1c, 0x08},
		{0xbcf1, 0x02},
};

/* 16 mpix 10fps */
static const struct imx519_reg mode_4656x3496_regs[] = {
	{0x0111, 0x02},
	{0x0112, 0x0a},
	{0x0113, 0x0a},
	{0x0114, 0x01},
	{0x0342, 0x31},
	{0x0343, 0x6a},
	{0x0340, 0x0d},
	{0x0341, 0xf4},
	{0x0344, 0x00},
	{0x0345, 0x00},
	{0x0346, 0x00},
	{0x0347, 0x00},
	{0x0348, 0x12},
	{0x0349, 0x2f},
	{0x034a, 0x0d},
	{0x034b, 0xa7},
	{0x0220, 0x00},
	{0x0221, 0x11},
	{0x0222, 0x01},
	{0x0900, 0x00},
	{0x0901, 0x11},
	{0x0902, 0x0a},
	{0x3f4c, 0x01},
	{0x3f4d, 0x01},
	{0x4254, 0x7f},
	{0x0401, 0x00},
	{0x0404, 0x00},
	{0x0405, 0x10},
	{0x0408, 0x00},
	{0x0409, 0x00},
	{0x040a, 0x00},
	{0x040b, 0x00},
	{0x040c, 0x12},
	{0x040d, 0x30},
	{0x040e, 0x0d},
	{0x040f, 0xa8},
	{0x034c, 0x12},
	{0x034d, 0x30},
	{0x034e, 0x0d},
	{0x034f, 0xa8},
	{0x0301, 0x06},
	{0x0303, 0x04},
	{0x0305, 0x06},
	{0x0306, 0x01},
	{0x0307, 0x40},
	{0x0309, 0x0a},
	{0x030b, 0x02},
	{0x030d, 0x04},
	{0x030e, 0x01},
	{0x030f, 0x10},
	{0x0310, 0x01},
	{0x0820, 0x0a},
	{0x0821, 0x20},
	{0x0822, 0x00},
	{0x0823, 0x00},
	{0x3e20, 0x01},
	{0x3e37, 0x00},
	{0x3e3b, 0x00},
	{0x0106, 0x00},
	{0x0b00, 0x00},
	{0x3230, 0x00},
	{0x3f14, 0x01},
	{0x3f3c, 0x01},
	{0x3f0d, 0x0a},
	{0x3fbc, 0x00},
	{0x3c06, 0x00},
	{0x3c07, 0x48},
	{0x3c0a, 0x00},
	{0x3c0b, 0x00},
	{0x3f78, 0x00},
	{0x3f79, 0x40},
	{0x3f7c, 0x00},
	{0x3f7d, 0x00},
};

/* 4k 21fps mode */
static const struct imx519_reg mode_3840x2160_regs[] = {
	{0x0111, 0x02},
	{0x0112, 0x0a},
	{0x0113, 0x0a},
	{0x0114, 0x01},
	{0x0342, 0x28},
	{0x0343, 0xf6},
	{0x0340, 0x08},
	{0x0341, 0xd4},
	{0x0344, 0x01},
	{0x0345, 0x98},
	{0x0346, 0x02},
	{0x0347, 0xa0},
	{0x0348, 0x10},
	{0x0349, 0x97},
	{0x034a, 0x0b},
	{0x034b, 0x17},
	{0x0220, 0x00},
	{0x0221, 0x11},
	{0x0222, 0x01},
	{0x0900, 0x00},
	{0x0901, 0x11},
	{0x0902, 0x0a},
	{0x3f4c, 0x01},
	{0x3f4d, 0x01},
	{0x4254, 0x7f},
	{0x0401, 0x00},
	{0x0404, 0x00},
	{0x0405, 0x10},
	{0x0408, 0x00},
	{0x0409, 0x00},
	{0x040a, 0x00},
	{0x040b, 0x00},
	{0x040c, 0x0f},
	{0x040d, 0x00},
	{0x040e, 0x08},
	{0x040f, 0x70},
	{0x034c, 0x0f},
	{0x034d, 0x00},
	{0x034e, 0x08},
	{0x034f, 0x70},
	{0x0301, 0x06},
	{0x0303, 0x04},
	{0x0305, 0x06},
	{0x0306, 0x01},
	{0x0307, 0x40},
	{0x0309, 0x0a},
	{0x030b, 0x02},
	{0x030d, 0x04},
	{0x030e, 0x01},
	{0x030f, 0x10},
	{0x0310, 0x01},
	{0x0820, 0x0a},
	{0x0821, 0x20},
	{0x0822, 0x00},
	{0x0823, 0x00},
	{0x3e20, 0x01},
	{0x3e37, 0x00},
	{0x3e3b, 0x00},
	{0x0106, 0x00},
	{0x0b00, 0x00},
	{0x3230, 0x00},
	{0x3f14, 0x01},
	{0x3f3c, 0x01},
	{0x3f0d, 0x0a},
	{0x3fbc, 0x00},
	{0x3c06, 0x00},
	{0x3c07, 0x48},
	{0x3c0a, 0x00},
	{0x3c0b, 0x00},
	{0x3f78, 0x00},
	{0x3f79, 0x40},
	{0x3f7c, 0x00},
	{0x3f7d, 0x00},
};

/* 2x2 binned 30fps mode */
static const struct imx519_reg mode_2328x1748_regs[] = {
	{0x0111, 0x02},
	{0x0112, 0x0a},
	{0x0113, 0x0a},
	{0x0114, 0x01},
	{0x0342, 0x19},
	{0x0343, 0x70},
	{0x0340, 0x08},
	{0x0341, 0x88},
	{0x0344, 0x00},
	{0x0345, 0x00},
	{0x0346, 0x00},
	{0x0347, 0x00},
	{0x0348, 0x12},
	{0x0349, 0x2f},
	{0x034a, 0x0d},
	{0x034b, 0xa7},
	{0x0220, 0x00},
	{0x0221, 0x11},
	{0x0222, 0x01},
	{0x0900, 0x01},
	{0x0901, 0x22},
	{0x0902, 0x0a},
	{0x3f4c, 0x01},
	{0x3f4d, 0x01},
	{0x4254, 0x7f},
	{0x0401, 0x00},
	{0x0404, 0x00},
	{0x0405, 0x10},
	{0x0408, 0x00},
	{0x0409, 0x00},
	{0x040a, 0x00},
	{0x040b, 0x00},
	{0x040c, 0x09},
	{0x040d, 0x18},
	{0x040e, 0x06},
	{0x040f, 0xd4},
	{0x034c, 0x09},
	{0x034d, 0x18},
	{0x034e, 0x06},
	{0x034f, 0xd4},
	{0x0301, 0x06},
	{0x0303, 0x04},
	{0x0305, 0x06},
	{0x0306, 0x01},
	{0x0307, 0x40},
	{0x0309, 0x0a},
	{0x030b, 0x02},
	{0x030d, 0x04},
	{0x030e, 0x01},
	{0x030f, 0x10},
	{0x0310, 0x01},
	{0x0820, 0x0a},
	{0x0821, 0x20},
	{0x0822, 0x00},
	{0x0823, 0x00},
	{0x3e20, 0x01},
	{0x3e37, 0x00},
	{0x3e3b, 0x00},
	{0x0106, 0x00},
	{0x0b00, 0x00},
	{0x3230, 0x00},
	{0x3f14, 0x01},
	{0x3f3c, 0x01},
	{0x3f0d, 0x0a},
	{0x3fbc, 0x00},
	{0x3c06, 0x00},
	{0x3c07, 0x48},
	{0x3c0a, 0x00},
	{0x3c0b, 0x00},
	{0x3f78, 0x00},
	{0x3f79, 0x40},
	{0x3f7c, 0x00},
	{0x3f7d, 0x00},
};

/* 1080p 60fps mode */
static const struct imx519_reg mode_1920x1080_regs[] = {
	{0x0111, 0x02},
	{0x0112, 0x0a},
	{0x0113, 0x0a},
	{0x0114, 0x01},
	{0x0342, 0x17},
	{0x0343, 0x8b},
	{0x0340, 0x04},
	{0x0341, 0x9c},
	{0x0344, 0x01},
	{0x0345, 0x98},
	{0x0346, 0x02},
	{0x0347, 0xa2},
	{0x0348, 0x10},
	{0x0349, 0x97},
	{0x034a, 0x0b},
	{0x034b, 0x15},
	{0x0220, 0x00},
	{0x0221, 0x11},
	{0x0222, 0x01},
	{0x0900, 0x01},
	{0x0901, 0x22},
	{0x0902, 0x0a},
	{0x3f4c, 0x01},
	{0x3f4d, 0x01},
	{0x4254, 0x7f},
	{0x0401, 0x00},
	{0x0404, 0x00},
	{0x0405, 0x10},
	{0x0408, 0x00},
	{0x0409, 0x00},
	{0x040a, 0x00},
	{0x040b, 0x00},
	{0x040c, 0x07},
	{0x040d, 0x80},
	{0x040e, 0x04},
	{0x040f, 0x38},
	{0x034c, 0x07},
	{0x034d, 0x80},
	{0x034e, 0x04},
	{0x034f, 0x38},
	{0x0301, 0x06},
	{0x0303, 0x04},
	{0x0305, 0x06},
	{0x0306, 0x01},
	{0x0307, 0x40},
	{0x0309, 0x0a},
	{0x030b, 0x02},
	{0x030d, 0x04},
	{0x030e, 0x01},
	{0x030f, 0x10},
	{0x0310, 0x01},
	{0x0820, 0x0a},
	{0x0821, 0x20},
	{0x0822, 0x00},
	{0x0823, 0x00},
	{0x3e20, 0x01},
	{0x3e37, 0x00},
	{0x3e3b, 0x00},
	{0x0106, 0x00},
	{0x0b00, 0x00},
	{0x3230, 0x00},
	{0x3f14, 0x01},
	{0x3f3c, 0x01},
	{0x3f0d, 0x0a},
	{0x3fbc, 0x00},
	{0x3c06, 0x00},
	{0x3c07, 0x48},
	{0x3c0a, 0x00},
	{0x3c0b, 0x00},
	{0x3f78, 0x00},
	{0x3f79, 0x40},
	{0x3f7c, 0x00},
	{0x3f7d, 0x00},
};

/* 720p 120fps mode */
static const struct imx519_reg mode_1280x720_regs[] = {
	{0x0111, 0x02},
	{0x0112, 0x0a},
	{0x0113, 0x0a},
	{0x0114, 0x01},
	{0x0342, 0x10},
	{0x0343, 0xf0},
	{0x0340, 0x03},
	{0x0341, 0x34},
	{0x0344, 0x04},
	{0x0345, 0x18},
	{0x0346, 0x04},
	{0x0347, 0x12},
	{0x0348, 0x0e},
	{0x0349, 0x17},
	{0x034a, 0x09},
	{0x034b, 0xb6},
	{0x0220, 0x00},
	{0x0221, 0x11},
	{0x0222, 0x01},
	{0x0900, 0x01},
	{0x0901, 0x22},
	{0x0902, 0x0a},
	{0x3f4c, 0x01},
	{0x3f4d, 0x01},
	{0x4254, 0x7f},
	{0x0401, 0x00},
	{0x0404, 0x00},
	{0x0405, 0x10},
	{0x0408, 0x00},
	{0x0409, 0x00},
	{0x040a, 0x00},
	{0x040b, 0x00},
	{0x040c, 0x05},
	{0x040d, 0x00},
	{0x040e, 0x02},
	{0x040f, 0xd0},
	{0x034c, 0x05},
	{0x034d, 0x00},
	{0x034e, 0x02},
	{0x034f, 0xd0},
	{0x0301, 0x06},
	{0x0303, 0x04},
	{0x0305, 0x06},
	{0x0306, 0x01},
	{0x0307, 0x40},
	{0x0309, 0x0a},
	{0x030b, 0x02},
	{0x030d, 0x04},
	{0x030e, 0x01},
	{0x030f, 0x10},
	{0x0310, 0x01},
	{0x0820, 0x0a},
	{0x0821, 0x20},
	{0x0822, 0x00},
	{0x0823, 0x00},
	{0x3e20, 0x01},
	{0x3e37, 0x00},
	{0x3e3b, 0x00},
	{0x0106, 0x00},
	{0x0b00, 0x00},
	{0x3230, 0x00},
	{0x3f14, 0x01},
	{0x3f3c, 0x01},
	{0x3f0d, 0x0a},
	{0x3fbc, 0x00},
	{0x3c06, 0x00},
	{0x3c07, 0x48},
	{0x3c0a, 0x00},
	{0x3c0b, 0x00},
	{0x3f78, 0x00},
	{0x3f79, 0x40},
	{0x3f7c, 0x00},
	{0x3f7d, 0x00},
};

// richard add
static const struct imx519_reg mode_user_regs[] = {
        //IMX519_REG_EXPOSURE 1147 (0x047B). keep 1080p with frame rate 60
        {0x0202, 0x04},
        {0x0203, 0x7B},

        // Analog gain 960 (0x03C0)
        // #define IMX519_REG_ANALOG_GAIN        0x0204
        {0x0204, 0x03},
        {0x0205, 0xC0},

		// Color Balance, blue 0x0213, red 0x1BC
		{0x3ff9, 0x00}, // 0: by color, 1: all color
		{0x0212, 0x02}, // blue grain
		{0x0213, 0x13},
		{0x0210, 0x01}, // red gain
		{0x0211, 0xBC},
		{0x020e, 0x01}, // restore global gain, IMX519_REG_DIGITAL_GAIN(0x020e) =  IMX519_DGTL_GAIN_DEFAULT(0x0100)
		{0x020f, 0x00},
};

/* Mode configs */
static const struct imx519_mode supported_modes_10bit[] = {
	{
		.width = 4656,
		.height = 3496,
		.line_length_pix = 0x316a,
		.crop = {
			.left = IMX519_PIXEL_ARRAY_LEFT,
			.top = IMX519_PIXEL_ARRAY_TOP,
			.width = 4656,
			.height = 3496,
		},
		.timeperframe_min = {
			.numerator = 100,
			.denominator = 900
		},
		.timeperframe_default = {
			.numerator = 100,
			.denominator = 900
		},
		.reg_list = {
			.num_of_regs = ARRAY_SIZE(mode_4656x3496_regs),
			.regs = mode_4656x3496_regs,
		}
	},
	{
		.width = 3840,
		.height = 2160,
		.line_length_pix = 0x28f6,
		.crop = {
			.left = IMX519_PIXEL_ARRAY_LEFT + 408,
			.top = IMX519_PIXEL_ARRAY_TOP + 672,
			.width = 3840,
			.height = 2160,
		},
		.timeperframe_min = {
			.numerator = 100,
			.denominator = 1800
		},
		.timeperframe_default = {
			.numerator = 100,
			.denominator = 1800
		},
		.reg_list = {
			.num_of_regs = ARRAY_SIZE(mode_3840x2160_regs),
			.regs = mode_3840x2160_regs,
		}
	},
	{
		.width = 2328,
		.height = 1748,
		.line_length_pix = 0x1970,
		.crop = {
			.left = IMX519_PIXEL_ARRAY_LEFT,
			.top = IMX519_PIXEL_ARRAY_TOP,
			.width = 4656,
			.height = 3496,
		},
		.timeperframe_min = {
			.numerator = 100,
			.denominator = 3000
		},
		.timeperframe_default = {
			.numerator = 100,
			.denominator = 3000
		},
		.reg_list = {
			.num_of_regs = ARRAY_SIZE(mode_2328x1748_regs),
			.regs = mode_2328x1748_regs,
		}
	},
	{
		.width = 1920,
		.height = 1080,
		.line_length_pix = 0x178b,
		.crop = {
			.left = IMX519_PIXEL_ARRAY_LEFT + 408,
			.top = IMX519_PIXEL_ARRAY_TOP + 674,
			.width = 3840,
			.height = 2160,
		},
		.timeperframe_min = {
			.numerator = 100,
			.denominator = 6000
		},
		.timeperframe_default = {
			.numerator = 100,
			.denominator = 6000
		},
		.reg_list = {
			.num_of_regs = ARRAY_SIZE(mode_1920x1080_regs),
			.regs = mode_1920x1080_regs,
		}
	},
	{
		.width = 1280,
		.height = 720,
		.line_length_pix = 0x10f0,
		.crop = {
			.left = IMX519_PIXEL_ARRAY_LEFT + 1048,
			.top = IMX519_PIXEL_ARRAY_TOP + 1042,
			.width = 2560,
			.height = 1440,
		},
		.timeperframe_min = {
			.numerator = 100,
			.denominator = 12000
		},
		.timeperframe_default = {
			.numerator = 100,
			.denominator = 12000
		},
		.reg_list = {
			.num_of_regs = ARRAY_SIZE(mode_1280x720_regs),
			.regs = mode_1280x720_regs,
		}
	}
};


/*
 * The supported formats.
 * This table MUST contain 4 entries per format, to cover the various flip
 * combinations in the order
 * - no flip
 * - h flip
 * - v flip
 * - h&v flips
 */



static const char * const imx519_test_pattern_menu[] = {
	"Disabled",
	"Color Bars",
	"Solid Color",
	"Grey Color Bars",
	"PN9"
};

static const int imx519_test_pattern_val[] = {
	IMX519_TEST_PATTERN_DISABLE,
	IMX519_TEST_PATTERN_COLOR_BARS,
	IMX519_TEST_PATTERN_SOLID_COLOR,
	IMX519_TEST_PATTERN_GREY_COLOR,
	IMX519_TEST_PATTERN_PN9,
};


IMX519::IMX519(const char *pDevName, alt_u32 nDeviceAddr){
	// TODO Auto-generated constructor stub
	strcpy(m_szDevName, pDevName);
	m_nDeviceAddr = nDeviceAddr;

	imx519_set_default_format();

}

IMX519::~IMX519() {
	// TODO Auto-generated destructor stub
}


bool IMX519::imx519_identify_module(){

	bool bSuccess;
	alt_u16 value16;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);


	// check CHIP ID
	if (bSuccess)
			bSuccess = imx519_read_reg16(IMX519_REG_CHIP_ID,  &value16);

	if (bSuccess){
		if (value16 != IMX519_CHIP_ID){
			printf("unexpected camera Chip ID 0x%04x(expected %04xh)\n", value16, IMX519_CHIP_ID);
			bSuccess = false;
		}else{
			printf("camera module ID 0x%04x\n", value16);
		}
	}else{
		printf("failed to read chip id\r\n");
	}

	Close();


	return bSuccess;

}

/* Initialize default format */
bool IMX519::imx519_set_default_format(){
#if 1

	// init m_imx708
	memset((void *)&m_imx519, 0x00, sizeof(m_imx519));
	m_imx519.link_freq_idx = imx519_LINK_FREQ_450MHZ ;
	m_imx519.streaming = false;
	m_imx519.common_regs_written = false;
	m_imx519.long_exp_shift = 0;

	/* Set default mode to max resolution */
	//m_imx519.mode = &supported_modes_10bit[0]; //mode_4656x3496_regs
	//m_imx519.mode = &supported_modes_10bit[3]; //mode_1920x1080_regs 1080P
	//m_imx519.mode = &supported_modes_10bit[4]; //mode_1280x720_regs 720P
	m_imx519.mode = &supported_modes_10bit[MODE_1080P_60FPS];
	//unsigned int num_modes;
	//const bool hdr_enable = true;
	//get_mode_table(MEDIA_BUS_FMT_SRGGB10_1X10, &m_imx708.mode,&num_modes,hdr_enable);

	return true;
#else
	struct v4l2_mbus_framefmt *fmt = &imx708->fmt;

	/* Set default mode to max resolution */
	imx708->mode = &supported_modes_10bit_no_hdr[0];

	/* fmt->code not set as it will always be computed based on flips */
	fmt->colorspace = V4L2_COLORSPACE_RAW;
	fmt->ycbcr_enc = V4L2_MAP_YCBCR_ENC_DEFAULT(fmt->colorspace);
	fmt->quantization = V4L2_MAP_QUANTIZATION_DEFAULT(true,
							  fmt->colorspace,
							  fmt->ycbcr_enc);
	fmt->xfer_func = V4L2_MAP_XFER_FUNC_DEFAULT(fmt->colorspace);
	fmt->width = imx708->mode->width;
	fmt->height = imx708->mode->height;
	fmt->field = V4L2_FIELD_NONE;
#endif
}


/* Start streaming */
bool IMX519::imx519_init()
{
	bool bSuccess = false;
	//struct i2c_client *client = v4l2_get_subdevdata(&imx708->sd);
	const struct imx519_reg_list *reg_list, *freq_regs;

	printf("mode:%dx%d\r\n", m_imx519.mode->width, m_imx519.mode->height);

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	if (!m_imx519.common_regs_written) {
		printf("set common settings\r\n");
		bSuccess = imx519_write_regs(mode_common_regs, ARRAY_SIZE(mode_common_regs));
		if (!bSuccess){
			printf("%s failed to set common settings\n",__func__);
			//return bSuccess;
		}else{
			m_imx519.common_regs_written = true; // just run once for common register
		}
	}

	/* Apply default values of current mode */
	if (bSuccess){
		printf("  Apply default values of current mode\r\n");
		reg_list = &(m_imx519.mode->reg_list);
		bSuccess = imx519_write_regs(reg_list->regs, reg_list->num_of_regs);
		if (!bSuccess) {
			printf("%s failed to set mode\n", __func__);
			//return bSuccess;
		}
	}

	/* Update the link frequency registers */
	if (bSuccess){
		printf("  Update the link frequency registers\r\n");
		freq_regs = &link_freq_regs[m_imx519.link_freq_idx];
		bSuccess = imx519_write_regs(freq_regs->regs, freq_regs->num_of_regs);
		if (!bSuccess) {
			printf("%s failed to set link frequency registers\n",__func__);
			//return bSuccess;
		}
	}


	/* Apply customized values from user */
	//ret =  __v4l2_ctrl_handler_setup(imx708->sd.ctrl_handler);
	//if (ret)
	//	return ret;
#if 1
	if (bSuccess){
		printf("  Apply customized values from user\r\n");
		bSuccess = imx519_write_regs(mode_user_regs, ARRAY_SIZE(mode_user_regs));
		if (!bSuccess){
			printf("%s failed to set user settings\n",__func__);
		}

//		bSuccess = imx519_user_customized_setting();

	}
#endif

	/* set stream on register */
	//return imx519_write_reg(imx708, IMX519_REG_MODE_SELECT,
	//			IMX519_REG_VALUE_08BIT, imx519_MODE_STREAMING);
	if (bSuccess){
		printf("  imx519 set stream on register\r\n");
		bSuccess = imx519_write_reg8(IMX519_REG_MODE_SELECT, IMX519_MODE_STREAMING);
		if (!bSuccess) {
			printf("%s failed to set stream on register\n",__func__);
			//return bSuccess;
		}
	}

	Close();

	return bSuccess;
}


/* Start streaming */
bool IMX519::imx519_start_streaming()
{
	bool bSuccess;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	/* set stream off register */
	if (bSuccess)
		bSuccess = imx519_write_reg8(IMX519_REG_MODE_SELECT, IMX519_MODE_STREAMING);


	Close();

	if (!bSuccess)
		printf("%s failed to start stream\n", __func__);



	return bSuccess;
}

/* Stop streaming */
bool IMX519::imx519_stop_streaming()
{
	bool bSuccess;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	/* set stream off register */
	if (bSuccess)
		bSuccess = imx519_write_reg8(IMX519_REG_MODE_SELECT, IMX519_MODE_STANDBY);

	Close();

	if (!bSuccess)
		printf("%s failed to stop stream\n", __func__);


	return bSuccess;
}

bool IMX519::imx519_write_regs(const struct imx519_reg *regs, alt_u32 len){
	bool bSuccess = true;
	unsigned int i;

	for (i = 0; i < len && bSuccess; i++) {

		bSuccess = imx519_write_reg8(regs[i].address, regs[i].val);
		if (!bSuccess) {
			printf("Failed to write reg 0x%4.4x.\n", regs[i].address);
		}
	}
	return bSuccess;

}


bool IMX519::imx519_write_reg8(alt_u16 nRegByteOffset, alt_u8 nValue8){
	bool bSuccess;
	bSuccess = I2C::WriteReg8(nRegByteOffset, nValue8);
	return bSuccess;
}

bool IMX519::imx519_read_reg8(alt_u16 nRegByteOffset, alt_u8 *pnRead8){

	bool bSuccess;
	bSuccess = I2C::ReadReg8(nRegByteOffset, pnRead8);
	return bSuccess;

}

bool IMX519::imx519_write_reg16(alt_u16 nRegByteOffset, alt_u16 nValue16){
	bool bSuccess;
#if 0
	bSuccess = I2C::WriteReg8(nRegByteOffset, (nValue16 >> 8) & 0x00FF); // MSB
	if (bSuccess)
		bSuccess = I2C::WriteReg8(nRegByteOffset+1, nValue16 & 0x00FF); // LSB
#else
	alt_u16 nSwapValue16;

	// Nios V is little-endian, IMX708 16-bit register is big-endian

	nSwapValue16 = (nValue16 >> 8) & 0x00FF;
	nSwapValue16 |= (nValue16 & 0x00FF ) << 8;

	bSuccess = I2C::WriteReg16(nRegByteOffset, nSwapValue16);
#endif
	return bSuccess;
}

bool IMX519::imx519_read_reg16(alt_u16 nRegByteOffset, alt_u16 *pnRead16){
	bool bSuccess;

#if 0
	bSuccess = I2C::ReadReg16(nRegByteOffset, pnRead16);
#else
	alt_u16 nSwapValue16;

	// Nios V is little-endian, IMX708 16-bit register is big-endian

	bSuccess = I2C::ReadReg16(nRegByteOffset, &nSwapValue16);

	if (bSuccess){
		*pnRead16 = (nSwapValue16 >> 8) & 0x00FF;
		*pnRead16 |= (nSwapValue16 & 0x00FF ) << 8;
	}
#endif
	return bSuccess;
}



bool IMX519::imx519_set_exposure(unsigned int val)
{
	/*
	 * In HDR mode this will set the longest exposure. The sensor
	 * will automatically divide the medium and short ones by 4,16.
	 */

	//val = max(val, m_imx519.mode->exposure_lines_min);
	//val -= val % m_imx519.mode->exposure_lines_step;
	bool bSuccess;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX519_REG_EXPOSURE,	val >> m_imx519.long_exp_shift);

	Close();

	if (!bSuccess)
		printf("%s failed to set exposure\n", __func__);

	return bSuccess;
}

bool IMX519::imx519_get_exposure(unsigned int *val)
{
	/*
	 * In HDR mode this will set the longest exposure. The sensor
	 * will automatically divide the medium and short ones by 4,16.
	 */

	//val = max(val, m_imx519.mode->exposure_lines_min);
	//val -= val % m_imx519.mode->exposure_lines_step;
	bool bSuccess;
	alt_u16 value16;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	if (bSuccess){
		bSuccess = imx519_read_reg16(IMX519_REG_EXPOSURE,	&value16);
		if (bSuccess)
			*val = value16 << m_imx519.long_exp_shift;
	}

	Close();

	if (!bSuccess)
		printf("%s failed to set exposure\n", __func__);

	return bSuccess;
}


bool IMX519::imx519_set_analogue_gain(unsigned int val)
{
	/*
	 * In HDR mode this will set the gain for the longest exposure,
	 * and by default the sensor uses the same gain for all of them.
	 */

	bool bSuccess;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX519_REG_ANALOG_GAIN,val);

	Close();

	if (!bSuccess)
		printf("%s failed to set analog gain\n", __func__);


	return bSuccess;


}

bool IMX519::imx519_get_analogue_gain(unsigned int *val)
{
	/*
	 * In HDR mode this will set the gain for the longest exposure,
	 * and by default the sensor uses the same gain for all of them.
	 */

	bool bSuccess;
	alt_u16 value16;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	if (bSuccess){
		bSuccess = imx519_read_reg16(IMX519_REG_ANALOG_GAIN,&value16);
		if (bSuccess)
			*val = value16;
	}


	Close();

	if (!bSuccess)
		printf("%s failed to set analog gain\n", __func__);


	return bSuccess;


}

bool IMX519::imx519_set_digital_gain(unsigned int val)
{
	bool bSuccess;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);


	if (bSuccess)
		bSuccess = imx519_write_reg8(IMX708_REG_DPGA_USE_GLOBAL_GAINB, 0x01); // //0: by color, 1: all color

	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX519_REG_DIGITAL_GAIN,val);

	Close();

	if (!bSuccess)
		printf("%s failed to set digital gain\n", __func__);


	return bSuccess;

}

bool IMX519::imx519_get_digital_gain(unsigned int *val)
{
	bool bSuccess;
	alt_u16 value16;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	if (bSuccess){
		bSuccess = imx519_write_reg16(IMX519_REG_DIGITAL_GAIN,value16);
		if (bSuccess)
			*val = value16;
	}

	Close();

	if (!bSuccess)
		printf("%s failed to set digital gain\n", __func__);


	return bSuccess;

}


bool IMX519::imx519_set_frame_length(unsigned int val)
{
	bool bSuccess;

	m_imx519.long_exp_shift = 0;

	while (val > IMX519_FRAME_LENGTH_MAX) {
		m_imx519.long_exp_shift++;
		val >>= 1;
	}

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX519_REG_FRAME_LENGTH,val);

	if (bSuccess)
		bSuccess = imx519_write_reg8(IMX519_LONG_EXP_SHIFT_REG, m_imx519.long_exp_shift);


	Close();

	if (!bSuccess)
		printf("%s failed to set frame length\n", __func__);

	return bSuccess;
}



bool IMX519::imx519_set_color_balance_BR(unsigned int blue, unsigned int red){
	bool bSuccess;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	if (bSuccess)
		bSuccess = imx519_write_reg8(IMX708_REG_DPGA_USE_GLOBAL_GAINB, 0x00); // //0: by color, 1: all color

	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX519_REG_DIGITAL_GAIN,IMX519_DGTL_GAIN_DEFAULT); // restore global gain to default

	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX708_REG_DIG_GAIN_B, blue);

	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX708_REG_DIG_GAIN_R, red);

	Close();

	if (!bSuccess)
		printf("%s failed\r\n", __func__);

	return bSuccess;
}

bool IMX519::imx519_get_color_balance_BR(unsigned int *blue, unsigned int *red){
	bool bSuccess;
	alt_u16 value16;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	if (bSuccess){
		bSuccess = imx519_read_reg16(IMX708_REG_DIG_GAIN_B, &value16);
		if (bSuccess)
			*blue = value16;
	}
	if (bSuccess){
		bSuccess = imx519_read_reg16(IMX708_REG_DIG_GAIN_R, &value16);
		if (bSuccess)
			*red = value16;
	}

	Close();

	if (!bSuccess)
		printf("%s failed\r\n", __func__);

	return bSuccess;
}



bool IMX519::imx519_set_test_patttern_colour(unsigned int R, unsigned int GR, unsigned int B,unsigned int GB){
	bool bSuccess;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);
	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX519_REG_TEST_PATTERN_R, R);
	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX519_REG_TEST_PATTERN_GR, GR);
	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX519_REG_TEST_PATTERN_B, B);
	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX519_REG_TEST_PATTERN_GB, GB);

	Close();

	if (!bSuccess)
		printf("%s failed\r\n", __func__);


	return bSuccess;
}

bool IMX519::imx519_set_test_patttern(unsigned int nPatternID){
	bool bSuccess;

	bSuccess = Open(m_szDevName, m_nDeviceAddr);
	if (bSuccess)
		bSuccess = imx519_write_reg16(IMX519_REG_TEST_PATTERN, nPatternID);

	Close();

	if (!bSuccess)
		printf("%s failed\r\n", __func__);


	return bSuccess;

}


bool IMX519::imx519_dump_setting(){
	alt_u16 value16;
	int i,num;
	bool bSuccess;


	struct REG_INFO{
		alt_u16 reg_id;
		char szRegName[64];
	};

	struct REG_INFO szRegInfo[] = {
			{IMX519_REG_CHIP_ID, "Chip ID (0x0519 expected)"},
			{IMX519_REG_EXPOSURE, "Exposure"},
			{IMX519_REG_ANALOG_GAIN, "Analog Gain"},
			{IMX519_REG_DIGITAL_GAIN, "Digital Gain"},
			{IMX708_REG_DIG_GAIN_B, "Color balance: Blue"},
			{IMX708_REG_DIG_GAIN_R, "Color balance: Red"},
			{IMX708_REG_DIG_GAIN_GB, "Color balance: Green/Blue"},
			{IMX708_REG_DIG_GAIN_GR, "Color balance: Green/Red"},


	};

	bSuccess = Open(m_szDevName, m_nDeviceAddr);

	num = sizeof(szRegInfo)/sizeof(szRegInfo[0]);
	for(i=0;i<num && bSuccess;i++){
		bSuccess = imx519_read_reg16(szRegInfo[i].reg_id,  &value16);
		if (bSuccess)
			printf("%s = %d(%04xh)\r\n", szRegInfo[i].szRegName, value16, value16);
	}

	Close();

	if (!bSuccess)
		printf("%s failed to dump register content\n", __func__);

	return bSuccess;


}


bool IMX519::imx519_user_customized_setting(){
	bool bSuccess = true;


	if (bSuccess)
		bSuccess = imx519_set_exposure(10000);

#if 0
	if (bSuccess)
		bSuccess = imx519_set_color_balance_BR(0x0213, 0x01BC);
#endif


	return bSuccess;
}

