#include "lis2dw12_api.h"
#include "lis2dw12_regs.h"


int32_t lis2dw12_read_reg(lis2dw12_dev_t *pdev, uint8_t reg, uint8_t *data, uint16_t len){
    int32_t ret;

    if (pdev == NULL){
        return -1;
    }
    ret = pdev->read_reg(pdev->handle, reg, data, len);
    return ret;
}
  
int32_t lis2dw12_write_reg(lis2dw12_dev_t *pdev, uint8_t reg, uint8_t *data, uint16_t len){
    int32_t ret;

    if (pdev == NULL){
        return -1;
    }
    ret = pdev->write_reg(pdev->handle, reg, data, len);
    return ret;
}
  
//These functions convert raw-data into engineering units.
float_t lis2dw12_from_fs2_to_mg(int16_t lsb){
    return ((float_t)lsb) * 0.061f;
}

float_t lis2dw12_from_fs4_to_mg(int16_t lsb){
    return ((float_t)lsb) * 0.122f;
}

float_t lis2dw12_from_fs8_to_mg(int16_t lsb){
    return ((float_t)lsb) * 0.244f;
}

float_t lis2dw12_from_fs16_to_mg(int16_t lsb){
    return ((float_t)lsb) * 0.488f;
}

float_t lis2dw12_from_fs2_lp1_to_mg(int16_t lsb){
    return ((float_t)lsb) * 0.061f;
}

float_t lis2dw12_from_fs4_lp1_to_mg(int16_t lsb){
    return ((float_t)lsb) * 0.122f;
}

float_t lis2dw12_from_fs8_lp1_to_mg(int16_t lsb){
    return ((float_t)lsb) * 0.244f;
}

float_t lis2dw12_from_fs16_lp1_to_mg(int16_t lsb){
    return ((float_t)lsb) * 0.488f;
}

float_t lis2dw12_from_lsb_to_celsius(int16_t lsb){
    return (((float_t)lsb / 256.0f) + 25.0f);
}
  
//This section groups all the functions concerning data generation

//Select accelerometer operating modes.
int32_t lis2dw12_power_mode_set(lis2dw12_dev_t *pdev, lis2dw12_mode_t val){
    lis2dw12_ctrl1_t ctrl1;
    lis2dw12_ctrl6_t ctrl6;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL1, (uint8_t *) &ctrl1, 1);
    if (ret == 0){
        ctrl1.mode = ((uint8_t) val & 0x0CU) >> 2;
        ctrl1.lp_mode = (uint8_t) val & 0x03U ;
        ret = lis2dw12_write_reg(pdev, REG_CTRL1, (uint8_t *) &ctrl1, 1);
    }
  
    if (ret == 0){
        ret = lis2dw12_read_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    }
  
    if (ret == 0){
        ctrl6.low_noise = ((uint8_t) val & 0x10U) >> 4;
        ret = lis2dw12_write_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_power_mode_get(lis2dw12_dev_t *pdev, lis2dw12_mode_t *val){
    lis2dw12_ctrl1_t ctrl1;
    lis2dw12_ctrl6_t ctrl6;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL1, (uint8_t *) &ctrl1, 1);
    if (ret == 0){
        ret = lis2dw12_read_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
        switch (((ctrl6.low_noise << 4) + (ctrl1.mode << 2) + ctrl1.lp_mode)){
            case LIS2DW12_CONTINUE_HIGH_PERFORMANCE:
                *val = LIS2DW12_CONTINUE_HIGH_PERFORMANCE;
                break;
    
            case LIS2DW12_CONTINUE_LOW_PWR_4:
                *val = LIS2DW12_CONTINUE_LOW_PWR_4;
                break;
    
            case LIS2DW12_CONTINUE_LOW_PWR_3:
                *val = LIS2DW12_CONTINUE_LOW_PWR_3;
                break;
    
            case LIS2DW12_CONTINUE_LOW_PWR_2:
                *val = LIS2DW12_CONTINUE_LOW_PWR_2;
                break;
    
            case LIS2DW12_CONTINUE_LOW_PWR_12bit:
                *val = LIS2DW12_CONTINUE_LOW_PWR_12bit;
                break;
    
            case LIS2DW12_SINGLE_LOW_PWR_4:
                *val = LIS2DW12_SINGLE_LOW_PWR_4;
                break;
    
            case LIS2DW12_SINGLE_LOW_PWR_3:
                *val = LIS2DW12_SINGLE_LOW_PWR_3;
                break;
    
            case LIS2DW12_SINGLE_LOW_PWR_2:
                *val = LIS2DW12_SINGLE_LOW_PWR_2;
                break;
    
            case LIS2DW12_SINGLE_LOW_PWR_12bit:
                *val = LIS2DW12_SINGLE_LOW_PWR_12bit;
                break;
    
            case LIS2DW12_HIGH_PERFORMANCE_LOW_NOISE:
                *val = LIS2DW12_HIGH_PERFORMANCE_LOW_NOISE;
                break;
    
            case LIS2DW12_CONTINUE_LOW_PWR_LOW_NOISE_4:
                *val = LIS2DW12_CONTINUE_LOW_PWR_LOW_NOISE_4;
                break;
    
            case LIS2DW12_CONTINUE_LOW_PWR_LOW_NOISE_3:
                *val = LIS2DW12_CONTINUE_LOW_PWR_LOW_NOISE_3;
                break;
    
            case LIS2DW12_CONTINUE_LOW_PWR_LOW_NOISE_2:
                *val = LIS2DW12_CONTINUE_LOW_PWR_LOW_NOISE_2;
                break;
    
            case LIS2DW12_CONTINUE_LOW_PWR_LOW_NOISE_12bit:
                *val = LIS2DW12_CONTINUE_LOW_PWR_LOW_NOISE_12bit;
                break;
    
            case LIS2DW12_SINGLE_LOW_PWR_LOW_NOISE_4:
                *val = LIS2DW12_SINGLE_LOW_PWR_LOW_NOISE_4;
                break;
    
            case LIS2DW12_SINGLE_LOW_PWR_LOW_NOISE_3:
                *val = LIS2DW12_SINGLE_LOW_PWR_LOW_NOISE_3;
                break;
    
            case LIS2DW12_SINGLE_LOW_PWR_LOW_NOISE_2:
                *val = LIS2DW12_SINGLE_LOW_PWR_LOW_NOISE_2;
                break;
    
            case LIS2DW12_SINGLE_LOW_LOW_NOISE_PWR_12bit:
                *val = LIS2DW12_SINGLE_LOW_LOW_NOISE_PWR_12bit;
                break;
    
            default:
                *val = LIS2DW12_CONTINUE_HIGH_PERFORMANCE;
                break;
        }
    }
    return ret;
}
  
//Accelerometer data rate selection.
int32_t lis2dw12_data_rate_set(lis2dw12_dev_t *pdev, lis2dw12_slp_and_odr_t val){
    lis2dw12_ctrl1_t ctrl1;
    lis2dw12_ctrl3_t ctrl3;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL1, (uint8_t *) &ctrl1, 1);
    if (ret == 0){
        ctrl1.odr = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL1, (uint8_t *) &ctrl1, 1);
    }
  
    if (ret == 0){
        ret = lis2dw12_read_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    }
  
    if (ret == 0){
        ctrl3.slp_mode = ((uint8_t) val & 0x30U) >> 4;
        ret = lis2dw12_write_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_data_rate_get(lis2dw12_dev_t *pdev, lis2dw12_slp_and_odr_t *val){
    lis2dw12_ctrl1_t ctrl1;
    lis2dw12_ctrl3_t ctrl3;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL1, (uint8_t *) &ctrl1, 1);
    if (ret == 0){
        ret = lis2dw12_read_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
        switch ((ctrl3.slp_mode << 4) + ctrl1.odr){
            case LIS2DW12_XL_ODR_OFF:
                *val = LIS2DW12_XL_ODR_OFF;
                break;
    
            case LIS2DW12_XL_ODR_1p6Hz_LP_ONLY:
                *val = LIS2DW12_XL_ODR_1p6Hz_LP_ONLY;
                break;
    
            case LIS2DW12_XL_ODR_12p5Hz:
                *val = LIS2DW12_XL_ODR_12p5Hz;
                break;
    
            case LIS2DW12_XL_ODR_25Hz:
                *val = LIS2DW12_XL_ODR_25Hz;
                break;
    
            case LIS2DW12_XL_ODR_50Hz:
                *val = LIS2DW12_XL_ODR_50Hz;
                break;
    
            case LIS2DW12_XL_ODR_100Hz:
                *val = LIS2DW12_XL_ODR_100Hz;
                break;
    
            case LIS2DW12_XL_ODR_200Hz:
                *val = LIS2DW12_XL_ODR_200Hz;
                break;
    
            case LIS2DW12_XL_ODR_400Hz:
                *val = LIS2DW12_XL_ODR_400Hz;
                break;
    
            case LIS2DW12_XL_ODR_800Hz:
                *val = LIS2DW12_XL_ODR_800Hz;
                break;
    
            case LIS2DW12_XL_ODR_1p6KHz:
                *val = LIS2DW12_XL_ODR_1p6KHz;
                break;
    
            case LIS2DW12_XL_SET_SW_TRIG:
                *val = LIS2DW12_XL_SET_SW_TRIG;
                break;
    
            case LIS2DW12_XL_SET_PIN_TRIG:
                *val = LIS2DW12_XL_SET_PIN_TRIG;
                break;
    
            default:
                *val = LIS2DW12_XL_ODR_OFF;
                break;
        }
    }
    return ret;
}
  
// Block data update.
int32_t lis2dw12_block_data_update_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    if (ret == 0){
        ctrl2.bdu = val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_block_data_update_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    *val = ctrl2.bdu;
  
    return ret;
  }
  
// Accelerometer full-scale selection.
int32_t lis2dw12_full_scale_set(lis2dw12_dev_t *pdev, lis2dw12_fullscale_t val){
    lis2dw12_ctrl6_t ctrl6;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    if (ret == 0){
        ctrl6.full_scale_sel = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_full_scale_get(lis2dw12_dev_t *pdev, lis2dw12_fullscale_t *val){
    lis2dw12_ctrl6_t ctrl6;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    switch (ctrl6.full_scale_sel){
        case LIS2DW12_2g:
            *val = LIS2DW12_2g;
            break;
  
        case LIS2DW12_4g:
            *val = LIS2DW12_4g;
            break;
  
        case LIS2DW12_8g:
            *val = LIS2DW12_8g;
            break;
    
        case LIS2DW12_16g:
            *val = LIS2DW12_16g;
            break;
    
        default:
            *val = LIS2DW12_2g;
            break;
    }
  
    return ret;
}
  
// The STATUS_REG register of the device.
int32_t lis2dw12_status_reg_get(lis2dw12_dev_t *pdev, lis2dw12_status_t *val){
    int32_t ret;

    ret = lis2dw12_read_reg(pdev, REG_STATUS, (uint8_t *) val, 1);

    return ret;
}
  
// Accelerometer new data available.
int32_t lis2dw12_flag_data_ready_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_status_t status;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_STATUS, (uint8_t *) &status, 1);
    *val = status.dat_rdy_xyz;
  
    return ret;
}

//Read all the interrupt/status flag of the device.
int32_t lis2dw12_all_sources_get(lis2dw12_dev_t *pdev, lis2dw12_all_sources_t *val){
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_STATUS_DUP, (uint8_t *) val, 5);
  
    return ret;
}
  
// Accelerometer X-axis user offset correction expressed in two’s
// complement, weight depends on bit USR_OFF_W. The value must be
// in the range [-127 127].
int32_t lis2dw12_usr_offset_x_set(lis2dw12_dev_t *pdev, uint8_t *buff){
    int32_t ret;
  
    ret = lis2dw12_write_reg(pdev, REG_X_OFS_USR, buff, 1);
  
    return ret;
}
  
int32_t lis2dw12_usr_offset_x_get(lis2dw12_dev_t *pdev, uint8_t *buff){
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_X_OFS_USR, buff, 1);
  
    return ret;
}
  
// Accelerometer Y-axis user offset correction expressed in two’s
// complement, weight depends on bit USR_OFF_W. The value must be
// in the range [-127 127].
int32_t lis2dw12_usr_offset_y_set(lis2dw12_dev_t *pdev, uint8_t *buff){
    int32_t ret;
  
    ret = lis2dw12_write_reg(pdev, REG_Y_OFS_USR, buff, 1);
  
    return ret;
}

int32_t lis2dw12_usr_offset_y_get(lis2dw12_dev_t *pdev, uint8_t *buff){
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_Y_OFS_USR, buff, 1);
  
    return ret;
}
  
// Accelerometer Z-axis user offset correction expressed in two’s
// complement, weight depends on bit USR_OFF_W. The value must be
// in the range [-127 127].
int32_t lis2dw12_usr_offset_z_set(lis2dw12_dev_t *pdev, uint8_t *buff){
    int32_t ret;
  
    ret = lis2dw12_write_reg(pdev, REG_Z_OFS_USR, buff, 1);
  
    return ret;
}
  
int32_t lis2dw12_usr_offset_z_get(lis2dw12_dev_t *pdev, uint8_t *buff){
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_Z_OFS_USR, buff, 1);
  
    return ret;
}
  
// Weight of XL user offset bits of registers X_OFS_USR,Y_OFS_USR, Z_OFS_USR.
int32_t lis2dw12_offset_weight_set(lis2dw12_dev_t *pdev, lis2dw12_usr_offset_weight_t val){
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    if (ret == 0){
        ctrl7.usr_off_w = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_offset_weight_get(lis2dw12_dev_t *pdev, lis2dw12_usr_offset_weight_t *val){
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    switch (ctrl7.usr_off_w){
        case LIS2DW12_LSB_977ug:
            *val = LIS2DW12_LSB_977ug;
            break;
    
        case LIS2DW12_LSB_15p6mg:
            *val = LIS2DW12_LSB_15p6mg;
            break;
    
        default:
            *val = LIS2DW12_LSB_977ug;
            break;
    }
  
    return ret;
}
  
// This section groups all the data output functions.

// Temperature data output register (r). L and H registers
// together express a 16-bit word in two’s complement.
int32_t lis2dw12_temperature_raw_get(lis2dw12_dev_t *pdev, int16_t *val){
    uint8_t buff[2];
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_OUT_T_L, buff, 2);
    *val = (int16_t)buff[1];
    *val = (*val * 256) + (int16_t)buff[0];
  
    return ret;
}
  
// Linear acceleration output register. The value is expressed as
// a 16-bit word in two’s complement.
int32_t lis2dw12_acceleration_raw_get(lis2dw12_dev_t *pdev, int16_t *val){
    uint8_t buff[6];
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_OUT_X_L, buff, 6);
    val[0] = (int16_t)buff[1];
    val[0] = (val[0] * 256) + (int16_t)buff[0];
    val[1] = (int16_t)buff[3];
    val[1] = (val[1] * 256) + (int16_t)buff[2];
    val[2] = (int16_t)buff[5];
    val[2] = (val[2] * 256) + (int16_t)buff[4];
  
    return ret;
}
  
// This section groups common useful functions.
  
// Device Who am I.
int32_t lis2dw12_device_id_get(lis2dw12_dev_t *pdev, uint8_t *buff){
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WHO_AM_I, buff, 1);
  
    return ret;
}
  
// Register address automatically incremented during multiple byte
// access with a serial interface.
int32_t lis2dw12_auto_increment_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    if (ret == 0){
        ctrl2.if_add_inc = val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_auto_increment_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    *val = ctrl2.if_add_inc;
  
    return ret;
}
  
// Software reset. Restore the default values in user registers.
int32_t lis2dw12_reset_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    if (ret == 0){
        ctrl2.soft_reset = val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_reset_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    *val = ctrl2.soft_reset;
  
    return ret;
}
  
// Reboot memory content. Reload the calibration parameters.
int32_t lis2dw12_boot_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    if (ret == 0){
        ctrl2.boot = val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_boot_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    *val = ctrl2.boot;
  
    return ret;
}
  
// Sensor self-test enable.
int32_t lis2dw12_self_test_set(lis2dw12_dev_t *pdev, lis2dw12_selftest_t val){
    lis2dw12_ctrl3_t ctrl3;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    if (ret == 0){
        ctrl3.self_test = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    }
  
    return ret;
}

int32_t lis2dw12_self_test_get(lis2dw12_dev_t *pdev, lis2dw12_selftest_t *val){
    lis2dw12_ctrl3_t ctrl3;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    switch (ctrl3.self_test){
        case LIS2DW12_XL_ST_DISABLE:
            *val = LIS2DW12_XL_ST_DISABLE;
            break;
    
        case LIS2DW12_XL_ST_POSITIVE:
            *val = LIS2DW12_XL_ST_POSITIVE;
            break;
    
        case LIS2DW12_XL_ST_NEGATIVE:
            *val = LIS2DW12_XL_ST_NEGATIVE;
            break;
    
        default:
            *val = LIS2DW12_XL_ST_DISABLE;
            break;
    }
  
    return ret;
}
  
// Data-ready pulsed / letched mode.
int32_t lis2dw12_data_ready_mode_set(lis2dw12_dev_t *pdev, lis2dw12_dat_rdy_pulsed_t val){
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    if (ret == 0){
        ctrl7.dat_rdy_pulsed = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    return ret;
}

int32_t lis2dw12_data_ready_mode_get(lis2dw12_dev_t *pdev,lis2dw12_dat_rdy_pulsed_t *val){
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    switch (ctrl7.dat_rdy_pulsed){
        case LIS2DW12_DRDY_LATCHED:
            *val = LIS2DW12_DRDY_LATCHED;
            break;
    
        case LIS2DW12_DRDY_PULSED:
            *val = LIS2DW12_DRDY_PULSED;
            break;
    
        default:
            *val = LIS2DW12_DRDY_LATCHED;
            break;
    }
  
    return ret;
}
  
// This section group all the functions concerning the filters

// Accelerometer filtering path for outputs.
int32_t lis2dw12_filter_path_set(lis2dw12_dev_t *pdev,lis2dw12_filt_dat_type_t val){
    lis2dw12_ctrl6_t ctrl6;
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    if (ret == 0){
        ctrl6.filt_path_sel = ((uint8_t) val & 0x10U) >> 4;
        ret = lis2dw12_write_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    }
  
    if (ret == 0){
        ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    if (ret == 0){
        ctrl7.usr_off_on_out = (uint8_t) val & 0x01U;
        ret = lis2dw12_write_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_filter_path_get(lis2dw12_dev_t *pdev,lis2dw12_filt_dat_type_t *val){
    lis2dw12_ctrl6_t ctrl6;
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    if (ret == 0){
        ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
        switch ((ctrl6.filt_path_sel << 4) + ctrl7.usr_off_on_out){
            case LIS2DW12_LPF_ON_OUT:
            *val = LIS2DW12_LPF_ON_OUT;
            break;
    
            case LIS2DW12_USER_OFFSET_ON_OUT:
            *val = LIS2DW12_USER_OFFSET_ON_OUT;
            break;
    
            case LIS2DW12_HIGH_PASS_ON_OUT:
            *val = LIS2DW12_HIGH_PASS_ON_OUT;
            break;
    
            default:
            *val = LIS2DW12_LPF_ON_OUT;
            break;
        }
    }
  
    return ret;
}
  
// Accelerometer cutoff filter frequency. Valid for low and high pass filter.
int32_t lis2dw12_filter_bandwidth_set(lis2dw12_dev_t *pdev, lis2dw12_bw_filt_t val){
    lis2dw12_ctrl6_t ctrl6;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    if (ret == 0){
        ctrl6.bnadwidth_filt = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_filter_bandwidth_get(lis2dw12_dev_t *pdev, lis2dw12_bw_filt_t *val){
    lis2dw12_ctrl6_t ctrl6;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL6, (uint8_t *) &ctrl6, 1);
    switch (ctrl6.bnadwidth_filt){
        case LIS2DW12_ODR_DIV_2:
            *val = LIS2DW12_ODR_DIV_2;
            break;
    
        case LIS2DW12_ODR_DIV_4:
            *val = LIS2DW12_ODR_DIV_4;
            break;
    
        case LIS2DW12_ODR_DIV_10:
            *val = LIS2DW12_ODR_DIV_10;
            break;
    
        case LIS2DW12_ODR_DIV_20:
            *val = LIS2DW12_ODR_DIV_20;
            break;
    
        default:
            *val = LIS2DW12_ODR_DIV_2;
            break;
    }
  
    return ret;
}
  
// Enable HP filter reference mode.
int32_t lis2dw12_reference_mode_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    if (ret == 0){
        ctrl7.hp_ref_mode = val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    return ret;
}

int32_t lis2dw12_reference_mode_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    *val = ctrl7.hp_ref_mode;
  
    return ret;
}
  
// This section groups all the functions concerning main serial interface management (not auxiliary)

// SPI Serial Interface Mode selection.
int32_t lis2dw12_spi_mode_set(lis2dw12_dev_t *pdev, lis2dw12_spi_mode_t val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    if (ret == 0){
        ctrl2.sim = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_spi_mode_get(lis2dw12_dev_t *pdev, lis2dw12_spi_mode_t *val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    switch (ctrl2.sim){
        case LIS2DW12_SPI_4_WIRE:
            *val = LIS2DW12_SPI_4_WIRE;
            break;
    
        case LIS2DW12_SPI_3_WIRE:
            *val = LIS2DW12_SPI_3_WIRE;
            break;
    
        default:
            *val = LIS2DW12_SPI_4_WIRE;
            break;
    }
  
    return ret;
  }
  
// Disable / Enable I2C interface.
int32_t lis2dw12_i2c_interface_set(lis2dw12_dev_t *pdev, lis2dw12_i2c_disable_t val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    if (ret == 0){
        ctrl2.i2c_disable = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_i2c_interface_get(lis2dw12_dev_t *pdev, lis2dw12_i2c_disable_t *val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    switch (ctrl2.i2c_disable){
        case LIS2DW12_I2C_ENABLE:
            *val = LIS2DW12_I2C_ENABLE;
            break;
    
        case LIS2DW12_I2C_DISABLE:
            *val = LIS2DW12_I2C_DISABLE;
            break;
    
        default:
            *val = LIS2DW12_I2C_ENABLE;
            break;
    }
  
    return ret;
}
  
// Disconnect CS pull-up.
int32_t lis2dw12_cs_mode_set(lis2dw12_dev_t *pdev, lis2dw12_cs_pullup_disc_t val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    if (ret == 0){
        ctrl2.cs_pu_disc = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    }
  
    return ret;
}

int32_t lis2dw12_cs_mode_get(lis2dw12_dev_t *pdev, lis2dw12_cs_pullup_disc_t *val){
    lis2dw12_ctrl2_t ctrl2;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL2, (uint8_t *) &ctrl2, 1);
    switch (ctrl2.cs_pu_disc){
        case LIS2DW12_PULL_UP_CONNECT:
            *val = LIS2DW12_PULL_UP_CONNECT;
            break;
    
        case LIS2DW12_PULL_UP_DISCONNECT:
            *val = LIS2DW12_PULL_UP_DISCONNECT;
            break;
    
        default:
            *val = LIS2DW12_PULL_UP_CONNECT;
            break;
    }
  
    return ret;
}
  
// This section groups all the functions that manage interrupt pins

// Interrupt active-high/low.
int32_t lis2dw12_pin_polarity_set(lis2dw12_dev_t *pdev,lis2dw12_h_l_active_t val){
    lis2dw12_ctrl3_t ctrl3;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    if (ret == 0){
        ctrl3.h_l_active = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_pin_polarity_get(lis2dw12_dev_t *pdev, lis2dw12_h_l_active_t *val){
    lis2dw12_ctrl3_t ctrl3;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    switch (ctrl3.h_l_active){
        case LIS2DW12_ACTIVE_HIGH:
            *val = LIS2DW12_ACTIVE_HIGH;
            break;
    
        case LIS2DW12_ACTIVE_LOW:
            *val = LIS2DW12_ACTIVE_LOW;
            break;
    
        default:
            *val = LIS2DW12_ACTIVE_HIGH;
            break;
    }
  
    return ret;
}
  
//Latched/pulsed interrupt.
int32_t lis2dw12_int_notification_set(lis2dw12_dev_t *pdev, lis2dw12_latch_int_req_t val){
    lis2dw12_ctrl3_t ctrl3;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    if (ret == 0){
        ctrl3.latch_int = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_int_notification_get(lis2dw12_dev_t *pdev, lis2dw12_latch_int_req_t *val){
    lis2dw12_ctrl3_t ctrl3;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    switch (ctrl3.latch_int){
        case LIS2DW12_INT_PULSED:
            *val = LIS2DW12_INT_PULSED;
            break;
    
        case LIS2DW12_INT_LATCHED:
            *val = LIS2DW12_INT_LATCHED;
            break;
    
        default:
            *val = LIS2DW12_INT_PULSED;
            break;
    }
  
    return ret;
}
  
// Push-pull/open drain selection on interrupt pads.
int32_t lis2dw12_pin_mode_set(lis2dw12_dev_t *pdev, lis2dw12_pp_od_t val){
    lis2dw12_ctrl3_t ctrl3;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    if (ret == 0){
        ctrl3.pp_od = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    }
  
    return ret;
}

int32_t lis2dw12_pin_mode_get(lis2dw12_dev_t *pdev, lis2dw12_pp_od_t *val){
    lis2dw12_ctrl3_t ctrl3;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL3, (uint8_t *) &ctrl3, 1);
    switch (ctrl3.pp_od){
        case LIS2DW12_PUSH_PULL:
            *val = LIS2DW12_PUSH_PULL;
            break;
    
        case LIS2DW12_OPEN_DRAIN:
            *val = LIS2DW12_OPEN_DRAIN;
            break;
    
        default:
            *val = LIS2DW12_PUSH_PULL;
            break;
    }
  
    return ret;
}
  
// Select the signal that need to route on int1 pad.
int32_t lis2dw12_pin_int1_route_set(lis2dw12_dev_t *pdev, lis2dw12_ctrl4_int1_pad_ctrl_t *val){
    lis2dw12_ctrl5_int2_pad_ctrl_t ctrl5;
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL5, (uint8_t *)&ctrl5, 1);
    if (ret == 0){
      ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    if (ret == 0){
        if ((val->int1_double_tap |
            val->int1_fifo_full |
            val->int1_wake_up |
            val->int1_single_tap |
            val->int1_6d |
            ctrl5.int2_sleep_state |
            ctrl5.int2_sleep_chg) != PROPERTY_DISABLE)
        {
            ctrl7.interrupts_enable = PROPERTY_ENABLE;
        }else{
            ctrl7.interrupts_enable = PROPERTY_DISABLE;
        }
    
        ret = lis2dw12_write_reg(pdev, REG_CTRL4,(uint8_t *) val, 1);
    }
  
    if (ret == 0){
        ret = lis2dw12_write_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_pin_int1_route_get(lis2dw12_dev_t *pdev, lis2dw12_ctrl4_int1_pad_ctrl_t *val){
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL4,(uint8_t *) val, 1);
  
    return ret;
}
  
// Select the signal that need to route on int2 pad.
int32_t lis2dw12_pin_int2_route_set(lis2dw12_dev_t *pdev, lis2dw12_ctrl5_int2_pad_ctrl_t *val){
    lis2dw12_ctrl4_int1_pad_ctrl_t ctrl4;
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL4, (uint8_t *) &ctrl4, 1);
    if (ret == 0){
      ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    if (ret == 0){
        if ((ctrl4.int1_double_tap |
            ctrl4.int1_fifo_full |
            ctrl4.int1_wake_up |
            ctrl4.int1_single_tap |
            ctrl4.int1_6d |
            val->int2_sleep_state | val->int2_sleep_chg) != PROPERTY_DISABLE)
        {
            ctrl7.interrupts_enable = PROPERTY_ENABLE;
        }else{
            ctrl7.interrupts_enable = PROPERTY_DISABLE;
        }
    
        ret = lis2dw12_write_reg(pdev, REG_CTRL5, (uint8_t *) val, 1);
    }
  
    if (ret == 0){
      ret = lis2dw12_write_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    return ret;
}

int32_t lis2dw12_pin_int2_route_get(lis2dw12_dev_t *pdev, lis2dw12_ctrl5_int2_pad_ctrl_t *val){
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL5, (uint8_t *) val, 1);
  
    return ret;
}
 
// All interrupt signals become available on INT1 pin.
int32_t lis2dw12_all_on_int1_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    if (ret == 0){
        ctrl7.int2_on_int1 = val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_all_on_int1_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_ctrl7_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &reg, 1);
    *val = reg.int2_on_int1;
  
    return ret;
}
  
// This section groups all the functions that manage the Wake Up event generation.

// Threshold for wakeup.1 LSB = FS_XL / 64.
int32_t lis2dw12_wkup_threshold_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_wake_up_ths_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_THS, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.wakeup_thsh = val;
        ret = lis2dw12_write_reg(pdev, REG_WAKE_UP_THS, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_wkup_threshold_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_wake_up_ths_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_THS, (uint8_t *) &reg, 1);
    *val = reg.wakeup_thsh;
  
    return ret;
}
  
// Wake up duration event.1LSb = 1 / ODR.
int32_t lis2dw12_wkup_dur_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_wake_up_dur_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.wakeup_dur = val;
        ret = lis2dw12_write_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &reg, 1);
    }
  
    return ret;
}

int32_t lis2dw12_wkup_dur_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_wake_up_dur_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &reg, 1);
    *val = reg.wakeup_dur;
  
    return ret;
}
  
// Data sent to wake-up interrupt function.
int32_t lis2dw12_wkup_feed_data_set(lis2dw12_dev_t *pdev, lis2dw12_usr_offset_on_wakeup_t val){
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    if (ret == 0){
        ctrl7.usr_off_on_wu = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_wkup_feed_data_get(lis2dw12_dev_t *pdev, lis2dw12_usr_offset_on_wakeup_t *val)
  {
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    switch (ctrl7.usr_off_on_wu){
        case LIS2DW12_HP_FEED:
            *val = LIS2DW12_HP_FEED;
            break;
    
        case LIS2DW12_USER_OFFSET_FEED:
            *val = LIS2DW12_USER_OFFSET_FEED;
            break;
    
        default:
            *val = LIS2DW12_HP_FEED;
            break;
    }
  
    return ret;
}
  
// This section groups all the functions concerning activity/inactivity detection.

// Config activity / inactivity or stationary / motion detection.
int32_t lis2dw12_act_mode_set(lis2dw12_dev_t *pdev, lis2dw12_sleep_on_t val){
    lis2dw12_wake_up_ths_t wake_up_ths;
    lis2dw12_wake_up_dur_t wake_up_dur;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_THS, (uint8_t *) &wake_up_ths, 1);
    if (ret == 0){
      ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &wake_up_dur, 1);
    }
  
    if (ret == 0){
      wake_up_ths.sleep_on = (uint8_t) val & 0x01U;
      ret = lis2dw12_write_reg(pdev, REG_WAKE_UP_THS, (uint8_t *) &wake_up_ths, 1);
    }
  
    if (ret == 0){
      wake_up_dur.stationary = ((uint8_t)val & 0x02U) >> 1;
      ret = lis2dw12_write_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &wake_up_dur, 1);
    }
  
    return ret;
}

int32_t lis2dw12_act_mode_get(lis2dw12_dev_t *pdev, lis2dw12_sleep_on_t *val){
    lis2dw12_wake_up_ths_t wake_up_ths;
    lis2dw12_wake_up_dur_t wake_up_dur;;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_THS, (uint8_t *) &wake_up_ths, 1);
    if (ret == 0){
        ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &wake_up_dur, 1);
        switch ((wake_up_dur.stationary << 1) + wake_up_ths.sleep_on){
            case LIS2DW12_NO_DETECTION:
                *val = LIS2DW12_NO_DETECTION;
                break;
    
            case LIS2DW12_DETECT_ACT_INACT:
                *val = LIS2DW12_DETECT_ACT_INACT;
                break;
    
            case LIS2DW12_DETECT_STAT_MOTION:
                *val = LIS2DW12_DETECT_STAT_MOTION;
                break;
    
            default:
                *val = LIS2DW12_NO_DETECTION;
                break;
        }
    }
    
    return ret;
}
  
// Duration to go in sleep mode (1 LSb = 512 / ODR).
int32_t lis2dw12_act_sleep_dur_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_wake_up_dur_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.sleep_dur = val;
        ret = lis2dw12_write_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_act_sleep_dur_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_wake_up_dur_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &reg, 1);
    *val = reg.sleep_dur;
  
    return ret;
}
  
// This section groups all the functions that manage the tap
// and double tap event generation.
 
// Threshold for tap recognition.
int32_t lis2dw12_tap_threshold_x_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_tap_ths_x_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_X, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.tap_x_thsh = val;
        ret = lis2dw12_write_reg(pdev, REG_TAP_THS_X, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_tap_threshold_x_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_tap_ths_x_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_X, (uint8_t *) &reg, 1);
    *val = reg.tap_x_thsh;
  
    return ret;
}
  
// Threshold for tap recognition.[set]
int32_t lis2dw12_tap_threshold_y_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_tap_ths_y_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Y, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.tap_y_thsh = val;
        ret = lis2dw12_write_reg(pdev, REG_TAP_THS_Y, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_tap_threshold_y_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_tap_ths_y_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Y, (uint8_t *) &reg, 1);
    *val = reg.tap_y_thsh;
  
    return ret;
}
  
// Selection of axis priority for TAP detection.
int32_t lis2dw12_tap_axis_priority_set(lis2dw12_dev_t *pdev, lis2dw12_tap_prior_t val){
    lis2dw12_tap_ths_y_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Y, (uint8_t *) &reg, 1);
    if (ret == 0){
      reg.tap_prior = (uint8_t) val;
      ret = lis2dw12_write_reg(pdev, REG_TAP_THS_Y, (uint8_t *) &reg, 1);
    }
  
    return ret;
}

int32_t lis2dw12_tap_axis_priority_get(lis2dw12_dev_t *pdev, lis2dw12_tap_prior_t *val){
    lis2dw12_tap_ths_y_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Y, (uint8_t *) &reg, 1);
    switch (reg.tap_prior){
        case LIS2DW12_XYZ:
            *val = LIS2DW12_XYZ;
            break;
    
        case LIS2DW12_YXZ:
            *val = LIS2DW12_YXZ;
            break;
    
        case LIS2DW12_XZY:
            *val = LIS2DW12_XZY;
            break;
    
        case LIS2DW12_ZYX:
            *val = LIS2DW12_ZYX;
            break;
    
        case LIS2DW12_YZX:
            *val = LIS2DW12_YZX;
            break;
    
        case LIS2DW12_ZXY:
            *val = LIS2DW12_ZXY;
            break;
    
        default:
            *val = LIS2DW12_XYZ;
            break;
    }
  
    return ret;
}
  
// Threshold for tap recognition.
int32_t lis2dw12_tap_threshold_z_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_tap_ths_z_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.tap_z_thsh = val;
        ret = lis2dw12_write_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_tap_threshold_z_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_tap_ths_z_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    *val = reg.tap_z_thsh;
  
    return ret;
}
  
// Enable Z direction in tap recognition.
int32_t lis2dw12_tap_detection_on_z_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_tap_ths_z_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.tap_z_en = val;
        ret = lis2dw12_write_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_tap_detection_on_z_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_tap_ths_z_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    *val = reg.tap_z_en;
  
    return ret;
}
  
// Enable Y direction in tap recognition.
int32_t lis2dw12_tap_detection_on_y_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_tap_ths_z_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.tap_y_en = val;
        ret = lis2dw12_write_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_tap_detection_on_y_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_tap_ths_z_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    *val = reg.tap_y_en;
  
    return ret;
}
  
// Enable X direction in tap recognition.
int32_t lis2dw12_tap_detection_on_x_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_tap_ths_z_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.tap_x_en = val;
        ret = lis2dw12_write_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    }
  
    return ret;
}

int32_t lis2dw12_tap_detection_on_x_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_tap_ths_z_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_Z, (uint8_t *) &reg, 1);
    *val = reg.tap_x_en;
  
    return ret;
}
  
// Maximum duration is the maximum time of an overthreshold signal
// detection to be recognized as a tap event. The default value
// of these bits is 00b which corresponds to 4*ODR_XL time.
// If the SHOCK[1:0] bits are set to a different value, 1LSB
// corresponds to 8*ODR_XL time.
int32_t lis2dw12_tap_shock_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_int_dur_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_INT_DUR, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.shock = val;
        ret = lis2dw12_write_reg(pdev, REG_INT_DUR, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_tap_shock_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_int_dur_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_INT_DUR, (uint8_t *) &reg, 1);
    *val = reg.shock;
  
    return ret;
}
  
// Quiet time is the time after the first detected tap in which
// there must not be any overthreshold event.
// The default value of these bits is 00b which corresponds
// to 2*ODR_XL time. If the QUIET[1:0] bits are set to a different
// value, 1LSB corresponds to 4*ODR_XL time.
int32_t lis2dw12_tap_quiet_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_int_dur_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_INT_DUR, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.quiet = val;
        ret = lis2dw12_write_reg(pdev, REG_INT_DUR, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_tap_quiet_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_int_dur_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_INT_DUR, (uint8_t *) &reg, 1);
    *val = reg.quiet;
  
    return ret;
}
  
// When double tap recognition is enabled, this register expresses
// the maximum time between two consecutive detected taps to
// determine a double tap event.
// The default value of these bits is 0000b which corresponds
// to 16*ODR_XL time. If the DUR[3:0] bits are set to a different
// value, 1LSB corresponds to 32*ODR_XL time.
int32_t lis2dw12_tap_dur_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_int_dur_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_INT_DUR, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.latency = val;
        ret = lis2dw12_write_reg(pdev, REG_INT_DUR, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_tap_dur_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_int_dur_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_INT_DUR, (uint8_t *) &reg, 1);
    *val = reg.latency;
  
    return ret;
}
  
// Single/double-tap event enable.
int32_t lis2dw12_tap_mode_set(lis2dw12_dev_t *pdev, lis2dw12_single_double_tap_t val){
    lis2dw12_wake_up_ths_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_THS, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.single_double_tap = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_WAKE_UP_THS, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_tap_mode_get(lis2dw12_dev_t *pdev, lis2dw12_single_double_tap_t *val){
    lis2dw12_wake_up_ths_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_THS, (uint8_t *) &reg, 1);
    switch (reg.single_double_tap){
        case LIS2DW12_ONLY_SINGLE:
            *val = LIS2DW12_ONLY_SINGLE;
            break;
    
        case LIS2DW12_BOTH_SINGLE_DOUBLE:
            *val = LIS2DW12_BOTH_SINGLE_DOUBLE;
            break;
    
        default:
            *val = LIS2DW12_ONLY_SINGLE;
            break;
    }
  
    return ret;
}
  
// Read the tap / double tap source register.
int32_t lis2dw12_tap_src_get(lis2dw12_dev_t *pdev, lis2dw12_tap_src_t *val){
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_SRC, (uint8_t *) val, 1);
  
    return ret;
}
  
// This section groups all the functions concerning six
// position detection (6D).

// Threshold for 4D/6D function.
int32_t lis2dw12_6d_threshold_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_tap_ths_x_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_X, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg._6d_thsh = val;
        ret = lis2dw12_write_reg(pdev, REG_TAP_THS_X, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_6d_threshold_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_tap_ths_x_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_X, (uint8_t *) &reg, 1);
    *val = reg._6d_thsh;
  
    return ret;
}
  
// 4D orientation detection enable.
int32_t lis2dw12_4d_mode_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_tap_ths_x_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_X, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg._4d_en = val;
        ret = lis2dw12_write_reg(pdev, REG_TAP_THS_X, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  

int32_t lis2dw12_4d_mode_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_tap_ths_x_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_TAP_THS_X, (uint8_t *) &reg, 1);
    *val = reg._4d_en;
  
    return ret;
}
  
// Read the 6D tap source register.
int32_t lis2dw12_6d_src_get(lis2dw12_dev_t *pdev, lis2dw12_sixd_src_t *val){
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_SIXD_SRC, (uint8_t *) val, 1);
  
    return ret;
}

//Data sent to 6D interrupt function.
int32_t lis2dw12_6d_feed_data_set(lis2dw12_dev_t *pdev, lis2dw12_lpass_on6d_t val){
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    if (ret == 0){
        ctrl7.lpass_on6d = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_6d_feed_data_get(lis2dw12_dev_t *pdev, lis2dw12_lpass_on6d_t *val)
  {
    lis2dw12_ctrl7_t ctrl7;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_CTRL7, (uint8_t *) &ctrl7, 1);
    switch (ctrl7.lpass_on6d){
        case LIS2DW12_ODR_DIV_2_FEED:
            *val = LIS2DW12_ODR_DIV_2_FEED;
            break;
    
        case LIS2DW12_LPF2_FEED:
            *val = LIS2DW12_LPF2_FEED;
            break;
    
        default:
            *val = LIS2DW12_ODR_DIV_2_FEED;
            break;
    }
  
    return ret;
}
  
// This section group all the functions concerning the free fall detection.

// Free fall duration event(1LSb = 1 / ODR).
int32_t lis2dw12_freefall_dur_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_wake_up_dur_t wake_up_dur;
    lis2dw12_free_fall_t free_fall;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &wake_up_dur, 1);
    if (ret == 0){
        ret = lis2dw12_read_reg(pdev, REG_FREE_FALL, (uint8_t *) &free_fall, 1);
    }
  
    if (ret == 0){
        wake_up_dur.freefall_dur5 = ((uint8_t) val & 0x20U) >> 5;
        free_fall.freefall_dur4_0 = (uint8_t) val & 0x1FU;
        ret = lis2dw12_write_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &wake_up_dur, 1);
    }
  
    if (ret == 0){
      ret = lis2dw12_write_reg(pdev, REG_FREE_FALL, (uint8_t *) &free_fall, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_freefall_dur_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_wake_up_dur_t wake_up_dur;
    lis2dw12_free_fall_t free_fall;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_WAKE_UP_DUR, (uint8_t *) &wake_up_dur, 1);
    if (ret == 0){
        ret = lis2dw12_read_reg(pdev, REG_FREE_FALL, (uint8_t *) &free_fall, 1);
        *val = (wake_up_dur.freefall_dur5 << 5) + free_fall.freefall_dur4_0;
    }
  
    return ret;
}
  
// Free fall threshold setting.
int32_t lis2dw12_freefall_threshold_set(lis2dw12_dev_t *pdev, lis2dw12_freefall_thsh_t val){
    lis2dw12_free_fall_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_FREE_FALL, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.freefall_thsh = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_FREE_FALL, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_freefall_threshold_get(lis2dw12_dev_t *pdev, lis2dw12_freefall_thsh_t *val){
    lis2dw12_free_fall_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_FREE_FALL, (uint8_t *) &reg, 1);
    switch (reg.freefall_thsh){
        case LIS2DW12_FF_TSH_5LSB_FS2g:
            *val = LIS2DW12_FF_TSH_5LSB_FS2g;
            break;
    
        case LIS2DW12_FF_TSH_7LSB_FS2g:
            *val = LIS2DW12_FF_TSH_7LSB_FS2g;
            break;
    
        case LIS2DW12_FF_TSH_8LSB_FS2g:
            *val = LIS2DW12_FF_TSH_8LSB_FS2g;
            break;
    
        case LIS2DW12_FF_TSH_10LSB_FS2g:
            *val = LIS2DW12_FF_TSH_10LSB_FS2g;
            break;
    
        case LIS2DW12_FF_TSH_11LSB_FS2g:
            *val = LIS2DW12_FF_TSH_11LSB_FS2g;
            break;
    
        case LIS2DW12_FF_TSH_13LSB_FS2g:
            *val = LIS2DW12_FF_TSH_13LSB_FS2g;
            break;
    
        case LIS2DW12_FF_TSH_15LSB_FS2g:
            *val = LIS2DW12_FF_TSH_15LSB_FS2g;
            break;
    
        case LIS2DW12_FF_TSH_16LSB_FS2g:
            *val = LIS2DW12_FF_TSH_16LSB_FS2g;
            break;
    
        default:
            *val = LIS2DW12_FF_TSH_5LSB_FS2g;
            break;
    }
  
    return ret;
}
  
// This section group all the functions concerning the fifo usage

// FIFO watermark level selection.
int32_t lis2dw12_fifo_watermark_set(lis2dw12_dev_t *pdev, uint8_t val){
    lis2dw12_fifo_ctrl_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_FIFO_CTRL, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.fifo_thsh = val;
        ret = lis2dw12_write_reg(pdev, REG_FIFO_CTRL, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_fifo_watermark_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_fifo_ctrl_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_FIFO_CTRL, (uint8_t *) &reg, 1);
    *val = reg.fifo_thsh;
  
    return ret;
}
  
// FIFO mode selection.
int32_t lis2dw12_fifo_mode_set(lis2dw12_dev_t *pdev, lis2dw12_fifo_mode_t val){
    lis2dw12_fifo_ctrl_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_FIFO_CTRL, (uint8_t *) &reg, 1);
    if (ret == 0){
        reg.fifo_mode = (uint8_t) val;
        ret = lis2dw12_write_reg(pdev, REG_FIFO_CTRL, (uint8_t *) &reg, 1);
    }
  
    return ret;
}
  
int32_t lis2dw12_fifo_mode_get(lis2dw12_dev_t *pdev, lis2dw12_fifo_mode_t *val){
    lis2dw12_fifo_ctrl_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_FIFO_CTRL, (uint8_t *) &reg, 1);
    switch (reg.fifo_mode){
        case LIS2DW12_BYPASS_MODE:
            *val = LIS2DW12_BYPASS_MODE;
            break;
    
        case LIS2DW12_FIFO_MODE:
            *val = LIS2DW12_FIFO_MODE;
            break;
    
        case LIS2DW12_STREAM_TO_FIFO_MODE:
            *val = LIS2DW12_STREAM_TO_FIFO_MODE;
            break;
    
        case LIS2DW12_BYPASS_TO_STREAM_MODE:
            *val = LIS2DW12_BYPASS_TO_STREAM_MODE;
            break;
    
        case LIS2DW12_STREAM_MODE:
            *val = LIS2DW12_STREAM_MODE;
            break;
    
        default:
            *val = LIS2DW12_BYPASS_MODE;
            break;
    }
  
    return ret;
}
  
// Read Number of unread samples stored in FIFO.
int32_t lis2dw12_fifo_data_level_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_fifo_samples_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_FIFO_SAMPLES, (uint8_t *) &reg, 1);
    *val = reg.unread_nums;
  
    return ret;
}

// Read FIFO overrun status.
int32_t lis2dw12_fifo_overrun_flag_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_fifo_samples_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_FIFO_SAMPLES, (uint8_t *) &reg, 1);
    *val = reg.fifo_overrun;
  
    return ret;
}

//Read FIFO threshold status flag.
int32_t lis2dw12_fifo_wtm_flag_get(lis2dw12_dev_t *pdev, uint8_t *val){
    lis2dw12_fifo_samples_t reg;
    int32_t ret;
  
    ret = lis2dw12_read_reg(pdev, REG_FIFO_SAMPLES, (uint8_t *) &reg, 1);
    *val = reg.fifo_thsh_flag;
  
    return ret;
}
  
