// ============================================================================
//
// Function: 
//   Plays a snippet of a retro-style tune using square waves.
//
// Clock:
//   - input Clock (clk): 12.272 MHz
//   - Generated sclk: 1.534 MHz (12.272 / 8)
//   - Resulting lrclk: 47.9375 kHz (Excellent Compatibility)
//
// ============================================================================

module AUDIO_IF(
    //
    reset_n,
    sclk,
    lrclk,
    i2s,
    clk  // Expecting 12.272 MHz input
);

/*****************************************************************************
 * Port Declarations                             *
 *****************************************************************************/

output                      sclk;   
output                      lrclk;
input                       reset_n;
output  [3:0]               i2s;
input                       clk;

parameter   DATA_WIDTH              =   16;
localparam  AMPLITUDE               =   20000; // Volume of the sound

/*****************************************************************************
 * Internal Wires and Registers Declarations                 *
 *****************************************************************************/

// Clock Divider
reg     sclk_reg = 1'b0;
reg     [2:0] clk_div_count = 3'b0;

// I2S Signal Generation
reg                         lrclk;
reg     [4:0]               sclk_Count; 
reg     [4:0]               Data_Count;

// Audio Data Generation
reg signed [DATA_WIDTH-1:0] Data_Bit;
reg     [3:0]               i2s;

// --- Melody and Tone Generation ---
localparam NOTE_C5_PD = 44; 
localparam NOTE_G4_PD = 60; 
localparam NOTE_E4_PD = 71; 
localparam NOTE_A4_PD = 53;
localparam NOTE_B4_PD = 47;
localparam NOTE_AS4_PD = 50;
localparam NOTE_PAUSE = 16'd0;

// Melody Note Sequence
reg [7:0] melody_notes [0:31]; 

// Melody Note Duration Sequence
reg [17:0] melody_durations [0:31]; 

initial begin
    // Melody Data Initialization
    melody_notes[0]  = NOTE_E4_PD; melody_durations[0]  = 18'd5000;
    melody_notes[1]  = NOTE_E4_PD; melody_durations[1]  = 18'd5000;
    melody_notes[2]  = NOTE_PAUSE; melody_durations[2]  = 18'd5000;
    melody_notes[3]  = NOTE_E4_PD; melody_durations[3]  = 18'd5000;
    melody_notes[4]  = NOTE_PAUSE; melody_durations[4]  = 18'd5000;
    melody_notes[5]  = NOTE_C5_PD; melody_durations[5]  = 18'd5000;
    melody_notes[6]  = NOTE_E4_PD; melody_durations[6]  = 18'd10000;
    melody_notes[7]  = NOTE_G4_PD; melody_durations[7]  = 18'd10000;
    melody_notes[8]  = NOTE_PAUSE; melody_durations[8]  = 18'd10000;
    melody_notes[9]  = NOTE_G4_PD; melody_durations[9]  = 18'd10000; 
    melody_notes[10] = NOTE_PAUSE; melody_durations[10] = 18'd10000;
    // loop back
    melody_notes[11] = NOTE_E4_PD; melody_durations[11] = 18'd5000;
    melody_notes[12] = NOTE_E4_PD; melody_durations[12] = 18'd5000;
    melody_notes[13]  = NOTE_PAUSE; melody_durations[13] = 18'd5000;
    melody_notes[14] = NOTE_E4_PD; melody_durations[14] = 18'd5000;
    melody_notes[15]  = NOTE_PAUSE; melody_durations[15] = 18'd5000;
    melody_notes[16] = NOTE_C5_PD; melody_durations[16] = 18'd5000;
    melody_notes[17] = NOTE_E4_PD; melody_durations[17] = 18'd10000;
    melody_notes[18] = NOTE_G4_PD; melody_durations[18] = 18'd10000;
    melody_notes[19] = NOTE_PAUSE; melody_durations[19] = 18'd10000;
    melody_notes[20] = NOTE_G4_PD; melody_durations[20] = 18'd10000; 
    melody_notes[21] = NOTE_PAUSE; melody_durations[21] = 18'd10000;
    melody_notes[22] = NOTE_PAUSE; melody_durations[22] = 18'd10000;
end

reg [15:0] note_period_counter;
reg [17:0] note_duration_counter;
reg [4:0]  melody_step;

/*****************************************************************************
 * Sequential logic                              *
 *****************************************************************************/

// Clock Divider for 12.272 MHz input
assign sclk = sclk_reg;
always @(posedge clk or negedge reset_n)
begin
    if (!reset_n) begin
        clk_div_count <= 3'b0; 
        sclk_reg <= 1'b0;
    end else begin
        clk_div_count <= clk_div_count + 1;
        if (clk_div_count == 3'b011) begin
            sclk_reg <= ~sclk_reg;
        end
    end
end

// LRCLK Generation
always@(negedge sclk or negedge reset_n)
begin
    if(!reset_n) begin
      lrclk<= 1'b0; sclk_Count<= 5'b0;
    end else if(sclk_Count >= (DATA_WIDTH-1)) begin
        sclk_Count <= 0; lrclk <= ~lrclk;
    end else begin
        sclk_Count <= sclk_Count + 1;
    end
end

// Melody Sequencer and Tone Generator
always @(posedge lrclk or negedge reset_n)
begin
    if (!reset_n)
    begin
        melody_step <= 0; note_duration_counter <= 0; note_period_counter <= 0; Data_Bit <= 0;
    end
    else
    begin
        if (note_duration_counter >= melody_durations[melody_step] - 1) begin
            note_duration_counter <= 0; melody_step <= (melody_step == 21) ? 0 : melody_step + 1;
        end else begin
            note_duration_counter <= note_duration_counter + 1;
        end
        if (melody_notes[melody_step] == NOTE_PAUSE) begin
            Data_Bit <= 0;
        end else begin
            if (note_period_counter >= melody_notes[melody_step] - 1) begin
                note_period_counter <= 0; Data_Bit <= -Data_Bit;
            end else begin
                note_period_counter <= note_period_counter + 1;
            end
        end
        if (note_duration_counter == 0) begin
            Data_Bit <= AMPLITUDE;
        end
    end
end

// I2S Data Serialization
always@(negedge sclk or negedge reset_n)
begin
  if(!reset_n) begin
    Data_Count <= 0;
  end else begin
    if(Data_Count >= DATA_WIDTH-1) Data_Count <= 0;
    else Data_Count <= Data_Count + 1;
  end
end

always@(negedge sclk or negedge reset_n)
begin
  if(!reset_n) begin
    i2s <= 4'b0;
  end else begin
    i2s[0] <= Data_Bit[DATA_WIDTH - 1 - Data_Count];
    i2s[1] <= Data_Bit[DATA_WIDTH - 1 - Data_Count];
    i2s[2] <= Data_Bit[DATA_WIDTH - 1 - Data_Count];
    i2s[3] <= Data_Bit[DATA_WIDTH - 1 - Data_Count];
  end
end

endmodule