// (C) 2001-2025 Altera Corporation. All rights reserved.
// Your use of Altera Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Altera Program License Subscription 
// Agreement, Altera IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Altera and sold by 
// Altera or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


// Module: refclk_recovery_xcvr_resync_std
//
// Description:
//  A general purpose resynchronization module that uses the recommended eth_f_altera_std_synchronizer
//  and eth_f_altera_std_synchronizer_nocut synchronizer
//  
//  Parameters:
//    SYNC_CHAIN_LENGTH
//      - Specifies the length of the synchronizer chain for metastability
//        retiming.
//    WIDTH
//      - Specifies the number of bits you want to synchronize. Controls the width of the
//        d and q ports.
//    SLOW_CLOCK - USE WITH CAUTION. 
//      - Leaving this setting at its default will create a standard resynch circuit that
//        merely passes the input data through a chain of flip-flops. This setting assumes
//        that the input data has a pulse width longer than one clock cycle sufficient to
//        satisfy setup and hold requirements on at least one clock edge.
//      - By setting this to 1 (USE CAUTION) you are creating an asynchronous
//        circuit that will capture the input data regardless of the pulse width and 
//        its relationship to the clock. However it is more difficult to apply static
//        timing constraints as it ties the data input to the clock input of the flop.
//        This implementation assumes the data rate is slow enough
//    INIT_VALUE
//      - Specifies the initial values of the synchronization registers.
//	  NO_CUT
//		- Specifies whether to apply embedded false path timing constraint. 
//		  0: Apply the constraint 1: Not applying the constraint
//
 
`timescale 1ps/1ps 
 
module refclk_recovery_xcvr_resync_std #(
    parameter SYNC_CHAIN_LENGTH = 2,  // Number of flip-flops for retiming. Must be >1
    parameter WIDTH             = 1,  // Number of bits to resync
    parameter SLOW_CLOCK        = 0,  // See description above
    parameter INIT_VALUE        = 0,
    parameter NO_CUT		= 1	  // See description above
  ) (
  input   wire              clk,
  input   wire              reset,
  input   wire  [WIDTH-1:0] d,
  output  wire  [WIDTH-1:0] q
  );
 
localparam  INT_LEN       = (SYNC_CHAIN_LENGTH > 1) ? SYNC_CHAIN_LENGTH : 2;
localparam  L_INIT_VALUE  = (INIT_VALUE == 1) ? 1'b1 : 1'b0;
 
genvar ig;
 
// Generate a synchronizer chain for each bit
generate for(ig=0;ig<WIDTH;ig=ig+1) begin : resync_chains
	wire                d_in;   // Input to sychronization chain.
	wire				sync_d_in;
	wire		        sync_q_out;
	
	// Adding inverter to the input of first sync register and output of the last sync register to implement power-up high for INIT_VALUE=1
	assign sync_d_in = (INIT_VALUE == 1) ? ~d_in : d_in;
	assign q[ig] = (INIT_VALUE == 1)  ? ~sync_q_out : sync_q_out;
	
	// NOT SUPPORTED if (NO_CUT == 0) begin		
	// NOT SUPPORTED 	eth_f_altera_std_synchronizer #(
	// NOT SUPPORTED 		.depth(INT_LEN)				
	// NOT SUPPORTED 	) synchronizer (
	// NOT SUPPORTED 		.clk		(clk),
	// NOT SUPPORTED 		.reset_n	(~reset),
	// NOT SUPPORTED 		.din		(sync_d_in),
	// NOT SUPPORTED 		.dout		(sync_q_out)
	// NOT SUPPORTED 	);
	// NOT SUPPORTED 	
	// NOT SUPPORTED 	//synthesis translate_off			
	// NOT SUPPORTED 	initial begin
	// NOT SUPPORTED 		synchronizer.dreg = {(INT_LEN-1){1'b0}};
	// NOT SUPPORTED 		synchronizer.din_s1 = 1'b0;
	// NOT SUPPORTED 	end
	// NOT SUPPORTED 	//synthesis translate_on
	// NOT SUPPORTED 			
	// NOT SUPPORTED end else begin
		refclk_recovery_altera_std_synchronizer_nocut #(
			.depth(INT_LEN)				
		) synchronizer_nocut (
			.clk		(clk),
			.reset_n	(~reset),
			.din		(sync_d_in),
			.dout		(sync_q_out)
		);
				
		//synthesis translate_off
		initial begin
			synchronizer_nocut.dreg = {(INT_LEN-1){1'b0}};
			synchronizer_nocut.din_s1 = 1'b0;
		end
		//synthesis translate_on	
	// NOT SUPPORTED end
	
    // Generate asynchronous capture circuit if specified.
    if(SLOW_CLOCK == 0) begin
      assign  d_in = d[ig];
    end else begin
      wire  d_clk;
      reg   d_r = L_INIT_VALUE;
      wire  clr_n;
 
      assign  d_clk = d[ig];
      assign  d_in  = d_r;
      assign  clr_n = ~q[ig] | d_clk; // Clear when output is logic 1 and input is logic 0
 
      // Asynchronously latch the input signal.
      always @(posedge d_clk or negedge clr_n)
        if(!clr_n)      d_r <= 1'b0;
        else if(d_clk)  d_r <= 1'b1;
    end // SLOW_CLOCK
  end // for loop
endgenerate
 
endmodule
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "KvY3cInsyNb25msnrLt7stVcYrfTnHuIYVMQ1737VTJbaDsQP/fg37OQ8BqprVvzIW+w5ltsgbh+KL63tK5E/46JxXrXQPHower03fou3CtkaHIVqFQFY/TqIUSi3JDAI38mKKcWurZB6Wc8ZG5hj06oOJlI+8DfIxJfa5D5HCsq1ylDZ6HHjPT51Z+H71nwlAGcdbIr1aaGa3JYsYd6B5rDj8hxcsdjszWqfgZ98/dZjSiyyMtdWqG9oPkXT9npPsantv9UyjUe0eOf2sYCK0L8GREs+4Q6kYPW+e3E9w6R9i8X/OPxU/FyIblyuC7D1Usex+F6IsPxAkvy4tpYO5oEzRsGRe6x7drGkwS29niC/kfyrPrYLlB6vTMWMtKwGRQdb5QPkD7ZCyRVDQMiOx3KRBpFZhULsqlMGchgVSpepXGFecFlQAcCzft829LB8YzALVtjhgykAzu/Vsx5L0nPZlzzb+VNVuT6Iysb+E67l+8+s1XCepvjjK0f6Cdvx8bdtu/ZOa5FOqnubwfw2XvrM/P5Mp8xD3GfT6xtpgpkxW27YF1rAhduVbTC5vGXnzPTZcyxZ0IR1G25996Xz+fn9uq+EW3azN3GOzs2C89chsO/lKSwRG4wXz+ZHxg8yvWMrQzqEioClKAJXGABVSCQjWDDNmUiyuSprmbLxQEnPUBc88V/0sgCGl+0QVfTLf3MvLEhBwkPu2L/6Vl1VPIaY4zqpjuMio0GOffgqYNbPTpSrOkdllDbmlx9549O5d9N8dPLQIbz0KNV0HpgA8RfNx1bCEb+yueO/Ld0/YQC1wJdd8akM14cLW0447FooUtvpR6FzT0+jMO/VtXaVR/fi0XvnRJyHu8SRB/7lAWM2+cprDCaL+igOCUEJfBSgViEDh2f+MZii/fUI/1ZoJFAgxbaBMXnRP5RpLugKIaFAI1m/ld4qggCl4YPcGaC9mGWqfKSAdFBB+re1IErmupv4CXkZ1/uQab9MrQV+xjrxVK9WDeLd5gHO8AZaji5"
`endif