// (C) 2001-2025 Altera Corporation. All rights reserved.
// Your use of Altera Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Altera Program License Subscription 
// Agreement, Altera IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Altera and sold by 
// Altera or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


// Clocked priority encoder with state
//
// On each clock cycle, updates state to show which request is granted.
// Most recent grant holder is always the highest priority.
// If current grant holder is not making a request, while others are, 
// then new grant holder is always the requester with lowest bit number.
// If no requests, current grant holder retains grant state
 
// $Header$
 
//altera message_off 16753
module refclk_recovery_arbiter #(
	parameter width = 2
) (
	input  wire clock,
	input  wire [width-1:0] req,	// req[n] requests for this cycle
	output reg  [width-1:0] grant	// grant[n] means requester n is grantee in this cycle
);
 
	wire idle;	// idle when no requests
	wire [width-1:0] keep;	// keep[n] means requester n is requesting, and already has the grant
							// Note: current grantee is always highest priority for next grant
	wire [width-1:0] take;	// take[n] means requester n is requesting, and there are no higher-priority requests
 
	assign keep = req & grant;	// current grantee is always highest priority for next grant
	assign idle = ~| req;		// idle when no requests
 
	initial begin
		grant = 0;
	end
 
	// grant next state depends on current grant and take priority
	always @(posedge clock) begin
		grant <= 
// synthesis translate_off
                    (grant === {width{1'bx}})? {width{1'b0}} :
// synthesis translate_on
				keep				// if current grantee is requesting, gets to keep grant
				 | ({width{idle}} & grant)	// if no requests, grant state remains unchanged
				 | take;			// take applies only if current grantee is not requesting
	end
 
	// 'take' bus encodes priority.  Request with lowest bit number wins when current grantee not requesting
	assign take[0] = req[0]
					 & (~| (keep & ({width{1'b1}} << 1)));	// no 'keep' from lower-priority inputs
	genvar i;
	generate
	for (i=1; i < width; i = i + 1) begin : arb
		assign take[i] = req[i]
						 & (~| (keep & ({width{1'b1}} << (i+1))))	// no 'keep' from lower-priority inputs
						 & (~| (req & {i{1'b1}}));	// no 'req' from higher-priority inputs
	end
	endgenerate
endmodule
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "KvY3cInsyNb25msnrLt7stVcYrfTnHuIYVMQ1737VTJbaDsQP/fg37OQ8BqprVvzIW+w5ltsgbh+KL63tK5E/46JxXrXQPHower03fou3CtkaHIVqFQFY/TqIUSi3JDAI38mKKcWurZB6Wc8ZG5hj06oOJlI+8DfIxJfa5D5HCsq1ylDZ6HHjPT51Z+H71nwlAGcdbIr1aaGa3JYsYd6B5rDj8hxcsdjszWqfgZ98/foLUPXGvalN+lsFcU7tXzkfUKrddtKbfMSYZIM2aJkJegzkXy+W/KJAkkV2fsINs7x5WjBKBwkdUjK1Ntyx2lopZpCHIA5Dv0rCMVZs+l7To9552PY0IEAMbfPFwSKtzVbP9uoRVzg6H/7LfTFi/2sDoF7j+3IjXD3HyPcuFGIz0TccWHlcA/Y7mwbTcX+DrxP/U9iVUIYT9+OizLvNx7/aa+GswsioJlOmDiWhYH8kMMqDUnL6gPuEccXgqcDDk1rC0s88v3T6hu2cohA4WGBrsvzJO8V2rHN42uIfV/A73DeRIyvWL/i9DT1XzEfNAdBfXFNxcNDGPK8n0PNpm1hxOcj81816g6sMFgxjp7mY+FYBjwltzh+Irn5TE/1/t7lURRRN7PHRe8nXWJ150pgaKAFBziXDKve14Kiw1JT8F7qzABqXd276TMYiiek9yuJtExT4MlEnwNwu2AI1KoxPd3L+1pbZ4LsJVsq3PxV3E+UCYhRK2FHcfFY71BJjG/w1qj30Tb7YU8RuFFBM5czchEelUgv6OLU5Ip13gJ295ipmWJekfhP02c+Zr8+/6rXISDByx+J6ingvIMHO69Y+SZLnUsTtpU4fnA+fjvD0VHqBxI4icRpU//R2Xt4PwKidKPQefKxEzHI+FecW9r1GMJPgs4nAK8YBPtcF2ZXmrdyPDuxxenZctWG6z68+IFKXFrchCOeTc1hJYIFtYikveKPaUu+JSAYW+cVaEJoR0WsnWY+4SQ/qhIHDJzckHBxSs4b0ovcO/Ez2CJBVubK"
`endif