//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		Video Timing Generator core.
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Jan. 16 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT		: Bit Width of color.
//		PLANE		: Color Plane.
//		VSI_FMIN	: Minimum frequency of VSI Clock.
//		VIDEO_FMAX	: Maximum frequency of Video I/F Clock.
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk 				: Global clock.
//		rst_n				: Async reset.
//		srst				: Sync reset.
//
//		frame_rst_o 		: Frame reset output
//		enable				: Enable of circuit.
//		param_update_i		: Mode setting signal
//
//		interlace_mode_i	: Scan mode select.
//							  1:Interlace 0:Progressive
//
//		ex_sync_pol_i		: External synchronization polarity.
//		ex_odd_field_pol_i	: Odd field polarity.
//		h_sync_pol_i		: H Sync polarity.(hsync_i)
//		v_sync_pol_i		: V Sync polarity.(vsync_i)
//		odd_field_pol_i 	: Odd field polarity.(field_i)
//		h_valid_pol_i		: H Valid polarity.(hvalid_i)
//		v_valid_pol_i		: V Valid polarity.(vvalid_i)
//		de_pol_i			: DE polarity.(de_i)
//
//		h_front_i			: H front porch.
//		h_interval_i		: H sync interval.
//		h_back_i			: H back porch.
//		h_video_i			: H active video.
//		v_front_i			: V front porch.
//		v_interval_i		: V sync interval.
//		v_back_i			: V back porch.
//		v_video_i			: V active video.
//		v_delay_i			: V sync delay.
//		even_v_front_i		: V front porch for even field.
//		even_v_interval_i	: V sync interval for even field.
//		even_v_back_i		: V back porch for even field.
//		even_v_video_i		: V active video for even field.
//		even_v_delay_i		: V sync delay for even field.
//		even2odd_v_delay_i	: Vertical change point of field startus signal.(even to odd field)
//		even2odd_h_delay_i	: Horizontal change point of field startus signal.(even to odd field)
//		odd2even_v_delay_i	: Vertical change point of field startus signal.(odd to even field)
//		odd2even_h_delay_i	: Horizontal change point of field startus signal.(odd to even field)
//		reset_timing_i		: Frame reset timing.(Frame unit)
//		reset_interval_i	: Frame reset interval.
//
//		const_data_out_i	: Constant data output.
//		const_out_mode_i	: Mode select for constant data output.
//							  1:const_data0_i & const_data1_i 0:const_data0_i
//		const_data0_i		: Constant data 0.
//		const_data1_i		: Constant data 1.
//
//		external_sync_i 	: External synchronization signal.
//		external_field_i	: Field status of external synchronization signal.
//
//		req_field_o 		: Request for next field status.
//		pixel_rdreq_o		: Read request.
//		pixel_oe_o			: Output enabe.
//		pixel_data_i		: Read pixel data.
//
//		act_video_o			: Active video.
//
//		h_sync_o			: H Sync signal.
//		v_sync_o			: V Sync signal.
//		field_o 			: Field status signal.
//		de_o				: DE signal.
//		h_valid_o			: H Valid signal.
//		v_valid_o			: V Valid signal.
//		pixel_o				: Pixel data signal.
//----------------------------------------------------------------------------------------
`timescale 1ps / 1ps
`default_nettype none

module	vtg11_timing_gen
	#(
		parameter	Q_BIT				= 10			,
		parameter	PLANE				= 2				,
		parameter	VSI_FMIN			= 100			,
		parameter	VIDEO_FMAX			= 148			,

		parameter	H_FRONT 			= 2				,
		parameter	H_INTERVAL			= 2				,
		parameter	H_BACK				= 2				,
		parameter	H_VIDEO 			= 2				,
		parameter	V_FRONT 			= 2				,
		parameter	V_INTERVAL			= 2				,
		parameter	V_BACK				= 2				,
		parameter	V_VIDEO 			= 2				,
		parameter	V_DELAY 			= 2				,
		parameter	EVEN_V_FRONT		= 2				,
		parameter	EVEN_V_INTERVAL 	= 2				,
		parameter	EVEN_V_BACK 		= 2				,
		parameter	EVEN_V_VIDEO		= 2				,
		parameter	EVEN_V_DELAY		= 2				,
		parameter	EVEN2ODD_V_DELAY	= 2				,
		parameter	EVEN2ODD_H_DELAY	= 2				,
		parameter	ODD2EVEN_V_DELAY	= 2				,
		parameter	ODD2EVEN_H_DELAY	= 2				,
		parameter	RESET_TIMING		= 1				,
		parameter	PIXEL_BIT			= Q_BIT * PLANE	,
		parameter	H_F_P				= 9				,		//	bit_width h_front_i
		parameter	H_S_I				= 8				,		//	bit_width h_interval_i
		parameter	H_B_P				= 9				,		//	bit_width h_back_i
		parameter	H_V_I				= 12			,		//	bit_width h_video_i
		parameter	V_F_P				= 6				,		//	bit_width v_front_i & even_v_front_i
		parameter	V_S_I				= 4				,		//	bit_width v_interval_i & even_v_interval_i
		parameter	V_B_P				= 7				,		//	bit_width v_back_i & even_v_back_i
		parameter	V_V_I				= 11			,		//	bit_width v_video_i & even_v_video_i
		parameter	H_COUNTER			= 12			,
		parameter	V_COUNTER			= 12
	)
	(
		input	wire							clk 				,
		input	wire							rst_n				,
		input	wire							srst				,

		output	wire							frame_rst_o 		,
		input	wire							enable				,
		input	wire							param_update_i		,

		input	wire							interlace_mode_i	,

		input	wire							ex_sync_pol_i		,
		input	wire							ex_odd_field_pol_i	,
		input	wire							h_sync_pol_i		,
		input	wire							v_sync_pol_i		,
		input	wire							odd_field_pol_i 	,
		input	wire							h_valid_pol_i		,
		input	wire							v_valid_pol_i		,
		input	wire							de_pol_i			,

		input	wire	[	  H_F_P-1 : 0 ] 	h_front_i			,
		input	wire	[	  H_S_I-1 : 0 ] 	h_interval_i		,
		input	wire	[	  H_B_P-1 : 0 ] 	h_back_i			,
		input	wire	[	  H_V_I-1 : 0 ] 	h_video_i			,
		input	wire	[	  V_F_P-1 : 0 ] 	v_front_i			,
		input	wire	[	  V_S_I-1 : 0 ] 	v_interval_i		,
		input	wire	[	  V_B_P-1 : 0 ] 	v_back_i			,
		input	wire	[	  V_V_I-1 : 0 ] 	v_video_i			,
		input	wire	[ H_COUNTER-1 : 0 ] 	v_delay_i			,
		input	wire	[	  V_F_P-1 : 0 ] 	even_v_front_i		,
		input	wire	[	  V_S_I-1 : 0 ] 	even_v_interval_i	,
		input	wire	[	  V_B_P-1 : 0 ] 	even_v_back_i		,
		input	wire	[	  V_V_I-1 : 0 ] 	even_v_video_i		,
		input	wire	[ H_COUNTER-1 : 0 ] 	even_v_delay_i		,
		input	wire	[ V_COUNTER-1 : 0 ] 	even2odd_v_delay_i	,
		input	wire	[ H_COUNTER-1 : 0 ] 	even2odd_h_delay_i	,
		input	wire	[ V_COUNTER-1 : 0 ] 	odd2even_v_delay_i	,
		input	wire	[ H_COUNTER-1 : 0 ] 	odd2even_h_delay_i	,
		input	wire	[ V_COUNTER-1 : 0 ] 	reset_timing_i		,
		input	wire	[			3 : 0 ] 	reset_interval_i	,

		input	wire							const_data_out_i	,
		input	wire							const_out_mode_i	,
		input	wire	[ PIXEL_BIT-1 : 0 ]		const_data0_i		,
		input	wire	[ PIXEL_BIT-1 : 0 ] 	const_data1_i		,

		input	wire							external_sync_i 	,
		input	wire							external_field_i	,

		output	wire							req_field_o 		,
		output	wire							pixel_rdreq_o		,
		output	wire							pixel_oe_o			,
		input	wire	[ PIXEL_BIT-1 : 0 ] 	pixel_data_i		,
		output	wire							act_video_o			,
		output	wire							h_sync_o			,
		output	wire							v_sync_o			,
		output	wire							field_o 			,
		output	wire							de_o				,
		output	wire							h_valid_o			,
		output	wire							v_valid_o			,
		output	wire	[ PIXEL_BIT-1 : 0 ] 	pixel_o
	);

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------
	localparam	RESET_INTERVAL		= ( VIDEO_FMAX / VSI_FMIN ) + 2 ;

	localparam	H_FRONT_END			= H_FRONT - 1 ;
	localparam	H_SYNC_END			= H_FRONT + H_INTERVAL - 1 ;
	localparam	H_BACK_END			= H_FRONT + H_INTERVAL + H_BACK - 1 ;
	localparam	H_VIDEO_END_N		= H_FRONT + H_INTERVAL + H_BACK + H_VIDEO - 2 ;
	localparam	H_VIDEO_END			= H_FRONT + H_INTERVAL + H_BACK + H_VIDEO - 1 ;
	localparam	V_FRONT_END			= V_FRONT - 1 ;
	localparam	V_FRONT_END_P		= V_FRONT ;
	localparam	V_SYNC_END			= V_FRONT + V_INTERVAL - 1 ;
	localparam	V_SYNC_END_P		= V_FRONT + V_INTERVAL ;
	localparam	V_BACK_END			= V_FRONT + V_INTERVAL + V_BACK - 1 ;
	localparam	V_VIDEO_END			= V_FRONT + V_INTERVAL + V_BACK + V_VIDEO - 1 ;
	localparam	EV_V_FRONT_END		= EVEN_V_FRONT - 1 ;
	localparam	EV_V_FRONT_END_P	= EVEN_V_FRONT ;
	localparam	EV_V_SYNC_END		= EVEN_V_FRONT + EVEN_V_INTERVAL - 1 ;
	localparam	EV_V_SYNC_END_P		= EVEN_V_FRONT + EVEN_V_INTERVAL ;
	localparam	EV_V_BACK_END		= EVEN_V_FRONT + EVEN_V_INTERVAL + EVEN_V_BACK - 1 ;
	localparam	EV_V_VIDEO_END		= EVEN_V_FRONT + EVEN_V_INTERVAL + EVEN_V_BACK + EVEN_V_VIDEO - 1 ;
	localparam	DATA_RD_START		= H_FRONT + H_INTERVAL + H_BACK - 2 ;
	localparam	DATA_RD_END			= H_FRONT + H_INTERVAL + H_BACK + H_VIDEO - 2 ;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	//	output register
	reg 							h_sync_o_ff 				;
	reg 							v_sync_o_ff 				;
	reg 							de_o_ff 					;
	reg 							h_valid_o_ff				;
	reg 							v_valid_o_ff				;
	reg 							field_o_ff					;
	reg 							frame_rst_o_ff				;
	reg								act_video_o_ff				;
	reg 	[	PIXEL_BIT-1 : 0 ]	pixel_o_ff					;

	//	parameter
	reg 							interlace_mode_ff			;
	reg 							ex_sync_mode_ff 			;

	reg 							ex_sync_pol_ff				;
	reg 							ex_odd_field_pol_ff 		;
	reg 							h_sync_pol_ff				;
	reg 							v_sync_pol_ff				;
	reg 							odd_field_pol_ff			;
	reg 							h_valid_pol_ff				;
	reg 							v_valid_pol_ff				;
	reg 							de_pol_ff					;

	reg 	[	H_COUNTER-1 : 0 ]	h_front_end_ff				;
	reg 	[	H_COUNTER-1 : 0 ]	h_sync_end_ff				;
	reg 	[	H_COUNTER-1 : 0 ]	h_back_end_ff				;
	reg 	[	H_COUNTER-1 : 0 ]	h_video_end_n_ff			;
	reg 	[	H_COUNTER-1 : 0 ]	h_video_end_ff				;
	reg 	[	V_COUNTER-1 : 0 ]	v_front_end_ff				;
	reg 	[	V_COUNTER-1 : 0 ]	v_front_end_p_ff			;
	reg 	[	V_COUNTER-1 : 0 ]	v_sync_end_ff				;
	reg 	[	V_COUNTER-1 : 0 ]	v_sync_end_p_ff 			;
	reg 	[	V_COUNTER-1 : 0 ]	v_back_end_ff				;
	reg 	[	V_COUNTER-1 : 0 ]	v_video_end_ff				;
	reg 	[	H_COUNTER-1 : 0 ]	v_delay_ff					;
	reg 	[	V_COUNTER-1 : 0 ]	even_v_front_end_ff 		;
	reg 	[	V_COUNTER-1 : 0 ]	even_v_front_end_p_ff		;
	reg 	[	V_COUNTER-1 : 0 ]	even_v_sync_end_ff			;
	reg 	[	V_COUNTER-1 : 0 ]	even_v_sync_end_p_ff		;
	reg 	[	V_COUNTER-1 : 0 ]	even_v_back_end_ff			;
	reg 	[	V_COUNTER-1 : 0 ]	even_v_video_end_ff 		;
	reg 	[	H_COUNTER-1 : 0 ]	even_v_delay_ff 			;
	reg 	[	V_COUNTER-1 : 0 ]	even2odd_v_delay_ff 		;
	reg 	[	V_COUNTER-1 : 0 ]	even2odd_v_delay_n_ff 		;
	reg 	[	H_COUNTER-1 : 0 ]	even2odd_h_delay_ff 		;
	reg 	[	V_COUNTER-1 : 0 ]	odd2even_v_delay_ff 		;
	reg 	[	V_COUNTER-1 : 0 ]	odd2even_v_delay_n_ff 		;
	reg 	[	H_COUNTER-1 : 0 ]	odd2even_h_delay_ff 		;
	reg 	[	H_COUNTER-1 : 0 ]	data_rd_start_ff			;
	reg 	[	H_COUNTER-1 : 0 ]	data_rd_end_ff				;
	reg 	[	V_COUNTER-1 : 0 ]	reset_timing_ff 			;
	reg 	[			  3 : 0 ]	reset_interval_ff			;
	reg								even2odd_v_delay_zero_ff	;
	reg								even2odd_h_delay_zero_ff	;
	reg								odd2even_v_delay_zero_ff	;
	reg								odd2even_h_delay_zero_ff	;

	reg 							v_delay_zero_ff 			;
	reg 							even_v_delay_zero_ff		;

	reg 							interlace_mode_lat_ff		;
	reg 							ex_sync_mode_lat_ff 		;

	reg 							ex_sync_pol_lat_ff			;
	reg 							ex_odd_field_pol_lat_ff 	;
	reg 							h_sync_pol_lat_ff			;
	reg 							v_sync_pol_lat_ff			;
	reg 							odd_field_pol_lat_ff		;
	reg 							h_valid_pol_lat_ff			;
	reg 							v_valid_pol_lat_ff			;
	reg 							de_pol_lat_ff				;
	reg 	[		H_F_P-1 : 0 ]	h_front_lat_ff				;
	reg 	[		H_S_I-1 : 0 ]	h_interval_lat_ff			;
	reg 	[		H_B_P-1 : 0 ]	h_back_lat_ff				;
	reg 	[		H_V_I-1 : 0 ]	h_video_lat_ff				;
	reg 	[		V_F_P-1 : 0 ]	v_front_lat_ff				;
	reg 	[		V_S_I-1 : 0 ]	v_interval_lat_ff			;
	reg 	[		V_B_P-1 : 0 ]	v_back_lat_ff				;
	reg 	[		V_V_I-1 : 0 ]	v_video_lat_ff				;
	reg 	[	H_COUNTER-1 : 0 ]	v_delay_lat_ff				;
	reg 	[		V_F_P-1 : 0 ]	even_v_front_lat_ff 		;
	reg 	[		V_S_I-1 : 0 ]	even_v_interval_lat_ff		;
	reg 	[		V_B_P-1 : 0 ]	even_v_back_lat_ff			;
	reg 	[		V_V_I-1 : 0 ]	even_v_video_lat_ff 		;
	reg 	[	H_COUNTER-1 : 0 ]	even_v_delay_lat_ff 		;
	reg 	[	V_COUNTER-1 : 0 ]	even2odd_v_delay_lat_ff 	;
	reg 	[	H_COUNTER-1 : 0 ]	even2odd_h_delay_lat_ff 	;
	reg 	[	V_COUNTER-1 : 0 ]	odd2even_v_delay_lat_ff 	;
	reg 	[	H_COUNTER-1 : 0 ]	odd2even_h_delay_lat_ff 	;
	reg 	[	V_COUNTER-1 : 0 ]	reset_timing_lat_ff 		;
	reg 	[			  3 : 0 ]	reset_interval_lat_ff		;

	wire	[	V_COUNTER-1 : 0 ]	even2odd_v_delay_n			;
	wire	[	H_COUNTER-1 : 0 ]	even2odd_h_delay_n			;
	wire	[	V_COUNTER-1 : 0 ]	odd2even_v_delay_n			;
	wire	[	H_COUNTER-1 : 0 ]	odd2even_h_delay_n			;

	reg		[		H_F_P	: 0 ]	h_f_add_i_ff				;
	reg		[	H_COUNTER-1 : 0 ]	h_b_add_v_ff				;
	reg		[		V_F_P	: 0 ]	v_f_add_i_ff				;
	reg		[	V_COUNTER-1 : 0 ]	v_b_add_v_ff				;
	reg		[		V_F_P	: 0 ]	ev_f_add_i_ff				;
	reg		[	V_COUNTER-1 : 0 ]	ev_b_add_v_ff				;
	reg								even2odd_vdelay_lat_z_ff	;
	reg								even2odd_hdelay_lat_z_ff	;
	reg								odd2even_vdelay_lat_z_ff	;
	reg								odd2even_hdelay_lat_z_ff	;

	//	set timing
	reg								param_update_lat_ff			;
	reg								param_update_lat1_ff		;
	reg								param_update_lat2_ff		;
	reg								param_set_ff				;
	reg								param_set_str_ff			;
	reg 							param_update_wait_ff		;
	wire							param_update_timing 		;

	//	internal sync signal
	reg 							h_sync_ff					;
	reg 							h_valid_ff					;
	reg 							v_sync_ff					;
	reg 							v_valid_ff					;
	reg 							field_ff					;

	reg 							field_sel_ff				;
	reg 							field_sel_str_ff			;

	reg 							external_sync_lat_ff		;
	reg 							external_sync_lat1_ff		;
	reg 							external_sync_lat2_ff		;
	reg 							external_field_ff			;
	wire							external_sync_pos			;
	reg 							ex_sync_pos_str_ff			;

	//	pixel data read
	reg 							pixel_rdreq_ff				;
	reg 							pixel_oe_ff 				;

	reg 							const_data_out_ff			;
	reg 							const_out_mode_ff			;
	reg 	[	PIXEL_BIT-1 : 0 ]	const_data0_ff				;
	reg 	[	PIXEL_BIT-1 : 0 ]	const_data1_ff				;
	reg 							const_data_sel_ff			;

	//	frame reset
	reg 							frame_rst_ff				;

	//	timing signal
	reg 							h_front_end_pls_ff			;
	reg 							h_sync_end_pls_ff			;
	reg 							h_back_end_pls_ff			;
	reg								h_video_end_n_pls_ff		;
	reg 							h_video_end_pls_ff			;
	reg 							v_front_end_pls_ff			;
	reg 							v_front_end_p_pls_ff		;
	reg 							v_sync_end_pls_ff			;
	reg 							v_sync_end_p_pls_ff 		;
	reg 							v_back_end_pls_ff			;
	reg 							v_video_end_pls_ff			;
	reg 							v_delay_pls_ff				;
	reg 							even_v_front_end_pls_ff 	;
	reg 							even_v_front_end_p_pls_ff	;
	reg 							even_v_sync_end_pls_ff		;
	reg 							even_v_sync_end_p_pls_ff	;
	reg 							even_v_back_end_pls_ff		;
	reg 							even_v_video_end_pls_ff 	;
	reg 							even_v_delay_pls_ff 		;
	reg 							even2odd_v_delay_pls_ff 	;
	reg 							even2odd_v_delay_n_pls_ff 	;
	reg 							even2odd_h_delay_pls_ff 	;
	reg 							odd2even_v_delay_pls_ff 	;
	reg								odd2even_v_delay_n_pls_ff	;
	reg 							odd2even_h_delay_pls_ff 	;
	reg 							data_rd_start_pls_ff		;
	reg 							data_rd_end_pls_ff			;

	//	pixel counter
	reg 	[	H_COUNTER-1 : 0 ]	h_count_ff					;
	wire							h_count_clr 				;
	reg 	[	V_COUNTER-1 : 0 ]	v_count_ff					;
	wire							v_count_clr 				;

	//	reset counter
	reg 	[			  3 : 0 ]	rst_count_ff				;
	wire							frame_rst_start 			;
	wire							frame_rst_tim				;
	reg 							frame_rst_tim_str_ff		;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================
	//===============//
	// output signal //
	//===============//
	// assign frame_rst_o = frame_rst ;

	assign	req_field_o		= field_sel_ff		;
	assign	pixel_rdreq_o	= pixel_rdreq_ff	;
	assign	pixel_oe_o		= pixel_oe_ff		;
	assign	h_sync_o		= h_sync_o_ff		;
	assign	v_sync_o		= v_sync_o_ff		;
	assign	de_o			= de_o_ff			;
	assign	pixel_o 		= pixel_o_ff		;
	assign	h_valid_o		= h_valid_o_ff		;
	assign	v_valid_o		= v_valid_o_ff		;
	assign	field_o 		= field_o_ff		;
	assign	frame_rst_o 	= frame_rst_o_ff	;
	assign	act_video_o		= act_video_o_ff	;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			h_sync_o_ff		<= 1'b0 ;
			v_sync_o_ff		<= 1'b0 ;
			de_o_ff			<= 1'b0 ;
			h_valid_o_ff	<= 1'b0 ;
			v_valid_o_ff	<= 1'b0 ;
			field_o_ff		<= 1'b0 ;
			frame_rst_o_ff	<= 1'b0 ;
			act_video_o_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			h_sync_o_ff		<= 1'b0 ;
			v_sync_o_ff		<= 1'b0 ;
			de_o_ff			<= 1'b0 ;
			h_valid_o_ff	<= 1'b0 ;
			v_valid_o_ff	<= 1'b0 ;
			field_o_ff		<= 1'b0 ;
			frame_rst_o_ff	<= 1'b0 ;
			act_video_o_ff	<= 1'b0 ;
		end
		else begin
			h_sync_o_ff		<= ( h_sync_pol_ff		) ?	h_sync_ff					: ~h_sync_ff ;
			v_sync_o_ff		<= ( v_sync_pol_ff		) ?	v_sync_ff					: ~v_sync_ff ;
			de_o_ff			<= ( de_pol_ff			) ?	( h_valid_ff & v_valid_ff )	: ~( h_valid_ff & v_valid_ff ) ;
			h_valid_o_ff	<= ( h_valid_pol_ff		) ?	h_valid_ff					: ~h_valid_ff ;
			v_valid_o_ff	<= ( v_valid_pol_ff		) ?	v_valid_ff					: ~v_valid_ff ;
			field_o_ff		<= ( !interlace_mode_ff	) ?	1'b0 :
							   (  odd_field_pol_ff	) ?	field_ff :
														~field_ff ;
			frame_rst_o_ff	<= frame_rst_ff ;
			act_video_o_ff	<= v_valid_ff ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			pixel_o_ff	<= { PIXEL_BIT{1'b0} } ;
		end
		else if ( srst ) begin
			pixel_o_ff	<= { PIXEL_BIT{1'b0} } ;
		end
		else if ( h_valid_ff && v_valid_ff ) begin
			if ( const_data_out_ff ) begin
				if ( const_out_mode_ff && const_data_sel_ff ) begin
					pixel_o_ff	<= const_data1_ff ;
				end
				else begin
					pixel_o_ff	<= const_data0_ff ;
				end
			end
			else begin
				pixel_o_ff	<= pixel_data_i ;
			end
		end
		else begin
			pixel_o_ff	<= { PIXEL_BIT{1'b0} } ;
		end
	end

	//===============//
	// parameter set //
	//===============//

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			interlace_mode_ff			<= 1'b0 ;
			ex_sync_mode_ff 			<= 1'b0 ;
			ex_sync_pol_ff				<= 1'b0 ;
			ex_odd_field_pol_ff			<= 1'b0 ;
			h_sync_pol_ff				<= 1'b0 ;
			v_sync_pol_ff				<= 1'b0 ;
			odd_field_pol_ff			<= 1'b0 ;
			h_valid_pol_ff				<= 1'b0 ;
			v_valid_pol_ff				<= 1'b0 ;
			de_pol_ff					<= 1'b0 ;
			h_front_end_ff				<= H_FRONT_END[ H_COUNTER-1 : 0 ] ;
			h_sync_end_ff				<= H_SYNC_END[ H_COUNTER-1 : 0 ] ;
			h_back_end_ff				<= H_BACK_END[ H_COUNTER-1 : 0 ] ;
			h_video_end_n_ff			<= H_VIDEO_END_N[ H_COUNTER-1 : 0 ] ;
			h_video_end_ff				<= H_VIDEO_END[ H_COUNTER-1 : 0 ] ;
			v_front_end_ff				<= V_FRONT_END[ V_COUNTER-1 : 0 ] ;
			v_front_end_p_ff			<= V_FRONT_END_P[ V_COUNTER-1 : 0 ] ;
			v_sync_end_ff				<= V_SYNC_END[ V_COUNTER-1 : 0 ] ;
			v_sync_end_p_ff 			<= V_SYNC_END_P[ V_COUNTER-1 : 0 ] ;
			v_back_end_ff				<= V_BACK_END[ V_COUNTER-1 : 0 ] ;
			v_video_end_ff				<= V_VIDEO_END[ V_COUNTER-1 : 0 ] ;
			v_delay_ff					<= V_DELAY[ H_COUNTER-1 : 0 ] ;
			even_v_front_end_ff 		<= EV_V_FRONT_END[ V_COUNTER-1 : 0 ] ;
			even_v_front_end_p_ff		<= EV_V_FRONT_END_P[ V_COUNTER-1 : 0 ] ;
			even_v_sync_end_ff			<= EV_V_SYNC_END[ V_COUNTER-1 : 0 ] ;
			even_v_sync_end_p_ff		<= EV_V_SYNC_END_P[ V_COUNTER-1 : 0 ] ;
			even_v_back_end_ff			<= EV_V_BACK_END[ V_COUNTER-1 : 0 ] ;
			even_v_video_end_ff 		<= EV_V_VIDEO_END[ V_COUNTER-1 : 0 ] ;
			even_v_delay_ff 			<= EVEN_V_DELAY[ H_COUNTER-1 : 0 ] ;
			even2odd_v_delay_ff 		<= EVEN2ODD_V_DELAY[ V_COUNTER-1 : 0 ] ;
			even2odd_v_delay_n_ff 		<= EVEN2ODD_V_DELAY[ V_COUNTER-1 : 0 ] - 1'b1 ;
			even2odd_h_delay_ff 		<= EVEN2ODD_H_DELAY[ H_COUNTER-1 : 0 ] ;
			odd2even_v_delay_ff 		<= ODD2EVEN_V_DELAY[ V_COUNTER-1 : 0 ] ;
			odd2even_v_delay_n_ff		<= ODD2EVEN_V_DELAY[ V_COUNTER-1 : 0 ] - 1'b1 ;
			odd2even_h_delay_ff 		<= ODD2EVEN_H_DELAY[ H_COUNTER-1 : 0 ] ;
			data_rd_start_ff			<= DATA_RD_START[ H_COUNTER-1 : 0 ] ;
			data_rd_end_ff				<= DATA_RD_END[ H_COUNTER-1 : 0 ] ;
			reset_timing_ff 			<= RESET_TIMING[ V_COUNTER-1 : 0 ] - 1'b1 ;
			reset_interval_ff			<= RESET_INTERVAL[ 3 : 0 ]		 ;
			even2odd_v_delay_zero_ff	<= 1'b0 ;
			even2odd_h_delay_zero_ff	<= 1'b0 ;
			odd2even_v_delay_zero_ff	<= 1'b0 ;
			odd2even_h_delay_zero_ff	<= 1'b0 ;
		end
		else if ( srst || param_update_timing ) begin
			interlace_mode_ff			<= interlace_mode_lat_ff ;
			ex_sync_mode_ff 			<= ex_sync_mode_lat_ff ;
			ex_sync_pol_ff				<= ex_sync_pol_lat_ff ;
			ex_odd_field_pol_ff 		<= ex_odd_field_pol_lat_ff ;
			h_sync_pol_ff				<= h_sync_pol_lat_ff ;
			v_sync_pol_ff				<= v_sync_pol_lat_ff ;
			odd_field_pol_ff			<= odd_field_pol_lat_ff ;
			h_valid_pol_ff				<= h_valid_pol_lat_ff ;
			v_valid_pol_ff				<= v_valid_pol_lat_ff ;
			de_pol_ff					<= de_pol_lat_ff ;
			h_front_end_ff				<= h_front_lat_ff - 1'b1 ;
			h_sync_end_ff				<= h_f_add_i_ff - 1'b1 ;
			h_back_end_ff				<= h_f_add_i_ff + h_back_lat_ff - 1'b1 ;
			h_video_end_n_ff			<= h_f_add_i_ff + h_b_add_v_ff - 2'b10 ;
			h_video_end_ff				<= h_f_add_i_ff + h_b_add_v_ff - 1'b1 ;
			v_front_end_ff				<= v_front_lat_ff - 1'b1 ;
			v_front_end_p_ff			<= v_front_lat_ff ;
			v_sync_end_ff				<= v_f_add_i_ff - 1'b1 ;
			v_sync_end_p_ff 			<= v_f_add_i_ff ;
			v_back_end_ff				<= v_f_add_i_ff + v_back_lat_ff - 1'b1 ;
			v_video_end_ff				<= v_f_add_i_ff + v_b_add_v_ff - 1'b1 ;
			v_delay_ff					<= v_delay_lat_ff ;
			even_v_front_end_ff 		<= even_v_front_lat_ff - 1'b1 ;
			even_v_front_end_p_ff		<= even_v_front_lat_ff ;
			even_v_sync_end_ff			<= ev_f_add_i_ff - 1'b1 ;
			even_v_sync_end_p_ff		<= ev_f_add_i_ff ;
			even_v_back_end_ff			<= ev_f_add_i_ff + even_v_back_lat_ff - 1'b1 ;
			even_v_video_end_ff 		<= ev_f_add_i_ff + ev_b_add_v_ff - 1'b1 ;
			even_v_delay_ff 			<= even_v_delay_lat_ff ;
			even2odd_v_delay_ff 		<= even2odd_v_delay_lat_ff;
			even2odd_v_delay_n_ff		<= even2odd_v_delay_n ;
			even2odd_h_delay_ff 		<= even2odd_h_delay_n ;
			odd2even_v_delay_ff 		<= odd2even_v_delay_lat_ff;
			odd2even_v_delay_n_ff		<= odd2even_v_delay_n ;
			odd2even_h_delay_ff 		<= odd2even_h_delay_n ;
			data_rd_start_ff			<= h_f_add_i_ff + h_back_lat_ff - 2'b10 ;
			data_rd_end_ff				<= h_f_add_i_ff + h_b_add_v_ff - 2'b10 ;
			reset_timing_ff 			<= reset_timing_lat_ff - 1'b1 ;
			reset_interval_ff			<= reset_interval_lat_ff ;
			even2odd_v_delay_zero_ff	<= even2odd_vdelay_lat_z_ff ;
			even2odd_h_delay_zero_ff	<= even2odd_hdelay_lat_z_ff ;
			odd2even_v_delay_zero_ff	<= odd2even_vdelay_lat_z_ff ;
			odd2even_h_delay_zero_ff	<= odd2even_hdelay_lat_z_ff ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			v_delay_zero_ff 		<= 1'b0 ;
			even_v_delay_zero_ff	<= 1'b0 ;
		end
		else begin
			v_delay_zero_ff 		<= ( v_delay_ff == {H_COUNTER{1'b0}} ) ? 1'b1 :1'b0 ;
			even_v_delay_zero_ff	<= ( even_v_delay_ff == {H_COUNTER{1'b0}} ) ? 1'b1 :1'b0 ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			interlace_mode_lat_ff	<= 1'b0 ;
			ex_sync_mode_lat_ff 	<= 1'b0 ;
			ex_sync_pol_lat_ff		<= 1'b0 ;
			ex_odd_field_pol_lat_ff <= 1'b0 ;
			h_sync_pol_lat_ff		<= 1'b0 ;
			v_sync_pol_lat_ff		<= 1'b0 ;
			odd_field_pol_lat_ff	<= 1'b0 ;
			h_valid_pol_lat_ff		<= 1'b0 ;
			v_valid_pol_lat_ff		<= 1'b0 ;
			de_pol_lat_ff			<= 1'b0 ;
			h_front_lat_ff			<= H_FRONT[ H_F_P-1 : 0 ] ;
			h_interval_lat_ff		<= H_INTERVAL[ H_S_I-1 : 0 ] ;
			h_back_lat_ff			<= H_BACK[ H_B_P-1 : 0 ] ;
			h_video_lat_ff			<= H_VIDEO[ H_V_I-1 : 0 ] ;
			v_front_lat_ff			<= V_FRONT[ V_F_P-1 : 0 ] ;
			v_interval_lat_ff		<= V_INTERVAL[ V_S_I-1 : 0 ] ;
			v_back_lat_ff			<= V_BACK[ V_B_P-1 : 0 ] ;
			v_video_lat_ff			<= V_VIDEO[ V_V_I-1 : 0 ] ;
			v_delay_lat_ff			<= V_DELAY[ H_COUNTER-1 : 0 ] ;
			even_v_front_lat_ff 	<= EVEN_V_FRONT[ V_F_P-1 : 0 ] ;
			even_v_interval_lat_ff	<= EVEN_V_INTERVAL[ V_S_I-1 : 0 ] ;
			even_v_back_lat_ff		<= EVEN_V_BACK[ V_B_P-1 : 0 ] ;
			even_v_video_lat_ff 	<= EVEN_V_VIDEO[ V_V_I-1 : 0 ] ;
			even_v_delay_lat_ff 	<= EVEN_V_DELAY[ H_COUNTER-1 : 0 ] ;
			even2odd_v_delay_lat_ff <= EVEN2ODD_V_DELAY[ V_COUNTER-1 : 0 ] ;
			even2odd_h_delay_lat_ff <= EVEN2ODD_H_DELAY[ H_COUNTER-1 : 0 ] ;
			odd2even_v_delay_lat_ff <= ODD2EVEN_V_DELAY[ V_COUNTER-1 : 0 ] ;
			odd2even_h_delay_lat_ff <= ODD2EVEN_H_DELAY[ H_COUNTER-1 : 0 ] ;
			reset_timing_lat_ff 	<= RESET_TIMING[ V_COUNTER-1 : 0 ] ;
			reset_interval_lat_ff	<= RESET_INTERVAL[ 3 : 0 ] ;
		end
		else if ( srst || param_set_ff ) begin
			interlace_mode_lat_ff	<= interlace_mode_i 	;
			ex_sync_mode_lat_ff 	<= 1'b0 	  			;
//			  ex_sync_mode_lat_ff	  <= ex_sync_mode_i 	  ;
			ex_sync_pol_lat_ff		<= ex_sync_pol_i		;
			ex_odd_field_pol_lat_ff <= ex_odd_field_pol_i	;
			h_sync_pol_lat_ff		<= h_sync_pol_i 		;
			v_sync_pol_lat_ff		<= v_sync_pol_i 		;
			odd_field_pol_lat_ff	<= odd_field_pol_i		;
			h_valid_pol_lat_ff		<= h_valid_pol_i		;
			v_valid_pol_lat_ff		<= v_valid_pol_i		;
			de_pol_lat_ff			<= de_pol_i 			;
			h_front_lat_ff			<= h_front_i			;
			h_interval_lat_ff		<= h_interval_i 		;
			h_back_lat_ff			<= h_back_i 			;
			h_video_lat_ff			<= h_video_i			;
			v_front_lat_ff			<= v_front_i			;
			v_interval_lat_ff		<= v_interval_i 		;
			v_back_lat_ff			<= v_back_i 			;
			v_video_lat_ff			<= v_video_i			;
			v_delay_lat_ff			<= v_delay_i			;
			even_v_front_lat_ff 	<= even_v_front_i		;
			even_v_interval_lat_ff	<= even_v_interval_i	;
			even_v_back_lat_ff		<= even_v_back_i		;
			even_v_video_lat_ff 	<= even_v_video_i		;
			even_v_delay_lat_ff 	<= even_v_delay_i		;
			even2odd_v_delay_lat_ff <= even2odd_v_delay_i	;
			even2odd_h_delay_lat_ff <= even2odd_h_delay_i	;
			odd2even_v_delay_lat_ff <= odd2even_v_delay_i	;
			odd2even_h_delay_lat_ff <= odd2even_h_delay_i	;
			reset_timing_lat_ff 	<= reset_timing_i		;
			reset_interval_lat_ff	<= reset_interval_i 	;
		end
	end

	assign	even2odd_v_delay_n	= even2odd_v_delay_lat_ff - 1'b1 ;
	assign	even2odd_h_delay_n	= even2odd_h_delay_lat_ff - 1'b1 ;
	assign	odd2even_v_delay_n	= odd2even_v_delay_lat_ff - 1'b1 ;
	assign	odd2even_h_delay_n	= odd2even_h_delay_lat_ff - 1'b1 ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			h_f_add_i_ff				<= {H_F_P+1{1'b0}} ;
			h_b_add_v_ff				<= {H_V_I{1'b0}} ;
			v_f_add_i_ff				<= {V_F_P+1{1'b0}} ;
			v_b_add_v_ff				<= {V_V_I{1'b0}} ;
			ev_f_add_i_ff				<= {V_F_P+1{1'b0}} ;
			ev_b_add_v_ff				<= {V_V_I{1'b0}} ;
			even2odd_vdelay_lat_z_ff	<= 1'b0 ;
			even2odd_hdelay_lat_z_ff	<= 1'b0 ;
			odd2even_vdelay_lat_z_ff	<= 1'b0 ;
			odd2even_hdelay_lat_z_ff	<= 1'b0 ;
		end
		else begin
			h_f_add_i_ff				<= h_front_lat_ff + h_interval_lat_ff ;
			h_b_add_v_ff				<= h_back_lat_ff + h_video_lat_ff ;
			v_f_add_i_ff				<= v_front_lat_ff + v_interval_lat_ff ;
			v_b_add_v_ff				<= v_back_lat_ff + v_video_lat_ff ;
			ev_f_add_i_ff				<= even_v_front_lat_ff + even_v_interval_lat_ff ;
			ev_b_add_v_ff				<= even_v_back_lat_ff + even_v_video_lat_ff ;
			even2odd_vdelay_lat_z_ff	<= & even2odd_v_delay_n ;
			even2odd_hdelay_lat_z_ff	<= & even2odd_h_delay_n ;
			odd2even_vdelay_lat_z_ff	<= & odd2even_v_delay_n ;
			odd2even_hdelay_lat_z_ff	<= & odd2even_h_delay_n ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			param_update_lat_ff		<= 1'b0 ;
			param_update_lat1_ff	<= 1'b0 ;
			param_update_lat2_ff	<= 1'b0 ;
			param_set_ff			<= 1'b0 ;
		end
		else begin
			param_update_lat_ff		<= param_update_i ;
			param_update_lat1_ff	<= param_update_lat_ff ;
			param_update_lat2_ff	<= param_update_lat1_ff ;
			param_set_ff			<= param_update_lat1_ff & ~param_update_lat2_ff ;
			param_set_str_ff		<= param_set_ff ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			param_update_wait_ff	<= 1'b0 ;
		end
		else if ( param_set_str_ff ) begin
			param_update_wait_ff	<= 1'b1 ;
		end
		else if ( param_update_timing ) begin
			param_update_wait_ff	<= 1'b0 ;
		end
	end

	assign	param_update_timing	= ( srst & param_update_wait_ff ) |
								  ( enable & param_update_wait_ff & h_video_end_n_pls_ff &
									( ( v_video_end_pls_ff & field_sel_str_ff ) |
									  ( even_v_video_end_pls_ff & ~field_sel_str_ff ) ) ) ;

	//===============//
	// sync generate //
	//===============//
	// h sync
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			h_sync_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			h_sync_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( h_front_end_pls_ff ) begin
				h_sync_ff	<= 1'b1 ;
			end
			else if ( h_sync_end_pls_ff || ex_sync_pos_str_ff ) begin
				h_sync_ff	<= 1'b0 ;
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			h_valid_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			h_valid_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( h_back_end_pls_ff ) begin
				h_valid_ff	<= 1'b1 ;
			end
			else if ( h_video_end_pls_ff || ex_sync_pos_str_ff ) begin
				h_valid_ff	<= 1'b0 ;
			end
		end
	end

	// v sync
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			v_sync_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			v_sync_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ex_sync_pos_str_ff ) begin
				v_sync_ff	<= 1'b0 ;
			end
			else if ( field_sel_str_ff ) begin
				if ( (	v_delay_zero_ff && v_front_end_pls_ff	&& h_video_end_pls_ff ) ||
					 ( !v_delay_zero_ff && v_front_end_p_pls_ff && v_delay_pls_ff ) ) begin
					v_sync_ff	<= 1'b1 ;
				end
				else if ( (	v_delay_zero_ff && v_sync_end_pls_ff   && h_video_end_pls_ff ) ||
							  ( !v_delay_zero_ff && v_sync_end_p_pls_ff && v_delay_pls_ff ) ) begin
					v_sync_ff	<= 1'b0 ;
				end
			end
			else begin
				if ( (	even_v_delay_zero_ff && even_v_front_end_pls_ff   && h_video_end_pls_ff ) ||
					 ( !even_v_delay_zero_ff && even_v_front_end_p_pls_ff && even_v_delay_pls_ff ) ) begin
					v_sync_ff	<= 1'b1 ;
				end
				else if ( (  even_v_delay_zero_ff && even_v_sync_end_pls_ff   && h_video_end_pls_ff ) ||
							  ( !even_v_delay_zero_ff && even_v_sync_end_p_pls_ff && even_v_delay_pls_ff ) ) begin
					v_sync_ff	<= 1'b0 ;
				end
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			v_valid_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			v_valid_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ex_sync_pos_str_ff ) begin
				v_valid_ff	<= 1'b0 ;
			end
			else if ( field_sel_str_ff ) begin
				if ( v_back_end_pls_ff && h_video_end_pls_ff ) begin
					v_valid_ff	<= 1'b1 ;
				end
				else if ( v_video_end_pls_ff && h_video_end_pls_ff ) begin
					v_valid_ff	<= 1'b0 ;
				end
			end
			else begin
				if ( even_v_back_end_pls_ff && h_video_end_pls_ff ) begin
					v_valid_ff	<= 1'b1 ;
				end
				else if ( even_v_video_end_pls_ff && h_video_end_pls_ff ) begin
					v_valid_ff	<= 1'b0 ;
				end
			end
		end
	end

	// field
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			field_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			field_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( field_sel_str_ff ) begin
				if ( ( odd2even_v_delay_zero_ff && odd2even_h_delay_zero_ff ) && ( v_video_end_pls_ff && h_video_end_pls_ff ) ) begin
					field_ff	<= 1'b0 ;
				end
				else if ( (  even2odd_h_delay_zero_ff && even2odd_v_delay_n_pls_ff && h_video_end_pls_ff ) ||
							  ( !even2odd_h_delay_zero_ff && even2odd_v_delay_pls_ff   && even2odd_h_delay_pls_ff ) ) begin
					field_ff	<= 1'b1 ;
				end
			end
			else begin
				if ( ( even2odd_v_delay_zero_ff && even2odd_h_delay_zero_ff ) && ( even_v_video_end_pls_ff && h_video_end_pls_ff ) ) begin
					field_ff	<= 1'b1 ;
				end
				else if ( (  odd2even_h_delay_zero_ff && odd2even_v_delay_n_pls_ff && h_video_end_pls_ff ) ||
							  ( !odd2even_h_delay_zero_ff && odd2even_v_delay_pls_ff   && odd2even_h_delay_pls_ff ) ) begin
					field_ff	<= 1'b0 ;
				end
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			field_sel_ff	<= 1'b1 ;
		end
		else if ( srst ) begin
			field_sel_ff	<= 1'b1 ;
		end
		else if ( enable ) begin
			if ( interlace_mode_ff ) begin
				if ( external_sync_pos ) begin
					field_sel_ff	<= external_field_ff ;
				end
				else if ( v_count_clr ) begin
					field_sel_ff	<= ~field_sel_ff ;
				end
			end
			else begin
				field_sel_ff	<= 1'b1 ;
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			field_sel_str_ff	<= 1'b1 ;
		end
		else if ( srst ) begin
			field_sel_str_ff	<= 1'b1 ;
		end
		else if ( enable ) begin
			field_sel_str_ff	<= field_sel_ff ;
		end
	end

	//===============//
	// external sync //
	//===============//
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			external_sync_lat_ff	<= 1'b0 ;
			external_sync_lat1_ff	<= 1'b0 ;
			external_sync_lat2_ff	<= 1'b0 ;
			external_field_ff		<= 1'b0 ;
		end
		else begin
			external_sync_lat_ff	<= external_sync_i ;
			external_sync_lat1_ff	<= external_sync_lat_ff ;
			external_sync_lat2_ff	<= external_sync_lat1_ff ;
			external_field_ff		<= ( ex_odd_field_pol_ff ) ? external_field_i : ~external_field_i ;
		end
	end

	assign	external_sync_pos	= ( ex_sync_pol_ff ) ?	~external_sync_lat2_ff &  external_sync_lat1_ff :
														external_sync_lat2_ff & ~external_sync_lat1_ff ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			ex_sync_pos_str_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			ex_sync_pos_str_ff	<= 1'b0 ;
		end
		else begin
			ex_sync_pos_str_ff	<= external_sync_pos ;
		end
	end

	//=================//
	// pixel data read //
	//=================//
	// read signal
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			pixel_rdreq_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			pixel_rdreq_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( v_valid_ff & data_rd_start_pls_ff ) begin
				pixel_rdreq_ff	<= 1'b1 ;
			end
			else if ( data_rd_end_pls_ff ) begin
				pixel_rdreq_ff	<= 1'b0 ;
			end
		end
	end

	// acknowledge signal
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			pixel_oe_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			pixel_oe_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			pixel_oe_ff	<= pixel_rdreq_ff ;
		end
	end

	// output constant data
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			const_data_out_ff	<= 1'b0 ;
			const_out_mode_ff	<= 1'b0 ;
			const_data0_ff		<= {PIXEL_BIT{1'b0}} ;
			const_data1_ff		<= {PIXEL_BIT{1'b0}} ;
		end
		else begin
			const_data_out_ff	<= const_data_out_i ;
			const_out_mode_ff	<= const_out_mode_i ;
			const_data0_ff		<= const_data0_i ;
			const_data1_ff		<= const_data1_i ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			const_data_sel_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			const_data_sel_ff	<= 1'b0 ;
		end
		else if ( h_count_clr ) begin
			const_data_sel_ff	<= 1'b0 ;
		end
		else if ( h_valid_ff && v_valid_ff ) begin
			const_data_sel_ff	<= ~const_data_sel_ff ;
		end
	end

	//====================//
	// frame reset signal //
	//====================//
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frame_rst_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			frame_rst_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( frame_rst_start ) begin
				frame_rst_ff	<= 1'b1 ;
			end
			else if ( rst_count_ff == reset_interval_ff ) begin
				frame_rst_ff	<= 1'b0 ;
			end
		end
	end

	//====================//
	// timing puls signal //
	//====================//
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			h_front_end_pls_ff			<= 1'b0 ;
			h_sync_end_pls_ff			<= 1'b0 ;
			h_back_end_pls_ff			<= 1'b0 ;
			h_video_end_n_pls_ff		<= 1'b0 ;
			h_video_end_pls_ff			<= 1'b0 ;
			v_front_end_pls_ff			<= 1'b0 ;
			v_front_end_p_pls_ff		<= 1'b0 ;
			v_sync_end_pls_ff			<= 1'b0 ;
			v_sync_end_p_pls_ff 		<= 1'b0 ;
			v_back_end_pls_ff			<= 1'b0 ;
			v_video_end_pls_ff			<= 1'b0 ;
			v_delay_pls_ff				<= 1'b0 ;
			even_v_front_end_pls_ff 	<= 1'b0 ;
			even_v_front_end_p_pls_ff	<= 1'b0 ;
			even_v_sync_end_pls_ff		<= 1'b0 ;
			even_v_sync_end_p_pls_ff	<= 1'b0 ;
			even_v_back_end_pls_ff		<= 1'b0 ;
			even_v_video_end_pls_ff 	<= 1'b0 ;
			even_v_delay_pls_ff 		<= 1'b0 ;
			even2odd_v_delay_pls_ff 	<= 1'b0 ;
			even2odd_h_delay_pls_ff 	<= 1'b0 ;
			odd2even_v_delay_pls_ff 	<= 1'b0 ;
			odd2even_h_delay_pls_ff 	<= 1'b0 ;
			data_rd_start_pls_ff		<= 1'b0 ;
			data_rd_end_pls_ff			<= 1'b0 ;
		end
		else if ( srst ) begin
			h_front_end_pls_ff			<= 1'b0 ;
			h_sync_end_pls_ff			<= 1'b0 ;
			h_back_end_pls_ff			<= 1'b0 ;
			h_video_end_n_pls_ff		<= 1'b0 ;
			h_video_end_pls_ff			<= 1'b0 ;
			v_front_end_pls_ff			<= 1'b0 ;
			v_front_end_p_pls_ff		<= 1'b0 ;
			v_sync_end_pls_ff			<= 1'b0 ;
			v_sync_end_p_pls_ff 		<= 1'b0 ;
			v_back_end_pls_ff			<= 1'b0 ;
			v_video_end_pls_ff			<= 1'b0 ;
			v_delay_pls_ff				<= 1'b0 ;
			even_v_front_end_pls_ff 	<= 1'b0 ;
			even_v_front_end_p_pls_ff	<= 1'b0 ;
			even_v_sync_end_pls_ff		<= 1'b0 ;
			even_v_sync_end_p_pls_ff	<= 1'b0 ;
			even_v_back_end_pls_ff		<= 1'b0 ;
			even_v_video_end_pls_ff 	<= 1'b0 ;
			even_v_delay_pls_ff 		<= 1'b0 ;
			even2odd_v_delay_pls_ff 	<= 1'b0 ;
			even2odd_h_delay_pls_ff 	<= 1'b0 ;
			odd2even_v_delay_pls_ff 	<= 1'b0 ;
			odd2even_h_delay_pls_ff 	<= 1'b0 ;
			data_rd_start_pls_ff		<= 1'b0 ;
			data_rd_end_pls_ff			<= 1'b0 ;
		end
		else if ( enable ) begin
			h_front_end_pls_ff			<= ( h_count_ff == h_front_end_ff			) ?	1'b1 : 1'b0 ;
			h_sync_end_pls_ff			<= ( h_count_ff == h_sync_end_ff			) ?	1'b1 : 1'b0 ;
			h_back_end_pls_ff			<= ( h_count_ff == h_back_end_ff			) ?	1'b1 : 1'b0 ;
			h_video_end_n_pls_ff		<= ( h_count_ff == h_video_end_n_ff 		) ?	1'b1 : 1'b0 ;
			h_video_end_pls_ff			<= ( h_count_ff == h_video_end_ff 			) ?	1'b1 : 1'b0 ;
			v_front_end_pls_ff			<= ( v_count_ff == v_front_end_ff 			) ?	1'b1 : 1'b0 ;
			v_front_end_p_pls_ff		<= ( v_count_ff == v_front_end_p_ff			) ?	1'b1 : 1'b0 ;
			v_sync_end_pls_ff			<= ( v_count_ff == v_sync_end_ff			) ?	1'b1 : 1'b0 ;
			v_sync_end_p_pls_ff 		<= ( v_count_ff == v_sync_end_p_ff			) ?	1'b1 : 1'b0 ;
			v_back_end_pls_ff			<= ( v_count_ff == v_back_end_ff			) ?	1'b1 : 1'b0 ;
			v_video_end_pls_ff			<= ( v_count_ff == v_video_end_ff 			) ?	1'b1 : 1'b0 ;
			v_delay_pls_ff				<= ( h_count_ff == v_delay_ff - 1'b1		) ?	1'b1 : 1'b0 ;
			even_v_front_end_pls_ff 	<= ( v_count_ff == even_v_front_end_ff		) ?	1'b1 : 1'b0 ;
			even_v_front_end_p_pls_ff	<= ( v_count_ff == even_v_front_end_p_ff	) ?	1'b1 : 1'b0 ;
			even_v_sync_end_pls_ff		<= ( v_count_ff == even_v_sync_end_ff		) ?	1'b1 : 1'b0 ;
			even_v_sync_end_p_pls_ff	<= ( v_count_ff == even_v_sync_end_p_ff		) ?	1'b1 : 1'b0 ;
			even_v_back_end_pls_ff		<= ( v_count_ff == even_v_back_end_ff		) ?	1'b1 : 1'b0 ;
			even_v_video_end_pls_ff 	<= ( v_count_ff == even_v_video_end_ff		) ?	1'b1 : 1'b0 ;
			even_v_delay_pls_ff 		<= ( h_count_ff == even_v_delay_ff - 1'b1	) ?	1'b1 : 1'b0 ;
			even2odd_v_delay_pls_ff 	<= ( v_count_ff == even2odd_v_delay_ff		) ?	1'b1 : 1'b0 ;
			even2odd_v_delay_n_pls_ff 	<= ( v_count_ff == even2odd_v_delay_n_ff	) ?	1'b1 : 1'b0 ;
			even2odd_h_delay_pls_ff 	<= ( h_count_ff == even2odd_h_delay_ff		) ?	1'b1 : 1'b0 ;
			odd2even_v_delay_pls_ff 	<= ( v_count_ff == odd2even_v_delay_ff		) ?	1'b1 : 1'b0 ;
			odd2even_v_delay_n_pls_ff 	<= ( v_count_ff == odd2even_v_delay_n_ff	) ?	1'b1 : 1'b0 ;
			odd2even_h_delay_pls_ff 	<= ( h_count_ff == odd2even_h_delay_ff		) ?	1'b1 : 1'b0 ;
			data_rd_start_pls_ff		<= ( h_count_ff == data_rd_start_ff			) ?	1'b1 : 1'b0 ;
			data_rd_end_pls_ff			<= ( h_count_ff == data_rd_end_ff 			) ?	1'b1 : 1'b0 ;
		end
	end

	//=========//
	// counter //
	//=========//
	// h counter
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			h_count_ff <= { H_COUNTER{1'b0} } ;
		end
		else if ( srst ) begin
			h_count_ff <= { H_COUNTER{1'b0} } ;
		end
		else if ( enable ) begin
			if ( h_count_clr ) begin
				h_count_ff <= { H_COUNTER{1'b0} } ;
			end
			else if ( h_video_end_n_pls_ff ) begin
				h_count_ff <= h_count_ff ;
			end
			else begin
				h_count_ff <= h_count_ff + 1'b1 ;
			end
		end
	end

	assign h_count_clr	= ( ( ( ex_sync_mode_ff ) & ( ( external_sync_pos ) |
													  ( ( ( ( ~v_video_end_pls_ff ) & ( field_sel_ff ) ) |
														  ( ( ~even_v_video_end_pls_ff ) & ( ~field_sel_ff ) ) ) & ( h_video_end_n_pls_ff ) ) ) ) |
						   ( ( ~ex_sync_mode_ff ) & ( ( h_video_end_n_pls_ff ) | ( external_sync_pos ) ) ) ) ;

	// v counter
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			v_count_ff <= { V_COUNTER{1'b0} } ;
		end
		else if ( srst ) begin
			v_count_ff <= { V_COUNTER{1'b0} } ;
		end
		else if ( enable ) begin
			if ( v_count_clr ) begin
				v_count_ff <= { V_COUNTER{1'b0} } ;
			end
			else if ( h_count_clr ) begin
				v_count_ff <= v_count_ff + 1'b1 ;
			end
		end
	end

	assign v_count_clr = ( ( external_sync_pos ) |
						   ( ( h_count_clr ) & ( ( ( v_video_end_pls_ff ) & ( field_sel_ff ) ) |
												 ( ( even_v_video_end_pls_ff ) & ( ~field_sel_ff ) ) ) ) ) ;

	// reset counter
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rst_count_ff <= 4'b0000 ;
		end
		else if ( srst ) begin
			rst_count_ff <= 4'b0000 ;
		end
		else if ( enable ) begin
			if ( rst_count_ff == reset_interval_ff ) begin
				rst_count_ff <= 4'b0000 ;
			end
			else if ( frame_rst_ff ) begin
				rst_count_ff <= rst_count_ff + 1'b1 ;
			end
		end
	end

	assign	frame_rst_start	= ( frame_rst_tim && !frame_rst_tim_str_ff ) ?	1'b1 : 1'b0 ;
	assign	frame_rst_tim	= ( v_count_ff == reset_timing_ff ) ?			1'b1 : 1'b0 ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frame_rst_tim_str_ff <= 1'b0 ;
		end
		else if ( srst ) begin
			frame_rst_tim_str_ff <= 1'b0 ;
		end
		else if ( enable ) begin
			frame_rst_tim_str_ff <= frame_rst_tim ;
		end
	end


endmodule
`default_nettype wire