//----------------------------------------------------------------------------------------
//Copyright (C) 2014 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		memory read data unpacking module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Dec.  1 2014	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT				: Bit par color
//		PLANE				: Color par pixel
//		LDATA_BIT			: high-performance controller local data bit width
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk					: clock for all circuit
//		rst_n				: asyncronous reset input
//		srst				: software reset (sync with clk)
//		enable				: clock enable
//
//		rd_start_i			: read start
//		fdata_rd_i			: one frame read operation end from memory read module
//		rd_end_i			: read end
//
//		rd_en_o				: fifo read enable
//		rd_data_i			: fifo read data
//		empty_i				: fifo empty flag
//
//		data_valid_o		: data valid of output data
//		data_o				: output data
//		data_ready_i		: ready signal from receriver module
//----------------------------------------------------------------------------------------
`default_nettype none
`timescale 1ns / 1ps

module	sfb16_unpacker (
	// global signal
	clk				,
	rst_n			,
	srst			,
	enable			,

	// control signal
	rd_start_i		,
	fdata_rd_i		,
	rd_end_i		,

	// fifo flag/data
	rd_en_o			,
	rd_data_i		,
	empty_i			,

	// timing_gen module I/F
	data_valid_o	,
	data_o			,
	data_ready_i

) ;

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	parameter	PPC				= 4				; // add
	parameter	Q_BIT			= 8 			;
	parameter	PLANE			= 3 			;
	parameter	LDATA_BIT		= 128			;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

	localparam	PIXEL_BIT		= Q_BIT * PLANE * PPC	;
//	localparam	PIXEL_BIT		= Q_BIT * PLANE			;
	localparam	RDATA_BIT		= LDATA_BIT + PIXEL_BIT	;

	function integer	log2 ;
		input integer	value ;
		begin
			value	= value - 1 ;
			for (log2 = 0 ; value > 0 ; log2 = log2 + 1 ) begin
				value	= value >> 1 ;
			end
		end
	endfunction
	localparam	RD_CNT_W		= log2(RDATA_BIT)	;

	// ---------------------------------------------------------------------
	// State Encode
	// ---------------------------------------------------------------------

	localparam	IDLE			= 4'h0 			;
	localparam	FIFO_WAIT		= 4'h1 			;
	localparam	FIFO_1STRD		= 4'h2 			;
	localparam	DATA_LAT		= 4'h3 			;
	localparam	D_VALID			= 4'h4 			;
	localparam	D_INVALID		= 4'h5 			;

// =============================================================================
// PORT DECLARATION
// =============================================================================

	// global signal
	input	wire						clk				;
	input	wire						srst			;
	input	wire						rst_n			;
	input	wire						enable			;

	// control signal
	input	wire						rd_start_i		;
	input	wire						fdata_rd_i		;
	input	wire						rd_end_i		;

	// fifo flag/data
	input	wire						empty_i			;
	output	wire						rd_en_o			;
	input	wire	[ LDATA_BIT-1 : 0 ]	rd_data_i		;

	// timing_gen module I/F
	output	wire						data_valid_o	;
	output	wire	[ PIXEL_BIT-1 : 0 ]	data_o			;
	input	wire						data_ready_i	;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg		[           3 : 0 ]	upc_nst				;
	reg		[           3 : 0 ]	upc_cst_ff			;

	reg		[ RDATA_BIT-1 : 0 ]	rdata_ff			;
	reg							data_valid_ff		;

	wire						rd_en				;
	reg		[  RD_CNT_W-1 : 0 ]	rd_cnt_ff			;
	reg							rd_en_flag_ff		;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	// -----------------------------
	//	state machine
	// -----------------------------

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			upc_cst_ff	<= IDLE ;
		end
		else if ( srst ) begin
			upc_cst_ff	<= IDLE ;
		end
		else if ( enable ) begin
			upc_cst_ff	<= upc_nst ;
		end
	end

	always @( * ) begin
		case ( upc_cst_ff )
			IDLE	: begin
				if ( rd_start_i ) begin
					upc_nst	<= FIFO_WAIT ;
				end
				else begin
					upc_nst	<= IDLE ;
				end
			end

			FIFO_WAIT	: begin
				if ( !empty_i ) begin
					upc_nst	<= FIFO_1STRD ;
				end
				else begin
					upc_nst	<= FIFO_WAIT ;
				end
			end


			FIFO_1STRD	: begin
				upc_nst	<= DATA_LAT	;
			end

			DATA_LAT	: begin
				if ( !empty_i )begin
					upc_nst	<= D_VALID	;
				end
				else begin
					upc_nst	<= DATA_LAT ;
				end
			end

			D_VALID		: begin
				if ( rd_end_i ) begin
					upc_nst	<= IDLE ;
				end
				else if ( empty_i && !fdata_rd_i ) begin
					upc_nst	<= D_INVALID ;
				end
				else begin
					upc_nst	<= D_VALID ;
				end
			end

			D_INVALID	: begin
				if ( empty_i && !fdata_rd_i ) begin
					upc_nst	<= D_INVALID ;
				end
				else begin
					upc_nst	<= D_VALID ;
				end
			end

			default	: begin
				upc_nst	<= IDLE ;
			end

		endcase
	end

	// -----------------------------
	//	output data valid
	// -----------------------------

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			data_valid_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			data_valid_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( upc_nst == D_VALID ) begin
				data_valid_ff	<= 1'b1 ;
			end
			else if ( ( upc_nst == D_INVALID && upc_cst_ff == D_VALID )
				   || ( upc_nst == IDLE ) ) begin
				data_valid_ff	<= 1'b0 ;
			end
		end
	end

	assign	data_valid_o	= data_valid_ff ;

	// -----------------------------
	// data unpacking
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rd_cnt_ff[ RD_CNT_W - 1 : 0 ]	<= LDATA_BIT[ RD_CNT_W - 1 : 0 ] ;
		end
		else if ( srst ) begin
			rd_cnt_ff[ RD_CNT_W - 1 : 0 ]	<= LDATA_BIT[ RD_CNT_W - 1 : 0 ] ;
		end
		else if ( upc_cst_ff == FIFO_WAIT ) begin
			rd_cnt_ff[ RD_CNT_W - 1 : 0 ]	<= LDATA_BIT[ RD_CNT_W - 1 : 0 ] ;
		end
		else if ( upc_cst_ff == D_VALID && data_ready_i ) begin
			if ( rd_cnt_ff <= 2 * PIXEL_BIT ) begin
				rd_cnt_ff[ RD_CNT_W -1 : 0 ]	<= rd_cnt_ff[ RD_CNT_W -1 : 0 ] - PIXEL_BIT[ RD_CNT_W -1 : 0 ] + LDATA_BIT[ RD_CNT_W -1 : 0 ] ;
			end
			else begin
				rd_cnt_ff[ RD_CNT_W -1 : 0 ]	<= rd_cnt_ff[ RD_CNT_W -1 : 0 ] - PIXEL_BIT[ RD_CNT_W -1 : 0 ] ;
			end
		end
	end

	assign	rd_en	= ( ( upc_nst == FIFO_1STRD && upc_cst_ff == FIFO_WAIT )
				      | ( upc_nst == D_VALID    && upc_cst_ff == DATA_LAT  )
				      | ( upc_cst_ff == D_VALID && data_ready_i && ( rd_cnt_ff <= 2 * PIXEL_BIT ) ) ) ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rd_en_flag_ff <= 1'b0 ;
		end
		else if ( empty_i && !fdata_rd_i && rd_en ) begin
			rd_en_flag_ff <= 1'b1 ;
		end
		else if ( !empty_i ) begin
			rd_en_flag_ff <= 1'b0 ;
		end
	end

	assign rd_en_o	= rd_en | rd_en_flag_ff ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rdata_ff[ RDATA_BIT - 1 : 0 ]		<= { RDATA_BIT{1'b0} } ;
		end
		else if ( srst ) begin
			rdata_ff[ RDATA_BIT - 1 : 0 ]		<= { RDATA_BIT{1'b0} } ;
		end
		else if ( upc_cst_ff == FIFO_WAIT ) begin
			rdata_ff[ RDATA_BIT - 1 : 0 ]		<= { RDATA_BIT{1'b0} } ;
		end
		else if ( rd_en ) begin
			if ( rd_cnt_ff - PIXEL_BIT <= PIXEL_BIT && ( upc_cst_ff != DATA_LAT ) ) begin
				rdata_ff[ RDATA_BIT - 1 : 0 ]		<= ( rd_data_i[ LDATA_BIT - 1 : 0 ] << ( rd_cnt_ff - PIXEL_BIT ) )
													 | ( rdata_ff[ RDATA_BIT - 1 : 0 ] >> PIXEL_BIT ) ;
			end
			else begin
				rdata_ff[ RDATA_BIT - 1 : 0 ]		<= rd_data_i[ LDATA_BIT - 1 : 0 ] ;
			end
		end
		else if ( upc_cst_ff == D_VALID && data_ready_i ) begin
			rdata_ff[ RDATA_BIT - 1 : 0 ]		<= ( rdata_ff[ RDATA_BIT - 1 : 0 ] >> PIXEL_BIT ) ;
		end
	end

	assign	data_o	= rdata_ff[ PIXEL_BIT - 1 : 0 ] ;

endmodule
`default_nettype wire
