//----------------------------------------------------------------------------------------
//Copyright (C) 2014 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		Simple Frame Buffer Controller IP top module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Dec.  1 2014	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT				: Bit par color
//		PLANE				: Color par pixel
//		SIZE_BIT			: size setting signal bit width
//		BASE_ADR			: logical base address for frame memory ( word address )
//		STEP_ADR			: address step between each frame ( word address )
//		LDATA_BIT			: high-performance controller local data bit width
//		LADR_BIT			: high-performance controller local address bit width
//		MODE				: set bank access
//----------------------------------------------------------------------------------------
// I/O PORTS
//		iclk				: clock for VSI input I/F
//		mclk				: clock for Memory I/F
//		oclk				: clock for VSI output I/F
//		rst_n				: asyncronous reset input
//		srst_iclk			: software reset (sync with iclk)
//		srst_mclk			: software reset (sync with mclk)
//		srst_oclk			: software reset (sync with oclk)
//		enable				: clock enable
//		param_update_i		: trigger for parameter update (sync with aclk)
//
//		scan_mode_i			: set the field status of streaming data
//		rd_mode_o			: field status of streaming data to vsi to Avalon-ST Bridge
//
//		frame_start_i		: frame start of input frame data
//		frame_end_i			: frame end of input frame data
//		valid_i				: data valid of input frame data
//		pixel_i				: pixel data of input frame data
//		field_i				: field status of input frame data  1:even field  0:odd field
//		width_i				: horizontal size of input frame data
//		height_i			: vertical size of input frame data
//		ready_o				: data reception ready of input frame data
//
//		frame_start_o		: frame start of output frame data
//		frame_end_o			: frame end of output frame data
//		valid_o				: data valid of output frame data
//		pixel_o				: pixel data of output frame data
//		field_o				: field status of output frame data  1:even field  0:odd field
//		width_o				: horizontal size of output frame data
//		height_o			: vertical size of output frame data
//		ready_i				: data reception ready of output frame data
//
//		avmm_address_o		: high-performance controller local address
//		avmm_burstcount_o	: high-performance controller local burst size
//		avmm_byteenable_o	: high-performance controller local byte enable
//		avmm_write_o		: high-performance controller local write request
//		avmm_writedata_o	: high-performance controller local write data
//		avmm_read_o			: high-performance controller local read request
//		avmm_readdatavalid_i: high-performance controller local read data valid
//		avmm_readdata_i		: high-performance controller local read data
//		avmm_waitrequest_i	: high-performance controller wait request
//----------------------------------------------------------------------------------------
`default_nettype none
`timescale 1ns / 1ps

module	sfb16_top (
	// Global Signal
	iclk					,
	mclk					,
	oclk					,
	rst_n					,
	srst_iclk				,
	srst_mclk				,
	srst_oclk				,
	enable					,
	param_update_i			,

	// Operation Setting Signal
	scan_mode_i				,
	rd_mode_o				,

	// VSI input I/F
	frame_start_i			,
	frame_end_i				,
	field_i					,
	valid_i					,
	width_i					,
	height_i				,
	pixel_i					,
	ready_o					,

	// VSI output I/F
	frame_start_o			,
	frame_end_o				,
	valid_o					,
	field_o					,
	width_o					,
	height_o				,
	pixel_o					,
	ready_i					,

	// Avalon-MM Master write side I/F
	avmm_address_o			,
	avmm_begintrans_o		,
	avmm_burstcount_o		,
	avmm_byteenable_o		,
	avmm_write_o			,
	avmm_writedata_o		,
	avmm_read_o				,
	avmm_readdatavalid_i	,
	avmm_readdata_i			,
	avmm_waitrequest_i
) ;

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	parameter	IN_PPC			= 4				; // add
	parameter	OUT_PPC			= 4				; // add

	parameter	Q_BIT			= 8 			;
	parameter	PLANE			= 3 			;
	parameter	SIZE_BIT		= 12 			;
	parameter	BASE_ADR		= 32'h0000_0000	;
	parameter	STEP_ADR		= 32'h0080_0000	;
	parameter	LDATA_BIT		= 64 			;
	parameter	LADR_BIT		= 23 			;
	parameter	MODE			= 0	;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

	localparam	LBE_BIT			= LDATA_BIT / 8 ;	//tkondoh
	localparam	LBST_BIT		= 5				;

	localparam	IN_PIX_BIT		= Q_BIT * PLANE * IN_PPC	;
	localparam	OUT_PIX_BIT		= Q_BIT * PLANE * OUT_PPC	;

// =============================================================================
// PORT DECLARATION
// =============================================================================

	// Global Signal
	input	wire							iclk					;
	input	wire							mclk					;
	input	wire							oclk					;
	input	wire							rst_n					;
	input	wire							srst_iclk				;
	input	wire							srst_mclk				;
	input	wire							srst_oclk				;
	input	wire							enable					;
	input	wire							param_update_i			;

	// Operation Setting Signal
	input	wire							scan_mode_i				;
	output	wire							rd_mode_o				;

	// VSI input I/F
	input	wire							frame_start_i			;
	input	wire							frame_end_i				;
	input	wire							valid_i					;
	input	wire							field_i					;
	input	wire	[  SIZE_BIT-1 : 0 ]		width_i					;
	input	wire	[  SIZE_BIT-1 : 0 ]		height_i				;
	input	wire	[ IN_PIX_BIT-1 : 0 ]	pixel_i					;
	output	wire							ready_o					;

	// VSI output I/F
	output	wire							frame_start_o			;
	output	wire							frame_end_o				;
	output	wire							valid_o					;
	output	wire							field_o					;
	output	wire	[  SIZE_BIT-1 : 0 ]		width_o					;
	output	wire	[  SIZE_BIT-1 : 0 ]		height_o				;
	output	wire	[ OUT_PIX_BIT-1 : 0 ]	pixel_o					;
	input	wire							ready_i					;

	// Avalon-MM Master write side I/F
	output	wire	[  LADR_BIT-1 : 0 ]		avmm_address_o			;
	output	wire							avmm_begintrans_o		;
	output	wire	[  LBST_BIT-1 : 0 ]		avmm_burstcount_o		;
	output	wire	[   LBE_BIT-1 : 0 ]		avmm_byteenable_o		;
	output	wire							avmm_write_o			;
	output	wire	[ LDATA_BIT-1 : 0 ]		avmm_writedata_o		;
	output	wire							avmm_read_o				;
	input	wire							avmm_readdatavalid_i	;
	input	wire	[ LDATA_BIT-1 : 0 ]		avmm_readdata_i			;
	input	wire							avmm_waitrequest_i		;


// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	// srst from arbiter
	wire							srst_arbiter		;

	// packer
	wire							pac_frame_start		;
	wire	[  SIZE_BIT-1 : 0 ]		pac_width			;
	wire	[  SIZE_BIT-1 : 0 ]		pac_height			;
	wire							pac_field			;
	wire							pac_frame_end		;
	wire							pac_pixel_lack		;
	wire							pac_wt_en			;
	wire	[ LDATA_BIT-1 : 0 ]		pac_wt_data			;

	// write fifo
	wire							wfifo_frame_start	;
	wire							wfifo_frame_end		;
	wire							wfifo_field			;
	wire	[  SIZE_BIT-1 : 0 ]		wfifo_width			;
	wire	[  SIZE_BIT-1 : 0 ]		wfifo_height		;
	wire							wfifo_pixel_lack	;
	wire							wfifo_afull			;
	wire							wfifo_aempty		;
	wire							wfifo_empty			;
	wire	[ LDATA_BIT-1 : 0 ]		wfifo_rdata			;

	wire							mw_trans_done		;
	wire							mw_rd_en			;

	// memory control
	wire							mr_wt_en			;
	wire	[ LDATA_BIT-1 : 0 ]		mr_wt_data			;
	wire							fdata_rd			;

	wire							bus_req_w			;
	wire							bus_grant_w			;
	wire							bus_busy_w			;

	wire							local_bbt_w			;
	wire	[  LADR_BIT-1 : 0 ]		local_adr_w			;
	wire	[  LBST_BIT-1 : 0 ]		local_size_w		;
	wire							local_write			;
	wire	[ LDATA_BIT-1 : 0 ]		local_wdata			;

	wire							bus_req_r			;
	wire							bus_grant_r			;
	wire							bus_busy_r			;

	wire							local_bbt_r			;
	wire	[  LADR_BIT-1 : 0 ]		local_adr_r			;
	wire	[  LBST_BIT-1 : 0 ]		local_size_r		;
	wire							local_read			;
	wire							local_rdatavalid	;
	wire	[ LDATA_BIT-1 : 0 ]		local_rdata			;

	wire							local_wait			;

	wire	[  SIZE_BIT-1 : 0 ]		frm0o_wdt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm0e_wdt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm1o_wdt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm1e_wdt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm2o_wdt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm2e_wdt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm3o_wdt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm3e_wdt			;

	wire	[  SIZE_BIT-1 : 0 ]		frm0o_hgt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm0e_hgt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm1o_hgt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm1e_hgt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm2o_hgt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm2e_hgt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm3o_hgt			;
	wire	[  SIZE_BIT-1 : 0 ]		frm3e_hgt			;

	wire	[  LADR_BIT-1 : 0 ]		frm0o_adr			;
	wire	[  LADR_BIT-1 : 0 ]		frm0e_adr			;
	wire	[  LADR_BIT-1 : 0 ]		frm1o_adr			;
	wire	[  LADR_BIT-1 : 0 ]		frm1e_adr			;
	wire	[  LADR_BIT-1 : 0 ]		frm2o_adr			;
	wire	[  LADR_BIT-1 : 0 ]		frm2e_adr			;
	wire	[  LADR_BIT-1 : 0 ]		frm3o_adr			;
	wire	[  LADR_BIT-1 : 0 ]		frm3e_adr			;

	wire							frm0_mode			;
	wire							frm1_mode			;
	wire							frm2_mode			;
	wire							frm3_mode			;
	wire	[          2 : 0 ]		wfrm_cnt			;
	wire	[          2 : 0 ]		rfrm_cnt			;


	// read fifo
	wire							rfifo_rd_start		;
	wire							rfifo_field			;

	wire							rfifo_afull			;
	wire							rfifo_empty			;
	wire	[ LDATA_BIT-1 : 0 ]		rfifo_rdata			;
	wire							rfifo_frm_valid		;
	wire	[  SIZE_BIT-1 : 0 ]		rfifo_width			;
	wire	[  SIZE_BIT-1 : 0 ]		rfifo_height		;
	wire							rfifo_fdata_rd		;

	// unpacker
	wire							upc_rd_en			;
	wire							upc_data_valid		;
	wire	[ OUT_PIX_BIT-1 : 0 ]	upc_data			;

	// timing generator
	wire							tg_rd_start			;
	wire							tg_field			;
	wire							tg_data_ready		;
	wire							fifo_clear			;
	wire							fifo_cend			;

	wire							rd_frm_vld			;
	wire	[  SIZE_BIT-1 : 0 ]		rd_width			;
	wire	[  SIZE_BIT-1 : 0 ]		rd_height			;
	wire							rd_mode_mclk		;
	wire							rd_mode_oclk		;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	sfb16_packer
		# (
			.PPC					( IN_PPC						),
			.Q_BIT					( Q_BIT							),
			.PLANE					( PLANE							),
			.SIZE_BIT				( SIZE_BIT						),
			.LDATA_BIT				( LDATA_BIT						)
		)
		u_sfb16_packer (
			.clk					( iclk							),
			.rst_n					( rst_n							),
			.srst					( srst_iclk						),
			.enable					( enable						),

			.frame_start_i			( frame_start_i					),
			.frame_end_i			( frame_end_i					),
			.valid_i				( valid_i						),
			.pixel_i				( pixel_i						),
			.field_i				( field_i						),
			.width_i				( width_i						),
			.height_i				( height_i						),
			.ready_o				( ready_o						),

			.frame_start_o			( pac_frame_start				),
			.frame_end_o			( pac_frame_end					),
			.field_o				( pac_field						),
			.width_o				( pac_width						),
			.height_o				( pac_height					),
			.pixel_lack_o			( pac_pixel_lack				),

			.wt_en_o				( pac_wt_en						),
			.wt_data_o				( pac_wt_data					),
			.a_full_i				( wfifo_afull					)

		) ;


	sfb16_wfifo
		# (
			.SIZE_BIT				( SIZE_BIT						),
			.DATA_BIT				( LDATA_BIT						)
		)
		u_sfb16_wfifo (
			.iclk					( iclk							),
			.mclk					( mclk							),
			.rst_n					( rst_n							),
			.srst_iclk				( srst_iclk						),
			.srst_mclk				( srst_arbiter					),

			.fs_iclk_i				( pac_frame_start				),
			.fe_iclk_i				( pac_frame_end					),
			.fld_iclk_i				( pac_field						),
			.width_iclk_i			( pac_width						),
			.height_iclk_i			( pac_height					),
			.pix_lack_iclk_i		( pac_pixel_lack				),

			.fs_mclk_o				( wfifo_frame_start				),
			.fe_mclk_o				( wfifo_frame_end				),
			.fld_mclk_o				( wfifo_field					),
			.width_mclk_o			( wfifo_width					),
			.height_mclk_o			( wfifo_height					),
			.pix_lack_mclk_o		( wfifo_pixel_lack				),
			.trans_done_i			( mw_trans_done					),

			.a_full_o				( wfifo_afull					),
			.a_empty_o				( wfifo_aempty					),
			.empty_o				( wfifo_empty					),

			.wen_i					( pac_wt_en						),
			.wdata_i				( pac_wt_data					),
			.ren_i					( mw_rd_en						),
			.rdata_o				( wfifo_rdata					)
		) ;


	sfb16_mem_write_ctl
		# (
			.SIZE_BIT				( SIZE_BIT						),
			.LADR_BIT				( LADR_BIT						),
			.BASE_ADR				( BASE_ADR / ( LDATA_BIT / 8 )	),
			.STEP_ADR				( STEP_ADR / ( LDATA_BIT / 8 )	),
			.LDATA_BIT				( LDATA_BIT						),
			.MODE					( MODE							)
		)
		u_sfb16_mem_write_ctl (
			.clk					( mclk							),
			.rst_n					( rst_n							),
			.srst					( srst_arbiter					),
			.enable					( enable						),
			.param_update_i			( param_update_i				),
			.scan_mode_i			( scan_mode_i					),

			.frame_start_i			( wfifo_frame_start				),
			.frame_end_i			( wfifo_frame_end				),
			.field_i				( wfifo_field					),
			.width_i				( wfifo_width					),
			.height_i				( wfifo_height					),
			.pix_lack_i				( wfifo_pixel_lack				),
			.trans_done_o			( mw_trans_done					),

			.rd_en_o				( mw_rd_en						),
			.rd_data_i				( wfifo_rdata					),
			.a_empty_i				( wfifo_aempty					),
			.empty_i				( wfifo_empty					),

			.avmm_address_o			( local_adr_w					),
			.avmm_begintrans_o		( local_bbt_w					),
			.avmm_burstcount_o		( local_size_w					),
			.avmm_write_o			( local_write					),
			.avmm_writedata_o		( local_wdata					),
			.avmm_waitrequest_i		( local_wait					),

			.wfrm_cnt_o				( wfrm_cnt						),
			.rfrm_cnt_i				( rfrm_cnt						),
			.rd_frm_vld_o			( rd_frm_vld					),

			.bus_req_o				( bus_req_w						),
			.bus_grant_i			( bus_grant_w					),
			.bus_busy_o				( bus_busy_w					),

			.frm0o_wdt_o			( frm0o_wdt						),
			.frm0o_hgt_o			( frm0o_hgt						),
			.frm0o_adr_o			( frm0o_adr						),
			.frm0e_wdt_o			( frm0e_wdt						),
			.frm0e_hgt_o			( frm0e_hgt						),
			.frm0e_adr_o			( frm0e_adr						),

			.frm1o_wdt_o			( frm1o_wdt						),
			.frm1o_hgt_o			( frm1o_hgt						),
			.frm1o_adr_o			( frm1o_adr						),
			.frm1e_wdt_o			( frm1e_wdt						),
			.frm1e_hgt_o			( frm1e_hgt						),
			.frm1e_adr_o			( frm1e_adr						),

			.frm2o_wdt_o			( frm2o_wdt						),
			.frm2o_hgt_o			( frm2o_hgt						),
			.frm2o_adr_o			( frm2o_adr						),
			.frm2e_wdt_o			( frm2e_wdt						),
			.frm2e_hgt_o			( frm2e_hgt						),
			.frm2e_adr_o			( frm2e_adr						),

			.frm3o_wdt_o			( frm3o_wdt						),
			.frm3o_hgt_o			( frm3o_hgt						),
			.frm3o_adr_o			( frm3o_adr						),
			.frm3e_wdt_o			( frm3e_wdt						),
			.frm3e_hgt_o			( frm3e_hgt						),
			.frm3e_adr_o			( frm3e_adr						),

			.frm0_mode_o			( frm0_mode						),
			.frm1_mode_o			( frm1_mode						),
			.frm2_mode_o			( frm2_mode						),
			.frm3_mode_o			( frm3_mode						)

		) ;

	sfb16_arbiter
		#(
			.LADR_BIT				( LADR_BIT						),
			.LDATA_BIT				( LDATA_BIT						)
		)
		u_sfb16_arbiter (
			.clk					( mclk							),
			.rst_n					( rst_n							),
			.srst					( srst_mclk						),
			.enable					( enable						),

			.wr_bus_req_i			( bus_req_w						),
			.wr_bus_grant_o			( bus_grant_w					),
			.wr_bus_busy_i			( bus_busy_w					),
			.rd_bus_req_i			( bus_req_r						),
			.rd_bus_grant_o			( bus_grant_r					),
			.rd_bus_busy_i			( bus_busy_r					),

			.wr_address_i			( local_adr_w					),
			.wr_begintrans_i		( local_bbt_w					),
			.wr_burstcount_i		( local_size_w					),
			.wr_write_i				( local_write					),
			.wr_writedata_i			( local_wdata					),

			.rd_address_i			( local_adr_r					),
			.rd_begintrans_i		( local_bbt_r					),
			.rd_burstcount_i		( local_size_r					),
			.rd_read_i				( local_read					),
			.rd_readdatavalid_o		( local_rdatavalid				),
			.rd_readdata_o			( local_rdata					),

			.local_waitrequest_o	( local_wait					),

			.avmm_address_o			( avmm_address_o				),
			.avmm_byteenable_o		( avmm_byteenable_o				),
			.avmm_begintrans_o		( avmm_begintrans_o				),
			.avmm_burstcount_o		( avmm_burstcount_o				),
			.avmm_write_o			( avmm_write_o					),
			.avmm_writedata_o		( avmm_writedata_o				),
			.avmm_read_o			( avmm_read_o					),
			.avmm_readdatavalid_i	( avmm_readdatavalid_i			),
			.avmm_readdata_i		( avmm_readdata_i				),

			.avmm_waitrequest_i		( avmm_waitrequest_i			),

			.srst_sfb_o				( srst_arbiter					)
		) ;

	sfb16_mem_read_ctl
		# (
			.SIZE_BIT				( SIZE_BIT						),
			.LADR_BIT				( LADR_BIT						),
			.BASE_ADR				( BASE_ADR / ( LDATA_BIT / 8 )	),
			.STEP_ADR				( STEP_ADR / ( LDATA_BIT / 8 )	),
			.LDATA_BIT				( LDATA_BIT						),
			.MODE					( MODE							)
		)
		u_sfb16_mem_read_ctl (
			.clk					( mclk							),
			.rst_n					( rst_n							),
			.srst					( srst_arbiter					),
			.enable					( enable						),

			.rd_start_i				( rfifo_rd_start				),
			.field_i				( rfifo_field					),
			.width_o				( rd_width						),
			.height_o				( rd_height						),
			.fdata_rd_o				( fdata_rd						),

			.wr_en_o				( mr_wt_en						),
			.wr_data_o				( mr_wt_data					),
			.a_full_i				( rfifo_afull					),

			.avmm_address_o			( local_adr_r					),
			.avmm_begintrans_o		( local_bbt_r					),
			.avmm_burstcount_o		( local_size_r					),
			.avmm_read_o			( local_read					),
			.avmm_readdatavalid_i	( local_rdatavalid				),
			.avmm_readdata_i		( local_rdata					),
			.avmm_waitrequest_i		( local_wait					),

			.rd_mode_o				( rd_mode_mclk					),
			.rd_frm_vld_i			( rd_frm_vld					),
			.wfrm_cnt_i				( wfrm_cnt						),
			.rfrm_cnt_o				( rfrm_cnt						),

			.bus_req_o				( bus_req_r						),
			.bus_grant_i			( bus_grant_r					),
			.bus_busy_o				( bus_busy_r					),

			.frm0o_wdt_i			( frm0o_wdt						),
			.frm0o_hgt_i			( frm0o_hgt						),
			.frm0o_adr_i			( frm0o_adr						),
			.frm0e_wdt_i			( frm0e_wdt						),
			.frm0e_hgt_i			( frm0e_hgt						),
			.frm0e_adr_i			( frm0e_adr						),

			.frm1o_wdt_i			( frm1o_wdt						),
			.frm1o_hgt_i			( frm1o_hgt						),
			.frm1o_adr_i			( frm1o_adr						),
			.frm1e_wdt_i			( frm1e_wdt						),
			.frm1e_hgt_i			( frm1e_hgt						),
			.frm1e_adr_i			( frm1e_adr						),

			.frm2o_wdt_i			( frm2o_wdt						),
			.frm2o_hgt_i			( frm2o_hgt						),
			.frm2o_adr_i			( frm2o_adr						),
			.frm2e_wdt_i			( frm2e_wdt						),
			.frm2e_hgt_i			( frm2e_hgt						),
			.frm2e_adr_i			( frm2e_adr						),

			.frm3o_wdt_i			( frm3o_wdt						),
			.frm3o_hgt_i			( frm3o_hgt						),
			.frm3o_adr_i			( frm3o_adr						),
			.frm3e_wdt_i			( frm3e_wdt						),
			.frm3e_hgt_i			( frm3e_hgt						),
			.frm3e_adr_i			( frm3e_adr						),

			.frm0_mode_i			( frm0_mode						),
			.frm1_mode_i			( frm1_mode						),
			.frm2_mode_i			( frm2_mode						),
			.frm3_mode_i			( frm3_mode						)
		) ;


	sfb16_rfifo
		# (
			.SIZE_BIT				( SIZE_BIT						),
			.DATA_BIT				( LDATA_BIT						)
		)
		u_sfb16_rfifo (
			.mclk					( mclk							),
			.oclk					( oclk							),
			.rst_n					( rst_n							),
			.srst_mclk				( srst_arbiter					),
			.srst_oclk				( srst_oclk						),

			.rs_mclk_o				( rfifo_rd_start				),
			.fld_mclk_o				( rfifo_field					),
			.width_mclk_i			( rd_width						),
			.height_mclk_i			( rd_height						),
			.frm_vld_mclk_i			( rd_frm_vld					),
			.fdata_rd_mclk_i		( fdata_rd						),
			.rd_mode_mclk_i			( rd_mode_mclk					),

			.rs_oclk_i				( tg_rd_start					),
			.fld_oclk_i				( tg_field						),
			.fifo_clear_oclk_i		( fifo_clear					),
			.width_oclk_o			( rfifo_width					),
			.height_oclk_o			( rfifo_height					),
			.frm_vld_oclk_o			( rfifo_frm_valid				),
			.fdata_rd_oclk_o		( rfifo_fdata_rd				),
			.fifo_cend_oclk_o		( fifo_cend						),
			.rd_mode_oclk_o			( rd_mode_oclk					),

			.a_full_o				( rfifo_afull					),
			.empty_o				( rfifo_empty					),

			.wen_i					( mr_wt_en						),
			.wdata_i				( mr_wt_data					),
			.ren_i					( upc_rd_en						),
			.rdata_o				( rfifo_rdata					)
		) ;


	sfb16_unpacker
		#(
			.PPC					( OUT_PPC						),
			.Q_BIT					( Q_BIT							),
			.PLANE					( PLANE							),
			.LDATA_BIT				( LDATA_BIT						)
		)
		u_sfb16_unpacker (
			.clk					( oclk							),
			.rst_n					( rst_n							),
			.srst					( srst_oclk						),
			.enable					( enable						),

			.rd_start_i				( tg_rd_start					),
			.fdata_rd_i				( rfifo_fdata_rd				),
			.rd_end_i				( frame_end_o					),

			.rd_en_o				( upc_rd_en						),
			.rd_data_i				( rfifo_rdata					),
			.empty_i				( rfifo_empty					),

			.data_valid_o			( upc_data_valid				),
			.data_o					( upc_data						),
			.data_ready_i			( tg_data_ready					)
		) ;

	sfb16_timing_gen
		# (
			.IN_PPC					( IN_PPC						),
			.OUT_PPC				( OUT_PPC						),
			.Q_BIT					( Q_BIT							),
			.PLANE					( PLANE							),
			.SIZE_BIT				( SIZE_BIT						)
		)
		u_sfb16_timing_gen (
			.clk					( oclk							),
			.rst_n					( rst_n							),
			.srst					( srst_oclk						),
			.enable					( enable						),

			.rd_start_o				( tg_rd_start					),
			.rd_field_o				( tg_field						),
			.frm_valid_i			( rfifo_frm_valid				),
			.rd_mode_i				( rd_mode_oclk					),
			.rd_mode_o				( rd_mode_o						),
			.width_i				( rfifo_width					),
			.height_i				( rfifo_height					),
			.fifo_clear_o			( fifo_clear					),
			.fifo_cend_i			( fifo_cend						),

			.data_valid_i			( upc_data_valid				),
			.data_i					( upc_data						),
			.data_ready_o			( tg_data_ready					),

			.frame_start_o			( frame_start_o					),
			.frame_end_o			( frame_end_o					),
			.valid_o				( valid_o						),
			.pixel_o				( pixel_o						),
			.field_o				( field_o						),
			.width_o				( width_o						),
			.height_o				( height_o						),
			.ready_i				( ready_i						)
		) ;

endmodule
`default_nettype wire
