//----------------------------------------------------------------------------------------
//Copyright (C) 2014 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		Asyncronus FIFO Controler
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Dec.  1 2014	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		ADR_BIT			: address bit width of ram
//		DATA_BIT		: data bit width
//		WATERMARK_W 	: watermark of asserting almost full flag
//						  ( word count of remaining writable space )
//		WATERMARK_R 	: watermark of asserting almost empty flag
//						  ( word count of remaining readable space )
//		RETIME			: read data retiming setting 0:no retiming / 1:retiming
//----------------------------------------------------------------------------------------
// I/O PORTS
//		wclk			: write clock
//		rclk			: read clock
//		rst_n			: asynchronous reset
//		srst_wclk		: wclk synchronous reset
//		srst_rclk		: rclk synchronous reset
//
//		fifo_wen_i		: fifo write enable
//		fifo_ren_i		: fifo read enable
//		fifo_wdata_i	: fifo write data
//		fifo_rdata_o	: fifo read data
//		fifo_aempty_o	: fifo almost empty flag
//		fifo_empty_o	: fifo empty flag
//		fifo_afull_o	: fifo almost full flag
//		fifo_full_o		: fifo full flag
//
//		ram_wadr_o		: ram write address
//		ram_radr_o		: ram read address
//		ram_wen_o		: ram write enable
//		ram_ren_o		: ram read enable
//		ram_wdata_o		: ram write data
//		ram_rdata_i		: ram read data
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	sfb16_async_fifo_ctl (
	// global signal
	wclk			,
	rclk			,
	rst_n			,
	srst_wclk		,
	srst_rclk		,

	// FIFO write side I/F
	fifo_wen_i		,
	fifo_wdata_i	,
	fifo_afull_o	,
	fifo_full_o		,

	// FIFO read side I/F
	fifo_ren_i		,
	fifo_rdata_o	,
	fifo_aempty_o	,
	fifo_empty_o	,

	// RAM I/F
	ram_wadr_o		,
	ram_radr_o		,
	ram_wen_o		,
	ram_ren_o		,
	ram_wdata_o		,
	ram_rdata_i
) ;

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------
	parameter	ADR_BIT			=  8 ;
	parameter	DATA_BIT		= 16 ;
	parameter	WATERMARK_W 	=  1 ;

	parameter	WATERMARK_R 	=  1 ;

	parameter	RETIME			=  0 ;


// =============================================================================
// PORT DECLARATION
// =============================================================================

	// global signal
	input	wire							wclk			;
	input	wire							rclk			;
	input	wire							rst_n			;
	input	wire							srst_wclk		;
	input	wire							srst_rclk		;

	// FIFO write side I/F
	input	wire							fifo_wen_i		;
	input	wire	[ DATA_BIT-1 : 0 ]		fifo_wdata_i	;
	output	wire							fifo_afull_o	;
	output	wire							fifo_full_o		;

	// FIFO read side I/F
	input	wire							fifo_ren_i		;
	output	wire	[ DATA_BIT-1 : 0 ]		fifo_rdata_o	;
	output	wire							fifo_aempty_o	;
	output	wire							fifo_empty_o	;

	// RAM I/F
	output	wire	[  ADR_BIT-1 : 0 ]		ram_wadr_o		;
	output	wire	[  ADR_BIT-1 : 0 ]		ram_radr_o		;
	output	wire							ram_wen_o		;
	output	wire							ram_ren_o		;
	output	wire	[ DATA_BIT-1 : 0 ]		ram_wdata_o		;
	input	wire	[ DATA_BIT-1 : 0 ]		ram_rdata_i		;


// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	wire							set_wrp_gray_v		;
	wire							set_rdp_gray_v		;

	reg		[  ADR_BIT-1 : 0 ]		wrp_b_ff			;
	wire	[  ADR_BIT-1 : 0 ]		wrp_b_nxt			;
	wire	[  ADR_BIT-1 : 0 ]		wrp_gray			;
	reg 	[  ADR_BIT-1 : 0 ]		wrp_gray_ff			;
	reg 	[  ADR_BIT-1 : 0 ]		wrp_gray_a_ff		;

	wire	[  ADR_BIT-1 : 0 ]		rdp_w_g2b			;
	wire	[  ADR_BIT-1 : 0 ]		sub_w_pointer		;
	wire	[  ADR_BIT-1 : 0 ]		sub_afw_pointer		;
	reg 							afull_ff			;
	reg 							full_ff				;

	reg 	[  ADR_BIT-1 : 0 ]		wrp_sync_ff			;
	reg 	[  ADR_BIT-1 : 0 ]		wrp_r_ff			;

	reg		[  ADR_BIT-1 : 0 ]      rdp_b_ff			;
	wire	[  ADR_BIT-1 : 0 ]		rdp_b_nxt			;
	wire	[  ADR_BIT-1 : 0 ]		rdp_gray			;
	reg 	[  ADR_BIT-1 : 0 ]		rdp_gray_ff			;
	reg 	[  ADR_BIT-1 : 0 ]		rdp_gray_a_ff		;

	wire	[  ADR_BIT-1 : 0 ]		wrp_r_g2b			;
	wire	[  ADR_BIT-1 : 0 ]		sub_r_pointer		;
	wire	[  ADR_BIT-1 : 0 ]		sub_aer_pointer		;
	reg 							aempty_ff			;
	reg 							empty_ff			;

	reg 	[  ADR_BIT-1 : 0 ]		rdp_sync_ff			;
	reg 	[  ADR_BIT-1 : 0 ]		rdp_w_ff			;
	reg 	[ DATA_BIT-1 : 0 ]		data_out_retime0_ff	;


// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	// -----------------------------
	//	output signal definition
	// -----------------------------

	assign	fifo_aempty_o 	= aempty_ff ;
	assign	fifo_empty_o	= empty_ff ;
	assign	fifo_afull_o	= afull_ff ;
	assign	fifo_full_o		= full_ff ;
	assign	fifo_rdata_o	= ( RETIME )? data_out_retime0_ff : ram_rdata_i ;

	assign	ram_wadr_o		= wrp_b_ff ;
	assign	ram_radr_o		= rdp_b_ff ;
	assign	ram_wen_o		= set_wrp_gray_v ;
	assign	ram_ren_o		= set_rdp_gray_v ;
	assign	ram_wdata_o		= fifo_wdata_i ;

	assign	set_wrp_gray_v	= fifo_wen_i & ~full_ff ;
	assign	set_rdp_gray_v	= fifo_ren_i & ~empty_ff ;

	// -----------------------------
	//	write pointer
	// -----------------------------

	assign	wrp_b_nxt	= wrp_b_ff + 1'b1 ;
	assign	wrp_gray	= ( { 1'b0 , wrp_b_nxt[ADR_BIT-1:1] } ) ^ wrp_b_nxt ;

	always @( posedge wclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			wrp_gray_ff 	<= { ADR_BIT{1'b0} } ;
			wrp_gray_a_ff	<= { ADR_BIT{1'b0} } ;
			wrp_b_ff		<= { ADR_BIT{1'b0} } ;
		end
		else if ( srst_wclk ) begin
			wrp_gray_ff 	<= { ADR_BIT{1'b0} } ;
			wrp_gray_a_ff	<= { ADR_BIT{1'b0} } ;
			wrp_b_ff		<= { ADR_BIT{1'b0} } ;
		end
		else if ( set_wrp_gray_v ) begin
			wrp_gray_ff 	<= wrp_gray ;
			wrp_gray_a_ff	<= wrp_gray ;
			wrp_b_ff		<= wrp_b_nxt ;
		end
	end

	// -----------------------------
	//	full / almost full flag
	// -----------------------------

	assign	rdp_w_g2b		= ( { 1'b0 , rdp_w_g2b[ADR_BIT-1:1] } ) ^ rdp_w_ff ;
	assign	sub_w_pointer	= rdp_w_g2b - wrp_b_nxt ;
	assign	sub_afw_pointer	= rdp_w_g2b - wrp_b_ff ;

	always @( posedge wclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			afull_ff	<= 1'b0 ;
		end
		else if ( srst_wclk ) begin
			afull_ff	<= 1'b0 ;
		end
		else if ( set_wrp_gray_v && ( sub_w_pointer <= WATERMARK_W ) ) begin
			afull_ff	<= 1'b1 ;
		end
		else if ( sub_afw_pointer > WATERMARK_W ) begin
			afull_ff	<= 1'b0 ;
		end
	end

	always @( posedge wclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			full_ff	<= 1'b0 ;
		end
		else if ( srst_wclk ) begin
			full_ff	<= 1'b0 ;
		end
		else if ( set_wrp_gray_v && ( rdp_w_ff == wrp_gray ) ) begin
			full_ff	<= 1'b1 ;
		end
		else if ( rdp_w_ff != wrp_gray_ff ) begin
			full_ff	<= 1'b0 ;
		end
	end

	//====================================//
	// clock translation ( wclk -> rclk ) //
	//====================================//

	always @( posedge rclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			wrp_sync_ff	<= { ADR_BIT{1'b0} } ;
			wrp_r_ff	<= { ADR_BIT{1'b0} } ;
		end
		else begin
			wrp_sync_ff	<= wrp_gray_a_ff ;
			wrp_r_ff	<= wrp_sync_ff ;
		end
	end

	// -----------------------------
	//	read pointer
	// -----------------------------

	assign	rdp_b_nxt		= rdp_b_ff + 1'b1 ;
	assign	rdp_gray	= ( { 1'b0 , rdp_b_nxt[ADR_BIT-1:1] } ) ^ rdp_b_nxt ;

	always @( posedge rclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rdp_gray_ff 	<= { ADR_BIT{1'b0} } ;
			rdp_gray_a_ff	<= { ADR_BIT{1'b0} } ;
			rdp_b_ff		<= { ADR_BIT{1'b0} } ;
		end
		else if ( srst_rclk ) begin
			rdp_gray_ff 	<= { ADR_BIT{1'b0} } ;
			rdp_gray_a_ff	<= { ADR_BIT{1'b0} } ;
			rdp_b_ff		<= { ADR_BIT{1'b0} } ;
		end
		else if ( set_rdp_gray_v ) begin
			rdp_gray_ff 	<= rdp_gray ;
			rdp_gray_a_ff	<= rdp_gray ;
			rdp_b_ff		<= rdp_b_nxt ;
		end
	end

	// -----------------------------
	//	empty / almost empty flag
	// -----------------------------

	assign	wrp_r_g2b		= { 1'b0 , wrp_r_g2b[ADR_BIT-1:1] } ^ wrp_r_ff ;
	assign	sub_r_pointer	= wrp_r_g2b - rdp_b_nxt ;
	assign	sub_aer_pointer	= wrp_r_g2b - rdp_b_ff ;

	always @( posedge rclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			aempty_ff	<= 1'b1 ;
		end
		else if ( srst_rclk ) begin
			aempty_ff	<= 1'b1 ;
		end
		else if ( set_rdp_gray_v && ( sub_r_pointer <= WATERMARK_R ) ) begin
			aempty_ff	<= 1'b1 ;
		end
		else if ( sub_aer_pointer > WATERMARK_R ) begin
			aempty_ff	<= 1'b0 ;
		end
	end

	always @( posedge rclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			empty_ff	<= 1'b1 ;
		end
		else if ( srst_rclk ) begin
			empty_ff	<= 1'b1 ;
		end
		else if ( set_rdp_gray_v && ( rdp_gray == wrp_r_ff ) ) begin
			empty_ff	<= 1'b1 ;
		end
		else if ( rdp_gray_ff != wrp_r_ff ) begin
			empty_ff	<= 1'b0 ;
		end
	end

	//====================================//
	// clock translation ( rclk -> wclk ) //
	//====================================//

	always @( posedge wclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rdp_sync_ff	<= { ADR_BIT{1'b0} } ;
			rdp_w_ff	<= { ADR_BIT{1'b0} } ;
		end
		else begin
			rdp_sync_ff	<= rdp_gray_a_ff ;
			rdp_w_ff	<= rdp_sync_ff ;
		end
	end

	always @( posedge rclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			data_out_retime0_ff	<= {DATA_BIT{1'b0}} ;
		end
		else if ( srst_rclk ) begin
			data_out_retime0_ff	<= {DATA_BIT{1'b0}} ;
		end
		else if ( set_rdp_gray_v ) begin
			data_out_retime0_ff	<= ram_rdata_i ;
		end
	end


endmodule
`default_nettype wire