//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		image scaling circuit
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Mar. 13 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT 			: quantized bit width per color plane
//		PLANE			: color plane count
//		SIZE_BIT		: size setting signal bit width
//		RATE_FRAC_BIT	: fractional part bit width for coordinate calculation
//						: recommended value ( SIZE_BIT ~ SIZE_BIT + 2 )
//
//		NEAREST			: nearest-neighbor filter enable
//		BILINEAR		: bi-linear filter enable
//		BICUBIC			: bi-cubic filter enable
//		GAUSSIAN		: gaussian filter enable
//
//		RATE_BIT		: scaling rate calculation bit width ( SIZE_BIT + RATE_FRAC_BIT )
//		DLT_FRAC_BIT	: depending on filter table resolution ( fixed value )
//		PIXEL_BIT		: pixel bit width ( Q_BIT * PLANE )
//		RAM_ADR_BIT		: ram address bus width ( SIZE_BIT )
//		RAM_DATA_BIT	: ram data bus width ( PIXEL_BIT )
//		RAM_COUNT		: number of line buffers ( nearest-neighbor, bi-linear : 5 / bi-cubic, gaussian : 3 )
//
//		DEVICE			: set FPGA device family. ( default:0 )
//
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk				: clock for all circuit
//		rst_n			: asynchronous reset ( low active )
//		srst			: synchronous reset
//		enable			: clock enable
//
//		param_update	: updating setting parameter registers
//
//		table_sel_i		: scaling filter table
//							0:nearest-neighbor / 1:bi-linear
//							2:bi-cubic / 3:gaussian
//		coeff_sel_i		: scaling filter function coefficient ( only for bi-cubic or gaussian )
//							0:a=-0.5,sigma=0.5 / 1:a=-1.0,sigma=0.8
//							2:a=-1.5,sigma=1.0 / 3:a=-2.0,sigma=1.5
//		smpl_mode_i		: chroma sampling mode 0:422 / 1:444
//		scan_mode_i		: output scan mode 0:interlace / 1:progressive
//
//		width_i			: horizontal size of input frame data
//		height_i		: vertical size of input frame data
//		tgt_wdt_i		: horizontal size of output frame data
//		tgt_hgt_i		: vertical size of output frame data
//
//		width_o			: output target size
//		height_o		: output target size
//
//		frame_start_i	: frame start of input frame data
//		frame_end_i		: frame end of input frame data
//		valid_i			: data valid of input frame data
//		pixel_i			: pixel data of input frame data
//		field_i			: field status of input frame data 1:even field / 0:odd field
//		ready_o			: data reception ready of input frame data
//
//		frame_start_o	: frame start of input frame data
//		frame_end_o		: frame end of input frame data
//		valid_o			: data valid of input frame data
//		pixel_o			: pixel data of input frame data
//		field_o			: field status of input frame data 1:even field / 0:odd field
//		ready_i			: data reception ready of input frame data
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	scl16_top (
	clk				,
	rst_n			,
	srst			,
	enable			,

	param_update_i	,

	table_sel_i		,
	coeff_sel_i		,
	smpl_mode_i		,
	scan_mode_i		,

	width_i			,
	height_i		,
	tgt_wdt_i		,
	tgt_hgt_i		,

	width_o			,
	height_o		,

	frame_start_i	,
	frame_end_i		,
	valid_i			,
	pixel_i			,
	field_i			,
	ready_o			,

	frame_start_o	,
	frame_end_o		,
	valid_o			,
	pixel_o			,
	field_o			,
	ready_i
) ;


// =============================================================================
// DEFINE INCLUDE
// =============================================================================

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================
	function integer	log2 ;
		input integer	value ;
		begin
			value	= value - 1 ;
			for (log2 = 0 ; value > 0 ; log2 = log2 + 1 ) begin
				value	= value >> 1 ;
			end
		end
	endfunction


	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------
	parameter PPC					= 4				; // add by sugino
	parameter Q_BIT					= 8				;
	parameter PLANE					= 3				;
	parameter SIZE_BIT				= 12			;
	parameter RATE_FRAC_BIT			= SIZE_BIT + 2	;

	parameter NEAREST				= 1				;
	parameter BILINEAR				= 1				;
	parameter BICUBIC				= 1				;
	parameter GAUSSIAN				= 1				;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

	parameter RATE_BIT 				= SIZE_BIT + RATE_FRAC_BIT							;
	parameter DLT_FRAC_BIT 			= 5													;

	parameter PIXEL_BIT				= Q_BIT * PLANE										;
	parameter PXL_BIT_PER_CLK		= PIXEL_BIT * PPC									; // add by sugino
	parameter DLT_BIT_PER_CLK		= DLT_FRAC_BIT * PPC								; // add by sugino

	parameter RAM_ADR_BIT			= SIZE_BIT - log2(PPC - 1)							; // add by sugino
//	parameter RAM_DATA_BIT 			= PIXEL_BIT											;
	parameter RAM_DATA_BIT			= PXL_BIT_PER_CLK									; // add by sugino
	parameter RAM_COUNT				= 3 + ( ( BICUBIC == 1 ) | ( GAUSSIAN == 1 ) ) * 2	;

	parameter DEVICE				= 0													;		// v1.1 ura

// =============================================================================
// PORT DECLARATION
// =============================================================================
	input	wire								clk				;
	input	wire								rst_n			;
	input	wire								srst			;
	input	wire								enable			;

	input	wire								param_update_i	;

	input	wire	[ 1 : 0 ]					table_sel_i		;
	input	wire	[ 1 : 0 ]					coeff_sel_i		;
	input	wire								smpl_mode_i		;
	input	wire								scan_mode_i		;

	input	wire	[ SIZE_BIT-1 : 0 ]			width_i			;
	input	wire	[ SIZE_BIT-1 : 0 ]			height_i		;
	input	wire	[ SIZE_BIT-1 : 0 ]			tgt_wdt_i		;
	input	wire	[ SIZE_BIT-1 : 0 ]			tgt_hgt_i		;

	output	wire	[ SIZE_BIT-1 : 0 ]			width_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]			height_o		;

	input	wire								frame_start_i	;
	input	wire								frame_end_i		;
	input	wire								valid_i			;
	input	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	pixel_i			;
	input	wire								field_i			;
	output	wire								ready_o			;

	output	wire								frame_start_o	;
	output	wire								frame_end_o		;
	output	wire								valid_o			;
	output	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	pixel_o			;
	output	wire								field_o			;
	input	wire								ready_i			;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================
	reg									pu_lat1_ff		;
	reg									pu_lat2_ff		;
	reg									pu_lat3_ff		;
	wire								pu_pls			;

	wire								f_rst_pls		;

	wire								fstart_asrt		;
	reg									fstart_asrt_ff	;
	wire								fstart_asrt_pls	;

	reg		[1:0]						table_lat1_ff	;
	reg		[1:0]						coeff_lat1_ff	;
	reg									smpl_lat1_ff	;
	reg									scan_lat1_ff	;

	reg		[1:0]						table_lat2_ff	;
	reg		[1:0]						coeff_lat2_ff	;
	reg									smpl_lat2_ff	;
	reg									scan_lat2_ff	;

	reg		[ SIZE_BIT-1 : 0 ]			swdt_lat1_ff	;
	reg		[ SIZE_BIT-1 : 0 ]			shgt_lat1_ff	;
	reg		[ SIZE_BIT-1 : 0 ]			twdt_lat1_ff	;
	reg		[ SIZE_BIT-1 : 0 ]			thgt_lat1_ff	;

	wire	[ RATE_BIT-1 : 0 ]			h_scl_rate		;
	wire	[ RATE_BIT-1 : 0 ]			v_scl_rate		;
	wire	[ SIZE_BIT-1 : 0 ]			src_wdt_int		;
	wire	[ SIZE_BIT-1 : 0 ]			src_hgt_int		;
	wire	[ SIZE_BIT-1 : 0 ]			tgt_wdt_int		;
	wire	[ SIZE_BIT-1 : 0 ]			tgt_hgt_int		;

	wire								init_end		;

	wire	[ SIZE_BIT-1 : 0 ]			wr_line_pos		;
	wire	[ SIZE_BIT-1 : 0 ]			rd_line_pos		;
	wire								tgt_end			;

	wire								valid_fx_dltx	;
	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	fxm1			;
	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	fx0				;
	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	fxp1			;
	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	fxp2			;
	wire	[ DLT_BIT_PER_CLK-1 : 0 ]	deltax			;

	wire								valid_fxd		;
	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	fxd				;

	wire								field			;
	wire								pix_error		;

	wire								en_vscl			;
	wire								valid_gy_dlty	;
	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	gym1			;
	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	gy0				;
	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	gyp1			;
	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	gyp2			;
	wire	[ DLT_FRAC_BIT-1 : 0 ]		deltay			;

	wire								valid_gyd		;
	wire	[ PXL_BIT_PER_CLK-1 : 0 ]	gyd				;

	wire	[ RAM_ADR_BIT-1 : 0 ]		ram_radr		;
	wire								ram_ren			;
	wire	[ RAM_DATA_BIT-1 : 0 ]		ram0_rdata		;
	wire	[ RAM_DATA_BIT-1 : 0 ]		ram1_rdata		;
	wire	[ RAM_DATA_BIT-1 : 0 ]		ram2_rdata		;
	wire	[ RAM_DATA_BIT-1 : 0 ]		ram3_rdata		;
	wire	[ RAM_DATA_BIT-1 : 0 ]		ram4_rdata		;
	wire	[ RAM_ADR_BIT-1 : 0 ]		ram_wadr		;
	wire	[ RAM_COUNT-1 : 0 ]			ram_wen			;
	wire	[ RAM_DATA_BIT-1 : 0 ]		ram_wdata		;

	genvar								i				;
	genvar								gv_i			;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================
	//-----------------------------
	// setting register controller
	//-----------------------------

	// parameter update signal latch register (1st~3rd)
	// parameter update signal rising edge register

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			pu_lat1_ff		<= 1'b0	;
			pu_lat2_ff		<= 1'b0	;
			pu_lat3_ff		<= 1'b0	;
		end
		else begin
			pu_lat1_ff		<= param_update_i	;
			pu_lat2_ff		<= pu_lat1_ff		;
			pu_lat3_ff		<= pu_lat2_ff		;
		end
	end

	assign	pu_pls	=	pu_lat2_ff & ~pu_lat3_ff ;

	// frame start assert signal flip flop

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			fstart_asrt_ff	<= 1'b0	;
		end
		else begin
			fstart_asrt_ff	<= fstart_asrt	;
		end
	end

	assign	fstart_asrt		=	frame_start_i & valid_i & ( pixel_i[3:0] == 4'h0 )	;
	assign	fstart_asrt_pls	=	fstart_asrt & ~fstart_asrt_ff	;

	// operation setting value 1st latch register ( pu_pls timing )

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			table_lat1_ff	<= 2'b00	;
			coeff_lat1_ff	<= 2'b00	;
			smpl_lat1_ff	<= 1'b0		;
			scan_lat1_ff	<= 1'b0		;
		end
		else if ( srst || pu_pls ) begin
			table_lat1_ff	<= table_sel_i	;
			coeff_lat1_ff	<= coeff_sel_i	;
			smpl_lat1_ff	<= smpl_mode_i	;
			scan_lat1_ff	<= scan_mode_i	;
		end
	end

	// target size setting 1nd latch register ( pu_pls timing )

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			twdt_lat1_ff 	<= {SIZE_BIT{1'b0}}	;
			thgt_lat1_ff 	<= {SIZE_BIT{1'b0}}	;
		end
		else if ( srst || pu_pls ) begin
			twdt_lat1_ff 	<= tgt_wdt_i	;
			thgt_lat1_ff 	<= tgt_hgt_i	;
		end
	end

	// source size setting 1nd latch register ( fstart_asrt_pls timing )

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			swdt_lat1_ff 	<= {SIZE_BIT{1'b0}}	;
			shgt_lat1_ff 	<= {SIZE_BIT{1'b0}}	;
		end
		else if ( srst || fstart_asrt_pls ) begin
			swdt_lat1_ff 	<= width_i	;
			shgt_lat1_ff 	<= height_i	;
		end
	end

	// operation setting value 2nd latch register ( f_rst_pls timing )

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			table_lat2_ff	<= 2'b00	;
			coeff_lat2_ff	<= 2'b00	;
			smpl_lat2_ff	<= 1'b0		;
			scan_lat2_ff	<= 1'b0		;
		end
		else if ( srst || f_rst_pls ) begin
			table_lat2_ff	<= table_lat1_ff	;
			coeff_lat2_ff	<= coeff_lat1_ff	;
			smpl_lat2_ff	<= smpl_lat1_ff		;
			scan_lat2_ff	<= scan_lat1_ff		;
		end
	end

	//--------------
	// lower module
	//--------------

	// scaling rates calculation ( f_rst_pls timing )

	scl16_rate_calc
		#(
			.SIZE_BIT		( SIZE_BIT				) ,
			.RATE_FRAC_BIT	( RATE_FRAC_BIT			)
		)
		u_scl_rate_calc (
			.clk			( clk					) ,
			.rst_n			( rst_n					) ,
			.srst			( srst | f_rst_pls		) ,

			.src_wdt_i		( swdt_lat1_ff			) ,
			.src_hgt_i		( shgt_lat1_ff			) ,
			.tgt_wdt_i		( twdt_lat1_ff			) ,
			.tgt_hgt_i		( thgt_lat1_ff			) ,

			.h_scl_rate_o 	( h_scl_rate			) ,
			.v_scl_rate_o 	( v_scl_rate			) ,
			.src_wdt_o		( src_wdt_int			) ,
			.src_hgt_o		( src_hgt_int			) ,
			.tgt_wdt_o		( tgt_wdt_int			) ,
			.tgt_hgt_o		( tgt_hgt_int			) ,
			.init_end_o		( init_end				)
		);

	scl16_din_ctl
		#(
			.PPC				( PPC			) , // add by sugino
			.Q_BIT				( Q_BIT			) ,
			.PLANE				( PLANE			) ,
			.SIZE_BIT			( SIZE_BIT		) ,
			.RATE_FRAC_BIT		( RATE_FRAC_BIT	) ,
			.RAM_COUNT			( RAM_COUNT		)
		)
		u_din_ctl (
			.clk				( clk			) ,
			.rst_n				( rst_n			) ,
			.srst				( srst			) ,

			.enable				( enable		) ,

			.smpl_mode_i		( smpl_lat2_ff	) ,
			.scan_mode_i		( scan_lat2_ff	) ,

			.src_wdt_i			( src_wdt_int	) ,
			.src_hgt_i			( src_hgt_int	) ,
			.tgt_wdt_i			( tgt_wdt_int	) ,
			.h_scl_rate_i 		( h_scl_rate	) ,
			.init_end_i			( init_end		) ,

			.rd_line_pos_i		( rd_line_pos	) ,
			.tgt_end_i			( tgt_end		) ,

			.valid_fx_dltx_o	( valid_fx_dltx	) ,
			.fxm1_o				( fxm1			) ,
			.fx0_o				( fx0			) ,
			.fxp1_o				( fxp1			) ,
			.fxp2_o				( fxp2			) ,
			.deltax_o 			( deltax		) ,

			.valid_fxd_i		( valid_fxd		) ,
			.fxd_i				( fxd			) ,

			.ram_wadr_o			( ram_wadr		) ,
			.ram_wen_o			( ram_wen		) ,
			.ram_wdata_o		( ram_wdata		) ,

			.wr_line_pos_o		( wr_line_pos	) ,
			.field_o			( field			) ,

			.frame_start_i		( frame_start_i	) ,
			.frame_end_i		( frame_end_i	) ,
			.valid_i			( valid_i		) ,
			.pixel_i			( pixel_i		) ,
			.field_i			( field_i		) ,
			.ready_o			( ready_o		) ,

			.pix_error_o		( pix_error		) ,
			.fstart_asrt_o		( f_rst_pls		)
	);

	generate
		for ( i = 0 ; i < PPC ; i = i + 1 ) begin	: fx_gen
			wire	[ PIXEL_BIT-1 : 0 ]			fxm1_tmp			[ 0 : PPC - 1 ]	;
			wire	[ PIXEL_BIT-1 : 0 ]			fx_tmp				[ 0 : PPC - 1 ]	;
			wire	[ PIXEL_BIT-1 : 0 ]			fxp1_tmp			[ 0 : PPC - 1 ]	;
			wire	[ PIXEL_BIT-1 : 0 ]			fxp2_tmp			[ 0 : PPC - 1 ]	;
			wire	[ DLT_FRAC_BIT-1 : 0 ]		deltax_tmp			[ 0 : PPC - 1 ]	;

			wire	[ PIXEL_BIT-1 : 0 ]			fxd_tmp				[ 0 : PPC - 1 ]		;

			assign fxm1_tmp[ i ]	= fxm1[ PIXEL_BIT * ( i + 1 ) -1 : PIXEL_BIT * i ]			;
			assign fx_tmp[ i ]		= fx0[ PIXEL_BIT * ( i + 1 ) -1 : PIXEL_BIT * i ]			;
			assign fxp1_tmp[ i ]	= fxp1[ PIXEL_BIT * ( i + 1 ) -1 : PIXEL_BIT * i ]			;
			assign fxp2_tmp[ i ]	= fxp2[ PIXEL_BIT * ( i + 1 ) -1 : PIXEL_BIT * i ]			;
			assign deltax_tmp[ i ]	= deltax[ DLT_FRAC_BIT * ( i + 1 ) -1 : DLT_FRAC_BIT * i ]	;

			for ( gv_i = 0 ; gv_i < PLANE ; gv_i = gv_i + 1 ) begin : h_scl
				wire	[ Q_BIT-1 : 0 ]	fxd_per_qbit	;
				wire					valid_tmp	;

				scl16_conv_core
					#(
						.Q_BIT			( Q_BIT									) ,
						.NEAREST		( NEAREST								) ,
						.BILINEAR		( BILINEAR								) ,
						.BICUBIC		( BICUBIC								) ,
						.GAUSSIAN		( GAUSSIAN								)
					)
					u_conv_core_h (
						.clk			( clk									) ,
						.rst_n			( rst_n									) ,
						.srst			( srst | pix_error | f_rst_pls			) ,

						.enable 		( enable								) ,

						.table_sel_i	( table_lat2_ff							) ,
						.coeff_sel_i	( coeff_lat2_ff							) ,

						.fxm1_i			( fxm1_tmp[ i ][ (gv_i+1)*Q_BIT-1 : gv_i*Q_BIT ]	) ,
						.fx0_i 			( fx_tmp[ i ][ (gv_i+1)*Q_BIT-1 : gv_i*Q_BIT ]		) ,
						.fxp1_i			( fxp1_tmp[ i ][ (gv_i+1)*Q_BIT-1 : gv_i*Q_BIT ]	) ,
						.fxp2_i			( fxp2_tmp[ i ][ (gv_i+1)*Q_BIT-1 : gv_i*Q_BIT ]	) ,
						.delta_i		( deltax_tmp[ i ]									) ,
						.valid_i		( valid_fx_dltx										) ,

						.fxd_o			( fxd_per_qbit										) ,
						.valid_o		( valid_tmp											)
					);

				assign	fxd_tmp[ i ][ (gv_i+1)*Q_BIT-1 : gv_i*Q_BIT ]	=	fxd_per_qbit ;

				if ( gv_i == 0 && i == 0) begin
					assign	valid_fxd	=	valid_tmp ;
				end

			end

			assign fxd[ PIXEL_BIT * ( i + 1 ) -1 : PIXEL_BIT * i ]	= fxd_tmp[ i ] ;


		end
	endgenerate

	scl16_dout_ctl
		#(
			.PPC				( PPC					) , // add by sugino
			.Q_BIT				( Q_BIT					) ,
			.PLANE				( PLANE					) ,
			.SIZE_BIT			( SIZE_BIT				) ,
			.RATE_FRAC_BIT		( RATE_FRAC_BIT			) ,
			.RAM_COUNT			( RAM_COUNT				)
		)
		u_dout_ctl (
			.clk				( clk					) ,
			.rst_n				( rst_n					) ,
			.srst				( srst | f_rst_pls		) ,

			.enable				( enable				) ,

			.scan_mode_i		( scan_lat2_ff			) ,

			.src_hgt_i			( src_hgt_int			) ,
			.tgt_wdt_i			( tgt_wdt_int			) ,
			.tgt_hgt_i			( tgt_hgt_int			) ,
			.v_scl_rate_i		( v_scl_rate			) ,
			.wr_line_pos_i		( wr_line_pos			) ,
			.init_end_i			( init_end				) ,

			.en_scl_o			( en_vscl				) ,
			.valid_gy_dlty_o	( valid_gy_dlty			) ,
			.gym1_o				( gym1					) ,
			.gy0_o				( gy0					),
			.gyp1_o				( gyp1					) ,
			.gyp2_o				( gyp2					) ,
			.deltay_o			( deltay				) ,

			.valid_gyd_i		( valid_gyd				) ,
			.gyd_i				( gyd					) ,

			.ram_radr_o			( ram_radr				) ,
			.ram_ren_o			( ram_ren				) ,
			.ram0_rdata_i		( ram0_rdata			) ,
			.ram1_rdata_i		( ram1_rdata			) ,
			.ram2_rdata_i		( ram2_rdata			) ,
			.ram3_rdata_i		( ram3_rdata			) ,
			.ram4_rdata_i		( ram4_rdata			) ,

			.rd_line_pos_o 		( rd_line_pos			) ,
			.tgt_end_o 			( tgt_end				) ,

			.width_o			( width_o				) ,
			.height_o			( height_o				) ,

			.frame_start_o		( frame_start_o			) ,
			.frame_end_o		( frame_end_o			) ,
			.valid_o			( valid_o				) ,
			.pixel_o			( pixel_o				) ,
			.field_o			( field_o				) ,
			.ready_i			( ready_i				) ,

			.field_i			( field					) ,

			.pix_err_i			( pix_error				)
	);

	generate
		for ( i = 0 ; i < PPC ; i = i + 1 ) begin	: gy_gen
			wire	[ PIXEL_BIT-1 : 0 ]			gym1_tmp			[ 0 : PPC - 1 ]	;
			wire	[ PIXEL_BIT-1 : 0 ]			gy_tmp				[ 0 : PPC - 1 ]	;
			wire	[ PIXEL_BIT-1 : 0 ]			gyp1_tmp			[ 0 : PPC - 1 ]	;
			wire	[ PIXEL_BIT-1 : 0 ]			gyp2_tmp			[ 0 : PPC - 1 ]	;

			wire	[ PIXEL_BIT-1 : 0 ]			gyd_tmp				[ 0 : PPC - 1 ]		;

			assign gym1_tmp[ i ]	= gym1[ PIXEL_BIT * ( i + 1 ) -1 : PIXEL_BIT * i ]			;
			assign gy_tmp[ i ]		= gy0[ PIXEL_BIT * ( i + 1 ) -1 : PIXEL_BIT * i ]			;
			assign gyp1_tmp[ i ]	= gyp1[ PIXEL_BIT * ( i + 1 ) -1 : PIXEL_BIT * i ]			;
			assign gyp2_tmp[ i ]	= gyp2[ PIXEL_BIT * ( i + 1 ) -1 : PIXEL_BIT * i ]			;

			for ( gv_i = 0 ; gv_i < PLANE ; gv_i = gv_i + 1 ) begin : v_scl

				wire	[ Q_BIT-1 : 0 ]	gyd_per_qbit	;
				wire					valid_tmp	;

				scl16_conv_core
					#(
						.Q_BIT			( Q_BIT									) ,
						.NEAREST		( NEAREST								) ,
						.BILINEAR 		( BILINEAR								) ,
						.BICUBIC		( BICUBIC								) ,
						.GAUSSIAN		( GAUSSIAN								)
					)
					u_conv_core_v (
						.clk			( clk									) ,
						.rst_n			( rst_n									) ,
						.srst			( srst | pix_error | f_rst_pls			) ,

						.enable 		( en_vscl								) ,

						.table_sel_i	( table_lat2_ff							) ,
						.coeff_sel_i	( coeff_lat2_ff							) ,

						.fxm1_i			( gym1_tmp[ i ][ (gv_i+1)*Q_BIT-1 : gv_i*Q_BIT ]	) ,
						.fx0_i			( gy_tmp[ i ][( gv_i+1)*Q_BIT-1 : gv_i*Q_BIT ]		) ,
						.fxp1_i			( gyp1_tmp[ i ][ (gv_i+1)*Q_BIT-1 : gv_i*Q_BIT ]	) ,
						.fxp2_i			( gyp2_tmp[ i ][ (gv_i+1)*Q_BIT-1 : gv_i*Q_BIT ]	) ,
						.delta_i		( deltay								) ,
						.valid_i		( valid_gy_dlty							) ,

						.fxd_o			( gyd_per_qbit							) ,
						.valid_o		( valid_tmp								)
					);

				assign	gyd_tmp[ i ][ (gv_i+1)*Q_BIT-1 : gv_i*Q_BIT ]	= gyd_per_qbit ;

				if ( gv_i == 0 && i == 0) begin
					assign	valid_gyd	= valid_tmp ;
				end

			end

			assign gyd[ PIXEL_BIT * ( i + 1 ) -1 : PIXEL_BIT * i ]	= gyd_tmp[ i ] ;

		end
	endgenerate

	generate
		if ( DEVICE == 0 ) begin	: default_ram

			scl16_srd_swr_ram
				#(
					.ADR_BIT	( RAM_ADR_BIT		) ,
					.DATA_BIT	( RAM_DATA_BIT		) ,
					.DATA_DEPTH	( 1 << RAM_ADR_BIT	)
				)
				u_line_buffer_0 (
					.wclk		( clk				) ,
					.rclk		( clk				) ,

					.wadr_i		( ram_wadr			) ,
					.radr_i		( ram_radr			) ,
					.wen_i		( ram_wen[0]		) ,
					.ren_i		( ram_ren			) ,

					.wdata_i	( ram_wdata			) ,
					.rdata_o	( ram0_rdata		)
				);

			scl16_srd_swr_ram
				#(
					.ADR_BIT	( RAM_ADR_BIT		) ,
					.DATA_BIT	( RAM_DATA_BIT		) ,
					.DATA_DEPTH	( 1 << RAM_ADR_BIT	)
				)
				u_line_buffer_1 (
					.wclk		( clk				) ,
					.rclk		( clk				) ,

					.wadr_i		( ram_wadr			) ,
					.radr_i		( ram_radr			) ,
					.wen_i		( ram_wen[1]		) ,
					.ren_i		( ram_ren			) ,

					.wdata_i	( ram_wdata			) ,
					.rdata_o	( ram1_rdata		)
				);

			scl16_srd_swr_ram
				#(
					.ADR_BIT	( RAM_ADR_BIT		) ,
					.DATA_BIT	( RAM_DATA_BIT		) ,
					.DATA_DEPTH	( 1 << RAM_ADR_BIT	)
				)
				u_line_buffer_2 (
					.wclk		( clk				) ,
					.rclk		( clk				) ,

					.wadr_i		( ram_wadr			) ,
					.radr_i		( ram_radr			) ,
					.wen_i		( ram_wen[2]		) ,
					.ren_i		( ram_ren			) ,

					.wdata_i	( ram_wdata			) ,
					.rdata_o	( ram2_rdata		)
				);

			if ( RAM_COUNT == 5 ) begin : conn_ram

				scl16_srd_swr_ram
					#(
						.ADR_BIT	( RAM_ADR_BIT		) ,
						.DATA_BIT	( RAM_DATA_BIT		) ,
						.DATA_DEPTH	( 1 << RAM_ADR_BIT	)
					)
					u_line_buffer_3 (
						.wclk		( clk				) ,
						.rclk		( clk				) ,

						.wadr_i		( ram_wadr			) ,
						.radr_i		( ram_radr			) ,
						.wen_i		( ram_wen[3]		) ,
						.ren_i		( ram_ren			) ,

						.wdata_i	( ram_wdata			) ,
						.rdata_o	( ram3_rdata		)
					);

				scl16_srd_swr_ram
					#(
						.ADR_BIT	( RAM_ADR_BIT		) ,
						.DATA_BIT	( RAM_DATA_BIT		) ,
						.DATA_DEPTH	( 1 << RAM_ADR_BIT	)
					)
					u_line_buffer_4 (
						.wclk		( clk				) ,
						.rclk		( clk				) ,

						.wadr_i		( ram_wadr			) ,
						.radr_i		( ram_radr			) ,
						.wen_i		( ram_wen[4]		) ,
						.ren_i		( ram_ren			) ,

						.wdata_i	( ram_wdata			) ,
						.rdata_o	( ram4_rdata		)
					);

			end
			else begin : conn_zero

				assign	ram3_rdata	= { RAM_DATA_BIT{1'b0} } ;
				assign	ram4_rdata	= { RAM_DATA_BIT{1'b0} } ;

			end

		end
		else begin					: lattice_ram_macro

			pmi_ram_dp
				#(
					.pmi_wr_addr_depth		( 1 << RAM_ADR_BIT	) ,
					.pmi_wr_addr_width		( RAM_ADR_BIT		) ,
					.pmi_wr_data_width		( RAM_DATA_BIT		) ,
					.pmi_rd_addr_depth		( 1 << RAM_ADR_BIT	) ,
					.pmi_rd_addr_width		( RAM_ADR_BIT		) ,
					.pmi_rd_data_width		( RAM_DATA_BIT		) ,
					.pmi_regmode			( "noreg"			) ,
					.pmi_gsr				( "disable"			) ,
					.pmi_resetmode			( "sync"			) ,
					.pmi_optimization		( "speed"			) ,
					.pmi_init_file			( "none"			) ,
					.pmi_init_file_format	( "binary"			) ,
					.pmi_family				( DEVICE			) ,
					.module_type			( "pmi_ram_dp"		)
				)
				u_line_buffer_0 (
					.Reset					( ~rst_n			) ,

					.WrClock				( clk				) ,
					.RdClock				( clk				) ,

					.WrAddress				( ram_wadr			) ,
					.RdAddress				( ram_radr			) ,
					.WE						( ram_wen[0]		) ,
					.WrClockEn				( 1'b1				) ,
					.RdClockEn				( ram_ren			) ,

					.Data					( ram_wdata			) ,
					.Q						( ram0_rdata		)
				);

			pmi_ram_dp
				#(
					.pmi_wr_addr_depth		( 1 << RAM_ADR_BIT	) ,
					.pmi_wr_addr_width		( RAM_ADR_BIT		) ,
					.pmi_wr_data_width		( RAM_DATA_BIT		) ,
					.pmi_rd_addr_depth		( 1 << RAM_ADR_BIT	) ,
					.pmi_rd_addr_width		( RAM_ADR_BIT		) ,
					.pmi_rd_data_width		( RAM_DATA_BIT		) ,
					.pmi_regmode			( "noreg"			) ,
					.pmi_gsr				( "disable"			) ,
					.pmi_resetmode			( "sync"			) ,
					.pmi_optimization		( "speed"			) ,
					.pmi_init_file			( "none"			) ,
					.pmi_init_file_format	( "binary"			) ,
					.pmi_family				( DEVICE			) ,
					.module_type			( "pmi_ram_dp"		)
				)
				u_line_buffer_1 (
					.Reset					( ~rst_n			) ,

					.WrClock				( clk				) ,
					.RdClock				( clk				) ,

					.WrAddress				( ram_wadr			) ,
					.RdAddress				( ram_radr			) ,
					.WE						( ram_wen[1]		) ,
					.WrClockEn				( 1'b1				) ,
					.RdClockEn				( ram_ren			) ,

					.Data					( ram_wdata			) ,
					.Q						( ram1_rdata		)
				);

			pmi_ram_dp
				#(
					.pmi_wr_addr_depth		( 1 << RAM_ADR_BIT	) ,
					.pmi_wr_addr_width		( RAM_ADR_BIT		) ,
					.pmi_wr_data_width		( RAM_DATA_BIT		) ,
					.pmi_rd_addr_depth		( 1 << RAM_ADR_BIT	) ,
					.pmi_rd_addr_width		( RAM_ADR_BIT		) ,
					.pmi_rd_data_width		( RAM_DATA_BIT		) ,
					.pmi_regmode			( "noreg"			) ,
					.pmi_gsr				( "disable"			) ,
					.pmi_resetmode			( "sync"			) ,
					.pmi_optimization		( "speed"			) ,
					.pmi_init_file			( "none"			) ,
					.pmi_init_file_format	( "binary"			) ,
					.pmi_family				( DEVICE			) ,
					.module_type			( "pmi_ram_dp"		)
				)
				u_line_buffer_2 (
					.Reset					( ~rst_n			) ,

					.WrClock				( clk				) ,
					.RdClock				( clk				) ,

					.WrAddress				( ram_wadr			) ,
					.RdAddress				( ram_radr			) ,
					.WE						( ram_wen[2]		) ,
					.WrClockEn				( 1'b1				) ,
					.RdClockEn				( ram_ren			) ,

					.Data					( ram_wdata			) ,
					.Q						( ram2_rdata		)
				);

			if ( RAM_COUNT == 5 ) begin : conn_ram

				pmi_ram_dp
					#(
						.pmi_wr_addr_depth		( 1 << RAM_ADR_BIT	) ,
						.pmi_wr_addr_width		( RAM_ADR_BIT		) ,
						.pmi_wr_data_width		( RAM_DATA_BIT		) ,
						.pmi_rd_addr_depth		( 1 << RAM_ADR_BIT	) ,
						.pmi_rd_addr_width		( RAM_ADR_BIT		) ,
						.pmi_rd_data_width		( RAM_DATA_BIT		) ,
						.pmi_regmode			( "noreg"			) ,
						.pmi_gsr				( "disable"			) ,
						.pmi_resetmode			( "sync"			) ,
						.pmi_optimization		( "speed"			) ,
						.pmi_init_file			( "none"			) ,
						.pmi_init_file_format	( "binary"			) ,
						.pmi_family				( DEVICE			) ,
						.module_type			( "pmi_ram_dp"		)
					)
					u_line_buffer_3 (
						.Reset					( ~rst_n			) ,

						.WrClock				( clk				) ,
						.RdClock				( clk				) ,

						.WrAddress				( ram_wadr			) ,
						.RdAddress				( ram_radr			) ,
						.WE						( ram_wen[3]		) ,
						.WrClockEn				( 1'b1				) ,
						.RdClockEn				( ram_ren			) ,

						.Data					( ram_wdata			) ,
						.Q						( ram3_rdata		)
					);

				pmi_ram_dp
					#(
						.pmi_wr_addr_depth		( 1 << RAM_ADR_BIT	) ,
						.pmi_wr_addr_width		( RAM_ADR_BIT		) ,
						.pmi_wr_data_width		( RAM_DATA_BIT		) ,
						.pmi_rd_addr_depth		( 1 << RAM_ADR_BIT	) ,
						.pmi_rd_addr_width		( RAM_ADR_BIT		) ,
						.pmi_rd_data_width		( RAM_DATA_BIT		) ,
						.pmi_regmode			( "noreg"			) ,
						.pmi_gsr				( "disable"			) ,
						.pmi_resetmode			( "sync"			) ,
						.pmi_optimization		( "speed"			) ,
						.pmi_init_file			( "none"			) ,
						.pmi_init_file_format	( "binary"			) ,
						.pmi_family				( DEVICE			) ,
						.module_type			( "pmi_ram_dp"		)
					)
					u_line_buffer_4 (
						.Reset					( ~rst_n			) ,

						.WrClock				( clk				) ,
						.RdClock				( clk				) ,

						.WrAddress				( ram_wadr			) ,
						.RdAddress				( ram_radr			) ,
						.WE						( ram_wen[4]		) ,
						.WrClockEn				( 1'b1				) ,
						.RdClockEn				( ram_ren			) ,

						.Data					( ram_wdata			) ,
						.Q						( ram4_rdata		)
					);

			end
			else begin : conn_zero

				assign	ram3_rdata	= { RAM_DATA_BIT{1'b0} } ;
				assign	ram4_rdata	= { RAM_DATA_BIT{1'b0} } ;

			end

		end
	endgenerate

endmodule

`default_nettype wire
