//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		divider circuit
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Mar. 13 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		DVEND			: dividend bit width 2 ~ 32
//		DVSOR			: divisor bit width 2 ~ 32
//		PIPELINE		: pipeline architecture 0:non pipeline / 1:pipeline
//
//		INT_DVEND		: dividend bit + divisor bit
//		BUSY_TIMER		: counter bit
//		BUSY_TIMER_OVF	: counter max value
//		PP_STEP			: pipeline step
//
// ------ divider calc state machine ------
//		DIV_IDLE		: idle state
//		DIV_CALC_BUSY	: calculation state
//		DIV_RSLT_OUT	: end of calculation state
//
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk				: clock for all circuit
//		rst_n			: asynchronous reset ( low active )
//		srst			: synchronous reset
//		enable			: clock enable
//
//		valid_i			: data valid of input data
//		dividend_i		: dividend data
//		divisor_i		: divisor data
//
//		ready_o			: data reception ready of input data
//		valid_o			: data valid of output data
//		quotient_o		: quotient data
//		remainder_o		: remainder data
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	scl16_divider (
	clk			,
	rst_n		,
	srst		,
	enable		,

	valid_i		,
	dividend_i	,
	divisor_i	,

	ready_o		,
	valid_o		,
	quotient_o	,
	remainder_o
) ;

// =============================================================================
// DEFINE INCLUDE
// =============================================================================

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------
	parameter	DVEND				= 8	;
	parameter	DVSOR				= 8	;
	parameter	PIPELINE			= 1	;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------
	parameter	INT_DVEND			= DVEND + DVSOR	;
	parameter	BUSY_TIMER			= 8				;
	parameter	BUSY_TIMER_OVF		= DVEND - 1		;
	parameter	PP_STEP				= DVEND			;

	parameter	DIV_IDLE			= 3'b001		;
	parameter	DIV_CALC_BUSY		= 3'b010		;
	parameter	DIV_RSLT_OUT		= 3'b100		;

// =============================================================================
// PORT DECLARATION
// =============================================================================
	input	wire							clk			;
	input	wire							rst_n		;
	input	wire							srst		;
	input	wire							enable		;

	input	wire							valid_i		;
	input	wire	[ DVEND-1 : 0 ] 		dividend_i	;
	input	wire	[ DVSOR-1 : 0 ] 		divisor_i	;

	output	wire							ready_o		;
	output	wire							valid_o		;
	output	wire	[ DVEND-1 : 0 ] 		quotient_o	;
	output	wire	[ DVSOR-1 : 0 ] 		remainder_o	;


// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================
	reg		[ 2 : 0 ]						div_state_ff	;
	reg										busy_ff			;
	reg		[ BUSY_TIMER-1 : 0 ]			busy_cnt_ff		;
	reg										valid_ff		;

	wire									enable_in		;
	wire	[ PP_STEP-1 : 0 ]				valid_int		;
	wire	[ INT_DVEND-1 : 0 ] 			dividend_in		;
	wire	[ (INT_DVEND*PP_STEP)-1 : 0 ]	dividend_int	;
	wire	[ INT_DVEND-1 : 0 ] 			dividend_out	;
	wire	[ DVSOR-1 : 0 ] 				divisor_in		;
	wire	[ (DVSOR*PP_STEP)-1 : 0 ]		divisor_int		;
	wire	[ DVSOR-1 : 0 ] 				divisor_out		;

	genvar									i				;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	generate

		//--------------------//
		//	 PIPELINE = YES   //
		//--------------------//

		if ( ( PIPELINE == 1 ) || ( PP_STEP == 1 ) ) begin	: IF_LABEL

			assign ready_o		= enable ;
			assign valid_o		= valid_int[PP_STEP-1] ;
			assign quotient_o	= dividend_int[INT_DVEND*(PP_STEP-1)+DVEND-1:INT_DVEND*(PP_STEP-1)] ;
			assign remainder_o	= dividend_int[(INT_DVEND*PP_STEP)-1:(INT_DVEND*PP_STEP)-DVSOR] ;

			for ( i = 0 ; i < PP_STEP ; i = i + 1 ) begin	: PP_RADIX2
				if ( i == 0 ) begin	: IF_LABEL
					scl16_div_radix2
						#(
							.DVEND			( DVEND											) ,
							.DVSOR			( DVSOR											)
						)
						u_div_radix2 (
							.clk			( clk											) ,
							.rst_n			( rst_n											) ,
							.srst			( srst											) ,
							.enable 		( enable										) ,

							.valid_i		( valid_i ),
							.dividend_i		( { {(DVSOR+DVEND-DVEND){1'b0}} , dividend_i }	) ,
							.divisor_i		( divisor_i										) ,

							.valid_o		( valid_int[i]									) ,
							.dividend_o		( dividend_int[(i+1)*INT_DVEND-1:i*INT_DVEND]	) ,
							.divisor_o		( divisor_int[(i+1)*DVSOR-1:i*DVSOR]			)
						) ;
				end
				else begin	: ELSE_LABEL
					scl16_div_radix2
						#(
							.DVEND			( DVEND											) ,
							.DVSOR			( DVSOR											)
						)
						u_div_radix2 (
							.clk			( clk 											) ,
							.rst_n			( rst_n											) ,
							.srst			( srst											) ,
							.enable 		( enable										) ,

							.valid_i		( valid_int[i-1]								) ,
							.dividend_i		( dividend_int[i*INT_DVEND-1:(i-1)*INT_DVEND]	) ,
							.divisor_i		( divisor_int[i*DVSOR-1:(i-1)*DVSOR]			) ,

							.valid_o		( valid_int[i]									) ,
							.dividend_o		( dividend_int[(i+1)*INT_DVEND-1:i*INT_DVEND]	) ,
							.divisor_o		( divisor_int[(i+1)*DVSOR-1:i*DVSOR]			)
						) ;
				end
			end
		end

		//--------------------//
		//	 PIPELINE = NO	  //
		//--------------------//

		else begin	: ELSE_LABEL

			assign ready_o			= ( div_state_ff == DIV_IDLE )
									| ( ( div_state_ff == DIV_RSLT_OUT ) & enable ) ;
			assign valid_o			= valid_ff ;
			assign quotient_o		= dividend_out[DVEND-1:0] ;
			assign remainder_o		= dividend_out[INT_DVEND-1:DVEND] ;
			assign enable_in		= ( ( div_state_ff == DIV_IDLE ) & valid_i )
									| ( div_state_ff == DIV_CALC_BUSY )
									| ( ( div_state_ff == DIV_RSLT_OUT ) & valid_i & enable ) ;
			assign dividend_in		= ( busy_ff ) ? dividend_out : { {(DVSOR+DVEND-DVEND){1'b0}} , dividend_i } ;
			assign divisor_in		= ( busy_ff ) ? divisor_out : divisor_i ;

			scl16_div_radix2
				#(
					.DVEND			( DVEND			) ,
					.DVSOR			( DVSOR			)
				)
				u_div_radix2 (
					.clk			( clk			) ,
					.rst_n			( rst_n			) ,
					.srst			( srst			) ,
					.enable 		( enable_in		) ,

					.valid_i		( 1'b0			) ,
					.dividend_i		( dividend_in	) ,
					.divisor_i		( divisor_in	) ,

					.valid_o		( ) ,
					.dividend_o		( dividend_out	) ,
					.divisor_o		( divisor_out	)
				) ;
		end

	endgenerate

	generate

		if ( ( PIPELINE != 1 ) && ( PP_STEP != 1 ) ) begin	: IF_LABEL2

			always @( posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					div_state_ff	<= DIV_IDLE ;
					busy_ff			<= 1'b0 ;
					busy_cnt_ff 	<= {BUSY_TIMER{1'b0}} ;
					valid_ff		<= 1'b0 ;
				end
				else if ( srst ) begin
					div_state_ff	<= DIV_IDLE ;
					busy_ff			<= 1'b0 ;
					busy_cnt_ff		<= {BUSY_TIMER{1'b0}} ;
					valid_ff		<= 1'b0 ;
				end
				else begin
					case ( div_state_ff )
						DIV_IDLE	: begin
							if ( valid_i ) begin
								div_state_ff	<= DIV_CALC_BUSY ;
								busy_ff			<= 1'b1 ;
								busy_cnt_ff 	<= busy_cnt_ff + 1'b1 ;
								valid_ff		<= 1'b0 ;
							end
							else begin
								busy_ff			<= 1'b0 ;
								busy_cnt_ff 	<= {BUSY_TIMER{1'b0}} ;
								valid_ff		<= 1'b0 ;
							end
						end
						DIV_CALC_BUSY	: begin
							if ( busy_cnt_ff == BUSY_TIMER_OVF ) begin
								div_state_ff	<= DIV_RSLT_OUT ;
								busy_ff			<= 1'b0 ;
								busy_cnt_ff 	<= {BUSY_TIMER{1'b0}} ;
								valid_ff		<= 1'b1 ;
							end
							else begin
								busy_cnt_ff 	<= busy_cnt_ff + 1'b1 ;
							end
						end
						DIV_RSLT_OUT	: begin
							if ( valid_i & enable ) begin
								div_state_ff	<= DIV_CALC_BUSY ;
								busy_ff			<= 1'b1 ;
								busy_cnt_ff 	<= busy_cnt_ff + 1'b1 ;
								valid_ff		<= 1'b0 ;
							end
							else if ( enable ) begin
								div_state_ff	<= DIV_IDLE ;
								valid_ff		<= 1'b0 ;
							end
						end
						default	: begin
							div_state_ff	<= DIV_IDLE ;
							busy_ff			<= 1'bx ;
							busy_cnt_ff 	<= {BUSY_TIMER{1'bx}} ;
							valid_ff		<= 1'bx ;
						end
					endcase
				end
			end

		end

	endgenerate

endmodule

`default_nettype wire
