//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		2's base divider component
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Mar. 13 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		DVEND		: dividend bit width 2 ~ 32
//		DVSOR		: divisor bit width 2 ~ 32
//		INT_DVEND	: dividend bit + divisor bit
//
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk				: clock for all circuit
//		rst_n			: asynchronous reset ( low active )
//		srst			: synchronous reset
//		enable			: clock enable
//
//		valid_i			: data valid of input data
//		dividend_i		: dividend data
//		divisor_i		: divisor data
//
//		valid_o			: data valid of output data
//		dividend_o		: shifted dividend data
//		divisor_o		: divisor data
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	scl16_div_radix2 (
	clk			,
	rst_n		,
	srst		,
	enable		,

	valid_i		,
	dividend_i	,
	divisor_i	,

	valid_o		,
	dividend_o	,
	divisor_o
) ;

// =============================================================================
// DEFINE INCLUDE
// =============================================================================

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------
	parameter	DVEND 				= 8	;
	parameter	DVSOR 				= 8	;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------
	parameter	INT_DVEND 			= DVEND + DVSOR	;

// =============================================================================
// PORT DECLARATION
// =============================================================================
	input	wire							clk			;
	input	wire							rst_n		;
	input	wire							srst		;
	input	wire							enable		;

	input	wire							valid_i		;
	input	wire	[ INT_DVEND-1 : 0 ]		dividend_i	;
	input	wire	[ DVSOR-1 : 0 ]			divisor_i	;

	output	wire							valid_o		;
	output	wire	[ INT_DVEND-1 : 0 ]		dividend_o	;
	output	wire	[ DVSOR-1 : 0 ]			divisor_o	;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	wire	[ DVSOR+1 : 0 ] 		diff_mult			;
	wire							diff_flag			;

	reg								valid_ff;
	reg		[ INT_DVEND-1 : 0 ] 	dividend_lat_ff		;
	reg		[ DVSOR-1 : 0 ] 		divisor_lat_ff		;

	wire	[ INT_DVEND-1 : 0 ] 	dividend_out_comb	;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	assign diff_mult	= { 1'b0 , dividend_lat_ff[INT_DVEND-1:DVEND-1] } - { 2'b00 , divisor_lat_ff } ;
	assign diff_flag	= diff_mult[DVSOR+1] ;

	assign	dividend_out_comb	= ( diff_flag )	? { dividend_lat_ff[INT_DVEND-2:0] , 1'b0 }
												: { diff_mult[DVSOR-1:0] , dividend_lat_ff[DVEND-2:0] , 1'b1 } ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			valid_ff		<= 1'b0 ;
			dividend_lat_ff	<= {INT_DVEND{1'b0}} ;
			divisor_lat_ff	<= {DVSOR{1'b0}} ;
		end
		else if ( srst ) begin
			valid_ff		<= 1'b0 ;
			dividend_lat_ff	<= {INT_DVEND{1'b0}} ;
			divisor_lat_ff	<= {DVSOR{1'b0}} ;
		end
		else if ( enable ) begin
			valid_ff		<= valid_i ;
			dividend_lat_ff	<= dividend_i ;
			divisor_lat_ff	<= divisor_i ;
		end
	end

	assign valid_o		= valid_ff ;
	assign dividend_o	= dividend_out_comb ;
	assign divisor_o	= divisor_lat_ff ;

endmodule

`default_nettype wire
