//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		horizontal direction scaling circuit
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Mar. 13 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT		: quantized bit width per color plane
//		NEAREST		: nearest-neighbor filter enable
//		BILINEAR	: bi-linear filter enable
//		BICUBIC		: bi-cubic filter enable
//		GAUSSIAN	: gaussian filter enable
//
//		DLT_BIT		: fractional value bit width ( fixed value )
//		COEFF_BIT	: coefficient bit width ( Q_BIT + 1 )
//		MA_BIT		: multiply accumulate result bit width ( Q_BIT + 1 + COEFF_BIT + 2 )
//		CORE_DLY	: pipeline delay cycle ( fixed value )
//
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk			: clock for all circuit
//		rst_n		: asynchronous reset ( low active )
//		srst		: synchronous reset
//		enable		: clock enable
//
//		table_sel_i	: scaling filter table
//						0:nearest-neighbor / 1:bi-linear
//						2:bi-cubic / 3:gaussian
//		coeff_sel_i	: scaling filter function coefficient ( only for bi-cubic or gaussian )
//						0:a=-0.5,sigma=0.5 / 1:a=-1.0,sigma=0.8
//						2:a=-1.5,sigma=1.0 / 3:a=-2.0,sigma=1.5
//
//		fxm1_i		: x-1 pixel data
//		fx0_i		: x   pixel data
//		fxp1_i		: x+1 pixel data
//		fxp2_i		: x+2 pixel data
//		delta_i		: fractional value of reference coordinate
//		valid_i		: data valid of input data
//
//		fxd_o		: interpolated pixel data
//		valid_o		: data valid of output data
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	scl16_conv_core (
	clk			,
	rst_n		,
	srst		,
	enable		,

	table_sel_i	,
	coeff_sel_i	,

	fxm1_i		,
	fx0_i		,
	fxp1_i		,
	fxp2_i		,
	delta_i		,
	valid_i		,

	fxd_o		,
	valid_o
) ;

// =============================================================================
// DEFINE INCLUDE
// =============================================================================

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------
	parameter Q_BIT					= 8								;

	parameter NEAREST				= 1								;
	parameter BILINEAR				= 1								;
	parameter BICUBIC				= 1								;
	parameter GAUSSIAN				= 1								;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------
	parameter DLT_BIT				= 5								;
	parameter COEFF_BIT				= Q_BIT + 1						;

	parameter MA_BIT				= ( Q_BIT + 1 ) + COEFF_BIT + 2	;

	parameter CORE_DLY				= 4								;

// =============================================================================
// PORT DECLARATION
// =============================================================================
	input	wire							clk			;
	input	wire							rst_n		;
	input	wire							srst		;
	input	wire							enable		;

	input	wire	[ 1 : 0 ]				table_sel_i	;
	input	wire	[ 1 : 0 ]				coeff_sel_i	;

	input	wire	[ Q_BIT-1 : 0 ]			fxm1_i		;
	input	wire	[ Q_BIT-1 : 0 ]			fx0_i		;
	input	wire	[ Q_BIT-1 : 0 ]			fxp1_i		;
	input	wire	[ Q_BIT-1 : 0 ]			fxp2_i		;
	input	wire	[ DLT_BIT-1 : 0 ]		delta_i		;
	input	wire	 						valid_i		;

	output	wire	[ Q_BIT-1 : 0 ]			fxd_o		;
	output	wire							valid_o		;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================
	reg		[ Q_BIT-1 : 0 ]			fxm1_lat_ff			;
	reg		[ Q_BIT-1 : 0 ]			fx0_lat_ff			;
	reg		[ Q_BIT-1 : 0 ]			fxp1_lat_ff			;
	reg		[ Q_BIT-1 : 0 ]			fxp2_lat_ff			;

	wire	[ COEFF_BIT-1 : 0 ]		nearest_coeff_zr	;
	wire	[ COEFF_BIT-1 : 0 ]		nearest_coeff_p1	;
	wire	[ COEFF_BIT-1 : 0 ]		bilinear_coeff_zr	;
	wire	[ COEFF_BIT-1 : 0 ]		bilinear_coeff_p1	;
	wire	[ COEFF_BIT-1 : 0 ]		bicubic_coeff_m1	;
	wire	[ COEFF_BIT-1 : 0 ]		bicubic_coeff_zr	;
	wire	[ COEFF_BIT-1 : 0 ]		bicubic_coeff_p1	;
	wire	[ COEFF_BIT-1 : 0 ]		bicubic_coeff_p2	;
	wire	[ COEFF_BIT-1 : 0 ]		gaussian_coeff_m1	;
	wire	[ COEFF_BIT-1 : 0 ]		gaussian_coeff_zr	;
	wire	[ COEFF_BIT-1 : 0 ]		gaussian_coeff_p1	;
	wire	[ COEFF_BIT-1 : 0 ]		gaussian_coeff_p2	;
	wire	[ COEFF_BIT-1 : 0 ]		coeff_m1			;
	wire	[ COEFF_BIT-1 : 0 ]		coeff_zr			;
	wire	[ COEFF_BIT-1 : 0 ]		coeff_p1			;
	wire	[ COEFF_BIT-1 : 0 ]		coeff_p2			;

	wire	[ MA_BIT-1 : 0 ]		multadd_rlt			;

	reg		[ CORE_DLY-1 : 0 ]		valid_dly_ff		;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================
	//--------------------
	// pipeline 1st stage
	//--------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			fxm1_lat_ff		<= {Q_BIT{1'b0}} ;
			fx0_lat_ff		<= {Q_BIT{1'b0}} ;
			fxp1_lat_ff		<= {Q_BIT{1'b0}} ;
			fxp2_lat_ff		<= {Q_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			fxm1_lat_ff		<= {Q_BIT{1'b0}} ;
			fx0_lat_ff		<= {Q_BIT{1'b0}} ;
			fxp1_lat_ff		<= {Q_BIT{1'b0}} ;
			fxp2_lat_ff		<= {Q_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			fxm1_lat_ff		<= fxm1_i ;
			fx0_lat_ff		<= fx0_i ;
			fxp1_lat_ff		<= fxp1_i ;
			fxp2_lat_ff		<= fxp2_i ;
		end
	end

	generate
		if ( NEAREST ) begin : nearst_table
			scl16_nearest_ph32_func
				#(
					.Q_BIT			( Q_BIT				)
				)
				u_nearest_ph32_func (
					.clk			( clk				) ,
					.rst_n			( rst_n				) ,
					.enable			( enable			) ,

					.x_i			( delta_i			) ,

					.coeff_zr_o		( nearest_coeff_zr	) ,
					.coeff_p1_o		( nearest_coeff_p1	)
				) ;
		end
		else begin : non_nearst
			assign nearest_coeff_zr		= {COEFF_BIT{1'b0}} ;
			assign nearest_coeff_p1		= {COEFF_BIT{1'b0}} ;
		end
	endgenerate

	generate
		if ( BILINEAR ) begin : bilinear_table
			scl16_bilinear_ph32_func
				#(
					.Q_BIT			( Q_BIT				)
				)
				u_bilinear_ph32_func (
					.clk			( clk				) ,
					.rst_n			( rst_n				) ,
					.enable			( enable			) ,

					.x_i			( delta_i			) ,

					.coeff_zr_o		( bilinear_coeff_zr	) ,
					.coeff_p1_o		( bilinear_coeff_p1	)
				) ;
		end
		else begin : non_bilinear
			assign bilinear_coeff_zr	= {COEFF_BIT{1'b0}} ;
			assign bilinear_coeff_p1	= {COEFF_BIT{1'b0}} ;
		end
	endgenerate

	generate
		if ( BICUBIC ) begin : bicubic_table
			scl16_bicubic_ph32_func
				#(
					.Q_BIT			( Q_BIT				)
				)
				u_bicubic_ph32_func (
					.clk			( clk				) ,
					.rst_n			( rst_n				) ,
					.enable			( enable			) ,

					.a_i			( coeff_sel_i		) ,
					.x_i			( delta_i			) ,

					.coeff_m1_o		( bicubic_coeff_m1	) ,
					.coeff_zr_o		( bicubic_coeff_zr	) ,
					.coeff_p1_o		( bicubic_coeff_p1	) ,
					.coeff_p2_o		( bicubic_coeff_p2	)
				);
		end
		else begin : non_bicubic
			assign bicubic_coeff_m1		= {COEFF_BIT{1'b0}} ;
			assign bicubic_coeff_zr		= {COEFF_BIT{1'b0}} ;
			assign bicubic_coeff_p1		= {COEFF_BIT{1'b0}} ;
			assign bicubic_coeff_p2		= {COEFF_BIT{1'b0}} ;
		end
	endgenerate

	generate
		if ( GAUSSIAN ) begin : gaussian_table
			scl16_gaussian_ph32_func
				#(
					.Q_BIT			( Q_BIT				)
				)
				u_gaussian_ph32_func (
					.clk			( clk				) ,
					.rst_n			( rst_n				) ,
					.enable			( enable			) ,

					.s_i			( coeff_sel_i		) ,
					.x_i			( delta_i			) ,

					.coeff_m1_o		( gaussian_coeff_m1	) ,
					.coeff_zr_o		( gaussian_coeff_zr	) ,
					.coeff_p1_o		( gaussian_coeff_p1	) ,
					.coeff_p2_o		( gaussian_coeff_p2	)
			);
		end
		else begin : non_gaussian
			assign gaussian_coeff_m1	= {COEFF_BIT{1'b0}} ;
			assign gaussian_coeff_zr	= {COEFF_BIT{1'b0}} ;
			assign gaussian_coeff_p1	= {COEFF_BIT{1'b0}} ;
			assign gaussian_coeff_p2	= {COEFF_BIT{1'b0}} ;
		end
	endgenerate

	assign coeff_m1		= ( {COEFF_BIT{( table_sel_i == 2'b10 )}} & bicubic_coeff_m1 )
						| ( {COEFF_BIT{( table_sel_i == 2'b11 )}} & gaussian_coeff_m1 ) ;

	assign coeff_zr		= ( {COEFF_BIT{( table_sel_i == 2'b00 )}} & nearest_coeff_zr )
						| ( {COEFF_BIT{( table_sel_i == 2'b01 )}} & bilinear_coeff_zr )
						| ( {COEFF_BIT{( table_sel_i == 2'b10 )}} & bicubic_coeff_zr )
						| ( {COEFF_BIT{( table_sel_i == 2'b11 )}} & gaussian_coeff_zr ) ;

	assign coeff_p1		= ( {COEFF_BIT{( table_sel_i == 2'b00 )}} & nearest_coeff_p1 )
						| ( {COEFF_BIT{( table_sel_i == 2'b01 )}} & bilinear_coeff_p1 )
						| ( {COEFF_BIT{( table_sel_i == 2'b10 )}} & bicubic_coeff_p1 )
						| ( {COEFF_BIT{( table_sel_i == 2'b11 )}} & gaussian_coeff_p1 ) ;

	assign coeff_p2		= ( {COEFF_BIT{( table_sel_i == 2'b10 )}} & bicubic_coeff_p2 )
						| ( {COEFF_BIT{( table_sel_i == 2'b11 )}} & gaussian_coeff_p2 ) ;

	// -------------------
	// pipeline 2nd stage
	// pipeline 3rd stage
	// pipeline 4th stage
	// -------------------

	scl16_signed_multadd4
		#(
			.APORT_BIT		( Q_BIT + 1					) ,
			.BPORT_BIT		( COEFF_BIT					)
		)
		u_multadd_for_conv_core (
			.clk			( clk						) ,
			.rst_n			( rst_n						) ,
			.enable			( enable					) ,

			.dataa_0_i		( { 1'b0 , fxm1_lat_ff }	) ,
			.dataa_1_i		( { 1'b0 , fx0_lat_ff }		) ,
			.dataa_2_i		( { 1'b0 , fxp1_lat_ff }	) ,
			.dataa_3_i		( { 1'b0 , fxp2_lat_ff }	) ,
			.datab_0_i		( coeff_m1					) ,
			.datab_1_i		( coeff_zr					) ,
			.datab_2_i		( coeff_p1					) ,
			.datab_3_i		( coeff_p2					) ,

			.result_o		( multadd_rlt				)
	);

	assign fxd_o			= ( multadd_rlt[MA_BIT-1] )				? {Q_BIT{1'b0}}	:
							  ( |multadd_rlt[MA_BIT-2:MA_BIT-5] )	? {Q_BIT{1'b1}}	: multadd_rlt[MA_BIT-6:MA_BIT-Q_BIT-5] ;

	// -------------------
	// valid signal shift
	// -------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			valid_dly_ff	<= {CORE_DLY{1'b0}} ;
		end
		else if ( srst ) begin
			valid_dly_ff	<= {CORE_DLY{1'b0}} ;
		end
		else if ( enable ) begin
			valid_dly_ff	<= { valid_dly_ff[CORE_DLY-2:0] , valid_i } ;
		end
	end

	assign valid_o			= valid_dly_ff[CORE_DLY-1] ;

endmodule

`default_nettype wire
