//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		bi-cubic scaling filter coefficient table for scaler adaptive
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Mar. 13 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT		: quantized bit width per color plane
//
//		DLT_BIT		: delta bit width
//		COEFF_BIT	: coefficient bit width
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk;		: clock for all circuit
//		rst_n;		: asynchronous reset ( low active )
//		enable;		: clock enable
//
//		a_i			: scaling filter function coefficient
//						0:a=-0.5 / 1:a=-1 / 2:a=-1.5 / 3:a=-2
//		x_i			: table index
//
//		coeff_m1_o	: table value ( -2 -> -1 section coefficient )
//		coeff_zr_o	: table value ( -1 ->  0 section coefficient )
//		coeff_p1_o	: table value (  0 ->  1 section coefficient )
//		coeff_p2_o	: table value (  1 ->  2 section coefficient )
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	scl16_bicubic_ph32_func (
	clk			,
	rst_n		,
	enable		,

	a_i			,
	x_i			,

	coeff_m1_o	,
	coeff_zr_o	,
	coeff_p1_o	,
	coeff_p2_o
) ;

// =============================================================================
// DEFINE INCLUDE
// =============================================================================

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------
	parameter Q_BIT					= 8			;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------
	parameter DLT_BIT				= 5			;
	parameter COEFF_BIT				= Q_BIT + 1	;

// =============================================================================
// PORT DECLARATION
// =============================================================================
	input	wire							clk			;
	input	wire							rst_n		;
	input	wire							enable		;

	input	wire	[ 1 : 0 ]				a_i			;
	input	wire	[ DLT_BIT-1 : 0 ]		x_i			;

	output	wire	[ COEFF_BIT-1 : 0 ]		coeff_m1_o	;
	output	wire	[ COEFF_BIT-1 : 0 ]		coeff_zr_o	;
	output	wire	[ COEFF_BIT-1 : 0 ]		coeff_p1_o	;
	output	wire	[ COEFF_BIT-1 : 0 ]		coeff_p2_o	;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================
	reg		[ COEFF_BIT-1 : 0 ]		coeff_m1_ff	;
	reg		[ COEFF_BIT-1 : 0 ]		coeff_zr_ff	;
	reg		[ COEFF_BIT-1 : 0 ]		coeff_p1_ff	;
	reg		[ COEFF_BIT-1 : 0 ]		coeff_p2_ff	;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================
	generate
		if ( Q_BIT == 6 ) begin : table_6bit
			//-------------------- Table Value Format -----------------------//
			//	 { Sign(1bit), Integral Part(1bit), Fraction Part(5bit) }	 //
			//---------------------------------------------------------------//
			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_m1_ff		<= 7'h00 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_m1_ff	<= 7'h00 ;
						7'b00_0_0001	: coeff_m1_ff	<= 7'h7f ;
						7'b00_0_0010	: coeff_m1_ff	<= 7'h7f ;
						7'b00_0_0011	: coeff_m1_ff	<= 7'h7e ;
						7'b00_0_0100	: coeff_m1_ff	<= 7'h7e ;
						7'b00_0_0101	: coeff_m1_ff	<= 7'h7e ;
						7'b00_0_0110	: coeff_m1_ff	<= 7'h7e ;
						7'b00_0_0111	: coeff_m1_ff	<= 7'h7d ;
						7'b00_0_1000	: coeff_m1_ff	<= 7'h7d ;
						7'b00_0_1001	: coeff_m1_ff	<= 7'h7d ;
						7'b00_0_1010	: coeff_m1_ff	<= 7'h7d ;
						7'b00_0_1011	: coeff_m1_ff	<= 7'h7d ;
						7'b00_0_1100	: coeff_m1_ff	<= 7'h7d ;
						7'b00_0_1101	: coeff_m1_ff	<= 7'h7d ;
						7'b00_0_1110	: coeff_m1_ff	<= 7'h7d ;
						7'b00_0_1111	: coeff_m1_ff	<= 7'h7d ;
						7'b00_1_0000	: coeff_m1_ff	<= 7'h7e ;
						7'b00_1_0001	: coeff_m1_ff	<= 7'h7e ;
						7'b00_1_0010	: coeff_m1_ff	<= 7'h7e ;
						7'b00_1_0011	: coeff_m1_ff	<= 7'h7e ;
						7'b00_1_0100	: coeff_m1_ff	<= 7'h7e ;
						7'b00_1_0101	: coeff_m1_ff	<= 7'h7e ;
						7'b00_1_0110	: coeff_m1_ff	<= 7'h7e ;
						7'b00_1_0111	: coeff_m1_ff	<= 7'h7f ;
						7'b00_1_1000	: coeff_m1_ff	<= 7'h7f ;
						7'b00_1_1001	: coeff_m1_ff	<= 7'h7f ;
						7'b00_1_1010	: coeff_m1_ff	<= 7'h7f ;
						7'b00_1_1011	: coeff_m1_ff	<= 7'h7f ;
						7'b00_1_1100	: coeff_m1_ff	<= 7'h7f ;
						7'b00_1_1101	: coeff_m1_ff	<= 7'h7f ;
						7'b00_1_1110	: coeff_m1_ff	<= 7'h7f ;
						7'b00_1_1111	: coeff_m1_ff	<= 7'h7f ;

						7'b01_0_0000	: coeff_m1_ff	<= 7'h00 ;
						7'b01_0_0001	: coeff_m1_ff	<= 7'h7f ;
						7'b01_0_0010	: coeff_m1_ff	<= 7'h7e ;
						7'b01_0_0011	: coeff_m1_ff	<= 7'h7d ;
						7'b01_0_0100	: coeff_m1_ff	<= 7'h7c ;
						7'b01_0_0101	: coeff_m1_ff	<= 7'h7c ;
						7'b01_0_0110	: coeff_m1_ff	<= 7'h7c ;
						7'b01_0_0111	: coeff_m1_ff	<= 7'h7b ;
						7'b01_0_1000	: coeff_m1_ff	<= 7'h7b ;
						7'b01_0_1001	: coeff_m1_ff	<= 7'h7b ;
						7'b01_0_1010	: coeff_m1_ff	<= 7'h7b ;
						7'b01_0_1011	: coeff_m1_ff	<= 7'h7b ;
						7'b01_0_1100	: coeff_m1_ff	<= 7'h7b ;
						7'b01_0_1101	: coeff_m1_ff	<= 7'h7b ;
						7'b01_0_1110	: coeff_m1_ff	<= 7'h7b ;
						7'b01_0_1111	: coeff_m1_ff	<= 7'h7b ;
						7'b01_1_0000	: coeff_m1_ff	<= 7'h7c ;
						7'b01_1_0001	: coeff_m1_ff	<= 7'h7c ;
						7'b01_1_0010	: coeff_m1_ff	<= 7'h7c ;
						7'b01_1_0011	: coeff_m1_ff	<= 7'h7c ;
						7'b01_1_0100	: coeff_m1_ff	<= 7'h7d ;
						7'b01_1_0101	: coeff_m1_ff	<= 7'h7d ;
						7'b01_1_0110	: coeff_m1_ff	<= 7'h7d ;
						7'b01_1_0111	: coeff_m1_ff	<= 7'h7e ;
						7'b01_1_1000	: coeff_m1_ff	<= 7'h7e ;
						7'b01_1_1001	: coeff_m1_ff	<= 7'h7e ;
						7'b01_1_1010	: coeff_m1_ff	<= 7'h7f ;
						7'b01_1_1011	: coeff_m1_ff	<= 7'h7f ;
						7'b01_1_1100	: coeff_m1_ff	<= 7'h7f ;
						7'b01_1_1101	: coeff_m1_ff	<= 7'h7f ;
						7'b01_1_1110	: coeff_m1_ff	<= 7'h7f ;
						7'b01_1_1111	: coeff_m1_ff	<= 7'h7f ;

						7'b10_0_0000	: coeff_m1_ff	<= 7'h00 ;
						7'b10_0_0001	: coeff_m1_ff	<= 7'h7e ;
						7'b10_0_0010	: coeff_m1_ff	<= 7'h7d ;
						7'b10_0_0011	: coeff_m1_ff	<= 7'h7c ;
						7'b10_0_0100	: coeff_m1_ff	<= 7'h7b ;
						7'b10_0_0101	: coeff_m1_ff	<= 7'h7a ;
						7'b10_0_0110	: coeff_m1_ff	<= 7'h7a ;
						7'b10_0_0111	: coeff_m1_ff	<= 7'h79 ;
						7'b10_0_1000	: coeff_m1_ff	<= 7'h79 ;
						7'b10_0_1001	: coeff_m1_ff	<= 7'h79 ;
						7'b10_0_1010	: coeff_m1_ff	<= 7'h78 ;
						7'b10_0_1011	: coeff_m1_ff	<= 7'h78 ;
						7'b10_0_1100	: coeff_m1_ff	<= 7'h78 ;
						7'b10_0_1101	: coeff_m1_ff	<= 7'h79 ;
						7'b10_0_1110	: coeff_m1_ff	<= 7'h79 ;
						7'b10_0_1111	: coeff_m1_ff	<= 7'h79 ;
						7'b10_1_0000	: coeff_m1_ff	<= 7'h7a ;
						7'b10_1_0001	: coeff_m1_ff	<= 7'h7a ;
						7'b10_1_0010	: coeff_m1_ff	<= 7'h7a ;
						7'b10_1_0011	: coeff_m1_ff	<= 7'h7b ;
						7'b10_1_0100	: coeff_m1_ff	<= 7'h7b ;
						7'b10_1_0101	: coeff_m1_ff	<= 7'h7c ;
						7'b10_1_0110	: coeff_m1_ff	<= 7'h7c ;
						7'b10_1_0111	: coeff_m1_ff	<= 7'h7d ;
						7'b10_1_1000	: coeff_m1_ff	<= 7'h7d ;
						7'b10_1_1001	: coeff_m1_ff	<= 7'h7e ;
						7'b10_1_1010	: coeff_m1_ff	<= 7'h7e ;
						7'b10_1_1011	: coeff_m1_ff	<= 7'h7f ;
						7'b10_1_1100	: coeff_m1_ff	<= 7'h7f ;
						7'b10_1_1101	: coeff_m1_ff	<= 7'h7f ;
						7'b10_1_1110	: coeff_m1_ff	<= 7'h7f ;
						7'b10_1_1111	: coeff_m1_ff	<= 7'h7f ;

						7'b11_0_0000	: coeff_m1_ff	<= 7'h00 ;
						7'b11_0_0001	: coeff_m1_ff	<= 7'h7e ;
						7'b11_0_0010	: coeff_m1_ff	<= 7'h7c ;
						7'b11_0_0011	: coeff_m1_ff	<= 7'h7b ;
						7'b11_0_0100	: coeff_m1_ff	<= 7'h79 ;
						7'b11_0_0101	: coeff_m1_ff	<= 7'h78 ;
						7'b11_0_0110	: coeff_m1_ff	<= 7'h78 ;
						7'b11_0_0111	: coeff_m1_ff	<= 7'h77 ;
						7'b11_0_1000	: coeff_m1_ff	<= 7'h77 ;
						7'b11_0_1001	: coeff_m1_ff	<= 7'h76 ;
						7'b11_0_1010	: coeff_m1_ff	<= 7'h76 ;
						7'b11_0_1011	: coeff_m1_ff	<= 7'h76 ;
						7'b11_0_1100	: coeff_m1_ff	<= 7'h76 ;
						7'b11_0_1101	: coeff_m1_ff	<= 7'h76 ;
						7'b11_0_1110	: coeff_m1_ff	<= 7'h77 ;
						7'b11_0_1111	: coeff_m1_ff	<= 7'h77 ;
						7'b11_1_0000	: coeff_m1_ff	<= 7'h78 ;
						7'b11_1_0001	: coeff_m1_ff	<= 7'h78 ;
						7'b11_1_0010	: coeff_m1_ff	<= 7'h79 ;
						7'b11_1_0011	: coeff_m1_ff	<= 7'h79 ;
						7'b11_1_0100	: coeff_m1_ff	<= 7'h7a ;
						7'b11_1_0101	: coeff_m1_ff	<= 7'h7b ;
						7'b11_1_0110	: coeff_m1_ff	<= 7'h7b ;
						7'b11_1_0111	: coeff_m1_ff	<= 7'h7c ;
						7'b11_1_1000	: coeff_m1_ff	<= 7'h7d ;
						7'b11_1_1001	: coeff_m1_ff	<= 7'h7d ;
						7'b11_1_1010	: coeff_m1_ff	<= 7'h7e ;
						7'b11_1_1011	: coeff_m1_ff	<= 7'h7e ;
						7'b11_1_1100	: coeff_m1_ff	<= 7'h7f ;
						7'b11_1_1101	: coeff_m1_ff	<= 7'h7f ;
						7'b11_1_1110	: coeff_m1_ff	<= 7'h7f ;
						7'b11_1_1111	: coeff_m1_ff	<= 7'h7f ;
						default			: coeff_m1_ff	<= {7{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_zr_ff		<= 7'h20 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_zr_ff	<= 7'h20 ;
						7'b00_0_0001	: coeff_zr_ff	<= 7'h20 ;
						7'b00_0_0010	: coeff_zr_ff	<= 7'h20 ;
						7'b00_0_0011	: coeff_zr_ff	<= 7'h20 ;
						7'b00_0_0100	: coeff_zr_ff	<= 7'h1f ;
						7'b00_0_0101	: coeff_zr_ff	<= 7'h1f ;
						7'b00_0_0110	: coeff_zr_ff	<= 7'h1e ;
						7'b00_0_0111	: coeff_zr_ff	<= 7'h1d ;
						7'b00_0_1000	: coeff_zr_ff	<= 7'h1c ;
						7'b00_0_1001	: coeff_zr_ff	<= 7'h1b ;
						7'b00_0_1010	: coeff_zr_ff	<= 7'h1a ;
						7'b00_0_1011	: coeff_zr_ff	<= 7'h19 ;
						7'b00_0_1100	: coeff_zr_ff	<= 7'h18 ;
						7'b00_0_1101	: coeff_zr_ff	<= 7'h17 ;
						7'b00_0_1110	: coeff_zr_ff	<= 7'h15 ;
						7'b00_0_1111	: coeff_zr_ff	<= 7'h14 ;
						7'b00_1_0000	: coeff_zr_ff	<= 7'h12 ;
						7'b00_1_0001	: coeff_zr_ff	<= 7'h11 ;
						7'b00_1_0010	: coeff_zr_ff	<= 7'h10 ;
						7'b00_1_0011	: coeff_zr_ff	<= 7'h0e ;
						7'b00_1_0100	: coeff_zr_ff	<= 7'h0d ;
						7'b00_1_0101	: coeff_zr_ff	<= 7'h0c ;
						7'b00_1_0110	: coeff_zr_ff	<= 7'h0b ;
						7'b00_1_0111	: coeff_zr_ff	<= 7'h09 ;
						7'b00_1_1000	: coeff_zr_ff	<= 7'h08 ;
						7'b00_1_1001	: coeff_zr_ff	<= 7'h07 ;
						7'b00_1_1010	: coeff_zr_ff	<= 7'h05 ;
						7'b00_1_1011	: coeff_zr_ff	<= 7'h04 ;
						7'b00_1_1100	: coeff_zr_ff	<= 7'h04 ;
						7'b00_1_1101	: coeff_zr_ff	<= 7'h03 ;
						7'b00_1_1110	: coeff_zr_ff	<= 7'h02 ;
						7'b00_1_1111	: coeff_zr_ff	<= 7'h02 ;

						7'b01_0_0000	: coeff_zr_ff	<= 7'h20 ;
						7'b01_0_0001	: coeff_zr_ff	<= 7'h20 ;
						7'b01_0_0010	: coeff_zr_ff	<= 7'h20 ;
						7'b01_0_0011	: coeff_zr_ff	<= 7'h20 ;
						7'b01_0_0100	: coeff_zr_ff	<= 7'h20 ;
						7'b01_0_0101	: coeff_zr_ff	<= 7'h1f ;
						7'b01_0_0110	: coeff_zr_ff	<= 7'h1e ;
						7'b01_0_0111	: coeff_zr_ff	<= 7'h1e ;
						7'b01_0_1000	: coeff_zr_ff	<= 7'h1d ;
						7'b01_0_1001	: coeff_zr_ff	<= 7'h1c ;
						7'b01_0_1010	: coeff_zr_ff	<= 7'h1b ;
						7'b01_0_1011	: coeff_zr_ff	<= 7'h1a ;
						7'b01_0_1100	: coeff_zr_ff	<= 7'h19 ;
						7'b01_0_1101	: coeff_zr_ff	<= 7'h18 ;
						7'b01_0_1110	: coeff_zr_ff	<= 7'h17 ;
						7'b01_0_1111	: coeff_zr_ff	<= 7'h16 ;
						7'b01_1_0000	: coeff_zr_ff	<= 7'h14 ;
						7'b01_1_0001	: coeff_zr_ff	<= 7'h13 ;
						7'b01_1_0010	: coeff_zr_ff	<= 7'h12 ;
						7'b01_1_0011	: coeff_zr_ff	<= 7'h11 ;
						7'b01_1_0100	: coeff_zr_ff	<= 7'h0f ;
						7'b01_1_0101	: coeff_zr_ff	<= 7'h0e ;
						7'b01_1_0110	: coeff_zr_ff	<= 7'h0d ;
						7'b01_1_0111	: coeff_zr_ff	<= 7'h0b ;
						7'b01_1_1000	: coeff_zr_ff	<= 7'h0a ;
						7'b01_1_1001	: coeff_zr_ff	<= 7'h09 ;
						7'b01_1_1010	: coeff_zr_ff	<= 7'h07 ;
						7'b01_1_1011	: coeff_zr_ff	<= 7'h06 ;
						7'b01_1_1100	: coeff_zr_ff	<= 7'h05 ;
						7'b01_1_1101	: coeff_zr_ff	<= 7'h04 ;
						7'b01_1_1110	: coeff_zr_ff	<= 7'h03 ;
						7'b01_1_1111	: coeff_zr_ff	<= 7'h02 ;

						7'b10_0_0000	: coeff_zr_ff	<= 7'h20 ;
						7'b10_0_0001	: coeff_zr_ff	<= 7'h20 ;
						7'b10_0_0010	: coeff_zr_ff	<= 7'h20 ;
						7'b10_0_0011	: coeff_zr_ff	<= 7'h20 ;
						7'b10_0_0100	: coeff_zr_ff	<= 7'h20 ;
						7'b10_0_0101	: coeff_zr_ff	<= 7'h1f ;
						7'b10_0_0110	: coeff_zr_ff	<= 7'h1f ;
						7'b10_0_0111	: coeff_zr_ff	<= 7'h1e ;
						7'b10_0_1000	: coeff_zr_ff	<= 7'h1e ;
						7'b10_0_1001	: coeff_zr_ff	<= 7'h1d ;
						7'b10_0_1010	: coeff_zr_ff	<= 7'h1c ;
						7'b10_0_1011	: coeff_zr_ff	<= 7'h1b ;
						7'b10_0_1100	: coeff_zr_ff	<= 7'h1b ;
						7'b10_0_1101	: coeff_zr_ff	<= 7'h1a ;
						7'b10_0_1110	: coeff_zr_ff	<= 7'h19 ;
						7'b10_0_1111	: coeff_zr_ff	<= 7'h18 ;
						7'b10_1_0000	: coeff_zr_ff	<= 7'h16 ;
						7'b10_1_0001	: coeff_zr_ff	<= 7'h15 ;
						7'b10_1_0010	: coeff_zr_ff	<= 7'h14 ;
						7'b10_1_0011	: coeff_zr_ff	<= 7'h12 ;
						7'b10_1_0100	: coeff_zr_ff	<= 7'h12 ;
						7'b10_1_0101	: coeff_zr_ff	<= 7'h11 ;
						7'b10_1_0110	: coeff_zr_ff	<= 7'h10 ;
						7'b10_1_0111	: coeff_zr_ff	<= 7'h0d ;
						7'b10_1_1000	: coeff_zr_ff	<= 7'h0c ;
						7'b10_1_1001	: coeff_zr_ff	<= 7'h0b ;
						7'b10_1_1010	: coeff_zr_ff	<= 7'h09 ;
						7'b10_1_1011	: coeff_zr_ff	<= 7'h08 ;
						7'b10_1_1100	: coeff_zr_ff	<= 7'h06 ;
						7'b10_1_1101	: coeff_zr_ff	<= 7'h05 ;
						7'b10_1_1110	: coeff_zr_ff	<= 7'h04 ;
						7'b10_1_1111	: coeff_zr_ff	<= 7'h03 ;

						7'b11_0_0000	: coeff_zr_ff	<= 7'h20 ;
						7'b11_0_0001	: coeff_zr_ff	<= 7'h20 ;
						7'b11_0_0010	: coeff_zr_ff	<= 7'h20 ;
						7'b11_0_0011	: coeff_zr_ff	<= 7'h20 ;
						7'b11_0_0100	: coeff_zr_ff	<= 7'h20 ;
						7'b11_0_0101	: coeff_zr_ff	<= 7'h20 ;
						7'b11_0_0110	: coeff_zr_ff	<= 7'h1f ;
						7'b11_0_0111	: coeff_zr_ff	<= 7'h1f ;
						7'b11_0_1000	: coeff_zr_ff	<= 7'h1e ;
						7'b11_0_1001	: coeff_zr_ff	<= 7'h1e ;
						7'b11_0_1010	: coeff_zr_ff	<= 7'h1d ;
						7'b11_0_1011	: coeff_zr_ff	<= 7'h1d ;
						7'b11_0_1100	: coeff_zr_ff	<= 7'h1c ;
						7'b11_0_1101	: coeff_zr_ff	<= 7'h1b ;
						7'b11_0_1110	: coeff_zr_ff	<= 7'h1a ;
						7'b11_0_1111	: coeff_zr_ff	<= 7'h19 ;
						7'b11_1_0000	: coeff_zr_ff	<= 7'h18 ;
						7'b11_1_0001	: coeff_zr_ff	<= 7'h18 ;
						7'b11_1_0010	: coeff_zr_ff	<= 7'h16 ;
						7'b11_1_0011	: coeff_zr_ff	<= 7'h16 ;
						7'b11_1_0100	: coeff_zr_ff	<= 7'h14 ;
						7'b11_1_0101	: coeff_zr_ff	<= 7'h12 ;
						7'b11_1_0110	: coeff_zr_ff	<= 7'h12 ;
						7'b11_1_0111	: coeff_zr_ff	<= 7'h10 ;
						7'b11_1_1000	: coeff_zr_ff	<= 7'h0e ;
						7'b11_1_1001	: coeff_zr_ff	<= 7'h0d ;
						7'b11_1_1010	: coeff_zr_ff	<= 7'h0b ;
						7'b11_1_1011	: coeff_zr_ff	<= 7'h0a ;
						7'b11_1_1100	: coeff_zr_ff	<= 7'h08 ;
						7'b11_1_1101	: coeff_zr_ff	<= 7'h06 ;
						7'b11_1_1110	: coeff_zr_ff	<= 7'h05 ;
						7'b11_1_1111	: coeff_zr_ff	<= 7'h03 ;
						default			: coeff_zr_ff	<= {7{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p1_ff		<= 7'h00 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_p1_ff	<= 7'h00 ;
						7'b00_0_0001	: coeff_p1_ff	<= 7'h02 ;
						7'b00_0_0010	: coeff_p1_ff	<= 7'h02 ;
						7'b00_0_0011	: coeff_p1_ff	<= 7'h03 ;
						7'b00_0_0100	: coeff_p1_ff	<= 7'h04 ;
						7'b00_0_0101	: coeff_p1_ff	<= 7'h04 ;
						7'b00_0_0110	: coeff_p1_ff	<= 7'h05 ;
						7'b00_0_0111	: coeff_p1_ff	<= 7'h07 ;
						7'b00_0_1000	: coeff_p1_ff	<= 7'h08 ;
						7'b00_0_1001	: coeff_p1_ff	<= 7'h09 ;
						7'b00_0_1010	: coeff_p1_ff	<= 7'h0b ;
						7'b00_0_1011	: coeff_p1_ff	<= 7'h0c ;
						7'b00_0_1100	: coeff_p1_ff	<= 7'h0d ;
						7'b00_0_1101	: coeff_p1_ff	<= 7'h0e ;
						7'b00_0_1110	: coeff_p1_ff	<= 7'h10 ;
						7'b00_0_1111	: coeff_p1_ff	<= 7'h11 ;
						7'b00_1_0000	: coeff_p1_ff	<= 7'h12 ;
						7'b00_1_0001	: coeff_p1_ff	<= 7'h14 ;
						7'b00_1_0010	: coeff_p1_ff	<= 7'h15 ;
						7'b00_1_0011	: coeff_p1_ff	<= 7'h17 ;
						7'b00_1_0100	: coeff_p1_ff	<= 7'h18 ;
						7'b00_1_0101	: coeff_p1_ff	<= 7'h19 ;
						7'b00_1_0110	: coeff_p1_ff	<= 7'h1a ;
						7'b00_1_0111	: coeff_p1_ff	<= 7'h1b ;
						7'b00_1_1000	: coeff_p1_ff	<= 7'h1c ;
						7'b00_1_1001	: coeff_p1_ff	<= 7'h1d ;
						7'b00_1_1010	: coeff_p1_ff	<= 7'h1e ;
						7'b00_1_1011	: coeff_p1_ff	<= 7'h1f ;
						7'b00_1_1100	: coeff_p1_ff	<= 7'h1f ;
						7'b00_1_1101	: coeff_p1_ff	<= 7'h20 ;
						7'b00_1_1110	: coeff_p1_ff	<= 7'h20 ;
						7'b00_1_1111	: coeff_p1_ff	<= 7'h20 ;

						7'b01_0_0000	: coeff_p1_ff	<= 7'h00 ;
						7'b01_0_0001	: coeff_p1_ff	<= 7'h02 ;
						7'b01_0_0010	: coeff_p1_ff	<= 7'h03 ;
						7'b01_0_0011	: coeff_p1_ff	<= 7'h04 ;
						7'b01_0_0100	: coeff_p1_ff	<= 7'h05 ;
						7'b01_0_0101	: coeff_p1_ff	<= 7'h06 ;
						7'b01_0_0110	: coeff_p1_ff	<= 7'h07 ;
						7'b01_0_0111	: coeff_p1_ff	<= 7'h09 ;
						7'b01_0_1000	: coeff_p1_ff	<= 7'h0a ;
						7'b01_0_1001	: coeff_p1_ff	<= 7'h0b ;
						7'b01_0_1010	: coeff_p1_ff	<= 7'h0d ;
						7'b01_0_1011	: coeff_p1_ff	<= 7'h0e ;
						7'b01_0_1100	: coeff_p1_ff	<= 7'h0f ;
						7'b01_0_1101	: coeff_p1_ff	<= 7'h11 ;
						7'b01_0_1110	: coeff_p1_ff	<= 7'h12 ;
						7'b01_0_1111	: coeff_p1_ff	<= 7'h13 ;
						7'b01_1_0000	: coeff_p1_ff	<= 7'h14 ;
						7'b01_1_0001	: coeff_p1_ff	<= 7'h16 ;
						7'b01_1_0010	: coeff_p1_ff	<= 7'h17 ;
						7'b01_1_0011	: coeff_p1_ff	<= 7'h18 ;
						7'b01_1_0100	: coeff_p1_ff	<= 7'h19 ;
						7'b01_1_0101	: coeff_p1_ff	<= 7'h1a ;
						7'b01_1_0110	: coeff_p1_ff	<= 7'h1b ;
						7'b01_1_0111	: coeff_p1_ff	<= 7'h1c ;
						7'b01_1_1000	: coeff_p1_ff	<= 7'h1d ;
						7'b01_1_1001	: coeff_p1_ff	<= 7'h1e ;
						7'b01_1_1010	: coeff_p1_ff	<= 7'h1e ;
						7'b01_1_1011	: coeff_p1_ff	<= 7'h1f ;
						7'b01_1_1100	: coeff_p1_ff	<= 7'h20 ;
						7'b01_1_1101	: coeff_p1_ff	<= 7'h20 ;
						7'b01_1_1110	: coeff_p1_ff	<= 7'h20 ;
						7'b01_1_1111	: coeff_p1_ff	<= 7'h20 ;

						7'b10_0_0000	: coeff_p1_ff	<= 7'h00 ;
						7'b10_0_0001	: coeff_p1_ff	<= 7'h03 ;
						7'b10_0_0010	: coeff_p1_ff	<= 7'h04 ;
						7'b10_0_0011	: coeff_p1_ff	<= 7'h05 ;
						7'b10_0_0100	: coeff_p1_ff	<= 7'h06 ;
						7'b10_0_0101	: coeff_p1_ff	<= 7'h08 ;
						7'b10_0_0110	: coeff_p1_ff	<= 7'h09 ;
						7'b10_0_0111	: coeff_p1_ff	<= 7'h0b ;
						7'b10_0_1000	: coeff_p1_ff	<= 7'h0c ;
						7'b10_0_1001	: coeff_p1_ff	<= 7'h0d ;
						7'b10_0_1010	: coeff_p1_ff	<= 7'h10 ;
						7'b10_0_1011	: coeff_p1_ff	<= 7'h11 ;
						7'b10_0_1100	: coeff_p1_ff	<= 7'h12 ;
						7'b10_0_1101	: coeff_p1_ff	<= 7'h12 ;
						7'b10_0_1110	: coeff_p1_ff	<= 7'h14 ;
						7'b10_0_1111	: coeff_p1_ff	<= 7'h15 ;
						7'b10_1_0000	: coeff_p1_ff	<= 7'h16 ;
						7'b10_1_0001	: coeff_p1_ff	<= 7'h18 ;
						7'b10_1_0010	: coeff_p1_ff	<= 7'h19 ;
						7'b10_1_0011	: coeff_p1_ff	<= 7'h1a ;
						7'b10_1_0100	: coeff_p1_ff	<= 7'h1b ;
						7'b10_1_0101	: coeff_p1_ff	<= 7'h1b ;
						7'b10_1_0110	: coeff_p1_ff	<= 7'h1c ;
						7'b10_1_0111	: coeff_p1_ff	<= 7'h1d ;
						7'b10_1_1000	: coeff_p1_ff	<= 7'h1e ;
						7'b10_1_1001	: coeff_p1_ff	<= 7'h1e ;
						7'b10_1_1010	: coeff_p1_ff	<= 7'h1f ;
						7'b10_1_1011	: coeff_p1_ff	<= 7'h1f ;
						7'b10_1_1100	: coeff_p1_ff	<= 7'h20 ;
						7'b10_1_1101	: coeff_p1_ff	<= 7'h20 ;
						7'b10_1_1110	: coeff_p1_ff	<= 7'h20 ;
						7'b10_1_1111	: coeff_p1_ff	<= 7'h20 ;

						7'b11_0_0000	: coeff_p1_ff	<= 7'h00 ;
						7'b11_0_0001	: coeff_p1_ff	<= 7'h03 ;
						7'b11_0_0010	: coeff_p1_ff	<= 7'h05 ;
						7'b11_0_0011	: coeff_p1_ff	<= 7'h06 ;
						7'b11_0_0100	: coeff_p1_ff	<= 7'h08 ;
						7'b11_0_0101	: coeff_p1_ff	<= 7'h0a ;
						7'b11_0_0110	: coeff_p1_ff	<= 7'h0b ;
						7'b11_0_0111	: coeff_p1_ff	<= 7'h0d ;
						7'b11_0_1000	: coeff_p1_ff	<= 7'h0e ;
						7'b11_0_1001	: coeff_p1_ff	<= 7'h10 ;
						7'b11_0_1010	: coeff_p1_ff	<= 7'h12 ;
						7'b11_0_1011	: coeff_p1_ff	<= 7'h12 ;
						7'b11_0_1100	: coeff_p1_ff	<= 7'h14 ;
						7'b11_0_1101	: coeff_p1_ff	<= 7'h16 ;
						7'b11_0_1110	: coeff_p1_ff	<= 7'h16 ;
						7'b11_0_1111	: coeff_p1_ff	<= 7'h18 ;
						7'b11_1_0000	: coeff_p1_ff	<= 7'h18 ;
						7'b11_1_0001	: coeff_p1_ff	<= 7'h19 ;
						7'b11_1_0010	: coeff_p1_ff	<= 7'h1a ;
						7'b11_1_0011	: coeff_p1_ff	<= 7'h1b ;
						7'b11_1_0100	: coeff_p1_ff	<= 7'h1c ;
						7'b11_1_0101	: coeff_p1_ff	<= 7'h1d ;
						7'b11_1_0110	: coeff_p1_ff	<= 7'h1d ;
						7'b11_1_0111	: coeff_p1_ff	<= 7'h1e ;
						7'b11_1_1000	: coeff_p1_ff	<= 7'h1e ;
						7'b11_1_1001	: coeff_p1_ff	<= 7'h1f ;
						7'b11_1_1010	: coeff_p1_ff	<= 7'h1f ;
						7'b11_1_1011	: coeff_p1_ff	<= 7'h20 ;
						7'b11_1_1100	: coeff_p1_ff	<= 7'h20 ;
						7'b11_1_1101	: coeff_p1_ff	<= 7'h20 ;
						7'b11_1_1110	: coeff_p1_ff	<= 7'h20 ;
						7'b11_1_1111	: coeff_p1_ff	<= 7'h20 ;
						default			: coeff_p1_ff	<= {7{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p2_ff		<= 7'h00 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_p2_ff	<= 7'h00 ;
						7'b00_0_0001	: coeff_p2_ff	<= 7'h7f ;
						7'b00_0_0010	: coeff_p2_ff	<= 7'h7f ;
						7'b00_0_0011	: coeff_p2_ff	<= 7'h7f ;
						7'b00_0_0100	: coeff_p2_ff	<= 7'h7f ;
						7'b00_0_0101	: coeff_p2_ff	<= 7'h7f ;
						7'b00_0_0110	: coeff_p2_ff	<= 7'h7f ;
						7'b00_0_0111	: coeff_p2_ff	<= 7'h7f ;
						7'b00_0_1000	: coeff_p2_ff	<= 7'h7f ;
						7'b00_0_1001	: coeff_p2_ff	<= 7'h7f ;
						7'b00_0_1010	: coeff_p2_ff	<= 7'h7e ;
						7'b00_0_1011	: coeff_p2_ff	<= 7'h7e ;
						7'b00_0_1100	: coeff_p2_ff	<= 7'h7e ;
						7'b00_0_1101	: coeff_p2_ff	<= 7'h7e ;
						7'b00_0_1110	: coeff_p2_ff	<= 7'h7e ;
						7'b00_0_1111	: coeff_p2_ff	<= 7'h7e ;
						7'b00_1_0000	: coeff_p2_ff	<= 7'h7e ;
						7'b00_1_0001	: coeff_p2_ff	<= 7'h7d ;
						7'b00_1_0010	: coeff_p2_ff	<= 7'h7d ;
						7'b00_1_0011	: coeff_p2_ff	<= 7'h7d ;
						7'b00_1_0100	: coeff_p2_ff	<= 7'h7d ;
						7'b00_1_0101	: coeff_p2_ff	<= 7'h7d ;
						7'b00_1_0110	: coeff_p2_ff	<= 7'h7d ;
						7'b00_1_0111	: coeff_p2_ff	<= 7'h7d ;
						7'b00_1_1000	: coeff_p2_ff	<= 7'h7d ;
						7'b00_1_1001	: coeff_p2_ff	<= 7'h7d ;
						7'b00_1_1010	: coeff_p2_ff	<= 7'h7e ;
						7'b00_1_1011	: coeff_p2_ff	<= 7'h7e ;
						7'b00_1_1100	: coeff_p2_ff	<= 7'h7e ;
						7'b00_1_1101	: coeff_p2_ff	<= 7'h7e ;
						7'b00_1_1110	: coeff_p2_ff	<= 7'h7f ;
						7'b00_1_1111	: coeff_p2_ff	<= 7'h7f ;

						7'b01_0_0000	: coeff_p2_ff	<= 7'h00 ;
						7'b01_0_0001	: coeff_p2_ff	<= 7'h7f ;
						7'b01_0_0010	: coeff_p2_ff	<= 7'h7f ;
						7'b01_0_0011	: coeff_p2_ff	<= 7'h7f ;
						7'b01_0_0100	: coeff_p2_ff	<= 7'h7f ;
						7'b01_0_0101	: coeff_p2_ff	<= 7'h7f ;
						7'b01_0_0110	: coeff_p2_ff	<= 7'h7f ;
						7'b01_0_0111	: coeff_p2_ff	<= 7'h7e ;
						7'b01_0_1000	: coeff_p2_ff	<= 7'h7e ;
						7'b01_0_1001	: coeff_p2_ff	<= 7'h7e ;
						7'b01_0_1010	: coeff_p2_ff	<= 7'h7d ;
						7'b01_0_1011	: coeff_p2_ff	<= 7'h7d ;
						7'b01_0_1100	: coeff_p2_ff	<= 7'h7d ;
						7'b01_0_1101	: coeff_p2_ff	<= 7'h7c ;
						7'b01_0_1110	: coeff_p2_ff	<= 7'h7c ;
						7'b01_0_1111	: coeff_p2_ff	<= 7'h7c ;
						7'b01_1_0000	: coeff_p2_ff	<= 7'h7c ;
						7'b01_1_0001	: coeff_p2_ff	<= 7'h7b ;
						7'b01_1_0010	: coeff_p2_ff	<= 7'h7b ;
						7'b01_1_0011	: coeff_p2_ff	<= 7'h7b ;
						7'b01_1_0100	: coeff_p2_ff	<= 7'h7b ;
						7'b01_1_0101	: coeff_p2_ff	<= 7'h7b ;
						7'b01_1_0110	: coeff_p2_ff	<= 7'h7b ;
						7'b01_1_0111	: coeff_p2_ff	<= 7'h7b ;
						7'b01_1_1000	: coeff_p2_ff	<= 7'h7b ;
						7'b01_1_1001	: coeff_p2_ff	<= 7'h7b ;
						7'b01_1_1010	: coeff_p2_ff	<= 7'h7c ;
						7'b01_1_1011	: coeff_p2_ff	<= 7'h7c ;
						7'b01_1_1100	: coeff_p2_ff	<= 7'h7c ;
						7'b01_1_1101	: coeff_p2_ff	<= 7'h7d ;
						7'b01_1_1110	: coeff_p2_ff	<= 7'h7e ;
						7'b01_1_1111	: coeff_p2_ff	<= 7'h7f ;

						7'b10_0_0000	: coeff_p2_ff	<= 7'h00 ;
						7'b10_0_0001	: coeff_p2_ff	<= 7'h7f ;
						7'b10_0_0010	: coeff_p2_ff	<= 7'h7f ;
						7'b10_0_0011	: coeff_p2_ff	<= 7'h7f ;
						7'b10_0_0100	: coeff_p2_ff	<= 7'h7f ;
						7'b10_0_0101	: coeff_p2_ff	<= 7'h7f ;
						7'b10_0_0110	: coeff_p2_ff	<= 7'h7e ;
						7'b10_0_0111	: coeff_p2_ff	<= 7'h7e ;
						7'b10_0_1000	: coeff_p2_ff	<= 7'h7d ;
						7'b10_0_1001	: coeff_p2_ff	<= 7'h7d ;
						7'b10_0_1010	: coeff_p2_ff	<= 7'h7c ;
						7'b10_0_1011	: coeff_p2_ff	<= 7'h7c ;
						7'b10_0_1100	: coeff_p2_ff	<= 7'h7b ;
						7'b10_0_1101	: coeff_p2_ff	<= 7'h7b ;
						7'b10_0_1110	: coeff_p2_ff	<= 7'h7a ;
						7'b10_0_1111	: coeff_p2_ff	<= 7'h7a ;
						7'b10_1_0000	: coeff_p2_ff	<= 7'h7a ;
						7'b10_1_0001	: coeff_p2_ff	<= 7'h79 ;
						7'b10_1_0010	: coeff_p2_ff	<= 7'h79 ;
						7'b10_1_0011	: coeff_p2_ff	<= 7'h79 ;
						7'b10_1_0100	: coeff_p2_ff	<= 7'h78 ;
						7'b10_1_0101	: coeff_p2_ff	<= 7'h78 ;
						7'b10_1_0110	: coeff_p2_ff	<= 7'h78 ;
						7'b10_1_0111	: coeff_p2_ff	<= 7'h79 ;
						7'b10_1_1000	: coeff_p2_ff	<= 7'h79 ;
						7'b10_1_1001	: coeff_p2_ff	<= 7'h79 ;
						7'b10_1_1010	: coeff_p2_ff	<= 7'h7a ;
						7'b10_1_1011	: coeff_p2_ff	<= 7'h7a ;
						7'b10_1_1100	: coeff_p2_ff	<= 7'h7b ;
						7'b10_1_1101	: coeff_p2_ff	<= 7'h7c ;
						7'b10_1_1110	: coeff_p2_ff	<= 7'h7d ;
						7'b10_1_1111	: coeff_p2_ff	<= 7'h7e ;

						7'b11_0_0000	: coeff_p2_ff	<= 7'h00 ;
						7'b11_0_0001	: coeff_p2_ff	<= 7'h7f ;
						7'b11_0_0010	: coeff_p2_ff	<= 7'h7f ;
						7'b11_0_0011	: coeff_p2_ff	<= 7'h7f ;
						7'b11_0_0100	: coeff_p2_ff	<= 7'h7f ;
						7'b11_0_0101	: coeff_p2_ff	<= 7'h7e ;
						7'b11_0_0110	: coeff_p2_ff	<= 7'h7e ;
						7'b11_0_0111	: coeff_p2_ff	<= 7'h7d ;
						7'b11_0_1000	: coeff_p2_ff	<= 7'h7d ;
						7'b11_0_1001	: coeff_p2_ff	<= 7'h7c ;
						7'b11_0_1010	: coeff_p2_ff	<= 7'h7b ;
						7'b11_0_1011	: coeff_p2_ff	<= 7'h7b ;
						7'b11_0_1100	: coeff_p2_ff	<= 7'h7a ;
						7'b11_0_1101	: coeff_p2_ff	<= 7'h79 ;
						7'b11_0_1110	: coeff_p2_ff	<= 7'h79 ;
						7'b11_0_1111	: coeff_p2_ff	<= 7'h78 ;
						7'b11_1_0000	: coeff_p2_ff	<= 7'h78 ;
						7'b11_1_0001	: coeff_p2_ff	<= 7'h77 ;
						7'b11_1_0010	: coeff_p2_ff	<= 7'h77 ;
						7'b11_1_0011	: coeff_p2_ff	<= 7'h76 ;
						7'b11_1_0100	: coeff_p2_ff	<= 7'h76 ;
						7'b11_1_0101	: coeff_p2_ff	<= 7'h76 ;
						7'b11_1_0110	: coeff_p2_ff	<= 7'h76 ;
						7'b11_1_0111	: coeff_p2_ff	<= 7'h76 ;
						7'b11_1_1000	: coeff_p2_ff	<= 7'h77 ;
						7'b11_1_1001	: coeff_p2_ff	<= 7'h77 ;
						7'b11_1_1010	: coeff_p2_ff	<= 7'h78 ;
						7'b11_1_1011	: coeff_p2_ff	<= 7'h78 ;
						7'b11_1_1100	: coeff_p2_ff	<= 7'h79 ;
						7'b11_1_1101	: coeff_p2_ff	<= 7'h7b ;
						7'b11_1_1110	: coeff_p2_ff	<= 7'h7c ;
						7'b11_1_1111	: coeff_p2_ff	<= 7'h7e ;
						default			: coeff_p2_ff	<= {7{1'bx}} ;
					endcase
				end
			end
		end
		else if ( Q_BIT == 8 ) begin : table_8bit
			//-------------------- Table Value Format -----------------------//
			//	 { Sign(1bit), Integral Part(1bit), Fraction Part(7bit) }	 //
			//---------------------------------------------------------------//
			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_m1_ff		<= 9'h000 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_m1_ff	<= 9'h000 ;
						7'b00_0_0001	: coeff_m1_ff	<= 9'h1fe ;
						7'b00_0_0010	: coeff_m1_ff	<= 9'h1fc ;
						7'b00_0_0011	: coeff_m1_ff	<= 9'h1fb ;
						7'b00_0_0100	: coeff_m1_ff	<= 9'h1f9 ;
						7'b00_0_0101	: coeff_m1_ff	<= 9'h1f8 ;
						7'b00_0_0110	: coeff_m1_ff	<= 9'h1f8 ;
						7'b00_0_0111	: coeff_m1_ff	<= 9'h1f7 ;
						7'b00_0_1000	: coeff_m1_ff	<= 9'h1f7 ;
						7'b00_0_1001	: coeff_m1_ff	<= 9'h1f6 ;
						7'b00_0_1010	: coeff_m1_ff	<= 9'h1f6 ;
						7'b00_0_1011	: coeff_m1_ff	<= 9'h1f6 ;
						7'b00_0_1100	: coeff_m1_ff	<= 9'h1f6 ;
						7'b00_0_1101	: coeff_m1_ff	<= 9'h1f6 ;
						7'b00_0_1110	: coeff_m1_ff	<= 9'h1f7 ;
						7'b00_0_1111	: coeff_m1_ff	<= 9'h1f7 ;
						7'b00_1_0000	: coeff_m1_ff	<= 9'h1f8 ;
						7'b00_1_0001	: coeff_m1_ff	<= 9'h1f8 ;
						7'b00_1_0010	: coeff_m1_ff	<= 9'h1f9 ;
						7'b00_1_0011	: coeff_m1_ff	<= 9'h1f9 ;
						7'b00_1_0100	: coeff_m1_ff	<= 9'h1fa ;
						7'b00_1_0101	: coeff_m1_ff	<= 9'h1fb ;
						7'b00_1_0110	: coeff_m1_ff	<= 9'h1fb ;
						7'b00_1_0111	: coeff_m1_ff	<= 9'h1fc ;
						7'b00_1_1000	: coeff_m1_ff	<= 9'h1fd ;
						7'b00_1_1001	: coeff_m1_ff	<= 9'h1fd ;
						7'b00_1_1010	: coeff_m1_ff	<= 9'h1fe ;
						7'b00_1_1011	: coeff_m1_ff	<= 9'h1fe ;
						7'b00_1_1100	: coeff_m1_ff	<= 9'h1ff ;
						7'b00_1_1101	: coeff_m1_ff	<= 9'h1ff ;
						7'b00_1_1110	: coeff_m1_ff	<= 9'h1ff ;
						7'b00_1_1111	: coeff_m1_ff	<= 9'h1ff ;

						7'b01_0_0000	: coeff_m1_ff	<= 9'h000 ;
						7'b01_0_0001	: coeff_m1_ff	<= 9'h1fc ;
						7'b01_0_0010	: coeff_m1_ff	<= 9'h1f8 ;
						7'b01_0_0011	: coeff_m1_ff	<= 9'h1f6 ;
						7'b01_0_0100	: coeff_m1_ff	<= 9'h1f3 ;
						7'b01_0_0101	: coeff_m1_ff	<= 9'h1f1 ;
						7'b01_0_0110	: coeff_m1_ff	<= 9'h1f0 ;
						7'b01_0_0111	: coeff_m1_ff	<= 9'h1ee ;
						7'b01_0_1000	: coeff_m1_ff	<= 9'h1ee ;
						7'b01_0_1001	: coeff_m1_ff	<= 9'h1ed ;
						7'b01_0_1010	: coeff_m1_ff	<= 9'h1ed ;
						7'b01_0_1011	: coeff_m1_ff	<= 9'h1ed ;
						7'b01_0_1100	: coeff_m1_ff	<= 9'h1ed ;
						7'b01_0_1101	: coeff_m1_ff	<= 9'h1ed ;
						7'b01_0_1110	: coeff_m1_ff	<= 9'h1ee ;
						7'b01_0_1111	: coeff_m1_ff	<= 9'h1ef ;
						7'b01_1_0000	: coeff_m1_ff	<= 9'h1f0 ;
						7'b01_1_0001	: coeff_m1_ff	<= 9'h1f1 ;
						7'b01_1_0010	: coeff_m1_ff	<= 9'h1f2 ;
						7'b01_1_0011	: coeff_m1_ff	<= 9'h1f3 ;
						7'b01_1_0100	: coeff_m1_ff	<= 9'h1f4 ;
						7'b01_1_0101	: coeff_m1_ff	<= 9'h1f6 ;
						7'b01_1_0110	: coeff_m1_ff	<= 9'h1f7 ;
						7'b01_1_0111	: coeff_m1_ff	<= 9'h1f8 ;
						7'b01_1_1000	: coeff_m1_ff	<= 9'h1fa ;
						7'b01_1_1001	: coeff_m1_ff	<= 9'h1fb ;
						7'b01_1_1010	: coeff_m1_ff	<= 9'h1fc ;
						7'b01_1_1011	: coeff_m1_ff	<= 9'h1fd ;
						7'b01_1_1100	: coeff_m1_ff	<= 9'h1fe ;
						7'b01_1_1101	: coeff_m1_ff	<= 9'h1fe ;
						7'b01_1_1110	: coeff_m1_ff	<= 9'h1ff ;
						7'b01_1_1111	: coeff_m1_ff	<= 9'h1ff ;

						7'b10_0_0000	: coeff_m1_ff	<= 9'h000 ;
						7'b10_0_0001	: coeff_m1_ff	<= 9'h1fa ;
						7'b10_0_0010	: coeff_m1_ff	<= 9'h1f5 ;
						7'b10_0_0011	: coeff_m1_ff	<= 9'h1f1 ;
						7'b10_0_0100	: coeff_m1_ff	<= 9'h1ed ;
						7'b10_0_0101	: coeff_m1_ff	<= 9'h1ea ;
						7'b10_0_0110	: coeff_m1_ff	<= 9'h1e8 ;
						7'b10_0_0111	: coeff_m1_ff	<= 9'h1e6 ;
						7'b10_0_1000	: coeff_m1_ff	<= 9'h1e5 ;
						7'b10_0_1001	: coeff_m1_ff	<= 9'h1e4 ;
						7'b10_0_1010	: coeff_m1_ff	<= 9'h1e3 ;
						7'b10_0_1011	: coeff_m1_ff	<= 9'h1e3 ;
						7'b10_0_1100	: coeff_m1_ff	<= 9'h1e3 ;
						7'b10_0_1101	: coeff_m1_ff	<= 9'h1e4 ;
						7'b10_0_1110	: coeff_m1_ff	<= 9'h1e5 ;
						7'b10_0_1111	: coeff_m1_ff	<= 9'h1e6 ;
						7'b10_1_0000	: coeff_m1_ff	<= 9'h1e8 ;
						7'b10_1_0001	: coeff_m1_ff	<= 9'h1e9 ;
						7'b10_1_0010	: coeff_m1_ff	<= 9'h1eb ;
						7'b10_1_0011	: coeff_m1_ff	<= 9'h1ed ;
						7'b10_1_0100	: coeff_m1_ff	<= 9'h1ef ;
						7'b10_1_0101	: coeff_m1_ff	<= 9'h1f1 ;
						7'b10_1_0110	: coeff_m1_ff	<= 9'h1f3 ;
						7'b10_1_0111	: coeff_m1_ff	<= 9'h1f5 ;
						7'b10_1_1000	: coeff_m1_ff	<= 9'h1f7 ;
						7'b10_1_1001	: coeff_m1_ff	<= 9'h1f8 ;
						7'b10_1_1010	: coeff_m1_ff	<= 9'h1fa ;
						7'b10_1_1011	: coeff_m1_ff	<= 9'h1fc ;
						7'b10_1_1100	: coeff_m1_ff	<= 9'h1fd ;
						7'b10_1_1101	: coeff_m1_ff	<= 9'h1fe ;
						7'b10_1_1110	: coeff_m1_ff	<= 9'h1ff ;
						7'b10_1_1111	: coeff_m1_ff	<= 9'h1ff ;

						7'b11_0_0000	: coeff_m1_ff	<= 9'h000 ;
						7'b11_0_0001	: coeff_m1_ff	<= 9'h1f8 ;
						7'b11_0_0010	: coeff_m1_ff	<= 9'h1f1 ;
						7'b11_0_0011	: coeff_m1_ff	<= 9'h1ec ;
						7'b11_0_0100	: coeff_m1_ff	<= 9'h1e7 ;
						7'b11_0_0101	: coeff_m1_ff	<= 9'h1e3 ;
						7'b11_0_0110	: coeff_m1_ff	<= 9'h1e0 ;
						7'b11_0_0111	: coeff_m1_ff	<= 9'h1dd ;
						7'b11_0_1000	: coeff_m1_ff	<= 9'h1dc ;
						7'b11_0_1001	: coeff_m1_ff	<= 9'h1da ;
						7'b11_0_1010	: coeff_m1_ff	<= 9'h1da ;
						7'b11_0_1011	: coeff_m1_ff	<= 9'h1da ;
						7'b11_0_1100	: coeff_m1_ff	<= 9'h1da ;
						7'b11_0_1101	: coeff_m1_ff	<= 9'h1db ;
						7'b11_0_1110	: coeff_m1_ff	<= 9'h1dc ;
						7'b11_0_1111	: coeff_m1_ff	<= 9'h1de ;
						7'b11_1_0000	: coeff_m1_ff	<= 9'h1e0 ;
						7'b11_1_0001	: coeff_m1_ff	<= 9'h1e2 ;
						7'b11_1_0010	: coeff_m1_ff	<= 9'h1e4 ;
						7'b11_1_0011	: coeff_m1_ff	<= 9'h1e6 ;
						7'b11_1_0100	: coeff_m1_ff	<= 9'h1e9 ;
						7'b11_1_0101	: coeff_m1_ff	<= 9'h1ec ;
						7'b11_1_0110	: coeff_m1_ff	<= 9'h1ee ;
						7'b11_1_0111	: coeff_m1_ff	<= 9'h1f1 ;
						7'b11_1_1000	: coeff_m1_ff	<= 9'h1f4 ;
						7'b11_1_1001	: coeff_m1_ff	<= 9'h1f6 ;
						7'b11_1_1010	: coeff_m1_ff	<= 9'h1f8 ;
						7'b11_1_1011	: coeff_m1_ff	<= 9'h1fa ;
						7'b11_1_1100	: coeff_m1_ff	<= 9'h1fc ;
						7'b11_1_1101	: coeff_m1_ff	<= 9'h1fd ;
						7'b11_1_1110	: coeff_m1_ff	<= 9'h1ff ;
						7'b11_1_1111	: coeff_m1_ff	<= 9'h1ff ;
						default			: coeff_m1_ff	<= {9{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_zr_ff		<= 9'h080 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_zr_ff	<= 9'h080 ;
						7'b00_0_0001	: coeff_zr_ff	<= 9'h080 ;
						7'b00_0_0010	: coeff_zr_ff	<= 9'h07f ;
						7'b00_0_0011	: coeff_zr_ff	<= 9'h07e ;
						7'b00_0_0100	: coeff_zr_ff	<= 9'h07c ;
						7'b00_0_0101	: coeff_zr_ff	<= 9'h079 ;
						7'b00_0_0110	: coeff_zr_ff	<= 9'h077 ;
						7'b00_0_0111	: coeff_zr_ff	<= 9'h073 ;
						7'b00_0_1000	: coeff_zr_ff	<= 9'h06f ;
						7'b00_0_1001	: coeff_zr_ff	<= 9'h06b ;
						7'b00_0_1010	: coeff_zr_ff	<= 9'h067 ;
						7'b00_0_1011	: coeff_zr_ff	<= 9'h062 ;
						7'b00_0_1100	: coeff_zr_ff	<= 9'h05e ;
						7'b00_0_1101	: coeff_zr_ff	<= 9'h059 ;
						7'b00_0_1110	: coeff_zr_ff	<= 9'h053 ;
						7'b00_0_1111	: coeff_zr_ff	<= 9'h04e ;
						7'b00_1_0000	: coeff_zr_ff	<= 9'h048 ;
						7'b00_1_0001	: coeff_zr_ff	<= 9'h043 ;
						7'b00_1_0010	: coeff_zr_ff	<= 9'h03d ;
						7'b00_1_0011	: coeff_zr_ff	<= 9'h038 ;
						7'b00_1_0100	: coeff_zr_ff	<= 9'h032 ;
						7'b00_1_0101	: coeff_zr_ff	<= 9'h02d ;
						7'b00_1_0110	: coeff_zr_ff	<= 9'h028 ;
						7'b00_1_0111	: coeff_zr_ff	<= 9'h023 ;
						7'b00_1_1000	: coeff_zr_ff	<= 9'h01d ;
						7'b00_1_1001	: coeff_zr_ff	<= 9'h019 ;
						7'b00_1_1010	: coeff_zr_ff	<= 9'h013 ;
						7'b00_1_1011	: coeff_zr_ff	<= 9'h011 ;
						7'b00_1_1100	: coeff_zr_ff	<= 9'h00c ;
						7'b00_1_1101	: coeff_zr_ff	<= 9'h008 ;
						7'b00_1_1110	: coeff_zr_ff	<= 9'h006 ;
						7'b00_1_1111	: coeff_zr_ff	<= 9'h003 ;

						7'b01_0_0000	: coeff_zr_ff	<= 9'h080 ;
						7'b01_0_0001	: coeff_zr_ff	<= 9'h080 ;
						7'b01_0_0010	: coeff_zr_ff	<= 9'h080 ;
						7'b01_0_0011	: coeff_zr_ff	<= 9'h07e ;
						7'b01_0_0100	: coeff_zr_ff	<= 9'h07d ;
						7'b01_0_0101	: coeff_zr_ff	<= 9'h07b ;
						7'b01_0_0110	: coeff_zr_ff	<= 9'h078 ;
						7'b01_0_0111	: coeff_zr_ff	<= 9'h076 ;
						7'b01_0_1000	: coeff_zr_ff	<= 9'h072 ;
						7'b01_0_1001	: coeff_zr_ff	<= 9'h06f ;
						7'b01_0_1010	: coeff_zr_ff	<= 9'h06b ;
						7'b01_0_1011	: coeff_zr_ff	<= 9'h067 ;
						7'b01_0_1100	: coeff_zr_ff	<= 9'h063 ;
						7'b01_0_1101	: coeff_zr_ff	<= 9'h05f ;
						7'b01_0_1110	: coeff_zr_ff	<= 9'h05a ;
						7'b01_0_1111	: coeff_zr_ff	<= 9'h055 ;
						7'b01_1_0000	: coeff_zr_ff	<= 9'h050 ;
						7'b01_1_0001	: coeff_zr_ff	<= 9'h04b ;
						7'b01_1_0010	: coeff_zr_ff	<= 9'h046 ;
						7'b01_1_0011	: coeff_zr_ff	<= 9'h041 ;
						7'b01_1_0100	: coeff_zr_ff	<= 9'h03c ;
						7'b01_1_0101	: coeff_zr_ff	<= 9'h036 ;
						7'b01_1_0110	: coeff_zr_ff	<= 9'h031 ;
						7'b01_1_0111	: coeff_zr_ff	<= 9'h02c ;
						7'b01_1_1000	: coeff_zr_ff	<= 9'h026 ;
						7'b01_1_1001	: coeff_zr_ff	<= 9'h021 ;
						7'b01_1_1010	: coeff_zr_ff	<= 9'h01c ;
						7'b01_1_1011	: coeff_zr_ff	<= 9'h017 ;
						7'b01_1_1100	: coeff_zr_ff	<= 9'h012 ;
						7'b01_1_1101	: coeff_zr_ff	<= 9'h00e ;
						7'b01_1_1110	: coeff_zr_ff	<= 9'h009 ;
						7'b01_1_1111	: coeff_zr_ff	<= 9'h005 ;

						7'b10_0_0000	: coeff_zr_ff	<= 9'h080 ;
						7'b10_0_0001	: coeff_zr_ff	<= 9'h080 ;
						7'b10_0_0010	: coeff_zr_ff	<= 9'h080 ;
						7'b10_0_0011	: coeff_zr_ff	<= 9'h07f ;
						7'b10_0_0100	: coeff_zr_ff	<= 9'h07e ;
						7'b10_0_0101	: coeff_zr_ff	<= 9'h07c ;
						7'b10_0_0110	: coeff_zr_ff	<= 9'h07a ;
						7'b10_0_0111	: coeff_zr_ff	<= 9'h078 ;
						7'b10_0_1000	: coeff_zr_ff	<= 9'h075 ;
						7'b10_0_1001	: coeff_zr_ff	<= 9'h073 ;
						7'b10_0_1010	: coeff_zr_ff	<= 9'h070 ;
						7'b10_0_1011	: coeff_zr_ff	<= 9'h06c ;
						7'b10_0_1100	: coeff_zr_ff	<= 9'h069 ;
						7'b10_0_1101	: coeff_zr_ff	<= 9'h065 ;
						7'b10_0_1110	: coeff_zr_ff	<= 9'h061 ;
						7'b10_0_1111	: coeff_zr_ff	<= 9'h05d ;
						7'b10_1_0000	: coeff_zr_ff	<= 9'h058 ;
						7'b10_1_0001	: coeff_zr_ff	<= 9'h054 ;
						7'b10_1_0010	: coeff_zr_ff	<= 9'h04f ;
						7'b10_1_0011	: coeff_zr_ff	<= 9'h04a ;
						7'b10_1_0100	: coeff_zr_ff	<= 9'h045 ;
						7'b10_1_0101	: coeff_zr_ff	<= 9'h040 ;
						7'b10_1_0110	: coeff_zr_ff	<= 9'h03a ;
						7'b10_1_0111	: coeff_zr_ff	<= 9'h034 ;
						7'b10_1_1000	: coeff_zr_ff	<= 9'h02f ;
						7'b10_1_1001	: coeff_zr_ff	<= 9'h02a ;
						7'b10_1_1010	: coeff_zr_ff	<= 9'h024 ;
						7'b10_1_1011	: coeff_zr_ff	<= 9'h01e ;
						7'b10_1_1100	: coeff_zr_ff	<= 9'h018 ;
						7'b10_1_1101	: coeff_zr_ff	<= 9'h012 ;
						7'b10_1_1110	: coeff_zr_ff	<= 9'h00c ;
						7'b10_1_1111	: coeff_zr_ff	<= 9'h007 ;

						7'b11_0_0000	: coeff_zr_ff	<= 9'h080 ;
						7'b11_0_0001	: coeff_zr_ff	<= 9'h080 ;
						7'b11_0_0010	: coeff_zr_ff	<= 9'h080 ;
						7'b11_0_0011	: coeff_zr_ff	<= 9'h07f ;
						7'b11_0_0100	: coeff_zr_ff	<= 9'h07e ;
						7'b11_0_0101	: coeff_zr_ff	<= 9'h07d ;
						7'b11_0_0110	: coeff_zr_ff	<= 9'h07c ;
						7'b11_0_0111	: coeff_zr_ff	<= 9'h07a ;
						7'b11_0_1000	: coeff_zr_ff	<= 9'h078 ;
						7'b11_0_1001	: coeff_zr_ff	<= 9'h076 ;
						7'b11_0_1010	: coeff_zr_ff	<= 9'h074 ;
						7'b11_0_1011	: coeff_zr_ff	<= 9'h071 ;
						7'b11_0_1100	: coeff_zr_ff	<= 9'h06e ;
						7'b11_0_1101	: coeff_zr_ff	<= 9'h06b ;
						7'b11_0_1110	: coeff_zr_ff	<= 9'h068 ;
						7'b11_0_1111	: coeff_zr_ff	<= 9'h064 ;
						7'b11_1_0000	: coeff_zr_ff	<= 9'h060 ;
						7'b11_1_0001	: coeff_zr_ff	<= 9'h05c ;
						7'b11_1_0010	: coeff_zr_ff	<= 9'h058 ;
						7'b11_1_0011	: coeff_zr_ff	<= 9'h054 ;
						7'b11_1_0100	: coeff_zr_ff	<= 9'h04f ;
						7'b11_1_0101	: coeff_zr_ff	<= 9'h049 ;
						7'b11_1_0110	: coeff_zr_ff	<= 9'h044 ;
						7'b11_1_0111	: coeff_zr_ff	<= 9'h03f ;
						7'b11_1_1000	: coeff_zr_ff	<= 9'h038 ;
						7'b11_1_1001	: coeff_zr_ff	<= 9'h033 ;
						7'b11_1_1010	: coeff_zr_ff	<= 9'h02c ;
						7'b11_1_1011	: coeff_zr_ff	<= 9'h026 ;
						7'b11_1_1100	: coeff_zr_ff	<= 9'h01f ;
						7'b11_1_1101	: coeff_zr_ff	<= 9'h018 ;
						7'b11_1_1110	: coeff_zr_ff	<= 9'h010 ;
						7'b11_1_1111	: coeff_zr_ff	<= 9'h009 ;
						default			: coeff_zr_ff	<= {9{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p1_ff		<= 9'h000 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_p1_ff	<= 9'h000 ;
						7'b00_0_0001	: coeff_p1_ff	<= 9'h003 ;
						7'b00_0_0010	: coeff_p1_ff	<= 9'h006 ;
						7'b00_0_0011	: coeff_p1_ff	<= 9'h008 ;
						7'b00_0_0100	: coeff_p1_ff	<= 9'h00c ;
						7'b00_0_0101	: coeff_p1_ff	<= 9'h011 ;
						7'b00_0_0110	: coeff_p1_ff	<= 9'h013 ;
						7'b00_0_0111	: coeff_p1_ff	<= 9'h019 ;
						7'b00_0_1000	: coeff_p1_ff	<= 9'h01d ;
						7'b00_0_1001	: coeff_p1_ff	<= 9'h023 ;
						7'b00_0_1010	: coeff_p1_ff	<= 9'h028 ;
						7'b00_0_1011	: coeff_p1_ff	<= 9'h02d ;
						7'b00_0_1100	: coeff_p1_ff	<= 9'h032 ;
						7'b00_0_1101	: coeff_p1_ff	<= 9'h038 ;
						7'b00_0_1110	: coeff_p1_ff	<= 9'h03d ;
						7'b00_0_1111	: coeff_p1_ff	<= 9'h043 ;
						7'b00_1_0000	: coeff_p1_ff	<= 9'h048 ;
						7'b00_1_0001	: coeff_p1_ff	<= 9'h04e ;
						7'b00_1_0010	: coeff_p1_ff	<= 9'h053 ;
						7'b00_1_0011	: coeff_p1_ff	<= 9'h059 ;
						7'b00_1_0100	: coeff_p1_ff	<= 9'h05e ;
						7'b00_1_0101	: coeff_p1_ff	<= 9'h062 ;
						7'b00_1_0110	: coeff_p1_ff	<= 9'h067 ;
						7'b00_1_0111	: coeff_p1_ff	<= 9'h06b ;
						7'b00_1_1000	: coeff_p1_ff	<= 9'h06f ;
						7'b00_1_1001	: coeff_p1_ff	<= 9'h073 ;
						7'b00_1_1010	: coeff_p1_ff	<= 9'h077 ;
						7'b00_1_1011	: coeff_p1_ff	<= 9'h079 ;
						7'b00_1_1100	: coeff_p1_ff	<= 9'h07c ;
						7'b00_1_1101	: coeff_p1_ff	<= 9'h07e ;
						7'b00_1_1110	: coeff_p1_ff	<= 9'h07f ;
						7'b00_1_1111	: coeff_p1_ff	<= 9'h080 ;

						7'b01_0_0000	: coeff_p1_ff	<= 9'h000 ;
						7'b01_0_0001	: coeff_p1_ff	<= 9'h005 ;
						7'b01_0_0010	: coeff_p1_ff	<= 9'h009 ;
						7'b01_0_0011	: coeff_p1_ff	<= 9'h00e ;
						7'b01_0_0100	: coeff_p1_ff	<= 9'h012 ;
						7'b01_0_0101	: coeff_p1_ff	<= 9'h017 ;
						7'b01_0_0110	: coeff_p1_ff	<= 9'h01c ;
						7'b01_0_0111	: coeff_p1_ff	<= 9'h021 ;
						7'b01_0_1000	: coeff_p1_ff	<= 9'h026 ;
						7'b01_0_1001	: coeff_p1_ff	<= 9'h02c ;
						7'b01_0_1010	: coeff_p1_ff	<= 9'h031 ;
						7'b01_0_1011	: coeff_p1_ff	<= 9'h036 ;
						7'b01_0_1100	: coeff_p1_ff	<= 9'h03c ;
						7'b01_0_1101	: coeff_p1_ff	<= 9'h041 ;
						7'b01_0_1110	: coeff_p1_ff	<= 9'h046 ;
						7'b01_0_1111	: coeff_p1_ff	<= 9'h04b ;
						7'b01_1_0000	: coeff_p1_ff	<= 9'h050 ;
						7'b01_1_0001	: coeff_p1_ff	<= 9'h055 ;
						7'b01_1_0010	: coeff_p1_ff	<= 9'h05a ;
						7'b01_1_0011	: coeff_p1_ff	<= 9'h05f ;
						7'b01_1_0100	: coeff_p1_ff	<= 9'h063 ;
						7'b01_1_0101	: coeff_p1_ff	<= 9'h067 ;
						7'b01_1_0110	: coeff_p1_ff	<= 9'h06b ;
						7'b01_1_0111	: coeff_p1_ff	<= 9'h06f ;
						7'b01_1_1000	: coeff_p1_ff	<= 9'h072 ;
						7'b01_1_1001	: coeff_p1_ff	<= 9'h076 ;
						7'b01_1_1010	: coeff_p1_ff	<= 9'h078 ;
						7'b01_1_1011	: coeff_p1_ff	<= 9'h07b ;
						7'b01_1_1100	: coeff_p1_ff	<= 9'h07d ;
						7'b01_1_1101	: coeff_p1_ff	<= 9'h07e ;
						7'b01_1_1110	: coeff_p1_ff	<= 9'h080 ;
						7'b01_1_1111	: coeff_p1_ff	<= 9'h080 ;

						7'b10_0_0000	: coeff_p1_ff	<= 9'h000 ;
						7'b10_0_0001	: coeff_p1_ff	<= 9'h007 ;
						7'b10_0_0010	: coeff_p1_ff	<= 9'h00c ;
						7'b10_0_0011	: coeff_p1_ff	<= 9'h012 ;
						7'b10_0_0100	: coeff_p1_ff	<= 9'h018 ;
						7'b10_0_0101	: coeff_p1_ff	<= 9'h01e ;
						7'b10_0_0110	: coeff_p1_ff	<= 9'h024 ;
						7'b10_0_0111	: coeff_p1_ff	<= 9'h02a ;
						7'b10_0_1000	: coeff_p1_ff	<= 9'h02f ;
						7'b10_0_1001	: coeff_p1_ff	<= 9'h034 ;
						7'b10_0_1010	: coeff_p1_ff	<= 9'h03a ;
						7'b10_0_1011	: coeff_p1_ff	<= 9'h040 ;
						7'b10_0_1100	: coeff_p1_ff	<= 9'h045 ;
						7'b10_0_1101	: coeff_p1_ff	<= 9'h04a ;
						7'b10_0_1110	: coeff_p1_ff	<= 9'h04f ;
						7'b10_0_1111	: coeff_p1_ff	<= 9'h054 ;
						7'b10_1_0000	: coeff_p1_ff	<= 9'h058 ;
						7'b10_1_0001	: coeff_p1_ff	<= 9'h05d ;
						7'b10_1_0010	: coeff_p1_ff	<= 9'h061 ;
						7'b10_1_0011	: coeff_p1_ff	<= 9'h065 ;
						7'b10_1_0100	: coeff_p1_ff	<= 9'h069 ;
						7'b10_1_0101	: coeff_p1_ff	<= 9'h06c ;
						7'b10_1_0110	: coeff_p1_ff	<= 9'h070 ;
						7'b10_1_0111	: coeff_p1_ff	<= 9'h073 ;
						7'b10_1_1000	: coeff_p1_ff	<= 9'h075 ;
						7'b10_1_1001	: coeff_p1_ff	<= 9'h078 ;
						7'b10_1_1010	: coeff_p1_ff	<= 9'h07a ;
						7'b10_1_1011	: coeff_p1_ff	<= 9'h07c ;
						7'b10_1_1100	: coeff_p1_ff	<= 9'h07e ;
						7'b10_1_1101	: coeff_p1_ff	<= 9'h07f ;
						7'b10_1_1110	: coeff_p1_ff	<= 9'h080 ;
						7'b10_1_1111	: coeff_p1_ff	<= 9'h080 ;

						7'b11_0_0000	: coeff_p1_ff	<= 9'h000 ;
						7'b11_0_0001	: coeff_p1_ff	<= 9'h009 ;
						7'b11_0_0010	: coeff_p1_ff	<= 9'h010 ;
						7'b11_0_0011	: coeff_p1_ff	<= 9'h018 ;
						7'b11_0_0100	: coeff_p1_ff	<= 9'h01f ;
						7'b11_0_0101	: coeff_p1_ff	<= 9'h026 ;
						7'b11_0_0110	: coeff_p1_ff	<= 9'h02c ;
						7'b11_0_0111	: coeff_p1_ff	<= 9'h033 ;
						7'b11_0_1000	: coeff_p1_ff	<= 9'h038 ;
						7'b11_0_1001	: coeff_p1_ff	<= 9'h03f ;
						7'b11_0_1010	: coeff_p1_ff	<= 9'h044 ;
						7'b11_0_1011	: coeff_p1_ff	<= 9'h049 ;
						7'b11_0_1100	: coeff_p1_ff	<= 9'h04f ;
						7'b11_0_1101	: coeff_p1_ff	<= 9'h054 ;
						7'b11_0_1110	: coeff_p1_ff	<= 9'h058 ;
						7'b11_0_1111	: coeff_p1_ff	<= 9'h05c ;
						7'b11_1_0000	: coeff_p1_ff	<= 9'h060 ;
						7'b11_1_0001	: coeff_p1_ff	<= 9'h064 ;
						7'b11_1_0010	: coeff_p1_ff	<= 9'h068 ;
						7'b11_1_0011	: coeff_p1_ff	<= 9'h06b ;
						7'b11_1_0100	: coeff_p1_ff	<= 9'h06e ;
						7'b11_1_0101	: coeff_p1_ff	<= 9'h071 ;
						7'b11_1_0110	: coeff_p1_ff	<= 9'h074 ;
						7'b11_1_0111	: coeff_p1_ff	<= 9'h076 ;
						7'b11_1_1000	: coeff_p1_ff	<= 9'h078 ;
						7'b11_1_1001	: coeff_p1_ff	<= 9'h07a ;
						7'b11_1_1010	: coeff_p1_ff	<= 9'h07c ;
						7'b11_1_1011	: coeff_p1_ff	<= 9'h07d ;
						7'b11_1_1100	: coeff_p1_ff	<= 9'h07e ;
						7'b11_1_1101	: coeff_p1_ff	<= 9'h07f ;
						7'b11_1_1110	: coeff_p1_ff	<= 9'h080 ;
						7'b11_1_1111	: coeff_p1_ff	<= 9'h080 ;
						default			: coeff_p1_ff	<= {9{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p2_ff		<= 9'h000 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_p2_ff	<= 9'h000 ;
						7'b00_0_0001	: coeff_p2_ff	<= 9'h1ff ;
						7'b00_0_0010	: coeff_p2_ff	<= 9'h1ff ;
						7'b00_0_0011	: coeff_p2_ff	<= 9'h1ff ;
						7'b00_0_0100	: coeff_p2_ff	<= 9'h1ff ;
						7'b00_0_0101	: coeff_p2_ff	<= 9'h1fe ;
						7'b00_0_0110	: coeff_p2_ff	<= 9'h1fe ;
						7'b00_0_0111	: coeff_p2_ff	<= 9'h1fd ;
						7'b00_0_1000	: coeff_p2_ff	<= 9'h1fd ;
						7'b00_0_1001	: coeff_p2_ff	<= 9'h1fc ;
						7'b00_0_1010	: coeff_p2_ff	<= 9'h1fb ;
						7'b00_0_1011	: coeff_p2_ff	<= 9'h1fb ;
						7'b00_0_1100	: coeff_p2_ff	<= 9'h1fa ;
						7'b00_0_1101	: coeff_p2_ff	<= 9'h1f9 ;
						7'b00_0_1110	: coeff_p2_ff	<= 9'h1f9 ;
						7'b00_0_1111	: coeff_p2_ff	<= 9'h1f8 ;
						7'b00_1_0000	: coeff_p2_ff	<= 9'h1f8 ;
						7'b00_1_0001	: coeff_p2_ff	<= 9'h1f7 ;
						7'b00_1_0010	: coeff_p2_ff	<= 9'h1f7 ;
						7'b00_1_0011	: coeff_p2_ff	<= 9'h1f6 ;
						7'b00_1_0100	: coeff_p2_ff	<= 9'h1f6 ;
						7'b00_1_0101	: coeff_p2_ff	<= 9'h1f6 ;
						7'b00_1_0110	: coeff_p2_ff	<= 9'h1f6 ;
						7'b00_1_0111	: coeff_p2_ff	<= 9'h1f6 ;
						7'b00_1_1000	: coeff_p2_ff	<= 9'h1f7 ;
						7'b00_1_1001	: coeff_p2_ff	<= 9'h1f7 ;
						7'b00_1_1010	: coeff_p2_ff	<= 9'h1f8 ;
						7'b00_1_1011	: coeff_p2_ff	<= 9'h1f8 ;
						7'b00_1_1100	: coeff_p2_ff	<= 9'h1f9 ;
						7'b00_1_1101	: coeff_p2_ff	<= 9'h1fb ;
						7'b00_1_1110	: coeff_p2_ff	<= 9'h1fc ;
						7'b00_1_1111	: coeff_p2_ff	<= 9'h1fe ;

						7'b01_0_0000	: coeff_p2_ff	<= 9'h000 ;
						7'b01_0_0001	: coeff_p2_ff	<= 9'h1ff ;
						7'b01_0_0010	: coeff_p2_ff	<= 9'h1ff ;
						7'b01_0_0011	: coeff_p2_ff	<= 9'h1fe ;
						7'b01_0_0100	: coeff_p2_ff	<= 9'h1fe ;
						7'b01_0_0101	: coeff_p2_ff	<= 9'h1fd ;
						7'b01_0_0110	: coeff_p2_ff	<= 9'h1fc ;
						7'b01_0_0111	: coeff_p2_ff	<= 9'h1fb ;
						7'b01_0_1000	: coeff_p2_ff	<= 9'h1fa ;
						7'b01_0_1001	: coeff_p2_ff	<= 9'h1f8 ;
						7'b01_0_1010	: coeff_p2_ff	<= 9'h1f7 ;
						7'b01_0_1011	: coeff_p2_ff	<= 9'h1f6 ;
						7'b01_0_1100	: coeff_p2_ff	<= 9'h1f4 ;
						7'b01_0_1101	: coeff_p2_ff	<= 9'h1f3 ;
						7'b01_0_1110	: coeff_p2_ff	<= 9'h1f2 ;
						7'b01_0_1111	: coeff_p2_ff	<= 9'h1f1 ;
						7'b01_1_0000	: coeff_p2_ff	<= 9'h1f0 ;
						7'b01_1_0001	: coeff_p2_ff	<= 9'h1ef ;
						7'b01_1_0010	: coeff_p2_ff	<= 9'h1ee ;
						7'b01_1_0011	: coeff_p2_ff	<= 9'h1ed ;
						7'b01_1_0100	: coeff_p2_ff	<= 9'h1ed ;
						7'b01_1_0101	: coeff_p2_ff	<= 9'h1ed ;
						7'b01_1_0110	: coeff_p2_ff	<= 9'h1ed ;
						7'b01_1_0111	: coeff_p2_ff	<= 9'h1ed ;
						7'b01_1_1000	: coeff_p2_ff	<= 9'h1ee ;
						7'b01_1_1001	: coeff_p2_ff	<= 9'h1ee ;
						7'b01_1_1010	: coeff_p2_ff	<= 9'h1f0 ;
						7'b01_1_1011	: coeff_p2_ff	<= 9'h1f1 ;
						7'b01_1_1100	: coeff_p2_ff	<= 9'h1f3 ;
						7'b01_1_1101	: coeff_p2_ff	<= 9'h1f6 ;
						7'b01_1_1110	: coeff_p2_ff	<= 9'h1f8 ;
						7'b01_1_1111	: coeff_p2_ff	<= 9'h1fc ;

						7'b10_0_0000	: coeff_p2_ff	<= 9'h000 ;
						7'b10_0_0001	: coeff_p2_ff	<= 9'h1ff ;
						7'b10_0_0010	: coeff_p2_ff	<= 9'h1ff ;
						7'b10_0_0011	: coeff_p2_ff	<= 9'h1fe ;
						7'b10_0_0100	: coeff_p2_ff	<= 9'h1fd ;
						7'b10_0_0101	: coeff_p2_ff	<= 9'h1fc ;
						7'b10_0_0110	: coeff_p2_ff	<= 9'h1fa ;
						7'b10_0_0111	: coeff_p2_ff	<= 9'h1f8 ;
						7'b10_0_1000	: coeff_p2_ff	<= 9'h1f7 ;
						7'b10_0_1001	: coeff_p2_ff	<= 9'h1f5 ;
						7'b10_0_1010	: coeff_p2_ff	<= 9'h1f3 ;
						7'b10_0_1011	: coeff_p2_ff	<= 9'h1f1 ;
						7'b10_0_1100	: coeff_p2_ff	<= 9'h1ef ;
						7'b10_0_1101	: coeff_p2_ff	<= 9'h1ed ;
						7'b10_0_1110	: coeff_p2_ff	<= 9'h1eb ;
						7'b10_0_1111	: coeff_p2_ff	<= 9'h1e9 ;
						7'b10_1_0000	: coeff_p2_ff	<= 9'h1e8 ;
						7'b10_1_0001	: coeff_p2_ff	<= 9'h1e6 ;
						7'b10_1_0010	: coeff_p2_ff	<= 9'h1e5 ;
						7'b10_1_0011	: coeff_p2_ff	<= 9'h1e4 ;
						7'b10_1_0100	: coeff_p2_ff	<= 9'h1e3 ;
						7'b10_1_0101	: coeff_p2_ff	<= 9'h1e3 ;
						7'b10_1_0110	: coeff_p2_ff	<= 9'h1e3 ;
						7'b10_1_0111	: coeff_p2_ff	<= 9'h1e4 ;
						7'b10_1_1000	: coeff_p2_ff	<= 9'h1e5 ;
						7'b10_1_1001	: coeff_p2_ff	<= 9'h1e6 ;
						7'b10_1_1010	: coeff_p2_ff	<= 9'h1e8 ;
						7'b10_1_1011	: coeff_p2_ff	<= 9'h1ea ;
						7'b10_1_1100	: coeff_p2_ff	<= 9'h1ed ;
						7'b10_1_1101	: coeff_p2_ff	<= 9'h1f1 ;
						7'b10_1_1110	: coeff_p2_ff	<= 9'h1f5 ;
						7'b10_1_1111	: coeff_p2_ff	<= 9'h1fa ;

						7'b11_0_0000	: coeff_p2_ff	<= 9'h000 ;
						7'b11_0_0001	: coeff_p2_ff	<= 9'h1ff ;
						7'b11_0_0010	: coeff_p2_ff	<= 9'h1ff ;
						7'b11_0_0011	: coeff_p2_ff	<= 9'h1fd ;
						7'b11_0_0100	: coeff_p2_ff	<= 9'h1fc ;
						7'b11_0_0101	: coeff_p2_ff	<= 9'h1fa ;
						7'b11_0_0110	: coeff_p2_ff	<= 9'h1f8 ;
						7'b11_0_0111	: coeff_p2_ff	<= 9'h1f6 ;
						7'b11_0_1000	: coeff_p2_ff	<= 9'h1f4 ;
						7'b11_0_1001	: coeff_p2_ff	<= 9'h1f1 ;
						7'b11_0_1010	: coeff_p2_ff	<= 9'h1ee ;
						7'b11_0_1011	: coeff_p2_ff	<= 9'h1ec ;
						7'b11_0_1100	: coeff_p2_ff	<= 9'h1e9 ;
						7'b11_0_1101	: coeff_p2_ff	<= 9'h1e6 ;
						7'b11_0_1110	: coeff_p2_ff	<= 9'h1e4 ;
						7'b11_0_1111	: coeff_p2_ff	<= 9'h1e2 ;
						7'b11_1_0000	: coeff_p2_ff	<= 9'h1e0 ;
						7'b11_1_0001	: coeff_p2_ff	<= 9'h1de ;
						7'b11_1_0010	: coeff_p2_ff	<= 9'h1dc ;
						7'b11_1_0011	: coeff_p2_ff	<= 9'h1db ;
						7'b11_1_0100	: coeff_p2_ff	<= 9'h1da ;
						7'b11_1_0101	: coeff_p2_ff	<= 9'h1da ;
						7'b11_1_0110	: coeff_p2_ff	<= 9'h1da ;
						7'b11_1_0111	: coeff_p2_ff	<= 9'h1da ;
						7'b11_1_1000	: coeff_p2_ff	<= 9'h1dc ;
						7'b11_1_1001	: coeff_p2_ff	<= 9'h1dd ;
						7'b11_1_1010	: coeff_p2_ff	<= 9'h1e0 ;
						7'b11_1_1011	: coeff_p2_ff	<= 9'h1e3 ;
						7'b11_1_1100	: coeff_p2_ff	<= 9'h1e7 ;
						7'b11_1_1101	: coeff_p2_ff	<= 9'h1ec ;
						7'b11_1_1110	: coeff_p2_ff	<= 9'h1f1 ;
						7'b11_1_1111	: coeff_p2_ff	<= 9'h1f8 ;
						default			: coeff_p2_ff	<= {9{1'bx}} ;
					endcase
				end
			end
		end
		else if ( Q_BIT == 10 ) begin : table_10bit
			//-------------------- Table Value Format -----------------------//
			//	 { Sign(1bit), Integral Part(1bit), Fraction Part(9bit) }	 //
			//---------------------------------------------------------------//
			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_m1_ff		<= 11'h000 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_m1_ff	<= 11'h000 ;
						7'b00_0_0001	: coeff_m1_ff	<= 11'h7f8 ;
						7'b00_0_0010	: coeff_m1_ff	<= 11'h7f1 ;
						7'b00_0_0011	: coeff_m1_ff	<= 11'h7ec ;
						7'b00_0_0100	: coeff_m1_ff	<= 11'h7e7 ;
						7'b00_0_0101	: coeff_m1_ff	<= 11'h7e3 ;
						7'b00_0_0110	: coeff_m1_ff	<= 11'h7e0 ;
						7'b00_0_0111	: coeff_m1_ff	<= 11'h7dd ;
						7'b00_0_1000	: coeff_m1_ff	<= 11'h7dc ;
						7'b00_0_1001	: coeff_m1_ff	<= 11'h7da ;
						7'b00_0_1010	: coeff_m1_ff	<= 11'h7da ;
						7'b00_0_1011	: coeff_m1_ff	<= 11'h7da ;
						7'b00_0_1100	: coeff_m1_ff	<= 11'h7da ;
						7'b00_0_1101	: coeff_m1_ff	<= 11'h7db ;
						7'b00_0_1110	: coeff_m1_ff	<= 11'h7dc ;
						7'b00_0_1111	: coeff_m1_ff	<= 11'h7de ;
						7'b00_1_0000	: coeff_m1_ff	<= 11'h7e0 ;
						7'b00_1_0001	: coeff_m1_ff	<= 11'h7e2 ;
						7'b00_1_0010	: coeff_m1_ff	<= 11'h7e4 ;
						7'b00_1_0011	: coeff_m1_ff	<= 11'h7e6 ;
						7'b00_1_0100	: coeff_m1_ff	<= 11'h7e9 ;
						7'b00_1_0101	: coeff_m1_ff	<= 11'h7ec ;
						7'b00_1_0110	: coeff_m1_ff	<= 11'h7ee ;
						7'b00_1_0111	: coeff_m1_ff	<= 11'h7f1 ;
						7'b00_1_1000	: coeff_m1_ff	<= 11'h7f4 ;
						7'b00_1_1001	: coeff_m1_ff	<= 11'h7f6 ;
						7'b00_1_1010	: coeff_m1_ff	<= 11'h7f8 ;
						7'b00_1_1011	: coeff_m1_ff	<= 11'h7fa ;
						7'b00_1_1100	: coeff_m1_ff	<= 11'h7fc ;
						7'b00_1_1101	: coeff_m1_ff	<= 11'h7fd ;
						7'b00_1_1110	: coeff_m1_ff	<= 11'h7ff ;
						7'b00_1_1111	: coeff_m1_ff	<= 11'h7ff ;

						7'b01_0_0000	: coeff_m1_ff	<= 11'h000 ;
						7'b01_0_0001	: coeff_m1_ff	<= 11'h7f0 ;
						7'b01_0_0010	: coeff_m1_ff	<= 11'h7e3 ;
						7'b01_0_0011	: coeff_m1_ff	<= 11'h7d8 ;
						7'b01_0_0100	: coeff_m1_ff	<= 11'h7cf ;
						7'b01_0_0101	: coeff_m1_ff	<= 11'h7c7 ;
						7'b01_0_0110	: coeff_m1_ff	<= 11'h7c0 ;
						7'b01_0_0111	: coeff_m1_ff	<= 11'h7bb ;
						7'b01_0_1000	: coeff_m1_ff	<= 11'h7b8 ;
						7'b01_0_1001	: coeff_m1_ff	<= 11'h7b5 ;
						7'b01_0_1010	: coeff_m1_ff	<= 11'h7b4 ;
						7'b01_0_1011	: coeff_m1_ff	<= 11'h7b4 ;
						7'b01_0_1100	: coeff_m1_ff	<= 11'h7b5 ;
						7'b01_0_1101	: coeff_m1_ff	<= 11'h7b6 ;
						7'b01_0_1110	: coeff_m1_ff	<= 11'h7b9 ;
						7'b01_0_1111	: coeff_m1_ff	<= 11'h7bc ;
						7'b01_1_0000	: coeff_m1_ff	<= 11'h7c0 ;
						7'b01_1_0001	: coeff_m1_ff	<= 11'h7c4 ;
						7'b01_1_0010	: coeff_m1_ff	<= 11'h7c8 ;
						7'b01_1_0011	: coeff_m1_ff	<= 11'h7cd ;
						7'b01_1_0100	: coeff_m1_ff	<= 11'h7d3 ;
						7'b01_1_0101	: coeff_m1_ff	<= 11'h7d8 ;
						7'b01_1_0110	: coeff_m1_ff	<= 11'h7dd ;
						7'b01_1_0111	: coeff_m1_ff	<= 11'h7e2 ;
						7'b01_1_1000	: coeff_m1_ff	<= 11'h7e8 ;
						7'b01_1_1001	: coeff_m1_ff	<= 11'h7ec ;
						7'b01_1_1010	: coeff_m1_ff	<= 11'h7f1 ;
						7'b01_1_1011	: coeff_m1_ff	<= 11'h7f5 ;
						7'b01_1_1100	: coeff_m1_ff	<= 11'h7f9 ;
						7'b01_1_1101	: coeff_m1_ff	<= 11'h7fb ;
						7'b01_1_1110	: coeff_m1_ff	<= 11'h7fe ;
						7'b01_1_1111	: coeff_m1_ff	<= 11'h7ff ;

						7'b10_0_0000	: coeff_m1_ff	<= 11'h000 ;
						7'b10_0_0001	: coeff_m1_ff	<= 11'h7e9 ;
						7'b10_0_0010	: coeff_m1_ff	<= 11'h7d5 ;
						7'b10_0_0011	: coeff_m1_ff	<= 11'h7c4 ;
						7'b10_0_0100	: coeff_m1_ff	<= 11'h7b6 ;
						7'b10_0_0101	: coeff_m1_ff	<= 11'h7aa ;
						7'b10_0_0110	: coeff_m1_ff	<= 11'h7a0 ;
						7'b10_0_0111	: coeff_m1_ff	<= 11'h799 ;
						7'b10_0_1000	: coeff_m1_ff	<= 11'h794 ;
						7'b10_0_1001	: coeff_m1_ff	<= 11'h790 ;
						7'b10_0_1010	: coeff_m1_ff	<= 11'h78e ;
						7'b10_0_1011	: coeff_m1_ff	<= 11'h78e ;
						7'b10_0_1100	: coeff_m1_ff	<= 11'h78f ;
						7'b10_0_1101	: coeff_m1_ff	<= 11'h792 ;
						7'b10_0_1110	: coeff_m1_ff	<= 11'h795 ;
						7'b10_0_1111	: coeff_m1_ff	<= 11'h79a ;
						7'b10_1_0000	: coeff_m1_ff	<= 11'h7a0 ;
						7'b10_1_0001	: coeff_m1_ff	<= 11'h7a6 ;
						7'b10_1_0010	: coeff_m1_ff	<= 11'h7ad ;
						7'b10_1_0011	: coeff_m1_ff	<= 11'h7b4 ;
						7'b10_1_0100	: coeff_m1_ff	<= 11'h7bc ;
						7'b10_1_0101	: coeff_m1_ff	<= 11'h7c4 ;
						7'b10_1_0110	: coeff_m1_ff	<= 11'h7cc ;
						7'b10_1_0111	: coeff_m1_ff	<= 11'h7d4 ;
						7'b10_1_1000	: coeff_m1_ff	<= 11'h7dc ;
						7'b10_1_1001	: coeff_m1_ff	<= 11'h7e3 ;
						7'b10_1_1010	: coeff_m1_ff	<= 11'h7ea ;
						7'b10_1_1011	: coeff_m1_ff	<= 11'h7f0 ;
						7'b10_1_1100	: coeff_m1_ff	<= 11'h7f5 ;
						7'b10_1_1101	: coeff_m1_ff	<= 11'h7f9 ;
						7'b10_1_1110	: coeff_m1_ff	<= 11'h7fd ;
						7'b10_1_1111	: coeff_m1_ff	<= 11'h7ff ;

						7'b11_0_0000	: coeff_m1_ff	<= 11'h000 ;
						7'b11_0_0001	: coeff_m1_ff	<= 11'h7e1 ;
						7'b11_0_0010	: coeff_m1_ff	<= 11'h7c7 ;
						7'b11_0_0011	: coeff_m1_ff	<= 11'h7b1 ;
						7'b11_0_0100	: coeff_m1_ff	<= 11'h79e ;
						7'b11_0_0101	: coeff_m1_ff	<= 11'h78e ;
						7'b11_0_0110	: coeff_m1_ff	<= 11'h781 ;
						7'b11_0_0111	: coeff_m1_ff	<= 11'h777 ;
						7'b11_0_1000	: coeff_m1_ff	<= 11'h770 ;
						7'b11_0_1001	: coeff_m1_ff	<= 11'h76b ;
						7'b11_0_1010	: coeff_m1_ff	<= 11'h768 ;
						7'b11_0_1011	: coeff_m1_ff	<= 11'h768 ;
						7'b11_0_1100	: coeff_m1_ff	<= 11'h76a ;
						7'b11_0_1101	: coeff_m1_ff	<= 11'h76d ;
						7'b11_0_1110	: coeff_m1_ff	<= 11'h772 ;
						7'b11_0_1111	: coeff_m1_ff	<= 11'h778 ;
						7'b11_1_0000	: coeff_m1_ff	<= 11'h780 ;
						7'b11_1_0001	: coeff_m1_ff	<= 11'h788 ;
						7'b11_1_0010	: coeff_m1_ff	<= 11'h791 ;
						7'b11_1_0011	: coeff_m1_ff	<= 11'h79b ;
						7'b11_1_0100	: coeff_m1_ff	<= 11'h7a6 ;
						7'b11_1_0101	: coeff_m1_ff	<= 11'h7b0 ;
						7'b11_1_0110	: coeff_m1_ff	<= 11'h7bb ;
						7'b11_1_0111	: coeff_m1_ff	<= 11'h7c5 ;
						7'b11_1_1000	: coeff_m1_ff	<= 11'h7d0 ;
						7'b11_1_1001	: coeff_m1_ff	<= 11'h7d9 ;
						7'b11_1_1010	: coeff_m1_ff	<= 11'h7e2 ;
						7'b11_1_1011	: coeff_m1_ff	<= 11'h7ea ;
						7'b11_1_1100	: coeff_m1_ff	<= 11'h7f2 ;
						7'b11_1_1101	: coeff_m1_ff	<= 11'h7f7 ;
						7'b11_1_1110	: coeff_m1_ff	<= 11'h7fc ;
						7'b11_1_1111	: coeff_m1_ff	<= 11'h7ff ;
						default			: coeff_m1_ff	<= {11{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_zr_ff		<= 11'h200 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_zr_ff	<= 11'h200 ;
						7'b00_0_0001	: coeff_zr_ff	<= 11'h1ff ;
						7'b00_0_0010	: coeff_zr_ff	<= 11'h1fc ;
						7'b00_0_0011	: coeff_zr_ff	<= 11'h1f6 ;
						7'b00_0_0100	: coeff_zr_ff	<= 11'h1ee ;
						7'b00_0_0101	: coeff_zr_ff	<= 11'h1e4 ;
						7'b00_0_0110	: coeff_zr_ff	<= 11'h1d9 ;
						7'b00_0_0111	: coeff_zr_ff	<= 11'h1cb ;
						7'b00_0_1000	: coeff_zr_ff	<= 11'h1bc ;
						7'b00_0_1001	: coeff_zr_ff	<= 11'h1ac ;
						7'b00_0_1010	: coeff_zr_ff	<= 11'h19b ;
						7'b00_0_1011	: coeff_zr_ff	<= 11'h188 ;
						7'b00_0_1100	: coeff_zr_ff	<= 11'h175 ;
						7'b00_0_1101	: coeff_zr_ff	<= 11'h161 ;
						7'b00_0_1110	: coeff_zr_ff	<= 11'h14c ;
						7'b00_0_1111	: coeff_zr_ff	<= 11'h136 ;
						7'b00_1_0000	: coeff_zr_ff	<= 11'h120 ;
						7'b00_1_0001	: coeff_zr_ff	<= 11'h10a ;
						7'b00_1_0010	: coeff_zr_ff	<= 11'h0f4 ;
						7'b00_1_0011	: coeff_zr_ff	<= 11'h0de ;
						7'b00_1_0100	: coeff_zr_ff	<= 11'h0c8 ;
						7'b00_1_0101	: coeff_zr_ff	<= 11'h0b2 ;
						7'b00_1_0110	: coeff_zr_ff	<= 11'h09d ;
						7'b00_1_0111	: coeff_zr_ff	<= 11'h089 ;
						7'b00_1_1000	: coeff_zr_ff	<= 11'h074 ;
						7'b00_1_1001	: coeff_zr_ff	<= 11'h062 ;
						7'b00_1_1010	: coeff_zr_ff	<= 11'h04f ;
						7'b00_1_1011	: coeff_zr_ff	<= 11'h03f ;
						7'b00_1_1100	: coeff_zr_ff	<= 11'h02f ;
						7'b00_1_1101	: coeff_zr_ff	<= 11'h021 ;
						7'b00_1_1110	: coeff_zr_ff	<= 11'h014 ;
						7'b00_1_1111	: coeff_zr_ff	<= 11'h00a ;

						7'b01_0_0000	: coeff_zr_ff	<= 11'h200 ;
						7'b01_0_0001	: coeff_zr_ff	<= 11'h200 ;
						7'b01_0_0010	: coeff_zr_ff	<= 11'h1fd ;
						7'b01_0_0011	: coeff_zr_ff	<= 11'h1f8 ;
						7'b01_0_0100	: coeff_zr_ff	<= 11'h1f1 ;
						7'b01_0_0101	: coeff_zr_ff	<= 11'h1e9 ;
						7'b01_0_0110	: coeff_zr_ff	<= 11'h1e0 ;
						7'b01_0_0111	: coeff_zr_ff	<= 11'h1d5 ;
						7'b01_0_1000	: coeff_zr_ff	<= 11'h1c8 ;
						7'b01_0_1001	: coeff_zr_ff	<= 11'h1bb ;
						7'b01_0_1010	: coeff_zr_ff	<= 11'h1ac ;
						7'b01_0_1011	: coeff_zr_ff	<= 11'h19c ;
						7'b01_0_1100	: coeff_zr_ff	<= 11'h18b ;
						7'b01_0_1101	: coeff_zr_ff	<= 11'h17a ;
						7'b01_0_1110	: coeff_zr_ff	<= 11'h167 ;
						7'b01_0_1111	: coeff_zr_ff	<= 11'h154 ;
						7'b01_1_0000	: coeff_zr_ff	<= 11'h140 ;
						7'b01_1_0001	: coeff_zr_ff	<= 11'h12c ;
						7'b01_1_0010	: coeff_zr_ff	<= 11'h118 ;
						7'b01_1_0011	: coeff_zr_ff	<= 11'h103 ;
						7'b01_1_0100	: coeff_zr_ff	<= 11'h0ed ;
						7'b01_1_0101	: coeff_zr_ff	<= 11'h0d8 ;
						7'b01_1_0110	: coeff_zr_ff	<= 11'h0c3 ;
						7'b01_1_0111	: coeff_zr_ff	<= 11'h0ae ;
						7'b01_1_1000	: coeff_zr_ff	<= 11'h098 ;
						7'b01_1_1001	: coeff_zr_ff	<= 11'h084 ;
						7'b01_1_1010	: coeff_zr_ff	<= 11'h06f ;
						7'b01_1_1011	: coeff_zr_ff	<= 11'h05b ;
						7'b01_1_1100	: coeff_zr_ff	<= 11'h047 ;
						7'b01_1_1101	: coeff_zr_ff	<= 11'h035 ;
						7'b01_1_1110	: coeff_zr_ff	<= 11'h022 ;
						7'b01_1_1111	: coeff_zr_ff	<= 11'h011 ;

						7'b10_0_0000	: coeff_zr_ff	<= 11'h200 ;
						7'b10_0_0001	: coeff_zr_ff	<= 11'h200 ;
						7'b10_0_0010	: coeff_zr_ff	<= 11'h1fe ;
						7'b10_0_0011	: coeff_zr_ff	<= 11'h1fa ;
						7'b10_0_0100	: coeff_zr_ff	<= 11'h1f5 ;
						7'b10_0_0101	: coeff_zr_ff	<= 11'h1ef ;
						7'b10_0_0110	: coeff_zr_ff	<= 11'h1e7 ;
						7'b10_0_0111	: coeff_zr_ff	<= 11'h1de ;
						7'b10_0_1000	: coeff_zr_ff	<= 11'h1d4 ;
						7'b10_0_1001	: coeff_zr_ff	<= 11'h1c9 ;
						7'b10_0_1010	: coeff_zr_ff	<= 11'h1bd ;
						7'b10_0_1011	: coeff_zr_ff	<= 11'h1b0 ;
						7'b10_0_1100	: coeff_zr_ff	<= 11'h1a2 ;
						7'b10_0_1101	: coeff_zr_ff	<= 11'h193 ;
						7'b10_0_1110	: coeff_zr_ff	<= 11'h183 ;
						7'b10_0_1111	: coeff_zr_ff	<= 11'h172 ;
						7'b10_1_0000	: coeff_zr_ff	<= 11'h160 ;
						7'b10_1_0001	: coeff_zr_ff	<= 11'h14e ;
						7'b10_1_0010	: coeff_zr_ff	<= 11'h13b ;
						7'b10_1_0011	: coeff_zr_ff	<= 11'h127 ;
						7'b10_1_0100	: coeff_zr_ff	<= 11'h113 ;
						7'b10_1_0101	: coeff_zr_ff	<= 11'h0fe ;
						7'b10_1_0110	: coeff_zr_ff	<= 11'h0e9 ;
						7'b10_1_0111	: coeff_zr_ff	<= 11'h0d3 ;
						7'b10_1_1000	: coeff_zr_ff	<= 11'h0bc ;
						7'b10_1_1001	: coeff_zr_ff	<= 11'h0a6 ;
						7'b10_1_1010	: coeff_zr_ff	<= 11'h08f ;
						7'b10_1_1011	: coeff_zr_ff	<= 11'h077 ;
						7'b10_1_1100	: coeff_zr_ff	<= 11'h060 ;
						7'b10_1_1101	: coeff_zr_ff	<= 11'h049 ;
						7'b10_1_1110	: coeff_zr_ff	<= 11'h030 ;
						7'b10_1_1111	: coeff_zr_ff	<= 11'h018 ;

						7'b11_0_0000	: coeff_zr_ff	<= 11'h200 ;
						7'b11_0_0001	: coeff_zr_ff	<= 11'h200 ;
						7'b11_0_0010	: coeff_zr_ff	<= 11'h1fe ;
						7'b11_0_0011	: coeff_zr_ff	<= 11'h1fc ;
						7'b11_0_0100	: coeff_zr_ff	<= 11'h1f8 ;
						7'b11_0_0101	: coeff_zr_ff	<= 11'h1f4 ;
						7'b11_0_0110	: coeff_zr_ff	<= 11'h1ee ;
						7'b11_0_0111	: coeff_zr_ff	<= 11'h1e8 ;
						7'b11_0_1000	: coeff_zr_ff	<= 11'h1e0 ;
						7'b11_0_1001	: coeff_zr_ff	<= 11'h1d8 ;
						7'b11_0_1010	: coeff_zr_ff	<= 11'h1ce ;
						7'b11_0_1011	: coeff_zr_ff	<= 11'h1c4 ;
						7'b11_0_1100	: coeff_zr_ff	<= 11'h1b8 ;
						7'b11_0_1101	: coeff_zr_ff	<= 11'h1ac ;
						7'b11_0_1110	: coeff_zr_ff	<= 11'h19e ;
						7'b11_0_1111	: coeff_zr_ff	<= 11'h190 ;
						7'b11_1_0000	: coeff_zr_ff	<= 11'h180 ;
						7'b11_1_0001	: coeff_zr_ff	<= 11'h170 ;
						7'b11_1_0010	: coeff_zr_ff	<= 11'h15f ;
						7'b11_1_0011	: coeff_zr_ff	<= 11'h14c ;
						7'b11_1_0100	: coeff_zr_ff	<= 11'h138 ;
						7'b11_1_0101	: coeff_zr_ff	<= 11'h124 ;
						7'b11_1_0110	: coeff_zr_ff	<= 11'h10f ;
						7'b11_1_0111	: coeff_zr_ff	<= 11'h0f8 ;
						7'b11_1_1000	: coeff_zr_ff	<= 11'h0e0 ;
						7'b11_1_1001	: coeff_zr_ff	<= 11'h0c8 ;
						7'b11_1_1010	: coeff_zr_ff	<= 11'h0af ;
						7'b11_1_1011	: coeff_zr_ff	<= 11'h094 ;
						7'b11_1_1100	: coeff_zr_ff	<= 11'h078 ;
						7'b11_1_1101	: coeff_zr_ff	<= 11'h05c ;
						7'b11_1_1110	: coeff_zr_ff	<= 11'h03f ;
						7'b11_1_1111	: coeff_zr_ff	<= 11'h020 ;
						default			: coeff_zr_ff	<= {11{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p1_ff		<= 11'h000 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_p1_ff	<= 11'h000 ;
						7'b00_0_0001	: coeff_p1_ff	<= 11'h00a ;
						7'b00_0_0010	: coeff_p1_ff	<= 11'h014 ;
						7'b00_0_0011	: coeff_p1_ff	<= 11'h021 ;
						7'b00_0_0100	: coeff_p1_ff	<= 11'h02f ;
						7'b00_0_0101	: coeff_p1_ff	<= 11'h03f ;
						7'b00_0_0110	: coeff_p1_ff	<= 11'h04f ;
						7'b00_0_0111	: coeff_p1_ff	<= 11'h062 ;
						7'b00_0_1000	: coeff_p1_ff	<= 11'h074 ;
						7'b00_0_1001	: coeff_p1_ff	<= 11'h089 ;
						7'b00_0_1010	: coeff_p1_ff	<= 11'h09d ;
						7'b00_0_1011	: coeff_p1_ff	<= 11'h0b2 ;
						7'b00_0_1100	: coeff_p1_ff	<= 11'h0c8 ;
						7'b00_0_1101	: coeff_p1_ff	<= 11'h0de ;
						7'b00_0_1110	: coeff_p1_ff	<= 11'h0f4 ;
						7'b00_0_1111	: coeff_p1_ff	<= 11'h10a ;
						7'b00_1_0000	: coeff_p1_ff	<= 11'h120 ;
						7'b00_1_0001	: coeff_p1_ff	<= 11'h136 ;
						7'b00_1_0010	: coeff_p1_ff	<= 11'h14c ;
						7'b00_1_0011	: coeff_p1_ff	<= 11'h161 ;
						7'b00_1_0100	: coeff_p1_ff	<= 11'h175 ;
						7'b00_1_0101	: coeff_p1_ff	<= 11'h188 ;
						7'b00_1_0110	: coeff_p1_ff	<= 11'h19b ;
						7'b00_1_0111	: coeff_p1_ff	<= 11'h1ac ;
						7'b00_1_1000	: coeff_p1_ff	<= 11'h1bc ;
						7'b00_1_1001	: coeff_p1_ff	<= 11'h1cb ;
						7'b00_1_1010	: coeff_p1_ff	<= 11'h1d9 ;
						7'b00_1_1011	: coeff_p1_ff	<= 11'h1e4 ;
						7'b00_1_1100	: coeff_p1_ff	<= 11'h1ee ;
						7'b00_1_1101	: coeff_p1_ff	<= 11'h1f6 ;
						7'b00_1_1110	: coeff_p1_ff	<= 11'h1fc ;
						7'b00_1_1111	: coeff_p1_ff	<= 11'h1ff ;

						7'b01_0_0000	: coeff_p1_ff	<= 11'h000 ;
						7'b01_0_0001	: coeff_p1_ff	<= 11'h011 ;
						7'b01_0_0010	: coeff_p1_ff	<= 11'h022 ;
						7'b01_0_0011	: coeff_p1_ff	<= 11'h035 ;
						7'b01_0_0100	: coeff_p1_ff	<= 11'h047 ;
						7'b01_0_0101	: coeff_p1_ff	<= 11'h05b ;
						7'b01_0_0110	: coeff_p1_ff	<= 11'h06f ;
						7'b01_0_0111	: coeff_p1_ff	<= 11'h084 ;
						7'b01_0_1000	: coeff_p1_ff	<= 11'h098 ;
						7'b01_0_1001	: coeff_p1_ff	<= 11'h0ae ;
						7'b01_0_1010	: coeff_p1_ff	<= 11'h0c3 ;
						7'b01_0_1011	: coeff_p1_ff	<= 11'h0d8 ;
						7'b01_0_1100	: coeff_p1_ff	<= 11'h0ed ;
						7'b01_0_1101	: coeff_p1_ff	<= 11'h103 ;
						7'b01_0_1110	: coeff_p1_ff	<= 11'h118 ;
						7'b01_0_1111	: coeff_p1_ff	<= 11'h12c ;
						7'b01_1_0000	: coeff_p1_ff	<= 11'h140 ;
						7'b01_1_0001	: coeff_p1_ff	<= 11'h154 ;
						7'b01_1_0010	: coeff_p1_ff	<= 11'h167 ;
						7'b01_1_0011	: coeff_p1_ff	<= 11'h17a ;
						7'b01_1_0100	: coeff_p1_ff	<= 11'h18b ;
						7'b01_1_0101	: coeff_p1_ff	<= 11'h19c ;
						7'b01_1_0110	: coeff_p1_ff	<= 11'h1ac ;
						7'b01_1_0111	: coeff_p1_ff	<= 11'h1bb ;
						7'b01_1_1000	: coeff_p1_ff	<= 11'h1c8 ;
						7'b01_1_1001	: coeff_p1_ff	<= 11'h1d5 ;
						7'b01_1_1010	: coeff_p1_ff	<= 11'h1e0 ;
						7'b01_1_1011	: coeff_p1_ff	<= 11'h1e9 ;
						7'b01_1_1100	: coeff_p1_ff	<= 11'h1f1 ;
						7'b01_1_1101	: coeff_p1_ff	<= 11'h1f8 ;
						7'b01_1_1110	: coeff_p1_ff	<= 11'h1fd ;
						7'b01_1_1111	: coeff_p1_ff	<= 11'h200 ;

						7'b10_0_0000	: coeff_p1_ff	<= 11'h000 ;
						7'b10_0_0001	: coeff_p1_ff	<= 11'h018 ;
						7'b10_0_0010	: coeff_p1_ff	<= 11'h030 ;
						7'b10_0_0011	: coeff_p1_ff	<= 11'h049 ;
						7'b10_0_0100	: coeff_p1_ff	<= 11'h060 ;
						7'b10_0_0101	: coeff_p1_ff	<= 11'h077 ;
						7'b10_0_0110	: coeff_p1_ff	<= 11'h08f ;
						7'b10_0_0111	: coeff_p1_ff	<= 11'h0a6 ;
						7'b10_0_1000	: coeff_p1_ff	<= 11'h0bc ;
						7'b10_0_1001	: coeff_p1_ff	<= 11'h0d3 ;
						7'b10_0_1010	: coeff_p1_ff	<= 11'h0e9 ;
						7'b10_0_1011	: coeff_p1_ff	<= 11'h0fe ;
						7'b10_0_1100	: coeff_p1_ff	<= 11'h113 ;
						7'b10_0_1101	: coeff_p1_ff	<= 11'h127 ;
						7'b10_0_1110	: coeff_p1_ff	<= 11'h13b ;
						7'b10_0_1111	: coeff_p1_ff	<= 11'h14e ;
						7'b10_1_0000	: coeff_p1_ff	<= 11'h160 ;
						7'b10_1_0001	: coeff_p1_ff	<= 11'h172 ;
						7'b10_1_0010	: coeff_p1_ff	<= 11'h183 ;
						7'b10_1_0011	: coeff_p1_ff	<= 11'h193 ;
						7'b10_1_0100	: coeff_p1_ff	<= 11'h1a2 ;
						7'b10_1_0101	: coeff_p1_ff	<= 11'h1b0 ;
						7'b10_1_0110	: coeff_p1_ff	<= 11'h1bd ;
						7'b10_1_0111	: coeff_p1_ff	<= 11'h1c9 ;
						7'b10_1_1000	: coeff_p1_ff	<= 11'h1d4 ;
						7'b10_1_1001	: coeff_p1_ff	<= 11'h1de ;
						7'b10_1_1010	: coeff_p1_ff	<= 11'h1e7 ;
						7'b10_1_1011	: coeff_p1_ff	<= 11'h1ef ;
						7'b10_1_1100	: coeff_p1_ff	<= 11'h1f5 ;
						7'b10_1_1101	: coeff_p1_ff	<= 11'h1fa ;
						7'b10_1_1110	: coeff_p1_ff	<= 11'h1fe ;
						7'b10_1_1111	: coeff_p1_ff	<= 11'h200 ;

						7'b11_0_0000	: coeff_p1_ff	<= 11'h000 ;
						7'b11_0_0001	: coeff_p1_ff	<= 11'h020 ;
						7'b11_0_0010	: coeff_p1_ff	<= 11'h03f ;
						7'b11_0_0011	: coeff_p1_ff	<= 11'h05c ;
						7'b11_0_0100	: coeff_p1_ff	<= 11'h078 ;
						7'b11_0_0101	: coeff_p1_ff	<= 11'h094 ;
						7'b11_0_0110	: coeff_p1_ff	<= 11'h0af ;
						7'b11_0_0111	: coeff_p1_ff	<= 11'h0c8 ;
						7'b11_0_1000	: coeff_p1_ff	<= 11'h0e0 ;
						7'b11_0_1001	: coeff_p1_ff	<= 11'h0f8 ;
						7'b11_0_1010	: coeff_p1_ff	<= 11'h10f ;
						7'b11_0_1011	: coeff_p1_ff	<= 11'h124 ;
						7'b11_0_1100	: coeff_p1_ff	<= 11'h138 ;
						7'b11_0_1101	: coeff_p1_ff	<= 11'h14c ;
						7'b11_0_1110	: coeff_p1_ff	<= 11'h15f ;
						7'b11_0_1111	: coeff_p1_ff	<= 11'h170 ;
						7'b11_1_0000	: coeff_p1_ff	<= 11'h180 ;
						7'b11_1_0001	: coeff_p1_ff	<= 11'h190 ;
						7'b11_1_0010	: coeff_p1_ff	<= 11'h19e ;
						7'b11_1_0011	: coeff_p1_ff	<= 11'h1ac ;
						7'b11_1_0100	: coeff_p1_ff	<= 11'h1b8 ;
						7'b11_1_0101	: coeff_p1_ff	<= 11'h1c4 ;
						7'b11_1_0110	: coeff_p1_ff	<= 11'h1ce ;
						7'b11_1_0111	: coeff_p1_ff	<= 11'h1d8 ;
						7'b11_1_1000	: coeff_p1_ff	<= 11'h1e0 ;
						7'b11_1_1001	: coeff_p1_ff	<= 11'h1e8 ;
						7'b11_1_1010	: coeff_p1_ff	<= 11'h1ee ;
						7'b11_1_1011	: coeff_p1_ff	<= 11'h1f4 ;
						7'b11_1_1100	: coeff_p1_ff	<= 11'h1f8 ;
						7'b11_1_1101	: coeff_p1_ff	<= 11'h1fc ;
						7'b11_1_1110	: coeff_p1_ff	<= 11'h1fe ;
						7'b11_1_1111	: coeff_p1_ff	<= 11'h200 ;
						default			: coeff_p1_ff	<= {11{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p2_ff		<= 11'h000 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_p2_ff	<= 11'h000 ;
						7'b00_0_0001	: coeff_p2_ff	<= 11'h7ff ;
						7'b00_0_0010	: coeff_p2_ff	<= 11'h7ff ;
						7'b00_0_0011	: coeff_p2_ff	<= 11'h7fd ;
						7'b00_0_0100	: coeff_p2_ff	<= 11'h7fc ;
						7'b00_0_0101	: coeff_p2_ff	<= 11'h7fa ;
						7'b00_0_0110	: coeff_p2_ff	<= 11'h7f8 ;
						7'b00_0_0111	: coeff_p2_ff	<= 11'h7f6 ;
						7'b00_0_1000	: coeff_p2_ff	<= 11'h7f4 ;
						7'b00_0_1001	: coeff_p2_ff	<= 11'h7f1 ;
						7'b00_0_1010	: coeff_p2_ff	<= 11'h7ee ;
						7'b00_0_1011	: coeff_p2_ff	<= 11'h7ec ;
						7'b00_0_1100	: coeff_p2_ff	<= 11'h7e9 ;
						7'b00_0_1101	: coeff_p2_ff	<= 11'h7e6 ;
						7'b00_0_1110	: coeff_p2_ff	<= 11'h7e4 ;
						7'b00_0_1111	: coeff_p2_ff	<= 11'h7e2 ;
						7'b00_1_0000	: coeff_p2_ff	<= 11'h7e0 ;
						7'b00_1_0001	: coeff_p2_ff	<= 11'h7de ;
						7'b00_1_0010	: coeff_p2_ff	<= 11'h7dc ;
						7'b00_1_0011	: coeff_p2_ff	<= 11'h7db ;
						7'b00_1_0100	: coeff_p2_ff	<= 11'h7da ;
						7'b00_1_0101	: coeff_p2_ff	<= 11'h7da ;
						7'b00_1_0110	: coeff_p2_ff	<= 11'h7da ;
						7'b00_1_0111	: coeff_p2_ff	<= 11'h7da ;
						7'b00_1_1000	: coeff_p2_ff	<= 11'h7dc ;
						7'b00_1_1001	: coeff_p2_ff	<= 11'h7dd ;
						7'b00_1_1010	: coeff_p2_ff	<= 11'h7e0 ;
						7'b00_1_1011	: coeff_p2_ff	<= 11'h7e3 ;
						7'b00_1_1100	: coeff_p2_ff	<= 11'h7e7 ;
						7'b00_1_1101	: coeff_p2_ff	<= 11'h7ec ;
						7'b00_1_1110	: coeff_p2_ff	<= 11'h7f1 ;
						7'b00_1_1111	: coeff_p2_ff	<= 11'h7f8 ;

						7'b01_0_0000	: coeff_p2_ff	<= 11'h000 ;
						7'b01_0_0001	: coeff_p2_ff	<= 11'h7ff ;
						7'b01_0_0010	: coeff_p2_ff	<= 11'h7fe ;
						7'b01_0_0011	: coeff_p2_ff	<= 11'h7fb ;
						7'b01_0_0100	: coeff_p2_ff	<= 11'h7f9 ;
						7'b01_0_0101	: coeff_p2_ff	<= 11'h7f5 ;
						7'b01_0_0110	: coeff_p2_ff	<= 11'h7f1 ;
						7'b01_0_0111	: coeff_p2_ff	<= 11'h7ec ;
						7'b01_0_1000	: coeff_p2_ff	<= 11'h7e8 ;
						7'b01_0_1001	: coeff_p2_ff	<= 11'h7e2 ;
						7'b01_0_1010	: coeff_p2_ff	<= 11'h7dd ;
						7'b01_0_1011	: coeff_p2_ff	<= 11'h7d8 ;
						7'b01_0_1100	: coeff_p2_ff	<= 11'h7d3 ;
						7'b01_0_1101	: coeff_p2_ff	<= 11'h7cd ;
						7'b01_0_1110	: coeff_p2_ff	<= 11'h7c8 ;
						7'b01_0_1111	: coeff_p2_ff	<= 11'h7c4 ;
						7'b01_1_0000	: coeff_p2_ff	<= 11'h7c0 ;
						7'b01_1_0001	: coeff_p2_ff	<= 11'h7bc ;
						7'b01_1_0010	: coeff_p2_ff	<= 11'h7b9 ;
						7'b01_1_0011	: coeff_p2_ff	<= 11'h7b6 ;
						7'b01_1_0100	: coeff_p2_ff	<= 11'h7b5 ;
						7'b01_1_0101	: coeff_p2_ff	<= 11'h7b4 ;
						7'b01_1_0110	: coeff_p2_ff	<= 11'h7b4 ;
						7'b01_1_0111	: coeff_p2_ff	<= 11'h7b5 ;
						7'b01_1_1000	: coeff_p2_ff	<= 11'h7b8 ;
						7'b01_1_1001	: coeff_p2_ff	<= 11'h7bb ;
						7'b01_1_1010	: coeff_p2_ff	<= 11'h7c0 ;
						7'b01_1_1011	: coeff_p2_ff	<= 11'h7c7 ;
						7'b01_1_1100	: coeff_p2_ff	<= 11'h7cf ;
						7'b01_1_1101	: coeff_p2_ff	<= 11'h7d8 ;
						7'b01_1_1110	: coeff_p2_ff	<= 11'h7e3 ;
						7'b01_1_1111	: coeff_p2_ff	<= 11'h7f0 ;

						7'b10_0_0000	: coeff_p2_ff	<= 11'h000 ;
						7'b10_0_0001	: coeff_p2_ff	<= 11'h7ff ;
						7'b10_0_0010	: coeff_p2_ff	<= 11'h7fd ;
						7'b10_0_0011	: coeff_p2_ff	<= 11'h7f9 ;
						7'b10_0_0100	: coeff_p2_ff	<= 11'h7f5 ;
						7'b10_0_0101	: coeff_p2_ff	<= 11'h7f0 ;
						7'b10_0_0110	: coeff_p2_ff	<= 11'h7ea ;
						7'b10_0_0111	: coeff_p2_ff	<= 11'h7e3 ;
						7'b10_0_1000	: coeff_p2_ff	<= 11'h7dc ;
						7'b10_0_1001	: coeff_p2_ff	<= 11'h7d4 ;
						7'b10_0_1010	: coeff_p2_ff	<= 11'h7cc ;
						7'b10_0_1011	: coeff_p2_ff	<= 11'h7c4 ;
						7'b10_0_1100	: coeff_p2_ff	<= 11'h7bc ;
						7'b10_0_1101	: coeff_p2_ff	<= 11'h7b4 ;
						7'b10_0_1110	: coeff_p2_ff	<= 11'h7ad ;
						7'b10_0_1111	: coeff_p2_ff	<= 11'h7a6 ;
						7'b10_1_0000	: coeff_p2_ff	<= 11'h7a0 ;
						7'b10_1_0001	: coeff_p2_ff	<= 11'h79a ;
						7'b10_1_0010	: coeff_p2_ff	<= 11'h795 ;
						7'b10_1_0011	: coeff_p2_ff	<= 11'h792 ;
						7'b10_1_0100	: coeff_p2_ff	<= 11'h78f ;
						7'b10_1_0101	: coeff_p2_ff	<= 11'h78e ;
						7'b10_1_0110	: coeff_p2_ff	<= 11'h78e ;
						7'b10_1_0111	: coeff_p2_ff	<= 11'h790 ;
						7'b10_1_1000	: coeff_p2_ff	<= 11'h794 ;
						7'b10_1_1001	: coeff_p2_ff	<= 11'h799 ;
						7'b10_1_1010	: coeff_p2_ff	<= 11'h7a0 ;
						7'b10_1_1011	: coeff_p2_ff	<= 11'h7aa ;
						7'b10_1_1100	: coeff_p2_ff	<= 11'h7b6 ;
						7'b10_1_1101	: coeff_p2_ff	<= 11'h7c4 ;
						7'b10_1_1110	: coeff_p2_ff	<= 11'h7d5 ;
						7'b10_1_1111	: coeff_p2_ff	<= 11'h7e9 ;

						7'b11_0_0000	: coeff_p2_ff	<= 11'h000 ;
						7'b11_0_0001	: coeff_p2_ff	<= 11'h7ff ;
						7'b11_0_0010	: coeff_p2_ff	<= 11'h7fc ;
						7'b11_0_0011	: coeff_p2_ff	<= 11'h7f7 ;
						7'b11_0_0100	: coeff_p2_ff	<= 11'h7f2 ;
						7'b11_0_0101	: coeff_p2_ff	<= 11'h7ea ;
						7'b11_0_0110	: coeff_p2_ff	<= 11'h7e2 ;
						7'b11_0_0111	: coeff_p2_ff	<= 11'h7d9 ;
						7'b11_0_1000	: coeff_p2_ff	<= 11'h7d0 ;
						7'b11_0_1001	: coeff_p2_ff	<= 11'h7c5 ;
						7'b11_0_1010	: coeff_p2_ff	<= 11'h7bb ;
						7'b11_0_1011	: coeff_p2_ff	<= 11'h7b0 ;
						7'b11_0_1100	: coeff_p2_ff	<= 11'h7a6 ;
						7'b11_0_1101	: coeff_p2_ff	<= 11'h79b ;
						7'b11_0_1110	: coeff_p2_ff	<= 11'h791 ;
						7'b11_0_1111	: coeff_p2_ff	<= 11'h788 ;
						7'b11_1_0000	: coeff_p2_ff	<= 11'h780 ;
						7'b11_1_0001	: coeff_p2_ff	<= 11'h778 ;
						7'b11_1_0010	: coeff_p2_ff	<= 11'h772 ;
						7'b11_1_0011	: coeff_p2_ff	<= 11'h76d ;
						7'b11_1_0100	: coeff_p2_ff	<= 11'h76a ;
						7'b11_1_0101	: coeff_p2_ff	<= 11'h768 ;
						7'b11_1_0110	: coeff_p2_ff	<= 11'h768 ;
						7'b11_1_0111	: coeff_p2_ff	<= 11'h76b ;
						7'b11_1_1000	: coeff_p2_ff	<= 11'h770 ;
						7'b11_1_1001	: coeff_p2_ff	<= 11'h777 ;
						7'b11_1_1010	: coeff_p2_ff	<= 11'h781 ;
						7'b11_1_1011	: coeff_p2_ff	<= 11'h78e ;
						7'b11_1_1100	: coeff_p2_ff	<= 11'h79e ;
						7'b11_1_1101	: coeff_p2_ff	<= 11'h7b1 ;
						7'b11_1_1110	: coeff_p2_ff	<= 11'h7c7 ;
						7'b11_1_1111	: coeff_p2_ff	<= 11'h7e1 ;
						default			: coeff_p2_ff	<= {11{1'bx}} ;
					endcase
				end
			end
		end
		else if ( Q_BIT == 12 ) begin : table_12bit
			//-------------------- Table Value Format -----------------------//
			//	 { Sign(1bit), Integral Part(1bit), Fraction Part(11bit) }	 //
			//---------------------------------------------------------------//
			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_m1_ff		<= 13'h0000 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_m1_ff	<= 13'h0000 ;
						7'b00_0_0001	: coeff_m1_ff	<= 13'h1fe1 ;
						7'b00_0_0010	: coeff_m1_ff	<= 13'h1fc7 ;
						7'b00_0_0011	: coeff_m1_ff	<= 13'h1fb1 ;
						7'b00_0_0100	: coeff_m1_ff	<= 13'h1f9e ;
						7'b00_0_0101	: coeff_m1_ff	<= 13'h1f8e ;
						7'b00_0_0110	: coeff_m1_ff	<= 13'h1f81 ;
						7'b00_0_0111	: coeff_m1_ff	<= 13'h1f77 ;
						7'b00_0_1000	: coeff_m1_ff	<= 13'h1f70 ;
						7'b00_0_1001	: coeff_m1_ff	<= 13'h1f6b ;
						7'b00_0_1010	: coeff_m1_ff	<= 13'h1f68 ;
						7'b00_0_1011	: coeff_m1_ff	<= 13'h1f68 ;
						7'b00_0_1100	: coeff_m1_ff	<= 13'h1f6a ;
						7'b00_0_1101	: coeff_m1_ff	<= 13'h1f6d ;
						7'b00_0_1110	: coeff_m1_ff	<= 13'h1f72 ;
						7'b00_0_1111	: coeff_m1_ff	<= 13'h1f78 ;
						7'b00_1_0000	: coeff_m1_ff	<= 13'h1f80 ;
						7'b00_1_0001	: coeff_m1_ff	<= 13'h1f88 ;
						7'b00_1_0010	: coeff_m1_ff	<= 13'h1f91 ;
						7'b00_1_0011	: coeff_m1_ff	<= 13'h1f9b ;
						7'b00_1_0100	: coeff_m1_ff	<= 13'h1fa6 ;
						7'b00_1_0101	: coeff_m1_ff	<= 13'h1fb0 ;
						7'b00_1_0110	: coeff_m1_ff	<= 13'h1fbb ;
						7'b00_1_0111	: coeff_m1_ff	<= 13'h1fc5 ;
						7'b00_1_1000	: coeff_m1_ff	<= 13'h1fd0 ;
						7'b00_1_1001	: coeff_m1_ff	<= 13'h1fd9 ;
						7'b00_1_1010	: coeff_m1_ff	<= 13'h1fe2 ;
						7'b00_1_1011	: coeff_m1_ff	<= 13'h1fea ;
						7'b00_1_1100	: coeff_m1_ff	<= 13'h1ff2 ;
						7'b00_1_1101	: coeff_m1_ff	<= 13'h1ff7 ;
						7'b00_1_1110	: coeff_m1_ff	<= 13'h1ffc ;
						7'b00_1_1111	: coeff_m1_ff	<= 13'h1fff ;

						7'b01_0_0000	: coeff_m1_ff	<= 13'h0000 ;
						7'b01_0_0001	: coeff_m1_ff	<= 13'h1fc3 ;
						7'b01_0_0010	: coeff_m1_ff	<= 13'h1f8f ;
						7'b01_0_0011	: coeff_m1_ff	<= 13'h1f62 ;
						7'b01_0_0100	: coeff_m1_ff	<= 13'h1f3c ;
						7'b01_0_0101	: coeff_m1_ff	<= 13'h1f1c ;
						7'b01_0_0110	: coeff_m1_ff	<= 13'h1f02 ;
						7'b01_0_0111	: coeff_m1_ff	<= 13'h1eee ;
						7'b01_0_1000	: coeff_m1_ff	<= 13'h1ee0 ;
						7'b01_0_1001	: coeff_m1_ff	<= 13'h1ed6 ;
						7'b01_0_1010	: coeff_m1_ff	<= 13'h1ed1 ;
						7'b01_0_1011	: coeff_m1_ff	<= 13'h1ed0 ;
						7'b01_0_1100	: coeff_m1_ff	<= 13'h1ed4 ;
						7'b01_0_1101	: coeff_m1_ff	<= 13'h1eda ;
						7'b01_0_1110	: coeff_m1_ff	<= 13'h1ee4 ;
						7'b01_0_1111	: coeff_m1_ff	<= 13'h1ef1 ;
						7'b01_1_0000	: coeff_m1_ff	<= 13'h1f00 ;
						7'b01_1_0001	: coeff_m1_ff	<= 13'h1f10 ;
						7'b01_1_0010	: coeff_m1_ff	<= 13'h1f23 ;
						7'b01_1_0011	: coeff_m1_ff	<= 13'h1f37 ;
						7'b01_1_0100	: coeff_m1_ff	<= 13'h1f4c ;
						7'b01_1_0101	: coeff_m1_ff	<= 13'h1f61 ;
						7'b01_1_0110	: coeff_m1_ff	<= 13'h1f76 ;
						7'b01_1_0111	: coeff_m1_ff	<= 13'h1f8b ;
						7'b01_1_1000	: coeff_m1_ff	<= 13'h1fa0 ;
						7'b01_1_1001	: coeff_m1_ff	<= 13'h1fb3 ;
						7'b01_1_1010	: coeff_m1_ff	<= 13'h1fc5 ;
						7'b01_1_1011	: coeff_m1_ff	<= 13'h1fd5 ;
						7'b01_1_1100	: coeff_m1_ff	<= 13'h1fe4 ;
						7'b01_1_1101	: coeff_m1_ff	<= 13'h1fef ;
						7'b01_1_1110	: coeff_m1_ff	<= 13'h1ff8 ;
						7'b01_1_1111	: coeff_m1_ff	<= 13'h1ffe ;

						7'b10_0_0000	: coeff_m1_ff	<= 13'h0000 ;
						7'b10_0_0001	: coeff_m1_ff	<= 13'h1fa5 ;
						7'b10_0_0010	: coeff_m1_ff	<= 13'h1f57 ;
						7'b10_0_0011	: coeff_m1_ff	<= 13'h1f13 ;
						7'b10_0_0100	: coeff_m1_ff	<= 13'h1eda ;
						7'b10_0_0101	: coeff_m1_ff	<= 13'h1eaa ;
						7'b10_0_0110	: coeff_m1_ff	<= 13'h1e83 ;
						7'b10_0_0111	: coeff_m1_ff	<= 13'h1e65 ;
						7'b10_0_1000	: coeff_m1_ff	<= 13'h1e50 ;
						7'b10_0_1001	: coeff_m1_ff	<= 13'h1e41 ;
						7'b10_0_1010	: coeff_m1_ff	<= 13'h1e3a ;
						7'b10_0_1011	: coeff_m1_ff	<= 13'h1e39 ;
						7'b10_0_1100	: coeff_m1_ff	<= 13'h1e3e ;
						7'b10_0_1101	: coeff_m1_ff	<= 13'h1e48 ;
						7'b10_0_1110	: coeff_m1_ff	<= 13'h1e56 ;
						7'b10_0_1111	: coeff_m1_ff	<= 13'h1e69 ;
						7'b10_1_0000	: coeff_m1_ff	<= 13'h1e80 ;
						7'b10_1_0001	: coeff_m1_ff	<= 13'h1e99 ;
						7'b10_1_0010	: coeff_m1_ff	<= 13'h1eb5 ;
						7'b10_1_0011	: coeff_m1_ff	<= 13'h1ed2 ;
						7'b10_1_0100	: coeff_m1_ff	<= 13'h1ef2 ;
						7'b10_1_0101	: coeff_m1_ff	<= 13'h1f11 ;
						7'b10_1_0110	: coeff_m1_ff	<= 13'h1f31 ;
						7'b10_1_0111	: coeff_m1_ff	<= 13'h1f51 ;
						7'b10_1_1000	: coeff_m1_ff	<= 13'h1f70 ;
						7'b10_1_1001	: coeff_m1_ff	<= 13'h1f8d ;
						7'b10_1_1010	: coeff_m1_ff	<= 13'h1fa8 ;
						7'b10_1_1011	: coeff_m1_ff	<= 13'h1fc0 ;
						7'b10_1_1100	: coeff_m1_ff	<= 13'h1fd6 ;
						7'b10_1_1101	: coeff_m1_ff	<= 13'h1fe7 ;
						7'b10_1_1110	: coeff_m1_ff	<= 13'h1ff4 ;
						7'b10_1_1111	: coeff_m1_ff	<= 13'h1ffd ;

						7'b11_0_0000	: coeff_m1_ff	<= 13'h0000 ;
						7'b11_0_0001	: coeff_m1_ff	<= 13'h1f87 ;
						7'b11_0_0010	: coeff_m1_ff	<= 13'h1f1f ;
						7'b11_0_0011	: coeff_m1_ff	<= 13'h1ec4 ;
						7'b11_0_0100	: coeff_m1_ff	<= 13'h1e78 ;
						7'b11_0_0101	: coeff_m1_ff	<= 13'h1e38 ;
						7'b11_0_0110	: coeff_m1_ff	<= 13'h1e05 ;
						7'b11_0_0111	: coeff_m1_ff	<= 13'h1ddd ;
						7'b11_0_1000	: coeff_m1_ff	<= 13'h1dc0 ;
						7'b11_0_1001	: coeff_m1_ff	<= 13'h1dac ;
						7'b11_0_1010	: coeff_m1_ff	<= 13'h1da3 ;
						7'b11_0_1011	: coeff_m1_ff	<= 13'h1da1 ;
						7'b11_0_1100	: coeff_m1_ff	<= 13'h1da8 ;
						7'b11_0_1101	: coeff_m1_ff	<= 13'h1db5 ;
						7'b11_0_1110	: coeff_m1_ff	<= 13'h1dc9 ;
						7'b11_0_1111	: coeff_m1_ff	<= 13'h1de2 ;
						7'b11_1_0000	: coeff_m1_ff	<= 13'h1e00 ;
						7'b11_1_0001	: coeff_m1_ff	<= 13'h1e21 ;
						7'b11_1_0010	: coeff_m1_ff	<= 13'h1e47 ;
						7'b11_1_0011	: coeff_m1_ff	<= 13'h1e6e ;
						7'b11_1_0100	: coeff_m1_ff	<= 13'h1e98 ;
						7'b11_1_0101	: coeff_m1_ff	<= 13'h1ec2 ;
						7'b11_1_0110	: coeff_m1_ff	<= 13'h1eed ;
						7'b11_1_0111	: coeff_m1_ff	<= 13'h1f17 ;
						7'b11_1_1000	: coeff_m1_ff	<= 13'h1f40 ;
						7'b11_1_1001	: coeff_m1_ff	<= 13'h1f66 ;
						7'b11_1_1010	: coeff_m1_ff	<= 13'h1f8b ;
						7'b11_1_1011	: coeff_m1_ff	<= 13'h1fab ;
						7'b11_1_1100	: coeff_m1_ff	<= 13'h1fc8 ;
						7'b11_1_1101	: coeff_m1_ff	<= 13'h1fdf ;
						7'b11_1_1110	: coeff_m1_ff	<= 13'h1ff1 ;
						7'b11_1_1111	: coeff_m1_ff	<= 13'h1ffc ;
						default			: coeff_m1_ff	<= {13{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_zr_ff		<= 13'h0800 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_zr_ff	<= 13'h0800 ;
						7'b00_0_0001	: coeff_zr_ff	<= 13'h07fc ;
						7'b00_0_0010	: coeff_zr_ff	<= 13'h07ed ;
						7'b00_0_0011	: coeff_zr_ff	<= 13'h07d6 ;
						7'b00_0_0100	: coeff_zr_ff	<= 13'h07b6 ;
						7'b00_0_0101	: coeff_zr_ff	<= 13'h078f ;
						7'b00_0_0110	: coeff_zr_ff	<= 13'h0761 ;
						7'b00_0_0111	: coeff_zr_ff	<= 13'h072c ;
						7'b00_0_1000	: coeff_zr_ff	<= 13'h06f0 ;
						7'b00_0_1001	: coeff_zr_ff	<= 13'h06b0 ;
						7'b00_0_1010	: coeff_zr_ff	<= 13'h066a ;
						7'b00_0_1011	: coeff_zr_ff	<= 13'h0620 ;
						7'b00_0_1100	: coeff_zr_ff	<= 13'h05d2 ;
						7'b00_0_1101	: coeff_zr_ff	<= 13'h0581 ;
						7'b00_0_1110	: coeff_zr_ff	<= 13'h052e ;
						7'b00_0_1111	: coeff_zr_ff	<= 13'h04d8 ;
						7'b00_1_0000	: coeff_zr_ff	<= 13'h0480 ;
						7'b00_1_0001	: coeff_zr_ff	<= 13'h0428 ;
						7'b00_1_0010	: coeff_zr_ff	<= 13'h03cf ;
						7'b00_1_0011	: coeff_zr_ff	<= 13'h0377 ;
						7'b00_1_0100	: coeff_zr_ff	<= 13'h031e ;
						7'b00_1_0101	: coeff_zr_ff	<= 13'h02c8 ;
						7'b00_1_0110	: coeff_zr_ff	<= 13'h0273 ;
						7'b00_1_0111	: coeff_zr_ff	<= 13'h0220 ;
						7'b00_1_1000	: coeff_zr_ff	<= 13'h01d0 ;
						7'b00_1_1001	: coeff_zr_ff	<= 13'h0184 ;
						7'b00_1_1010	: coeff_zr_ff	<= 13'h013c ;
						7'b00_1_1011	: coeff_zr_ff	<= 13'h00f9 ;
						7'b00_1_1100	: coeff_zr_ff	<= 13'h00ba ;
						7'b00_1_1101	: coeff_zr_ff	<= 13'h0082 ;
						7'b00_1_1110	: coeff_zr_ff	<= 13'h0050 ;
						7'b00_1_1111	: coeff_zr_ff	<= 13'h0024 ;

						7'b01_0_0000	: coeff_zr_ff	<= 13'h0800 ;
						7'b01_0_0001	: coeff_zr_ff	<= 13'h07fd ;
						7'b01_0_0010	: coeff_zr_ff	<= 13'h07f1 ;
						7'b01_0_0011	: coeff_zr_ff	<= 13'h07de ;
						7'b01_0_0100	: coeff_zr_ff	<= 13'h07c4 ;
						7'b01_0_0101	: coeff_zr_ff	<= 13'h07a4 ;
						7'b01_0_0110	: coeff_zr_ff	<= 13'h077e ;
						7'b01_0_0111	: coeff_zr_ff	<= 13'h0752 ;
						7'b01_0_1000	: coeff_zr_ff	<= 13'h0720 ;
						7'b01_0_1001	: coeff_zr_ff	<= 13'h06ea ;
						7'b01_0_1010	: coeff_zr_ff	<= 13'h06af ;
						7'b01_0_1011	: coeff_zr_ff	<= 13'h0670 ;
						7'b01_0_1100	: coeff_zr_ff	<= 13'h062c ;
						7'b01_0_1101	: coeff_zr_ff	<= 13'h05e6 ;
						7'b01_0_1110	: coeff_zr_ff	<= 13'h059c ;
						7'b01_0_1111	: coeff_zr_ff	<= 13'h054f ;
						7'b01_1_0000	: coeff_zr_ff	<= 13'h0500 ;
						7'b01_1_0001	: coeff_zr_ff	<= 13'h04b0 ;
						7'b01_1_0010	: coeff_zr_ff	<= 13'h045d ;
						7'b01_1_0011	: coeff_zr_ff	<= 13'h0409 ;
						7'b01_1_0100	: coeff_zr_ff	<= 13'h03b4 ;
						7'b01_1_0101	: coeff_zr_ff	<= 13'h035f ;
						7'b01_1_0110	: coeff_zr_ff	<= 13'h030a ;
						7'b01_1_0111	: coeff_zr_ff	<= 13'h02b5 ;
						7'b01_1_1000	: coeff_zr_ff	<= 13'h0260 ;
						7'b01_1_1001	: coeff_zr_ff	<= 13'h020d ;
						7'b01_1_1010	: coeff_zr_ff	<= 13'h01bb ;
						7'b01_1_1011	: coeff_zr_ff	<= 13'h016b ;
						7'b01_1_1100	: coeff_zr_ff	<= 13'h011c ;
						7'b01_1_1101	: coeff_zr_ff	<= 13'h00d1 ;
						7'b01_1_1110	: coeff_zr_ff	<= 13'h0088 ;
						7'b01_1_1111	: coeff_zr_ff	<= 13'h0042 ;

						7'b10_0_0000	: coeff_zr_ff	<= 13'h0800 ;
						7'b10_0_0001	: coeff_zr_ff	<= 13'h07fe ;
						7'b10_0_0010	: coeff_zr_ff	<= 13'h07f5 ;
						7'b10_0_0011	: coeff_zr_ff	<= 13'h07e6 ;
						7'b10_0_0100	: coeff_zr_ff	<= 13'h07d2 ;
						7'b10_0_0101	: coeff_zr_ff	<= 13'h07b9 ;
						7'b10_0_0110	: coeff_zr_ff	<= 13'h079b ;
						7'b10_0_0111	: coeff_zr_ff	<= 13'h0778 ;
						7'b10_0_1000	: coeff_zr_ff	<= 13'h0750 ;
						7'b10_0_1001	: coeff_zr_ff	<= 13'h0724 ;
						7'b10_0_1010	: coeff_zr_ff	<= 13'h06f4 ;
						7'b10_0_1011	: coeff_zr_ff	<= 13'h06bf ;
						7'b10_0_1100	: coeff_zr_ff	<= 13'h0686 ;
						7'b10_0_1101	: coeff_zr_ff	<= 13'h064a ;
						7'b10_0_1110	: coeff_zr_ff	<= 13'h060a ;
						7'b10_0_1111	: coeff_zr_ff	<= 13'h05c7 ;
						7'b10_1_0000	: coeff_zr_ff	<= 13'h0580 ;
						7'b10_1_0001	: coeff_zr_ff	<= 13'h0537 ;
						7'b10_1_0010	: coeff_zr_ff	<= 13'h04eb ;
						7'b10_1_0011	: coeff_zr_ff	<= 13'h049c ;
						7'b10_1_0100	: coeff_zr_ff	<= 13'h044a ;
						7'b10_1_0101	: coeff_zr_ff	<= 13'h03f7 ;
						7'b10_1_0110	: coeff_zr_ff	<= 13'h03a1 ;
						7'b10_1_0111	: coeff_zr_ff	<= 13'h034a ;
						7'b10_1_1000	: coeff_zr_ff	<= 13'h02f0 ;
						7'b10_1_1001	: coeff_zr_ff	<= 13'h0296 ;
						7'b10_1_1010	: coeff_zr_ff	<= 13'h023a ;
						7'b10_1_1011	: coeff_zr_ff	<= 13'h01dd ;
						7'b10_1_1100	: coeff_zr_ff	<= 13'h017e ;
						7'b10_1_1101	: coeff_zr_ff	<= 13'h0120 ;
						7'b10_1_1110	: coeff_zr_ff	<= 13'h00c0 ;
						7'b10_1_1111	: coeff_zr_ff	<= 13'h0060 ;

						7'b11_0_0000	: coeff_zr_ff	<= 13'h0800 ;
						7'b11_0_0001	: coeff_zr_ff	<= 13'h07fe ;
						7'b11_0_0010	: coeff_zr_ff	<= 13'h07f8 ;
						7'b11_0_0011	: coeff_zr_ff	<= 13'h07ee ;
						7'b11_0_0100	: coeff_zr_ff	<= 13'h07e0 ;
						7'b11_0_0101	: coeff_zr_ff	<= 13'h07ce ;
						7'b11_0_0110	: coeff_zr_ff	<= 13'h07b8 ;
						7'b11_0_0111	: coeff_zr_ff	<= 13'h079e ;
						7'b11_0_1000	: coeff_zr_ff	<= 13'h0780 ;
						7'b11_0_1001	: coeff_zr_ff	<= 13'h075e ;
						7'b11_0_1010	: coeff_zr_ff	<= 13'h0738 ;
						7'b11_0_1011	: coeff_zr_ff	<= 13'h070e ;
						7'b11_0_1100	: coeff_zr_ff	<= 13'h06e0 ;
						7'b11_0_1101	: coeff_zr_ff	<= 13'h06ae ;
						7'b11_0_1110	: coeff_zr_ff	<= 13'h0678 ;
						7'b11_0_1111	: coeff_zr_ff	<= 13'h063e ;
						7'b11_1_0000	: coeff_zr_ff	<= 13'h0600 ;
						7'b11_1_0001	: coeff_zr_ff	<= 13'h05bf ;
						7'b11_1_0010	: coeff_zr_ff	<= 13'h0578 ;
						7'b11_1_0011	: coeff_zr_ff	<= 13'h052f ;
						7'b11_1_0100	: coeff_zr_ff	<= 13'h04e0 ;
						7'b11_1_0101	: coeff_zr_ff	<= 13'h048f ;
						7'b11_1_0110	: coeff_zr_ff	<= 13'h0438 ;
						7'b11_1_0111	: coeff_zr_ff	<= 13'h03df ;
						7'b11_1_1000	: coeff_zr_ff	<= 13'h0380 ;
						7'b11_1_1001	: coeff_zr_ff	<= 13'h031f ;
						7'b11_1_1010	: coeff_zr_ff	<= 13'h02b8 ;
						7'b11_1_1011	: coeff_zr_ff	<= 13'h024f ;
						7'b11_1_1100	: coeff_zr_ff	<= 13'h01e0 ;
						7'b11_1_1101	: coeff_zr_ff	<= 13'h016f ;
						7'b11_1_1110	: coeff_zr_ff	<= 13'h00f8 ;
						7'b11_1_1111	: coeff_zr_ff	<= 13'h007f ;
						default			: coeff_zr_ff	<= {13{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p1_ff		<= 13'h0000 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_p1_ff	<= 13'h0000 ;
						7'b00_0_0001	: coeff_p1_ff	<= 13'h0024 ;
						7'b00_0_0010	: coeff_p1_ff	<= 13'h0050 ;
						7'b00_0_0011	: coeff_p1_ff	<= 13'h0082 ;
						7'b00_0_0100	: coeff_p1_ff	<= 13'h00ba ;
						7'b00_0_0101	: coeff_p1_ff	<= 13'h00f9 ;
						7'b00_0_0110	: coeff_p1_ff	<= 13'h013c ;
						7'b00_0_0111	: coeff_p1_ff	<= 13'h0184 ;
						7'b00_0_1000	: coeff_p1_ff	<= 13'h01d0 ;
						7'b00_0_1001	: coeff_p1_ff	<= 13'h0220 ;
						7'b00_0_1010	: coeff_p1_ff	<= 13'h0273 ;
						7'b00_0_1011	: coeff_p1_ff	<= 13'h02c8 ;
						7'b00_0_1100	: coeff_p1_ff	<= 13'h031e ;
						7'b00_0_1101	: coeff_p1_ff	<= 13'h0377 ;
						7'b00_0_1110	: coeff_p1_ff	<= 13'h03cf ;
						7'b00_0_1111	: coeff_p1_ff	<= 13'h0428 ;
						7'b00_1_0000	: coeff_p1_ff	<= 13'h0480 ;
						7'b00_1_0001	: coeff_p1_ff	<= 13'h04d8 ;
						7'b00_1_0010	: coeff_p1_ff	<= 13'h052e ;
						7'b00_1_0011	: coeff_p1_ff	<= 13'h0581 ;
						7'b00_1_0100	: coeff_p1_ff	<= 13'h05d2 ;
						7'b00_1_0101	: coeff_p1_ff	<= 13'h0620 ;
						7'b00_1_0110	: coeff_p1_ff	<= 13'h066a ;
						7'b00_1_0111	: coeff_p1_ff	<= 13'h06b0 ;
						7'b00_1_1000	: coeff_p1_ff	<= 13'h06f0 ;
						7'b00_1_1001	: coeff_p1_ff	<= 13'h072c ;
						7'b00_1_1010	: coeff_p1_ff	<= 13'h0761 ;
						7'b00_1_1011	: coeff_p1_ff	<= 13'h078f ;
						7'b00_1_1100	: coeff_p1_ff	<= 13'h07b6 ;
						7'b00_1_1101	: coeff_p1_ff	<= 13'h07d6 ;
						7'b00_1_1110	: coeff_p1_ff	<= 13'h07ed ;
						7'b00_1_1111	: coeff_p1_ff	<= 13'h07fc ;

						7'b01_0_0000	: coeff_p1_ff	<= 13'h0000 ;
						7'b01_0_0001	: coeff_p1_ff	<= 13'h0042 ;
						7'b01_0_0010	: coeff_p1_ff	<= 13'h0088 ;
						7'b01_0_0011	: coeff_p1_ff	<= 13'h00d1 ;
						7'b01_0_0100	: coeff_p1_ff	<= 13'h011c ;
						7'b01_0_0101	: coeff_p1_ff	<= 13'h016b ;
						7'b01_0_0110	: coeff_p1_ff	<= 13'h01bb ;
						7'b01_0_0111	: coeff_p1_ff	<= 13'h020d ;
						7'b01_0_1000	: coeff_p1_ff	<= 13'h0260 ;
						7'b01_0_1001	: coeff_p1_ff	<= 13'h02b5 ;
						7'b01_0_1010	: coeff_p1_ff	<= 13'h030a ;
						7'b01_0_1011	: coeff_p1_ff	<= 13'h035f ;
						7'b01_0_1100	: coeff_p1_ff	<= 13'h03b4 ;
						7'b01_0_1101	: coeff_p1_ff	<= 13'h0409 ;
						7'b01_0_1110	: coeff_p1_ff	<= 13'h045d ;
						7'b01_0_1111	: coeff_p1_ff	<= 13'h04b0 ;
						7'b01_1_0000	: coeff_p1_ff	<= 13'h0500 ;
						7'b01_1_0001	: coeff_p1_ff	<= 13'h054f ;
						7'b01_1_0010	: coeff_p1_ff	<= 13'h059c ;
						7'b01_1_0011	: coeff_p1_ff	<= 13'h05e6 ;
						7'b01_1_0100	: coeff_p1_ff	<= 13'h062c ;
						7'b01_1_0101	: coeff_p1_ff	<= 13'h0670 ;
						7'b01_1_0110	: coeff_p1_ff	<= 13'h06af ;
						7'b01_1_0111	: coeff_p1_ff	<= 13'h06ea ;
						7'b01_1_1000	: coeff_p1_ff	<= 13'h0720 ;
						7'b01_1_1001	: coeff_p1_ff	<= 13'h0752 ;
						7'b01_1_1010	: coeff_p1_ff	<= 13'h077e ;
						7'b01_1_1011	: coeff_p1_ff	<= 13'h07a4 ;
						7'b01_1_1100	: coeff_p1_ff	<= 13'h07c4 ;
						7'b01_1_1101	: coeff_p1_ff	<= 13'h07de ;
						7'b01_1_1110	: coeff_p1_ff	<= 13'h07f1 ;
						7'b01_1_1111	: coeff_p1_ff	<= 13'h07fd ;

						7'b10_0_0000	: coeff_p1_ff	<= 13'h0000 ;
						7'b10_0_0001	: coeff_p1_ff	<= 13'h0060 ;
						7'b10_0_0010	: coeff_p1_ff	<= 13'h00c0 ;
						7'b10_0_0011	: coeff_p1_ff	<= 13'h0120 ;
						7'b10_0_0100	: coeff_p1_ff	<= 13'h017e ;
						7'b10_0_0101	: coeff_p1_ff	<= 13'h01dd ;
						7'b10_0_0110	: coeff_p1_ff	<= 13'h023a ;
						7'b10_0_0111	: coeff_p1_ff	<= 13'h0296 ;
						7'b10_0_1000	: coeff_p1_ff	<= 13'h02f0 ;
						7'b10_0_1001	: coeff_p1_ff	<= 13'h034a ;
						7'b10_0_1010	: coeff_p1_ff	<= 13'h03a1 ;
						7'b10_0_1011	: coeff_p1_ff	<= 13'h03f7 ;
						7'b10_0_1100	: coeff_p1_ff	<= 13'h044a ;
						7'b10_0_1101	: coeff_p1_ff	<= 13'h049c ;
						7'b10_0_1110	: coeff_p1_ff	<= 13'h04eb ;
						7'b10_0_1111	: coeff_p1_ff	<= 13'h0537 ;
						7'b10_1_0000	: coeff_p1_ff	<= 13'h0580 ;
						7'b10_1_0001	: coeff_p1_ff	<= 13'h05c7 ;
						7'b10_1_0010	: coeff_p1_ff	<= 13'h060a ;
						7'b10_1_0011	: coeff_p1_ff	<= 13'h064a ;
						7'b10_1_0100	: coeff_p1_ff	<= 13'h0686 ;
						7'b10_1_0101	: coeff_p1_ff	<= 13'h06bf ;
						7'b10_1_0110	: coeff_p1_ff	<= 13'h06f4 ;
						7'b10_1_0111	: coeff_p1_ff	<= 13'h0724 ;
						7'b10_1_1000	: coeff_p1_ff	<= 13'h0750 ;
						7'b10_1_1001	: coeff_p1_ff	<= 13'h0778 ;
						7'b10_1_1010	: coeff_p1_ff	<= 13'h079b ;
						7'b10_1_1011	: coeff_p1_ff	<= 13'h07b9 ;
						7'b10_1_1100	: coeff_p1_ff	<= 13'h07d2 ;
						7'b10_1_1101	: coeff_p1_ff	<= 13'h07e6 ;
						7'b10_1_1110	: coeff_p1_ff	<= 13'h07f5 ;
						7'b10_1_1111	: coeff_p1_ff	<= 13'h07fe ;

						7'b11_0_0000	: coeff_p1_ff	<= 13'h0000 ;
						7'b11_0_0001	: coeff_p1_ff	<= 13'h007f ;
						7'b11_0_0010	: coeff_p1_ff	<= 13'h00f8 ;
						7'b11_0_0011	: coeff_p1_ff	<= 13'h016f ;
						7'b11_0_0100	: coeff_p1_ff	<= 13'h01e0 ;
						7'b11_0_0101	: coeff_p1_ff	<= 13'h024f ;
						7'b11_0_0110	: coeff_p1_ff	<= 13'h02b8 ;
						7'b11_0_0111	: coeff_p1_ff	<= 13'h031f ;
						7'b11_0_1000	: coeff_p1_ff	<= 13'h0380 ;
						7'b11_0_1001	: coeff_p1_ff	<= 13'h03df ;
						7'b11_0_1010	: coeff_p1_ff	<= 13'h0438 ;
						7'b11_0_1011	: coeff_p1_ff	<= 13'h048f ;
						7'b11_0_1100	: coeff_p1_ff	<= 13'h04e0 ;
						7'b11_0_1101	: coeff_p1_ff	<= 13'h052f ;
						7'b11_0_1110	: coeff_p1_ff	<= 13'h0578 ;
						7'b11_0_1111	: coeff_p1_ff	<= 13'h05bf ;
						7'b11_1_0000	: coeff_p1_ff	<= 13'h0600 ;
						7'b11_1_0001	: coeff_p1_ff	<= 13'h063e ;
						7'b11_1_0010	: coeff_p1_ff	<= 13'h0678 ;
						7'b11_1_0011	: coeff_p1_ff	<= 13'h06ae ;
						7'b11_1_0100	: coeff_p1_ff	<= 13'h06e0 ;
						7'b11_1_0101	: coeff_p1_ff	<= 13'h070e ;
						7'b11_1_0110	: coeff_p1_ff	<= 13'h0738 ;
						7'b11_1_0111	: coeff_p1_ff	<= 13'h075e ;
						7'b11_1_1000	: coeff_p1_ff	<= 13'h0780 ;
						7'b11_1_1001	: coeff_p1_ff	<= 13'h079e ;
						7'b11_1_1010	: coeff_p1_ff	<= 13'h07b8 ;
						7'b11_1_1011	: coeff_p1_ff	<= 13'h07ce ;
						7'b11_1_1100	: coeff_p1_ff	<= 13'h07e0 ;
						7'b11_1_1101	: coeff_p1_ff	<= 13'h07ee ;
						7'b11_1_1110	: coeff_p1_ff	<= 13'h07f8 ;
						7'b11_1_1111	: coeff_p1_ff	<= 13'h07fe ;
						default			: coeff_p1_ff	<= {13{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p2_ff		<= 13'h0000 ;
				end
				else if ( enable ) begin
					case ( { a_i , x_i } )
						7'b00_0_0000	: coeff_p2_ff	<= 13'h0000 ;
						7'b00_0_0001	: coeff_p2_ff	<= 13'h1fff ;
						7'b00_0_0010	: coeff_p2_ff	<= 13'h1ffc ;
						7'b00_0_0011	: coeff_p2_ff	<= 13'h1ff7 ;
						7'b00_0_0100	: coeff_p2_ff	<= 13'h1ff2 ;
						7'b00_0_0101	: coeff_p2_ff	<= 13'h1fea ;
						7'b00_0_0110	: coeff_p2_ff	<= 13'h1fe2 ;
						7'b00_0_0111	: coeff_p2_ff	<= 13'h1fd9 ;
						7'b00_0_1000	: coeff_p2_ff	<= 13'h1fd0 ;
						7'b00_0_1001	: coeff_p2_ff	<= 13'h1fc5 ;
						7'b00_0_1010	: coeff_p2_ff	<= 13'h1fbb ;
						7'b00_0_1011	: coeff_p2_ff	<= 13'h1fb0 ;
						7'b00_0_1100	: coeff_p2_ff	<= 13'h1fa6 ;
						7'b00_0_1101	: coeff_p2_ff	<= 13'h1f9b ;
						7'b00_0_1110	: coeff_p2_ff	<= 13'h1f91 ;
						7'b00_0_1111	: coeff_p2_ff	<= 13'h1f88 ;
						7'b00_1_0000	: coeff_p2_ff	<= 13'h1f80 ;
						7'b00_1_0001	: coeff_p2_ff	<= 13'h1f78 ;
						7'b00_1_0010	: coeff_p2_ff	<= 13'h1f72 ;
						7'b00_1_0011	: coeff_p2_ff	<= 13'h1f6d ;
						7'b00_1_0100	: coeff_p2_ff	<= 13'h1f6a ;
						7'b00_1_0101	: coeff_p2_ff	<= 13'h1f68 ;
						7'b00_1_0110	: coeff_p2_ff	<= 13'h1f68 ;
						7'b00_1_0111	: coeff_p2_ff	<= 13'h1f6b ;
						7'b00_1_1000	: coeff_p2_ff	<= 13'h1f70 ;
						7'b00_1_1001	: coeff_p2_ff	<= 13'h1f77 ;
						7'b00_1_1010	: coeff_p2_ff	<= 13'h1f81 ;
						7'b00_1_1011	: coeff_p2_ff	<= 13'h1f8e ;
						7'b00_1_1100	: coeff_p2_ff	<= 13'h1f9e ;
						7'b00_1_1101	: coeff_p2_ff	<= 13'h1fb1 ;
						7'b00_1_1110	: coeff_p2_ff	<= 13'h1fc7 ;
						7'b00_1_1111	: coeff_p2_ff	<= 13'h1fe1 ;

						7'b01_0_0000	: coeff_p2_ff	<= 13'h0000 ;
						7'b01_0_0001	: coeff_p2_ff	<= 13'h1ffe ;
						7'b01_0_0010	: coeff_p2_ff	<= 13'h1ff8 ;
						7'b01_0_0011	: coeff_p2_ff	<= 13'h1fef ;
						7'b01_0_0100	: coeff_p2_ff	<= 13'h1fe4 ;
						7'b01_0_0101	: coeff_p2_ff	<= 13'h1fd5 ;
						7'b01_0_0110	: coeff_p2_ff	<= 13'h1fc5 ;
						7'b01_0_0111	: coeff_p2_ff	<= 13'h1fb3 ;
						7'b01_0_1000	: coeff_p2_ff	<= 13'h1fa0 ;
						7'b01_0_1001	: coeff_p2_ff	<= 13'h1f8b ;
						7'b01_0_1010	: coeff_p2_ff	<= 13'h1f76 ;
						7'b01_0_1011	: coeff_p2_ff	<= 13'h1f61 ;
						7'b01_0_1100	: coeff_p2_ff	<= 13'h1f4c ;
						7'b01_0_1101	: coeff_p2_ff	<= 13'h1f37 ;
						7'b01_0_1110	: coeff_p2_ff	<= 13'h1f23 ;
						7'b01_0_1111	: coeff_p2_ff	<= 13'h1f10 ;
						7'b01_1_0000	: coeff_p2_ff	<= 13'h1f00 ;
						7'b01_1_0001	: coeff_p2_ff	<= 13'h1ef1 ;
						7'b01_1_0010	: coeff_p2_ff	<= 13'h1ee4 ;
						7'b01_1_0011	: coeff_p2_ff	<= 13'h1eda ;
						7'b01_1_0100	: coeff_p2_ff	<= 13'h1ed4 ;
						7'b01_1_0101	: coeff_p2_ff	<= 13'h1ed0 ;
						7'b01_1_0110	: coeff_p2_ff	<= 13'h1ed1 ;
						7'b01_1_0111	: coeff_p2_ff	<= 13'h1ed6 ;
						7'b01_1_1000	: coeff_p2_ff	<= 13'h1ee0 ;
						7'b01_1_1001	: coeff_p2_ff	<= 13'h1eee ;
						7'b01_1_1010	: coeff_p2_ff	<= 13'h1f02 ;
						7'b01_1_1011	: coeff_p2_ff	<= 13'h1f1c ;
						7'b01_1_1100	: coeff_p2_ff	<= 13'h1f3c ;
						7'b01_1_1101	: coeff_p2_ff	<= 13'h1f62 ;
						7'b01_1_1110	: coeff_p2_ff	<= 13'h1f8f ;
						7'b01_1_1111	: coeff_p2_ff	<= 13'h1fc3 ;

						7'b10_0_0000	: coeff_p2_ff	<= 13'h0000 ;
						7'b10_0_0001	: coeff_p2_ff	<= 13'h1ffd ;
						7'b10_0_0010	: coeff_p2_ff	<= 13'h1ff4 ;
						7'b10_0_0011	: coeff_p2_ff	<= 13'h1fe7 ;
						7'b10_0_0100	: coeff_p2_ff	<= 13'h1fd6 ;
						7'b10_0_0101	: coeff_p2_ff	<= 13'h1fc0 ;
						7'b10_0_0110	: coeff_p2_ff	<= 13'h1fa8 ;
						7'b10_0_0111	: coeff_p2_ff	<= 13'h1f8d ;
						7'b10_0_1000	: coeff_p2_ff	<= 13'h1f70 ;
						7'b10_0_1001	: coeff_p2_ff	<= 13'h1f51 ;
						7'b10_0_1010	: coeff_p2_ff	<= 13'h1f31 ;
						7'b10_0_1011	: coeff_p2_ff	<= 13'h1f11 ;
						7'b10_0_1100	: coeff_p2_ff	<= 13'h1ef2 ;
						7'b10_0_1101	: coeff_p2_ff	<= 13'h1ed2 ;
						7'b10_0_1110	: coeff_p2_ff	<= 13'h1eb5 ;
						7'b10_0_1111	: coeff_p2_ff	<= 13'h1e99 ;
						7'b10_1_0000	: coeff_p2_ff	<= 13'h1e80 ;
						7'b10_1_0001	: coeff_p2_ff	<= 13'h1e69 ;
						7'b10_1_0010	: coeff_p2_ff	<= 13'h1e56 ;
						7'b10_1_0011	: coeff_p2_ff	<= 13'h1e48 ;
						7'b10_1_0100	: coeff_p2_ff	<= 13'h1e3e ;
						7'b10_1_0101	: coeff_p2_ff	<= 13'h1e39 ;
						7'b10_1_0110	: coeff_p2_ff	<= 13'h1e3a ;
						7'b10_1_0111	: coeff_p2_ff	<= 13'h1e41 ;
						7'b10_1_1000	: coeff_p2_ff	<= 13'h1e50 ;
						7'b10_1_1001	: coeff_p2_ff	<= 13'h1e65 ;
						7'b10_1_1010	: coeff_p2_ff	<= 13'h1e83 ;
						7'b10_1_1011	: coeff_p2_ff	<= 13'h1eaa ;
						7'b10_1_1100	: coeff_p2_ff	<= 13'h1eda ;
						7'b10_1_1101	: coeff_p2_ff	<= 13'h1f13 ;
						7'b10_1_1110	: coeff_p2_ff	<= 13'h1f57 ;
						7'b10_1_1111	: coeff_p2_ff	<= 13'h1fa5 ;

						7'b11_0_0000	: coeff_p2_ff	<= 13'h0000 ;
						7'b11_0_0001	: coeff_p2_ff	<= 13'h1ffc ;
						7'b11_0_0010	: coeff_p2_ff	<= 13'h1ff1 ;
						7'b11_0_0011	: coeff_p2_ff	<= 13'h1fdf ;
						7'b11_0_0100	: coeff_p2_ff	<= 13'h1fc8 ;
						7'b11_0_0101	: coeff_p2_ff	<= 13'h1fab ;
						7'b11_0_0110	: coeff_p2_ff	<= 13'h1f8b ;
						7'b11_0_0111	: coeff_p2_ff	<= 13'h1f66 ;
						7'b11_0_1000	: coeff_p2_ff	<= 13'h1f40 ;
						7'b11_0_1001	: coeff_p2_ff	<= 13'h1f17 ;
						7'b11_0_1010	: coeff_p2_ff	<= 13'h1eed ;
						7'b11_0_1011	: coeff_p2_ff	<= 13'h1ec2 ;
						7'b11_0_1100	: coeff_p2_ff	<= 13'h1e98 ;
						7'b11_0_1101	: coeff_p2_ff	<= 13'h1e6e ;
						7'b11_0_1110	: coeff_p2_ff	<= 13'h1e47 ;
						7'b11_0_1111	: coeff_p2_ff	<= 13'h1e21 ;
						7'b11_1_0000	: coeff_p2_ff	<= 13'h1e00 ;
						7'b11_1_0001	: coeff_p2_ff	<= 13'h1de2 ;
						7'b11_1_0010	: coeff_p2_ff	<= 13'h1dc9 ;
						7'b11_1_0011	: coeff_p2_ff	<= 13'h1db5 ;
						7'b11_1_0100	: coeff_p2_ff	<= 13'h1da8 ;
						7'b11_1_0101	: coeff_p2_ff	<= 13'h1da1 ;
						7'b11_1_0110	: coeff_p2_ff	<= 13'h1da3 ;
						7'b11_1_0111	: coeff_p2_ff	<= 13'h1dac ;
						7'b11_1_1000	: coeff_p2_ff	<= 13'h1dc0 ;
						7'b11_1_1001	: coeff_p2_ff	<= 13'h1ddd ;
						7'b11_1_1010	: coeff_p2_ff	<= 13'h1e05 ;
						7'b11_1_1011	: coeff_p2_ff	<= 13'h1e38 ;
						7'b11_1_1100	: coeff_p2_ff	<= 13'h1e78 ;
						7'b11_1_1101	: coeff_p2_ff	<= 13'h1ec4 ;
						7'b11_1_1110	: coeff_p2_ff	<= 13'h1f1f ;
						7'b11_1_1111	: coeff_p2_ff	<= 13'h1f87 ;
						default			: coeff_p2_ff	<= {13{1'bx}} ;
					endcase
				end
			end
		end
	endgenerate

	assign coeff_m1_o	= coeff_m1_ff ;
	assign coeff_zr_o	= coeff_zr_ff ;
	assign coeff_p1_o	= coeff_p1_ff ;
	assign coeff_p2_o	= coeff_p2_ff ;

endmodule

`default_nettype wire
