//----------------------------------------------------------------------------------------
//Copyright (C) 2013 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		Synchronizer for pulse
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Feb. 19 2013	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		DATA_WIDTH	: Bitwidh of Input/Output data to be synchronized
//		GRAY_DEPTH 	: Width of the gray code counter used to synchronize input pulse
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk_in	: Clock signal that data_i is generated by
//		clk_out : Clock signal that data_o is to be synchronized with
//		rst_n	: reset_signal
//		data_i	: Input data
//		data_o	: Output data
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	cdc_multi_trns
	#(
		parameter								DATA_WIDTH	= 8 ,
		parameter								GRAY_DEPTH	= 1
	)
	(
		input	wire							clk_in			,
		input	wire							clk_out			,
		input	wire							rst_n			,

		input	wire	[ DATA_WIDTH-1	: 0 ]	data_i			,
		output	wire	[ DATA_WIDTH-1	: 0 ]	data_o
	) ;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg		[ DATA_WIDTH-1	: 0	]	hold_data_ff	;
	wire							din_hld_set		;
	wire							dout_set		;
	reg								din_hld_ff		;
	reg		[ DATA_WIDTH-1	: 0 ]	data_out_ff		;
	wire							din_hld_clr		;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	//clk_in
	always @( posedge clk_in or negedge rst_n ) begin
		if ( !rst_n ) begin
			hold_data_ff	<= { DATA_WIDTH{1'b0} } ;
		end
		else begin
			if ( din_hld_ff ==1'b0 )begin
				hold_data_ff	<= data_i ;
			end
			else begin
				hold_data_ff	<= hold_data_ff ;
			end
		end
	end

	assign din_hld_set	=	~din_hld_ff && ( hold_data_ff != data_i);

	always @( posedge clk_in or negedge rst_n ) begin
		if ( !rst_n ) begin
			din_hld_ff	<=1'b0 ;
		end
		else begin
			if ( din_hld_clr )begin
				din_hld_ff	<= 1'b0 ;
			end
			else if ( din_hld_set ) begin
				din_hld_ff	<= 1'b1 ;
			end
			else begin
				din_hld_ff	<= din_hld_ff ;
			end
		end
	end

	//clk_in -> clk_out
	cdc_pls_trns
		#(
			.GRAY_DEPTH	( GRAY_DEPTH )
		)
		u_cdc_pls_trns_io (
			.clk_in		( clk_in		),
			.clk_out	( clk_out		),
			.rst_n		( rst_n			),
			.pls_i		( din_hld_set	),
			.pls_o		( dout_set		)
		);

	//clk_out
	always @( posedge clk_out or negedge rst_n ) begin
		if ( !rst_n ) begin
			data_out_ff	<= { DATA_WIDTH{1'b0} } ;
		end
		else begin
			if ( dout_set ) begin
				data_out_ff	<= hold_data_ff ;
			end
			else begin
				data_out_ff	<= data_out_ff ;
			end
		end
	end

	assign	data_o		= data_out_ff ;

	//clk_out -> clk_in
	cdc_pls_trns
		#(
			.GRAY_DEPTH	( GRAY_DEPTH )
		)
		u_cdc_pls_trns_oi (
			.clk_in		( clk_out		),
			.clk_out	( clk_in		),
			.rst_n		( rst_n			),
			.pls_i		( dout_set		),
			.pls_o		( din_hld_clr	)
		);

endmodule
`default_nettype wire