//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		Picture Combination IP top module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Nov. 13 2012	: Initial version release
//		v1.1 May  28 2013	: Fix a bug that the last pixel_o is unexpected if the valid_o is made waited at the previous pixel
//		v1.2 May  28 2013	: Fix a bug that the pixel_o is unexpected if each field status inputed are not the same
//		v1.3 May  28 2013	: Fix a bug that ready_o is set for 2or3 clocks long at start timing of output frame
//		v1.4 Jun. 04 2013	: Fix a bug that Pic_comb goes wrong if ch#_on_i is changed during running
//		v1.5 Mar. 27 2014	: Signal end_flag_set is added
//
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT			data bit width,         user can select from among 6, 8, 10, and 12.
//		SIZE_BIT		display size bit width, user can select from 8 to 12.
//		ALPHA_BIT		alpha bit width,        user can select from 8 to 12.
//
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk 				: clock for all circuit
//		rst_n				: asynchronous reset ( low active )
//		srst				: synchronous reset
//		enable				: clock enable
//		param_update_i		: input updating setting parameter registers
//
//		smpl_mode_i			: input color format mode select
//							  2'b00 : 422 input
//							  2'b01 : 444 input
//		constant_data_i		: a value when outputting a constant data
//
//		width_i				: horizontal size of layer
//		height_i			: vertical size of layer
//
//		layer#_ch_i			: allocate channel for layer# ( # = 0, 1, 2, 3 )
//							  2'd0 : channel_0
//							  2'd1 : channel_1
//							  2'd2 : channel_2
//							  2'd3 : channel_3
//		ch#_on_i			: channel  ON/OFF
//							  1'b0 : OFF
//							  1'b1 : ON
//		ch#_alpha_i			: alpha value of channel#
//		ch#_hpos_i			: horizontal position of channel#
//		ch#_vpos_i			: vertical position of channel#
//
//		ch#_valid_i			: data valid of channel# input frame data.
//		ch#_frame_start_i	: frame start of channel# input frame data.
//		ch#_frame_end_i		: frame end of channel# input frame data.
//		ch#_pixel_i			: pixel data of channel# input frame data.
//		ch#_field_i			: field status of channel# input frame data.  1:even field  0:odd field
//		ch#_width_i			: horizontal size of channel# input frame data.
//		ch#_height_i		: vertical size of channel# input frame data.
//		ch#_ready_o			: data reception ready of channel# input frame data.
//
//		valid_o				: data valid of output frame data.
//		frame_start_o		: frame start of output frame data.
//		frame_end_o			: frame end of output frame data.
//		pixel_o				: pixel data of output frame data.
//		field_o				: field status of output frame data.  1:even field	0:odd field
//		width_o				: horizontal size of output frame data
//		height_o			: vertical size of input frame data
//		ready_i				: data reception ready of output frame data.
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	pc11_top (
	clk					,
	rst_n				,
	srst				,
	enable				,
	param_update_i		,

	smpl_mode_i			,
	constant_data_i		,		//	444 input
	layer3_con_i		, // custom
	layer3_cdata_i		, // custom

	width_i				,
	height_i			,

	layer0_ch_i			,
	layer1_ch_i			,
	layer2_ch_i			,
	layer3_ch_i			,

	ch0_on_i			,
	ch0_alpha_i			,
	ch0_hpos_i			,
	ch0_vpos_i			,

	ch1_on_i			,
	ch1_alpha_i			,
	ch1_hpos_i			,
	ch1_vpos_i			,

	ch2_on_i			,
	ch2_alpha_i			,
	ch2_hpos_i			,
	ch2_vpos_i			,

	ch3_on_i			,
	ch3_alpha_i			,
	ch3_hpos_i			,
	ch3_vpos_i			,

	ch0_valid_i			,
	ch0_frame_start_i	,
	ch0_frame_end_i		,
	ch0_pixel_i			,
	ch0_field_i			,
	ch0_width_i			,
	ch0_height_i		,
	ch0_ready_o			,

	ch1_valid_i			,
	ch1_frame_start_i	,
	ch1_frame_end_i		,
	ch1_pixel_i			,
	ch1_field_i			,
	ch1_width_i			,
	ch1_height_i		,
	ch1_ready_o			,

	ch2_valid_i			,
	ch2_frame_start_i	,
	ch2_frame_end_i		,
	ch2_pixel_i			,
	ch2_field_i			,
	ch2_width_i			,
	ch2_height_i		,
	ch2_ready_o			,

	ch3_valid_i			,
	ch3_frame_start_i	,
	ch3_frame_end_i		,
	ch3_pixel_i			,
	ch3_field_i			,
	ch3_width_i			,
	ch3_height_i		,
	ch3_ready_o			,

	valid_o				,
	frame_start_o		,
	frame_end_o			,
	pixel_o				,
	field_o				,
	width_o				,
	height_o			,
	ready_i
	);

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------

	parameter 	Q_BIT			=  8 ;
	parameter 	SIZE_BIT		= 11 ;
	parameter	ALPHA_BIT		=  8 ;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

	localparam	PLANE			= 3 ;	// only 3 ( only 444 format )
	localparam	PIXEL_BIT		= Q_BIT * PLANE ;
	localparam	MLTRSLT_BIT		= ( ALPHA_BIT + 1 ) * 2 ;
	localparam	MLTPIX_BIT		= Q_BIT + ALPHA_BIT ;

// =============================================================================
// PORT DECLARATION
// =============================================================================

	input	wire							clk					;
	input	wire							rst_n				;
	input	wire							srst				;
	input	wire							enable				;
	input	wire							param_update_i		;

	input	wire							smpl_mode_i			;	//	1:RGB444 or YUV444	0:YUV422
	input	wire	[ PIXEL_BIT - 1 : 0 ]	constant_data_i		;
	input	wire							layer3_con_i		; // custom
	input	wire	[ PIXEL_BIT - 1 : 0 ]	layer3_cdata_i		; // custom

	input	wire	[  SIZE_BIT - 1 : 0 ]	width_i				;
	input	wire	[  SIZE_BIT - 1 : 0 ]	height_i			;

	input	wire	[             1 : 0 ]	layer0_ch_i			;
	input	wire	[             1 : 0 ]	layer1_ch_i			;
	input	wire	[             1 : 0 ]	layer2_ch_i			;
	input	wire	[             1 : 0 ]	layer3_ch_i			;

	input	wire							ch0_on_i			;
	input	wire	[ ALPHA_BIT - 1 : 0 ]	ch0_alpha_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch0_hpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch0_vpos_i			;

	input	wire							ch1_on_i			;
	input	wire	[ ALPHA_BIT - 1 : 0 ]	ch1_alpha_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch1_hpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch1_vpos_i			;

	input	wire							ch2_on_i			;
	input	wire	[ ALPHA_BIT - 1 : 0 ]	ch2_alpha_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch2_hpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch2_vpos_i			;

	input	wire							ch3_on_i			;
	input	wire	[ ALPHA_BIT - 1 : 0 ]	ch3_alpha_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch3_hpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch3_vpos_i			;

	input	wire							ch0_valid_i			;
	input	wire							ch0_frame_start_i	;
	input	wire							ch0_frame_end_i		;
	input	wire	[ PIXEL_BIT - 1 : 0 ]	ch0_pixel_i			;
	input	wire							ch0_field_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch0_width_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch0_height_i		;
	output	wire							ch0_ready_o			;

	input	wire							ch1_valid_i			;
	input	wire							ch1_frame_start_i	;
	input	wire							ch1_frame_end_i		;
	input	wire	[ PIXEL_BIT - 1 : 0 ]	ch1_pixel_i			;
	input	wire							ch1_field_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch1_width_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch1_height_i		;
	output	wire							ch1_ready_o			;

	input	wire							ch2_valid_i			;
	input	wire							ch2_frame_start_i	;
	input	wire							ch2_frame_end_i		;
	input	wire	[ PIXEL_BIT - 1 : 0 ]	ch2_pixel_i			;
	input	wire							ch2_field_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch2_width_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch2_height_i		;
	output	wire							ch2_ready_o			;

	input	wire							ch3_valid_i			;
	input	wire							ch3_frame_start_i	;
	input	wire							ch3_frame_end_i		;
	input	wire	[ PIXEL_BIT - 1 : 0 ]	ch3_pixel_i			;
	input	wire							ch3_field_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch3_width_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch3_height_i		;
	output	wire							ch3_ready_o			;

	output	wire							valid_o				;
	output	wire							frame_start_o		;
	output	wire							frame_end_o			;
	output	wire	[ PIXEL_BIT - 1 : 0 ]	pixel_o				;
	output	wire							field_o				;
	output	wire	[  SIZE_BIT - 1 : 0 ]	width_o				;
	output	wire	[  SIZE_BIT - 1 : 0 ]	height_o			;
	input	wire							ready_i				;


// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	wire							init_end				;
	// operation parameter
	wire							smpl_mode				;
	wire	[   PIXEL_BIT - 1 : 0 ]	constant_data			;
	wire							layer3_con				; // custom
	wire	[   PIXEL_BIT - 1 : 0 ]	layer3_cdata			; // custom
	wire	[    SIZE_BIT - 1 : 0 ]	width					;
	wire	[    SIZE_BIT - 1 : 0 ]	height					;
	wire	[    SIZE_BIT - 1 : 0 ]	width_lat				;
	wire	[    SIZE_BIT - 1 : 0 ]	height_lat				;
	wire	[               1 : 0 ]	layer0_ch				;
	wire	[               1 : 0 ]	layer1_ch				;
	wire	[               1 : 0 ]	layer2_ch				;
	wire	[               1 : 0 ]	layer3_ch				;
	wire							ch0_on					;
	wire							ch1_on					;
	wire							ch2_on					;
	wire							ch3_on					;
	wire							layer0_on				;
	wire	[    SIZE_BIT - 1 : 0 ]	layer0_hstart			;
	wire	[    SIZE_BIT - 1 : 0 ]	layer0_vstart			;
	wire	[    SIZE_BIT - 1 : 0 ]	layer0_hend				;
	wire	[    SIZE_BIT - 1 : 0 ]	layer0_vend				;
	wire	[   ALPHA_BIT     : 0 ]	layer0_alpha0 			;
	wire	[   ALPHA_BIT     : 0 ]	layer0_alpha01 			;
	wire	[   ALPHA_BIT     : 0 ]	layer0_alpha02 			;
	wire	[   ALPHA_BIT     : 0 ]	layer0_alpha03 			;
	wire	[   ALPHA_BIT     : 0 ]	layer0_alpha012 		;
	wire	[   ALPHA_BIT     : 0 ]	layer0_alpha013 		;
	wire	[   ALPHA_BIT     : 0 ]	layer0_alpha023			;
	wire	[   ALPHA_BIT     : 0 ]	layer0_alpha0123		;
	wire							layer1_on				;
	wire	[    SIZE_BIT - 1 : 0 ]	layer1_hstart			;
	wire	[    SIZE_BIT - 1 : 0 ]	layer1_vstart			;
	wire	[    SIZE_BIT - 1 : 0 ]	layer1_hend				;
	wire	[    SIZE_BIT - 1 : 0 ]	layer1_vend				;
	wire	[   ALPHA_BIT     : 0 ]	layer1_alpha1			;
	wire	[   ALPHA_BIT     : 0 ]	layer1_alpha12			;
	wire	[   ALPHA_BIT     : 0 ]	layer1_alpha13			;
	wire	[   ALPHA_BIT     : 0 ]	layer1_alpha123			;
	wire							layer2_on				;
	wire	[    SIZE_BIT - 1 : 0 ]	layer2_hstart			;
	wire	[    SIZE_BIT - 1 : 0 ]	layer2_vstart			;
	wire	[    SIZE_BIT - 1 : 0 ]	layer2_hend				;
	wire	[    SIZE_BIT - 1 : 0 ]	layer2_vend				;
	wire	[   ALPHA_BIT     : 0 ]	layer2_alpha2			;
	wire	[   ALPHA_BIT     : 0 ]	layer2_alpha23			;
	wire							layer3_on				;
	wire	[    SIZE_BIT - 1 : 0 ]	layer3_hstart			;
	wire	[    SIZE_BIT - 1 : 0 ]	layer3_vstart			;
	wire	[    SIZE_BIT - 1 : 0 ]	layer3_hend				;
	wire	[    SIZE_BIT - 1 : 0 ]	layer3_vend				;
	wire	[   ALPHA_BIT     : 0 ]	layer3_alpha3			;

	// Channel 0 Flow control
	wire							ch0_de					;
	wire							ch0_rd					;
	wire							ch0_valid				;
	wire	[   PIXEL_BIT - 1 : 0 ]	ch0_pixel				;
	wire							ch0_field				;
	wire	[    SIZE_BIT - 1 : 0 ]	ch0_width				;
	wire	[    SIZE_BIT - 1 : 0 ]	ch0_height				;
	wire							ch0_frame_start			;
	wire							ch0_next_frame			;
	// Channel 1 Flow control
	wire							ch1_de					;
	wire							ch1_rd					;
	wire							ch1_valid				;
	wire	[   PIXEL_BIT - 1 : 0 ]	ch1_pixel				;
	wire							ch1_field				;
	wire	[    SIZE_BIT - 1 : 0 ]	ch1_width				;
	wire	[    SIZE_BIT - 1 : 0 ]	ch1_height				;
	wire							ch1_frame_start			;
	wire							ch1_next_frame			;
	// Channel 2 Flow control
	wire							ch2_de					;
	wire							ch2_rd					;
	wire							ch2_valid				;
	wire	[   PIXEL_BIT - 1 : 0 ]	ch2_pixel				;
	wire							ch2_field				;
	wire	[    SIZE_BIT - 1 : 0 ]	ch2_width				;
	wire	[    SIZE_BIT - 1 : 0 ]	ch2_height				;
	wire							ch2_frame_start			;
	wire							ch2_next_frame			;
	// Channel 3 Flow control
	wire							ch3_de					;
	wire							ch3_rd					;
	wire							ch3_valid				;
	wire	[   PIXEL_BIT - 1 : 0 ]	ch3_pixel				;
	wire							ch3_field				;
	wire	[    SIZE_BIT - 1 : 0 ]	ch3_width				;
	wire	[    SIZE_BIT - 1 : 0 ]	ch3_height				;
	wire							ch3_frame_start			;
	wire							ch3_next_frame			;
	// Output Timing
	// background
	reg		[    SIZE_BIT - 1 : 0 ]	h_count_ff				;
	reg		[    SIZE_BIT - 1 : 0 ]	v_count_ff				;
	wire							h_max					;
	wire							v_max					;
	reg								count_en_ff				;
	wire							count_en				;
	reg								output_sel_ff			;
	// Layer Output Timing
	// Layer 0
	wire							layer0_hde				;
	wire							layer0_vde				;
	reg								layer0_de_ff			;
	wire							layer0_valid			;
	wire	[   PIXEL_BIT - 1 : 0 ]	layer0_pixel			;
	wire							layer0_field			;
	wire							layer0_0123				;
	wire							layer0_023				;
	wire							layer0_013				;
	wire							layer0_012				;
	wire							layer0_03				;
	wire							layer0_02				;
	wire							layer0_01				;
	wire							layer0_0				;
	wire	[   ALPHA_BIT     : 0 ]	layer0_alpha			;
	wire							layer0_data_valid		;
	//	Layer 1
	wire							layer1_hde				;
	wire							layer1_vde				;
	reg								layer1_de_ff			;
	wire							layer1_valid			;
	wire	[   PIXEL_BIT - 1 : 0 ]	layer1_pixel			;
	wire							layer1_field			;
	wire							layer1_123				;
	wire							layer1_13				;
	wire							layer1_12				;
	wire							layer1_1				;
	wire	[   ALPHA_BIT     : 0 ]	layer1_alpha			;
	wire							layer1_data_valid		;
	//	Layer 2
	wire							layer2_hde				;
	wire							layer2_vde				;
	reg								layer2_de_ff			;
	wire							layer2_valid			;
	wire	[   PIXEL_BIT - 1 : 0 ]	layer2_pixel			;
	wire							layer2_field			;
	wire							layer2_23				;
	wire							layer2_2				;
	wire	[   ALPHA_BIT     : 0 ]	layer2_alpha			;
	wire							layer2_data_valid		;
	//	Layer 3
	wire							layer3_hde				;
	wire							layer3_vde				;
	reg								layer3_de_ff			;
	wire							layer3_de				;
	wire							layer3_valid			;
	wire	[   PIXEL_BIT - 1 : 0 ]	layer3_pixel			;
	wire							layer3_field			;
	wire							layer3_3				;
	wire	[   ALPHA_BIT     : 0 ]	layer3_alpha			;
	wire							layer3_data_valid		;
	wire							data_valid				;
	// combination
	reg		[   PIXEL_BIT - 1 : 0 ]	const_data_lat1_ff		;
	reg		[   PIXEL_BIT - 1 : 0 ]	layer0_pixel_lat1_ff	;
	reg		[   PIXEL_BIT - 1 : 0 ]	layer1_pixel_lat1_ff	;
	reg		[   PIXEL_BIT - 1 : 0 ]	layer2_pixel_lat1_ff	;
	reg		[   PIXEL_BIT - 1 : 0 ]	layer3_pixel_lat1_ff	;
	reg		[   ALPHA_BIT     : 0 ]	layer0_alpha_lat1_ff	;
	reg		[   ALPHA_BIT     : 0 ]	layer1_alpha_lat1_ff	;
	reg		[   ALPHA_BIT     : 0 ]	layer2_alpha_lat1_ff	;
	reg		[   ALPHA_BIT     : 0 ]	layer3_alpha_lat1_ff	;
	wire	[               2 : 0 ]	ry_dummy_bit			;
	wire	[  MLTPIX_BIT - 1 : 0 ]	ry_multadd_pixel		;
	wire	[               2 : 0 ]	gu_dummy_bit			;
	wire	[  MLTPIX_BIT - 1 : 0 ]	gu_multadd_pixel		;
	wire	[               2 : 0 ]	bv_dummy_bit			;
	wire	[  MLTPIX_BIT - 1 : 0 ]	bv_multadd_pixel		;
	//	Background Alpha
	wire	[   ALPHA_BIT     : 0 ]	bg_alpha				;
	reg		[   PIXEL_BIT - 1 : 0 ]	const_data_lat2_ff		;
	reg		[   ALPHA_BIT     : 0 ]	bg_alpha_ff				;
	wire						   	const_ry_dummy_bit		;
	wire	[  MLTPIX_BIT - 1 : 0 ]	const_ry_mult			;
	wire						   	const_gu_dummy_bit		;
	wire	[  MLTPIX_BIT - 1 : 0 ]	const_gu_mult			;
	wire						   	const_bv_dummy_bit		;
	wire	[  MLTPIX_BIT - 1 : 0 ]	const_bv_mult			;
	// Flag Generate
	reg								base_field_ff			;
	reg								last_field_ff			;
	wire							prog_frame				;
	reg								base_input_ff			;
	reg								all_off_ff				;
	reg								base_input_lat_ff		;
	reg								base_input_ris_ff		;
	reg								start_flag_ff			;
	wire							end_flag				;
	reg		[    SIZE_BIT - 1 : 0 ]	width_value_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	height_value_ff			;
	reg								count_en_lat_ff			;
	reg								start_flag_lat_ff		;
	reg								start_flag_lat1_ff		;
	reg								start_flag_lat2_ff		;
	reg								start_flag_lat3_ff		;
	reg								start_flag_lat4_ff		;
//	reg								end_flag_lat_ff			; //v1.1
	wire							end_flag_adj			;
	wire							end_flag_set			; //v1.5

	reg								end_flag_lat1_ff		;
	reg								end_flag_lat2_ff		;
	reg								end_flag_lat3_ff		;
	reg								end_flag_lat4_ff		;
	reg								data_valid_lat1_ff		;
	reg								data_valid_lat2_ff		;
	reg								data_valid_lat3_ff		;
	reg								data_valid_lat4_ff		;
	reg								output_sel_lat1_ff		;
	reg								output_sel_lat2_ff		;
	reg								output_sel_lat3_ff		;
	reg								output_sel_lat4_ff		;
	// frame output
	reg								valid_o_ff				;
	reg								frame_start_o_ff		;
	reg								frame_end_o_ff			;
	wire	[  MLTPIX_BIT - 1 : 0 ]	ry_pixel_add_const		;
	wire	[  MLTPIX_BIT - 1 : 0 ]	gu_pixel_add_const		;
	wire	[  MLTPIX_BIT - 1 : 0 ]	gubv_pixel_add_const	;
	wire	[  MLTPIX_BIT - 1 : 0 ]	bv_pixel_add_const		;
	reg		[   PIXEL_BIT - 1 : 0 ]	pixel_o_ff				;
	reg								field_o_ff				;
	reg		[    SIZE_BIT - 1 : 0 ]	width_o_ff				;
	reg		[    SIZE_BIT - 1 : 0 ]	height_o_ff				;

	reg								end_flag_lat_adj_ff		; //v1.1

	reg								base_param_valid_ff		; //v1.2

	wire							ch0_on_prm_lat			; //v1.4
	wire							ch1_on_prm_lat			; //v1.4
	wire							ch2_on_prm_lat			; //v1.4
	wire							ch3_on_prm_lat			; //v1.4

	wire							ch0_on_fs_lat			; //v1.4
	wire							ch1_on_fs_lat			; //v1.4
	wire							ch2_on_fs_lat			; //v1.4
	wire							ch3_on_fs_lat			; //v1.4

	wire							old_mode				; //v1.4

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

//v1.4
	assign	ch0_on	= ( ( ch0_on_fs_lat & old_mode ) | ( ch0_on_prm_lat & ~old_mode ) ) ;
	assign	ch1_on	= ( ( ch1_on_fs_lat & old_mode ) | ( ch1_on_prm_lat & ~old_mode ) ) ;
	assign	ch2_on	= ( ( ch2_on_fs_lat & old_mode ) | ( ch2_on_prm_lat & ~old_mode ) ) ;
	assign	ch3_on	= ( ( ch3_on_fs_lat & old_mode ) | ( ch3_on_prm_lat & ~old_mode ) ) ;

// ch#_on latch timing adjustment
	assign	old_mode	= 1'b0	;

// ================================== //
//  Parameter Latch & Update Timing = //
// ================================== //

	pc11_param
		# (
			.ALPHA_BIT		( ALPHA_BIT	) ,
			.SIZE_BIT		( SIZE_BIT	) ,
			.PIXEL_BIT		( PIXEL_BIT	)
		)
		u_pc11_param (
			.clk					( clk				) ,
			.rst_n					( rst_n				) ,
			.srst					( srst				) ,
			.param_update_i			( param_update_i	) ,
			.end_flag_i				( end_flag_adj		) ,
			.start_flag_i			( start_flag_ff		) ,
			.all_off_i				( all_off_ff		) ,

			.smpl_mode_i			( smpl_mode_i		) ,
			.constant_data_i		( constant_data_i	) ,
			.layer3_con_i			( layer3_con_i		) , // custom
			.layer3_cdata_i			( layer3_cdata_i	) , // custom

			.width_i				( width_i			) ,
			.height_i				( height_i			) ,

			.layer0_ch_i			( layer0_ch_i		) ,
			.layer1_ch_i			( layer1_ch_i		) ,
			.layer2_ch_i			( layer2_ch_i		) ,
			.layer3_ch_i			( layer3_ch_i		) ,

			.ch0_on_i				( ch0_on_i			) ,
			.ch0_alpha_i			( ch0_alpha_i		) ,
			.ch0_hpos_i				( ch0_hpos_i		) ,
			.ch0_vpos_i				( ch0_vpos_i		) ,
			.ch0_width_i			( ch0_width			) ,
			.ch0_height_i			( ch0_height		) ,

			.ch1_on_i				( ch1_on_i			) ,
			.ch1_alpha_i			( ch1_alpha_i		) ,
			.ch1_hpos_i				( ch1_hpos_i		) ,
			.ch1_vpos_i				( ch1_vpos_i		) ,
			.ch1_width_i			( ch1_width			) ,
			.ch1_height_i			( ch1_height		) ,

			.ch2_on_i				( ch2_on_i			) ,
			.ch2_alpha_i			( ch2_alpha_i		) ,
			.ch2_hpos_i				( ch2_hpos_i		) ,
			.ch2_vpos_i				( ch2_vpos_i		) ,
			.ch2_width_i			( ch2_width			) ,
			.ch2_height_i			( ch2_height		) ,

			.ch3_on_i				( ch3_on_i			) ,
			.ch3_alpha_i			( ch3_alpha_i		) ,
			.ch3_hpos_i				( ch3_hpos_i		) ,
			.ch3_vpos_i				( ch3_vpos_i		) ,
			.ch3_width_i			( ch3_width			) ,
			.ch3_height_i			( ch3_height		) ,

			.ch0_frame_start_i		( ch0_frame_start	) ,
			.ch1_frame_start_i		( ch1_frame_start	) ,
			.ch2_frame_start_i		( ch2_frame_start	) ,
			.ch3_frame_start_i		( ch3_frame_start	) ,

			.smpl_mode_o			( smpl_mode			) ,
			.constant_data_o		( constant_data		) ,
			.layer3_con_o			( layer3_con		) , // custom
			.layer3_cdata_o			( layer3_cdata		) , // custom

			.width_o				( width				) ,
			.height_o				( height			) ,
			.width_lat_o			( width_lat			) ,
			.height_lat_o			( height_lat		) ,
//////////////////////////////////////////////////////////////////////////
 //v1.4
//			.ch0_on_o				( ch0_on			) ,
//			.ch1_on_o				( ch1_on			) ,
//			.ch2_on_o				( ch2_on			) ,
//			.ch3_on_o				( ch3_on			) ,

			.ch0_on_o				( ch0_on_fs_lat		) ,
			.ch1_on_o				( ch1_on_fs_lat		) ,
			.ch2_on_o				( ch2_on_fs_lat		) ,
			.ch3_on_o				( ch3_on_fs_lat		) ,

			.ch0_on_prm_lat_o		( ch0_on_prm_lat	) ,
			.ch1_on_prm_lat_o		( ch1_on_prm_lat	) ,
			.ch2_on_prm_lat_o		( ch2_on_prm_lat	) ,
			.ch3_on_prm_lat_o		( ch3_on_prm_lat	) ,

////////////////////////////////////////////////////////////////////////////////
			.layer0_on_o			( layer0_on			) ,
			.layer0_ch_o			( layer0_ch			) ,
			.layer0_hstart_o		( layer0_hstart		) ,
			.layer0_vstart_o		( layer0_vstart		) ,
			.layer0_hend_o			( layer0_hend		) ,
			.layer0_vend_o			( layer0_vend		) ,
			.layer1_on_o			( layer1_on			) ,
			.layer1_ch_o			( layer1_ch			) ,
			.layer1_hstart_o		( layer1_hstart		) ,
			.layer1_vstart_o		( layer1_vstart		) ,
			.layer1_hend_o			( layer1_hend		) ,
			.layer1_vend_o			( layer1_vend		) ,
			.layer2_on_o			( layer2_on			) ,
			.layer2_ch_o			( layer2_ch			) ,
			.layer2_hstart_o		( layer2_hstart		) ,
			.layer2_vstart_o		( layer2_vstart		) ,
			.layer2_hend_o			( layer2_hend		) ,
			.layer2_vend_o			( layer2_vend		) ,
			.layer3_on_o			( layer3_on			) ,
			.layer3_ch_o			( layer3_ch			) ,
			.layer3_hstart_o		( layer3_hstart		) ,
			.layer3_vstart_o		( layer3_vstart		) ,
			.layer3_hend_o			( layer3_hend		) ,
			.layer3_vend_o			( layer3_vend		) ,

			.layer0_alpha0_o 		( layer0_alpha0 	) ,
			.layer0_alpha01_o 		( layer0_alpha01 	) ,
			.layer0_alpha02_o 		( layer0_alpha02 	) ,
			.layer0_alpha03_o 		( layer0_alpha03 	) ,
			.layer0_alpha012_o 		( layer0_alpha012 	) ,
			.layer0_alpha013_o 		( layer0_alpha013 	) ,
			.layer0_alpha023_o		( layer0_alpha023	) ,
			.layer0_alpha0123_o		( layer0_alpha0123	) ,
			.layer1_alpha1_o		( layer1_alpha1		) ,
			.layer1_alpha12_o		( layer1_alpha12	) ,
			.layer1_alpha13_o		( layer1_alpha13	) ,
			.layer1_alpha123_o		( layer1_alpha123	) ,
			.layer2_alpha2_o		( layer2_alpha2		) ,
			.layer2_alpha23_o		( layer2_alpha23	) ,
			.layer3_alpha3_o		( layer3_alpha3		) ,

			.init_end_o				( init_end			)
		) ;

// ========================= //
//  Input Data Flow control  //
// ========================= //
	// ------------------------ //
	//  Channel 0 Flow control  //
	// ------------------------ //

	assign	ch0_de	= ( (   ready_i == 1'b0  ) & 1'b0 ) |
					  ( ( layer0_ch == 2'b00 ) & layer0_de_ff ) |
					  ( ( layer1_ch == 2'b00 ) & layer1_de_ff ) |
					  ( ( layer2_ch == 2'b00 ) & layer2_de_ff ) |
					  ( ( layer3_ch == 2'b00 ) & layer3_de_ff ) ;
	assign	ch0_rd	= count_en & ~ch0_next_frame & ch0_de ;

	pc11_in_buf
		# (
			.SIZE_BIT			( SIZE_BIT				) ,
			.PIXEL_BIT			( PIXEL_BIT				)
		)
		u_ch0_in_buf (
			.clk				( clk					) ,
			.rst_n				( rst_n					) ,
			.srst				( srst					) ,
			.enable				( enable				) ,

			.count_en_i			( count_en_ff			) ,
			.end_flag_i			( end_flag_adj			) ,
			.ready_i			( ready_i				) ,
			.end_flag_set_i		( end_flag_set			) , //v1.5

			.base_param_valid_i	( base_param_valid_ff	) , //v1.2
			.base_field_i		( base_field_ff			) , //v1.2

			.ch_on_i			( ch0_on				) , //v1.3

			.valid_i			( ch0_valid_i			) ,
			.frame_start_i		( ch0_frame_start_i		) ,
			.frame_end_i		( ch0_frame_end_i		) ,
			.pixel_i			( ch0_pixel_i			) ,
			.field_i			( ch0_field_i			) ,
			.width_i			( ch0_width_i			) ,
			.height_i			( ch0_height_i			) ,
			.ready_o			( ch0_ready_o			) ,

			.rd_i				( ch0_rd				) ,
			.valid_o			( ch0_valid				) ,
			.pixel_o			( ch0_pixel				) ,
			.field_o			( ch0_field				) ,
			.width_o			( ch0_width				) ,
			.height_o			( ch0_height			) ,
			.frame_start_o		( ch0_frame_start		) ,
			.next_frame_o		( ch0_next_frame		)
		) ;


	// ------------------------ //
	//  Channel 1 Flow control  //
	// ------------------------ //

	assign	ch1_de	= ( (   ready_i == 1'b0  ) & 1'b0 ) |
					  ( ( layer0_ch == 2'b01 ) & layer0_de_ff ) |
					  ( ( layer1_ch == 2'b01 ) & layer1_de_ff ) |
					  ( ( layer2_ch == 2'b01 ) & layer2_de_ff ) |
					  ( ( layer3_ch == 2'b01 ) & layer3_de_ff ) ;
	assign	ch1_rd	= count_en & ~ch1_next_frame & ch1_de ;

	pc11_in_buf
		# (
			.SIZE_BIT			( SIZE_BIT				) ,
			.PIXEL_BIT			( PIXEL_BIT				)
		)
		u_ch1_in_buf (
			.clk				( clk					) ,
			.rst_n				( rst_n					) ,
			.srst				( srst					) ,
			.enable				( enable				) ,

			.count_en_i			( count_en_ff			) ,
			.end_flag_i			( end_flag_adj			) ,
			.ready_i			( ready_i				) ,
			.end_flag_set_i		( end_flag_set			) , //v1.5

			.base_param_valid_i	( base_param_valid_ff	) , //v1.2
			.base_field_i		( base_field_ff			) , //v1.2

			.ch_on_i			( ch1_on				) , //v1.3

			.valid_i			( ch1_valid_i			) ,
			.frame_start_i		( ch1_frame_start_i		) ,
			.frame_end_i		( ch1_frame_end_i		) ,
			.pixel_i			( ch1_pixel_i			) ,
			.field_i			( ch1_field_i			) ,
			.width_i			( ch1_width_i			) ,
			.height_i			( ch1_height_i			) ,
			.ready_o			( ch1_ready_o			) ,

			.rd_i				( ch1_rd				) ,
			.valid_o			( ch1_valid				) ,
			.pixel_o			( ch1_pixel				) ,
			.field_o			( ch1_field				) ,
			.width_o			( ch1_width				) ,
			.height_o			( ch1_height			) ,
			.frame_start_o		( ch1_frame_start		) ,
			.next_frame_o		( ch1_next_frame		)
		) ;


	// ------------------------ //
	//  Channel 2 Flow control  //
	// ------------------------ //

	assign	ch2_de	= ( (   ready_i == 1'b0  ) & 1'b0 ) |
					  ( ( layer0_ch == 2'b10 ) & layer0_de_ff ) |
					  ( ( layer1_ch == 2'b10 ) & layer1_de_ff ) |
					  ( ( layer2_ch == 2'b10 ) & layer2_de_ff ) |
					  ( ( layer3_ch == 2'b10 ) & layer3_de_ff ) ;
	assign	ch2_rd	= count_en & ~ch2_next_frame & ch2_de ;

	pc11_in_buf
		# (
			.SIZE_BIT			( SIZE_BIT				) ,
			.PIXEL_BIT			( PIXEL_BIT				)
		)
		u_ch2_in_buf (
			.clk				( clk					) ,
			.rst_n				( rst_n					) ,
			.srst				( srst					) ,
			.enable				( enable				) ,

			.count_en_i			( count_en_ff			) ,
			.end_flag_i			( end_flag_adj			) ,
			.ready_i			( ready_i				) ,
			.end_flag_set_i		( end_flag_set			) , //v1.5

			.base_param_valid_i	( base_param_valid_ff	) , //v1.2
			.base_field_i		( base_field_ff			) , //v1.2

			.ch_on_i			( ch2_on				) , //v1.3

			.valid_i			( ch2_valid_i			) ,
			.frame_start_i		( ch2_frame_start_i		) ,
			.frame_end_i		( ch2_frame_end_i		) ,
			.pixel_i			( ch2_pixel_i			) ,
			.field_i			( ch2_field_i			) ,
			.width_i			( ch2_width_i			) ,
			.height_i			( ch2_height_i			) ,
			.ready_o			( ch2_ready_o			) ,

			.rd_i				( ch2_rd				) ,
			.valid_o			( ch2_valid				) ,
			.pixel_o			( ch2_pixel				) ,
			.field_o			( ch2_field				) ,
			.width_o			( ch2_width				) ,
			.height_o			( ch2_height			) ,
			.frame_start_o		( ch2_frame_start		) ,
			.next_frame_o		( ch2_next_frame		)
		) ;


	// ------------------------ //
	//  Channel 3 Flow control  //
	// ------------------------ //

	assign	ch3_de	= ( (   ready_i == 1'b0  ) & 1'b0 ) |
					  ( ( layer0_ch == 2'b11 ) & layer0_de_ff ) |
					  ( ( layer1_ch == 2'b11 ) & layer1_de_ff ) |
					  ( ( layer2_ch == 2'b11 ) & layer2_de_ff ) |
					  ( ( layer3_ch == 2'b11 ) & layer3_de_ff ) ;
	assign	ch3_rd	= count_en & ~ch3_next_frame & ch3_de ;

	pc11_in_buf
		# (
			.SIZE_BIT			( SIZE_BIT				) ,
			.PIXEL_BIT			( PIXEL_BIT				)
		)
		u_ch3_in_buf (
			.clk				( clk					) ,
			.rst_n				( rst_n					) ,
			.srst				( srst					) ,
			.enable				( enable				) ,

			.count_en_i			( count_en_ff			) ,
			.end_flag_i			( end_flag_adj			) ,
			.ready_i			( ready_i				) ,
			.end_flag_set_i		( end_flag_set			) , //v1.5

			.base_param_valid_i	( base_param_valid_ff	) , //v1.2
			.base_field_i		( base_field_ff			) , //v1.2

			.ch_on_i			( ch3_on				) , //v1.3

			.valid_i			( ch3_valid_i			) ,
			.frame_start_i		( ch3_frame_start_i		) ,
			.frame_end_i		( ch3_frame_end_i		) ,
			.pixel_i			( ch3_pixel_i			) ,
			.field_i			( ch3_field_i			) ,
			.width_i			( ch3_width_i			) ,
			.height_i			( ch3_height_i			) ,
			.ready_o			( ch3_ready_o			) ,

			.rd_i				( ch3_rd				) ,
			.valid_o			( ch3_valid				) ,
			.pixel_o			( ch3_pixel				) ,
			.field_o			( ch3_field				) ,
			.width_o			( ch3_width				) ,
			.height_o			( ch3_height			) ,
			.frame_start_o		( ch3_frame_start		) ,
			.next_frame_o		( ch3_next_frame		)
		) ;


// =============== //
//  Output Timing  //
// =============== //

	// -------------------------- //
	//  background pixel counter  //
	// -------------------------- //

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			h_count_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( srst ) begin
			h_count_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( enable ) begin
			if ( count_en ) begin
				if ( h_max ) begin
					h_count_ff	<= { SIZE_BIT{1'b0} } ;
				end
				else begin
					h_count_ff	<= h_count_ff + 1'b1 ;
				end
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			v_count_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( srst ) begin
			v_count_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( enable ) begin
			if ( count_en && h_max ) begin
				if ( v_max ) begin
					v_count_ff	<= { SIZE_BIT{1'b0} } ;
				end
				else begin
					v_count_ff	<= v_count_ff + 1'b1 ;
				end
			end
		end
	end

	assign	h_max	= ( h_count_ff == width ) ?	1'b1	: 1'b0 ;
	assign	v_max	= ( v_count_ff == height ) ?	1'b1	: 1'b0 ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			count_en_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			count_en_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				if ( end_flag && count_en ) begin
					count_en_ff	<= 1'b0 ;
				end
				else if ( start_flag_ff ) begin
					count_en_ff	<= 1'b1 ;
				end
			end
		end
	end

	assign	count_en	= data_valid & count_en_ff & ready_i & init_end ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			output_sel_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			output_sel_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ready_i && data_valid ) begin
				output_sel_ff	<= h_count_ff[0] ;
			end
		end
	end


	// --------------------- //
	//  Layer Output Timing  //
	// --------------------- //
	// Layer 0
	assign	layer0_hde	= ( ( layer0_hstart <= h_count_ff ) & ( h_count_ff <= layer0_hend ) ) ?	1'b1	: 1'b0 ;
	assign	layer0_vde	= ( ( layer0_vstart <= v_count_ff ) & ( v_count_ff <= layer0_vend ) ) ?	1'b1	: 1'b0 ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer0_de_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			layer0_de_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
// v1.1
			if ( ready_i && end_flag_adj ) begin
					layer0_de_ff	<= 1'b0 ;
			end
			else if ( ready_i && end_flag_lat_adj_ff && layer0_de_ff ) begin
					layer0_de_ff	<= 1'b1 ;
			end
//			if ( count_en ) begin
			else if ( count_en ) begin
				if ( layer0_hde && layer0_vde && ( prog_frame | ( base_field_ff == layer0_field ) ) ) begin
					layer0_de_ff	<= 1'b1 ;
				end
				else begin
					layer0_de_ff	<= 1'b0 ;
				end
			end
			else if ( !count_en_ff && ready_i ) begin
				layer0_de_ff	<= 1'b0 ;
			end
		end
	end

	assign	layer0_valid		= ( layer0_ch == 2'b00 & ch0_valid ) |
								  ( layer0_ch == 2'b01 & ch1_valid ) |
								  ( layer0_ch == 2'b10 & ch2_valid ) |
								  ( layer0_ch == 2'b11 & ch3_valid ) ;

	assign	layer0_pixel		= ( { PIXEL_BIT{layer0_ch == 2'b00} } & ch0_pixel ) |
								  ( { PIXEL_BIT{layer0_ch == 2'b01} } & ch1_pixel ) |
								  ( { PIXEL_BIT{layer0_ch == 2'b10} } & ch2_pixel ) |
								  ( { PIXEL_BIT{layer0_ch == 2'b11} } & ch3_pixel ) ;

	assign	layer0_field		= ( layer0_ch == 2'b00 & ch0_field ) |
								  ( layer0_ch == 2'b01 & ch1_field ) |
								  ( layer0_ch == 2'b10 & ch2_field ) |
								  ( layer0_ch == 2'b11 & ch3_field ) ;

	assign	layer3_de	= 	( layer3_con ) ? ( layer3_de_ff & ( layer3_pixel != layer3_cdata ) ) : layer3_de_ff ;  // custom

	assign	layer0_0123			=  ( layer0_de_ff & layer0_on ) &  ( layer1_de_ff & layer1_on ) &
								   ( layer2_de_ff & layer2_on ) &  ( layer3_de    & layer3_on ) ; // custom
	assign	layer0_023			=  ( layer0_de_ff & layer0_on ) & ~( layer1_de_ff & layer1_on ) &
								   ( layer2_de_ff & layer2_on ) &  ( layer3_de    & layer3_on ) ; // custom
	assign	layer0_013			=  ( layer0_de_ff & layer0_on ) &  ( layer1_de_ff & layer1_on ) &
								  ~( layer2_de_ff & layer2_on ) &  ( layer3_de    & layer3_on ) ; // custom
	assign	layer0_012			=  ( layer0_de_ff & layer0_on ) &  ( layer1_de_ff & layer1_on ) &
								   ( layer2_de_ff & layer2_on ) & ~( layer3_de    & layer3_on ) ; // custom
	assign	layer0_03			=  ( layer0_de_ff & layer0_on ) & ~( layer1_de_ff & layer1_on ) &
								  ~( layer2_de_ff & layer2_on ) &  ( layer3_de    & layer3_on ) ; // custom
	assign	layer0_02			=  ( layer0_de_ff & layer0_on ) & ~( layer1_de_ff & layer1_on ) &
								   ( layer2_de_ff & layer2_on ) & ~( layer3_de    & layer3_on ) ; // custom
	assign	layer0_01			=  ( layer0_de_ff & layer0_on ) &  ( layer1_de_ff & layer1_on ) &
								  ~( layer2_de_ff & layer2_on ) & ~( layer3_de    & layer3_on ) ; // custom
	assign	layer0_0			=  ( layer0_de_ff & layer0_on ) & ~( layer1_de_ff & layer1_on ) &
								  ~( layer2_de_ff & layer2_on ) & ~( layer3_de    & layer3_on ) ; // custom
	assign	layer0_alpha		= ( { (ALPHA_BIT+1){layer0_0123} } & layer0_alpha0123 ) |
								  ( { (ALPHA_BIT+1){layer0_023 } } & layer0_alpha023  ) |
								  ( { (ALPHA_BIT+1){layer0_013 } } & layer0_alpha013  ) |
								  ( { (ALPHA_BIT+1){layer0_03  } } & layer0_alpha03   ) |
								  ( { (ALPHA_BIT+1){layer0_012 } } & layer0_alpha012  ) |
								  ( { (ALPHA_BIT+1){layer0_02  } } & layer0_alpha02   ) |
								  ( { (ALPHA_BIT+1){layer0_01  } } & layer0_alpha01   ) |
								  ( { (ALPHA_BIT+1){layer0_0   } } & layer0_alpha0    ) ;

	assign	layer0_data_valid	= ( layer0_de_ff & layer0_on & layer0_valid ) |
								   ( ~layer0_de_ff & layer0_valid ) |
								   ( ~layer0_on ) ;


	//	Layer 1
	assign	layer1_hde	= ( ( layer1_hstart <= h_count_ff ) && ( h_count_ff <= layer1_hend ) ) ? 1'b1 : 1'b0 ;
	assign	layer1_vde	= ( ( layer1_vstart <= v_count_ff ) && ( v_count_ff <= layer1_vend ) ) ? 1'b1 : 1'b0 ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer1_de_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			layer1_de_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
// v1.1
			if ( ready_i && end_flag_adj ) begin
					layer1_de_ff	<= 1'b0 ;
			end
			else if ( ready_i && end_flag_lat_adj_ff && layer1_de_ff ) begin
					layer1_de_ff	<= 1'b1 ;
			end
//			if ( count_en ) begin
			else if ( count_en ) begin
				if ( layer1_hde && layer1_vde && ( prog_frame | ( base_field_ff == layer1_field ) ) ) begin
					layer1_de_ff	<= 1'b1 ;
				end
				else begin
					layer1_de_ff	<= 1'b0 ;
				end
			end
			else if ( !count_en_ff && ready_i ) begin
				layer1_de_ff	<= 1'b0 ;
			end
		end
	end

	assign	layer1_valid	= ( layer1_ch == 2'b00 & ch0_valid ) |
							  ( layer1_ch == 2'b01 & ch1_valid ) |
							  ( layer1_ch == 2'b10 & ch2_valid ) |
							  ( layer1_ch == 2'b11 & ch3_valid ) ;

	assign	layer1_pixel	= ( { PIXEL_BIT{layer1_ch == 2'b00} } & ch0_pixel ) |
							  ( { PIXEL_BIT{layer1_ch == 2'b01} } & ch1_pixel ) |
							  ( { PIXEL_BIT{layer1_ch == 2'b10} } & ch2_pixel ) |
							  ( { PIXEL_BIT{layer1_ch == 2'b11} } & ch3_pixel ) ;

	assign	layer1_field	= ( layer1_ch == 2'b00 & ch0_field ) |
							  ( layer1_ch == 2'b01 & ch1_field ) |
							  ( layer1_ch == 2'b10 & ch2_field ) |
							  ( layer1_ch == 2'b11 & ch3_field ) ;

	assign	layer1_123			=  ( layer1_de_ff & layer1_on ) &  ( layer2_de_ff & layer2_on ) &
								   ( layer3_de    & layer3_on ) ; // custom
	assign	layer1_13			=  ( layer1_de_ff & layer1_on ) & ~( layer2_de_ff & layer2_on ) &
								   ( layer3_de    & layer3_on ) ; // custom
	assign	layer1_12			=  ( layer1_de_ff & layer1_on ) &  ( layer2_de_ff & layer2_on ) &
								  ~( layer3_de    & layer3_on ) ; // custom
	assign	layer1_1			=  ( layer1_de_ff & layer1_on ) & ~( layer2_de_ff & layer2_on ) &
								  ~( layer3_de    & layer3_on ) ; // custom
	assign	layer1_alpha		= ( { (ALPHA_BIT+1){layer1_123} } & layer1_alpha123 ) |
								  ( { (ALPHA_BIT+1){layer1_13 } } & layer1_alpha13  ) |
								  ( { (ALPHA_BIT+1){layer1_12 } } & layer1_alpha12  ) |
								  ( { (ALPHA_BIT+1){layer1_1  } } & layer1_alpha1   ) ;

	assign	layer1_data_valid	= ( layer1_de_ff & layer1_on & layer1_valid ) |
								  ( ~layer1_de_ff & layer1_valid ) |
								  ( ~layer1_on ) ;


	//	Layer 2
	assign	layer2_hde	= ( ( layer2_hstart <= h_count_ff ) && ( h_count_ff <= layer2_hend ) ) ? 1'b1 : 1'b0 ;
	assign	layer2_vde	= ( ( layer2_vstart <= v_count_ff ) && ( v_count_ff <= layer2_vend ) ) ? 1'b1 : 1'b0 ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer2_de_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			layer2_de_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
// v1.1
			if ( ready_i && end_flag_adj ) begin
					layer2_de_ff	<= 1'b0 ;
			end
			else if ( ready_i && end_flag_lat_adj_ff && layer2_de_ff ) begin
					layer2_de_ff	<= 1'b1 ;
			end
//			if ( count_en ) begin
			else if ( count_en ) begin
				if ( layer2_hde && layer2_vde && ( prog_frame | ( base_field_ff == layer2_field ) ) ) begin
					layer2_de_ff	<= 1'b1 ;
				end
				else begin
					layer2_de_ff	<= 1'b0 ;
				end
			end
			else if ( !count_en_ff && ready_i ) begin
				layer2_de_ff	<= 1'b0 ;
			end
		end
	end

	assign	layer2_valid		= ( layer2_ch == 2'b00 & ch0_valid ) |
								  ( layer2_ch == 2'b01 & ch1_valid ) |
								  ( layer2_ch == 2'b10 & ch2_valid ) |
								  ( layer2_ch == 2'b11 & ch3_valid ) ;

	assign	layer2_pixel		= ( { PIXEL_BIT{layer2_ch == 2'b00} } & ch0_pixel ) |
								  ( { PIXEL_BIT{layer2_ch == 2'b01} } & ch1_pixel ) |
								  ( { PIXEL_BIT{layer2_ch == 2'b10} } & ch2_pixel ) |
								  ( { PIXEL_BIT{layer2_ch == 2'b11} } & ch3_pixel ) ;

	assign	layer2_field		= ( layer2_ch == 2'b00 & ch0_field ) |
								  ( layer2_ch == 2'b01 & ch1_field ) |
								  ( layer2_ch == 2'b10 & ch2_field ) |
								  ( layer2_ch == 2'b11 & ch3_field ) ;

	assign	layer2_23			= ( layer2_de_ff & layer2_on ) &  ( layer3_de & layer3_on ) ; // custom
	assign	layer2_2			= ( layer2_de_ff & layer2_on ) & ~( layer3_de & layer3_on ) ; // custom
	assign	layer2_alpha		= ( { (ALPHA_BIT+1){layer2_23 } } & layer2_alpha23 ) |
								  ( { (ALPHA_BIT+1){layer2_2  } } & layer2_alpha2  ) ;

	assign	layer2_data_valid	= ( layer2_de_ff & layer2_on & layer2_valid ) |
								  ( ~layer2_de_ff & layer2_valid ) |
								  ( ~layer2_on ) ;

	//	Layer 3
	assign	layer3_hde	= ( ( layer3_hstart <= h_count_ff ) && ( h_count_ff <= layer3_hend ) ) ? 1'b1 : 1'b0 ;
	assign	layer3_vde	= ( ( layer3_vstart <= v_count_ff ) && ( v_count_ff <= layer3_vend ) ) ? 1'b1 : 1'b0 ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer3_de_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			layer3_de_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
// v1.1
			if ( ready_i && end_flag_adj ) begin
					layer3_de_ff	<= 1'b0 ;
			end
			else if ( ready_i && end_flag_lat_adj_ff && layer3_de_ff ) begin
					layer3_de_ff	<= 1'b1 ;
			end
//			if ( count_en ) begin
			else if ( count_en ) begin
				if ( layer3_hde && layer3_vde && ( prog_frame | ( base_field_ff == layer3_field ) ) ) begin
					layer3_de_ff	<= 1'b1 ;
				end
				else begin
					layer3_de_ff	<= 1'b0 ;
				end
			end
			else if ( !count_en_ff && ready_i ) begin
				layer3_de_ff	<= 1'b0 ;
			end
		end
	end

	assign	layer3_valid		= ( layer3_ch == 2'b00 & ch0_valid ) |
								  ( layer3_ch == 2'b01 & ch1_valid ) |
								  ( layer3_ch == 2'b10 & ch2_valid ) |
								  ( layer3_ch == 2'b11 & ch3_valid ) ;

	assign	layer3_pixel		= ( {PIXEL_BIT{layer3_ch == 2'b00}} & ch0_pixel ) |
								  ( {PIXEL_BIT{layer3_ch == 2'b01}} & ch1_pixel ) |
								  ( {PIXEL_BIT{layer3_ch == 2'b10}} & ch2_pixel ) |
								  ( {PIXEL_BIT{layer3_ch == 2'b11}} & ch3_pixel ) ;

	assign	layer3_field		= ( layer3_ch == 2'b00 & ch0_field ) |
								  ( layer3_ch == 2'b01 & ch1_field ) |
								  ( layer3_ch == 2'b10 & ch2_field ) |
								  ( layer3_ch == 2'b11 & ch3_field ) ;

	assign	layer3_3			= ( layer3_de & layer3_on ) ; // custom
	assign	layer3_alpha		= ( { (ALPHA_BIT+1){layer3_3  } } & layer3_alpha3 ) ;

	assign	layer3_data_valid	= ( layer3_de & layer3_on & layer3_valid ) | // custom
								  ( ~layer3_de & layer3_valid ) | // custom
								  ( ~layer3_on ) ;

	assign	data_valid			= layer0_data_valid & layer1_data_valid & layer2_data_valid & layer3_data_valid ;


	// ------------- //
	//  combination  //
	// ------------- //

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			const_data_lat1_ff		<= { PIXEL_BIT{1'b0} } ;
			layer0_pixel_lat1_ff	<= { PIXEL_BIT{1'b0} } ;
			layer1_pixel_lat1_ff	<= { PIXEL_BIT{1'b0} } ;
			layer2_pixel_lat1_ff	<= { PIXEL_BIT{1'b0} } ;
			layer3_pixel_lat1_ff	<= { PIXEL_BIT{1'b0} } ;
			layer0_alpha_lat1_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer1_alpha_lat1_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer2_alpha_lat1_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer3_alpha_lat1_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			const_data_lat1_ff 		<= { PIXEL_BIT{1'b0} } ;
			layer0_pixel_lat1_ff	<= { PIXEL_BIT{1'b0} } ;
			layer1_pixel_lat1_ff	<= { PIXEL_BIT{1'b0} } ;
			layer2_pixel_lat1_ff	<= { PIXEL_BIT{1'b0} } ;
			layer3_pixel_lat1_ff	<= { PIXEL_BIT{1'b0} } ;
			layer0_alpha_lat1_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer1_alpha_lat1_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer2_alpha_lat1_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer3_alpha_lat1_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				const_data_lat1_ff		<= constant_data ;
				layer0_pixel_lat1_ff	<= layer0_pixel ;
				layer1_pixel_lat1_ff	<= layer1_pixel ;
				layer2_pixel_lat1_ff	<= layer2_pixel ;
				layer3_pixel_lat1_ff	<= layer3_pixel ;
				layer0_alpha_lat1_ff	<= layer0_alpha ;
				layer1_alpha_lat1_ff	<= layer1_alpha ;
				layer2_alpha_lat1_ff	<= layer2_alpha ;
				layer3_alpha_lat1_ff	<= layer3_alpha ;
			end
		end
	end

	pc11_unsigned_multadd4
		# (
			.APORT_BIT	( Q_BIT 																) ,
			.BPORT_BIT	( ALPHA_BIT+1															)
		)
		u_ry_multadd4 (
			.clk		( clk																	) ,
			.rst_n		( rst_n																	) ,
			.enable		( enable & ready_i														) ,
			.dataa_0_i	( layer0_pixel_lat1_ff[Q_BIT*3-1:Q_BIT*2]								) ,
			.dataa_1_i	( layer1_pixel_lat1_ff[Q_BIT*3-1:Q_BIT*2]								) ,
			.dataa_2_i	( layer2_pixel_lat1_ff[Q_BIT*3-1:Q_BIT*2]								) ,
			.dataa_3_i	( layer3_pixel_lat1_ff[Q_BIT*3-1:Q_BIT*2]								) ,
			.datab_0_i	( layer0_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.datab_1_i	( layer1_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.datab_2_i	( layer2_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.datab_3_i	( layer3_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.result_o	( { ry_dummy_bit , ry_multadd_pixel }	)
		) ;

	pc11_unsigned_multadd4
		# (
			.APORT_BIT	( Q_BIT																	) ,
			.BPORT_BIT	( ALPHA_BIT+1															)
		)
		u_gu_multadd4 (
			.clk		( clk																	) ,
			.rst_n		( rst_n																	) ,
			.enable		( enable & ready_i														) ,
			.dataa_0_i	( layer0_pixel_lat1_ff[Q_BIT*2-1:Q_BIT]									) ,
			.dataa_1_i	( layer1_pixel_lat1_ff[Q_BIT*2-1:Q_BIT]									) ,
			.dataa_2_i	( layer2_pixel_lat1_ff[Q_BIT*2-1:Q_BIT]									) ,
			.dataa_3_i	( layer3_pixel_lat1_ff[Q_BIT*2-1:Q_BIT]									) ,
			.datab_0_i	( layer0_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.datab_1_i	( layer1_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.datab_2_i	( layer2_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.datab_3_i	( layer3_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.result_o	( { gu_dummy_bit , gu_multadd_pixel }	)
		) ;

	pc11_unsigned_multadd4
		# (
			.APORT_BIT	( Q_BIT																	) ,
			.BPORT_BIT	( ALPHA_BIT+1															)
		)
		u_bv_multadd4 (
			.clk		( clk																	) ,
			.rst_n		( rst_n																	) ,
			.enable		( enable & ready_i														) ,
			.dataa_0_i	( layer0_pixel_lat1_ff[Q_BIT-1:0]										) ,
			.dataa_1_i	( layer1_pixel_lat1_ff[Q_BIT-1:0]										) ,
			.dataa_2_i	( layer2_pixel_lat1_ff[Q_BIT-1:0]										) ,
			.dataa_3_i	( layer3_pixel_lat1_ff[Q_BIT-1:0]										) ,
			.datab_0_i	( layer0_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.datab_1_i	( layer1_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.datab_2_i	( layer2_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.datab_3_i	( layer3_alpha_lat1_ff[ALPHA_BIT:0]										) ,
			.result_o	( { bv_dummy_bit , bv_multadd_pixel }			)
		) ;

	//	Background Alpha
	assign	bg_alpha	= { 1'b1 , {ALPHA_BIT{1'b0}} } - layer0_alpha_lat1_ff - layer1_alpha_lat1_ff - layer2_alpha_lat1_ff - layer3_alpha_lat1_ff ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			const_data_lat2_ff	<= { PIXEL_BIT{1'b0} } ;
			bg_alpha_ff			<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			const_data_lat2_ff	<= { PIXEL_BIT{1'b0} } ;
			bg_alpha_ff			<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				const_data_lat2_ff	<= const_data_lat1_ff ;
				bg_alpha_ff			<= bg_alpha ;
			end
		end
	end

	pc11_unsigned_mult
		# (
			.APORT_BIT	( Q_BIT																			) ,
			.BPORT_BIT	( ALPHA_BIT + 1																	)
		)
		u_const_ry_mult (
			.clk		( clk																			) ,
			.rst_n		( rst_n																			) ,
			.enable		( enable & ready_i																) ,
			.dataa_i	( const_data_lat2_ff[Q_BIT*3-1:Q_BIT*2]											) ,
			.datab_i	( bg_alpha_ff																	) ,
			.result_o	( { const_ry_dummy_bit , const_ry_mult }	)
		) ;

	pc11_unsigned_mult
		# (
			.APORT_BIT	( Q_BIT																			) ,
			.BPORT_BIT	( ALPHA_BIT + 1																	)
		)
		u_const_gu_mult (
			.clk		( clk																			) ,
			.rst_n		( rst_n																			) ,
			.enable		( enable & ready_i																) ,
			.dataa_i	( const_data_lat2_ff[Q_BIT*2-1:Q_BIT]											) ,
			.datab_i	( bg_alpha_ff																	) ,
			.result_o	( { const_gu_dummy_bit , const_gu_mult }		)
		) ;

	pc11_unsigned_mult
		# (
			.APORT_BIT	( Q_BIT																			) ,
			.BPORT_BIT	( ALPHA_BIT + 1																	)
		)
		u_const_bv_mult (
			.clk		( clk																			) ,
			.rst_n		( rst_n																			) ,
			.enable		( enable & ready_i																) ,
			.dataa_i	( const_data_lat2_ff[Q_BIT-1:0]													) ,
			.datab_i	( bg_alpha_ff																	) ,
			.result_o	( { const_bv_dummy_bit , const_bv_mult }			)
		) ;


	// --------------- //
	//  Flag Generate  //
	// --------------- //

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			base_field_ff	<= 1'b1 ;
		end
		else if ( srst ) begin
			base_field_ff	<= 1'b1 ;
		end
		else if ( enable ) begin
			if ( base_input_ris_ff ) begin
				if ( ch0_on ) begin
					base_field_ff	<= ch0_field ;
				end
				else if ( ch1_on ) begin
					base_field_ff	<= ch1_field ;
				end
				else if ( ch2_on ) begin
					base_field_ff	<= ch2_field ;
				end
				else if ( ch3_on ) begin
					base_field_ff	<= ch3_field ;
				end
			end
		end
	end

	//v1.2
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			base_param_valid_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			base_param_valid_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( base_input_ris_ff ) begin
				base_param_valid_ff	<=1'b1 ;
			end
			else if ( end_flag_adj ) begin
				base_param_valid_ff	<= 1'b0 ;
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			last_field_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			last_field_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( base_input_ris_ff ) begin
				last_field_ff	<= base_field_ff ;
			end
		end
	end

	assign	prog_frame	= ~last_field_ff & ~base_field_ff ;

	//	start flag
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			base_input_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			base_input_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( init_end ) begin
				if ( end_flag_adj ) begin
					base_input_ff	<= 1'b0 ;
				end
				else if ( ch0_on ) begin
					base_input_ff	<= ch0_frame_start ;
				end
				else if ( ch1_on ) begin
					base_input_ff	<= ch1_frame_start ;
				end
				else if ( ch2_on ) begin
					base_input_ff	<= ch2_frame_start ;
				end
				else if ( ch3_on ) begin
					base_input_ff	<= ch3_frame_start ;
				end
				else begin
					base_input_ff	<= 1'b1 ;
				end
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			all_off_ff		<= 1'b0 ;
		end
		else if ( srst ) begin
			all_off_ff		<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( init_end ) begin
				if ( end_flag & data_valid ) begin
					all_off_ff		<= 1'b0 ;
				end
				else if ( ( base_input_ris_ff && !count_en_ff ) && ( !ch0_on && !ch1_on && !ch2_on && !ch3_on ) ) begin
					all_off_ff		<= 1'b1 ;
				end
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			base_input_lat_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			base_input_lat_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			base_input_lat_ff	<= base_input_ff ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			base_input_ris_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			base_input_ris_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			base_input_ris_ff	<= base_input_ff & ~base_input_lat_ff ;
		end
	end


	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			start_flag_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			start_flag_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( base_input_ris_ff && ~count_en_ff ) begin
				start_flag_ff	<= 1'b1 ;
			end
			else if ( ready_i ) begin
				start_flag_ff	<= 1'b0 ;
			end
		end
	end

	assign	end_flag	= init_end & h_max & v_max ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			width_value_ff	<= { SIZE_BIT{1'b0} } ;
			height_value_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( srst ) begin
			width_value_ff	<= { SIZE_BIT{1'b0} } ;
			height_value_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( enable ) begin
			if ( base_input_ris_ff ) begin
				width_value_ff	<= width_lat ;
				height_value_ff	<= height_lat ;
			end
		end
	end


	//	PipeLine delay
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			count_en_lat_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			count_en_lat_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ready_i && data_valid ) begin
				count_en_lat_ff	<= count_en_ff ;
			end
		end
	end

////////////////////////////////////////////////////////////////////////////////
// v1.1
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			end_flag_lat_adj_ff	<= 1'b0	;
		end
		else if ( srst ) begin
			end_flag_lat_adj_ff	<= 1'b0	;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				if ( end_flag  && data_valid ) begin
					end_flag_lat_adj_ff	<= 1'b1	;
				end
				else if ( end_flag_adj ) begin
					end_flag_lat_adj_ff	<= 1'b0	;
				end
			end
		end
	end

	assign end_flag_adj	= end_flag_lat_adj_ff && data_valid ;
	assign end_flag_set	= ready_i && end_flag && data_valid	; //v1.5

////////////////////////////////////////////////////////////////////////////////

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			start_flag_lat_ff 	<= 1'b0 ;
			start_flag_lat1_ff	<= 1'b0 ;
			start_flag_lat2_ff	<= 1'b0 ;
			start_flag_lat3_ff	<= 1'b0 ;
			start_flag_lat4_ff	<= 1'b0 ;
//			end_flag_lat_ff 	<= 1'b0 ; // v1.1
			end_flag_lat1_ff	<= 1'b0 ;
			end_flag_lat2_ff	<= 1'b0 ;
			end_flag_lat3_ff	<= 1'b0 ;
			end_flag_lat4_ff	<= 1'b0 ;
			data_valid_lat1_ff	<= 1'b0 ;
			data_valid_lat2_ff	<= 1'b0 ;
			data_valid_lat3_ff	<= 1'b0 ;
			data_valid_lat4_ff	<= 1'b0 ;
			output_sel_lat1_ff	<= 1'b0 ;
			output_sel_lat2_ff	<= 1'b0 ;
			output_sel_lat3_ff	<= 1'b0 ;
			output_sel_lat4_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			start_flag_lat_ff 	<= 1'b0 ;
			start_flag_lat1_ff	<= 1'b0 ;
			start_flag_lat2_ff	<= 1'b0 ;
			start_flag_lat3_ff	<= 1'b0 ;
			start_flag_lat4_ff	<= 1'b0 ;
//			end_flag_lat_ff		<= 1'b0 ; // v1.1
			end_flag_lat1_ff	<= 1'b0 ;
			end_flag_lat2_ff	<= 1'b0 ;
			end_flag_lat3_ff	<= 1'b0 ;
			end_flag_lat4_ff	<= 1'b0 ;
			data_valid_lat1_ff	<= 1'b0 ;
			data_valid_lat2_ff	<= 1'b0 ;
			data_valid_lat3_ff	<= 1'b0 ;
			data_valid_lat4_ff	<= 1'b0 ;
			output_sel_lat1_ff	<= 1'b0 ;
			output_sel_lat2_ff	<= 1'b0 ;
			output_sel_lat3_ff	<= 1'b0 ;
			output_sel_lat4_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				start_flag_lat_ff 	<= start_flag_ff ;
				start_flag_lat1_ff	<= start_flag_lat_ff ;
				start_flag_lat2_ff	<= start_flag_lat1_ff ;
				start_flag_lat3_ff	<= start_flag_lat2_ff ;
				start_flag_lat4_ff	<= start_flag_lat3_ff ;
//				end_flag_lat_ff		<= end_flag & data_valid ; // v1.1
				end_flag_lat1_ff	<= end_flag_adj ;
				end_flag_lat2_ff	<= end_flag_lat1_ff ;
				end_flag_lat3_ff	<= end_flag_lat2_ff ;
				end_flag_lat4_ff	<= end_flag_lat3_ff ;
				data_valid_lat1_ff	<= ( data_valid & count_en_lat_ff ) | end_flag_adj ;
				data_valid_lat2_ff	<= data_valid_lat1_ff ;
				data_valid_lat3_ff	<= data_valid_lat2_ff ;
				data_valid_lat4_ff	<= data_valid_lat3_ff ;
				output_sel_lat1_ff	<= output_sel_ff ;
				output_sel_lat2_ff	<= output_sel_lat1_ff ;
				output_sel_lat3_ff	<= output_sel_lat2_ff ;
				output_sel_lat4_ff	<= output_sel_lat3_ff ;
			end
		end
	end

	// -------------- //
	//  frame output  //
	// -------------- //
	// valid
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			valid_o_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			valid_o_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				valid_o_ff	<= data_valid_lat4_ff | start_flag_lat4_ff ;
			end
		end
	end

	// frame start
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frame_start_o_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			frame_start_o_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				frame_start_o_ff	<= start_flag_lat4_ff ;
			end
		end
	end

	// frame end
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frame_end_o_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			frame_end_o_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				frame_end_o_ff	<= end_flag_lat4_ff ;
			end
		end
	end

	// pixel
	assign	ry_pixel_add_const		= const_ry_mult + ry_multadd_pixel ;
	assign	gu_pixel_add_const		= const_gu_mult + gu_multadd_pixel ;
	assign	gubv_pixel_add_const	= const_bv_mult + gu_multadd_pixel ;
	assign	bv_pixel_add_const		= const_bv_mult + bv_multadd_pixel ;
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			pixel_o_ff	<= { PIXEL_BIT{1'b0} } ;
		end
		else if ( srst ) begin
			pixel_o_ff	<= { PIXEL_BIT{1'b0} } ;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				if ( start_flag_lat4_ff ) begin
					pixel_o_ff	<= { PIXEL_BIT{1'b0} } ;
				end
				else if ( !smpl_mode ) begin
					if ( output_sel_lat4_ff ) begin
						pixel_o_ff[Q_BIT*3-1:Q_BIT*2]	<= ry_pixel_add_const[MLTPIX_BIT-1:ALPHA_BIT] ;
						pixel_o_ff[Q_BIT*2-1:Q_BIT  ]	<= gubv_pixel_add_const[MLTPIX_BIT-1:ALPHA_BIT] ;
						pixel_o_ff[Q_BIT  -1:      0]	<= bv_pixel_add_const[MLTPIX_BIT-1:ALPHA_BIT] ;
					end
					else begin
						pixel_o_ff[Q_BIT*3-1:Q_BIT*2]	<= ry_pixel_add_const[MLTPIX_BIT-1:ALPHA_BIT] ;
						pixel_o_ff[Q_BIT*2-1:Q_BIT  ]	<= gu_pixel_add_const[MLTPIX_BIT-1:ALPHA_BIT] ;
						pixel_o_ff[Q_BIT  -1:      0]	<= bv_pixel_add_const[MLTPIX_BIT-1:ALPHA_BIT] ;
					end
				end
				else begin
					pixel_o_ff[Q_BIT*3-1:Q_BIT*2]	<= ry_pixel_add_const[MLTPIX_BIT-1:ALPHA_BIT] ;
					pixel_o_ff[Q_BIT*2-1:Q_BIT  ]	<= gu_pixel_add_const[MLTPIX_BIT-1:ALPHA_BIT] ;
					pixel_o_ff[Q_BIT  -1:      0]	<= bv_pixel_add_const[MLTPIX_BIT-1:ALPHA_BIT] ;
				end
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			field_o_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			field_o_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				if ( start_flag_lat4_ff ) begin
					field_o_ff	<= base_field_ff ;
				end
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			width_o_ff	<= { SIZE_BIT{1'b0} } ;
			height_o_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( srst ) begin
			width_o_ff	<= { SIZE_BIT{1'b0} } ;
			height_o_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( enable ) begin
			if ( ready_i ) begin
				if ( start_flag_lat4_ff ) begin
					width_o_ff	<= width_value_ff ;
					height_o_ff	<= height_value_ff ;
				end
			end
		end
	end


	assign	valid_o			= valid_o_ff ;
	assign	frame_start_o	= frame_start_o_ff ;
	assign	frame_end_o		= frame_end_o_ff ;
	assign	pixel_o			= pixel_o_ff ;
	assign	field_o			= field_o_ff ;
	assign	width_o			= width_o_ff ;
	assign	height_o		= height_o_ff ;


endmodule

`default_nettype wire
