//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		VSI to Video parallel I/F Decoder.
//----------------------------------------------------------------------------------------
// REVISION HISTORY	:
//		v1.0 Jan. 12 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT		: Bit Width of Vertical Counter.
//		PLANE		: Color Plane.
//		FIFO_DEPTH	: FIFO Depth.
//		VSI_FMIN	: Minimum frequency of VSI Clock.
//		VIDEO_FMAX	: Maximum frequency of Video I/F Clock.
//
//		DEVICE		: set FPGA device family. ( default:0 )
//----------------------------------------------------------------------------------------
// I/O PORTS
//		vsi_clk 			: VSI Clock.
//		out_clk				: Video parallel I/F clock.
//		rst_n				: Async reset.
//		vsi_srst			: Synchronous reset.
//		out_srst			: Synchronous reset.
//
//		enable				: Enable of circuit.
//		param_update_i		: Mode setting signal.
//
//		interlace_mode_i	: Scan mode select.
//							  1:Interlace 0:Progressive
//
//		ex_sync_pol_i		: External synchronization polarity.
//		ex_odd_field_pol_i	: Odd field polarity.
//		h_sync_pol_i		: H Sync polarity.(hsync_i)
//		v_sync_pol_i		: V Sync polarity.(vsync_i)
//		odd_field_pol_i 	: Odd field polarity.(field_i)
//		h_valid_pol_i		: H Valid polarity.(hvalid_i)
//		v_valid_pol_i		: V Valid polarity.(vvalid_i)
//		de_pol_i			: DE polarity.(de_i)
//
//		h_front_i			: H front porch.
//		h_interval_i		: H sync interval.
//		h_back_i			: H back porch.
//		h_video_i			: H active video.
//		v_front_i			: V front porch.
//		v_interval_i		: V sync interval.
//		v_back_i			: V back porch.
//		v_video_i			: V active video.
//		v_delay_i			: V sync delay.
//		even_v_front_i		: V front porch for even field.
//		even_v_interval_i	: V sync interval for even field.
//		even_v_back_i		: V back porch for even field.
//		even_v_video_i		: V active video for even field.
//		even_v_delay_i		: V sync delay for even field.
//		even2odd_v_delay_i	: Vertical change point of field startus signal.(even to odd field)
//		even2odd_h_delay_i	: Horizontal change point of field startus signal.(even to odd field)
//		odd2even_v_delay_i	: Vertical change point of field startus signal.(odd to even field)
//		odd2even_h_delay_i	: Horizontal change point of field startus signal.(odd to even field)
//
//		const_data_out_i	: Constant data output.
//		const_out_mode_i	: Mode select for constant data output.
//							  1:const_data0_i & const_data1_i 0:const_data0_i
//		const_data0_i		: Constant data 0.
//		const_data1_i		: Constant data 1.
//
//		external_sync_i 	: External synchronization signal.
//		external_field_i	: Field status of external synchronization signal.
//
//		frame_start_i		: Frame start of input frame data.
//		frame_end_i			: Frame end of input frame data.
//		valid_i				: Data valid of input frame data.
//		pixel_i				: Pixel data of input frame data.
//		field_i				: Field status of input frame data. 1:even field / 0:odd field
//		width_i				: Horizontal size of input frame data.
//		height_i			: Vertical size of input frame data.
//		ready_o				: Data reception ready of input frame data.
//
//		h_sync_o			: H Sync signal.
//		v_sync_o			: V Sync signal.
//		field_o 			: Field status signal.
//		de_o				: DE signal.
//		h_valid_o			: H Valid signal.
//		v_valid_o			: V Valid signal.
//		pixel_o				: Pixel data signal.
//----------------------------------------------------------------------------------------
`timescale 1ps / 1ps
`default_nettype none

module	vtg11_top
	#(
		parameter	Q_BIT				= 8		,
		parameter	PLANE				= 3		,
		parameter	FIFO_DEPTH			= 10	,
		parameter	VSI_FMIN			= 130	,
		parameter	VIDEO_FMAX			= 149	,
		parameter	DEVICE				= 0		,

		// Local Parameter
		parameter	H_F_P				= 9		,
		parameter	H_S_I				= 8		,
		parameter	H_B_P				= 9		,
		parameter	H_V_I				= 12	,
		parameter	V_F_P				= 6		,
		parameter	V_S_I				= 4		,
		parameter	V_B_P				= 7		,
		parameter	V_V_I				= 12	,
		parameter	H_COUNTER			= 12	,
		parameter	V_COUNTER			= 12	,
		parameter 	PIXEL_BIT			= Q_BIT * PLANE
	)
	(
		input	wire							vsi_clk 			,
		input	wire							out_clk 			,
		input	wire							rst_n				,
		input	wire							vsi_srst			,
		input	wire							out_srst			,

		input	wire							enable				,
		input	wire							param_update_i		,

		input	wire							interlace_mode_i	,

		input	wire							ex_sync_pol_i		,
		input	wire							ex_odd_field_pol_i	,
		input	wire							h_sync_pol_i		,
		input	wire							v_sync_pol_i		,
		input	wire							odd_field_pol_i 	,
		input	wire							h_valid_pol_i		,
		input	wire							v_valid_pol_i		,
		input	wire							de_pol_i			,

		input	wire	[	  H_F_P-1 : 0 ] 	h_front_i			,
		input	wire	[	  H_S_I-1 : 0 ] 	h_interval_i		,
		input	wire	[	  H_B_P-1 : 0 ] 	h_back_i			,
		input	wire	[	  H_V_I-1 : 0 ] 	h_video_i			,
		input	wire	[	  V_F_P-1 : 0 ] 	v_front_i			,
		input	wire	[	  V_S_I-1 : 0 ] 	v_interval_i		,
		input	wire	[	  V_B_P-1 : 0 ] 	v_back_i			,
		input	wire	[	  V_V_I-1 : 0 ] 	v_video_i			,
		input	wire	[ H_COUNTER-1 : 0 ] 	v_delay_i			,
		input	wire	[	  V_F_P-1 : 0 ] 	even_v_front_i		,
		input	wire	[	  V_S_I-1 : 0 ] 	even_v_interval_i	,
		input	wire	[	  V_B_P-1 : 0 ] 	even_v_back_i		,
		input	wire	[	  V_V_I-1 : 0 ] 	even_v_video_i		,
		input	wire	[ H_COUNTER-1 : 0 ] 	even_v_delay_i		,
		input	wire	[ V_COUNTER-1 : 0 ] 	even2odd_v_delay_i	,
		input	wire	[ H_COUNTER-1 : 0 ] 	even2odd_h_delay_i	,
		input	wire	[ V_COUNTER-1 : 0 ] 	odd2even_v_delay_i	,
		input	wire	[ H_COUNTER-1 : 0 ] 	odd2even_h_delay_i	,

		input	wire							const_data_out_i	,
		input	wire							const_out_mode_i	,
		input	wire	[ PIXEL_BIT-1 : 0 ]		const_data0_i		,
		input	wire	[ PIXEL_BIT-1 : 0 ] 	const_data1_i		,

		input	wire							external_sync_i		,
		input	wire							external_field_i	,

		input	wire							frame_start_i		,
		input	wire							frame_end_i			,
		input	wire							valid_i				,
		input	wire	[ PIXEL_BIT-1 : 0 ] 	pixel_i				,
		input	wire							field_i				,
		input	wire	[ H_COUNTER-1 : 0 ]		width_i				,
		input	wire	[ V_COUNTER-1 : 0 ]		height_i			,
		output	wire							ready_o				,

		output	wire							h_sync_o			,
		output	wire							v_sync_o			,
		output	wire							field_o 			,
		output	wire							de_o				,
		output	wire							h_valid_o			,
		output	wire							v_valid_o			,
		output	wire	[ PIXEL_BIT-1 : 0 ] 	pixel_o
	) ;

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Please do not change the following localparams
	// ---------------------------------------------------------------------
	localparam	H_FRONT 			= 16 ;
	localparam	H_INTERVAL			= 62 ;
	localparam	H_BACK				= 60 ;
	localparam	H_VIDEO 			= 720 ;
	localparam	V_FRONT 			= 9 ;
	localparam	V_INTERVAL			= 6 ;
	localparam	V_BACK				= 30 ;
	localparam	V_VIDEO 			= 480 ;
	localparam	V_DELAY 			= 16 ;
	localparam	EVEN_V_FRONT		= 9 ;
	localparam	EVEN_V_INTERVAL 	= 6 ;
	localparam	EVEN_V_BACK 		= 30 ;
	localparam	EVEN_V_VIDEO		= 480 ;
	localparam	EVEN_V_DELAY		= 16 ;
	localparam	EVEN2ODD_V_DELAY	= 2 ;
	localparam	EVEN2ODD_H_DELAY	= 2 ;
	localparam	ODD2EVEN_V_DELAY	= 2 ;
	localparam	ODD2EVEN_H_DELAY	= 2 ;
	localparam	RESET_TIMING		= 1 ;
	localparam	RESET_INTERVAL		= ( VIDEO_FMAX / VSI_FMIN ) + 2 ;

	localparam	FRST_WAIT			= 4'h0 ;
	localparam	START_WAIT			= 4'h1 ;
	localparam	NEXT_FRAME_WAIT		= 4'h2 ;
	localparam	FRM_PERIOD			= 4'h3 ;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg								param_update_lat_ff		;
	reg								param_update_lat1_ff	;
	reg								param_update_lat2_ff	;
	reg								param_ris_ff			;
	reg								frame_rst_lat_ff		;
	reg								frame_rst_lat1_ff		;
	reg								frame_rst_lat2_ff		;
	reg								frame_rst_fal_ff		;
	reg								req_field_lat_ff		;
	reg								act_video_lat_ff		;
	reg								act_video_lat1_ff		;
	reg								act_video_lat2_ff		;
	reg								act_video_fal_ff		;
	reg								act_video_ris_ff		;

	wire							frame_start				;
	wire							frame_end				;
	reg		[			 2 : 0 ]	vsi_state_ff			;
	reg								ready_int_ff			;

	reg								field_str_ff			;

	reg								frame_input_ff			;
	reg								frame_input_lat_ff		;
	reg								frame_input_lat1_ff		;
	reg								const_data_out_lat_ff	;
	reg								const_data_out_lat1_ff	;
	wire							const_data_out			;
	wire	[  V_COUNTER-1 : 0 ]	rst_timing				;

	reg								fifo_we_ff				;
	wire							fifo_re					;
	reg		[  PIXEL_BIT-1 : 0 ]	fifo_wdata_ff			;
	wire	[  PIXEL_BIT-1 : 0 ]	fifo_rdata				;
	wire							fifo_empty				;
	wire							fifo_afull				;

	wire							ram_wen					;
	wire							ram_ren					;
	wire	[ FIFO_DEPTH-1 : 0 ]	ram_wadr				;
	wire	[ FIFO_DEPTH-1 : 0 ]	ram_radr				;
	wire	[  PIXEL_BIT-1 : 0 ]	ram_wdata				;
	wire	[  PIXEL_BIT-1 : 0 ]	ram_rdata				;

	wire							frame_rst				;
	wire							req_field				;

	wire							act_video				;


// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	always @( posedge vsi_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			param_update_lat_ff		<= 1'b0 ;
			param_update_lat1_ff	<= 1'b0 ;
			param_update_lat2_ff	<= 1'b0 ;
			param_ris_ff			<= 1'b0 ;
		end
		else begin
			param_update_lat_ff		<= param_update_i ;
			param_update_lat1_ff	<= param_update_lat_ff ;
			param_update_lat2_ff	<= param_update_lat1_ff ;
			param_ris_ff			<= ~param_update_lat2_ff & param_update_lat1_ff ;
		end
	end

	always @( posedge vsi_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frame_rst_lat_ff	<= 1'b0 ;
			frame_rst_lat1_ff	<= 1'b0 ;
			frame_rst_lat2_ff	<= 1'b0 ;
			frame_rst_fal_ff	<= 1'b0 ;
			req_field_lat_ff	<= 1'b0 ;
			act_video_lat_ff	<= 1'b0 ;
			act_video_lat1_ff	<= 1'b0 ;
			act_video_lat2_ff	<= 1'b0 ;
			act_video_fal_ff	<= 1'b0 ;
			act_video_ris_ff	<= 1'b0 ;
		end
		else begin
			frame_rst_lat_ff	<= frame_rst ;
			frame_rst_lat1_ff	<= frame_rst_lat_ff ;
			frame_rst_lat2_ff	<= frame_rst_lat1_ff ;
			frame_rst_fal_ff	<= frame_rst_lat2_ff & ~frame_rst_lat1_ff ;
			req_field_lat_ff	<= req_field ;
			act_video_lat_ff	<= act_video ;
			act_video_lat1_ff	<= act_video_lat_ff ;
			act_video_lat2_ff	<= act_video_lat1_ff ;
			act_video_fal_ff	<=	act_video_lat2_ff & ~act_video_lat1_ff ;
			act_video_ris_ff	<= ~act_video_lat2_ff &  act_video_lat1_ff ;
		end
	end


////////////////////////
//	Video Stream I/F  //
////////////////////////

	assign	ready_o		= ready_int_ff ;

	assign	frame_start	= frame_start_i & valid_i & ( pixel_i[3:0] == 4'h0 ) & ready_int_ff ;
	assign	frame_end	= frame_end_i & valid_i & ready_int_ff ;

	always @( posedge vsi_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			vsi_state_ff	<= FRST_WAIT ;
			ready_int_ff	<= 1'b0 ;
			field_str_ff	<= 1'b0 ;
			frame_input_ff	<= 1'b0 ;
		end
		else if ( vsi_srst ) begin
			vsi_state_ff	<= FRST_WAIT ;
			ready_int_ff	<= 1'b0 ;
			field_str_ff	<= 1'b0 ;
			frame_input_ff	<= 1'b0 ;
		end
		else if ( param_ris_ff ) begin
			vsi_state_ff	<= FRST_WAIT ;
			ready_int_ff	<= 1'b0 ;
			field_str_ff	<= 1'b0 ;
			frame_input_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			case ( vsi_state_ff )
				FRST_WAIT	:
					if ( frame_rst_fal_ff ) begin
						vsi_state_ff	<= START_WAIT ;
						ready_int_ff	<= 1'b0 ;
						field_str_ff	<= field_str_ff ;
						frame_input_ff	<= 1'b0 ;
					end

				START_WAIT	:
					if ( act_video_ris_ff ) begin
						vsi_state_ff	<= FRST_WAIT ;
						ready_int_ff	<= 1'b0 ;
						field_str_ff	<= field_str_ff ;
						frame_input_ff	<= 1'b0 ;
					end
					else if ( frame_start && ( ( !req_field_lat_ff ) == field_i ) ) begin
						vsi_state_ff	<= FRM_PERIOD ;
						ready_int_ff	<= ~fifo_afull ;
						field_str_ff	<= field_i ;
						frame_input_ff	<= 1'b1 ;
					end
					else if ( frame_start && ( ( !req_field_lat_ff ) != field_i ) ) begin
						vsi_state_ff	<= NEXT_FRAME_WAIT ;
						ready_int_ff	<= 1'b0 ;
						field_str_ff	<= field_i ;
						frame_input_ff	<= 1'b0 ;
					end
					else begin
						vsi_state_ff	<= START_WAIT ;
						ready_int_ff	<= ~fifo_afull ;
						field_str_ff	<= field_str_ff ;
						frame_input_ff	<= 1'b0 ;
					end

				NEXT_FRAME_WAIT :
					if ( frame_rst_fal_ff && ( ( !req_field_lat_ff ) == field_str_ff ) ) begin
						vsi_state_ff	<= FRM_PERIOD ;
						ready_int_ff	<= ~fifo_afull ;
						field_str_ff	<= field_str_ff ;
						frame_input_ff	<= 1'b1 ;
					end

				FRM_PERIOD	:
					if ( act_video_fal_ff || frame_end ) begin
						vsi_state_ff	<= FRST_WAIT ;
						ready_int_ff	<= 1'b0 ;
						field_str_ff	<= field_str_ff ;
						frame_input_ff	<= 1'b0 ;
					end
					else begin
						vsi_state_ff	<= FRM_PERIOD ;
						ready_int_ff	<= ~fifo_afull ;
						field_str_ff	<= field_str_ff ;
						frame_input_ff	<= 1'b1 ;
					end

				default		:
					begin
						vsi_state_ff	<= FRST_WAIT ;
						ready_int_ff	<= 1'b0 ;
						field_str_ff	<= field_str_ff ;
						frame_input_ff	<= 1'b0 ;
					end

			endcase
		end
	end

	always @( posedge vsi_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			fifo_we_ff		<= 1'b0 ;
			fifo_wdata_ff	<= {PIXEL_BIT{1'b0}} ;
		end
		else if ( vsi_srst ) begin
			fifo_we_ff		<= 1'b0 ;
			fifo_wdata_ff	<= {PIXEL_BIT{1'b0}} ;
		end
		else if ( vsi_state_ff != FRM_PERIOD ) begin
			fifo_we_ff		<= 1'b0 ;
			fifo_wdata_ff	<= {PIXEL_BIT{1'b0}} ;
		end
		else begin
			fifo_we_ff		<= valid_i & ready_int_ff ;
			fifo_wdata_ff	<= pixel_i ;
		end
	end


///////////////////////////////////////////////////////
//	Video Stream I/F to Parallel Output signal FIFO  //
///////////////////////////////////////////////////////

	vtg11_async_fifo_cont
		#(
			.FIFO_DEPTH			( FIFO_DEPTH	) ,
			.DATA_BIT_WIDTH		( PIXEL_BIT		) ,
			.WATERMARK_W		( 3				) ,
			.WATERMARK_R		( 2				) ,
			.RETIME				( 0				)
		)
		u_vtg11_async_fifo_cont (
			.wt_clk				( vsi_clk		) ,
			.rd_clk				( out_clk		) ,
			.rst_n				( rst_n			) ,
			.srst				( frame_rst		) ,

			.wt_en_i			( fifo_we_ff	) ,
			.rd_en_i			( fifo_re		) ,
			.data_i				( fifo_wdata_ff	) ,
			.data_o				( fifo_rdata	) ,
			.a_empty_o			( 				) ,
			.empty_o			( fifo_empty	) ,
			.a_full_o			( fifo_afull	) ,
			.full_o				( 				) ,

			.ram_wt_en_o		( ram_wen		) ,
			.ram_rd_en_o		( ram_ren		) ,
			.ram_wt_adr_o		( ram_wadr		) ,
			.ram_rd_adr_o		( ram_radr		) ,
			.ram_wt_data_o		( ram_wdata		) ,
			.ram_rd_data_i		( ram_rdata		)
		) ;

	generate
		if ( DEVICE == 0 ) begin	: default_ram

			vtg11_srd_swr_ram
				#(
					.ADR_BIT				( FIFO_DEPTH		) ,
					.DATA_BIT				( PIXEL_BIT			) ,
					.DATA_DEPTH				( 1 << FIFO_DEPTH	)
				)
				u_ram_for_async_fifo (
					.wclk					( vsi_clk			) ,
					.rclk					( out_clk			) ,

					.wadr_i					( ram_wadr			) ,
					.radr_i					( ram_radr			) ,
					.wen_i					( ram_wen			) ,
					.ren_i					( ram_ren			) ,

					.wdata_i				( ram_wdata			) ,
					.rdata_o				( ram_rdata			)
				) ;

		end
		else begin					: lattice_ram_macro

			pmi_ram_dp
				#(
					.pmi_wr_addr_depth		( 1 << FIFO_DEPTH	) ,
					.pmi_wr_addr_width		( FIFO_DEPTH		) ,
					.pmi_wr_data_width		( PIXEL_BIT			) ,
					.pmi_rd_addr_depth		( 1 << FIFO_DEPTH	) ,
					.pmi_rd_addr_width		( FIFO_DEPTH		) ,
					.pmi_rd_data_width		( PIXEL_BIT			) ,
					.pmi_regmode			( "noreg"			) ,
					.pmi_gsr				( "disable"			) ,
					.pmi_resetmode			( "sync"			) ,
					.pmi_optimization		( "speed"			) ,
					.pmi_init_file			( "none"			) ,
					.pmi_init_file_format	( "binary"			) ,
					.pmi_family				( DEVICE			) ,
					.module_type			( "pmi_ram_dp"		)
				)
				u_ram_for_async_fifo (
					.Reset					( ~rst_n			) ,

					.WrClock				( vsi_clk			) ,
					.RdClock				( out_clk			) ,

					.WrAddress				( ram_wadr			) ,
					.RdAddress				( ram_radr			) ,
					.WE						( ram_wen			) ,
					.WrClockEn				( 1'b1				) ,
					.RdClockEn				( ram_ren			) ,

					.Data					( ram_wdata			) ,
					.Q						( ram_rdata			)
				);
		end
	endgenerate

/////////////////////////////////////
//	Output Video Timing Generator  //
/////////////////////////////////////

	always @( posedge out_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frame_input_lat_ff		<= 1'b0 ;
			frame_input_lat1_ff		<= 1'b0 ;
			const_data_out_lat_ff	<= 1'b0 ;
			const_data_out_lat1_ff	<= 1'b0 ;
		end
		else begin
			frame_input_lat_ff		<= frame_input_ff ;
			frame_input_lat1_ff		<= frame_input_lat_ff ;
			const_data_out_lat_ff	<= const_data_out_i ;
			const_data_out_lat1_ff	<= const_data_out_lat_ff ;
		end
	end

	assign	const_data_out	= const_data_out_lat1_ff | ( ~frame_input_lat1_ff & fifo_empty ) ;
	assign	rst_timing		= { {V_COUNTER-1{1'b0}} , 1'b1 } ;

	vtg11_timing_gen
		# (
			.Q_BIT				( Q_BIT				) ,
			.PLANE				( PLANE				) ,
			.VSI_FMIN			( VSI_FMIN			) ,
			.VIDEO_FMAX			( VIDEO_FMAX		) ,
			.H_FRONT			( H_FRONT			) ,
			.H_INTERVAL			( H_INTERVAL		) ,
			.H_BACK				( H_BACK			) ,
			.H_VIDEO			( H_VIDEO			) ,
			.V_FRONT			( V_FRONT			) ,
			.V_INTERVAL			( V_INTERVAL		) ,
			.V_BACK				( V_BACK			) ,
			.V_VIDEO			( V_VIDEO			) ,
			.V_DELAY			( V_DELAY			) ,
			.EVEN_V_FRONT		( EVEN_V_FRONT		) ,
			.EVEN_V_INTERVAL	( EVEN_V_INTERVAL	) ,
			.EVEN_V_BACK		( EVEN_V_BACK 		) ,
			.EVEN_V_VIDEO		( EVEN_V_VIDEO		) ,
			.EVEN_V_DELAY		( EVEN_V_DELAY		) ,
			.EVEN2ODD_V_DELAY	( EVEN2ODD_V_DELAY	) ,
			.EVEN2ODD_H_DELAY	( EVEN2ODD_H_DELAY	) ,
			.ODD2EVEN_V_DELAY	( ODD2EVEN_V_DELAY	) ,
			.ODD2EVEN_H_DELAY	( ODD2EVEN_H_DELAY	) ,
			.RESET_TIMING		( RESET_TIMING		) ,
			.H_F_P				( H_F_P				) ,
			.H_S_I				( H_S_I				) ,
			.H_B_P				( H_B_P				) ,
			.H_V_I				( H_V_I				) ,
			.V_F_P				( V_F_P				) ,
			.V_S_I				( V_S_I				) ,
			.V_B_P				( V_B_P				) ,
			.V_V_I				( V_V_I				)
		)
		u_vtg11_timing_gen (
			.clk 				( out_clk				) ,
			.rst_n				( rst_n					) ,
			.srst				( out_srst				) ,

			.frame_rst_o 		( frame_rst 			) ,
			.enable				( enable				) ,
			.param_update_i		( param_update_i		) ,

			.interlace_mode_i	( interlace_mode_i		) ,

			.ex_sync_pol_i		( ex_sync_pol_i			) ,
			.ex_odd_field_pol_i	( ex_odd_field_pol_i	) ,
			.h_sync_pol_i		( h_sync_pol_i			) ,
			.v_sync_pol_i		( v_sync_pol_i			) ,
			.odd_field_pol_i 	( odd_field_pol_i 		) ,
			.h_valid_pol_i		( h_valid_pol_i			) ,
			.v_valid_pol_i		( v_valid_pol_i			) ,
			.de_pol_i			( de_pol_i				) ,

			.h_front_i			( h_front_i				) ,
			.h_interval_i		( h_interval_i			) ,
			.h_back_i			( h_back_i				) ,
			.h_video_i			( h_video_i				) ,
			.v_front_i			( v_front_i				) ,
			.v_interval_i		( v_interval_i			) ,
			.v_back_i			( v_back_i				) ,
			.v_video_i			( v_video_i				) ,
			.v_delay_i			( v_delay_i				) ,
			.even_v_front_i		( even_v_front_i		) ,
			.even_v_interval_i	( even_v_interval_i		) ,
			.even_v_back_i		( even_v_back_i			) ,
			.even_v_video_i		( even_v_video_i		) ,
			.even_v_delay_i		( even_v_delay_i		) ,
			.even2odd_v_delay_i	( even2odd_v_delay_i	) ,
			.even2odd_h_delay_i	( even2odd_h_delay_i	) ,
			.odd2even_v_delay_i	( odd2even_v_delay_i	) ,
			.odd2even_h_delay_i	( odd2even_h_delay_i	) ,
			.reset_timing_i		( rst_timing			) ,
			.reset_interval_i	( RESET_INTERVAL[3:0]	) ,

			.const_data_out_i	( const_data_out		) ,
			.const_out_mode_i	( const_out_mode_i		) ,
			.const_data0_i		( const_data0_i			) ,
			.const_data1_i		( const_data1_i			) ,

			.external_sync_i 	( external_sync_i 		) ,
			.external_field_i	( external_field_i		) ,

			.req_field_o 		( req_field				) ,
			.pixel_rdreq_o		( fifo_re				) ,
			.pixel_oe_o			( 						) ,
			.pixel_data_i		( fifo_rdata			) ,

			.act_video_o		( act_video				) ,

			.h_sync_o			( h_sync_o				) ,
			.v_sync_o			( v_sync_o				) ,
			.field_o 			( field_o 				) ,
			.de_o				( de_o					) ,
			.h_valid_o			( h_valid_o				) ,
			.v_valid_o			( v_valid_o				) ,
			.pixel_o 			( pixel_o 				)
		) ;

endmodule
`default_nettype wire