//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		Asynchronous First In First Out
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Jan. 16 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		FIFO_DEPTH		: FIFO depth.
//		DATA_BIT_WIDTH	: Data bit width.
//		WATERMARK_W 	: Write watermark.
//		WATERMARK_R 	: Read watermark.
//		RETIME			: Read data retiming mode.
//						  1:Retimng 0: unretiming
//----------------------------------------------------------------------------------------
// I/O PORTS
//		wt_clk			: Write clock.
//		rd_clk			: Read clock.
//		rst_n			: Asynchronous reset.
//		srst			: Synchronous reset.
//
//		wt_en_i			: Write enable.
//		rd_en_i			: Read enable.
//		data_i 			: Write data.
//		data_o			: Read data.
//		a_empty_o 		: Almost empty flag.
//		empty_o			: Empty flag.
//		a_full_o		: Almost full flag.
//		full_o			: Full flag.
//
//		ram_wt_en_o		: RAM write enable.
//		ram_rd_en_o		: RAM read enable.
//		ram_wt_adr_o	: RAM write adress.
//		ram_rd_adr_o	: RAM read adress.
//		ram_wt_data_o	: RAM write data.
//		ram_rd_data_i	: RAM read data.
//----------------------------------------------------------------------------------------
`default_nettype none
`timescale 1ns / 1ps

module	vtg11_async_fifo_cont
	#(
		parameter	FIFO_DEPTH		= 8 	,
		parameter	DATA_BIT_WIDTH	= 16	,
		parameter	WATERMARK_W 	= 1 	,
		parameter	WATERMARK_R 	= 1 	,
		parameter	RETIME			= 1
	)
	(
		input	wire								wt_clk			,
		input	wire								rd_clk			,
		input	wire								rst_n			,
		input	wire								srst			,

		input	wire								wt_en_i			,
		input	wire								rd_en_i			,
		input	wire	[ DATA_BIT_WIDTH-1 : 0 ]	data_i			,
		output	wire	[ DATA_BIT_WIDTH-1 : 0 ]	data_o			,
		output	wire								a_empty_o 		,
		output	wire								empty_o			,
		output	wire								a_full_o		,
		output	wire								full_o			,

		output	wire								ram_wt_en_o		,
		output	wire								ram_rd_en_o		,
		output	wire	[	  FIFO_DEPTH-1 : 0 ]	ram_wt_adr_o	,
		output	wire	[	  FIFO_DEPTH-1 : 0 ]	ram_rd_adr_o	,
		output	wire	[ DATA_BIT_WIDTH-1 : 0 ]	ram_wt_data_o	,
		input	wire	[ DATA_BIT_WIDTH-1 : 0 ]	ram_rd_data_i
	) ;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg 	[	  FIFO_DEPTH-1 : 0 ]	wrp_gray_ff 			;
	reg 	[	  FIFO_DEPTH-1 : 0 ]	rdp_sync_ff 			;
	reg 	[	  FIFO_DEPTH-1 : 0 ]	rdp_w_ff				;
	wire	[	  FIFO_DEPTH-1 : 0 ]	sub_pointer 			;
	wire	[	  FIFO_DEPTH-1 : 0 ]	sub_afr_pointer 		;
	reg 								afull_ff				;
	reg 								full_ff 				;

	reg 	[	  FIFO_DEPTH-1 : 0 ]	wrp_sync_ff 			;
	reg 	[	  FIFO_DEPTH-1 : 0 ]	wrp_r_ff				;
	reg 	[	  FIFO_DEPTH-1 : 0 ]	rdp_gray_ff 			;
	reg 								empty_ff				;
	reg 								aempty_ff				;
	wire	[	  FIFO_DEPTH-1 : 0 ]	sub_r_pointer			;
	wire	[	  FIFO_DEPTH-1 : 0 ]	sub_aer_pointer 		;

	reg		[	  FIFO_DEPTH-1 : 0 ]	wrp_b_ff 				;
	wire	[	  FIFO_DEPTH-1 : 0 ]	wrp_b_nxt				;
	wire	[	  FIFO_DEPTH-1 : 0 ]	wrp_gray				;

	reg		[	  FIFO_DEPTH-1 : 0 ]	rdp_b_ff 				;
	wire	[	  FIFO_DEPTH-1 : 0 ]	rdp_b_nxt				;
	wire	[	  FIFO_DEPTH-1 : 0 ]	rdp_gray				;

	wire	[	  FIFO_DEPTH-1 : 0 ]	rdp_w_g2b				;
	wire	[	  FIFO_DEPTH-1 : 0 ]	wrp_r_g2b				;

	reg 	[ DATA_BIT_WIDTH-1 : 0 ]	data_out_retime0_ff 	;

	wire								set_wrp_gray_v			;
	wire								set_rdp_gray_v			;

	reg 								srst_wclk_lat1_ff		;
	reg 								srst_wclk_lat2_ff		;
	reg 								wt_srst_ff				;
	reg 								wt_srst_lat_ff			;
	reg 								wt_srst_rclk_lat1_ff	;
	reg 								wt_srst_rclk_lat2_ff	;
	reg 								rd_srst_clr_ff			;
	reg 								srst_rclk_lat1_ff		;
	reg 								srst_rclk_lat2_ff		;
	reg 								rd_srst_ff				;
	reg 								rd_srst_lat_ff			;
	reg 								rd_srst_wclk_lat1_ff	;
	reg 								rd_srst_wclk_lat2_ff	;
	reg 								wt_srst_clr_ff			;


// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	assign	a_empty_o	 	= aempty_ff ;
	assign	empty_o			= empty_ff ;
	assign	a_full_o		= afull_ff ;
	assign	full_o			= full_ff ;
	assign	data_o			= ( RETIME ) ?	data_out_retime0_ff : ram_rd_data_i ;

	assign	ram_wt_adr_o	= wrp_b_ff ;
	assign	ram_rd_adr_o	= rdp_b_ff ;

	assign	ram_wt_en_o		= set_wrp_gray_v ;
	assign	ram_rd_en_o		= set_rdp_gray_v ;
	assign	ram_wt_data_o	= data_i ;

	assign	set_wrp_gray_v	= wt_en_i & ( ~full_ff ) ;
	assign	set_rdp_gray_v	= rd_en_i & ( ~empty_ff ) ;

	//===================================//
	// Write Pointer (Gray Code Counter) //
	//===================================//
	assign	wrp_b_nxt	= wrp_b_ff + 1'b1 ;
	assign	wrp_gray	= ( { 1'b0 , wrp_b_nxt[ FIFO_DEPTH-1 : 1 ] } ) ^ wrp_b_nxt ;

	always @( posedge wt_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			wrp_gray_ff	<= { FIFO_DEPTH{1'b0} } ;
			wrp_b_ff	<= { FIFO_DEPTH{1'b0} } ;
		end
		else if ( wt_srst_ff ) begin
			wrp_gray_ff	<= { FIFO_DEPTH{1'b0} } ;
			wrp_b_ff	<= { FIFO_DEPTH{1'b0} } ;
		end
		else if ( set_wrp_gray_v ) begin
			wrp_gray_ff	<= wrp_gray ;
			wrp_b_ff	<= wrp_b_nxt ;
		end
	end

	//================//
	// clk_w -> clk_r //
	//================//
	always @ ( posedge rd_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			wrp_sync_ff	<= { FIFO_DEPTH{1'b0} } ;
			wrp_r_ff	<= { FIFO_DEPTH{1'b0} } ;
		end
		else begin
			wrp_sync_ff <= wrp_gray_ff ;
			wrp_r_ff	<= wrp_sync_ff ;
		end
	end

	//==================================//
	// Read Pointer (Gray Code Counter) //
	//==================================//
	assign	rdp_b_nxt	= rdp_b_ff + 1'b1 ;
	assign	rdp_gray	= ( { 1'b0 , rdp_b_nxt[FIFO_DEPTH-1:1] } ) ^ rdp_b_nxt ;

	always @ ( posedge rd_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rdp_gray_ff	<= { FIFO_DEPTH{1'b0} } ;
			rdp_b_ff	<= { FIFO_DEPTH{1'b0} } ;
		end
		else if ( rd_srst_ff ) begin
			rdp_gray_ff	<= { FIFO_DEPTH{1'b0} } ;
			rdp_b_ff	<= { FIFO_DEPTH{1'b0} } ;
		end
		else if ( set_rdp_gray_v ) begin
			rdp_gray_ff	<= rdp_gray ;
			rdp_b_ff	<= rdp_b_nxt ;
		end
	end

	//=================================//
	// almost empty and empty checking //
	//=================================//
	assign	wrp_r_g2b		= { 1'b0 , wrp_r_g2b[FIFO_DEPTH-1:1] } ^ wrp_r_ff ;
	assign	sub_r_pointer	= wrp_r_g2b - rdp_b_nxt ;
	assign	sub_aer_pointer = wrp_r_g2b - rdp_b_ff ;

	always @ ( posedge rd_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			aempty_ff	<= 1'b1 ;
		end
		else if ( rd_srst_ff ) begin
			aempty_ff	<= 1'b1 ;
		end
		else if ( set_rdp_gray_v && ( sub_r_pointer <= WATERMARK_R ) ) begin
			aempty_ff	<= 1'b1 ;
		end
		else if ( sub_aer_pointer > WATERMARK_R ) begin
			aempty_ff	<= 1'b0 ;
		end
	end

	always @ ( posedge rd_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			empty_ff	<= 1'b1 ;
		end
		else if ( rd_srst_ff ) begin
			empty_ff	<= 1'b1 ;
		end
		else if ( set_rdp_gray_v && ( rdp_gray == wrp_r_ff ) ) begin
			empty_ff	<= 1'b1 ;
		end
		else if ( rdp_gray_ff != wrp_r_ff ) begin
			empty_ff	<= 1'b0 ;
		end
	end

	//================//
	// clk_r -> clk_w //
	//================//
	always @ ( posedge wt_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rdp_sync_ff	<= { FIFO_DEPTH{1'b0} } ;
			rdp_w_ff	<= { FIFO_DEPTH{1'b0} } ;
		end
		else begin
			rdp_sync_ff	<= rdp_gray_ff ;
			rdp_w_ff	<= rdp_sync_ff ;
		end
	end

	//===============================//
	// almost full and full checking //
	//===============================//
	assign	rdp_w_g2b		= ( { 1'b0 , rdp_w_g2b[FIFO_DEPTH-1:1] } ) ^ rdp_w_ff ;
	assign	sub_pointer		= rdp_w_g2b - wrp_b_nxt ;
	assign	sub_afr_pointer	= rdp_w_g2b - wrp_b_ff ;

	always @ ( posedge wt_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			afull_ff	<= 1'b0 ;
		end
		else if ( wt_srst_ff || srst_wclk_lat2_ff ) begin
			afull_ff	<= 1'b1 ;
		end
		else if ( wt_srst_clr_ff ) begin
			afull_ff	<= 1'b0 ;
		end
		else if ( set_wrp_gray_v && ( sub_pointer <= WATERMARK_W ) ) begin
			afull_ff	<= 1'b1 ;
		end
		else if ( sub_afr_pointer > WATERMARK_W ) begin
			afull_ff	<= 1'b0 ;
		end
	end

	always @ ( posedge wt_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			full_ff	<= 1'b0 ;
		end
		else if ( wt_srst_ff || srst_wclk_lat2_ff ) begin
			full_ff	<= 1'b1 ;
		end
		else if ( wt_srst_clr_ff ) begin
			full_ff	<= 1'b0 ;
		end
		else if ( set_wrp_gray_v && ( rdp_w_ff == wrp_gray ) ) begin
			full_ff	<= 1'b1 ;
		end
		else if ( rdp_w_ff != wrp_gray_ff ) begin
			full_ff	<= 1'b0 ;
		end
	end

	//==============//
	// FiFo Control //
	//==============//
	always @ ( posedge rd_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			data_out_retime0_ff	<= { DATA_BIT_WIDTH{1'b0} } ;
		end
		else if ( rd_srst_ff ) begin
			data_out_retime0_ff	<= { DATA_BIT_WIDTH{1'b0} } ;
		end
		else if ( set_rdp_gray_v ) begin
			data_out_retime0_ff	<= ram_rd_data_i;
		end
	end

	//===================//
	// synchronous reset //
	//===================//
	// write clock srst
	always @ ( posedge wt_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			wt_srst_ff	<= 1'b0 ;
		end
		else if ( srst_wclk_lat2_ff ) begin
			wt_srst_ff	<= 1'b1 ;
		end
		else if ( wt_srst_clr_ff && !srst_wclk_lat2_ff ) begin
			wt_srst_ff	<= 1'b0 ;
		end
	end

	always @ ( posedge wt_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			srst_wclk_lat1_ff	<= 1'b0 ;
			srst_wclk_lat2_ff	<= 1'b0 ;
			wt_srst_lat_ff		<= 1'b0 ;
		end
		else begin
			srst_wclk_lat1_ff	<= srst ;
			srst_wclk_lat2_ff	<= srst_wclk_lat1_ff ;
			wt_srst_lat_ff		<= wt_srst_ff ;
		end
	end

	always @ ( posedge wt_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rd_srst_wclk_lat1_ff	<= 1'b0 ;
			rd_srst_wclk_lat2_ff	<= 1'b0 ;
		end
		else begin
			rd_srst_wclk_lat1_ff	<= rd_srst_lat_ff ;
			rd_srst_wclk_lat2_ff	<= rd_srst_wclk_lat1_ff ;
		end
	end

	always @ ( posedge wt_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			wt_srst_clr_ff	<= 1'b0 ;
		end
		else if ( !wt_srst_ff ) begin
			wt_srst_clr_ff	<= 1'b0 ;
		end
		else if ( rd_srst_wclk_lat2_ff ) begin
			wt_srst_clr_ff	<= 1'b1 ;
		end
	end

	// read clock srst
	always @ ( posedge rd_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rd_srst_ff	<= 1'b0 ;
		end
		else if ( srst_rclk_lat2_ff ) begin
			rd_srst_ff	<= 1'b1 ;
		end
		else if ( rd_srst_clr_ff && !srst_rclk_lat2_ff ) begin
			rd_srst_ff	<= 1'b0 ;
		end
	end

	always @ ( posedge rd_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			srst_rclk_lat1_ff	<= 1'b0 ;
			srst_rclk_lat2_ff	<= 1'b0 ;
			rd_srst_lat_ff		<= 1'b0 ;
		end
		else begin
			srst_rclk_lat1_ff	<= srst ;
			srst_rclk_lat2_ff	<= srst_rclk_lat1_ff ;
			rd_srst_lat_ff		<= rd_srst_ff ;
		end
	end

	always @ ( posedge rd_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			wt_srst_rclk_lat1_ff	<= 1'b0 ;
			wt_srst_rclk_lat2_ff	<= 1'b0 ;
		end
		else begin
			wt_srst_rclk_lat1_ff	<= wt_srst_lat_ff ;
			wt_srst_rclk_lat2_ff	<= wt_srst_rclk_lat1_ff ;
		end
	end

	always @ ( posedge rd_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rd_srst_clr_ff	<= 1'b0 ;
		end
		else if ( !rd_srst_ff ) begin
			rd_srst_clr_ff	<= 1'b0 ;
		end
		else if ( wt_srst_rclk_lat2_ff ) begin
			rd_srst_clr_ff	<= 1'b1 ;
		end
	end

endmodule
`default_nettype wire