//----------------------------------------------------------------------------------------
//Copyright (C) 2025 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		SLVS-EC RX UDL I/F to VSI I/F Bridge module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 May 28 2025	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT				: data-bit-width
//		H_PIXEL				: H Pixel count setting
//		V_PIXEL				: V Pixel count setting
//----------------------------------------------------------------------------------------
// I/O PORTS
//		inclk				: clock signal
//		outclk				: clock signal
//		rst_inclk_n			: reset signal inclk sync
//		rst_outclk_n		: reset signal outclk sync
//
//		through_i			: through white balance adjust
//		r_gain_i			: red gain setting
//		b_gain_i			: blue gain setting
//		frame_start_i		: frame start
//		frame_end_i			: frame end
//		valid_line_start_i	: line start
//		valid_line_end_i	: line end
//		pixel_valid_i		: pixel valid
//		pixel_data_i		: pixel data
//
//		frame_start_o		: frame strat
//		frame_end_o			: frame end
//		valid_o				: valid
//		pixel_o				: pixel
//		field_o				: field
//		ready_i				: ready
//		width_o				: width
//		height_o			: height
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module slec_rx_vsi_bridge #(
	parameter									H_PIXEL		= 13'd2472		,
	parameter									V_LINE		= 13'd2128		,
	parameter									Q_BIT		= 12			,
	parameter									FIFO_DEPTH	= 2048
	)
	(
	input	wire								inclk					,
	input	wire								outclk					,
	input	wire								rst_inclk_n				,
	input	wire								rst_outclk_n			,

	input	wire								frame_start_i			,
	input	wire								frame_end_i				,
	input	wire								valid_line_start_i		,
	input	wire								valid_line_end_i		,
	input	wire	[   7 :  0 ] 				pixel_valid_i			,
	input	wire	[ 511 :  0 ]				pixel_data_i			,

	output	wire								frame_start_o			,
	output	wire								frame_end_o				,
	output	wire								valid_o					,
	output	wire	[ Q_BIT*2-1  : 0 ]			pixel_o					,
	output	wire								field_o					,
	input	wire								ready_i					,
	output	wire	[  12  : 0 ]				width_o					,
	output	wire	[  12  : 0 ]				height_o
);

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	function integer	log2 ;
		input integer	value ;
		begin
			value	= value - 1 ;
			for (log2 = 0 ; value > 0 ; log2 = log2 + 1 ) begin
				value	= value >> 1 ;
			end
		end
	endfunction

	localparam	FIFO_AD_BIT	= log2(FIFO_DEPTH)							;

// =============================================================================
// PORT DECLARATION
// =============================================================================
// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg											frame_valid_ff			;

	wire			[ 511 :  0 ]				data_fifo_din			;
	wire			[ 511 :  0 ]				data_fifo_dout			;
	reg				[ 511 :  0 ]				pixel_data_ff			;

	wire										frame_start_sync		;
	wire										frame_end_sync			;

	wire			[   7 :  0 ]				pixel_valid 			;
	wire										fifo_wen				;
	wire										fifo_ren				;
	reg											fifo_ren_ff				;
	reg											fifo_ren_mask_ff		;
	reg				[ FIFO_AD_BIT-1 :  0 ]		wadr_ff					;
	wire			[ FIFO_AD_BIT-1 :  0 ]		wadr_sync				;
	reg				[ FIFO_AD_BIT-1 :  0 ]		radr_ff					;
	reg				[   4 :  0 ]				ctrl_count_ff			;
	reg				[   7 :  0 ]				pixel_valid_ff			;

	reg				[  12 :  0 ]				pixel_count_ff			;
	reg				[  12 :  0 ]				line_count_ff			;

	reg											frame_start_ff			;
	reg											frame_end_ff			;
	reg											line_valid_ff			;
	wire										frame_end				;
	reg				[ Q_BIT*2-1  : 0 ]			pixel_tmp				;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	assign data_fifo_din = pixel_data_i ;

	always @( posedge inclk or negedge rst_inclk_n ) begin
		if ( ~rst_inclk_n ) begin
			frame_valid_ff	<= 1'b0;
		end
		else begin
			if ( frame_end_i ) begin
				frame_valid_ff	<= 1'b0;
			end
			else if( frame_start_i ) begin
				frame_valid_ff	<= 1'b1;
			end
		end
	end

	always @( posedge inclk or negedge rst_inclk_n ) begin
		if ( ~rst_inclk_n ) begin
			line_valid_ff	<= 1'b0;
		end
		else begin
			if ( valid_line_end_i ) begin
				line_valid_ff	<= 1'b0;
			end
			else if( valid_line_start_i ) begin
				line_valid_ff	<= 1'b1;
			end
		end
	end

	assign fifo_wen = pixel_valid_i[0] & line_valid_ff & frame_valid_ff;

	srd_swr_ram
		#(
			.ADR_BIT			( FIFO_AD_BIT							),
			.DATA_BIT			( 8										),
			.DATA_DEPTH			( FIFO_DEPTH							)
		)
		u_valid_fifo (
			.wclk				( inclk									),
			.rclk				( outclk								),
			.wadr_i				( wadr_ff								),
			.radr_i				( radr_ff								),
			.wen_i				( fifo_wen								),
			.ren_i				( fifo_ren								),
			.wdata_i			( pixel_valid_i							),
			.rdata_o			( pixel_valid							)
		);

	srd_swr_ram
		#(
			.ADR_BIT			( FIFO_AD_BIT							),
			.DATA_BIT			( 512									),
			.DATA_DEPTH			( FIFO_DEPTH							)
		)
		u_data_fifo (
			.wclk				( inclk									),
			.rclk				( outclk								),
			.wadr_i				( wadr_ff	 							),
			.radr_i				( radr_ff								),
			.wen_i				( fifo_wen								),
			.ren_i				( fifo_ren								),
			.wdata_i			( data_fifo_din							),
			.rdata_o			( data_fifo_dout						)
		);

	always @( posedge inclk or negedge rst_inclk_n ) begin
		if ( ~rst_inclk_n ) begin
			wadr_ff	<= { FIFO_AD_BIT{1'b0} };
		end
		else begin
			if ( frame_start_i ) begin
				wadr_ff	<= { FIFO_AD_BIT{1'b0} };
			end
			else if( fifo_wen ) begin
				wadr_ff	<= wadr_ff + 1'b1;
			end
		end
	end

	always @( posedge outclk or negedge rst_outclk_n ) begin
		if ( ~rst_outclk_n ) begin
			radr_ff	<= { FIFO_AD_BIT{1'b0} };
		end
		else begin
			if ( frame_start_sync ) begin
				radr_ff	<= { FIFO_AD_BIT{1'b0} };
			end
			else if( fifo_ren ) begin
				radr_ff	<= radr_ff + 1'b1;
			end
		end
	end

	cdc_multi_trns
		#(
			.DATA_WIDTH			( FIFO_AD_BIT							)
		)
		u_cdc_multi_trns (
			.clk_in				( inclk									),
			.clk_out			( outclk								),
			.rst_n				( rst_inclk_n							),
			.data_i				( wadr_ff								),
			.data_o				( wadr_sync								)
		);

	cdc_pls_trns
		u_fs_pls_trns (
			.clk_in				( inclk									),
			.clk_out			( outclk								),
			.rst_n				( rst_inclk_n							),
			.pls_i				( frame_start_i							),
			.pls_o				( frame_start_sync						)
		);

	cdc_pls_trns
		u_fe_pls_trns (
			.clk_in				( inclk									),
			.clk_out			( outclk								),
			.rst_n				( rst_inclk_n							),
			.pls_i				( frame_end_i							),
			.pls_o				( frame_end_sync						)
		);

	always @( posedge outclk or negedge rst_outclk_n ) begin
		if ( ~rst_outclk_n ) begin
			fifo_ren_mask_ff	<= 1'b0;
		end
		else begin
			if ( frame_start_sync ) begin
				fifo_ren_mask_ff	<= 1'b1;
			end
			else if( wadr_sync == radr_ff ) begin
				fifo_ren_mask_ff	<= 1'b0;
			end
		end
	end

	assign fifo_ren = ~fifo_ren_ff & ~fifo_ren_mask_ff & ( ( wadr_sync != radr_ff ) & ( ( ctrl_count_ff == 0 )
																				    | ( pixel_valid_ff == 8'h01 & ctrl_count_ff == 5'h03 )
																				    | ( pixel_valid_ff == 8'h03 & ctrl_count_ff == 5'h07 )
																				    | ( pixel_valid_ff == 8'h07 & ctrl_count_ff == 5'h0B )
																				    | ( pixel_valid_ff == 8'h0F & ctrl_count_ff == 5'h0F )
																				    | ( pixel_valid_ff == 8'h1F & ctrl_count_ff == 5'h13 )
																				    | ( pixel_valid_ff == 8'h3F & ctrl_count_ff == 5'h17 )
																				    | ( pixel_valid_ff == 8'h7F & ctrl_count_ff == 5'h1B )
																				    | ( pixel_valid_ff == 8'hFF & ctrl_count_ff == 5'h1F )  ) ) ;

	always @( posedge outclk or negedge rst_outclk_n ) begin
		if ( ~rst_outclk_n ) begin
			fifo_ren_ff	<= 0;
		end
		else begin
			fifo_ren_ff <= fifo_ren;
		end
	end

	always @( posedge outclk or negedge rst_outclk_n ) begin
		if ( ~rst_outclk_n ) begin
			pixel_data_ff	<= 0;
		end
		else begin
			if ( fifo_ren_ff ) begin
				pixel_data_ff <= data_fifo_dout;
			end
		end
	end

	always @( posedge outclk or negedge rst_outclk_n ) begin
		if ( ~rst_outclk_n ) begin
			pixel_valid_ff	<= 0;
		end
		else begin
			if ( fifo_ren_ff ) begin
				pixel_valid_ff <= pixel_valid;
			end
		end
	end


	always @( posedge outclk or negedge rst_outclk_n ) begin
		if ( ~rst_outclk_n ) begin
			ctrl_count_ff	<= 5'd0;
		end
		else begin
			if ( frame_start_sync ) begin
				ctrl_count_ff <= 5'd0;
			end
			else if ( fifo_ren_ff ) begin
				ctrl_count_ff <= 5'd1;
			end
//			else if(      ctrl_count_ff[4]
			else if(     &ctrl_count_ff[4:0]
					 ||	( ctrl_count_ff[4:0] == 5'h1B &&  pixel_valid_ff == 8'b0111_1111 )
					 ||	( ctrl_count_ff[4:0] == 5'h17 &&  pixel_valid_ff == 8'b0011_1111 )
					 || ( ctrl_count_ff[4:0] == 5'h13 &&  pixel_valid_ff == 8'b0001_1111 )
					 || ( ctrl_count_ff[4:0] == 5'h0F &&  pixel_valid_ff == 8'b0000_1111 )
					 ||	( ctrl_count_ff[4:0] == 5'h0B &&  pixel_valid_ff == 8'b0000_0111 )
					 || ( ctrl_count_ff[4:0] == 5'h07 &&  pixel_valid_ff == 8'b0000_0011 )
					 || ( ctrl_count_ff[4:0] == 5'h03 &&  pixel_valid_ff == 8'b0000_0001 )) begin
				ctrl_count_ff <= 5'd0;
			end
			else if ( ~frame_start_ff & |ctrl_count_ff & ready_i ) begin
				ctrl_count_ff <= ctrl_count_ff + 5'd2;
			end
		end
	end

	always @( posedge outclk or negedge rst_outclk_n ) begin
		if ( ~rst_outclk_n ) begin
			pixel_count_ff	<= 13'd0;
		end
		else begin
			if ( frame_start_sync ) begin
				pixel_count_ff <= 13'd0;
			end
			else if ( ready_i ) begin
				if ( pixel_count_ff ==  ( H_PIXEL - 13'd2 ) ) begin
					pixel_count_ff <= 13'd0;
				end
				else if ( ~frame_start_ff & valid_o ) begin
					pixel_count_ff <= pixel_count_ff + 13'd2;
				end
			end
		end
	end


	always @( posedge outclk or negedge rst_outclk_n ) begin
		if ( ~rst_outclk_n ) begin
			line_count_ff	<= 13'd0;
		end
		else begin
			if ( frame_start_sync ) begin
				line_count_ff <= 13'd0;
			end
			else if ( ready_i ) begin
				if ( pixel_count_ff == (H_PIXEL - 13'd2) ) begin
					line_count_ff <= line_count_ff + 13'd1;
				end
			end
		end
	end

	always @( * ) begin
		case ( ctrl_count_ff )
			5'd01	: pixel_tmp <= { pixel_data_ff[ 16* 1+Q_BIT-1 : 16* 1 ] , pixel_data_ff[       Q_BIT-1 :     0 ] };
			5'd03	: pixel_tmp <= { pixel_data_ff[ 16* 3+Q_BIT-1 : 16* 3 ] , pixel_data_ff[ 16* 2+Q_BIT-1 : 16* 2 ] };
			5'd05	: pixel_tmp <= { pixel_data_ff[ 16* 5+Q_BIT-1 : 16* 5 ] , pixel_data_ff[ 16* 4+Q_BIT-1 : 16* 4 ] };
			5'd07	: pixel_tmp <= { pixel_data_ff[ 16* 7+Q_BIT-1 : 16* 7 ] , pixel_data_ff[ 16* 6+Q_BIT-1 : 16* 6 ] };
			5'd09	: pixel_tmp <= { pixel_data_ff[ 16* 9+Q_BIT-1 : 16* 9 ] , pixel_data_ff[ 16* 8+Q_BIT-1 : 16* 8 ] };
			5'd11	: pixel_tmp <= { pixel_data_ff[ 16*11+Q_BIT-1 : 16*11 ] , pixel_data_ff[ 16*10+Q_BIT-1 : 16*10 ] };
			5'd13	: pixel_tmp <= { pixel_data_ff[ 16*13+Q_BIT-1 : 16*13 ] , pixel_data_ff[ 16*12+Q_BIT-1 : 16*12 ] };
			5'd15	: pixel_tmp <= { pixel_data_ff[ 16*15+Q_BIT-1 : 16*15 ] , pixel_data_ff[ 16*14+Q_BIT-1 : 16*14 ] };
			5'd17	: pixel_tmp <= { pixel_data_ff[ 16*17+Q_BIT-1 : 16*17 ] , pixel_data_ff[ 16*16+Q_BIT-1 : 16*16 ] };
			5'd19	: pixel_tmp <= { pixel_data_ff[ 16*19+Q_BIT-1 : 16*19 ] , pixel_data_ff[ 16*18+Q_BIT-1 : 16*18 ] };
			5'd21	: pixel_tmp <= { pixel_data_ff[ 16*21+Q_BIT-1 : 16*21 ] , pixel_data_ff[ 16*20+Q_BIT-1 : 16*20 ] };
			5'd23	: pixel_tmp <= { pixel_data_ff[ 16*23+Q_BIT-1 : 16*23 ] , pixel_data_ff[ 16*22+Q_BIT-1 : 16*22 ] };
			5'd25	: pixel_tmp <= { pixel_data_ff[ 16*25+Q_BIT-1 : 16*25 ] , pixel_data_ff[ 16*24+Q_BIT-1 : 16*24 ] };
			5'd27	: pixel_tmp <= { pixel_data_ff[ 16*27+Q_BIT-1 : 16*27 ] , pixel_data_ff[ 16*26+Q_BIT-1 : 16*26 ] };
			5'd29	: pixel_tmp <= { pixel_data_ff[ 16*29+Q_BIT-1 : 16*29 ] , pixel_data_ff[ 16*28+Q_BIT-1 : 16*28 ] };
			5'd31	: pixel_tmp <= { pixel_data_ff[ 16*31+Q_BIT-1 : 16*31 ] , pixel_data_ff[ 16*30+Q_BIT-1 : 16*30 ] };
			default	: pixel_tmp <= 0;
		endcase
	end

	always @( posedge outclk or negedge rst_outclk_n ) begin
		if ( ~rst_outclk_n ) begin
			frame_start_ff	<= 1'b0;
		end
		else begin
			if ( frame_start_sync ) begin
				frame_start_ff	<= 1'b0;
			end
			else if ( ( line_count_ff == 13'd0 ) & ( pixel_count_ff == 13'd0 ) & fifo_ren ) begin
				frame_start_ff	<= 1'b1;
			end
			else if ( ready_i ) begin
				frame_start_ff	<= 1'b0;
			end
		end
	end

	assign	frame_end = ( line_count_ff == ( V_LINE - 13'd1 )) & ( pixel_count_ff == ( H_PIXEL - 13'd2 ));

	assign	valid_o = frame_start_ff || ( (|ctrl_count_ff) & !fifo_ren_ff );

	assign	frame_start_o = frame_start_ff	;
	assign	frame_end_o   = frame_end		;
	assign	field_o       = 1'b0			;
	assign	width_o       = H_PIXEL			;
	assign	height_o      = V_LINE			;
	assign	pixel_o       = pixel_tmp		;

endmodule
`default_nettype wire
