//----------------------------------------------------------------------------------------
//Copyright (C) 2025 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		Reference Design top layer for implementation SLVS-EC v3.0 RX IP to Agilex5
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 May 28 2025	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//
//----------------------------------------------------------------------------------------
// I/O PORTS
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module slec3_rx_ref_top
	(
		input	wire				CLK_100_B2B_p		,
		input	wire				CLK_50_B5A			,
		input	wire				CLK_50_B6A			,

		output	wire	[  3 : 0 ]	FPGA_LED_O			,

		input	wire				LPDDR4B_REFCLK_p	,
		output	wire				LPDDR4B_CS_n		,
		output	wire	[  5 : 0 ]	LPDDR4B_CA			,
		output	wire				LPDDR4B_CK			,
		output	wire				LPDDR4B_CKE			,
		output	wire				LPDDR4B_CK_n		,
		inout	wire	[  3 : 0 ]	LPDDR4B_DM			,
		inout	wire	[ 31 : 0 ]	LPDDR4B_DQ			,
		inout	wire	[  3 : 0 ]	LPDDR4B_DQS			,
		inout	wire	[  3 : 0 ]	LPDDR4B_DQS_n		,
		output	wire				LPDDR4B_RESET_n		,
		input	wire				LPDDR4B_RZQ			,

		output	wire				HDMI_TX_CLK			,
		output	wire				HDMI_TX_HS			,
		output	wire				HDMI_TX_VS			,
		output	wire	[ 23 : 0 ]	HDMI_TX_D			,
		output	wire				HDMI_TX_DE			,
		/*inout 	wire				HDMI_I2C_SCL		,
		inout 	wire				HDMI_I2C_SDA		,
		input 	wire				HDMI_TX_INT			,*/

		input	wire	[  1 : 0 ]	SLEC_REFCLK_P		,
		output	wire				SLEC_XCE			,
		output	wire				SLEC_SCK			,
		output	wire				SLEC_SDI			,
		input	wire				SLEC_SDO			,
		output	wire				SLEC_XVS			,
		output	wire				SLEC_XHS			,
		output	wire				SLEC_XCLR			,
		output	wire				SLEC_XTRIG1			,
		output	wire				SLEC_XTRIG2			,
		output	wire				SLEC_INCK_OE		,
		output	wire				SLEC_PON			,
		input	wire				SLEC_PGOOD			,
		output	wire				SLEC_XMASTER		,
		output	wire				SLEC_OMODE			,
		input	wire	[  7 : 0 ]	SLEC_RXD_P
	);

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	localparam						MAX_LANE		= 8				;
	localparam						SPC	 			= 32			;
	localparam						GCC				= 0				;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	// Reset Sequencer
	wire	[    1 : 0 ]			pma_cu_clk			;
	wire	[    7 : 0 ]			src_rs_req			;
	wire	[    7 : 0 ]			src_rs_grant		;

	// Reset Sequencer
	wire							pll_locked			;

	// NiosV Sub System
	wire	[  31 :  0 ]			nios_gpio_o			;
	wire	[  31 :  0 ]			nios_gpio_i			;
	wire	[   7 :  0 ]			avmm_addr			;
	wire							avmm_write			;
	wire							avmm_read			;
	wire	[  31 :  0 ]			avmm_writedata		;
	wire	[   3 :  0 ]			avmm_byteenable		;
	wire	[  31 :  0 ]			avmm_readdata		;
	wire							avmm_waitrequest	;
	wire							avmm_readdatavalid	;
	wire							slec_sdi_int		;
	wire							slec_sck_int		;
	wire							slec_xce_int		;

	// SLVS-EC
	wire							st_clk				;
	wire							hdr_fstart			;
	wire							hdr_fend			;
	wire							valid_line_start	;
	wire							valid_line_end		;
	wire	[   7 :  0 ]			pixel_valid			;
	wire	[ 511 :  0 ]			pixel_data			;

	// SLVS-EC Local I/F Bridge
	wire							brg_frame_start		;
	wire							brg_frame_end		;
	wire							brg_valid			;
	wire	[  23 :  0 ]			brg_pixel			;
	wire							brg_field			;
	wire	[  12 :  0 ]			brg_width			;
	wire	[  12 :  0 ]			brg_height			;
	wire							brg_ready			;

	// Scaler
	wire							scl_frame_start		;
	wire							scl_frame_end		;
	wire							scl_valid			;
	wire	[  23 :  0 ]			scl_pixel			;
	wire							scl_field			;
	wire	[  12 :  0 ]			scl_width			;
	wire	[  12 :  0 ]			scl_height			;
	wire							scl_ready			;

	// Frame Buffer
	wire							frm_frame_start		;
	wire							frm_frame_end		;
	wire							frm_valid			;
	wire	[  11 :  0 ]			frm_pixel			;
	wire							frm_field			;
	wire	[  12 :  0 ]			frm_width			;
	wire	[  12 :  0 ]			frm_height			;
	wire							frm_ready			;
	wire	[  23 :  0 ]			rgb_pixel			;

	// Picture Combination
	wire							pc_frame_start		;
	wire							pc_frame_end		;
	wire							pc_valid			;
	wire	[  23 :  0 ]			pc_pixel			;
	wire							pc_field			;
	wire	[  12 :  0 ]			pc_width			;
	wire	[  12 :  0 ]			pc_height			;
	wire							pc_ready			;

	// EMIF
	wire							axi_clk				;
	wire							ctrl_ready			;
	wire	[  24 :  0 ]			avl_addr			;
	wire							avl_burstbegin		;
	wire	[   4 :  0 ]			avl_burstcount		;
	wire							avl_write			;
	wire	[ 255 :  0 ]			avl_wdata			;
	wire	[  31 :  0 ]			avl_byteenable		;
	wire							avl_read			;
	wire							avl_rdata_valid		;
	wire	[ 255 :  0 ]			avl_rdata			;
	wire							avl_waitrequest		;

	// VTG and HDMI
	wire							vtg_hsync			;
	wire							vtg_vsync			;
	wire							vtg_de				;
	wire	[  23 :  0 ]			vtg_pixel			;
	wire							vtg_ready			;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	// ---------------------------------------------------------------------
	// Output Assignment
	// ---------------------------------------------------------------------
	assign	FPGA_LED_O[0]	= pll_locked	;
	assign	FPGA_LED_O[1]	= ctrl_ready	;
	assign	FPGA_LED_O[2]	= 1'b0	;
	assign	FPGA_LED_O[3]	= 1'b0	;

	// ---------------------------------------------------------------------
	// Reset Sequencer for BANK R(SLVS-EC)
	// ---------------------------------------------------------------------
	reset_seq
		u_reset_seq (
			.o_src_rs_grant 								( src_rs_grant					) ,
			.i_src_rs_priority								( 8'h0							) ,
			.i_src_rs_req									( src_rs_req					) ,
			.o_pma_cu_clk									( pma_cu_clk					)
		);

	// ---------------------------------------------------------------------
	// PLL
	// ---------------------------------------------------------------------
	pll
		u_pll (
		  .refclk											( CLK_50_B6A					),
		  .rst												( 1'b0							),
		  .outclk_0											( HDMI_TX_CLK					),	// 148.5MHz
		  .locked											( pll_locked					)
	   );

	// ---------------------------------------------------------------------
	// HDMI I2C
	// ---------------------------------------------------------------------
	/*i2c_hdmi_config
		u_i2c_hdmi_config (
			.clk											( CLK_50_B5A					),
			.rst_n											( 1'b1							),
			.i2c_scl_io										( HDMI_I2C_SCL					),
			.i2c_sda_io										( HDMI_I2C_SDA					),
			.hdmi_tx_int_i									( HDMI_TX_INT					)
		);*/

	// ---------------------------------------------------------------------
	//  NiosV Sub System
	// ---------------------------------------------------------------------
	niosv_ss
		u_niosv_ss (
			.clk_clk										( CLK_100_B2B_p					) , // input  wire
			.pio_0_export									( nios_gpio_o					) , // output wire [31:0]
			.pio_1_export									( nios_gpio_i					) , // input  wire [31:0]
			.rst_n_reset_n									( ctrl_ready					) , // input  wire
			.slec_rx_csr_waitrequest						( avmm_waitrequest				) , //
			.slec_rx_csr_readdata							( avmm_readdata					) , //
			.slec_rx_csr_readdatavalid						( avmm_readdatavalid			) , //
			.slec_rx_csr_writedata							( avmm_writedata				) , //
			.slec_rx_csr_address							( avmm_addr						) , //
			.slec_rx_csr_write								( avmm_write					) , //
			.slec_rx_csr_read								( avmm_read						) , //
			.slec_rx_csr_byteenable 						( avmm_byteenable				) , //
			.spi_MISO										( SLEC_SDO						) , // input  wire
			.spi_MOSI										( slec_sdi_int					) , // output wire
			.spi_SCLK										( slec_sck_int					) , // output wire
			.spi_SS_n										( slec_xce_int					)   // output wire
		);

	assign SLEC_XVS			= nios_gpio_o[28] ;
	assign SLEC_XHS			= nios_gpio_o[28] ;
	assign SLEC_XTRIG1		= 1'b0 ;
	assign SLEC_XTRIG2		= 1'b0 ;
	assign SLEC_XCE			= ( nios_gpio_o[28] ) ? slec_xce_int : 1'b0 ;
	assign SLEC_SCK			= ( nios_gpio_o[28] ) ? slec_sck_int : 1'b0 ;
	assign SLEC_SDI			= ( nios_gpio_o[28] ) ? slec_sdi_int : 1'b0 ;
	assign SLEC_XCLR		= nios_gpio_o[31];
	assign SLEC_INCK_OE		= nios_gpio_o[30] ;
	assign SLEC_PON			= nios_gpio_o[29] ;
	assign SLEC_XMASTER		= 1'b0 ;
	assign SLEC_OMODE		= 1'b1 ;

	assign nios_gpio_i = { 31'd0 , SLEC_PGOOD };

	// ---------------------------------------------------------------------
	//  SLVS-EC
	// ---------------------------------------------------------------------
	slec3_rx_wrap
		#(
			.MAX_LANE										( MAX_LANE						) ,
			.SPC											( SPC							) ,
			.GCC											( GCC							)
		)
		u_slec (
			.phy_cdr_refclk									( SLEC_REFCLK_P					) ,
			.cpu_clk										( CLK_100_B2B_p					) ,
			.st_clk											( st_clk						) ,
			.rst_n											( ctrl_ready					) ,

			.slec_pma_cu_clk								( pma_cu_clk					),
			.slec_rs_req_o									( src_rs_req					),
			.slec_rs_grant_i								( src_rs_grant					),

			.avms_addr_i									( { 2'b00 , avmm_addr[7:2] }	) ,
			.avms_write_i									( avmm_write					) ,
			.avms_read_i									( avmm_read						) ,
			.avms_writedata_i								( avmm_writedata				) ,
			.avms_byteenable_i								( avmm_byteenable				) ,
			.avms_readdata_o								( avmm_readdata					) ,
			.avms_waitrequest_o								( avmm_waitrequest				) ,
			.avms_readdatavalid_o							( avmm_readdatavalid			) ,
			.int_o											( 								) ,

			.rx_serial_data									( SLEC_RXD_P					) ,
	//		.rx_serial_data_n								( 								) ,

			.hdr_fstart_o									( hdr_fstart					) ,
			.hdr_fend_o										( hdr_fend						) ,
			.hdr_line_valid_o								( 								) ,
			.hdr_line_num_o									( 								) ,
			.hdr_ebd_line_o									( 								) ,
			.hdr_data_id_o									( 								) ,
			.hdr_info_type_o								( 								) ,
			.hdr_info_o										( 								) ,
			.hdr_corrupt_o									( 								) ,

			.pixel_bit_o									( 								) ,
			.valid_line_start_o								( valid_line_start				) ,
			.valid_line_end_o								( valid_line_end				) ,
			.ebd_line_start_o								( 								) ,
			.ebd_line_end_o									( 								) ,
			.blank_line_start_o								( 								) ,
			.blank_line_end_o								( 								) ,
			.hc_line_start_o								( 								) ,
			.hc_line_end_o									( 								) ,
			.line_len_o										( 								) ,
			.pixel_data_o									( pixel_data					) ,
			.pixel_valid_o									( pixel_valid					) ,
			.truncated_packet_o								( 								) ,
			.crc_err_detect_o								( 								) ,
			.crc_no_error_o									( 								) ,
			.ecc_err_correct_o								( 								) ,
			.ecc_err_uncorrect_o							( 								)
		) ;

	// ---------------------------------------------------------------------
	// User Interface bridge
	// ---------------------------------------------------------------------
	slec_rx_vsi_bridge
		#(
			.Q_BIT											( 12							),
			.H_PIXEL										( 2472							),
			.V_LINE											( 2128							)
		)
		u_slec_rx_vsi_bridge (
			.inclk											( st_clk						),
			.outclk											( axi_clk						),
			.rst_inclk_n									( ctrl_ready					),
			.rst_outclk_n									( ctrl_ready					),

			.frame_start_i									( hdr_fstart					),
			.frame_end_i									( hdr_fend						),
			.valid_line_start_i								( valid_line_start				),
			.valid_line_end_i								( valid_line_end				),
			.pixel_valid_i									( pixel_valid					),
			.pixel_data_i									( pixel_data					),

			.frame_start_o									( brg_frame_start				),
			.frame_end_o									( brg_frame_end					),
			.valid_o										( brg_valid						),
			.pixel_o										( brg_pixel						),
			.field_o										( brg_field						),
			.width_o										( brg_width						),
			.height_o										( brg_height					),
			.ready_i										( scl_ready						)
		) ;

	// ---------------------------------------------------------------------
	// Scaler
	// ---------------------------------------------------------------------
	scl16_top
		#(
			.PPC											( 2								) ,
			.Q_BIT											( 12							) ,
			.PLANE											( 1								) ,
			.SIZE_BIT										( 13							)
		)
		u_scl16_top (
			.clk											( axi_clk						) ,
			.rst_n											( ctrl_ready					) ,
			.srst											( ~ctrl_ready					) ,
			.enable											( 1'b1							) ,

			.param_update_i									( 1'b1							) ,

			.table_sel_i									( 2'b11							) ,
			.coeff_sel_i									( 2'b00							) ,
			.smpl_mode_i									( 1'b1							) ,
			.scan_mode_i									( 1'b1							) ,

			.width_i										( brg_width						) ,
			.height_i										( brg_height					) ,
			.tgt_wdt_i										( 13'd1254						) ,
			.tgt_hgt_i										( 13'd1080						) ,

			.width_o										( scl_width						) ,
			.height_o										( scl_height					) ,

			.frame_start_i									( brg_frame_start				) ,
			.frame_end_i									( brg_frame_end					) ,
			.valid_i										( brg_valid						) ,
			.pixel_i										( brg_pixel						) ,
			.field_i										( brg_field						) ,
			.ready_o										( scl_ready						) ,

			.frame_start_o									( scl_frame_start				) ,
			.frame_end_o									( scl_frame_end					) ,
			.valid_o										( scl_valid						) ,
			.pixel_o										( scl_pixel						) ,
			.field_o										( scl_field						) ,
			.ready_i										( frm_ready						)
		) ;

	// ---------------------------------------------------------------------
	// Frame Buffer
	// ---------------------------------------------------------------------
	sfb16_top
		#(
			.IN_PPC											( 2								),
			.OUT_PPC										( 1								),
			.Q_BIT											( 12							),
			.PLANE											( 1								),
			.SIZE_BIT										( 13							),
			.BASE_ADR										( 32'h0000_0000					),
			.STEP_ADR										( 32'h0080_0000					),
			.LDATA_BIT										( 256							),
			.LADR_BIT										( 25							),
			.MODE											( 0								)
		)
		u_sfb16_top (
			.iclk											( axi_clk						),
			.mclk											( axi_clk						),
			.oclk											( HDMI_TX_CLK					),
			.rst_n											( pll_locked					),
			.srst_iclk										( ~pll_locked					),
			.srst_mclk										( ~pll_locked					),
			.srst_oclk										( ~pll_locked					),
			.enable											( 1'b1							),
			.param_update_i									( 1'b1							),

			.scan_mode_i									( 1'b0							), // Progressive Scan Mode
			.rd_mode_o										( 								), // Progressive Scan Mode

			.frame_start_i									( scl_frame_start				),
			.frame_end_i									( scl_frame_end					),
			.valid_i										( scl_valid						),
			.pixel_i										( scl_pixel						),
			.field_i										( scl_field						),
			.width_i										( scl_width						),
			.height_i										( scl_height					),
			.ready_o										( frm_ready						),

			.frame_start_o									( frm_frame_start				),
			.frame_end_o									( frm_frame_end					),
			.valid_o										( frm_valid						),
			.pixel_o										( frm_pixel						),
			.field_o										( frm_field						),
			.width_o										( frm_width						),
			.height_o										( frm_height					),
			.ready_i										( pc_ready						),

			.avmm_address_o									( avl_addr						),
			.avmm_begintrans_o								( 								),
			.avmm_burstcount_o								( avl_burstcount				),
			.avmm_byteenable_o								( avl_byteenable				),
			.avmm_write_o									( avl_write						),
			.avmm_writedata_o								( avl_wdata						),
			.avmm_read_o									( avl_read						),
			.avmm_readdatavalid_i							( avl_rdata_valid				),
			.avmm_readdata_i								( avl_rdata						),
			.avmm_waitrequest_i								( avl_waitrequest				)
		) ;

	assign rgb_pixel	= { frm_pixel[11:4] , frm_pixel[11:4] , frm_pixel[11:4] } ;

	// ---------------------------------------------------------------------
	// EMIF
	// ---------------------------------------------------------------------
	ed_synth
		ed_synth_Inst(
			.clk_out_clk										( axi_clk						) ,
			.reset_out_reset_n									( ctrl_ready					) ,
			.mm_ccb_0_s0_waitrequest							( avl_waitrequest				) ,
			.mm_ccb_0_s0_readdata								( avl_rdata						) ,
			.mm_ccb_0_s0_readdatavalid							( avl_rdata_valid				) ,
			.mm_ccb_0_s0_burstcount								( avl_burstcount				) ,
			.mm_ccb_0_s0_writedata								( avl_wdata						) ,
			.mm_ccb_0_s0_address								( { 7'd0 , avl_addr }			) ,
			.mm_ccb_0_s0_write									( avl_write						) ,
			.mm_ccb_0_s0_read									( avl_read						) ,
			.mm_ccb_0_s0_byteenable								( avl_byteenable				) ,
			.mm_ccb_0_s0_debugaccess							( 1'b0							) ,
			.ed_synth_emif_io96b_lpddr4_mem_mem_cs				( LPDDR4B_CS_n					) ,
			.ed_synth_emif_io96b_lpddr4_mem_mem_ca				( LPDDR4B_CA					) ,
			.ed_synth_emif_io96b_lpddr4_mem_mem_cke				( LPDDR4B_CKE					) ,
			.ed_synth_emif_io96b_lpddr4_mem_mem_dq				( LPDDR4B_DQ					) ,
			.ed_synth_emif_io96b_lpddr4_mem_mem_dqs_t			( LPDDR4B_DQS					) ,
			.ed_synth_emif_io96b_lpddr4_mem_mem_dqs_c			( LPDDR4B_DQS_n					) ,
			.ed_synth_emif_io96b_lpddr4_mem_mem_dmi				( LPDDR4B_DM					) ,
			.ed_synth_emif_io96b_lpddr4_mem_ck_mem_ck_t			( LPDDR4B_CK					) ,
			.ed_synth_emif_io96b_lpddr4_mem_ck_mem_ck_c			( LPDDR4B_CK_n					) ,
			.ed_synth_emif_io96b_lpddr4_mem_reset_n_mem_reset_n	( LPDDR4B_RESET_n				) ,
			.ed_synth_emif_io96b_lpddr4_oct_oct_rzqin			( LPDDR4B_RZQ					) ,
			.ed_synth_emif_io96b_lpddr4_ref_clk_clk				( LPDDR4B_REFCLK_p				) ,
			.ref_clk_usr_pll_clk								( CLK_100_B2B_p					)
		);

	// ---------------------------------------------------------------------
	// Picture Combination
	// ---------------------------------------------------------------------
	pc11_top
		#(
			.Q_BIT											( 8								),
			.SIZE_BIT										( 13							)
		)
		u_pc11_top (
			.clk											( HDMI_TX_CLK					) ,
			.rst_n											( pll_locked					) ,
			.srst											( ~pll_locked					) ,
			.enable											( 1'b1							) ,
			.param_update_i									( 1'b1							) ,

			.smpl_mode_i									( 1'b1							) ,
			.constant_data_i								( 24'h0							) ,
			.layer3_con_i									( 1'b0							) ,
			.layer3_cdata_i									( 24'hFF_FFFF					) ,

			.width_i										( 13'd1920						) ,
			.height_i										( 13'd1080						) ,

			.layer0_ch_i									( 2'b00							) ,
			.layer1_ch_i									( 2'b01							) ,
			.layer2_ch_i									( 2'b10							) ,
			.layer3_ch_i									( 2'b11							) ,

			.ch0_on_i										( 1'b1							) ,
			.ch0_alpha_i									( 8'hFF							) ,
			.ch0_hpos_i										( 13'd333						) ,
			.ch0_vpos_i										( 13'd000						) ,

			.ch1_on_i										( 1'b0							) ,
			.ch1_alpha_i									( 8'hFF							) ,
			.ch1_hpos_i										( 13'h000						) ,
			.ch1_vpos_i										( 13'h000						) ,

			.ch2_on_i										( 1'b0							) ,
			.ch2_alpha_i									( 8'hFF							) ,
			.ch2_hpos_i										( 13'h000						) ,
			.ch2_vpos_i										( 13'h000						) ,

			.ch3_on_i										( 1'b0							) ,
			.ch3_alpha_i									( 8'hFF							) ,
			.ch3_hpos_i										( 13'h000						) ,
			.ch3_vpos_i										( 13'h000						) ,

			.ch0_valid_i									( frm_valid						) ,
			.ch0_frame_start_i								( frm_frame_start				) ,
			.ch0_frame_end_i								( frm_frame_end					) ,
			.ch0_pixel_i									( rgb_pixel						) ,
			.ch0_field_i									( frm_field						) ,
			.ch0_width_i									( frm_width						) ,
			.ch0_height_i									( frm_height					) ,
			.ch0_ready_o									( pc_ready						) ,

			.ch1_valid_i									( 1'b0							) ,
			.ch1_frame_start_i								( 1'b0							) ,
			.ch1_frame_end_i								( 1'b0							) ,
			.ch1_pixel_i									( 24'd0							) ,
			.ch1_field_i									( 1'b0							) ,
			.ch1_width_i									( 13'd0							) ,
			.ch1_height_i									( 13'd0							) ,
			.ch1_ready_o									( 								) ,

			.ch2_valid_i									( 1'b0							) ,
			.ch2_frame_start_i								( 1'b0							) ,
			.ch2_frame_end_i								( 1'b0							) ,
			.ch2_pixel_i									( 24'd0							) ,
			.ch2_field_i									( 1'b0							) ,
			.ch2_width_i									( 13'd0							) ,
			.ch2_height_i									( 13'd0							) ,
			.ch2_ready_o									( 								) ,

			.ch3_valid_i									( 1'b0							) ,
			.ch3_frame_start_i								( 1'b0							) ,
			.ch3_frame_end_i								( 1'b0							) ,
			.ch3_pixel_i									( 24'd0							) ,
			.ch3_field_i									( 1'b0							) ,
			.ch3_width_i									( 13'd0							) ,
			.ch3_height_i									( 13'd0							) ,
			.ch3_ready_o									( 								) ,

			.valid_o										( pc_valid						) ,
			.frame_start_o									( pc_frame_start				) ,
			.frame_end_o									( pc_frame_end					) ,
			.pixel_o										( pc_pixel						) ,
			.field_o										( pc_field						) ,
			.width_o										( pc_width						) ,
			.height_o										( pc_height						) ,
			.ready_i										( vtg_ready						)
		);

	// ---------------------------------------------------------------------
	// Video Timing Generation
	// ---------------------------------------------------------------------
	vtg11_top
		#(
			.Q_BIT											( 8								),
			.PLANE											( 3								),
			.FIFO_DEPTH										( 10							),
			.VSI_FMIN										( 148							),
			.VIDEO_FMAX										( 149							)
		) u_vtg11_top(
			.vsi_clk 										( HDMI_TX_CLK					),
			.out_clk										( HDMI_TX_CLK					),
			.rst_n											( pll_locked					),
			.vsi_srst										( ~pll_locked					),
			.out_srst										( ~pll_locked					),

			.enable											( 1'b1							),
			.param_update_i									( 1'b1							),

			.interlace_mode_i								( 1'b0							),

			.ex_sync_pol_i									( 1'b0							),
			.ex_odd_field_pol_i								( 1'b0							),
			.h_sync_pol_i									( 1'b1							),
			.v_sync_pol_i									( 1'b1							),
			.odd_field_pol_i 								( 1'b0							),
			.h_valid_pol_i									( 1'b0							),
			.v_valid_pol_i									( 1'b0							),
			.de_pol_i										( 1'b1							),

			.h_front_i										( 9'd88							), // H Front Porch + H Right Border
			.h_interval_i									( 8'd44							), // H Sync Time
			.h_back_i										( 9'd148						), // H Back Porch + H Right Border
			.h_video_i										( 12'd1920						), // H Pixels
			.v_front_i										( 6'd4							), // V Front Porch + V Bottom Border
			.v_interval_i									( 4'd5							), // V Sync Time
			.v_back_i										( 7'd36							), // V Back Porch + V Top Border
			.v_video_i										( 12'd1080						), // V Pixels
			.v_delay_i										( 12'd88						), // Same as H Front Porch

			.even_v_front_i									( 6'h00							),
			.even_v_interval_i								( 4'h0							),
			.even_v_back_i									( 7'h00							),
			.even_v_video_i									( 12'h000						),
			.even_v_delay_i									( 12'h000						),
			.even2odd_v_delay_i								( 12'h000						),
			.even2odd_h_delay_i								( 12'h000						),
			.odd2even_v_delay_i								( 12'h000						),
			.odd2even_h_delay_i								( 12'h000						),

			.const_data_out_i								( 1'b0							),
			.const_out_mode_i								( 1'b0							),
			.const_data0_i									( 24'h50_5050					),
			.const_data1_i									( 24'hA0_A0A0					),

			.external_sync_i 								( 1'b0							),
			.external_field_i								( 1'b0							),

			.frame_start_i									( pc_frame_start				),
			.frame_end_i									( pc_frame_end					),
			.valid_i										( pc_valid						),
			.pixel_i										( pc_pixel						),
			.field_i										( pc_field						),
//			.width_i										( pc_width						),
//			.height_i										( pc_height						),
			.ready_o										( vtg_ready						),

			.h_sync_o										( HDMI_TX_HS					),
			.v_sync_o										( HDMI_TX_VS					),
			.field_o 										( 								),
			.de_o											( HDMI_TX_DE					),
			.h_valid_o										( 								),
			.v_valid_o										( 								),
			.pixel_o										( HDMI_TX_D						)
		) ;

endmodule
`default_nettype wire