//----------------------------------------------------------------------------------------
//Copyright (C) 2025 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		SLVS-EC v3.0 RX IP Wrapper Module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 May 28 2025		: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		MAX_LANE				: Number of lanes
//		SPC						: Number of pixels output in parallel
//		GCC						: Physical Layer Coding
//									1 : GCC
//									0 : 8b10b
//----------------------------------------------------------------------------------------
// I/O PORTS
//		phy_cdr_refclk			: clock for RX PMA
//		cpu_clk					: clock for CPU Interface
//		st_clk					: clock for image streaming
//		rst_n					: asynchronous reset
//
//		slec_pma_cu_clk			: PMA control unit clock source
//		slec_rs_req_o			: Grant signal from GTS Reset Sequencer
//		slec_rs_grant_i			: Request signal from GTS Reset Sequencer
//
//		avms_addr_i				: Avalon-MM Slave I/F address
//		avms_write_i			: Avalon-MM Slave I/F write request
//		avms_read_i				: Avalon-MM Slave I/F read request
//		avms_writedata_i		: Avalon-MM Slave I/F writedata
//		avms_byteenable_i		: Avalon-MM Slave I/F byteenable
//		avms_readdata_o			: Avalon-MM Slave I/F readdata
//		avms_waitrequest_o		: Avalon-MM Slave I/F waitrequest
//		avms_readdatavalid_o	: Avalon-MM Slave I/F readdatavalid
//		int_o					: Level-Interrupt for CPU
//
//		rx_serial_data			: RX serial data port
//
//		hdr_fstart_o			: Frame Start bit of the Header
//		hdr_fend_o				: Frame End bit of the Header
//		hdr_line_valid_o		: Line Valid bit of the Header
//		hdr_line_num_o			: Line Number Field of the Header
//		hdr_ebd_line_o			: EBD Line bit of the Header
//		hdr_data_id_o			: Data ID Field of the Header
//		hdr_info_type_o			: Header Info Type Filed of the Header
//		hdr_info_o				: bit[39:16] of the Header
//		hdr_corrupt_o			: Packet header corruption indicator
//
//		pixel_bit_o				: PIXEL_BIT field of CONFIG register
//		valid_line_start_o		: Valid line start pulse
//		valid_line_end_o		: Valid line end pulse
//		ebd_line_start_o		: Embedded data line start pulse
//		ebd_line_end_o			: Embedded data line end pulse
//		blank_line_start_o		: Blank line data start pulse
//		blank_line_end_o		: Blank line data end pulse
//		hc_line_start_o			: Header corrupted line data start pulse
//		hc_line_end_o			: Header corrupted line data end pulse
//		line_len_o				: Number of pixels in the line
//		pixel_data_o			: 64/32 parallel pixel I/F
//      pixel_valid_o			: Indicate valid of each pixel
//		truncated_packet_o		: Truncated packet reception indicator pulse
//		crc_err_detect_o		: Error on footer CRC calculation indicator pulse
//		crc_no_error_o			: Error on footer CRC calculation indicator pulse
//		ecc_err_correct_o		: Payload error is detected and corrected by ECC function
//		ecc_err_uncorrect_o		: Uncorrectable error is detected by ECC function
//----------------------------------------------------------------------------------------
`default_nettype none
`timescale 1ps / 1ps

module slec3_rx_wrap
	#(
		parameter									MAX_LANE	= 8			,
		parameter									SPC			= 32		,
		parameter									GCC			= 0
	)
	(
	//--- Clock and Reset
		input	wire	[              1 : 0 ]		phy_cdr_refclk			,
		input	wire								cpu_clk					,
		output	wire								st_clk					,
		input	wire								rst_n					,

		input	wire	[              1 : 0 ]		slec_pma_cu_clk			,
		output	wire	[   MAX_LANE - 1 : 0 ]		slec_rs_req_o			,
		input	wire	[   MAX_LANE - 1 : 0 ]		slec_rs_grant_i			,

	//--- AvalonMM Interface
		input	wire	[            7 : 0 ]		avms_addr_i				,
		input	wire								avms_write_i			,
		input	wire								avms_read_i				,
		input	wire	[           31 : 0 ]		avms_writedata_i		,
		input	wire	[            3 : 0 ]		avms_byteenable_i		,
		output	wire	[           31 : 0 ]		avms_readdata_o			,
		output	wire								avms_waitrequest_o		,
		output	wire								avms_readdatavalid_o	,
		output	wire								int_o					,

	//--- Transceiver Interface
		input	wire	[ MAX_LANE - 1 : 0 ]		rx_serial_data			,
//		input	wire	[ MAX_LANE - 1 : 0 ]		rx_serial_data_n		,

	//Packet Header RAW Information Interface
		output	wire								hdr_fstart_o			,
		output	wire								hdr_fend_o				,
		output	wire								hdr_line_valid_o		,
		output	wire	[           12 : 0 ]		hdr_line_num_o			,
		output	wire								hdr_ebd_line_o			,
		output	wire	[            3 : 0 ]		hdr_data_id_o			,
		output	wire	[            2 : 0 ]		hdr_info_type_o			,
		output	wire	[           23 : 0 ]		hdr_info_o				,
		output	wire								hdr_corrupt_o			,

	// Pixel Data Interface
		output	wire	[            2 : 0 ]		pixel_bit_o				,
		output	wire								valid_line_start_o		,
		output	wire								valid_line_end_o		,
		output	wire								ebd_line_start_o		,
		output	wire								ebd_line_end_o			,
		output	wire								blank_line_start_o		,
		output	wire								blank_line_end_o		,
		output	wire								hc_line_start_o			,
		output	wire								hc_line_end_o			,
		output	wire	[           15 : 0 ]		line_len_o				,
		output	wire	[   16*SPC - 1 : 0 ]		pixel_data_o			,
		output	wire	[    SPC/4 - 1 : 0 ]		pixel_valid_o			,
		output	wire								truncated_packet_o		,
		output	wire								crc_err_detect_o		,
		output	wire								crc_no_error_o			,
		output	wire								ecc_err_correct_o		,
		output	wire								ecc_err_uncorrect_o
	) ;


// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	genvar	    						i							;

	// Transceiver
	wire	[       MAX_LANE - 1 : 0 ]	rx_clkout					;
	wire	[       MAX_LANE - 1 : 0 ]	rx_ready					;
	wire	[       MAX_LANE - 1 : 0 ]	rx_is_lockedtodata			;
	wire	[  80 * MAX_LANE - 1 : 0 ]	rx_parallel_data_fr_pma		;
	wire	[  40 * MAX_LANE - 1 : 0 ]	rx_parallel_data			;

	reg		[                  1 : 0 ]	st_rst_n_ff					;
	reg		[                  1 : 0 ]	cpu_rst_n_ff				;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	//-------------------------------------------------
	// Clock Output
	//-------------------------------------------------
	assign st_clk	= rx_clkout[0] ;

	// ---------------------------------------------------------------------
	// Transceiver
	// ---------------------------------------------------------------------
	xcvr_rx
		u_xcvr_rx_0(
			.i_rx_cdr_refclk_p			( phy_cdr_refclk[0]					),
			.i_pma_cu_clk				( slec_pma_cu_clk[0]				),
			.i_rx_reset					( ~rst_n	 						),
			.o_rx_reset_ack				( 									),
			.o_rx_ready					( rx_ready[0]						),
			.i_rx_coreclkin				( rx_clkout[0]						),
			.o_rx_clkout				( rx_clkout[0]						),
			.i_src_rs_grant				( slec_rs_grant_i[0]				),
			.o_src_rs_req				( slec_rs_req_o[0]					),
			.i_rx_serial_data			( rx_serial_data[0]					),
			.i_rx_serial_data_n			( 									),
			.o_rx_is_lockedtodata		( rx_is_lockedtodata[0]				),
			.o_rx_is_lockedtoref		( 									),
			.o_rx_parallel_data 		( rx_parallel_data_fr_pma[79:0]		)
		);

	xcvr_rx
		u_xcvr_rx_1(
			.i_rx_cdr_refclk_p			( phy_cdr_refclk[0]					),
			.i_pma_cu_clk				( slec_pma_cu_clk[0]				),
			.i_rx_reset					( ~rst_n	 						),
			.o_rx_reset_ack				( 									),
			.o_rx_ready					( rx_ready[1]						),
			.i_rx_coreclkin				( rx_clkout[0]						),
			.o_rx_clkout				( rx_clkout[1]						),
			.i_src_rs_grant				( slec_rs_grant_i[1]				),
			.o_src_rs_req				( slec_rs_req_o[1]					),
			.i_rx_serial_data			( rx_serial_data[1]					),
			.i_rx_serial_data_n			( 									),
			.o_rx_is_lockedtodata		( rx_is_lockedtodata[1]				),
			.o_rx_is_lockedtoref		( 									),
			.o_rx_parallel_data 		( rx_parallel_data_fr_pma[159:80]	)
		);

	xcvr_rx
		u_xcvr_rx_2(
			.i_rx_cdr_refclk_p			( phy_cdr_refclk[0]					),
			.i_pma_cu_clk				( slec_pma_cu_clk[0]				),
			.i_rx_reset					( ~rst_n	 						),
			.o_rx_reset_ack				( 									),
			.o_rx_ready					( rx_ready[2]						),
			.i_rx_coreclkin				( rx_clkout[0]						),
			.o_rx_clkout				( rx_clkout[2]						),
			.i_src_rs_grant				( slec_rs_grant_i[2]				),
			.o_src_rs_req				( slec_rs_req_o[2]					),
			.i_rx_serial_data			( rx_serial_data[2]					),
			.i_rx_serial_data_n			( 									),
			.o_rx_is_lockedtodata		( rx_is_lockedtodata[2]				),
			.o_rx_is_lockedtoref		( 									),
			.o_rx_parallel_data 		( rx_parallel_data_fr_pma[239:160]	)
		);

	xcvr_rx
		u_xcvr_rx_3(
			.i_rx_cdr_refclk_p			( phy_cdr_refclk[0]					),
			.i_pma_cu_clk				( slec_pma_cu_clk[0]				),
			.i_rx_reset					( ~rst_n	 						),
			.o_rx_reset_ack				( 									),
			.o_rx_ready					( rx_ready[3]						),
			.i_rx_coreclkin				( rx_clkout[0]						),
			.o_rx_clkout				( rx_clkout[3]						),
			.i_src_rs_grant				( slec_rs_grant_i[3]				),
			.o_src_rs_req				( slec_rs_req_o[3]					),
			.i_rx_serial_data			( rx_serial_data[3]					),
			.i_rx_serial_data_n			( 									),
			.o_rx_is_lockedtodata		( rx_is_lockedtodata[3]				),
			.o_rx_is_lockedtoref		( 									),
			.o_rx_parallel_data 		( rx_parallel_data_fr_pma[319:240]	)
		);

	xcvr_rx
		u_xcvr_rx_4(
			.i_rx_cdr_refclk_p			( phy_cdr_refclk[1]					),
			.i_pma_cu_clk				( slec_pma_cu_clk[1]				),
			.i_rx_reset					( ~rst_n	 						),
			.o_rx_reset_ack				( 									),
			.o_rx_ready					( rx_ready[4]						),
			.i_rx_coreclkin				( rx_clkout[0]						),
			.o_rx_clkout				( rx_clkout[4]						),
			.i_src_rs_grant				( slec_rs_grant_i[4]				),
			.o_src_rs_req				( slec_rs_req_o[4]					),
			.i_rx_serial_data			( rx_serial_data[4]					),
			.i_rx_serial_data_n			( 									),
			.o_rx_is_lockedtodata		( rx_is_lockedtodata[4]				),
			.o_rx_is_lockedtoref		( 									),
			.o_rx_parallel_data			( rx_parallel_data_fr_pma[399:320]	)
		);

	xcvr_rx
		u_xcvr_rx_5(
			.i_rx_cdr_refclk_p			( phy_cdr_refclk[1]					),
			.i_pma_cu_clk				( slec_pma_cu_clk[1]				),
			.i_rx_reset					( ~rst_n	 						),
			.o_rx_reset_ack				( 									),
			.o_rx_ready					( rx_ready[5]						),
			.i_rx_coreclkin				( rx_clkout[0]						),
			.o_rx_clkout				( rx_clkout[5]						),
			.i_src_rs_grant				( slec_rs_grant_i[5]				),
			.o_src_rs_req				( slec_rs_req_o[5]					),
			.i_rx_serial_data			( rx_serial_data[5]					),
			.i_rx_serial_data_n			( 									),
			.o_rx_is_lockedtodata		( rx_is_lockedtodata[5]				),
			.o_rx_is_lockedtoref		( 									),
			.o_rx_parallel_data			( rx_parallel_data_fr_pma[479:400]	)
		);

	xcvr_rx
		u_xcvr_rx_6(
			.i_rx_cdr_refclk_p			( phy_cdr_refclk[1]					),
			.i_pma_cu_clk				( slec_pma_cu_clk[1]				),
			.i_rx_reset					( ~rst_n	 						),
			.o_rx_reset_ack				( 									),
			.o_rx_ready					( rx_ready[6]						),
			.i_rx_coreclkin				( rx_clkout[0]						),
			.o_rx_clkout				( rx_clkout[6]						),
			.i_src_rs_grant				( slec_rs_grant_i[6]				),
			.o_src_rs_req				( slec_rs_req_o[6]					),
			.i_rx_serial_data			( rx_serial_data[6]					),
			.i_rx_serial_data_n			( 									),
			.o_rx_is_lockedtodata		( rx_is_lockedtodata[6]				),
			.o_rx_is_lockedtoref		( 									),
			.o_rx_parallel_data			( rx_parallel_data_fr_pma[559:480]	)
		);

	xcvr_rx
		u_xcvr_rx_7(
			.i_rx_cdr_refclk_p			( phy_cdr_refclk[1]					),
			.i_pma_cu_clk				( slec_pma_cu_clk[1]				),
			.i_rx_reset					( ~rst_n	 						),
			.o_rx_reset_ack				( 									),
			.o_rx_ready					( rx_ready[7]						),
			.i_rx_coreclkin				( rx_clkout[0]						),
			.o_rx_clkout				( rx_clkout[7]						),
			.i_src_rs_grant				( slec_rs_grant_i[7]				),
			.o_src_rs_req				( slec_rs_req_o[7]					),
			.i_rx_serial_data			( rx_serial_data[7]					),
			.i_rx_serial_data_n			( 									),
			.o_rx_is_lockedtodata		( rx_is_lockedtodata[7]				),
			.o_rx_is_lockedtoref		( 									),
			.o_rx_parallel_data			( rx_parallel_data_fr_pma[639:560]	)
		);

	//-------------------------------------------------
	// Macnica IP
	//-------------------------------------------------
	always @( posedge st_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			st_rst_n_ff[ 1 : 0 ]	<= 2'b00 ;
		end
		else begin
			st_rst_n_ff[ 1 : 0 ]	<= { st_rst_n_ff[ 0 ] , 1'b1 } ;
		end
	end

	always @( posedge cpu_clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			cpu_rst_n_ff[ 1 : 0 ]	<= 2'b00 ;
		end
		else begin
			cpu_rst_n_ff[ 1 : 0 ]	<= { cpu_rst_n_ff[ 0 ] , 1'b1 } ;
		end
	end

	generate
		for ( i = 0 ; i < MAX_LANE ; i = i + 1 ) begin : rx_parallel_data_gen
			assign	rx_parallel_data[i*40   +:20]	= rx_parallel_data_fr_pma[i*80   +:20];
			assign	rx_parallel_data[i*40+20+:20]	= rx_parallel_data_fr_pma[i*80+40+:20];
		end
	endgenerate

	slec3_rx_top
		#(
			.MAX_LANE					( MAX_LANE							),
			.SPC						( SPC								),
			.GCC						( GCC								)
		)
		u_slec3_rx_top (
			.pma_clk					( st_clk							),
			.st_clk						( st_clk							),
			.cpu_clk					( cpu_clk							),
			.st_rst_n					( st_rst_n_ff[1]					),
			.pma_rst_n					( st_rst_n_ff[1]					),
			.cpu_rst_n					( cpu_rst_n_ff[1]					),
			.avms_addr_i				( avms_addr_i						),
			.avms_write_i				( avms_write_i						),
			.avms_read_i				( avms_read_i						),
			.avms_writedata_i			( avms_writedata_i					),
			.avms_byteenable_i			( avms_byteenable_i					),
			.avms_readdata_o			( avms_readdata_o					),
			.avms_waitrequest_o			( avms_waitrequest_o				),
			.avms_readdatavalid_o		( avms_readdatavalid_o				),
			.int_o						( int_o								),
			.data_i						( rx_parallel_data					),
			.gcc_block_aligned_o		( 									),
			.gcc_parity_correct_o		( 									),
			.gcc_parity_err_o			( 									),
			.gcc_deskew_locked_o		( 									),
			.a8b10b_syncstatus_o		( 									),
			.a8b10b_patterndetect_o 	( 									),
			.a8b10b_disp_err_o			( 									),
			.a8b10b_code_err_o			( 									),
			.a8b10b_deskew_locked_o 	( 									),
			.pixel_bit_o				( pixel_bit_o						),
			.hdr_fstart_o				( hdr_fstart_o						),
			.hdr_fend_o					( hdr_fend_o						),
			.hdr_line_valid_o			( hdr_line_valid_o					),
			.hdr_line_num_o				( hdr_line_num_o					),
			.hdr_ebd_line_o				( hdr_ebd_line_o					),
			.hdr_data_id_o				( hdr_data_id_o						),
			.hdr_info_type_o			( hdr_info_type_o					),
			.hdr_info_o					( hdr_info_o						),
			.hdr_corrupt_o				( hdr_corrupt_o						),
			.valid_line_start_o			( valid_line_start_o				),
			.valid_line_end_o			( valid_line_end_o					),
			.ebd_line_start_o			( ebd_line_start_o					),
			.ebd_line_end_o				( ebd_line_end_o					),
			.blank_line_start_o			( blank_line_start_o				),
			.blank_line_end_o			( blank_line_end_o					),
			.hc_line_start_o			( hc_line_start_o					),
			.hc_line_end_o				( hc_line_end_o						),
			.line_len_o					( line_len_o						),
			.pixel_data_o				( pixel_data_o						),
			.pixel_valid_o				( pixel_valid_o						),
			.truncated_packet_o			( truncated_packet_o				),
			.crc_err_detect_o			( crc_err_detect_o					),
			.crc_no_error_o				( crc_no_error_o					),
			.ecc_err_correct_o			( ecc_err_correct_o					),
			.ecc_err_uncorrect_o		( ecc_err_uncorrect_o				)
	) ;

endmodule
`default_nettype wire