//----------------------------------------------------------------------------------------
//Copyright (C) 2014 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		FIFO Write Module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Dec.  1 2014	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		DATA_BIT		: data bit width
//		SIZE_BIT		: size setting signal bit width
//----------------------------------------------------------------------------------------
// I/O PORTS
//		iclk			: write clock
//		mclk			: read clock
//		rst_n			: asynchronous reset
//		srst_iclk		: software reset (sync with iclk)
//		srst_mclk		: software reset (sync with mclk)
//
//		fs_iclk_i		: frame start of input data
//		fe_iclk_i		: frame end of input data
//		fld_iclk_i		: field status of input data  1:even field  0:odd field
//		width_iclk_i	: width of input data
//		height_iclk_i	: height of input data
//		pix_lack_iclk_i	: indicates input data pixel missing
//
//		fs_mclk_o		: frame start of output data
//		fe_mclk_o		: frame end of output data
//		fld_mclk_o		: field status of output data  1:even field	0:odd field
//		width_mclk_o	: width of output data
//		height_mclk_o	: height of output data
//		pix_lack_mclk_o	: indicates output data pixel missing
//		trans_done_i	: transaction end
//
//		a_full_o		: fifo almost full flag
//		aempty_o		: fifo almost empty flag
//		empty_o			: fifo empty flag
//
//		wen_i			: fifo write enable
//		wdata_i			: fifo write data
//		ren_i			: fifo read enable
//		rdata_o			: fifo read data
//----------------------------------------------------------------------------------------
`default_nettype none
`timescale 1ns / 1ps

module	sfb16_wfifo (
	// global signal
	iclk			,
	mclk			,
	rst_n			,
	srst_iclk		,
	srst_mclk		,

	// iclk syncronous signal
	fs_iclk_i		,
	fe_iclk_i		,
	fld_iclk_i		,
	width_iclk_i	,
	height_iclk_i	,
	pix_lack_iclk_i	,

	// mclk syncronous signal
	fs_mclk_o		,
	fe_mclk_o		,
	fld_mclk_o		,
	width_mclk_o	,
	height_mclk_o	,
	pix_lack_mclk_o	,
	trans_done_i	,

	// fifo flag
	a_full_o		,
	a_empty_o		,
	empty_o			,

	// fifo data i/f
	wen_i			,
	wdata_i			,
	ren_i			,
	rdata_o
) ;


// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------

	parameter	DATA_BIT	= 64	;
	parameter	SIZE_BIT	= 11	;
	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

	localparam	ADR_BIT		= 6		;
	localparam	WATERMARK_R	= 31	;

// =============================================================================
// PORT DECLARATION
// =============================================================================

	// global signal
	input	wire						iclk			;
	input	wire						mclk			;
	input	wire						rst_n			;
	input	wire						srst_iclk		;
	input	wire						srst_mclk		;

	// iclk syncronous signal
	input	wire						fs_iclk_i		;
	input	wire						fe_iclk_i		;
	input	wire						fld_iclk_i		;
	input	wire	[  SIZE_BIT-1 : 0 ]	width_iclk_i	;
	input	wire	[  SIZE_BIT-1 : 0 ]	height_iclk_i	;
	input	wire						pix_lack_iclk_i	;

	// mclk syncronous signal
	output	wire						fs_mclk_o		;
	output	wire						fe_mclk_o		;
	output	wire						fld_mclk_o		;
	output	wire	[  SIZE_BIT-1 : 0 ]	width_mclk_o	;
	output	wire	[  SIZE_BIT-1 : 0 ]	height_mclk_o	;
	output	wire						pix_lack_mclk_o	;
	input	wire						trans_done_i	;

	// fifo flag
	output	wire						a_full_o		;
	output	wire						a_empty_o		;
	output	wire						empty_o			;

	// fifo data i/f
	input	wire						wen_i			;
	input	wire	[ DATA_BIT-1 : 0 ]	wdata_i			;
	input	wire						ren_i			;
	output	wire	[ DATA_BIT-1 : 0 ]	rdata_o			;


// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg								fe_lat_ff		;
	wire							fe_redg			;
	reg		[          2 : 0 ]		tdone_lat_ff	;
	wire							tdone_redg		;

	wire							a_full			;
	reg								a_full_ff		;
	reg								a_full_int_ff	;

	wire							ram_wen			;
	wire							ram_ren			;
	wire	[  ADR_BIT-1 : 0 ]		ram_wadr		;
	wire	[  ADR_BIT-1 : 0 ]		ram_radr		;
	wire	[ DATA_BIT-1 : 0 ]		ram_wdata		;
	wire	[ DATA_BIT-1 : 0 ]		ram_rdata		;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	//===============================//
	// put change clock iclk -> mclk //
	//===============================//

	// -----------------------------
	//	frame start
	// -----------------------------

	cdc_pls_trns
		#(
			.GRAY_DEPTH	( 1				)
		)
		u_fs_cdc_pls_trns (
			.clk_in		( iclk			),
			.clk_out	( mclk			),
			.rst_n		( rst_n			),
			.pls_i		( fs_iclk_i		),
			.pls_o		( fs_mclk_o		)
		);

	// -----------------------------
	//	frame end //
	// -----------------------------

	cdc_ff
		#(
			.BIT_WIDTH	( 1					),
			.SYNC_INPUT	( 1					)
		)
		u_fe_cdc_ff (
			.clk_in		( iclk				),
			.clk_out	( mclk				),
			.rst_n		( rst_n				),
			.data_i		( fe_iclk_i			),
			.data_o		( fe_mclk_o			)
		);

	// -----------------------------
	//	field
	// -----------------------------

	cdc_ff
		#(
			.BIT_WIDTH	( 1					),
			.SYNC_INPUT	( 1					)
		)
		u_fld_cdc_ff (
			.clk_in		( iclk				),
			.clk_out	( mclk				),
			.rst_n		( rst_n				),
			.data_i		( fld_iclk_i		),
			.data_o		( fld_mclk_o		)
		);

	// -----------------------------
	//	width
	// -----------------------------

	cdc_multi_trns
		#(
			.DATA_WIDTH	( SIZE_BIT			)
		)
		u_width_cdc_multi_trns (
			.clk_in		( iclk				),
			.clk_out	( mclk				),
			.rst_n		( rst_n				),
			.data_i		( width_iclk_i		),
			.data_o		( width_mclk_o		)
		);

	// -----------------------------
	//	height
	// -----------------------------

	cdc_multi_trns
		#(
			.DATA_WIDTH	( SIZE_BIT			)
		)
		u_height_cdc_multi_trns (
			.clk_in		( iclk				),
			.clk_out	( mclk				),
			.rst_n		( rst_n				),
			.data_i		( height_iclk_i		),
			.data_o		( height_mclk_o		)
		);
	// -----------------------------
	//	pixel lack
	// -----------------------------

	cdc_ff
		#(
			.BIT_WIDTH	( 1					),
			.SYNC_INPUT	( 1					)
		)
		u_pix_lack_cdc_ff (
			.clk_in		( iclk				),
			.clk_out	( mclk				),
			.rst_n		( rst_n				),
			.data_i		( pix_lack_iclk_i	),
			.data_o		( pix_lack_mclk_o	)
		);

	//===============================//
	// put change clock mclk -> iclk //
	//===============================//

	// -----------------------------
	//	trans done
	// -----------------------------

	always @ ( posedge iclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			tdone_lat_ff	<= 3'b000 ;
		end else begin
			tdone_lat_ff	<= { tdone_lat_ff[1:0], trans_done_i } ;
		end
	end

	assign	tdone_redg	= ~tdone_lat_ff[2] & tdone_lat_ff[1] ;

	// -----------------------------
	//	almost full flag
	// -----------------------------

	always @( posedge iclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			fe_lat_ff	<= 1'b0 ;
		end
		else begin
			fe_lat_ff	<= fe_iclk_i ;
		end
	end

	assign	fe_redg = ~fe_lat_ff & fe_iclk_i ;

	always @( posedge iclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			a_full_int_ff	<= 1'b0 ;
		end
		else if ( fe_redg ) begin
			a_full_int_ff	<= 1'b1 ;
		end
		else if ( tdone_redg ) begin
			a_full_int_ff	<= 1'b0 ;
		end
	end

	always @ ( posedge iclk or negedge rst_n ) begin
		if ( !rst_n ) begin
			a_full_ff	<= 1'b1 ;
		end
		else if ( fe_redg || a_full_int_ff ) begin
			a_full_ff	<= 1'b1 ;
		end
		else begin
			a_full_ff	<= a_full ;
		end
	end

	assign a_full_o	= a_full_ff ;

	//=====================================//
	// asynchronous fifo controller module //
	//=====================================//

	sfb16_async_fifo_ctl
		#(
			.ADR_BIT			( ADR_BIT		),
			.DATA_BIT			( DATA_BIT		),
			.WATERMARK_W		( 3				),
			.WATERMARK_R		( WATERMARK_R	),
			.RETIME				( 0				)
		)
		u_sfb16_async_fifo_ctl (
			.wclk				( iclk			),
			.rclk				( mclk			),
			.rst_n				( rst_n			),
			.srst_wclk			( srst_iclk		),
			.srst_rclk			( srst_mclk		),

			.fifo_wen_i			( wen_i			),
			.fifo_ren_i			( ren_i			),
			.fifo_wdata_i		( wdata_i		),
			.fifo_rdata_o		( rdata_o		),
			.fifo_aempty_o		( a_empty_o		),
			.fifo_empty_o		( empty_o		),
			.fifo_afull_o		( a_full		),
			.fifo_full_o		( 				),

			.ram_wadr_o			( ram_wadr		),
			.ram_radr_o			( ram_radr		),
			.ram_wen_o			( ram_wen		),
			.ram_ren_o			( ram_ren		),
			.ram_wdata_o		( ram_wdata		),
			.ram_rdata_i		( ram_rdata		)
		) ;

	sfb16_srd_swr_ram
		# (
			.ADR_BIT			( ADR_BIT			),
			.DATA_BIT			( DATA_BIT			)
		)
		u_sfb16_srd_swr_ram(
			.wclk				( iclk				),
			.rclk				( mclk				),

			.wadr_i				( ram_wadr			),
			.radr_i				( ram_radr			),
			.wen_i				( ram_wen			),
			.ren_i				( ram_ren			),

			.wdata_i			( ram_wdata			),
			.rdata_o			( ram_rdata			)
		);



endmodule
`default_nettype wire
