//----------------------------------------------------------------------------------------
//Copyright (C) 2014 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		vsi output timing control module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Dec.  1 2014	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT				: Bit par color
//		PLANE				: Color par pixel
//		SIZE_BIT			: size setting signal bit width
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk					: clock for all circuit
//		rst_n				: asyncronous reset input
//		srst				: software reset (sync with clk)
//		enable				: clock enable
//
//		rd_start_o			: frame start of read frame data
//		rd_field_o			: field status of read frame data
//		frm_valid_i			: read can be started
//		rd_mode_i			: set the field status of streaming data
//		rd_mode_o			: set the field status of streaming data
//		width_i				: horizontal size of read frame data
//		height_i			: vertical size of read frame data
//		fifo_clear_o		: read fifo clear signal
//		fifo_cend_i			: read fifo clear complete signal
//
//		data_valid_i		: data valid of input data
//		data_i				: input data
//		data_ready_o		: ack signal from receriver module
//
//		frame_start_o		: frame start of output frame data
//		frame_end_o			: frame end of output frame data
//		valid_o				: data valid of output frame data
//		pixel_o				: pixel data of output frame data
//		field_o				: field status of output frame data  1:even field  0:odd field
//		width_o				: horizontal size of output frame data
//		height_o			: vertical size of output frame data
//		ready_i				: data reception ready of output frame data
//----------------------------------------------------------------------------------------
`default_nettype none
`timescale 1ns / 1ps

module	sfb16_timing_gen (
	// global signal
	clk					,
	rst_n				,
	srst				,
	enable				,

	// control signal
	rd_start_o			,
	rd_field_o			,
	frm_valid_i			,
	rd_mode_i			,
	rd_mode_o			,
	width_i				,
	height_i			,
	fifo_clear_o		,
	fifo_cend_i			,

	// unpacker module I/F
	data_valid_i		,
	data_i				,
	data_ready_o		,

	// VSI output I/F
	frame_start_o		,
	frame_end_o			,
	valid_o				,
	pixel_o				,
	field_o				,
	width_o				,
	height_o			,
	ready_i
	) ;

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------
	parameter	IN_PPC		= 4		; // add
	parameter	OUT_PPC		= 4		; // add
	parameter	Q_BIT		= 8		;
	parameter	PLANE		= 3		;
	parameter	SIZE_BIT	= 11	;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------
	localparam	PIXEL_BIT		= Q_BIT * PLANE * OUT_PPC	;
//	parameter	PIXEL_BIT	= Q_BIT * PLANE	;

	// ---------------------------------------------------------------------
	// State Encode
	// ---------------------------------------------------------------------
	parameter	IDLE			= 4'h0 ;
	parameter	RD_START		= 4'h1 ;
	parameter	PARAM_UP		= 4'h2 ;
	parameter	F_START			= 4'h3 ;
	parameter	DATA_WAIT		= 4'h4 ;
	parameter	READY_WAIT		= 4'h5 ;
	parameter	DATA_OUT		= 4'h6 ;
	parameter	NONE_FLAG		= 4'h7 ;
	parameter	F_END			= 4'h8 ;

// =============================================================================
// PORT DECLARATION
// =============================================================================

	// global signal
	input	wire						clk					;
	input	wire						rst_n				;
	input	wire						srst				;
	input	wire						enable				;

	// control signal
	output	wire						rd_start_o			;
	output	wire						rd_field_o			;
	input	wire						frm_valid_i			;
	input	wire						rd_mode_i			;
	output	wire						rd_mode_o			;
	input	wire	[  SIZE_BIT-1 : 0 ]	width_i				;
	input	wire	[  SIZE_BIT-1 : 0 ]	height_i			;
	output	wire						fifo_clear_o		;
	input	wire						fifo_cend_i			;

	// unpacker module I/F
	input	wire						data_valid_i		;
	input	wire	[ PIXEL_BIT-1 : 0 ]	data_i				;
	output	wire						data_ready_o		;


	// VSI output I/F
	output	wire						frame_start_o		;
	output	wire						frame_end_o			;
	output	wire						valid_o				;
	output	wire	[ PIXEL_BIT-1 : 0 ]	pixel_o				;
	output	wire						field_o				;
	output	wire	[  SIZE_BIT-1 : 0 ]	width_o				;
	output	wire	[  SIZE_BIT-1 : 0 ]	height_o			;
	input	wire						ready_i				;


// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg		[           3 : 0 ]	tim_nst				;
	reg		[           3 : 0 ]	tim_cst_ff			;

	reg		[           2 : 0 ]	fifo_wait_cnt_ff	;

	reg							fifo_clear_ff		;
	reg		[           2 : 0 ]	fifo_cend_lat_ff	;
	wire						fifo_cend_fedg		;

	reg							rd_start_ff			;
	reg							rd_mode_ff			;
	reg							rd_field_ff			;

	reg							data_ready_ff		;

	reg							valid_ff			;
	reg							frame_start_ff		;
	reg							frame_end_ff		;
	reg							field_ff			;
	reg		[ PIXEL_BIT-1 : 0 ]	pixel_ff			;
	reg		[ PIXEL_BIT-1 : 0 ]	pixel_lat_ff		;
	reg		[ PIXEL_BIT-1 : 0 ]	data_buf_ff			;
	reg 						buf_enable_ff 		;

	reg		[  SIZE_BIT-1 : 0 ]	width_ff			;
	reg		[  SIZE_BIT-1 : 0 ]	height_ff			;

	reg		[  SIZE_BIT-1 : 0 ]	h_count_ff			;
	reg		[  SIZE_BIT-1 : 0 ]	v_count_ff			;

	reg		[ SIZE_BIT -1 : 0 ] h_max_ff			;
	reg		[ SIZE_BIT -1 : 0 ] h_max_m_ff			;
	reg		[ SIZE_BIT -1 : 0 ] v_max_ff			;

	wire						h_max				;
	wire						h_max_m				;
	wire						v_max				;


// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	// -----------------------------
	//	output signal
	// -----------------------------

	assign	rd_start_o			= rd_start_ff		;
	assign	rd_field_o			= rd_field_ff		;
	assign	rd_mode_o			= rd_mode_ff		;
	assign	data_ready_o		= data_ready_ff		;
	assign	width_o				= width_ff			;
	assign	height_o			= height_ff			;
	assign	valid_o				= valid_ff			;
	assign	frame_start_o		= frame_start_ff	;
	assign	frame_end_o			= frame_end_ff		;
	assign	field_o				= field_ff			;
	assign	pixel_o				= pixel_ff			;

	// -----------------------------
	// fifo clear
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			fifo_clear_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			fifo_clear_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( tim_nst == IDLE && tim_cst_ff == F_END ) begin
				fifo_clear_ff	<= 1'b1 ;
			end
			else if ( fifo_cend_fedg ) begin
				fifo_clear_ff	<= 1'b0 ;
			end
		end
	end

	assign fifo_clear_o	= fifo_clear_ff ;


	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			fifo_cend_lat_ff	<= 3'b000 ;
		end
		else if ( srst ) begin
			fifo_cend_lat_ff	<= 3'b000 ;
		end
		else if ( enable ) begin
			fifo_cend_lat_ff	<= { fifo_cend_lat_ff[1:0] , fifo_cend_i } ;
		end
	end

	assign fifo_cend_fedg = !fifo_cend_lat_ff[2] & fifo_cend_lat_ff[1] ;


	// -----------------------------
	// fifo wait
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			fifo_wait_cnt_ff	<= 3'b000 ;
		end
		else if ( srst ) begin
			fifo_wait_cnt_ff	<= 3'b000 ;
		end
		else if ( enable ) begin
			if ( tim_cst_ff == PARAM_UP && tim_nst == F_START )begin
				fifo_wait_cnt_ff	<= 3'b000 ;
			end
			else if ( tim_cst_ff == PARAM_UP )begin
				fifo_wait_cnt_ff	<= fifo_wait_cnt_ff + 1'b1 ;
			end
		end
	end

	// -----------------------------
	// state machine
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			tim_cst_ff	<= IDLE ;
		end
		else if ( srst ) begin
			tim_cst_ff	<= IDLE ;
		end
		else if ( enable ) begin
			tim_cst_ff	<= tim_nst ;
		end
	end

	always @( * ) begin
		case ( tim_cst_ff )
			IDLE	: begin
				if ( !fifo_clear_ff && frm_valid_i ) begin
					tim_nst	<= RD_START ;
				end
				else begin
					tim_nst	<= IDLE ;
				end
			end

			RD_START	: begin
				tim_nst	<= PARAM_UP ;
			end

			PARAM_UP	: begin
				if ( data_valid_i ) begin
					tim_nst	<= F_START ;
				end
				else begin
					tim_nst	<= PARAM_UP ;
				end
			end

			F_START	: begin
				if ( ready_i ) begin
					tim_nst	<= DATA_WAIT ;
				end
				else begin
					tim_nst	<= F_START ;
				end
			end

			DATA_OUT	: begin
				if ( ready_i && data_valid_i && h_max_m && v_max ) begin
					tim_nst	<= F_END ;
				end
				else if ( !ready_i && !data_valid_i ) begin
					tim_nst	<= NONE_FLAG ;
				end
				else if ( !ready_i && data_valid_i ) begin
					tim_nst	<= READY_WAIT ;
				end
				else if ( ready_i && !data_valid_i ) begin
					tim_nst	<= DATA_WAIT ;
				end
				else begin
					tim_nst	<= DATA_OUT ;
				end
			end

			DATA_WAIT	: begin
				if ( data_valid_i && h_max && v_max ) begin
					tim_nst	<= F_END ;
				end
				else if ( !ready_i && !data_valid_i ) begin
					tim_nst	<= NONE_FLAG ;
				end
				else if ( !ready_i && data_valid_i ) begin
					tim_nst	<= READY_WAIT ;
				end
				else if ( ready_i && data_valid_i ) begin
					tim_nst	<= DATA_OUT ;
				end
				else begin
					tim_nst	<= DATA_WAIT ;
				end
			end

			READY_WAIT	: begin
				if ( ready_i && data_valid_i && h_max_m && v_max ) begin
					tim_nst	<= F_END ;
				end
				else if ( !ready_i && !data_valid_i ) begin
					tim_nst	<= NONE_FLAG ;
				end
				else if ( ready_i && !data_valid_i ) begin
					tim_nst	<= DATA_WAIT ;
				end
				else if ( ready_i && data_valid_i ) begin
					tim_nst	<= DATA_OUT ;
				end
				else begin
					tim_nst	<= READY_WAIT ;
				end
			end

			NONE_FLAG	: begin
				if ( ready_i && data_valid_i ) begin
					tim_nst	<= DATA_OUT ;
				end
				else if ( !ready_i && data_valid_i ) begin
					tim_nst	<= READY_WAIT ;
				end
				else if ( ready_i && !data_valid_i ) begin
					tim_nst	<= DATA_WAIT ;
				end
				else begin
					tim_nst	<= NONE_FLAG ;
				end
			end

			F_END	: begin
				if ( ready_i ) begin
					tim_nst	<= IDLE ;
				end
				else begin
					tim_nst	<= F_END ;
				end
			end

			default	:	begin
				tim_nst	<= IDLE ;
			end
		endcase
	end

	// -----------------------------
	//	read start
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rd_start_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			rd_start_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if (  tim_cst_ff == IDLE && tim_nst == RD_START ) begin
				rd_start_ff	<= 1'b1 ;
			end
			else if ( tim_cst_ff == RD_START && tim_nst == PARAM_UP ) begin
				rd_start_ff	<= 1'b0 ;
			end
		end
	end

	// -----------------------------
	//	read field
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rd_mode_ff	<= 1'b0 ;
			rd_field_ff	<= 1'b1 ;
		end
		else if ( srst ) begin
			rd_mode_ff	<= 1'b0 ;
			rd_field_ff	<= 1'b1 ;
		end
		else if ( enable ) begin
			if ( tim_nst == RD_START ) begin
				rd_mode_ff	<= rd_mode_i ;
				rd_field_ff	<= rd_mode_i & !rd_field_ff ;
			end
		end
	end

	// -----------------------------
	//	data ready
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			data_ready_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			data_ready_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( tim_nst == DATA_WAIT
			  || tim_nst == DATA_OUT
			  || ( tim_nst == NONE_FLAG && tim_cst_ff == DATA_WAIT ) ) begin
				data_ready_ff	<= 1'b1 ;
			end
			else if ( tim_nst == READY_WAIT
				   || tim_nst == F_END ) begin
				data_ready_ff	<= 1'b0 ;
			end

		end
	end

	// -----------------------------
	//	frame start
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frame_start_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			frame_start_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( tim_nst == F_START ) begin
				frame_start_ff	<= 1'b1 ;
			end
			else if ( tim_nst != F_START ) begin
				frame_start_ff	<= 1'b0 ;
			end
		end
	end

	// -----------------------------
	//	frame end
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frame_end_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			frame_end_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( tim_nst == F_END ) begin
				frame_end_ff	<= 1'b1 ;
			end
			else if ( tim_cst_ff == F_END && tim_nst == IDLE ) begin
				frame_end_ff	<= 1'b0 ;
			end

		end
	end

	// -----------------------------
	//	valid
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			valid_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			valid_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( tim_nst == F_START
			  || tim_nst == F_END
			  || tim_nst == DATA_OUT
			  || ( tim_nst == DATA_WAIT && tim_cst_ff == NONE_FLAG && buf_enable_ff )
			  || ( tim_nst == DATA_WAIT && tim_cst_ff == READY_WAIT ) ) begin
				valid_ff	<= 1'b1 ;
			end
			else if ( ( tim_nst == DATA_WAIT && !buf_enable_ff )
				   || ( tim_nst == IDLE ) ) begin
				valid_ff	<= 1'b0 ;
			end

		end
	end

	// -----------------------------
	//	width / height / field
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			width_ff	<= {SIZE_BIT{1'b0}} ;
			height_ff	<= {SIZE_BIT{1'b0}} ;
			h_max_ff	<= {SIZE_BIT{1'b0}} ;
			h_max_m_ff	<= {SIZE_BIT{1'b0}} ;
			v_max_ff	<= {SIZE_BIT{1'b0}} ;
			field_ff	<= 1'b0				;
		end
		else if ( srst ) begin
			width_ff	<= {SIZE_BIT{1'b0}} ;
			height_ff	<= {SIZE_BIT{1'b0}} ;
			h_max_ff	<= {SIZE_BIT{1'b0}} ;
			h_max_m_ff	<= {SIZE_BIT{1'b0}} ;
			v_max_ff	<= {SIZE_BIT{1'b0}} ;
			field_ff	<= 1'b0				;
		end
		else if ( enable ) begin
			if ( tim_cst_ff == PARAM_UP && tim_nst == F_START ) begin
				width_ff	<= width_i * IN_PPC	;
				height_ff	<= height_i			;
				h_max_ff	<= width_i * IN_PPC / OUT_PPC - 1'd1	;
				h_max_m_ff	<= width_i * IN_PPC / OUT_PPC - 2'd2	;
				v_max_ff	<= height_i - 1'd1	;
				field_ff	<= rd_field_ff	 	;
			end
		end
	end

	// -----------------------------
	//	line / pixel counter
	// -----------------------------

	assign	h_max	= ( h_count_ff == h_max_ff   ) ? 1'b1 : 1'b0 ;
	assign	h_max_m	= ( h_count_ff == h_max_m_ff ) ? 1'b1 : 1'b0 ;
	assign	v_max	= ( v_count_ff == v_max_ff   ) ? 1'b1 : 1'b0 ;

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			h_count_ff	<= {SIZE_BIT{1'b0}} ;
			v_count_ff	<= {SIZE_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			h_count_ff	<= {SIZE_BIT{1'b0}} ;
			v_count_ff	<= {SIZE_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( frame_start_ff ) begin
				h_count_ff	<= {SIZE_BIT{1'b0}} ;
				v_count_ff	<= {SIZE_BIT{1'b0}} ;
			end
			else if ( !frame_start_ff && valid_ff && ready_i ) begin
				if ( h_max ) begin
					h_count_ff	<= {SIZE_BIT{1'b0}} ;
					if ( v_max ) begin
						v_count_ff	<= {SIZE_BIT{1'b0}} ;
					end
					else begin
						v_count_ff	<= v_count_ff + 1'b1 ;
					end
				end
				else begin
					h_count_ff	<= h_count_ff + 1'b1 ;
					v_count_ff	<= v_count_ff ;
				end
			end
		end
	end


	// -----------------------------
	//	data buffer
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			buf_enable_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			buf_enable_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( tim_nst == NONE_FLAG && tim_cst_ff == READY_WAIT ) begin
				buf_enable_ff	<= 1'b1 ;
			end
			else if ( tim_nst != NONE_FLAG && tim_cst_ff == NONE_FLAG ) begin
				buf_enable_ff	<= 1'b0 ;
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			data_buf_ff	<= {PIXEL_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			data_buf_ff	<= {PIXEL_BIT{1'b0}} ;
		end
		else if ( enable && !buf_enable_ff ) begin
			if ( tim_nst == READY_WAIT && ( ( tim_cst_ff == DATA_WAIT )
										 || ( tim_cst_ff == DATA_OUT  )
										 || ( tim_cst_ff == NONE_FLAG ) ) ) begin
				data_buf_ff	<= data_i ;
			end
			else if ( tim_cst_ff == F_END && tim_nst == IDLE ) begin
				data_buf_ff	<= {PIXEL_BIT{1'b0}} ;
			end
		end
	end

	// -----------------------------
	//	pixel
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			pixel_ff	<= {PIXEL_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			pixel_ff	<= {PIXEL_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( tim_cst_ff == F_END && tim_nst == IDLE ) begin
				pixel_ff	<= {PIXEL_BIT{1'b0}} ;
			end
			else if ( tim_cst_ff == DATA_OUT && tim_nst == READY_WAIT ) begin
				pixel_ff	<= pixel_ff ;
			end
			else if ( ( tim_cst_ff == READY_WAIT &&  ( ( tim_nst == DATA_WAIT )
													|| ( tim_nst == DATA_OUT  )
													|| ( tim_nst == F_END     ) ) )
				   || ( buf_enable_ff && tim_cst_ff == NONE_FLAG && ( ( tim_nst == DATA_OUT  )
				   												   || ( tim_nst == DATA_WAIT ) ) ) )begin
				pixel_ff	<= data_buf_ff ;
			end
			else if ( ( tim_nst == DATA_OUT )
				   || ( tim_nst == F_END && tim_cst_ff != F_END ) ) begin
				pixel_ff	<= data_i ;
			end
		end
	end


endmodule
`default_nettype wire
