//----------------------------------------------------------------------------------------
//Copyright (C) 2014 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		fifo read module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Dec.  1 2014	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		DATA_BIT			: data bit width
//		SIZE_BIT			: size setting signal bit width
//----------------------------------------------------------------------------------------
// I/O PORTS
//		mclk				: write clock
//		rclk				: read clock
//		rst_n				: asynchronous reset
//		srst_mclk			: software reset (sync with mclk)
//		srst_oclk			: software reset (sync with oclk)
//
//		rs_mclk_o			: frame start of read frame data.
//		fld_mclk_o			: field status of read frame data.  1:even field	0:odd field
//		width_mclk_i		: horizontal size of read frame data
//		height_mclk_i		: vertical size of read frame data
//		frm_vld_mclk_i		: read can be started
//		fdata_rd_mclk_i		: read completed
//		rd_mode_mclk_i		: read mode
//
//		rs_oclk_i			: frame start of read frame data.
//		fld_oclk_i			: field status of read frame data.  1:even field  0:odd field
//		fifo_clear_oclk_i	: fifo clear signal
//		width_oclk_o		: horizontal size of read frame data
//		height_oclk_o		: vertical size of read frame data
//		frm_vld_oclk_o		: read can be started
//		fdata_rd_oclk_o		: read completed
//		fifo_cend_oclk_o	: fifo clear complete signal
//		rd_mode_oclk_o		: read mode
//
//		a_full_o			: fifo almost full flag
//		empty_o				: fifo empty flag
//
//		wen_i				: fifo write enable
//		wdata_i				: fifo write data
//		ren_i				: fifo read enable
//		rdata_o				: fifo read data
//----------------------------------------------------------------------------------------
`default_nettype none
`timescale 1ns / 1ps

module	sfb16_rfifo (
	// global signal
	mclk				,
	oclk				,
	rst_n				,
	srst_mclk			,
	srst_oclk			,

	// mclk syncronous signal
	rs_mclk_o			,
	fld_mclk_o			,
	width_mclk_i		,
	height_mclk_i		,
	frm_vld_mclk_i		,
	fdata_rd_mclk_i		,
	rd_mode_mclk_i		,

	// oclk syncronous signal
	rs_oclk_i			,
	fld_oclk_i			,
	fifo_clear_oclk_i	,
	width_oclk_o		,
	height_oclk_o		,
	frm_vld_oclk_o		,
	fdata_rd_oclk_o		,
	fifo_cend_oclk_o	,
	rd_mode_oclk_o		,

	// fifo flag
	a_full_o			,
	empty_o				,

	// fifo data i/f
	wen_i				,
	wdata_i				,
	ren_i				,
	rdata_o
	) ;

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------

	parameter	SIZE_BIT	= 11 ;
	parameter	DATA_BIT	= 128 ;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

	localparam	ADR_BIT		= 6 ;
	localparam	WATERMARK_W	= 31 ;

// =============================================================================
// PORT DECLARATION
// =============================================================================

	// global signal
	input	wire						mclk				;
	input	wire						oclk				;
	input	wire						rst_n				;
	input	wire						srst_mclk			;
	input	wire						srst_oclk			;

	//mclk syncronous signal
	output	wire						rs_mclk_o			;
	output	wire						fld_mclk_o			;
	input	wire	[  SIZE_BIT-1 : 0 ]	width_mclk_i		;
	input	wire	[  SIZE_BIT-1 : 0 ]	height_mclk_i		;
	input	wire						frm_vld_mclk_i		;
	input	wire						fdata_rd_mclk_i		;
	input	wire						rd_mode_mclk_i		;

	//oclk syncronous signal
	input	wire						rs_oclk_i			;
	input	wire						fld_oclk_i			;
	input	wire						fifo_clear_oclk_i	;
	output	wire	[  SIZE_BIT-1 : 0 ]	width_oclk_o		;
	output	wire	[  SIZE_BIT-1 : 0 ]	height_oclk_o		;
	output	wire						frm_vld_oclk_o		;
	output	wire						fdata_rd_oclk_o		;
	output	wire						fifo_cend_oclk_o	;
	output	wire						rd_mode_oclk_o		;

	// fifo flag
	output	wire						a_full_o			;
	output	wire						empty_o				;

	// fifo data I/F
	input	wire						wen_i				;
	input	wire	[ DATA_BIT-1 : 0 ]	wdata_i				;
	input	wire						ren_i				;
	output	wire	[ DATA_BIT-1 : 0 ]	rdata_o				;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	wire	[  DATA_BIT-1 : 0 ]		fifo_rdata			;
	wire							ram_wen				;
	wire							ram_ren				;
	wire	[   ADR_BIT-1 : 0 ]		ram_wadr			;
	wire	[   ADR_BIT-1 : 0 ]		ram_radr			;
	wire	[  DATA_BIT-1 : 0 ]		ram_wdata			;
	wire	[  DATA_BIT-1 : 0 ]		ram_rdata			;

	wire							fifo_clear_mclk		;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	//===============================//
	// put change clock mclk -> oclk //
	//===============================//

	// -----------------------------
	//	width
	// -----------------------------

	cdc_multi_trns
		#(
			.DATA_WIDTH	( SIZE_BIT			)
		)
		u_width_cdc_multi_trns (
			.clk_in		( mclk				),
			.clk_out	( oclk				),
			.rst_n		( rst_n				),
			.data_i		( width_mclk_i		),
			.data_o		( width_oclk_o		)
		);

	// -----------------------------
	//	height
	// -----------------------------

	cdc_multi_trns
		#(
			.DATA_WIDTH	( SIZE_BIT			)
		)
		u_height_cdc_multi_trns (
			.clk_in		( mclk				),
			.clk_out	( oclk				),
			.rst_n		( rst_n				),
			.data_i		( height_mclk_i		),
			.data_o		( height_oclk_o		)
		);

	// -----------------------------
	//	final data read
	// -----------------------------

	cdc_ff
		#(
			.BIT_WIDTH	( 1					),
			.SYNC_INPUT	( 1					)
		)
		u_fdata_rd_cdc_ff (
			.clk_in		( mclk				),
			.clk_out	( oclk				),
			.rst_n		( rst_n				),
			.data_i		( fdata_rd_mclk_i	),
			.data_o		( fdata_rd_oclk_o	)
		);

	// -----------------------------
	//	fifo clear
	// -----------------------------

	cdc_ff
		#(
			.BIT_WIDTH	( 1					),
			.SYNC_INPUT	( 0					)
		)
		u_fifo_cend_cdc_ff (
			.clk_in		( mclk				),
			.clk_out	( oclk				),
			.rst_n		( rst_n				),
			.data_i		( fifo_clear_mclk	),
			.data_o		( fifo_cend_oclk_o	)
		);

	// -----------------------------
	//	frame valid
	// -----------------------------

	cdc_ff
		#(
			.BIT_WIDTH	( 1					),
			.SYNC_INPUT	( 1					)
		)
		u_frm_vld_cdc_ff (
			.clk_in		( mclk				),
			.clk_out	( oclk				),
			.rst_n		( rst_n				),
			.data_i		( frm_vld_mclk_i	),
			.data_o		( frm_vld_oclk_o	)
		);

	// -----------------------------
	//	read mode
	// -----------------------------

	cdc_ff
		#(
			.BIT_WIDTH	( 1					),
			.SYNC_INPUT	( 1					)
		)
		u_rd_mode_cdc_ff (
			.clk_in		( mclk				),
			.clk_out	( oclk				),
			.rst_n		( rst_n				),
			.data_i		( rd_mode_mclk_i	),
			.data_o		( rd_mode_oclk_o	)
		);


	//===============================//
	// put change clock oclk -> mclk //
	//===============================//

	// -----------------------------
	//	read start
	// -----------------------------

	cdc_pls_trns
		#(
			.GRAY_DEPTH	( 1				)
		)
		u_fs_cdc_pls_trns (
			.clk_in		( oclk			),
			.clk_out	( mclk			),
			.rst_n		( rst_n			),
			.pls_i		( rs_oclk_i		),
			.pls_o		( rs_mclk_o		)
		);

	// -----------------------------
	//	field
	// -----------------------------

	cdc_ff
		#(
			.BIT_WIDTH	( 1					),
			.SYNC_INPUT	( 0					)
		)
		u_fld_cdc_ff (
			.clk_in		( oclk				),
			.clk_out	( mclk				),
			.rst_n		( rst_n				),
			.data_i		( fld_oclk_i		),
			.data_o		( fld_mclk_o		)
		);



	// -----------------------------
	//	fifo clear
	// -----------------------------

	cdc_ff
		#(
			.BIT_WIDTH	( 1					),
			.SYNC_INPUT	( 0					)
		)
		u_fifo_clear_cdc_ff (
			.clk_in		( oclk				),
			.clk_out	( mclk				),
			.rst_n		( rst_n				),
			.data_i		( fifo_clear_oclk_i	),
			.data_o		( fifo_clear_mclk	)
		);

	//=====================================//
	// asynchronous fifo controller module //
	//=====================================//

	sfb16_async_fifo_ctl
		#(
			.ADR_BIT			( ADR_BIT						),
			.DATA_BIT			( DATA_BIT						),
			.WATERMARK_W		( WATERMARK_W					),
			.WATERMARK_R		( 3								),
			.RETIME				( 0								)
		)
		u_sfb16_async_fifo_ctl (
			.wclk				( mclk							),
			.rclk				( oclk							),
			.rst_n				( rst_n							),
			.srst_wclk			( srst_mclk	| fifo_clear_mclk	),
			.srst_rclk			( srst_oclk	| fifo_clear_oclk_i	),

			.fifo_wen_i			( wen_i							),
			.fifo_ren_i			( ren_i							),
			.fifo_wdata_i		( wdata_i						),
			.fifo_rdata_o		( fifo_rdata					),
			.fifo_aempty_o		( 								),
			.fifo_empty_o		( empty_o						),
			.fifo_afull_o		( a_full_o						),
			.fifo_full_o		( 								),

			.ram_wadr_o			( ram_wadr						),
			.ram_radr_o			( ram_radr						),
			.ram_wen_o			( ram_wen						),
			.ram_ren_o			( ram_ren						),
			.ram_wdata_o		( ram_wdata						),
			.ram_rdata_i		( ram_rdata						)
		) ;

	sfb16_srd_swr_ram
		# (
			.ADR_BIT			( ADR_BIT						),
			.DATA_BIT			( DATA_BIT						)
		)
		u_sfb16_srd_swr_ram(
			.wclk				( mclk							),
			.rclk				( oclk							),

			.wadr_i				( ram_wadr						),
			.radr_i				( ram_radr						),
			.wen_i				( ram_wen						),
			.ren_i				( ram_ren						),

			.wdata_i			( ram_wdata						),
			.rdata_o			( ram_rdata						)
		);

	assign rdata_o	= fifo_rdata			;

endmodule
`default_nettype wire
