//----------------------------------------------------------------------------------------
//Copyright (C) 2014 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		 input data packing module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Dec.  1 2014	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT			: Bit par color
//		PLANE			: Color par pixel
//		SIZE_BIT		: size setting signal bit width
//		LDATA_BIT		: high-performance controller local data bit width
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk				: clock for all circuit
//		rst_n			: asyncronous reset input
//		srst			: software reset (sync with clk)
//		enable			: clock enable
//
//		frame_start_i	: frame start of input frame data
//		frame_end_i		: frame end of input frame data
//		valid_i			: data valid of input frame data
//		pixel_i			: pixel data of input frame data
//		field_i			: field status of input frame data  1:even field  0:odd field
//		width_i			: horizontal size of input frame data
//		height_i		: vertical size of input frame data
//		ready_o			: data reception ready of input frame data
//
//		frame_start_o	: frame start of output packing data
//		frame_end_o		: frame end of output packing data
//		field_o			: field status of output packing data  1:even field	0:odd field
//		width_o			: horizontal size of output packing data
//		height_o		: vertical size of output packing data
//		pixel_lack_o	: signal informing 1 frame packing completion
//
//		wt_en_o			: fifo write enable
//		wt_data_o		: fifo write data
//		a_full_i		: fifo almost full flag
//----------------------------------------------------------------------------------------
`default_nettype none
`timescale 1ns / 1ps

module	sfb16_packer (
	// global Signal
	clk				,
	rst_n			,
	srst			,
	enable			,

	// VSI input signal
	frame_start_i	,
	frame_end_i		,
	valid_i			,
	pixel_i			,
	field_i			,
	width_i			,
	height_i		,
	ready_o			,

	// wfifo module I/F
	frame_start_o	,
	frame_end_o		,
	field_o			,
	width_o			,
	height_o		,
	pixel_lack_o	,

	// fifo flag signal
	wt_en_o			,
	wt_data_o		,
	a_full_i

) ;

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------

	parameter	PPC					= 4		; // add
	parameter	Q_BIT				= 8		;
	parameter	PLANE				= 3		;
	parameter	SIZE_BIT			= 12	;
	parameter	LDATA_BIT			= 128	;


	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

//	parameter	PIXEL_BIT			= Q_BIT * PLANE	;
	localparam	PIXEL_BIT			= Q_BIT * PLANE * PPC	;
	localparam	PDATA_BIT			= LDATA_BIT + PIXEL_BIT	;

	function integer	log2 ;
		input integer	value ;
		begin
			value	= value - 1 ;
			for (log2 = 0 ; value > 0 ; log2 = log2 + 1 ) begin
				value	= value >> 1 ;
			end
		end
	endfunction

	localparam	CNT_W				= log2(PDATA_BIT)	;

	// ---------------------------------------------------------------------
	// State Encode
	// ---------------------------------------------------------------------

	localparam	IDLE		= 3'd0 ;
	localparam	PACK		= 3'd1 ;
	localparam	WAIT		= 3'd2 ;
	localparam	F_END		= 3'd3 ;
	localparam	ERROR		= 3'd4 ;

// =============================================================================
// PORT DECLARATION
// =============================================================================

	// global Signal
	input	wire						clk				;
	input	wire						rst_n			;
	input	wire						srst			;
	input	wire						enable			;

	// VSI input signal
	input	wire						frame_start_i	;
	input	wire						frame_end_i		;
	input	wire						valid_i			;
	input	wire	[ PIXEL_BIT-1 : 0 ]	pixel_i			;
	input	wire						field_i			;
	input	wire	[  SIZE_BIT-1 : 0 ]	width_i			;
	input	wire	[  SIZE_BIT-1 : 0 ]	height_i		;
	output	wire						ready_o			;

	// wfifo module I/F
	output	wire						frame_start_o	;
	output	wire						frame_end_o		;
	output	wire						field_o			;
	output	wire						pixel_lack_o	;

	// fifo flag signal
	output	wire						wt_en_o			;
	output	wire	[ LDATA_BIT-1 : 0 ]	wt_data_o		;
	input	wire						a_full_i		;

	// manager module I/F
	output	wire	[  SIZE_BIT-1 : 0 ]	width_o			;
	output	wire	[  SIZE_BIT-1 : 0 ]	height_o		;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg		[           2 : 0 ]	pac_cst_ff			;
	reg		[           2 : 0 ]	pac_nst				;

	wire						frame_start			;
	wire						frame_end			;
	reg							frame_start_ff		;
	reg							frame_end_ff		;
	reg							ready_ff			;
	reg							field_ff			;
	reg		[  SIZE_BIT-1 : 0 ]	width_ff			;
	reg		[  SIZE_BIT-1 : 0 ]	height_ff			;
	reg		[  SIZE_BIT-1 : 0 ]	h_max_ff			;
	reg		[  SIZE_BIT-1 : 0 ]	v_max_ff			;

	reg		[     CNT_W-1 : 0 ]	cnt_ff				;
	reg		[ PDATA_BIT-1 : 0 ]	pdata_ff			;

	wire						h_count_max			;
	wire						v_count_max			;
	reg		[  SIZE_BIT-1 : 0 ]	h_count_ff			;
	reg		[  SIZE_BIT-1 : 0 ]	v_count_ff			;
	reg							count_max_ff		;

	reg							a_full_lat_ff		;
	wire						a_full_fedg			;
	reg							fdata_wten_ff		;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	// -----------------------------
	//	Output signal
	// -----------------------------

	assign	ready_o			= ready_ff ;
	assign	frame_start_o	= frame_start_ff ;
	assign	field_o			= field_ff ;
	assign	frame_end_o		= frame_end_ff ;
	assign	pixel_lack_o	= !count_max_ff ;
	assign	width_o			= width_ff ;
	assign	height_o		= height_ff ;


	// -----------------------------
	//	flag signal
	// -----------------------------


	assign	frame_start	= frame_start_i & ready_ff & valid_i & ( pixel_i[3:0]==4'h0 ) ;
	assign	frame_end	= frame_end_i & ready_ff & valid_i ;

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			a_full_lat_ff	<= 1'b0 ;
		end
		else begin
			a_full_lat_ff	<= a_full_i ;
		end
	end

	assign a_full_fedg	=  a_full_lat_ff & !a_full_i ;


	// -----------------------------
	//	state machine
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			pac_cst_ff	<= IDLE ;
		end
		else if ( srst ) begin
			pac_cst_ff	<= IDLE ;
		end
		else begin
			pac_cst_ff	<= pac_nst ;
		end
	end

	always @( * ) begin
		case ( pac_cst_ff )
			IDLE	: begin
				if ( frame_start && !a_full_i ) begin
					pac_nst	<= PACK ;
				end
				else begin
					pac_nst	<= IDLE ;
				end
			end

			PACK	: begin
				if  ( ( frame_start )
				   || ( !frame_end_i && ready_o && valid_i && v_count_max && h_count_max ) ) begin
					pac_nst	<= ERROR ;
				end
				else if ( frame_end && v_count_max && h_count_max ) begin
					pac_nst	<= F_END ;
				end
				else if ( a_full_i ) begin
					pac_nst	<= WAIT ;
				end
				else begin
					pac_nst	<= PACK ;
				end
			end

			WAIT	: begin
				if ( !a_full_i ) begin
					pac_nst	<= PACK ;
				end
				else begin
					pac_nst	<= WAIT ;
				end
			end

			F_END	: begin
				if ( a_full_fedg ) begin
					pac_nst	<= IDLE ;
				end
				else begin
					pac_nst	<= F_END ;
				end
			end

			ERROR	: begin
				if ( a_full_fedg ) begin
					pac_nst	<= WAIT ;
				end
				else begin
					pac_nst	<= ERROR ;
				end
			end

			default	: begin
				pac_nst	<= IDLE ;
			end

		endcase
	end

	// -----------------------------
	//	frame start
	// -----------------------------

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frame_start_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			frame_start_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ( pac_cst_ff == IDLE  && pac_nst == PACK )
			  || ( pac_cst_ff == ERROR && pac_nst == WAIT ) ) begin
				frame_start_ff	<= 1'b1 ;
			end
			else begin
				frame_start_ff	<= 1'b0 ;
			end
		end
	end

	// -----------------------------
	//	frame end
	// -----------------------------

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frame_end_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			frame_end_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( pac_cst_ff == PACK && ( ( pac_nst == ERROR )
									  || ( pac_nst == F_END ) ) ) begin
				frame_end_ff	<= 1'b1 ;
			end
			else if ( ( pac_cst_ff == F_END && pac_nst == IDLE )
				   || ( pac_cst_ff == ERROR && pac_nst == WAIT ) ) begin
				frame_end_ff	<= 1'b0 ;
			end
		end
	end

	// -----------------------------
	//	ready
	// -----------------------------

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			ready_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			ready_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( ( pac_cst_ff == IDLE && pac_nst == PACK )
			  || ( pac_cst_ff == WAIT && pac_nst == PACK ) ) begin
				ready_ff	<= 1'b1 ;
			end
			else if ( ( pac_cst_ff == PACK && ( ( pac_nst == ERROR )
											 || ( pac_nst == F_END )
											 || ( pac_nst == WAIT  ) ) ) ) begin
				ready_ff	<= 1'b0 ;
			end
			else if ( pac_cst_ff == IDLE ) begin
				ready_ff	<= !a_full_i ;
			end
		end
	end

	// -----------------------------
	//	status latch
	// -----------------------------

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			field_ff	<= 1'b0 			;
			width_ff	<= {SIZE_BIT{1'b0}} ;
			height_ff	<= {SIZE_BIT{1'b0}} ;
			h_max_ff	<= {SIZE_BIT{1'b0}} ;
			v_max_ff	<= {SIZE_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			field_ff	<= 1'b0 			;
			width_ff	<= {SIZE_BIT{1'b0}} ;
			height_ff	<= {SIZE_BIT{1'b0}} ;
			h_max_ff	<= {SIZE_BIT{1'b0}} ;
			v_max_ff	<= {SIZE_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( pac_cst_ff == IDLE && pac_nst == PACK ) begin
				field_ff	<= field_i 			;
				width_ff	<= width_i / PPC			;
				height_ff	<= height_i			;
				h_max_ff	<= width_i / PPC - 1'b1	;
				v_max_ff	<= height_i - 1'b1	;
			end
		end
	end

	// -----------------------------
	//	pixel count
	// -----------------------------

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			h_count_ff	<= {SIZE_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			h_count_ff	<= {SIZE_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( pac_cst_ff != PACK && pac_cst_ff != WAIT ) begin
				h_count_ff	<= {SIZE_BIT{1'b0}} ;
			end
			else if ( valid_i && ready_o ) begin
				if ( h_count_max ) begin
					h_count_ff	<= {SIZE_BIT{1'b0}} ;
				end
				else begin
					h_count_ff	<= h_count_ff + 1'b1 ;
				end
			end
		end
	end

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			v_count_ff	<= {SIZE_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			v_count_ff	<= {SIZE_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( pac_cst_ff != PACK && pac_cst_ff != WAIT ) begin
				v_count_ff	<= {SIZE_BIT{1'b0}} ;
			end
			else if ( valid_i && ready_o && h_count_max ) begin
				if ( v_count_max ) begin
					v_count_ff	<= {SIZE_BIT{1'b0}} ;
				end
				else begin
					v_count_ff	<= v_count_ff + 1'b1 ;
				end
			end
		end
	end

	assign	h_count_max	= ( h_count_ff == h_max_ff ) ?	1'b1 : 1'b0 ;
	assign	v_count_max	= ( v_count_ff == v_max_ff ) ?	1'b1 : 1'b0 ;

	// -----------------------------
	//	pixel_lack flag
	// -----------------------------

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			count_max_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			count_max_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( pac_cst_ff == IDLE ) begin
				count_max_ff	<= 1'b0 ;
			end
			else if ( valid_i && h_count_max && v_count_max && frame_end ) begin
//			else if ( valid_i && !a_full_i && h_count_max && v_count_max && frame_end ) begin
				count_max_ff	<= 1'b1 ;
			end
		end
	end

	// -----------------------------
	//	data_packing
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			cnt_ff[ CNT_W - 1 : 0 ]	<= { CNT_W{1'b0} } ;
		end
		else if ( srst ) begin
			cnt_ff[ CNT_W - 1 : 0 ]	<= { CNT_W{1'b0} } ;
		end
		else if ( ( pac_nst == F_END )
			   || ( pac_nst == ERROR )
			   || ( pac_nst == IDLE  ) ) begin
			cnt_ff[ CNT_W - 1 : 0 ]	<= { CNT_W{1'b0} } ;
		end
		else if ( ( pac_cst_ff == PACK ) && valid_i ) begin
			if ( cnt_ff >= LDATA_BIT ) begin
				cnt_ff[ CNT_W - 1 : 0 ]	<= cnt_ff[ CNT_W - 1 : 0 ] + PIXEL_BIT[ CNT_W - 1 : 0 ] - LDATA_BIT[ CNT_W - 1 : 0 ] ;
			end
			else begin
				cnt_ff[ CNT_W - 1 : 0 ]	<= cnt_ff[ CNT_W - 1 : 0 ] + PIXEL_BIT[ CNT_W - 1 : 0 ] ;
			end
		end
		else if ( cnt_ff >= LDATA_BIT ) begin
			cnt_ff[ CNT_W - 1 : 0 ]	<= cnt_ff[ CNT_W - 1 : 0 ] - LDATA_BIT[ CNT_W - 1 : 0 ] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			pdata_ff[ PDATA_BIT - 1 : 0 ]	<= { PDATA_BIT{1'b0} } ;
		end
		else if ( srst ) begin
			pdata_ff[ PDATA_BIT - 1 : 0 ]	<= { PDATA_BIT{1'b0} } ;
		end
		else if ( ( pac_nst == IDLE  )
			   || ( pac_nst == ERROR ) ) begin
			pdata_ff[ PDATA_BIT - 1 : 0 ]	<= { PDATA_BIT{1'b0} } ;
		end
		else if ( ( pac_cst_ff == PACK ) && valid_i ) begin
			if ( cnt_ff >= LDATA_BIT ) begin
				pdata_ff[ PDATA_BIT - 1 : 0 ]	<= ( pdata_ff[ PDATA_BIT - 1 : 0 ] >> LDATA_BIT )
												 | ( pixel_i[ PIXEL_BIT - 1 : 0 ] << ( cnt_ff - LDATA_BIT ) ) ;
			end
			else begin
				pdata_ff[ PDATA_BIT - 1 : 0 ]	<= pdata_ff[ PDATA_BIT - 1 : 0 ]
												 | ( pixel_i[ PIXEL_BIT - 1 : 0 ] << cnt_ff ) ;
			end
		end
		else if ( cnt_ff >= LDATA_BIT ) begin
			pdata_ff[ PDATA_BIT - 1 : 0 ]	<= pdata_ff[ PDATA_BIT - 1 : 0 ] >> LDATA_BIT ;
		end
	end

	// -----------------------------
	//	fifo flag (wt_en)
	// -----------------------------

	always @ ( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			fdata_wten_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			fdata_wten_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( frame_end && v_count_max && h_count_max ) begin
				fdata_wten_ff	<= 1'b1 ;
			end
			else begin
//			else if ( !a_full_i ) begin
				fdata_wten_ff	<= 1'b0 ;
			end
		end
	end

	assign	wt_en_o		= fdata_wten_ff | ( cnt_ff >= LDATA_BIT ) ;
	assign	wt_data_o	= pdata_ff[ LDATA_BIT - 1 : 0 ] ;


endmodule
`default_nettype wire
