//----------------------------------------------------------------------------------------
//Copyright (C) 2014 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		memory write i/f control module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Dec.  1 2014	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		BASE_ADR			: logical base address for frame memory ( word address )
//		STEP_ADR			: address step between each frame ( word address )
//		LDATA_BIT			: high-performance controller local data bit width
//		LADR_BIT			: high-performance controller local address bit width
//		SIZE_BIT			: size setting signal bit width
//		MODE				: set bank access
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk					: clock for all circuit
//		rst_n				: asyncronous reset input
//		srst				: software reset (sync with clk)
//		enable				: clock enable
//		param_update_i		: trigger for parameter update (sync with aclk)
//		scan_mode_i			: set the field status of streaming data
//
//		frame_start_i		: frame start of input data
//		frame_end_i			: frame end of input data
//		field_i				: field of input data
//		width_i				: width of input data
//		height_i			: height of input data
//		pix_lack_i			: signal informing 1 frame packing completion
//		trans_done_o		: transaction end
//
//		rd_en_o				: fifo read enable
//		rd_data_i			: fifo read data
//		a_empty_i			: fifo almost empty flag
//		empty_i				: fifo empty flag
//
//		avmm_address_o		: high-performance controller local address
//		avmm_burstcount_o	: high-performance controller local burst size
//		avmm_write_o		: high-performance controller local write request
//		avmm_writedata_o	: high-performance controller local write data
//		avmm_waitrequest_i	: high-performance controller wait request
//
//		bus_req_o			: Avalon-MM bus request to memory controller
//		bus_grant_i			: Avalon-MM bus grant from memory controller
//		bus_busy_o			: Avalon-MM bus busy to memory controller
//
//		wfrm_cnt_o			: next write frame number
//		rfrm_cnt_i			: next read frame number
//		rd_frm_vld_o		: read can be started
//
//		frm*o_wdt_o			: width of a frame in bank* odd field
//		frm*o_hgt_o			: height of a frame in bank* odd field
//		frm*o_adr_o			: address of a frame in bank* odd field
//		frm*e_wdt_o			: width of a frame in bank* even field
//		frm*e_hgt_o			: height of a frame in bank* even field
//		frm*e_adr_o			: address of a frame in bank* even field
//		frm*_mode_o			: scan mode of a frame
//----------------------------------------------------------------------------------------
`default_nettype none
`timescale 1ns / 1ps

module	sfb16_mem_write_ctl (
	// global signal
	clk						,
	rst_n					,
	srst					,
	enable					,
	param_update_i			,
	scan_mode_i				,

	// control signal
	frame_start_i			,
	frame_end_i				,
	field_i					,
	width_i					,
	height_i				,
	pix_lack_i				,
	trans_done_o			,

	// fifo flag/data
	rd_en_o					,
	rd_data_i				,
	a_empty_i				,
	empty_i					,

	// avalon-mm master I/F
	avmm_address_o			,
	avmm_begintrans_o		,
	avmm_burstcount_o		,
	avmm_write_o			,
	avmm_writedata_o		,
	avmm_waitrequest_i		,

	// arbiter I/F
	bus_req_o				,
	bus_grant_i				,
	bus_busy_o				,

	// memory read module I/F
	wfrm_cnt_o				,
	rfrm_cnt_i				,
	rd_frm_vld_o			,

	frm0o_wdt_o				,
	frm0o_hgt_o				,
	frm0o_adr_o				,
	frm0e_wdt_o				,
	frm0e_hgt_o				,
	frm0e_adr_o				,

	frm1o_wdt_o				,
	frm1o_hgt_o				,
	frm1o_adr_o				,
	frm1e_wdt_o				,
	frm1e_hgt_o				,
	frm1e_adr_o				,

	frm2o_wdt_o				,
	frm2o_hgt_o				,
	frm2o_adr_o				,
	frm2e_wdt_o				,
	frm2e_hgt_o				,
	frm2e_adr_o				,

	frm3o_wdt_o				,
	frm3o_hgt_o				,
	frm3o_adr_o				,
	frm3e_wdt_o				,
	frm3e_hgt_o				,
	frm3e_adr_o				,

	frm0_mode_o				,
	frm1_mode_o				,
	frm2_mode_o				,
	frm3_mode_o

);

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------

	parameter BASE_ADR		= 32'h0000_0000					;
	parameter STEP_ADR		= 32'h0080_0000					;
	parameter LDATA_BIT		= 128							;
	parameter LADR_BIT		= 25							;
	parameter SIZE_BIT		= 11							;
	parameter MODE			= 0								;
	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

	parameter LBST_BIT		= 5								;
	parameter ADR_CNT_BIT	= 18							;

	parameter FRAME1_BASE	= BASE_ADR + ( STEP_ADR * 0 )	;
	parameter FRAME2_BASE	= BASE_ADR + ( STEP_ADR * 1 )	;
	parameter FRAME3_BASE	= BASE_ADR + ( STEP_ADR * 2 )	;
	parameter FRAME4_BASE	= BASE_ADR + ( STEP_ADR * 3 )	;
	parameter FRAME5_BASE	= BASE_ADR + ( STEP_ADR * 4 )	;
	parameter FRAME6_BASE	= BASE_ADR + ( STEP_ADR * 5 )	;
	parameter FRAME7_BASE	= BASE_ADR + ( STEP_ADR * 6 )	;
	parameter FRAME8_BASE	= BASE_ADR + ( STEP_ADR * 7 )	;

// =============================================================================
// PORT DECLARATION
// =============================================================================

	// global signal
	input	wire							clk					;
	input	wire							rst_n				;
	input	wire							srst				;
	input	wire							enable				;
	input	wire							param_update_i		;
	input	wire							scan_mode_i			;

	// control signal
	input	wire							frame_start_i		;
	input	wire							frame_end_i			;
	input	wire							field_i				;
	input	wire	[  SIZE_BIT-1 : 0 ]		width_i				;
	input	wire	[  SIZE_BIT-1 : 0 ]		height_i			;
	input	wire							pix_lack_i			;
	output	wire							trans_done_o		;

	// fifo flag/data
	output	wire							rd_en_o				;
	input	wire	[ LDATA_BIT-1 : 0 ]		rd_data_i			;
	input	wire							a_empty_i			;
	input	wire							empty_i				;

	// avalon-mm master I/F
	output	wire	[  LADR_BIT-1 : 0 ]		avmm_address_o		;
	output wire								avmm_begintrans_o	;
	output	wire	[  LBST_BIT-1 : 0 ]		avmm_burstcount_o	;
	output	wire							avmm_write_o		;
	output	wire	[ LDATA_BIT-1 : 0 ]		avmm_writedata_o	;
	input	wire							avmm_waitrequest_i	;

	output	wire							bus_req_o			;
	input	wire							bus_grant_i			;
	output	wire							bus_busy_o			;

	output	wire	[ 2 : 0 ]				wfrm_cnt_o			;
	input	wire	[ 2 : 0 ]				rfrm_cnt_i			;
	output	wire							rd_frm_vld_o		;

	output	wire	[ SIZE_BIT-1 : 0 ]		frm0o_wdt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm0e_wdt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm1o_wdt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm1e_wdt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm2o_wdt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm2e_wdt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm3o_wdt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm3e_wdt_o			;

	output	wire	[ SIZE_BIT-1 : 0 ]		frm0o_hgt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm0e_hgt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm1o_hgt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm1e_hgt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm2o_hgt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm2e_hgt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm3o_hgt_o			;
	output	wire	[ SIZE_BIT-1 : 0 ]		frm3e_hgt_o			;

	output	wire	[ LADR_BIT-1 : 0 ]		frm0o_adr_o			;
	output	wire	[ LADR_BIT-1 : 0 ]		frm0e_adr_o			;
	output	wire	[ LADR_BIT-1 : 0 ]		frm1o_adr_o			;
	output	wire	[ LADR_BIT-1 : 0 ]		frm1e_adr_o			;
	output	wire	[ LADR_BIT-1 : 0 ]		frm2o_adr_o			;
	output	wire	[ LADR_BIT-1 : 0 ]		frm2e_adr_o			;
	output	wire	[ LADR_BIT-1 : 0 ]		frm3o_adr_o			;
	output	wire	[ LADR_BIT-1 : 0 ]		frm3e_adr_o			;

	output	wire							frm0_mode_o			;
	output	wire							frm1_mode_o			;
	output	wire							frm2_mode_o			;
	output	wire							frm3_mode_o			;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg									scan_mode_lat_ff	;
	reg									scan_mode_ff		;

	reg									fs_ff				;
	wire								trans_end			;
	reg									trans_done_ff		;
	reg									write_end_ff		;
	reg		[ LDATA_BIT-1 : 0 ]			avmm_wdata_ff		;
	reg		[  LBST_BIT-1 : 0 ]			burst_count_ff		;
//	reg		[  LBST_BIT-1 : 0 ]			avmm_bstcnt_ff		;
	reg									avmm_begintrans_ff	;

	reg									en_ff				;
	reg									en_dly1_ff			;
	reg									en_dly2_ff			;
	reg									bus_req_ff			;

	reg									rd_data_vld_ff		;
	reg									rd_data_lat_en_ff	;
	reg		[ LDATA_BIT-1 : 0 ]			rd_data_lat_ff		;

	reg		[  LADR_BIT-1 : 0 ]			adr_ff				;
	reg		[  LADR_BIT-1 : 0 ]			adr_cnt_ff			;
	wire	[  LADR_BIT-1 : 0 ]			base_address		;

	wire								ren_timing_1		;
	wire								ren_timing_2		;
	wire								rdata_lat_timing	;
	wire								wt_data_en			;

	reg		[ SIZE_BIT-1 : 0 ]			width_i_ff			;
	reg		[ SIZE_BIT-1 : 0 ]			height_i_ff			;
	reg									field_i_ff			;

	reg		[ SIZE_BIT-1 : 0 ]			frm0o_wdt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm0e_wdt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm1o_wdt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm1e_wdt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm2o_wdt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm2e_wdt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm3o_wdt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm3e_wdt_ff		;

	reg		[ SIZE_BIT-1 : 0 ]			frm0o_hgt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm0e_hgt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm1o_hgt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm1e_hgt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm2o_hgt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm2e_hgt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm3o_hgt_ff		;
	reg		[ SIZE_BIT-1 : 0 ]			frm3e_hgt_ff		;

	reg		[ LADR_BIT-1 : 0 ]			frm0o_adr_ff		;
	reg		[ LADR_BIT-1 : 0 ]			frm0e_adr_ff		;
	reg		[ LADR_BIT-1 : 0 ]			frm1o_adr_ff		;
	reg		[ LADR_BIT-1 : 0 ]			frm1e_adr_ff		;
	reg		[ LADR_BIT-1 : 0 ]			frm2o_adr_ff		;
	reg		[ LADR_BIT-1 : 0 ]			frm2e_adr_ff		;
	reg		[ LADR_BIT-1 : 0 ]			frm3o_adr_ff		;
	reg		[ LADR_BIT-1 : 0 ]			frm3e_adr_ff		;

	reg									frm0_mode_ff		;
	reg									frm1_mode_ff		;
	reg									frm2_mode_ff		;
	reg									frm3_mode_ff		;

	reg		[          2 : 0 ]			wfrm_cnt_ff			;
	reg									odd_fld_vld_ff		;
	reg									fld_hld_ff			;
	reg									rd_frm_vld_ff		;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	// -----------------------------
	//	reflect status to internal
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			scan_mode_lat_ff	<= 1'b0 ;
		end
		else if ( param_update_i || srst ) begin
			scan_mode_lat_ff	<= scan_mode_i ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			scan_mode_ff	<= 1'b0 ;
		end
		else if ( trans_done_ff || srst ) begin
			scan_mode_ff	<= scan_mode_lat_ff ;
		end
	end

	// -----------------------------
	//	 status hold register
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			fs_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			fs_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			fs_ff	<= frame_start_i ;
		end
	end
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			width_i_ff	<= {SIZE_BIT{1'b0}}	;
			height_i_ff	<= {SIZE_BIT{1'b0}}	;
			field_i_ff	<= 1'b0 			;
		end
		else if ( srst ) begin
			width_i_ff	<= {SIZE_BIT{1'b0}}	;
			height_i_ff	<= {SIZE_BIT{1'b0}}	;
			field_i_ff	<= 1'b0 			;
		end
		else if ( enable ) begin
			if ( fs_ff ) begin
				width_i_ff	<= width_i	;
				height_i_ff	<= height_i	;
				field_i_ff	<= field_i	;
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			frm0o_wdt_ff	<= 11'd0000			;
			frm0e_wdt_ff	<= 11'd0000			;
			frm1o_wdt_ff	<= 11'd0000			;
			frm1e_wdt_ff	<= 11'd0000			;
			frm2o_wdt_ff	<= 11'd0000			;
			frm2e_wdt_ff	<= 11'd0000			;
			frm3o_wdt_ff	<= 11'd0000			;
			frm3e_wdt_ff	<= 11'd0000			;
			frm0o_hgt_ff	<= 11'd0000			;
			frm0e_hgt_ff	<= 11'd0000			;
			frm1o_hgt_ff	<= 11'd0000			;
			frm1e_hgt_ff	<= 11'd0000			;
			frm2o_hgt_ff	<= 11'd0000			;
			frm2e_hgt_ff	<= 11'd0000			;
			frm3o_hgt_ff	<= 11'd0000			;
			frm3e_hgt_ff	<= 11'd0000			;
			frm0o_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm0e_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm1o_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm1e_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm2o_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm2e_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm3o_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm3e_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm0_mode_ff	<= 1'b0				;
			frm1_mode_ff	<= 1'b0				;
			frm2_mode_ff	<= 1'b0				;
			frm3_mode_ff	<= 1'b0				;
		end
		else if ( srst ) begin
			frm0o_wdt_ff	<= 11'd0000			;
			frm0e_wdt_ff	<= 11'd0000			;
			frm1o_wdt_ff	<= 11'd0000			;
			frm1e_wdt_ff	<= 11'd0000			;
			frm2o_wdt_ff	<= 11'd0000			;
			frm2e_wdt_ff	<= 11'd0000			;
			frm3o_wdt_ff	<= 11'd0000			;
			frm3e_wdt_ff	<= 11'd0000			;
			frm0o_hgt_ff	<= 11'd0000			;
			frm0e_hgt_ff	<= 11'd0000			;
			frm1o_hgt_ff	<= 11'd0000			;
			frm1e_hgt_ff	<= 11'd0000			;
			frm2o_hgt_ff	<= 11'd0000			;
			frm2e_hgt_ff	<= 11'd0000			;
			frm3o_hgt_ff	<= 11'd0000			;
			frm3e_hgt_ff	<= 11'd0000			;
			frm0o_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm0e_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm1o_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm1e_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm2o_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm2e_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm3o_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm3e_adr_ff	<= {LADR_BIT{1'b0}}	;
			frm0_mode_ff	<= 1'b0				;
			frm1_mode_ff	<= 1'b0				;
			frm2_mode_ff	<= 1'b0				;
			frm3_mode_ff	<= 1'b0				;
		end
		else if ( enable ) begin
			if ( write_end_ff && !pix_lack_i ) begin
				if ( wfrm_cnt_o == 3'd0 ) begin
					frm0o_wdt_ff		<= width_i_ff	;
					frm0o_hgt_ff		<= height_i_ff	;
					frm0o_adr_ff		<= adr_cnt_ff	;
					frm0_mode_ff		<= scan_mode_ff	;
				end
				else if ( wfrm_cnt_o == 3'd1 ) begin
					frm0e_wdt_ff		<= width_i_ff	;
					frm0e_hgt_ff		<= height_i_ff	;
					frm0e_adr_ff		<= adr_cnt_ff	;
				end
				else if ( wfrm_cnt_o == 3'd2 ) begin
					frm1o_wdt_ff		<= width_i_ff	;
					frm1o_hgt_ff		<= height_i_ff	;
					frm1o_adr_ff		<= adr_cnt_ff	;
					frm1_mode_ff		<= scan_mode_ff	;
				end
				else if ( wfrm_cnt_o == 3'd3 ) begin
					frm1e_wdt_ff		<= width_i_ff	;
					frm1e_hgt_ff		<= height_i_ff	;
					frm1e_adr_ff		<= adr_cnt_ff	;
				end
				else if ( wfrm_cnt_o == 3'd4 ) begin
					frm2o_wdt_ff		<= width_i_ff	;
					frm2o_hgt_ff		<= height_i_ff	;
					frm2o_adr_ff		<= adr_cnt_ff	;
					frm2_mode_ff		<= scan_mode_ff	;
				end
				else if ( wfrm_cnt_o == 3'd5 ) begin
					frm2e_wdt_ff		<= width_i_ff	;
					frm2e_hgt_ff		<= height_i_ff	;
					frm2e_adr_ff		<= adr_cnt_ff	;
				end
				else if ( wfrm_cnt_o == 3'd6 ) begin
					frm3o_wdt_ff		<= width_i_ff	;
					frm3o_hgt_ff		<= height_i_ff	;
					frm3o_adr_ff		<= adr_cnt_ff	;
					frm3_mode_ff		<= scan_mode_ff	;
				end
				else if ( wfrm_cnt_o == 3'd7 ) begin
					frm3e_wdt_ff		<= width_i_ff	;
					frm3e_hgt_ff		<= height_i_ff	;
					frm3e_adr_ff		<= adr_cnt_ff	;
				end
			end
		end
	end

	assign frm0o_wdt_o	= frm0o_wdt_ff ;
	assign frm0o_hgt_o	= frm0o_hgt_ff ;
	assign frm0o_adr_o	= frm0o_adr_ff ;
	assign frm0e_wdt_o	= ( frm0_mode_ff ) ? frm0e_wdt_ff : frm0o_wdt_ff ;
	assign frm0e_hgt_o	= ( frm0_mode_ff ) ? frm0e_hgt_ff : frm0o_hgt_ff ;
	assign frm0e_adr_o	= ( frm0_mode_ff ) ? frm0e_adr_ff : frm0o_adr_ff ;

	assign frm1o_wdt_o	= frm1o_wdt_ff ;
	assign frm1o_hgt_o	= frm1o_hgt_ff ;
	assign frm1o_adr_o	= frm1o_adr_ff ;
	assign frm1e_wdt_o	= ( frm1_mode_ff ) ? frm1e_wdt_ff : frm1o_wdt_ff ;
	assign frm1e_hgt_o	= ( frm1_mode_ff ) ? frm1e_hgt_ff : frm1o_hgt_ff ;
	assign frm1e_adr_o	= ( frm1_mode_ff ) ? frm1e_adr_ff : frm1o_adr_ff ;

	assign frm2o_wdt_o	= frm2o_wdt_ff ;
	assign frm2o_hgt_o	= frm2o_hgt_ff ;
	assign frm2o_adr_o	= frm2o_adr_ff ;
	assign frm2e_wdt_o	= ( frm2_mode_ff ) ? frm2e_wdt_ff : frm2o_wdt_ff ;
	assign frm2e_hgt_o	= ( frm2_mode_ff ) ? frm2e_hgt_ff : frm2o_hgt_ff ;
	assign frm2e_adr_o	= ( frm2_mode_ff ) ? frm2e_adr_ff : frm2o_adr_ff ;

	assign frm3o_wdt_o	= frm3o_wdt_ff ;
	assign frm3o_hgt_o	= frm3o_hgt_ff ;
	assign frm3o_adr_o	= frm3o_adr_ff ;
	assign frm3e_wdt_o	= ( frm3_mode_ff ) ? frm3e_wdt_ff : frm3o_wdt_ff ;
	assign frm3e_hgt_o	= ( frm3_mode_ff ) ? frm3e_hgt_ff : frm3o_hgt_ff ;
	assign frm3e_adr_o	= ( frm3_mode_ff ) ? frm3e_adr_ff : frm3o_adr_ff ;

	assign frm0_mode_o	= frm0_mode_ff ;
	assign frm1_mode_o	= frm1_mode_ff ;
	assign frm2_mode_o	= frm2_mode_ff ;
	assign frm3_mode_o	= frm3_mode_ff ;



	generate
		if ( MODE == 0 ) begin

			// -----------------------------
			//	odd field valid
			// -----------------------------
			// odd field write access is completed successfully
			always @( posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					odd_fld_vld_ff	<= 1'b0 ;
				end
				else if ( srst ) begin
					odd_fld_vld_ff	<= 1'b0 ;
				end
				else if ( enable ) begin
					if ( write_end_ff && !pix_lack_i && !field_i_ff && scan_mode_ff ) begin
						odd_fld_vld_ff	<= 1'b1 ;
					end
					else if ( write_end_ff ) begin
						odd_fld_vld_ff	<= 1'b0 ;
					end
				end
			end

			// -----------------------------
			//	write frame counter
			// -----------------------------
			always @( posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					wfrm_cnt_ff		<= 3'd0 ;
				end
				else if ( srst ) begin
					wfrm_cnt_ff		<= 3'd0 ;
				end
				else if ( enable ) begin
					if ( write_end_ff && !pix_lack_i ) begin
						if ( scan_mode_ff ) begin
							if ( field_i_ff && odd_fld_vld_ff && ( ( rfrm_cnt_i[2:1] != ( wfrm_cnt_o[2:1] + 2'd1 ) ) ) ) begin
								wfrm_cnt_ff	<= wfrm_cnt_ff + 3'd2 ;
							end
						end
						else begin
							if ( !field_i_ff && ( rfrm_cnt_i != wfrm_cnt_o + 3'd2 ) ) begin
								wfrm_cnt_ff	<= wfrm_cnt_ff + 3'd2 ;
							end
						end
					end
				end
			end

			assign wfrm_cnt_o = wfrm_cnt_ff | field_i_ff ;

			// -----------------------------
			//	read valid
			// -----------------------------
			always @( posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					rd_frm_vld_ff	<= 1'b0 ;
				end
				else if ( srst ) begin
					rd_frm_vld_ff	<= 1'b0 ;
				end
				else if ( enable ) begin
					if ( write_end_ff && !pix_lack_i && ( (  scan_mode_ff &&  field_i_ff && ( wfrm_cnt_o == 3'd1 ) && odd_fld_vld_ff )
													   || ( !scan_mode_ff && !field_i_ff && ( wfrm_cnt_o == 3'd0 ) ) ) ) begin
						rd_frm_vld_ff	<= 1'b1 ;
					end
				end
			end

			assign rd_frm_vld_o = rd_frm_vld_ff ;

		end

		else begin

			// -----------------------------
			//	field hold
			// -----------------------------
			always @( posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					fld_hld_ff	<= 1'b0 ;
				end
				else if ( srst ) begin
					fld_hld_ff	<= 1'b0 ;
				end
				else if ( enable ) begin
					if ( !scan_mode_ff ) begin
						fld_hld_ff	<= 1'b0 ;
					end
					else if ( write_end_ff && !pix_lack_i && ( fld_hld_ff == field_i_ff ) ) begin
						fld_hld_ff	<= ~fld_hld_ff ;
					end
				end
			end

			// -----------------------------
			//	write frame counter
			// -----------------------------
			always @( posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					wfrm_cnt_ff	<= 3'd0 ;
				end
				else if ( srst ) begin
					wfrm_cnt_ff	<= 3'd0 ;
				end
				else if ( enable ) begin
					if ( write_end_ff && !pix_lack_i ) begin
						if ( scan_mode_ff ) begin
							if ( ( fld_hld_ff == field_i_ff ) && ( rfrm_cnt_i != wfrm_cnt_ff + 3'd1 ) ) begin
								wfrm_cnt_ff	<= wfrm_cnt_ff + 3'd1 ;
							end
						end
						else begin
							if ( wfrm_cnt_ff[0] == 1'b1 ) begin
								wfrm_cnt_ff	<= wfrm_cnt_ff + 3'd1 ;
							end
							else if ( !field_i_ff && ( rfrm_cnt_i != wfrm_cnt_o + 3'd2 ) ) begin
								wfrm_cnt_ff	<= wfrm_cnt_ff + 3'd2 ;
							end
						end
					end
				end
			end

			assign wfrm_cnt_o = wfrm_cnt_ff ;

			// -----------------------------
			//	read valid
			// -----------------------------
			always @( posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					rd_frm_vld_ff	<= 1'b0 ;
				end
				else if ( srst ) begin
					rd_frm_vld_ff	<= 1'b0 ;
				end
				else if ( enable ) begin
					if ( write_end_ff && !pix_lack_i && !field_i_ff && ( wfrm_cnt_o == 3'd0 ) ) begin
						rd_frm_vld_ff	<= 1'b1 ;
					end
				end
			end

			assign rd_frm_vld_o = rd_frm_vld_ff ;

		end

	endgenerate


	// -----------------------------
	//	operation setting signal
	// -----------------------------

//	always @( posedge clk or negedge rst_n ) begin
//		if ( !rst_n ) begin
//			avmm_bstcnt_ff	<= 5'h00 ;
//		end
//		else if ( srst ) begin
//			avmm_bstcnt_ff	<= 5'h00 ;
//		end
//		else if ( enable ) begin
//			avmm_bstcnt_ff	<= 5'h10 ;
//		end
//	end

//	assign avmm_burstcount_o	= avmm_bstcnt_ff ;

	assign avmm_burstcount_o	= 5'h10 ;

	// -----------------------------
	//	begintranstransfer
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			avmm_begintrans_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			avmm_begintrans_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			avmm_begintrans_ff	<= en_dly1_ff & ~en_dly2_ff ;
		end
	end

	assign avmm_begintrans_o = avmm_begintrans_ff;

	// -----------------------------
	//	transaction request
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			bus_req_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			bus_req_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( !avmm_write_o && !bus_busy_o && !bus_grant_i && ( !a_empty_i || ( frame_end_i && !empty_i ) ) ) begin
				bus_req_ff	<= 1'b1 ;
			end
			else if ( bus_grant_i ) begin
				bus_req_ff	<= 1'b0 ;
			end
		end
	end

	assign	bus_req_o	= bus_req_ff ;
	assign	bus_busy_o	= en_ff | en_dly1_ff | en_dly2_ff ;

	// -----------------------------
	//	transaction enable
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			en_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			en_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( !en_ff && bus_req_ff && bus_grant_i ) begin
				en_ff	<= 1'b1 ;
			end
			if ( trans_end && !avmm_waitrequest_i ) begin
				en_ff	<= 1'b0 ;
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			en_dly1_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			en_dly1_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( avmm_waitrequest_i  && en_dly1_ff ) begin
				en_dly1_ff	<= en_dly1_ff ;
			end
			else begin
				en_dly1_ff	<= en_ff ;
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			en_dly2_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			en_dly2_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( avmm_waitrequest_i && en_dly2_ff ) begin
				en_dly2_ff	<= en_dly2_ff ;
			end
			else begin
				en_dly2_ff	<= en_dly1_ff ;
			end
		end
	end

	assign avmm_write_o 	 = en_dly2_ff ;

	assign ren_timing_1	 	 =  en_ff      && !en_dly1_ff ;
	assign ren_timing_2	 	 =  en_dly1_ff && !en_dly2_ff ;
	assign rdata_lat_timing  = !en_ff      &&  en_dly1_ff && !avmm_waitrequest_i ;

	assign rd_en_o			 = ren_timing_1 | ren_timing_2     | ( en_ff && !avmm_waitrequest_i ) ;
	assign wt_data_en		 = ren_timing_2 | rdata_lat_timing | ( en_ff && !avmm_waitrequest_i ) ;

	// -----------------------------
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rd_data_vld_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			rd_data_vld_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			rd_data_vld_ff	<= rd_en_o ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rd_data_lat_en_ff	<= 1'b0 ;
			rd_data_lat_ff		<= { LDATA_BIT{1'b0} } ;
		end
		else if ( srst ) begin
			rd_data_lat_en_ff	<= 1'b0 ;
			rd_data_lat_ff		<= { LDATA_BIT{1'b0} } ;
		end
		else if ( enable ) begin
			if ( rd_data_vld_ff && avmm_waitrequest_i ) begin
				rd_data_lat_en_ff	<= 1'b1 ;
				rd_data_lat_ff		<= rd_data_i ;
			end
			else if ( rd_data_lat_en_ff && !avmm_waitrequest_i ) begin
				rd_data_lat_en_ff	<= 1'b0 ;
				rd_data_lat_ff		<= { LDATA_BIT{1'b0} } ;
			end
		end
	end

	// -----------------------------
	//	write data
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			avmm_wdata_ff	<= {LDATA_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			avmm_wdata_ff	<= {LDATA_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( rd_data_lat_en_ff && !avmm_waitrequest_i ) begin
				avmm_wdata_ff	<= rd_data_lat_ff ;
			end
			else if ( wt_data_en ) begin
				avmm_wdata_ff	<= rd_data_i ;
			end
			else begin
				avmm_wdata_ff	<= avmm_wdata_ff ;
			end
		end
	end

	assign avmm_writedata_o	= avmm_wdata_ff ;

	// -----------------------------
	//	burst count
	// -----------------------------
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			burst_count_ff	<= 4'h0 ;
		end
		else if ( srst ) begin
			burst_count_ff	<= 4'h0 ;
		end
		else if ( enable ) begin
			if ( rd_en_o ) begin
				if ( burst_count_ff == 4'hf ) begin
					burst_count_ff	<= 4'h0 ;
				end
				else begin
					burst_count_ff	<= burst_count_ff + 1'b1 ;
				end
			end
		end
	end

	assign	trans_end = ( burst_count_ff == 4'hf ) ? 1'b1 : 1'b0 ;

	// -----------------------------
	//	address
	// -----------------------------
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			adr_ff	<= {LADR_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			adr_ff	<= {LADR_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( frame_start_i ) begin
				adr_ff	<= {LADR_BIT{1'b0}} ;
			end
			else if ( en_ff && !en_dly1_ff ) begin
				adr_ff	<= base_address + adr_cnt_ff ;
			end
		end
	end

	assign avmm_address_o	= adr_ff ;

	// -----------------------------
	//	address counter
	// -----------------------------
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			adr_cnt_ff	<= {ADR_CNT_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			adr_cnt_ff	<= {ADR_CNT_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( frame_start_i ) begin
				adr_cnt_ff	<= {ADR_CNT_BIT{1'b0}} ;
			end
			else if ( avmm_write_o && !avmm_waitrequest_i ) begin
				adr_cnt_ff	<= adr_cnt_ff + 1'b1 ;
			end
		end
	end

	assign base_address			= ( {LADR_BIT{(wfrm_cnt_o == 3'b000)}} & FRAME1_BASE[LADR_BIT-1:0] )
								| ( {LADR_BIT{(wfrm_cnt_o == 3'b001)}} & FRAME2_BASE[LADR_BIT-1:0] )
								| ( {LADR_BIT{(wfrm_cnt_o == 3'b010)}} & FRAME3_BASE[LADR_BIT-1:0] )
								| ( {LADR_BIT{(wfrm_cnt_o == 3'b011)}} & FRAME4_BASE[LADR_BIT-1:0] )
								| ( {LADR_BIT{(wfrm_cnt_o == 3'b100)}} & FRAME5_BASE[LADR_BIT-1:0] )
								| ( {LADR_BIT{(wfrm_cnt_o == 3'b101)}} & FRAME6_BASE[LADR_BIT-1:0] )
								| ( {LADR_BIT{(wfrm_cnt_o == 3'b110)}} & FRAME7_BASE[LADR_BIT-1:0] )
								| ( {LADR_BIT{(wfrm_cnt_o == 3'b111)}} & FRAME8_BASE[LADR_BIT-1:0] ) ;

	// -----------------------------
	//	transaction done
	// -----------------------------
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			trans_done_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			trans_done_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( frame_end_i && !en_ff && empty_i ) begin
				trans_done_ff	<= 1'b1 ;
			end
			else begin
				trans_done_ff	<= 1'b0 ;
			end
		end
	end

	assign trans_done_o	= trans_done_ff ;

	// -----------------------------
	//	write end
	// -----------------------------
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			write_end_ff	<= 1'b0;
		end
		else if ( srst ) begin
			write_end_ff	<= 1'b0;
		end
		else if ( enable ) begin
			write_end_ff	<= frame_end_i & !en_ff & empty_i & !trans_done_ff ;
		end
	end


endmodule
`default_nettype wire
