//----------------------------------------------------------------------------------------
//Copyright (C) 2014 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		memory read i/f control module
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Dec.  1 2014	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		BASE_ADR				: logical base address for frame memory ( word address )
//		STEP_ADR				: address step between each frame ( word address )
//		LDATA_BIT				: high-performance controller local data bit width
//		LADR_BIT				: high-performance controller local address bit width
//		SIZE_BIT				: size setting signal bit width
//		MODE					: set bank access
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk						: clock for all circuit
//		rst_n					: asyncronous reset input
//		srst					: software reset (sync with clk)
//		enable					: clock enable
//
//		rd_start_i				: frame start of read frame data
//		field_o					: field of output data
//		fdata_rd_o				: one frame read operation end
//		width_o					: width of output data
//		height_o				: height of output data
//
//		wr_en_o					: fifo write enable
//		wr_data_o				: fifo write data
//		a_full_i				: fifo almost full flag
//
//		avmm_address_o			: high-performance controller local address
//		avmm_begintrans_o		: high-performance controller local burst begin
//		avmm_burstcount_o		: high-performance controller local burst size
//		avmm_read_o				: high-performance controller local read request
//		avmm_readdatavalid_i	: high-performance controller local read data valid
//		avmm_readdata_i			: high-performance controller local read data
//		avmm_waitrequest_i		: high-performance controller wait request
//
//		bus_req_o				: Avalon-MM bus request to memory controller
//		bus_grant_i				: Avalon-MM bus grant from memory controller
//		bus_busy_o				: Avalon-MM bus busy to memory controller
//
//		rd_frm_vld_i			: read can be started
//		rd_mode_o				: read frame information to timing gen module
//		wfrm_cnt_i				: next write frame number
//		rfrm_cnt_o				: next read frame number
//
//		frm*o_wdt_i				: width of a frame in bank* odd field
//		frm*o_hgt_i				: height of a frame in bank* odd field
//		frm*o_adr_i				: address of a frame in bank* odd field
//		frm*e_wdt_i				: width of a frame in bank* even field
//		frm*e_hgt_i				: height of a frame in bank* even field
//		frm*e_adr_i				: address of a frame in bank* even field
//		frm*_mode_i				: scan mode of a frame
//----------------------------------------------------------------------------------------
`default_nettype none
`timescale 1ns / 1ps

module	sfb16_mem_read_ctl (
	// global signal
	clk					,
	rst_n				,
	srst				,
	enable				,

	// control signal
	rd_start_i			,
	field_i				,
	fdata_rd_o			,
	width_o				,
	height_o			,

	// fifo flag/data
	wr_en_o				,
	wr_data_o			,
	a_full_i			,

	// memory controller i/f
	avmm_address_o		,
	avmm_begintrans_o	,
	avmm_burstcount_o	,
	avmm_read_o			,
	avmm_readdatavalid_i,
	avmm_readdata_i		,
	avmm_waitrequest_i	,

	// arbiter module i/f
	bus_req_o			,
	bus_grant_i			,
	bus_busy_o			,


	// memory write module i/f
	rd_frm_vld_i		,
	rd_mode_o			,
	wfrm_cnt_i			,
	rfrm_cnt_o			,

	frm0o_wdt_i			,
	frm0o_hgt_i			,
	frm0o_adr_i			,
	frm0e_wdt_i			,
	frm0e_hgt_i			,
	frm0e_adr_i			,

	frm1o_wdt_i			,
	frm1o_hgt_i			,
	frm1o_adr_i			,
	frm1e_wdt_i			,
	frm1e_hgt_i			,
	frm1e_adr_i			,

	frm2o_wdt_i			,
	frm2o_hgt_i			,
	frm2o_adr_i			,
	frm2e_wdt_i			,
	frm2e_hgt_i			,
	frm2e_adr_i			,

	frm3o_wdt_i			,
	frm3o_hgt_i			,
	frm3o_adr_i			,
	frm3e_wdt_i			,
	frm3e_hgt_i			,
	frm3e_adr_i			,

	frm0_mode_i			,
	frm1_mode_i			,
	frm2_mode_i			,
	frm3_mode_i

);

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------

	parameter BASE_ADR		= 32'h0000_0000					;
	parameter STEP_ADR		= 32'h0080_0000					;
	parameter LDATA_BIT		= 128 							;
	parameter LADR_BIT		= 25 							;
	parameter SIZE_BIT		= 11							;
	parameter MODE			= 0								;
	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

	parameter LBST_BIT		=  5 							;
	parameter ADR_CNT_BIT	= 18 							;

	parameter FRAME1_BASE	= BASE_ADR + ( STEP_ADR * 0 )	;
	parameter FRAME2_BASE	= BASE_ADR + ( STEP_ADR * 1 )	;
	parameter FRAME3_BASE	= BASE_ADR + ( STEP_ADR * 2 )	;
	parameter FRAME4_BASE	= BASE_ADR + ( STEP_ADR * 3 )	;
	parameter FRAME5_BASE	= BASE_ADR + ( STEP_ADR * 4 )	;
	parameter FRAME6_BASE	= BASE_ADR + ( STEP_ADR * 5 )	;
	parameter FRAME7_BASE	= BASE_ADR + ( STEP_ADR * 6 )	;
	parameter FRAME8_BASE	= BASE_ADR + ( STEP_ADR * 7 )	;

// =============================================================================
// PORT DECLARATION
// =============================================================================

	// global signal
	input	wire							clk						;
	input	wire							rst_n					;
	input	wire							srst					;
	input	wire							enable					;

	// control signal
	input	wire							rd_start_i				;
	input	wire							field_i					;
	output	wire							fdata_rd_o				;
	output	wire	[  SIZE_BIT-1 : 0 ]		width_o					;
	output	wire	[  SIZE_BIT-1 : 0 ]		height_o				;

	// fifo flag/data
	output	wire							wr_en_o					;
	output	wire	[ LDATA_BIT-1 : 0 ]		wr_data_o				;
	input	wire							a_full_i				;

	// Avalon-MM Master I/F
	output	wire	[  LADR_BIT-1 : 0 ]		avmm_address_o			;
	output wire								avmm_begintrans_o		;
	output	wire	[  LBST_BIT-1 : 0 ]		avmm_burstcount_o		;
	output	wire							avmm_read_o				;
	input	wire							avmm_readdatavalid_i	;
	input	wire	[ LDATA_BIT-1 : 0 ]		avmm_readdata_i			;
	input	wire							avmm_waitrequest_i		;

	output	wire							bus_req_o				;
	input	wire							bus_grant_i				;
	output	wire							bus_busy_o				;

	input	wire	[ 2 : 0 ]				wfrm_cnt_i				;
	output	wire	[ 2 : 0 ]				rfrm_cnt_o				;
	input	wire							rd_frm_vld_i			;
	output	wire							rd_mode_o				;

	input	wire	[ SIZE_BIT-1 : 0 ]		frm0o_wdt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm0e_wdt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm1o_wdt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm1e_wdt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm2o_wdt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm2e_wdt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm3o_wdt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm3e_wdt_i				;

	input	wire	[ SIZE_BIT-1 : 0 ]		frm0o_hgt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm0e_hgt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm1o_hgt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm1e_hgt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm2o_hgt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm2e_hgt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm3o_hgt_i				;
	input	wire	[ SIZE_BIT-1 : 0 ]		frm3e_hgt_i				;

	input	wire	[ LADR_BIT-1 : 0 ]		frm0o_adr_i				;
	input	wire	[ LADR_BIT-1 : 0 ]		frm0e_adr_i				;
	input	wire	[ LADR_BIT-1 : 0 ]		frm1o_adr_i				;
	input	wire	[ LADR_BIT-1 : 0 ]		frm1e_adr_i				;
	input	wire	[ LADR_BIT-1 : 0 ]		frm2o_adr_i				;
	input	wire	[ LADR_BIT-1 : 0 ]		frm2e_adr_i				;
	input	wire	[ LADR_BIT-1 : 0 ]		frm3o_adr_i				;
	input	wire	[ LADR_BIT-1 : 0 ]		frm3e_adr_i				;

	input	wire							frm0_mode_i				;
	input	wire							frm1_mode_i				;
	input	wire							frm2_mode_i				;
	input	wire							frm3_mode_i				;


// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg									trans_enable_ff		;
	reg									en_ff				;
	reg									en_dly1_ff			;
	reg									busy_ff				;
	reg									bus_req_ff			;

	reg		[  LADR_BIT-1 : 0 ]			adr_ff				;
	reg		[  LADR_BIT-1 : 0 ]			adr_cnt_ff			;
	wire	[  LADR_BIT-1 : 0 ]			base_address		;
//	reg		[  LBST_BIT-1 : 0 ]			avmm_bstcnt_ff		;
	reg		[  LBST_BIT-1 : 0 ]			ack_cnt_ff			;
	reg									avmm_begintrans_ff	;

	reg									fdata_rd_ff			;
	reg		[  LADR_BIT-1 : 0 ]			rd_final_ff			;
	reg									rd_mode_ff			;
	reg		[          2 : 0 ]			rfrm_cnt_ff			;
	reg		[ SIZE_BIT-1 : 0 ]			width_ff			;
	reg		[ SIZE_BIT-1 : 0 ]			height_ff			;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	// -----------------------------
	//	transaction enable
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			trans_enable_ff		<= 1'd0;
		end
		else if ( srst ) begin
			trans_enable_ff		<= 1'd0;
		end
		else if ( enable ) begin
			if ( rd_start_i ) begin
				trans_enable_ff	<= 1'b1 ;
			end
			else if ( fdata_rd_o ) begin
				trans_enable_ff	<= 1'b0 ;
			end
		end
	end

	// -----------------------------
	//	read frame ring counter
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			rfrm_cnt_ff		<= 3'd0;
		end
		else if ( srst ) begin
			rfrm_cnt_ff		<= 3'd0;
		end
		else if ( enable && rd_frm_vld_i && rd_start_i ) begin
			if ( rd_mode_ff ) begin
				if ( MODE == 0 ) begin
					if ( !field_i ) begin
						if ( wfrm_cnt_i[0] == 1'd0 ) begin
							rfrm_cnt_ff	<= wfrm_cnt_i - 3'd2 ;
						end
						else if ( wfrm_cnt_i[0] == 1'd1 ) begin
							rfrm_cnt_ff	<= wfrm_cnt_i - 3'd3 ;
						end
					end
					else if ( field_i ) begin
						rfrm_cnt_ff	<= rfrm_cnt_ff + 3'd1 ;
					end
				end
				else if ( MODE == 1 ) begin
					rfrm_cnt_ff	<= wfrm_cnt_i - 3'd1 ;
				end
			end
			else begin
				if ( wfrm_cnt_i[0] == 1'd0 ) begin
					rfrm_cnt_ff	<= wfrm_cnt_i - 3'd2 ;
				end
				else if ( wfrm_cnt_i[0] == 1'd1 ) begin
					rfrm_cnt_ff	<= wfrm_cnt_i - 3'd3 ;
				end
			end
		end
	end

	assign rfrm_cnt_o	= rfrm_cnt_ff ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			width_ff	<= {(SIZE_BIT){1'b0}} ;
			height_ff	<= {(SIZE_BIT){1'b0}} ;
			rd_final_ff	<= {(LADR_BIT){1'b0}} ;
		end
		else if ( enable ) begin
			case ( rfrm_cnt_ff )
				3'd0	: begin
					width_ff	<= frm0o_wdt_i ;
					height_ff	<= frm0o_hgt_i ;
					rd_final_ff	<= frm0o_adr_i ;
					rd_mode_ff	<= frm0_mode_i ;
				end
				3'd1	: begin
					width_ff	<= frm0e_wdt_i ;
					height_ff	<= frm0e_hgt_i ;
					rd_final_ff	<= frm0e_adr_i ;
					rd_mode_ff	<= frm0_mode_i ;
				end
				3'd2	: begin
					width_ff	<= frm1o_wdt_i ;
					height_ff	<= frm1o_hgt_i ;
					rd_final_ff	<= frm1o_adr_i ;
					rd_mode_ff	<= frm1_mode_i ;
				end
				3'd3	: begin
					width_ff	<= frm1e_wdt_i ;
					height_ff	<= frm1e_hgt_i ;
					rd_final_ff	<= frm1e_adr_i ;
					rd_mode_ff	<= frm1_mode_i ;
				end
				3'd4	: begin
					width_ff	<= frm2o_wdt_i ;
					height_ff	<= frm2o_hgt_i ;
					rd_final_ff	<= frm2o_adr_i ;
					rd_mode_ff	<= frm2_mode_i ;
				end
				3'd5	: begin
					width_ff	<= frm2e_wdt_i ;
					height_ff	<= frm2e_hgt_i ;
					rd_final_ff	<= frm2e_adr_i ;
					rd_mode_ff	<= frm2_mode_i ;
				end
				3'd6	: begin
					width_ff	<= frm3o_wdt_i ;
					height_ff	<= frm3o_hgt_i ;
					rd_final_ff	<= frm3o_adr_i ;
					rd_mode_ff	<= frm3_mode_i ;
				end
				default	: begin
					width_ff	<= frm3e_wdt_i ;
					height_ff	<= frm3e_hgt_i ;
					rd_final_ff	<= frm3e_adr_i ;
					rd_mode_ff	<= frm3_mode_i ;
				end
			endcase
		end
	end

	assign width_o  	= width_ff  ;
	assign height_o 	= height_ff ;
	assign rd_mode_o	= rd_mode_ff ;


	// -----------------------------
	//	transration request
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			bus_req_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			bus_req_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( trans_enable_ff && !rd_start_i && !en_ff && !a_full_i && !busy_ff && !bus_grant_i && ( adr_cnt_ff < rd_final_ff ) && rd_frm_vld_i ) begin
				bus_req_ff	<= 1'b1 ;
			end
			else if ( bus_grant_i ) begin
				bus_req_ff	<= 1'b0 ;
			end
		end
	end

	assign	bus_req_o	= bus_req_ff ;

	// -----------------------------
	//	transration enable
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			en_ff	<= 1'b0;
		end
		else if ( srst ) begin
			en_ff	<= 1'b0;
		end
		else if ( enable ) begin
			if ( rd_start_i ) begin
				en_ff	<= 1'b0;
			end
			else if ( !en_ff && bus_req_ff && bus_grant_i ) begin
				en_ff	<= 1'b1;
			end
			else begin
				if ( !avmm_waitrequest_i ) begin
					en_ff	<= 1'b0;
				end
			end
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			en_dly1_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			en_dly1_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			en_dly1_ff	<= en_ff ;
		end
	end

	assign avmm_read_o = en_dly1_ff ;

	// -----------------------------
	//	transration busy
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			busy_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			busy_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( bus_req_ff && bus_grant_i ) begin
				busy_ff	<= 1'b1 ;
			end
			else if ( ack_cnt_ff == 4'hF && avmm_readdatavalid_i ) begin
				busy_ff	<= 1'b0 ;
			end
		end
	end

	assign bus_busy_o = busy_ff ;

	// -----------------------------
	//	begintranstransfer
	// -----------------------------
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			avmm_begintrans_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			avmm_begintrans_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			avmm_begintrans_ff	<= en_ff & ~en_dly1_ff ;
		end
	end

	assign avmm_begintrans_o = avmm_begintrans_ff;

	// -----------------------------
	//	operation setting signal
	// -----------------------------

//	always @( posedge clk or negedge rst_n ) begin
//		if ( !rst_n ) begin
//			avmm_bstcnt_ff	<= 5'h00 ;
//		end
//		else if ( srst ) begin
//			avmm_bstcnt_ff	<= 5'h00 ;
//		end
//		else if ( enable ) begin
//			avmm_bstcnt_ff	<= 5'h10 ;
//		end
//	end

//	assign avmm_burstcount_o	= avmm_bstcnt_ff ;

	assign avmm_burstcount_o	= 5'h10 ;

	// -----------------------------
	//	ack count
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			ack_cnt_ff	<= 4'h0 ;
		end
		else if ( srst ) begin
			ack_cnt_ff	<= 4'h0 ;
		end
		else if ( enable ) begin
			if ( avmm_readdatavalid_i ) begin
				if ( rd_start_i || ack_cnt_ff == 4'hF ) begin
					ack_cnt_ff	<= 4'h0 ;
				end
				else begin
					ack_cnt_ff	<= ack_cnt_ff + 1'b1 ;
				end
			end
		end
	end

	assign wr_en_o		= avmm_readdatavalid_i;
	assign wr_data_o	= avmm_readdata_i;

	// -----------------------------
	//	address
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			adr_ff	<= {LADR_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			adr_ff	<= {LADR_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( rd_start_i ) begin
				adr_ff	<= {LADR_BIT{1'b0}} ;
			end
			else if ( en_ff && !en_dly1_ff ) begin
				adr_ff	<= base_address + adr_cnt_ff ;
			end
		end
	end

	assign avmm_address_o	= adr_ff ;

	// -----------------------------
	//	address counter
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			adr_cnt_ff	<= {ADR_CNT_BIT{1'b0}} ;
		end
		else if ( srst ) begin
			adr_cnt_ff	<= {ADR_CNT_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( rd_start_i ) begin
				adr_cnt_ff	<= {ADR_CNT_BIT{1'b0}} ;
			end
			else if ( avmm_readdatavalid_i ) begin
				adr_cnt_ff	<= adr_cnt_ff + 1'b1 ;
			end
		end
	end

	assign base_address	= ( {LADR_BIT{(rfrm_cnt_ff == 3'b000)}} & FRAME1_BASE[LADR_BIT-1:0] )
						| ( {LADR_BIT{(rfrm_cnt_ff == 3'b001)}} & FRAME2_BASE[LADR_BIT-1:0] )
						| ( {LADR_BIT{(rfrm_cnt_ff == 3'b010)}} & FRAME3_BASE[LADR_BIT-1:0] )
						| ( {LADR_BIT{(rfrm_cnt_ff == 3'b011)}} & FRAME4_BASE[LADR_BIT-1:0] )
						| ( {LADR_BIT{(rfrm_cnt_ff == 3'b100)}} & FRAME5_BASE[LADR_BIT-1:0] )
						| ( {LADR_BIT{(rfrm_cnt_ff == 3'b101)}} & FRAME6_BASE[LADR_BIT-1:0] )
						| ( {LADR_BIT{(rfrm_cnt_ff == 3'b110)}} & FRAME7_BASE[LADR_BIT-1:0] )
						| ( {LADR_BIT{(rfrm_cnt_ff == 3'b111)}} & FRAME8_BASE[LADR_BIT-1:0] ) ;

	// -----------------------------
	//	final data read
	// -----------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			fdata_rd_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			fdata_rd_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( rd_start_i ) begin
				fdata_rd_ff	<= 1'b0 ;
			end
			else if ( ( adr_cnt_ff >= rd_final_ff ) && rd_frm_vld_i && trans_enable_ff ) begin
				fdata_rd_ff	<= 1'b1 ;
			end
		end
	end

	assign fdata_rd_o = fdata_rd_ff ;

endmodule
`default_nettype wire
