//----------------------------------------------------------------------------------------
//Copyright (C) 2014 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		Avalon-MM I/F Arbitration Mocule
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Dec.  1 2014	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		LADR_BIT				: Avalon-MM I/F address bit width
//								:	 must be greater than LBST_BIT + floor( log2( LDATA_BIT / 8 ) )
//		LDATA_BIT				:	 Avalon-MM I/F writedata/readdata bit width
//								: must be 2^n ( n = 3 ~ 10 ) and equal or greater than VSI_DATA_BIT
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk						: clock for all circuit
//		rst_n					: asynchronous reset ( low active )
//		srst					: synchronous reset
//		enable					: clock enable
//		srst_sfb_o				: Reset Output for SCV11
//
//		--- Avalon-MM I/F bus arbitration signals ---
//		wr_bus_req_i			: Avalon-MM bus request from write controller
//		wr_bus_grant_o			: Avalon-MM bus grant to write controller
//		wr_bus_busy_i			: Avalon-MM bus busy from write controller
//		rd_bus_req_i			: Avalon-MM bus request from read controller
//		rd_bus_grant_o			: Avalon-MM bus grant to read controller
//		rd_bus_busy_i			: Avalon-MM bus busy from read controller
//
//		--- Avalon-MM I/F from/to write/read controller ---
//		wr_address_i			: Avalon-MM address from write controller
//		wr_begintrans_i			: Avalon-MM begintrans from write controller
//		wr_burstcount_i			: Avalon-MM burstcount from write controller
//		wr_write_i				: Avalon-MM write from write controller
//		wr_writedata_i			: Avalon-MM writedata from write controller
//
//		rd_address_i			: Avalon-MM address from read controller
//		rd_begintrans_i			: Avalon-MM begintrans from read controller
//		rd_burstcount_i			: Avalon-MM burstcount from read controller
//		rd_read_i				: Avalon-MM read from read controller
//		rd_readdatavalid_o		: Avalon-MM readdatavalid to read controller
//		rd_readdata_o			: Avalon-MM readdata to read controller
//
//		local_waitrequest_o		: Avalon-MM waitrequest to write/read controller
//
//		--- Avalon-MM I/F from/to High-Performance Memory Controller ---
//		avmm_address_o			: Avalon-MM address
//		avmm_begintrans_o		: Avalon-MM begintranstransfer
//		avmm_burstcount_o		: Avalon-MM burstcount
//		avmm_write_o			: Avalon-MM write
//		avmm_writedata_o		: Avalon-MM writedata
//		avmm_read_o				: Avalon-MM read
//		avmm_readdatavalid_i	: Avalon-MM readdatavalid
//		avmm_readdata_i			: Avalon-MM readdata
//		avmm_waitrequest_i		: Avalon-MM waitrequest
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	sfb16_arbiter (
	clk						,
	rst_n					,
	srst					,
	enable					,
	srst_sfb_o				,

	// Avalon-MM I/F bus arbitration signals
	wr_bus_req_i			,
	wr_bus_grant_o			,
	wr_bus_busy_i			,
	rd_bus_req_i			,
	rd_bus_grant_o			,
	rd_bus_busy_i			,

	// Avalon-MM I/F from/to write/read controller
	wr_address_i			,
	wr_begintrans_i			,
	wr_burstcount_i			,
	wr_write_i				,
	wr_writedata_i			,

	rd_address_i			,
	rd_begintrans_i			,
	rd_burstcount_i			,
	rd_read_i				,
	rd_readdatavalid_o		,
	rd_readdata_o			,

	local_waitrequest_o		,

	// Avalon-MM I/F from/to High-Performance Memory Controller
	avmm_address_o			,
	avmm_byteenable_o		,
	avmm_begintrans_o		,
	avmm_burstcount_o		,
	avmm_write_o			,
	avmm_writedata_o		,
	avmm_read_o				,
	avmm_readdatavalid_i	,
	avmm_readdata_i			,
	avmm_waitrequest_i
) ;



// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------

	parameter	LADR_BIT		= 25 			;
	parameter	LDATA_BIT		= 128			;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

	localparam	LBST_BIT		= 5 			;
	localparam	LBE_BIT			= LDATA_BIT / 8 ;

	// ---------------------------------------------------------------------
	// State Encode
	// ---------------------------------------------------------------------

	parameter	ARBITER_WR_WAIT		= 3'b000 ;
	parameter	ARBITER_WR_BUSY		= 3'b001 ;
	parameter	ARBITER_RD_WAIT		= 3'b010 ;
	parameter	ARBITER_RD_BUSY		= 3'b011 ;
	parameter	ARBITER_WR_END		= 3'b100 ;
	parameter	ARBITER_SRST		= 3'b101 ;

// =============================================================================
// PORT DECLARATION
// =============================================================================

	input	wire						clk						;
	input	wire						rst_n					;
	input	wire						srst					;
	input	wire						enable					;
	output	wire						srst_sfb_o				;

	// Avalon-MM I/F bus arbitration signals
	input	wire						wr_bus_req_i			;
	output	wire						wr_bus_grant_o			;
	input	wire						wr_bus_busy_i			;
	input	wire						rd_bus_req_i			;
	output	wire						rd_bus_grant_o			;
	input	wire						rd_bus_busy_i			;

	// Avalon-MM I/F from/to write/read controller
	input	wire	[  LADR_BIT-1 : 0 ]	wr_address_i			;
	input	wire						wr_begintrans_i			;
	input	wire	[  LBST_BIT-1 : 0 ]	wr_burstcount_i			;
	input	wire						wr_write_i				;
	input	wire	[ LDATA_BIT-1 : 0 ]	wr_writedata_i			;

	input	wire	[  LADR_BIT-1 : 0 ]	rd_address_i			;
	input	wire						rd_begintrans_i			;
	input	wire	[  LBST_BIT-1 : 0 ]	rd_burstcount_i			;
	input	wire						rd_read_i				;
	output	wire						rd_readdatavalid_o		;
	output	wire	[ LDATA_BIT-1 : 0 ]	rd_readdata_o			;

	output	wire						local_waitrequest_o		;

	// Avalon-MM I/F from/to High-Performance Memory Controller
	output	wire	[  LADR_BIT-1 : 0 ]	avmm_address_o			;
	output	wire	[   LBE_BIT-1 : 0 ]	avmm_byteenable_o		;
	output	wire						avmm_begintrans_o		;
	output	wire	[  LBST_BIT-1 : 0 ]	avmm_burstcount_o		;
	output	wire						avmm_write_o			;
	output	wire	[ LDATA_BIT-1 : 0 ]	avmm_writedata_o		;
	output	wire						avmm_read_o				;
	input	wire						avmm_readdatavalid_i	;
	input	wire	[ LDATA_BIT-1 : 0 ]	avmm_readdata_i			;

	input	wire						avmm_waitrequest_i		;



// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	wire							pipe_en					;

	reg		[ 2 : 0 ]				grant_state_ff			;
	reg								write_grant_ff			;
	reg								read_grant_ff			;

	wire	[  LADR_BIT-1 : 0 ]		avmm_addr_sel			;
	wire							avmm_begintrans_sel		;
	wire	[  LBST_BIT-1 : 0 ]		avmm_burstcount_sel		;
	wire							avmm_write_sel			;
	wire	[ LDATA_BIT-1 : 0 ]		avmm_writedata_sel		;
	wire							avmm_read_sel			;

	reg		[  LADR_BIT-1 : 0 ]		avmm_addr_ff			;
	reg		[   LBE_BIT-1: 0 ]		avmm_byteenable_ff		;
	reg								avmm_begintrans_ff		;
	reg		[  LBST_BIT-1 : 0 ]		avmm_burstcount_ff		;
	reg								avmm_write_ff			;
	reg		[ LDATA_BIT-1 : 0 ]		avmm_writedata_ff		;
	reg								avmm_read_ff			;

	reg								srst_sfb_ff				;
	reg								srst_hld_ff				;
	reg								srst_mlat1_ff			;
	reg								srst_mlat2_ff			;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	assign pipe_en	= ~( avmm_waitrequest_i & ( avmm_write_ff | avmm_read_ff ) ) ;

	//===============//
	// SRST Conrtrol //
	//===============//

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			srst_mlat1_ff	<= 1'b0 ;
			srst_mlat2_ff	<= 1'b0 ;
		end
		else begin
			srst_mlat1_ff	<= srst ;
			srst_mlat2_ff	<= srst_mlat1_ff ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			srst_hld_ff		<= 1'b0 ;
		end
		else if( ( grant_state_ff == ARBITER_SRST ) && (srst_mlat2_ff == 1'b0) )begin
			srst_hld_ff <= 1'b0 ;
		end
		else if( srst_mlat2_ff )begin
			srst_hld_ff <= 1'b1 ;
		end
		else begin
			srst_hld_ff <= srst_hld_ff ;
		end
	end

	//====================================//
	// write/read bus arbitration control //
	//====================================//

	// bus grant state register
	// write grant register
	// read grant register
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			grant_state_ff	<= ARBITER_WR_WAIT ;
			write_grant_ff	<= 1'b0 ;
			read_grant_ff	<= 1'b0 ;
			srst_sfb_ff		<= 1'b0 ;
		end
		else if ( enable ) begin
			case ( grant_state_ff )

				// write request waiting state
				ARBITER_WR_WAIT	: begin
					if ( srst_hld_ff ) begin
						grant_state_ff	<= ARBITER_SRST ;
						srst_sfb_ff		<= 1'b1 ;
					end
					else if ( wr_bus_req_i ) begin
						grant_state_ff	<= ARBITER_WR_BUSY ;
						write_grant_ff	<= 1'b1 ;
					end
					else if ( !wr_bus_req_i ) begin
						grant_state_ff	<= ARBITER_RD_WAIT ;
					end
				end

				// write busy state
				ARBITER_WR_BUSY	: begin
					write_grant_ff	<= 1'b0 ;
					if ( !wr_bus_req_i && !wr_bus_busy_i ) begin
						grant_state_ff	<= ARBITER_WR_END ;
					end
				end

				// confirm the end of write cycle added by tkondoh
				ARBITER_WR_END : begin
					if ( !avmm_write_ff ) begin
						grant_state_ff <= ARBITER_RD_WAIT ;
					end
				end

				// read request waiting state
				ARBITER_RD_WAIT	: begin
					if ( srst_hld_ff ) begin
						grant_state_ff	<= ARBITER_SRST ;
						srst_sfb_ff		<= 1'b1 ;
					end
					else if ( rd_bus_req_i ) begin
						grant_state_ff	<= ARBITER_RD_BUSY ;
						read_grant_ff	<= 1'b1 ;
					end
					else if ( !rd_bus_req_i ) begin
						grant_state_ff	<= ARBITER_WR_WAIT ;
					end
				end

				// read busy state
				ARBITER_RD_BUSY	: begin

					read_grant_ff	<= 1'b0 ;

					if ( !rd_bus_req_i && !rd_bus_busy_i ) begin
						grant_state_ff	<= ARBITER_WR_WAIT ;
					end
				end

				ARBITER_SRST : begin
					if ( srst_mlat2_ff == 1'b0 ) begin
						grant_state_ff	<= ARBITER_WR_WAIT ;
						srst_sfb_ff		<= 1'b0 ;
					end
				end

				// exception state
				default			: begin
					grant_state_ff	<= ARBITER_WR_WAIT ;
					write_grant_ff	<= 1'b0 ;
					read_grant_ff	<= 1'b0 ;
				end
			endcase
		end
	end

	assign wr_bus_grant_o	= write_grant_ff ;
	assign rd_bus_grant_o	= read_grant_ff ;
	assign srst_sfb_o		= srst_sfb_ff ;

	//==============================//
	// Avalon-MM bus switch control //
	//==============================//

	assign avmm_addr_sel		= ( {LADR_BIT{wr_bus_busy_i}} & wr_address_i )
								| ( {LADR_BIT{rd_bus_busy_i}} & rd_address_i ) ;

	assign avmm_begintrans_sel	= ( wr_bus_busy_i & wr_begintrans_i )
								| ( rd_bus_busy_i & rd_begintrans_i ) ;

	assign avmm_burstcount_sel	= ( {LBST_BIT{wr_bus_busy_i}} & wr_burstcount_i )
								| ( {LBST_BIT{rd_bus_busy_i}} & rd_burstcount_i ) ;

	assign avmm_write_sel		= wr_bus_busy_i & wr_write_i ;

	assign avmm_writedata_sel	= {LDATA_BIT{wr_bus_busy_i}} & wr_writedata_i ;

	assign avmm_read_sel		= rd_bus_busy_i & rd_read_i ;


	// Avalon-MM address register
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			avmm_addr_ff	<= {LADR_BIT{1'b0}} ;
		end
		else if ( srst_sfb_ff ) begin
			avmm_addr_ff	<= {LADR_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( pipe_en && avmm_begintrans_sel ) begin
				avmm_addr_ff	<= avmm_addr_sel ;
			end
		end
	end

	// Avalon-MM byte enable register
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			avmm_byteenable_ff	<= {LBE_BIT{1'b0}} ;
		end
		else if ( srst_sfb_ff ) begin
			avmm_byteenable_ff	<= {LBE_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( pipe_en ) begin
				avmm_byteenable_ff	<= {LBE_BIT{(avmm_write_sel | avmm_read_sel)}} ;
			end
		end
	end

	// Avalon-MM begintranstransfer register
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			avmm_begintrans_ff	<= 1'b0 ;
		end
		else if ( srst_sfb_ff ) begin
			avmm_begintrans_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			avmm_begintrans_ff	<= pipe_en & avmm_begintrans_sel ;
		end
	end

	// Avalon-MM burstcount register
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			avmm_burstcount_ff	<= {LBST_BIT{1'b0}} ;
		end
		else if ( srst_sfb_ff ) begin
			avmm_burstcount_ff	<= {LBST_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( pipe_en && avmm_begintrans_sel ) begin
				avmm_burstcount_ff	<= avmm_burstcount_sel ;
			end
		end
	end

	// Avalon-MM write register
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			avmm_write_ff	<= 1'b0 ;
		end
		else if ( srst_sfb_ff ) begin
			avmm_write_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( pipe_en ) begin
				avmm_write_ff	<= avmm_write_sel ;
			end
		end
	end

	// Avalon-MM writedata register
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			avmm_writedata_ff	<= {LDATA_BIT{1'b0}} ;
		end
		else if ( srst_sfb_ff ) begin
			avmm_writedata_ff	<= {LDATA_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			if ( pipe_en ) begin
				avmm_writedata_ff	<= avmm_writedata_sel ;
			end
		end
	end

	// Avalon-MM read register
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			avmm_read_ff	<= 1'b0 ;
		end
		else if ( srst_sfb_ff ) begin
			avmm_read_ff	<= 1'b0 ;
		end
		else if ( enable ) begin
			if ( pipe_en ) begin
				avmm_read_ff	<= avmm_read_sel ;
			end
		end
	end

	assign avmm_address_o			= avmm_addr_ff		 	;
	assign avmm_byteenable_o		= avmm_byteenable_ff	;
	assign avmm_begintrans_o		= avmm_begintrans_ff 	;
	assign avmm_burstcount_o		= avmm_burstcount_ff	;
	assign avmm_write_o				= avmm_write_ff		 	;
	assign avmm_writedata_o			= avmm_writedata_ff	 	;
	assign avmm_read_o				= avmm_read_ff		 	;
	assign rd_readdatavalid_o		= avmm_readdatavalid_i	;
	assign rd_readdata_o			= avmm_readdata_i		;
	assign local_waitrequest_o		= ~pipe_en				;


endmodule
`default_nettype wire
