//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		multiply accumulator for scaler adaptive
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Mar. 13 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		APORT_BIT	: port A bit width
//		BPORT_BIT	: port B bit width
//
//		MULT_BIT	: multiply result bit width ( APORT_BIT + BPORT_BIT )
//		YPORT_BIT	: addition result bit width ( APORT_BIT + BPORT_BIT + 2 )
//
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk			: clock for all circuit
//		rst_n		: asynchronous reset ( low active )
//		enable		: clock enable
//
//		dataa_0_i	: port A0 data ( multiplier )
//		dataa_1_i	: port A1 data ( multiplier )
//		dataa_2_i	: port A2 data ( multiplier )
//		dataa_3_i	: port A3 data ( multiplier )
//		datab_0_i	: port B0 data ( multiplicand )
//		datab_1_i	: port B1 data ( multiplicand )
//		datab_2_i	: port B2 data ( multiplicand )
//		datab_3_i	: port B3 data ( multiplicand )
//
//		result_o	: port Y data ( result )
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	scl16_signed_multadd4 (
	clk			,
	rst_n		,
	enable		,

	dataa_0_i	,
	dataa_1_i	,
	dataa_2_i	,
	dataa_3_i	,
	datab_0_i	,
	datab_1_i	,
	datab_2_i	,
	datab_3_i	,

	result_o
) ;

// =============================================================================
// DEFINE INCLUDE
// =============================================================================

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------
	parameter APORT_BIT				= 8	;
	parameter BPORT_BIT				= 8	;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------
	parameter MULT_BIT 				= APORT_BIT + BPORT_BIT		;
	parameter YPORT_BIT				= APORT_BIT + BPORT_BIT + 2	;

// =============================================================================
// PORT DECLARATION
// =============================================================================
	input	wire									clk			;
	input	wire									rst_n		;
	input	wire									enable		;

	input	wire	signed	[ APORT_BIT-1 : 0 ]		dataa_0_i	;
	input	wire	signed	[ APORT_BIT-1 : 0 ]		dataa_1_i	;
	input	wire	signed	[ APORT_BIT-1 : 0 ]		dataa_2_i	;
	input	wire	signed	[ APORT_BIT-1 : 0 ]		dataa_3_i	;
	input	wire	signed	[ BPORT_BIT-1 : 0 ]		datab_0_i	;
	input	wire	signed	[ BPORT_BIT-1 : 0 ]		datab_1_i	;
	input	wire	signed	[ BPORT_BIT-1 : 0 ]		datab_2_i	;
	input	wire	signed	[ BPORT_BIT-1 : 0 ]		datab_3_i	;

	output	wire	[YPORT_BIT-1:0] 				result_o	;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================
	reg		signed	[ APORT_BIT-1 : 0 ]		dataa_0_ff	;
	reg		signed	[ APORT_BIT-1 : 0 ]		dataa_1_ff	;
	reg		signed	[ APORT_BIT-1 : 0 ]		dataa_2_ff	;
	reg		signed	[ APORT_BIT-1 : 0 ]		dataa_3_ff	;
	reg		signed	[ BPORT_BIT-1 : 0 ]		datab_0_ff	;
	reg		signed	[ BPORT_BIT-1 : 0 ]		datab_1_ff	;
	reg		signed	[ BPORT_BIT-1 : 0 ]		datab_2_ff	;
	reg		signed	[ BPORT_BIT-1 : 0 ]		datab_3_ff	;

	wire	signed	[ MULT_BIT-1 : 0 ]		dataab_0	;
	wire	signed	[ MULT_BIT-1 : 0 ]		dataab_1	;
	wire	signed	[ MULT_BIT-1 : 0 ]		dataab_2	;
	wire	signed	[ MULT_BIT-1 : 0 ]		dataab_3	;

	reg		signed	[ MULT_BIT-1 : 0 ]		dataab_0_ff	;
	reg		signed	[ MULT_BIT-1 : 0 ]		dataab_1_ff	;
	reg		signed	[ MULT_BIT-1 : 0 ]		dataab_2_ff	;
	reg		signed	[ MULT_BIT-1 : 0 ]		dataab_3_ff	;

	wire	signed	[ YPORT_BIT-1 : 0 ]		result		;
	reg		signed	[ YPORT_BIT-1 : 0 ]		result_ff	;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			dataa_0_ff		<= {APORT_BIT{1'b0}} ;
			dataa_1_ff		<= {APORT_BIT{1'b0}} ;
			dataa_2_ff		<= {APORT_BIT{1'b0}} ;
			dataa_3_ff		<= {APORT_BIT{1'b0}} ;
			datab_0_ff		<= {BPORT_BIT{1'b0}} ;
			datab_1_ff		<= {BPORT_BIT{1'b0}} ;
			datab_2_ff		<= {BPORT_BIT{1'b0}} ;
			datab_3_ff		<= {BPORT_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			dataa_0_ff		<= dataa_0_i ;
			dataa_1_ff		<= dataa_1_i ;
			dataa_2_ff		<= dataa_2_i ;
			dataa_3_ff		<= dataa_3_i ;
			datab_0_ff		<= datab_0_i ;
			datab_1_ff		<= datab_1_i ;
			datab_2_ff		<= datab_2_i ;
			datab_3_ff		<= datab_3_i ;
		end
	end

	assign dataab_0			= dataa_0_ff * datab_0_ff ;
	assign dataab_1			= dataa_1_ff * datab_1_ff ;
	assign dataab_2			= dataa_2_ff * datab_2_ff ;
	assign dataab_3			= dataa_3_ff * datab_3_ff ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			dataab_0_ff		<= {MULT_BIT{1'b0}} ;
			dataab_1_ff		<= {MULT_BIT{1'b0}} ;
			dataab_2_ff		<= {MULT_BIT{1'b0}} ;
			dataab_3_ff		<= {MULT_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			dataab_0_ff		<= dataab_0 ;
			dataab_1_ff		<= dataab_1 ;
			dataab_2_ff		<= dataab_2 ;
			dataab_3_ff		<= dataab_3 ;
		end
	end

	assign result			= dataab_0_ff + dataab_1_ff + dataab_2_ff + dataab_3_ff ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			result_ff		<= {YPORT_BIT{1'b0}} ;
		end
		else if ( enable ) begin
			result_ff		<= result ;
		end
	end

	assign result_o			= result_ff ;

endmodule

`default_nettype wire
