//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		scaling rate calculation circuit
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Mar. 13 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		SIZE_BIT		: size setting signal bit width
//		RATE_FRAC_BIT	: fractional part bit width for coordinate calculation
//
//		RATE_BIT 		: scaling rate calculation bit width ( SIZE_BIT + RATE_FRAC_BIT )
//		DIV_ARG1_BIT	: dividend bit width ( RATE_BIT )
//		DIV_ARG2_BIT	: divisor bit width ( SIZE_BIT )
//		DIV_RTN_BIT		: quotient bit width ( DIV_ARG1_BIT )
//		DIV_STEP		: divider pipeline steps ( DIV_ARG1_BIT )
//
// ------ calculate state machine ------
//		CALC_IDLE		: initial state
//		CALC_DIV1		: horizontal scale rate calculation state
//		CALC_DIV2		: vertical scale rate calculation state
//		CALC_WAIT		: waiting for srst state
//
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk				: clock for all circuit
//		rst_n			: asynchronous reset ( low active )
//		srst			: synchronous reset
//
//		src_wdt_i		: horizontal size of input frame data
//		src_hgt_i		: vertical size of input frame data
//		tgt_wdt_i		: horizontal size of output frame data
//		tgt_hgt_i		: vertical size of output frame data
//
//		h_scl_rate_o	: horizontal scaling rate
//		v_scl_rate_o	: vertical scaling rate
//		src_wdt_o		: gated horizontal size of input frame data
//		src_hgt_o		: gated vertical size of input frame data
//		tgt_wdt_o		: gated horizontal size of output frame data
//		tgt_hgt_o		: gated vertical size of output frame data
//		init_end_o		: initial calculation end pulse at
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	scl16_rate_calc (
	clk				,
	rst_n			,
	srst			,

	src_wdt_i		,
	src_hgt_i		,
	tgt_wdt_i		,
	tgt_hgt_i		,

	h_scl_rate_o	,
	v_scl_rate_o	,
	src_wdt_o		,
	src_hgt_o		,
	tgt_wdt_o		,
	tgt_hgt_o		,
	init_end_o
) ;

// =============================================================================
// DEFINE INCLUDE
// =============================================================================

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------
	parameter SIZE_BIT				= 12			;
	parameter RATE_FRAC_BIT			= SIZE_BIT + 2	;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------
	parameter RATE_BIT 				= SIZE_BIT + RATE_FRAC_BIT	;

	parameter DIV_ARG1_BIT			= RATE_BIT					;
	parameter DIV_ARG2_BIT			= SIZE_BIT					;
	parameter DIV_RTN_BIT			= DIV_ARG1_BIT				;

	parameter DIV_STEP				= DIV_ARG1_BIT				;

	parameter CALC_IDLE				= 2'h0						;
	parameter CALC_DIV1				= 2'h1						;
	parameter CALC_DIV2				= 2'h2						;
	parameter CALC_WAIT				= 2'h3						;

// =============================================================================
// PORT DECLARATION
// =============================================================================
	input	wire							clk				;
	input	wire							rst_n			;
	input	wire							srst			;

	input	wire	[ SIZE_BIT-1 : 0 ]		src_wdt_i		;
	input	wire	[ SIZE_BIT-1 : 0 ]		src_hgt_i		;
	input	wire	[ SIZE_BIT-1 : 0 ]		tgt_wdt_i		;
	input	wire	[ SIZE_BIT-1 : 0 ]		tgt_hgt_i		;

	output	wire	[ RATE_BIT-1 : 0 ]		h_scl_rate_o	;
	output	wire	[ RATE_BIT-1 : 0 ]		v_scl_rate_o	;
	output	wire	[ SIZE_BIT-1 : 0 ]		src_wdt_o		;
	output	wire	[ SIZE_BIT-1 : 0 ]		src_hgt_o		;
	output	wire	[ SIZE_BIT-1 : 0 ]		tgt_wdt_o		;
	output	wire	[ SIZE_BIT-1 : 0 ]		tgt_hgt_o		;
	output	wire							init_end_o		;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================
	reg		[ SIZE_BIT-1 : 0 ]		swdt_lat_ff		;
	reg		[ SIZE_BIT-1 : 0 ]		shgt_lat_ff		;
	reg		[ SIZE_BIT-1 : 0 ]		twdt_lat_ff		;
	reg		[ SIZE_BIT-1 : 0 ]		thgt_lat_ff		;

	reg		[ 1 : 0 ]				calc_state_ff	;
	reg		[ 4 : 0 ]				cyc_cnt_ff		;

	reg		[ DIV_ARG1_BIT-1 :0 ]	div_arg1_ff		;
	reg		[ DIV_ARG2_BIT-1 :0 ]	div_arg2_ff		;
	reg								div_valid_ff	;
	wire	[ DIV_RTN_BIT-1 :0 ]	div_rtn			;

	reg		[ RATE_BIT-1 : 0 ]		hscl_rate_ff	;
	reg		[ RATE_BIT-1 : 0 ]		vscl_rate_ff	;

	reg		[ SIZE_BIT-1 : 0 ]		swdt_out_ff		;
	reg		[ SIZE_BIT-1 : 0 ]		shgt_out_ff		;
	reg		[ SIZE_BIT-1 : 0 ]		twdt_out_ff		;
	reg		[ SIZE_BIT-1 : 0 ]		thgt_out_ff		;

	reg								init_end_ff		;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			swdt_lat_ff		<= { SIZE_BIT{1'b0} } ;
			shgt_lat_ff		<= { SIZE_BIT{1'b0} } ;
			twdt_lat_ff		<= { SIZE_BIT{1'b0} } ;
			thgt_lat_ff		<= { SIZE_BIT{1'b0} } ;
		end
		else if ( srst ) begin
			swdt_lat_ff		<= src_wdt_i ;
			shgt_lat_ff		<= src_hgt_i ;
			twdt_lat_ff		<= tgt_wdt_i ;
			thgt_lat_ff		<= tgt_hgt_i ;
		end
	end

	// --------------------------------------------------
	// scaling rate calculation process @ CORRECTION = 0
	// --------------------------------------------------

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			calc_state_ff	<= CALC_IDLE ;
			cyc_cnt_ff		<= 5'h00 ;

			div_arg1_ff		<= {DIV_ARG1_BIT{1'b0}} ;
			div_arg2_ff		<= {DIV_ARG2_BIT{1'b0}} ;
			div_valid_ff	<= 1'b0 ;

			hscl_rate_ff	<= {RATE_BIT{1'b0}} ;
			vscl_rate_ff	<= {RATE_BIT{1'b0}} ;

			swdt_out_ff		<= {SIZE_BIT{1'b0}} ;
			shgt_out_ff		<= {SIZE_BIT{1'b0}} ;
			twdt_out_ff		<= {SIZE_BIT{1'b0}} ;
			thgt_out_ff		<= {SIZE_BIT{1'b0}} ;

			init_end_ff		<= 1'b0 ;
		end
		else if ( srst ) begin
			calc_state_ff	<= CALC_IDLE ;
			cyc_cnt_ff		<= 5'h00 ;

			div_arg1_ff		<= {DIV_ARG1_BIT{1'b0}} ;
			div_arg2_ff		<= {DIV_ARG2_BIT{1'b0}} ;
			div_valid_ff	<= 1'b0 ;

			hscl_rate_ff	<= {RATE_BIT{1'b0}} ;
			vscl_rate_ff	<= {RATE_BIT{1'b0}} ;

			swdt_out_ff		<= {SIZE_BIT{1'b0}} ;
			shgt_out_ff		<= {SIZE_BIT{1'b0}} ;
			twdt_out_ff		<= {SIZE_BIT{1'b0}} ;
			thgt_out_ff		<= {SIZE_BIT{1'b0}} ;

			init_end_ff		<= 1'b0 ;
		end
		else begin
			case ( calc_state_ff )
				CALC_IDLE : begin
					calc_state_ff	<= CALC_DIV1;
					cyc_cnt_ff		<= 5'h00;
					div_arg1_ff		<= { swdt_lat_ff , {RATE_FRAC_BIT{1'b0}} } - { 1'b1 , {RATE_FRAC_BIT{1'b0}} } ;
					div_arg2_ff		<= twdt_lat_ff - 1'b1;
					div_valid_ff	<= 1'b1 ;
				end
				CALC_DIV1 : begin
					if ( cyc_cnt_ff == DIV_STEP ) begin
						calc_state_ff	<= CALC_DIV2 ;
						cyc_cnt_ff		<= 5'h00 ;
						hscl_rate_ff	<= div_rtn ;
						div_arg1_ff		<= { shgt_lat_ff , {RATE_FRAC_BIT{1'b0}} } - { 1'b1 , {RATE_FRAC_BIT{1'b0}} } ;
						div_arg2_ff		<= thgt_lat_ff - 1'b1 ;
						div_valid_ff	<= 1'b1 ;
					end
					else begin
						cyc_cnt_ff		<= cyc_cnt_ff + 1'b1 ;
						div_valid_ff	<= 1'b0 ;
					end
				end
				CALC_DIV2 : begin
					if ( cyc_cnt_ff == DIV_STEP ) begin
						calc_state_ff	<= CALC_WAIT ;
						cyc_cnt_ff		<= 5'h00 ;
						vscl_rate_ff	<= div_rtn ;
						swdt_out_ff		<= swdt_lat_ff - 1'b1 ;
						shgt_out_ff		<= shgt_lat_ff - 1'b1 ;
						twdt_out_ff		<= twdt_lat_ff - 1'b1 ;
						thgt_out_ff		<= thgt_lat_ff - 1'b1 ;
						init_end_ff		<= 1'b1 ;
						div_valid_ff	<= 1'b0 ;
					end
					else begin
						cyc_cnt_ff		<= cyc_cnt_ff + 1'b1 ;
						div_valid_ff	<= 1'b0 ;
					end
				end
				CALC_WAIT : begin
					init_end_ff		<= 1'b0 ;
				end
				default : begin
					calc_state_ff	<= CALC_IDLE ;
					cyc_cnt_ff		<= {5{1'bx}} ;

					div_arg1_ff		<= {DIV_ARG1_BIT{1'bx}} ;
					div_arg2_ff		<= {DIV_ARG2_BIT{1'bx}} ;
					div_valid_ff	<= 1'bx ;

					hscl_rate_ff	<= {RATE_BIT{1'bx}} ;
					vscl_rate_ff	<= {RATE_BIT{1'bx}} ;

					swdt_out_ff		<= {SIZE_BIT{1'bx}} ;
					shgt_out_ff		<= {SIZE_BIT{1'bx}} ;
					twdt_out_ff		<= {SIZE_BIT{1'bx}} ;
					thgt_out_ff		<= {SIZE_BIT{1'bx}} ;

					init_end_ff		<= 1'bx ;
				end
			endcase
		end
	end

	assign h_scl_rate_o		= hscl_rate_ff ;
	assign v_scl_rate_o		= vscl_rate_ff ;
	assign src_wdt_o		= swdt_out_ff ;
	assign src_hgt_o		= shgt_out_ff ;
	assign tgt_wdt_o		= twdt_out_ff ;
	assign tgt_hgt_o		= thgt_out_ff ;

	assign init_end_o		= init_end_ff ;

	// divider for scaling rate calculation
	scl16_divider
		#(
			.DVEND			( DIV_ARG1_BIT	),
			.DVSOR			( DIV_ARG2_BIT	),
			.PIPELINE		( 0 )
		)
		u_div_for_scl_rate_calc (
			.clk			( clk			),
			.rst_n			( rst_n			),
			.srst			( srst			),
			.enable 		( 1'b1			),

			.valid_i		( div_valid_ff	),
			.dividend_i		( div_arg1_ff	),
			.divisor_i		( div_arg2_ff	),

			.ready_o		( ),
			.valid_o		( ),
			.quotient_o		( div_rtn		),
			.remainder_o	( )
		);

endmodule

`default_nettype wire
