//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		nearest-neighbor scaling filter coefficient table for scaler adaptive
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Mar. 13 2012	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		Q_BIT		: quantized bit width per color plane
//
//		DLT_BIT		: delta bit width
//		COEFF_BIT	: coefficient bit width
//
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk;		: clock for all circuit
//		rst_n;		: asynchronous reset ( low active )
//		enable;		: clock enable
//
//		x_i;		: table index
//
//		coeff_zr_o;	: table value ( -1 -> 0 section coefficient )
//		coeff_p1_o;	: table value (  0 -> 1 section coefficient )
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	scl16_nearest_ph32_func (
	clk			,
	rst_n		,
	enable		,

	x_i			,

	coeff_zr_o	,
	coeff_p1_o
) ;

// =============================================================================
// DEFINE INCLUDE
// =============================================================================

// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------
	parameter Q_BIT					= 8			;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------
	parameter DLT_BIT				= 5			;
	parameter COEFF_BIT				= Q_BIT + 1	;

// =============================================================================
// PORT DECLARATION
// =============================================================================
	input	wire							clk			;
	input	wire							rst_n		;
	input	wire							enable		;

	input	wire	[ DLT_BIT-1 : 0 ]		x_i			;

	output	wire	[ COEFF_BIT-1 : 0 ]		coeff_zr_o	;
	output	wire	[ COEFF_BIT-1 : 0 ]		coeff_p1_o	;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================
	reg		[ COEFF_BIT-1 : 0 ]		coeff_zr_ff	;
	reg		[ COEFF_BIT-1 : 0 ]		coeff_p1_ff	;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================
	generate
		if ( Q_BIT == 6 ) begin : table_6bit
			//-------------------- Table Value Format -----------------------//
			//	 { Sign(1bit), Integral Part(1bit), Fraction Part(5bit) }	 //
			//---------------------------------------------------------------//
			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_zr_ff		<= 7'h20 ;
				end
				else if ( enable ) begin
					case ( x_i[DLT_BIT-1] )
						1'b0	: coeff_zr_ff	<= 7'h20 ;
						1'b1	: coeff_zr_ff	<= 7'h00 ;
						default	: coeff_zr_ff	<= {7{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p1_ff		<= 7'h00 ;
				end
				else if ( enable ) begin
					case ( x_i[DLT_BIT-1] )
						1'b0	: coeff_p1_ff	<= 7'h00 ;
						1'b1	: coeff_p1_ff	<= 7'h20 ;
						default	: coeff_p1_ff	<= {7{1'bx}} ;
					endcase
				end
			end
		end
		else if ( Q_BIT == 8 ) begin : table_8bit
			//-------------------- Table Value Format -----------------------//
			//	 { Sign(1bit), Integral Part(1bit), Fraction Part(7bit) }	 //
			//---------------------------------------------------------------//
			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_zr_ff		<= 9'h080 ;
				end
				else if ( enable ) begin
					case ( x_i[DLT_BIT-1] )
						1'b0	: coeff_zr_ff	<= 9'h080 ;
						1'b1	: coeff_zr_ff	<= 9'h000 ;
						default	: coeff_zr_ff	<= {9{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p1_ff		<= 9'h000 ;
				end
				else if ( enable ) begin
					case ( x_i[DLT_BIT-1] )
						1'b0	: coeff_p1_ff	<= 9'h000 ;
						1'b1	: coeff_p1_ff	<= 9'h080 ;
						default	: coeff_p1_ff	<= {9{1'bx}} ;
					endcase
				end
			end
		end
		else if ( Q_BIT == 10 ) begin : table_10bit
			//-------------------- Table Value Format -----------------------//
			//	 { Sign(1bit), Integral Part(1bit), Fraction Part(9bit) }	 //
			//---------------------------------------------------------------//
			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_zr_ff		<= 11'h200 ;
				end
				else if ( enable ) begin
					case ( x_i[DLT_BIT-1] )
						1'b0	: coeff_zr_ff	<= 11'h200 ;
						1'b1	: coeff_zr_ff	<= 11'h000 ;
						default	: coeff_zr_ff	<= {11{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p1_ff		<= 11'h000 ;
				end
				else if ( enable ) begin
					case ( x_i[DLT_BIT-1] )
						1'b0	: coeff_p1_ff	<= 11'h000 ;
						1'b1	: coeff_p1_ff	<= 11'h200 ;
						default	: coeff_p1_ff	<= {11{1'bx}} ;
					endcase
				end
			end
		end
		else if ( Q_BIT == 12 ) begin : table_12bit
			//-------------------- Table Value Format -----------------------//
			//	 { Sign(1bit), Integral Part(1bit), Fraction Part(11bit) }	 //
			//---------------------------------------------------------------//
			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_zr_ff		<= 13'h0800 ;
				end
				else if ( enable ) begin
					case ( x_i[DLT_BIT-1] )
						1'b0	: coeff_zr_ff	<= 13'h0800 ;
						1'b1	: coeff_zr_ff	<= 13'h0000 ;
						default	: coeff_zr_ff	<= {13{1'bx}} ;
					endcase
				end
			end

			always @(  posedge clk or negedge rst_n ) begin
				if ( !rst_n ) begin
					coeff_p1_ff		<= 13'h0000;
				end
				else if ( enable ) begin
					case ( x_i[DLT_BIT-1] )
						1'b0	: coeff_p1_ff	<= 13'h0000 ;
						1'b1	: coeff_p1_ff	<= 13'h0800 ;
						default	: coeff_p1_ff	<= {13{1'bx}} ;
					endcase
				end
			end
		end
	endgenerate

	assign coeff_zr_o	= coeff_zr_ff ;
	assign coeff_p1_o	= coeff_p1_ff ;

endmodule

`default_nettype wire
