//----------------------------------------------------------------------------------------
//Copyright (C) 2013 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		Synchronizer for pulse
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Feb. 19 2013	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		GRAY_DEPTH 	: Width of the gray code counter used to synchronize input pulse
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk_in	: Clock signal that pls_i is generated by
//		clk_out : Clock signal that pls_o is to be synchronized with
//		rst_n	: reset_signal
//		pls_i	: Input pulse
//		pls_o	: Output pulse
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	cdc_pls_trns
	#(
		parameter						GRAY_DEPTH	= 2
	)
	(
		input	wire					clk_in			,
		input	wire					clk_out			,
		input	wire					rst_n			,

		input	wire					pls_i			,
		output	wire					pls_o
	) ;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg		[ GRAY_DEPTH-1	: 0	]	gray_i_ff	;
	wire	[ GRAY_DEPTH-1	: 0	]	gray_io		;
	reg		[ GRAY_DEPTH-1	: 0	]	gray_o_ff	;

	wire	[ GRAY_DEPTH-1	: 0 ]	gray_i_nxt	;
	wire	[ GRAY_DEPTH-1	: 0 ]	gray_o_nxt	;

	wire							pls_o_tmp	;
	reg								pls_o_ff	;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	//clk_in
	always @( posedge clk_in or negedge rst_n ) begin
		if ( !rst_n ) begin
			gray_i_ff	<= { GRAY_DEPTH{1'b0} } ;
		end
		else begin
			if ( pls_i )begin
				gray_i_ff	<= gray_i_nxt ;
			end
			else begin
				gray_i_ff	<= gray_i_ff ;
			end
		end
	end

	//clk_in -> clk_out
	cdc_ff
		#(
			.BIT_WIDTH	( GRAY_DEPTH	),
			.SYNC_INPUT	( 0				)
		)
		u_cdc_ff (
			.clk_in		( clk_in		),
			.clk_out	( clk_out		),
			.rst_n		( rst_n			),
			.data_i		( gray_i_ff		),
			.data_o		( gray_io		)
	);

	//clk_out
	always @( posedge clk_out or negedge rst_n ) begin
		if ( !rst_n ) begin
			gray_o_ff	<= { GRAY_DEPTH{1'b0} } ;
		end
		else begin
			if ( gray_io != gray_o_ff ) begin
				gray_o_ff	<= gray_o_nxt ;
			end
			else begin
				gray_o_ff	<= gray_o_ff ;
			end
		end
	end

	assign	pls_o_tmp	= ( gray_io != gray_o_ff ) ;

	always @( posedge clk_out or negedge rst_n ) begin
		if ( !rst_n ) begin
			pls_o_ff	<= 1'b0 ;
		end
		else begin
			pls_o_ff	<= pls_o_tmp ;
		end
	end

	assign	pls_o	= pls_o_ff ;

	//======================//
	// function declaration //
	//======================//

	generate
		if ( GRAY_DEPTH == 1 ) begin
			function	gray_nxt   ;
			input		gray_cur_i ;
				gray_nxt = ~gray_cur_i ;
			endfunction
			assign	gray_i_nxt	= gray_nxt(gray_i_ff) ;
			assign	gray_o_nxt	= gray_nxt(gray_o_ff) ;
		end
		else if ( GRAY_DEPTH == 2 ) begin
			function	[ 1 : 0 ]	gray_nxt   ;
			input		[ 1 : 0 ]	gray_cur_i ;
				case ( gray_cur_i )
					2'b00	:	gray_nxt	= 2'b01 ;
					2'b01	:	gray_nxt	= 2'b11 ;
					2'b11	:	gray_nxt	= 2'b10 ;
					default	:	gray_nxt	= 2'b00 ; //2'b10 case
				endcase
			endfunction
			assign	gray_i_nxt	= gray_nxt(gray_i_ff) ;
			assign	gray_o_nxt	= gray_nxt(gray_o_ff) ;
		end
		else if ( GRAY_DEPTH == 3 ) begin
			function	[ 2 : 0 ]	gray_nxt   ;
			input		[ 2 : 0 ]	gray_cur_i ;
				case ( gray_cur_i )
					3'b000	:	gray_nxt	= 3'b001 ;
					3'b001	:	gray_nxt	= 3'b011 ;
					3'b011	:	gray_nxt	= 3'b010 ;
					3'b010	:	gray_nxt	= 3'b110 ;
					3'b110	:	gray_nxt	= 3'b111 ;
					3'b111	:	gray_nxt	= 3'b101 ;
					3'b101	:	gray_nxt	= 3'b100 ;
					default	:	gray_nxt	= 3'b000 ; //3'b100 case
				endcase
			endfunction
			assign	gray_i_nxt	= gray_nxt(gray_i_ff) ;
			assign	gray_o_nxt	= gray_nxt(gray_o_ff) ;
		end
		else if ( GRAY_DEPTH == 4 ) begin
			function	[ 3 : 0 ]	gray_nxt   ;
			input		[ 3 : 0 ]	gray_cur_i ;
				case ( gray_cur_i )
					4'b0000	:	gray_nxt	= 4'b0001 ;
					4'b0001 :	gray_nxt	= 4'b0011 ;
					4'b0011	:	gray_nxt	= 4'b0010 ;
					4'b0010 :	gray_nxt	= 4'b0110 ;
					4'b0110	:	gray_nxt	= 4'b0111 ;
					4'b0111 :	gray_nxt	= 4'b0101 ;
					4'b0101 :	gray_nxt	= 4'b0100 ;
					4'b0100 :	gray_nxt	= 4'b1100 ;
					4'b1100 :	gray_nxt	= 4'b1101 ;
					4'b1101 :	gray_nxt	= 4'b1111 ;
					4'b1111 :	gray_nxt	= 4'b1110 ;
					4'b1110 :	gray_nxt	= 4'b1010 ;
					4'b1010	:	gray_nxt	= 4'b1011 ;
					4'b1011 :	gray_nxt	= 4'b1001 ;
					4'b1001	:	gray_nxt	= 4'b1000 ;
					default	:	gray_nxt	= 4'b0000 ; //4'b1000 case
				endcase
			endfunction
			assign	gray_i_nxt	= gray_nxt(gray_i_ff) ;
			assign	gray_o_nxt	= gray_nxt(gray_o_ff) ;
		end
	endgenerate

endmodule
`default_nettype wire