//----------------------------------------------------------------------------------------
//Copyright (C) 2013 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		Simple Synchronizer with two Flip-Flops
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Feb. 19 2013	: Initial Version Release
//----------------------------------------------------------------------------------------
// PARAMETERS
//		BIT_WIDTH 	: Bitwidh of Input/Output data to be synchronized
//		SYNC_INPUT	: a F/F is needed or not just after data_i
//                    0: Not needed, 1:Needed
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk_in	: Clock signal that data_i is generated by
//		clk_out : Clock signal that data_o is to be synchronized with
//		rst_n	: reset_signal
//		data_i	: Input data
//		data_o	: Output data
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	cdc_ff
	#(
		parameter							BIT_WIDTH	= 8	,
		parameter							SYNC_INPUT	= 1
	)
	(
		input	wire						clk_in			,
		input	wire						clk_out			,
		input	wire						rst_n			,

		input	wire	[ BIT_WIDTH-1 : 0 ]	data_i			,
		output	wire	[ BIT_WIDTH-1 : 0 ]	data_o
	) ;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	wire	[ BIT_WIDTH-1	: 0	]			data_i_sync ;
	reg		[ BIT_WIDTH-1	: 0 ]			data_i_dly_b1_ff ;
	reg		[ BIT_WIDTH-1	: 0 ]			data_i_dly_b2_ff ;

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	// Sync clk_in
	generate
		if ( SYNC_INPUT == 1 ) begin //Synchronization is needed
			reg	[ BIT_WIDTH-1	: 0 ]		data_i_dly1_ff ;
			always @( posedge clk_in or negedge rst_n ) begin
				if ( !rst_n ) begin
					data_i_dly1_ff	<= { BIT_WIDTH{1'b0} } ;
				end
				else begin
					data_i_dly1_ff	<= data_i ;
				end
			end
			assign	data_i_sync	= data_i_dly1_ff ;
		end
		else begin //Synchronization is not needed
			assign	data_i_sync	= data_i ;
		end
	endgenerate

	// clk_out
	always @( posedge clk_out or negedge rst_n ) begin
		if ( !rst_n ) begin
			data_i_dly_b1_ff	<= { BIT_WIDTH{1'b0} } ;
			data_i_dly_b2_ff	<= { BIT_WIDTH{1'b0} } ;
		end
		else begin
			data_i_dly_b1_ff	<= data_i_sync ;
			data_i_dly_b2_ff	<= data_i_dly_b1_ff ;
		end
	end

	assign	data_o	= data_i_dly_b2_ff ;

endmodule
`default_nettype wire