//----------------------------------------------------------------------------------------
//Copyright (C) 2012 Macnica Inc. All Rights Reserved.
//
//Use in source and binary forms, with or without modification, are permitted provided
//by agreeing to the following terms and conditions:
//
//REDISTRIBUTIONS OR SUBLICENSING IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS "AS IS"
//AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE
//FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
//DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
//SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
//CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//AND ALSO REGARDING THE REFERENCE SOFTWARE, REDISTRIBUTION OR SUBLICENSING
//IN SOURCE AND BINARY FORM ARE NOT ALLOWED.
//----------------------------------------------------------------------------------------
// DESCRIPTION
//		multiplication circuit
//----------------------------------------------------------------------------------------
// REVISION HISTORY
//		v1.0 Nov. 13 2012	: Initial version release
//		v1.1 Jun. 04 2013	: Added ch#_on_prm_lat_o
//		v1.2 Apr. 25 2014	: Signal ch#_frame_start_rise are added
//
//----------------------------------------------------------------------------------------
// PARAMETERS
//		APORT_BIT	: "dataa_i" port bit width
//		BPORT_BIT	: "datab_i" port bit width
//
//----------------------------------------------------------------------------------------
// I/O PORTS
//		clk 				: clock for all circuit
//		rst_n				: asynchronous reset ( low active )
//		srst				: synchronous reset
//		param_update_i		: input updating setting parameter registers
//		end_flag_i			: frame_end generate timing
//		start_flag_i		: frame_start generate timing
//		all_off_i			: time VSI input off of frame_end
//
//	<< Input of the setting mode >>
//		smpl_mode_i			: input color format mode select
//							  2'b00 : 422 input
//							  2'b01 : 444 input
//		constant_data_i		: a value when outputting a constant data
//
//		width_i				: horizontal size of layer
//		height_i			: vertical size of layer
//
//		layer#_ch_i			: allocate channel for layer# ( # = 0, 1, 2, 3 )
//							  2'd0 : channel_0
//							  2'd1 : channel_1
//							  2'd2 : channel_2
//							  2'd3 : channel_3
//		ch#_on_i			: channel  ON/OFF
//							  1'b0 : OFF
//							  1'b1 : ON
//		ch#_alpha_i			: alpha value of channel#
//		ch#_hpos_i			: horizontal position of channel#
//		ch#_vpos_i			: vertical position of channel#
//		ch#_width_i			: horizontal size of channel#
//		ch#_height_i		: vertical size of channel#
//
//		ch#_frame_start_i	: frame input period of  channel#
//
//	<< Internal operation mode >>
//		smpl_mode_o			: input color format mode select
//							  2'b00 : 422 input
//							  2'b01 : 444 input
//		constant_data_o		: a value when outputting a constant data
//
//		width_o				: horizontal size of layer
//		height_o			: vertical size of layer
//		width_lat_o			: horizontal size of layer(  )
//		height_lat_o		: vertical size of layer(  )
//
//		ch#_on_o			: channel# ON/OFF 1'b1:OFF 1'b0:OFF
//
//		layer#_on_o			: layer# ON/OFF 1'b1:OFF 1'b0:OFF
//		layer#_ch_o			: allocate channel for layer# ( # = 0, 1, 2, 3 )
//		layer#_hstart_o		: horizontal start position of layer#
//		layer#_vstart_o		: vertical start position of layer#
//		layer#_hend_o		: horizontal end position of layer#
//		layer#_vend_o		: vertical end position of layer#
//
//		layer0_alpha0_o 	: alpha of layer0
//		layer0_alpha01_o 	: alpha of layer0 ( overlay laey 1 )
//		layer0_alpha02_o 	: alpha of layer0 ( overlay laey 2 )
//		layer0_alpha03_o 	: alpha of layer0 ( overlay laey 3 )
//		layer0_alpha012_o 	: alpha of layer0 ( overlay laey 1,2 )
//		layer0_alpha013_o 	: alpha of layer0 ( overlay laey 1,3 )
//		layer0_alpha023_o	: alpha of layer0 ( overlay laey 2,3 )
//		layer0_alpha0123_o	: alpha of layer0 ( overlay laey 1,2,3 )
//		layer1_alpha1_o		: alpha of layer1
//		layer1_alpha12_o	: alpha of layer1 ( overlay laey 2 )
//		layer1_alpha13_o	: alpha of layer1 ( overlay laey 3 )
//		layer1_alpha123_o	: alpha of layer1 ( overlay laey 2,3 )
//		layer2_alpha2_o		: alpha of layer2
//		layer2_alpha23_o	: alpha of layer2 ( overlay laey 3 )
//		layer3_alpha3_o		: alpha of layer3
//
//		init_end_o			: alpha calculation end
//
//----------------------------------------------------------------------------------------
`timescale 1ps/1ps
`default_nettype none

module	pc11_param (
	clk					,
	rst_n				,
	srst				,
	param_update_i		,
	end_flag_i			,
	start_flag_i		,
	all_off_i			,

	smpl_mode_i			,
	constant_data_i		,		//	444 input
	layer3_con_i		, // custom
	layer3_cdata_i		, // custom

	width_i				,
	height_i			,

	layer0_ch_i			,
	layer1_ch_i			,
	layer2_ch_i			,
	layer3_ch_i			,

	ch0_on_i			,
	ch0_alpha_i			,
	ch0_hpos_i			,
	ch0_vpos_i			,
	ch0_width_i			,
	ch0_height_i		,

	ch1_on_i			,
	ch1_alpha_i			,
	ch1_hpos_i			,
	ch1_vpos_i			,
	ch1_width_i			,
	ch1_height_i		,

	ch2_on_i			,
	ch2_alpha_i			,
	ch2_hpos_i			,
	ch2_vpos_i			,
	ch2_width_i			,
	ch2_height_i		,

	ch3_on_i			,
	ch3_alpha_i			,
	ch3_hpos_i			,
	ch3_vpos_i			,
	ch3_width_i			,
	ch3_height_i		,

	ch0_frame_start_i	,
	ch1_frame_start_i	,
	ch2_frame_start_i	,
	ch3_frame_start_i	,

	smpl_mode_o			,
	constant_data_o		,
	layer3_con_o		, // custom
	layer3_cdata_o		, // custom

	width_o				,
	height_o			,
	width_lat_o			,
	height_lat_o		,

	ch0_on_o			,
	ch1_on_o			,
	ch2_on_o			,
	ch3_on_o			,

	ch0_on_prm_lat_o	,		// v1.1
	ch1_on_prm_lat_o	,		// v1.1
	ch2_on_prm_lat_o	,		// v1.1
	ch3_on_prm_lat_o	,		// v1.1

	layer0_on_o			,
	layer0_ch_o			,
	layer0_hstart_o		,
	layer0_vstart_o		,
	layer0_hend_o		,
	layer0_vend_o		,
	layer1_on_o			,
	layer1_ch_o			,
	layer1_hstart_o		,
	layer1_vstart_o		,
	layer1_hend_o		,
	layer1_vend_o		,
	layer2_on_o			,
	layer2_ch_o			,
	layer2_hstart_o		,
	layer2_vstart_o		,
	layer2_hend_o		,
	layer2_vend_o		,
	layer3_on_o			,
	layer3_ch_o			,
	layer3_hstart_o		,
	layer3_vstart_o		,
	layer3_hend_o		,
	layer3_vend_o		,

	layer0_alpha0_o 	,
	layer0_alpha01_o 	,
	layer0_alpha02_o 	,
	layer0_alpha03_o 	,
	layer0_alpha012_o 	,
	layer0_alpha013_o 	,
	layer0_alpha023_o	,
	layer0_alpha0123_o	,
	layer1_alpha1_o		,
	layer1_alpha12_o	,
	layer1_alpha13_o	,
	layer1_alpha123_o	,
	layer2_alpha2_o		,
	layer2_alpha23_o	,
	layer3_alpha3_o		,
	init_end_o
	) ;


// =============================================================================
// PARAMETER DEFINITION
// =============================================================================

	// ---------------------------------------------------------------------
	// Below parameters have to be defined from upper module
	// ---------------------------------------------------------------------

	parameter	ALPHA_BIT		= 8		;
	parameter	SIZE_BIT		= 11	;
	parameter	PIXEL_BIT		= 24	;

	// ---------------------------------------------------------------------
	// Please do not change the following parameters
	// ---------------------------------------------------------------------

	localparam	MLTRSLT_BIT		= ( ALPHA_BIT + 1 ) * 2 ;


// =============================================================================
// PORT DECLARATION
// =============================================================================

	input	wire							clk					;
	input	wire							rst_n				;
	input	wire							srst				;
	input	wire							param_update_i		;
	input	wire							end_flag_i			;
	input	wire							start_flag_i		;
	input	wire							all_off_i			;

	input	wire							smpl_mode_i			;
	input	wire	[ PIXEL_BIT - 1 : 0 ]	constant_data_i		;
	input	wire							layer3_con_i		; // custom
	input	wire	[ PIXEL_BIT - 1 : 0 ]	layer3_cdata_i		; // custom

	input	wire	[  SIZE_BIT - 1 : 0 ]	width_i				;
	input	wire	[  SIZE_BIT - 1 : 0 ]	height_i			;

	input	wire	[             1 : 0 ]	layer0_ch_i			;
	input	wire	[             1 : 0 ]	layer1_ch_i			;
	input	wire	[             1 : 0 ]	layer2_ch_i			;
	input	wire	[             1 : 0 ]	layer3_ch_i			;

	input	wire							ch0_on_i			;
	input	wire	[ ALPHA_BIT - 1 : 0 ]	ch0_alpha_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch0_hpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch0_vpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch0_width_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch0_height_i		;

	input	wire							ch1_on_i			;
	input	wire	[ ALPHA_BIT - 1 : 0 ]	ch1_alpha_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch1_hpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch1_vpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch1_width_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch1_height_i		;

	input	wire							ch2_on_i			;
	input	wire	[ ALPHA_BIT - 1 : 0 ]	ch2_alpha_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch2_hpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch2_vpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch2_width_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch2_height_i		;

	input	wire							ch3_on_i			;
	input	wire	[ ALPHA_BIT - 1 : 0 ]	ch3_alpha_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch3_hpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch3_vpos_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch3_width_i			;
	input	wire	[  SIZE_BIT - 1 : 0 ]	ch3_height_i		;

	input	wire							ch0_frame_start_i	;
	input	wire							ch1_frame_start_i	;
	input	wire							ch2_frame_start_i	;
	input	wire							ch3_frame_start_i	;

	output	wire							smpl_mode_o			;
	output	wire	[ PIXEL_BIT - 1 : 0 ]	constant_data_o		;
	output	wire							layer3_con_o		; // custom
	output	wire	[ PIXEL_BIT - 1 : 0 ]	layer3_cdata_o		; // custom

	output	wire	[  SIZE_BIT - 1 : 0 ]	width_o				;
	output	wire	[  SIZE_BIT - 1 : 0 ]	height_o			;
	output	wire	[  SIZE_BIT - 1 : 0 ]	width_lat_o			;
	output	wire	[  SIZE_BIT - 1 : 0 ]	height_lat_o		;

	output	wire							ch0_on_o			;
	output	wire							ch1_on_o			;
	output	wire							ch2_on_o			;
	output	wire							ch3_on_o			;

	output	wire							layer0_on_o			;
	output	wire	[             1 : 0 ]	layer0_ch_o			;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer0_hstart_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer0_vstart_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer0_hend_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer0_vend_o		;
	output	wire							layer1_on_o			;
	output	wire	[             1 : 0 ]	layer1_ch_o			;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer1_hstart_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer1_vstart_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer1_hend_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer1_vend_o		;
	output	wire							layer2_on_o			;
	output	wire	[             1 : 0 ]	layer2_ch_o			;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer2_hstart_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer2_vstart_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer2_hend_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer2_vend_o		;
	output	wire							layer3_on_o			;
	output	wire	[             1 : 0 ]	layer3_ch_o			;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer3_hstart_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer3_vstart_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer3_hend_o		;
	output	wire	[  SIZE_BIT - 1 : 0 ]	layer3_vend_o		;

	output	wire	[     ALPHA_BIT : 0 ]	layer0_alpha0_o 	;
	output	wire	[     ALPHA_BIT : 0 ]	layer0_alpha01_o 	;
	output	wire	[     ALPHA_BIT : 0 ]	layer0_alpha02_o 	;
	output	wire	[     ALPHA_BIT : 0 ]	layer0_alpha03_o 	;
	output	wire	[     ALPHA_BIT : 0 ]	layer0_alpha012_o 	;
	output	wire	[     ALPHA_BIT : 0 ]	layer0_alpha013_o 	;
	output	wire	[     ALPHA_BIT : 0 ]	layer0_alpha023_o	;
	output	wire	[     ALPHA_BIT : 0 ]	layer0_alpha0123_o	;
	output	wire	[     ALPHA_BIT : 0 ]	layer1_alpha1_o		;
	output	wire	[     ALPHA_BIT : 0 ]	layer1_alpha12_o	;
	output	wire	[     ALPHA_BIT : 0 ]	layer1_alpha13_o	;
	output	wire	[     ALPHA_BIT : 0 ]	layer1_alpha123_o	;
	output	wire	[     ALPHA_BIT : 0 ]	layer2_alpha2_o		;
	output	wire	[     ALPHA_BIT : 0 ]	layer2_alpha23_o	;
	output	wire	[     ALPHA_BIT : 0 ]	layer3_alpha3_o		;

	output	wire							init_end_o			;

	output	wire							ch0_on_prm_lat_o	;
	output	wire							ch1_on_prm_lat_o	;
	output	wire							ch2_on_prm_lat_o	;
	output	wire							ch3_on_prm_lat_o	;

// =============================================================================
// REG / WIRE DECLARATION
// =============================================================================

	reg								param_up_lat0_ff		;
	reg								param_up_lat1_ff		;
	reg								param_up_lat2_ff		;
	wire							param_lat_timing		;
	reg								param_upwait_ff			;
	reg								param_update_ff			;
	wire							param_update			;

	reg								smpl_mode_lat_ff		;
	reg		[   PIXEL_BIT - 1 : 0 ]	constant_data_lat_ff	;
	reg		[    SIZE_BIT - 1 : 0 ]	width_lat_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	height_lat_ff			;
	reg		[               1 : 0 ]	layer0_ch_lat_ff		;
	reg		[               1 : 0 ]	layer1_ch_lat_ff		;
	reg		[               1 : 0 ]	layer2_ch_lat_ff		;
	reg		[               1 : 0 ]	layer3_ch_lat_ff		;
	reg								ch0_on_lat_ff			;
	reg		[   ALPHA_BIT - 1 : 0 ]	ch0_alpha_lat_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	ch0_hpos_lat_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	ch0_vpos_lat_ff			;
	reg								ch1_on_lat_ff			;
	reg		[   ALPHA_BIT - 1 : 0 ]	ch1_alpha_lat_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	ch1_hpos_lat_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	ch1_vpos_lat_ff			;
	reg								ch2_on_lat_ff			;
	reg		[   ALPHA_BIT - 1 : 0 ]	ch2_alpha_lat_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	ch2_hpos_lat_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	ch2_vpos_lat_ff			;
	reg								ch3_on_lat_ff			;
	reg		[   ALPHA_BIT - 1 : 0 ]	ch3_alpha_lat_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	ch3_hpos_lat_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	ch3_vpos_lat_ff			;
//  Alpha Calculate  //
	wire	[       ALPHA_BIT : 0 ]	layer0_alpha_base		;
	wire	[       ALPHA_BIT : 0 ]	layer1_alpha_base		;
	wire	[       ALPHA_BIT : 0 ]	layer2_alpha_base		;
	wire	[       ALPHA_BIT : 0 ]	layer3_alpha_base		;
	reg		[       ALPHA_BIT : 0 ]	layer0_alpha_base_ff	;
	reg		[       ALPHA_BIT : 0 ]	layer1_alpha_base_ff	;
	reg		[       ALPHA_BIT : 0 ]	layer2_alpha_base_ff	;
	reg		[       ALPHA_BIT : 0 ]	layer3_alpha_base_ff	;
	reg		[       ALPHA_BIT : 0 ]	mult_data_a				;
	reg		[       ALPHA_BIT : 0 ]	mult_data_b				;
	wire	[ MLTRSLT_BIT - 1 : 0 ]	mult_result				;
	reg		[       ALPHA_BIT : 0 ]	mult23_ff				;
	reg		[       ALPHA_BIT : 0 ]	mult13_ff				;
	reg		[       ALPHA_BIT : 0 ]	mult12_ff				;
	reg		[       ALPHA_BIT : 0 ]	mult123_ff				;
	reg		[       ALPHA_BIT : 0 ]	mult03_ff				;
	reg		[       ALPHA_BIT : 0 ]	mult02_ff				;
	reg		[       ALPHA_BIT : 0 ]	mult01_ff				;
	reg		[       ALPHA_BIT : 0 ]	mult023_ff				;
	reg		[       ALPHA_BIT : 0 ]	mult013_ff				;
	reg		[       ALPHA_BIT : 0 ]	mult012_ff				;
	reg		[       ALPHA_BIT : 0 ]	mult0123_ff				;
	reg		[       ALPHA_BIT : 0 ]	add_01_02_03_ff			;
	reg		[       ALPHA_BIT : 0 ]	add_012_013_023_ff		;
	//	Alpha calculater control
	reg		[               3 : 0 ]	init_count_ff			;
	reg								init_start_ff			;
	reg								init_end_ff				;
	//  parameter Update  //
	reg								ch0_frame_start_lat_ff	;
	reg								ch1_frame_start_lat_ff	;
	reg								ch2_frame_start_lat_ff	;
	reg								ch3_frame_start_lat_ff	;
	wire							frame_start_pls			;
	reg								smpl_mode_ff			;
	reg		[   PIXEL_BIT - 1 : 0 ]	constant_data_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	width_ff				;
	reg		[    SIZE_BIT - 1 : 0 ]	height_ff				;
	reg		[               1 : 0 ]	layer0_ch_ff			;
	reg		[               1 : 0 ]	layer1_ch_ff			;
	reg		[               1 : 0 ]	layer2_ch_ff			;
	reg		[               1 : 0 ]	layer3_ch_ff			;
	reg								ch0_on_ff				;
	reg								ch1_on_ff				;
	reg								ch2_on_ff				;
	reg								ch3_on_ff				;

	reg								layer0_on_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	layer0_hstart_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	layer0_vstart_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	layer0_hend_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	layer0_vend_ff			;
	reg		[       ALPHA_BIT : 0 ]	layer0_alpha0_ff 		;
	reg		[       ALPHA_BIT : 0 ]	layer0_alpha01_ff 		;
	reg		[       ALPHA_BIT : 0 ]	layer0_alpha02_ff 		;
	reg		[       ALPHA_BIT : 0 ]	layer0_alpha03_ff 		;
	reg		[       ALPHA_BIT : 0 ]	layer0_alpha012_ff 		;
	reg		[       ALPHA_BIT : 0 ]	layer0_alpha013_ff 		;
	reg		[       ALPHA_BIT : 0 ]	layer0_alpha023_ff		;
	reg		[       ALPHA_BIT : 0 ]	layer0_alpha0123_ff		;

	reg								layer1_on_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	layer1_hstart_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	layer1_vstart_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	layer1_hend_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	layer1_vend_ff			;
	reg		[       ALPHA_BIT : 0 ]	layer1_alpha1_ff		;
	reg		[       ALPHA_BIT : 0 ]	layer1_alpha12_ff		;
	reg		[       ALPHA_BIT : 0 ]	layer1_alpha13_ff		;
	reg		[       ALPHA_BIT : 0 ]	layer1_alpha123_ff		;
	reg								layer2_on_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	layer2_hstart_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	layer2_vstart_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	layer2_hend_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	layer2_vend_ff			;
	reg		[       ALPHA_BIT : 0 ]	layer2_alpha2_ff		;
	reg		[       ALPHA_BIT : 0 ]	layer2_alpha23_ff		;

	reg								layer3_on_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	layer3_hstart_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	layer3_vstart_ff		;
	reg		[    SIZE_BIT - 1 : 0 ]	layer3_hend_ff			;
	reg		[    SIZE_BIT - 1 : 0 ]	layer3_vend_ff			;
	reg		[       ALPHA_BIT : 0 ]	layer3_alpha3_ff		;

	reg								layer3_con_lat_ff		; // custom
	reg		[   PIXEL_BIT - 1 : 0 ]	layer3_cdata_lat_ff		; // custom
	reg								layer3_con_ff			; // custom
	reg		[   PIXEL_BIT - 1 : 0 ]	layer3_cdata_ff			; // custom



	wire	ch0_frame_start_rise							; //v1.2
	wire	ch1_frame_start_rise							; //v1.2
	wire	ch2_frame_start_rise							; //v1.2
	wire	ch3_frame_start_rise							; //v1.2

// =============================================================================
// FUNCTION DESCRIPTION
// =============================================================================

	assign	ch0_on_prm_lat_o	= ch0_on_lat_ff	;		// v1.1
	assign	ch1_on_prm_lat_o	= ch1_on_lat_ff	;		// v1.1
	assign	ch2_on_prm_lat_o	= ch2_on_lat_ff	;		// v1.1
	assign	ch3_on_prm_lat_o	= ch3_on_lat_ff	;		// v1.1


// ================================= //
//  Parameter Latch & Update Timing  //
// ================================= //

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			param_up_lat0_ff	<= 1'b0 ;
			param_up_lat1_ff	<= 1'b0 ;
			param_up_lat2_ff	<= 1'b0 ;
		end
		else begin
			param_up_lat0_ff	<= param_update_i ;
			param_up_lat1_ff	<= param_up_lat0_ff ;
			param_up_lat2_ff	<= param_up_lat1_ff  ;
		end
	end

	assign	param_lat_timing	= param_up_lat1_ff & ~param_up_lat2_ff ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			param_upwait_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			param_upwait_ff	<= 1'b0 ;
		end
		else if ( init_count_ff == 4'hE ) begin
			param_upwait_ff	<= 1'b1 ;
		end
		else if ( param_update_ff ) begin
			param_upwait_ff	<= 1'b0 ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			param_update_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			param_update_ff	<= 1'b0 ;
		end
		else if ( param_upwait_ff && ( start_flag_i || ( !init_end_ff && ( init_count_ff == 4'hF ) ) ) ) begin
			param_update_ff	<= 1'b1 ;
		end
		else begin
			param_update_ff	<= 1'b0 ;
		end
	end

	assign	param_update	= param_update_ff | ( init_end_ff & start_flag_i ) ;

	//	parameter latch

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			smpl_mode_lat_ff		<= 1'b0 ;
			constant_data_lat_ff	<= { PIXEL_BIT{1'b0} } ;
			layer3_con_lat_ff		<= 1'b0 ;					// custom
			layer3_cdata_lat_ff		<= { PIXEL_BIT{1'b0} } ;	// custom
			width_lat_ff			<= { SIZE_BIT{1'b0} } ;
			height_lat_ff			<= { SIZE_BIT{1'b0} } ;
			layer0_ch_lat_ff		<= 2'h0 ;
			layer1_ch_lat_ff		<= 2'h0 ;
			layer2_ch_lat_ff		<= 2'h0 ;
			layer3_ch_lat_ff		<= 2'h0 ;
			ch0_on_lat_ff			<= 1'b0 ;
			ch0_alpha_lat_ff		<= { ALPHA_BIT{1'b0} } ;
			ch0_hpos_lat_ff			<= { SIZE_BIT{1'b0} } ;
			ch0_vpos_lat_ff			<= { SIZE_BIT{1'b0} } ;
			ch1_on_lat_ff			<= 1'b0 ;
			ch1_alpha_lat_ff		<= { ALPHA_BIT{1'b0} } ;
			ch1_hpos_lat_ff			<= { SIZE_BIT{1'b0} } ;
			ch1_vpos_lat_ff			<= { SIZE_BIT{1'b0} } ;
			ch2_on_lat_ff			<= 1'b0 ;
			ch2_alpha_lat_ff		<= { ALPHA_BIT{1'b0} } ;
			ch2_hpos_lat_ff			<= { SIZE_BIT{1'b0} } ;
			ch2_vpos_lat_ff			<= { SIZE_BIT{1'b0} } ;
			ch3_on_lat_ff			<= 1'b0 ;
			ch3_alpha_lat_ff		<= { ALPHA_BIT{1'b0} } ;
			ch3_hpos_lat_ff			<= { SIZE_BIT{1'b0} } ;
			ch3_vpos_lat_ff			<= { SIZE_BIT{1'b0} } ;
		end
		else if ( param_lat_timing || srst ) begin
			smpl_mode_lat_ff		<= smpl_mode_i ;
			constant_data_lat_ff	<= constant_data_i ;
			layer3_con_lat_ff		<= layer3_con_i ;	// custom
			layer3_cdata_lat_ff		<= layer3_cdata_i ;	// custom
			width_lat_ff			<= width_i ;
			height_lat_ff			<= height_i ;
			layer0_ch_lat_ff		<= layer0_ch_i ;
			layer1_ch_lat_ff		<= layer1_ch_i ;
			layer2_ch_lat_ff		<= layer2_ch_i ;
			layer3_ch_lat_ff		<= layer3_ch_i ;
			ch0_on_lat_ff			<= ch0_on_i ;
			ch0_alpha_lat_ff		<= ch0_alpha_i ;
			ch0_hpos_lat_ff			<= ch0_hpos_i ;
			ch0_vpos_lat_ff			<= ch0_vpos_i ;
			ch1_on_lat_ff			<= ch1_on_i ;
			ch1_alpha_lat_ff		<= ch1_alpha_i ;
			ch1_hpos_lat_ff			<= ch1_hpos_i ;
			ch1_vpos_lat_ff			<= ch1_vpos_i ;
			ch2_on_lat_ff			<= ch2_on_i ;
			ch2_alpha_lat_ff		<= ch2_alpha_i ;
			ch2_hpos_lat_ff			<= ch2_hpos_i ;
			ch2_vpos_lat_ff			<= ch2_vpos_i ;
			ch3_on_lat_ff			<= ch3_on_i ;
			ch3_alpha_lat_ff		<= ch3_alpha_i ;
			ch3_hpos_lat_ff			<= ch3_hpos_i ;
			ch3_vpos_lat_ff			<= ch3_vpos_i ;
		end
	end

// ================= //
//  Alpha Calculate  //
// ================= //
	assign	layer0_alpha_base	= ( { ALPHA_BIT+1{layer0_ch_lat_ff == 2'b00} } & ( ch0_alpha_lat_ff + 1'b1 ) ) |
							  	  ( { ALPHA_BIT+1{layer0_ch_lat_ff == 2'b01} } & ( ch1_alpha_lat_ff + 1'b1 ) ) |
							  	  ( { ALPHA_BIT+1{layer0_ch_lat_ff == 2'b10} } & ( ch2_alpha_lat_ff + 1'b1 ) ) |
							  	  ( { ALPHA_BIT+1{layer0_ch_lat_ff == 2'b11} } & ( ch3_alpha_lat_ff + 1'b1 ) ) ;

	assign	layer1_alpha_base	= ( { ALPHA_BIT+1{layer1_ch_lat_ff == 2'b00} } & ( ch0_alpha_lat_ff + 1'b1 ) ) |
							  	  ( { ALPHA_BIT+1{layer1_ch_lat_ff == 2'b01} } & ( ch1_alpha_lat_ff + 1'b1 ) ) |
							  	  ( { ALPHA_BIT+1{layer1_ch_lat_ff == 2'b10} } & ( ch2_alpha_lat_ff + 1'b1 ) ) |
							  	  ( { ALPHA_BIT+1{layer1_ch_lat_ff == 2'b11} } & ( ch3_alpha_lat_ff + 1'b1 ) ) ;

	assign	layer2_alpha_base	= ( { ALPHA_BIT+1{layer2_ch_lat_ff == 2'b00} } & ( ch0_alpha_lat_ff + 1'b1 ) ) |
								  ( { ALPHA_BIT+1{layer2_ch_lat_ff == 2'b01} } & ( ch1_alpha_lat_ff + 1'b1 ) ) |
								  ( { ALPHA_BIT+1{layer2_ch_lat_ff == 2'b10} } & ( ch2_alpha_lat_ff + 1'b1 ) ) |
								  ( { ALPHA_BIT+1{layer2_ch_lat_ff == 2'b11} } & ( ch3_alpha_lat_ff + 1'b1 ) ) ;

	assign	layer3_alpha_base	= ( { ALPHA_BIT+1{layer3_ch_lat_ff == 2'b00} } & ( ch0_alpha_lat_ff + 1'b1 ) ) |
								  ( { ALPHA_BIT+1{layer3_ch_lat_ff == 2'b01} } & ( ch1_alpha_lat_ff + 1'b1 ) ) |
								  ( { ALPHA_BIT+1{layer3_ch_lat_ff == 2'b10} } & ( ch2_alpha_lat_ff + 1'b1 ) ) |
								  ( { ALPHA_BIT+1{layer3_ch_lat_ff == 2'b11} } & ( ch3_alpha_lat_ff + 1'b1 ) ) ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer0_alpha_base_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer1_alpha_base_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer2_alpha_base_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer3_alpha_base_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			layer0_alpha_base_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer1_alpha_base_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer2_alpha_base_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer3_alpha_base_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else begin
			layer0_alpha_base_ff	<= layer0_alpha_base ;
			layer1_alpha_base_ff	<= layer1_alpha_base ;
			layer2_alpha_base_ff	<= layer2_alpha_base ;
			layer3_alpha_base_ff	<= layer3_alpha_base ;
		end
	end


	always @( * ) begin
		case ( init_count_ff )
			4'h0	: mult_data_a	<= layer2_alpha_base ;
			4'h1	: mult_data_a	<= layer1_alpha_base ;
			4'h2	: mult_data_a	<= layer1_alpha_base ;
			4'h3	: mult_data_a	<= layer1_alpha_base ;
			4'h4	: mult_data_a	<= layer0_alpha_base ;
			4'h5	: mult_data_a	<= layer0_alpha_base ;
			4'h6	: mult_data_a	<= layer0_alpha_base ;
			4'h7	: mult_data_a	<= layer0_alpha_base ;
			4'h8	: mult_data_a	<= layer0_alpha_base ;
			4'h9	: mult_data_a	<= layer0_alpha_base ;
			default	: mult_data_a	<= mult01_ff ;
		endcase
	end

	always @( * ) begin
		case ( init_count_ff )
			4'h0	: mult_data_b	<= layer3_alpha_base ;
			4'h1	: mult_data_b	<= layer3_alpha_base ;
			4'h2	: mult_data_b	<= layer2_alpha_base ;
			4'h3	: mult_data_b	<= mult23_ff ;
			4'h4	: mult_data_b	<= layer3_alpha_base ;
			4'h5	: mult_data_b	<= layer2_alpha_base ;
			4'h6	: mult_data_b	<= layer1_alpha_base ;
			4'h7	: mult_data_b	<= mult23_ff ;
			4'h8	: mult_data_b	<= mult13_ff ;
			4'h9	: mult_data_b	<= mult12_ff ;
			default	: mult_data_b	<= mult23_ff ;
		endcase
	end

	pc11_unsigned_mult
		# (
			.APORT_BIT	( ALPHA_BIT + 1 ) ,
			.BPORT_BIT	( ALPHA_BIT + 1 )
		)
		alpha_calc_mult (
			.clk		( clk			) ,
			.rst_n		( rst_n			) ,
			.enable		( 1'b1			) ,
			.dataa_i	( mult_data_a	) ,
			.datab_i	( mult_data_b	) ,
			.result_o	( mult_result	)
		) ;

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult23_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult23_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'h2 ) begin
			mult23_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult13_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult13_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'h3 ) begin
			mult13_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult12_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult12_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'h4 ) begin
			mult12_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult123_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult123_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'h5 ) begin
			mult123_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult03_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult03_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'h6 ) begin
			mult03_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult02_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult02_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'h7 ) begin
			mult02_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult01_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult01_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'h8 ) begin
			mult01_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult023_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult023_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'h9 ) begin
			mult023_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult013_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult013_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'hA ) begin
			mult013_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult012_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult012_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'hB ) begin
			mult012_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			mult0123_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			mult0123_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'hC ) begin
			mult0123_ff	<= mult_result[MLTRSLT_BIT-2:ALPHA_BIT] ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			add_01_02_03_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			add_01_02_03_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'hD ) begin
			add_01_02_03_ff	<= mult01_ff + mult02_ff + mult03_ff ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			add_012_013_023_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			add_012_013_023_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( init_count_ff == 4'hD ) begin
			add_012_013_023_ff	<= mult012_ff + mult013_ff + mult023_ff ;
		end
	end


	//	Alpha calculater control
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			init_count_ff	<= 4'h0 ;
		end
		else if ( srst ) begin
			init_count_ff	<= 4'h0 ;
		end
		else if ( init_count_ff != 4'hF ) begin
			init_count_ff	<= init_count_ff + 1'b1 ;
		end
		else if ( init_start_ff ) begin
			init_count_ff	<= 4'h0 ;
		end
		else if ( init_end_ff == 1'b0 ) begin
			init_count_ff	<= 4'h0 ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			init_start_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			init_start_ff	<= 1'b0 ;
		end
		else begin
			init_start_ff	<= param_lat_timing ;
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			init_end_ff	<= 1'b0 ;
		end
		else if ( srst ) begin
			init_end_ff	<= 1'b0 ;
		end
		else if ( param_update ) begin
			init_end_ff	<= 1'b1 ;
		end
	end


// ================== //
//  parameter Update  //
// ================== //

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			ch0_frame_start_lat_ff	<= 1'b0 ;
			ch1_frame_start_lat_ff	<= 1'b0 ;
			ch2_frame_start_lat_ff	<= 1'b0 ;
			ch3_frame_start_lat_ff	<= 1'b0 ;
		end
		else begin
			ch0_frame_start_lat_ff	<= ch0_frame_start_i ;
			ch1_frame_start_lat_ff	<= ch1_frame_start_i ;
			ch2_frame_start_lat_ff	<= ch2_frame_start_i ;
			ch3_frame_start_lat_ff	<= ch3_frame_start_i ;
		end
	end

	assign	frame_start_pls	= ( ~ch0_frame_start_lat_ff & ch0_frame_start_i ) |
							  ( ~ch1_frame_start_lat_ff & ch1_frame_start_i ) |
							  ( ~ch2_frame_start_lat_ff & ch2_frame_start_i ) |
							  ( ~ch3_frame_start_lat_ff & ch3_frame_start_i ) ;

	assign	ch0_frame_start_rise	= ~ch0_frame_start_lat_ff & ch0_frame_start_i ; //v1.2
	assign	ch1_frame_start_rise	= ~ch1_frame_start_lat_ff & ch1_frame_start_i ; //v1.2
	assign	ch2_frame_start_rise	= ~ch2_frame_start_lat_ff & ch2_frame_start_i ; //v1.2
	assign	ch3_frame_start_rise	= ~ch3_frame_start_lat_ff & ch3_frame_start_i ; //v1.2

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			smpl_mode_ff		<= 1'b0 ;
			constant_data_ff	<= { PIXEL_BIT{1'b0} } ;
			layer3_con_ff		<= 1'b0 ;					// custom
			layer3_cdata_ff		<= { PIXEL_BIT{1'b0} } ;	// custom
			width_ff			<= { SIZE_BIT{1'b1} } ;
			height_ff			<= { SIZE_BIT{1'b1} } ;
			layer0_ch_ff		<= 2'h0 ;
			layer1_ch_ff		<= 2'h0 ;
			layer2_ch_ff		<= 2'h0 ;
			layer3_ch_ff		<= 2'h0 ;
			ch0_on_ff			<= 1'b0 ;
			ch1_on_ff			<= 1'b0 ;
			ch2_on_ff			<= 1'b0 ;
			ch3_on_ff			<= 1'b0 ;
		end
		else if ( param_update || end_flag_i ) begin
			smpl_mode_ff		<= smpl_mode_lat_ff ;
			constant_data_ff	<= constant_data_lat_ff ;
			layer3_con_ff		<= layer3_con_lat_ff & ~all_off_i ;	// custom
			layer3_cdata_ff		<= layer3_cdata_lat_ff ;			// custom
			width_ff			<= width_lat_ff - 1'b1 ;
			height_ff			<= height_lat_ff - 1'b1 ;
			layer0_ch_ff		<= layer0_ch_lat_ff ;
			layer1_ch_ff		<= layer1_ch_lat_ff ;
			layer2_ch_ff		<= layer2_ch_lat_ff ;
			layer3_ch_ff		<= layer3_ch_lat_ff ;
			ch0_on_ff			<= ch0_on_lat_ff & ~all_off_i ;
			ch1_on_ff			<= ch1_on_lat_ff & ~all_off_i ;
			ch2_on_ff			<= ch2_on_lat_ff & ~all_off_i ;
			ch3_on_ff			<= ch3_on_lat_ff & ~all_off_i ;
		end
	end

	//	layer 0 parameter
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer0_on_ff		<= 1'b0 ;
			layer0_hstart_ff	<= { SIZE_BIT{1'b0} } ;
			layer0_vstart_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( param_update || end_flag_i ) begin
			case ( layer0_ch_lat_ff )
				2'b00 : begin
					layer0_on_ff		<= ch0_on_lat_ff & ~all_off_i ;
					layer0_hstart_ff	<= ch0_hpos_lat_ff ;
					layer0_vstart_ff	<= ch0_vpos_lat_ff ;
				end
				2'b01 : begin
					layer0_on_ff		<= ch1_on_lat_ff & ~all_off_i ;
					layer0_hstart_ff	<= ch1_hpos_lat_ff ;
					layer0_vstart_ff	<= ch1_vpos_lat_ff ;
				end
				2'b10 : begin
					layer0_on_ff		<= ch2_on_lat_ff & ~all_off_i ;
					layer0_hstart_ff	<= ch2_hpos_lat_ff ;
					layer0_vstart_ff	<= ch2_vpos_lat_ff ;
				end
				2'b11 : begin
					layer0_on_ff		<= ch3_on_lat_ff & ~all_off_i ;
					layer0_hstart_ff	<= ch3_hpos_lat_ff ;
					layer0_vstart_ff	<= ch3_vpos_lat_ff ;
				end
				default : begin
					layer0_on_ff		<= ch0_on_lat_ff & ~all_off_i ;
					layer0_hstart_ff	<= ch0_hpos_lat_ff ;
					layer0_vstart_ff	<= ch0_vpos_lat_ff ;
				end
			endcase
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer0_hend_ff		<= { SIZE_BIT{1'b1} } ;
			layer0_vend_ff		<= { SIZE_BIT{1'b1} } ;
		end
		else if ( param_update || frame_start_pls ) begin
			case ( layer0_ch_lat_ff )
				2'b00 : begin
//v1.2					layer0_hend_ff		<= ( !ch0_frame_start_i ) ?	layer0_hend_ff :
					layer0_hend_ff		<= ( !ch0_frame_start_rise ) ?	layer0_hend_ff : //v1.2
																		ch0_hpos_lat_ff + ch0_width_i - 1'b1 ;
//v1.2					layer0_vend_ff		<= ( !ch0_frame_start_i ) ?	layer0_vend_ff :
					layer0_vend_ff		<= ( !ch0_frame_start_rise ) ?	layer0_vend_ff : //v1.2
																		ch0_vpos_lat_ff + ch0_height_i - 1'b1 ;
				end
				2'b01 : begin
//v1.2					layer0_hend_ff		<= ( !ch1_frame_start_i ) ?	layer0_hend_ff :
					layer0_hend_ff		<= ( !ch1_frame_start_rise ) ?	layer0_hend_ff : //v1.2
																	ch1_hpos_lat_ff + ch1_width_i - 1'b1 ;
//v1.2					layer0_vend_ff		<= ( !ch1_frame_start_i ) ?	layer0_vend_ff :
					layer0_vend_ff		<= ( !ch1_frame_start_rise ) ?	layer0_vend_ff : //v1.2
																	ch1_vpos_lat_ff + ch1_height_i - 1'b1 ;
				end
				2'b10 : begin
//v1.2					layer0_hend_ff		<= ( !ch2_frame_start_i ) ?	layer0_hend_ff :
					layer0_hend_ff		<= ( !ch2_frame_start_rise ) ?	layer0_hend_ff : //v1.2
																	ch2_hpos_lat_ff + ch2_width_i - 1'b1 ;
//v1.2					layer0_vend_ff		<= ( !ch2_frame_start_i ) ?	layer0_vend_ff :
					layer0_vend_ff		<= ( !ch2_frame_start_rise ) ?	layer0_vend_ff : //v1.2
																	ch2_vpos_lat_ff + ch2_height_i - 1'b1 ;
				end
				default : begin //v1.2
//v1.2				2'b11 : begin
//v1.2					layer0_hend_ff		<= ( !ch3_frame_start_i ) ?	layer0_hend_ff :
					layer0_hend_ff		<= ( !ch3_frame_start_rise ) ?	layer0_hend_ff : //v1.2
																	ch3_hpos_lat_ff + ch3_width_i - 1'b1 ;
//v1.2					layer0_vend_ff		<= ( !ch3_frame_start_i ) ?	layer0_vend_ff :
					layer0_vend_ff		<= ( !ch3_frame_start_rise ) ?	layer0_vend_ff : //v1.2
																	ch3_vpos_lat_ff + ch3_height_i - 1'b1 ;
				end
//v1.2				default : begin
//v1.2					layer0_hend_ff		<= ( !ch0_frame_start_i ) ?	layer0_hend_ff :
//v1.2																	ch0_hpos_lat_ff + ch0_width_i - 1'b1 ;
//v1.2					layer0_vend_ff		<= ( !ch0_frame_start_i ) ?	layer0_vend_ff :
//v1.2																	ch0_vpos_lat_ff + ch0_height_i - 1'b1 ;
//v1.2				end
			endcase
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer0_alpha0_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha01_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha02_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha03_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha012_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha013_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha023_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha0123_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			layer0_alpha0_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha01_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha02_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha03_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha012_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha013_ff 	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha023_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer0_alpha0123_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( param_update ) begin
			layer0_alpha0_ff 	<= layer0_alpha_base_ff ;
			layer0_alpha01_ff 	<= layer0_alpha_base_ff - mult01_ff ;
			layer0_alpha02_ff 	<= layer0_alpha_base_ff - mult02_ff ;
			layer0_alpha03_ff 	<= layer0_alpha_base_ff - mult03_ff ;
			layer0_alpha012_ff 	<= layer0_alpha_base_ff - mult01_ff - mult02_ff + mult012_ff ;
			layer0_alpha013_ff 	<= layer0_alpha_base_ff - mult01_ff - mult03_ff + mult013_ff ;
			layer0_alpha023_ff	<= layer0_alpha_base_ff - mult02_ff - mult03_ff + mult023_ff ;
			layer0_alpha0123_ff	<= layer0_alpha_base_ff - add_01_02_03_ff + add_012_013_023_ff - mult0123_ff ;
		end
	end

	//	layer 1 parameter
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer1_on_ff		<= 1'b0 ;
			layer1_hstart_ff	<= { SIZE_BIT{1'b0} } ;
			layer1_vstart_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( param_update || end_flag_i ) begin
			case ( layer1_ch_lat_ff )
				2'b00 : begin
					layer1_on_ff		<= ch0_on_lat_ff & ~all_off_i ;
					layer1_hstart_ff	<= ch0_hpos_lat_ff ;
					layer1_vstart_ff	<= ch0_vpos_lat_ff ;
				end
				2'b01 : begin
					layer1_on_ff		<= ch1_on_lat_ff & ~all_off_i ;
					layer1_hstart_ff	<= ch1_hpos_lat_ff ;
					layer1_vstart_ff	<= ch1_vpos_lat_ff ;
				end
				2'b10 : begin
					layer1_on_ff		<= ch2_on_lat_ff & ~all_off_i ;
					layer1_hstart_ff	<= ch2_hpos_lat_ff ;
					layer1_vstart_ff	<= ch2_vpos_lat_ff ;
				end
				2'b11 : begin
					layer1_on_ff		<= ch3_on_lat_ff & ~all_off_i ;
					layer1_hstart_ff	<= ch3_hpos_lat_ff ;
					layer1_vstart_ff	<= ch3_vpos_lat_ff ;
				end
				default : begin
					layer1_on_ff		<= ch0_on_lat_ff & ~all_off_i ;
					layer1_hstart_ff	<= ch0_hpos_lat_ff ;
					layer1_vstart_ff	<= ch0_vpos_lat_ff ;
				end
			endcase
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer1_hend_ff		<= { SIZE_BIT{1'b1} } ;
			layer1_vend_ff		<= { SIZE_BIT{1'b1} } ;
		end
		else if ( param_update || frame_start_pls ) begin
			case ( layer1_ch_lat_ff )
				2'b00 : begin
//v1.2					layer1_hend_ff		<= ( !ch0_frame_start_i ) ?	layer1_hend_ff :
					layer1_hend_ff		<= ( !ch0_frame_start_rise ) ?	layer1_hend_ff : //v1.2
																	ch0_hpos_lat_ff + ch0_width_i - 1'b1 ;
//v1.2					layer1_vend_ff		<= ( !ch0_frame_start_i ) ?	layer1_vend_ff :
					layer1_vend_ff		<= ( !ch0_frame_start_rise ) ?	layer1_vend_ff : //v1.2
																	ch0_vpos_lat_ff + ch0_height_i - 1'b1 ;
				end
				2'b01 : begin
//v1.2					layer1_hend_ff		<= ( !ch1_frame_start_i ) ?	layer1_hend_ff :
					layer1_hend_ff		<= ( !ch1_frame_start_rise ) ?	layer1_hend_ff : //v1.2
																	ch1_hpos_lat_ff + ch1_width_i - 1'b1 ;
//v1.2					layer1_vend_ff		<= ( !ch1_frame_start_i ) ?	layer1_vend_ff :
					layer1_vend_ff		<= ( !ch1_frame_start_rise ) ?	layer1_vend_ff : //v1.2
																	ch1_vpos_lat_ff + ch1_height_i - 1'b1 ;
				end
				2'b10 : begin
//v1.2					layer1_hend_ff		<= ( !ch2_frame_start_i ) ?	layer1_hend_ff :
					layer1_hend_ff		<= ( !ch2_frame_start_rise ) ?	layer1_hend_ff : //v1.2
																	ch2_hpos_lat_ff + ch2_width_i - 1'b1 ;
//v1.2					layer1_vend_ff		<= ( !ch2_frame_start_i ) ?	layer1_vend_ff :
					layer1_vend_ff		<= ( !ch2_frame_start_rise ) ?	layer1_vend_ff : //v1.2
																	ch2_vpos_lat_ff + ch2_height_i - 1'b1 ;
				end
				default : begin //v1.2
//v1.2				2'b11 : begin
//v1.2					layer1_hend_ff		<= ( !ch3_frame_start_i ) ?	layer1_hend_ff :
					layer1_hend_ff		<= ( !ch3_frame_start_rise ) ?	layer1_hend_ff :
																	ch3_hpos_lat_ff + ch3_width_i - 1'b1 ;
//v1.2					layer1_vend_ff		<= ( !ch3_frame_start_i ) ?	layer1_vend_ff :
					layer1_vend_ff		<= ( !ch3_frame_start_rise ) ?	layer1_vend_ff : //v1.2
																	ch3_vpos_lat_ff + ch3_height_i - 1'b1 ;
				end
//v1.2				default : begin
//v1.2					layer1_hend_ff		<= ( !ch0_frame_start_i ) ?	layer1_hend_ff :
//v1.2																	ch0_hpos_lat_ff + ch0_width_i - 1'b1 ;
//v1.2					layer1_vend_ff		<= ( !ch0_frame_start_i ) ?	layer1_vend_ff :
//v1.2																	ch0_vpos_lat_ff + ch0_height_i - 1'b1 ;
//v1.2				end
			endcase
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer1_alpha1_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer1_alpha12_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer1_alpha13_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer1_alpha123_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			layer1_alpha1_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer1_alpha12_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer1_alpha13_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer1_alpha123_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( param_update ) begin
			layer1_alpha1_ff	<= layer1_alpha_base_ff ;
			layer1_alpha12_ff	<= layer1_alpha_base_ff - mult12_ff ;
			layer1_alpha13_ff	<= layer1_alpha_base_ff - mult13_ff ;
			layer1_alpha123_ff	<= layer1_alpha_base_ff - mult12_ff - mult13_ff + mult123_ff ;
		end
	end

	//	layer 2 parameter
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer2_on_ff		<= 1'b0 ;
			layer2_hstart_ff	<= { SIZE_BIT{1'b0} } ;
			layer2_vstart_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( param_update || end_flag_i ) begin
			case ( layer2_ch_lat_ff )
				2'b00 : begin
					layer2_on_ff		<= ch0_on_lat_ff & ~all_off_i ;
					layer2_hstart_ff	<= ch0_hpos_lat_ff ;
					layer2_vstart_ff	<= ch0_vpos_lat_ff ;
				end
				2'b01 : begin
					layer2_on_ff		<= ch1_on_lat_ff & ~all_off_i ;
					layer2_hstart_ff	<= ch1_hpos_lat_ff ;
					layer2_vstart_ff	<= ch1_vpos_lat_ff ;
				end
				2'b10 : begin
					layer2_on_ff		<= ch2_on_lat_ff & ~all_off_i ;
					layer2_hstart_ff	<= ch2_hpos_lat_ff ;
					layer2_vstart_ff	<= ch2_vpos_lat_ff ;
				end
				2'b11 : begin
					layer2_on_ff		<= ch3_on_lat_ff & ~all_off_i ;
					layer2_hstart_ff	<= ch3_hpos_lat_ff ;
					layer2_vstart_ff	<= ch3_vpos_lat_ff ;
				end
				default : begin
					layer2_on_ff		<= ch0_on_lat_ff & ~all_off_i ;
					layer2_hstart_ff	<= ch0_hpos_lat_ff ;
					layer2_vstart_ff	<= ch0_vpos_lat_ff ;
				end
			endcase
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer2_hend_ff		<= { SIZE_BIT{1'b1} } ;
			layer2_vend_ff		<= { SIZE_BIT{1'b1} } ;
		end
		else if ( param_update || frame_start_pls ) begin
			case ( layer2_ch_lat_ff )
				2'b00 : begin
//v1.2					layer2_hend_ff		<= ( !ch0_frame_start_i ) ?	layer2_hend_ff :
					layer2_hend_ff		<= ( !ch0_frame_start_rise ) ?	layer2_hend_ff : //v1.2
																	ch0_hpos_lat_ff + ch0_width_i - 1'b1 ;
//v1.2					layer2_vend_ff		<= ( !ch0_frame_start_i ) ?	layer2_vend_ff :
					layer2_vend_ff		<= ( !ch0_frame_start_rise ) ?	layer2_vend_ff : //v1.2
																	ch0_vpos_lat_ff + ch0_height_i - 1'b1 ;
				end
				2'b01 : begin
//v1.2					layer2_hend_ff		<= ( !ch1_frame_start_i ) ?	layer2_hend_ff :
					layer2_hend_ff		<= ( !ch1_frame_start_rise ) ?	layer2_hend_ff : //v1.2
																	ch1_hpos_lat_ff + ch1_width_i - 1'b1 ;
//v1.2					layer2_vend_ff		<= ( !ch1_frame_start_i ) ?	layer2_vend_ff :
					layer2_vend_ff		<= ( !ch1_frame_start_rise ) ?	layer2_vend_ff : //v1.2
																	ch1_vpos_lat_ff + ch1_height_i - 1'b1 ;
				end
				2'b10 : begin
//v1.2					layer2_hend_ff		<= ( !ch2_frame_start_i ) ?	layer2_hend_ff :
					layer2_hend_ff		<= ( !ch2_frame_start_rise ) ?	layer2_hend_ff : //v1.2
																	ch2_hpos_lat_ff + ch2_width_i - 1'b1 ;
//v1.2					layer2_vend_ff		<= ( !ch2_frame_start_i ) ?	layer2_vend_ff :
					layer2_vend_ff		<= ( !ch2_frame_start_rise ) ?	layer2_vend_ff : //v1.2
																	ch2_vpos_lat_ff + ch2_height_i - 1'b1 ;
				end
				default : begin //v1.2
//v1.2				2'b11 : begin
//v1.2					layer2_hend_ff		<= ( !ch3_frame_start_i ) ?	layer2_hend_ff :
					layer2_hend_ff		<= ( !ch3_frame_start_rise ) ?	layer2_hend_ff : //v1.2
																	ch3_hpos_lat_ff + ch3_width_i - 1'b1 ;
//v1.2					layer2_vend_ff		<= ( !ch3_frame_start_i ) ?	layer2_vend_ff :
					layer2_vend_ff		<= ( !ch3_frame_start_rise ) ?	layer2_vend_ff : //v1.2
																	ch3_vpos_lat_ff + ch3_height_i - 1'b1 ;
				end
//v1.2				default : begin
//v1.2					layer2_hend_ff		<= ( !ch0_frame_start_i ) ?	layer2_hend_ff :
//v1.2																	ch0_hpos_lat_ff + ch0_width_i - 1'b1 ;
//v1.2					layer2_vend_ff		<= ( !ch0_frame_start_i ) ?	layer2_vend_ff :
//v1.2																	ch0_vpos_lat_ff + ch0_height_i - 1'b1 ;
//v1.2				end
			endcase
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer2_alpha2_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer2_alpha23_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			layer2_alpha2_ff	<= { ALPHA_BIT+1{1'b0} } ;
			layer2_alpha23_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( param_update ) begin
			layer2_alpha2_ff	<= layer2_alpha_base_ff ;
			layer2_alpha23_ff	<= layer2_alpha_base_ff - mult23_ff ;
		end
	end

	//	layer 3 parameter
	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer3_on_ff		<= 1'b0 ;
			layer3_hstart_ff	<= { SIZE_BIT{1'b0} } ;
			layer3_vstart_ff	<= { SIZE_BIT{1'b0} } ;
		end
		else if ( param_update || end_flag_i ) begin
			case ( layer3_ch_lat_ff )
				2'b00 : begin
					layer3_on_ff		<= ch0_on_lat_ff & ~all_off_i ;
					layer3_hstart_ff	<= ch0_hpos_lat_ff ;
					layer3_vstart_ff	<= ch0_vpos_lat_ff ;
				end
				2'b01 : begin
					layer3_on_ff		<= ch1_on_lat_ff & ~all_off_i ;
					layer3_hstart_ff	<= ch1_hpos_lat_ff ;
					layer3_vstart_ff	<= ch1_vpos_lat_ff ;
				end
				2'b10 : begin
					layer3_on_ff		<= ch2_on_lat_ff & ~all_off_i ;
					layer3_hstart_ff	<= ch2_hpos_lat_ff ;
					layer3_vstart_ff	<= ch2_vpos_lat_ff ;
				end
				2'b11 : begin
					layer3_on_ff		<= ch3_on_lat_ff & ~all_off_i ;
					layer3_hstart_ff	<= ch3_hpos_lat_ff ;
					layer3_vstart_ff	<= ch3_vpos_lat_ff ;
				end
				default : begin
					layer3_on_ff		<= ch0_on_lat_ff & ~all_off_i ;
					layer3_hstart_ff	<= ch0_hpos_lat_ff ;
					layer3_vstart_ff	<= ch0_vpos_lat_ff ;
				end
			endcase
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer3_hend_ff		<= { SIZE_BIT{1'b1} } ;
			layer3_vend_ff		<= { SIZE_BIT{1'b1} } ;
		end
		else if ( param_update || frame_start_pls ) begin
			case ( layer3_ch_lat_ff )
				2'b00 : begin
//v1.2					layer3_hend_ff		<= ( !ch0_frame_start_i ) ?	layer3_hend_ff :
					layer3_hend_ff		<= ( !ch0_frame_start_rise ) ?	layer3_hend_ff : //v1.2
																	ch0_hpos_lat_ff + ch0_width_i - 1'b1 ;
//v1.2					layer3_vend_ff		<= ( !ch0_frame_start_i ) ?	layer3_vend_ff :
					layer3_vend_ff		<= ( !ch0_frame_start_rise ) ?	layer3_vend_ff : //v1.2
																	ch0_vpos_lat_ff + ch0_height_i - 1'b1 ;
				end
				2'b01 : begin
//v1.2					layer3_hend_ff		<= ( !ch1_frame_start_i ) ?	layer3_hend_ff :
					layer3_hend_ff		<= ( !ch1_frame_start_rise ) ?	layer3_hend_ff : //v1.2
																	ch1_hpos_lat_ff + ch1_width_i - 1'b1 ;
//v1.2					layer3_vend_ff		<= ( !ch1_frame_start_i ) ?	layer3_vend_ff :
					layer3_vend_ff		<= ( !ch1_frame_start_rise ) ?	layer3_vend_ff : //v1.2
																	ch1_vpos_lat_ff + ch1_height_i - 1'b1 ;
				end
				2'b10 : begin
//v1.2					layer3_hend_ff		<= ( !ch2_frame_start_i ) ?	layer3_hend_ff :
					layer3_hend_ff		<= ( !ch2_frame_start_rise ) ?	layer3_hend_ff : //v1.2
																	ch2_hpos_lat_ff + ch2_width_i - 1'b1 ;
//v1.2					layer3_vend_ff		<= ( !ch2_frame_start_i ) ?	layer3_vend_ff :
					layer3_vend_ff		<= ( !ch2_frame_start_rise ) ?	layer3_vend_ff : //v1.2
																	ch2_vpos_lat_ff + ch2_height_i - 1'b1 ;
				end
				default : begin
//v1.2				2'b11 : begin
//v1.2					layer3_hend_ff		<= ( !ch3_frame_start_i ) ?	layer3_hend_ff :
					layer3_hend_ff		<= ( !ch3_frame_start_rise ) ?	layer3_hend_ff : //v1.2
																	ch3_hpos_lat_ff + ch3_width_i - 1'b1 ;
//v1.2					layer3_vend_ff		<= ( !ch3_frame_start_i ) ?	layer3_vend_ff :
					layer3_vend_ff		<= ( !ch3_frame_start_rise ) ?	layer3_vend_ff : //v1.2
																	ch3_vpos_lat_ff + ch3_height_i - 1'b1 ;
				end
//v1.2				default : begin
//v1.2					layer3_hend_ff		<= ( !ch0_frame_start_i ) ?	layer3_hend_ff :
//v1.2																	ch0_hpos_lat_ff + ch0_width_i - 1'b1 ;
//v1.2					layer3_vend_ff		<= ( !ch0_frame_start_i ) ?	layer3_vend_ff :
//v1.2																	ch0_vpos_lat_ff + ch0_height_i - 1'b1 ;
//v1.2				end
			endcase
		end
	end

	always @( posedge clk or negedge rst_n ) begin
		if ( !rst_n ) begin
			layer3_alpha3_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( srst ) begin
			layer3_alpha3_ff	<= { ALPHA_BIT+1{1'b0} } ;
		end
		else if ( param_update ) begin
			layer3_alpha3_ff	<= layer3_alpha_base_ff ;
		end
	end

// =============== //
//  output signal  //
// =============== //
	assign	smpl_mode_o			= smpl_mode_ff			;
	assign	constant_data_o		= constant_data_ff		;

	assign	width_o				= width_ff				;
	assign	height_o			= height_ff				;
	assign	width_lat_o			= width_lat_ff			;
	assign	height_lat_o		= height_lat_ff			;

	assign	ch0_on_o			= ch0_on_ff				;
	assign	ch1_on_o			= ch1_on_ff				;
	assign	ch2_on_o			= ch2_on_ff				;
	assign	ch3_on_o			= ch3_on_ff				;

	assign	layer0_ch_o			= layer0_ch_ff			;
	assign	layer1_ch_o			= layer1_ch_ff			;
	assign	layer2_ch_o			= layer2_ch_ff			;
	assign	layer3_ch_o			= layer3_ch_ff			;

	assign	layer0_hstart_o		= layer0_hstart_ff		;
	assign	layer0_vstart_o		= layer0_vstart_ff		;
	assign	layer0_hend_o		= layer0_hend_ff		;
	assign	layer0_vend_o		= layer0_vend_ff		;
	assign	layer1_hstart_o		= layer1_hstart_ff		;
	assign	layer1_vstart_o		= layer1_vstart_ff		;
	assign	layer1_hend_o		= layer1_hend_ff		;
	assign	layer1_vend_o		= layer1_vend_ff		;
	assign	layer2_hstart_o		= layer2_hstart_ff		;
	assign	layer2_vstart_o		= layer2_vstart_ff		;
	assign	layer2_hend_o		= layer2_hend_ff		;
	assign	layer2_vend_o		= layer2_vend_ff		;
	assign	layer3_hstart_o		= layer3_hstart_ff		;
	assign	layer3_vstart_o		= layer3_vstart_ff		;
	assign	layer3_hend_o		= layer3_hend_ff		;
	assign	layer3_vend_o		= layer3_vend_ff		;

	assign	layer0_on_o			= layer0_on_ff			;
	assign	layer0_alpha0_o 	= layer0_alpha0_ff 		;
	assign	layer0_alpha01_o 	= layer0_alpha01_ff 	;
	assign	layer0_alpha02_o 	= layer0_alpha02_ff 	;
	assign	layer0_alpha03_o 	= layer0_alpha03_ff 	;
	assign	layer0_alpha012_o 	= layer0_alpha012_ff 	;
	assign	layer0_alpha013_o 	= layer0_alpha013_ff 	;
	assign	layer0_alpha023_o	= layer0_alpha023_ff	;
	assign	layer0_alpha0123_o	= layer0_alpha0123_ff	;
	assign	layer1_on_o			= layer1_on_ff			;
	assign	layer1_alpha1_o		= layer1_alpha1_ff		;
	assign	layer1_alpha12_o	= layer1_alpha12_ff		;
	assign	layer1_alpha13_o	= layer1_alpha13_ff		;
	assign	layer1_alpha123_o	= layer1_alpha123_ff	;
	assign	layer2_on_o			= layer2_on_ff			;
	assign	layer2_alpha2_o		= layer2_alpha2_ff		;
	assign	layer2_alpha23_o	= layer2_alpha23_ff		;
	assign	layer3_on_o			= layer3_on_ff			;
	assign	layer3_alpha3_o		= layer3_alpha3_ff		;

	assign	layer3_con_o		= layer3_con_ff			;	// custom
	assign	layer3_cdata_o		= layer3_cdata_ff		;	// custom

	assign	init_end_o			= init_end_ff			;

endmodule

`default_nettype wire
