// (C) 2001-2025 Altera Corporation. All rights reserved.
// Your use of Altera Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Altera Program License Subscription 
// Agreement, Altera IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Altera and sold by 
// Altera or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


// Module: intel_eth_gts_xcvr_resync_std
//
// Description:
//  A general purpose resynchronization module that uses the recommended intel_eth_gts_altera_std_synchronizer
//  and intel_eth_gts_altera_std_synchronizer_nocut synchronizer
//  
//  Parameters:
//    SYNC_CHAIN_LENGTH
//      - Specifies the length of the synchronizer chain for metastability
//        retiming.
//    WIDTH
//      - Specifies the number of bits you want to synchronize. Controls the width of the
//        d and q ports.
//    SLOW_CLOCK - USE WITH CAUTION. 
//      - Leaving this setting at its default will create a standard resynch circuit that
//        merely passes the input data through a chain of flip-flops. This setting assumes
//        that the input data has a pulse width longer than one clock cycle sufficient to
//        satisfy setup and hold requirements on at least one clock edge.
//      - By setting this to 1 (USE CAUTION) you are creating an asynchronous
//        circuit that will capture the input data regardless of the pulse width and 
//        its relationship to the clock. However it is more difficult to apply static
//        timing constraints as it ties the data input to the clock input of the flop.
//        This implementation assumes the data rate is slow enough
//    INIT_VALUE
//      - Specifies the initial values of the synchronization registers.
//	  NO_CUT
//		- Specifies whether to apply embedded false path timing constraint. 
//		  0: Apply the constraint 1: Not applying the constraint
//

module dphy_gts_xcvr_resync_std #(
    parameter SYNC_CHAIN_LENGTH = 2,  // Number of flip-flops for retiming. Must be >1
    parameter WIDTH             = 1,  // Number of bits to resync
    parameter SLOW_CLOCK        = 0,  // See description above
    parameter INIT_VALUE        = 0,
    parameter NO_CUT		= 1	  // See description above
  ) (
  input   wire              clk,
  input   wire              reset,
  input   wire  [WIDTH-1:0] d,
  output  wire  [WIDTH-1:0] q
  );

localparam  INT_LEN       = (SYNC_CHAIN_LENGTH > 1) ? SYNC_CHAIN_LENGTH : 2;
localparam  L_INIT_VALUE  = (INIT_VALUE == 1) ? 1'b1 : 1'b0;

genvar ig;

// Generate a synchronizer chain for each bit
generate for(ig=0;ig<WIDTH;ig=ig+1) begin : resync_chains
	wire                d_in;   // Input to sychronization chain.
	wire				sync_d_in;
	wire		        sync_q_out;
	
	// Adding inverter to the input of first sync register and output of the last sync register to implement power-up high for INIT_VALUE=1
	assign sync_d_in = (INIT_VALUE == 1) ? ~d_in : d_in;
	assign q[ig] = (INIT_VALUE == 1)  ? ~sync_q_out : sync_q_out;
	
	// NOT SUPPORTED if (NO_CUT == 0) begin		
	// NOT SUPPORTED 	intel_eth_gts_altera_std_synchronizer #(
	// NOT SUPPORTED 		.depth(INT_LEN)				
	// NOT SUPPORTED 	) synchronizer (
	// NOT SUPPORTED 		.clk		(clk),
	// NOT SUPPORTED 		.reset_n	(~reset),
	// NOT SUPPORTED 		.din		(sync_d_in),
	// NOT SUPPORTED 		.dout		(sync_q_out)
	// NOT SUPPORTED 	);
	// NOT SUPPORTED 	
	// NOT SUPPORTED 	//synthesis translate_off			
	// NOT SUPPORTED 	initial begin
	// NOT SUPPORTED 		synchronizer.dreg = {(INT_LEN-1){1'b0}};
	// NOT SUPPORTED 		synchronizer.din_s1 = 1'b0;
	// NOT SUPPORTED 	end
	// NOT SUPPORTED 	//synthesis translate_on
	// NOT SUPPORTED 			
	// NOT SUPPORTED end else begin
		dphy_gts_std_synchronizer_nocut #(
			.depth(INT_LEN)				
		) synchronizer_nocut (
			.clk		(clk),
			.reset_n	(~reset),
			.din		(sync_d_in),
			.dout		(sync_q_out)
		);
				
		//synthesis translate_off
		initial begin
			synchronizer_nocut.dreg = {(INT_LEN-1){1'b0}};
			synchronizer_nocut.din_s1 = 1'b0;
		end
		//synthesis translate_on	
	// NOT SUPPORTED end
	
    // Generate asynchronous capture circuit if specified.
    if(SLOW_CLOCK == 0) begin
      assign  d_in = d[ig];
    end else begin
      wire  d_clk;
      reg   d_r = L_INIT_VALUE;
      wire  clr_n;

      assign  d_clk = d[ig];
      assign  d_in  = d_r;
      assign  clr_n = ~q[ig] | d_clk; // Clear when output is logic 1 and input is logic 0

      // Asynchronously latch the input signal.
      always @(posedge d_clk or negedge clr_n)
        if(!clr_n)      d_r <= 1'b0;
        else if(d_clk)  d_r <= 1'b1;
    end // SLOW_CLOCK
  end // for loop
endgenerate

endmodule


`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "5Conti+gYNYo9cSL/gyDnw/m0Cu7CADdtO6YoN4G91pw+rbOEGOK98vhOwNwkl4AW674AE6MpJOjDkdGUtFAsBC2xNImiT/XyMfyEPyCWOM8ZpzbZM0QOzl/DgRH1iDvBEWZ8SIwvWC9QH/PsUWf5suzYwhMelcG6f3lg94mhnCkB2HTY9JeYYVIrUn6e3Gn2DNSg4KlNfi3zSU3SPv5miApSjh43Tk5sGHlkLyXvXUc9eLf/s9liuV9gfoEjSRN87F/BPmNOtzSIACN3JSx1O+wEDYfL4fEDvy0/KrlO4cbXvAl2LDmY5Im0L8a0brTlXm1jHQqdGhBlWUsxEAf0ld5ZCufU14k5yCFBTZCQnP57Yx4bUp5jl9WHMgjifsqfdZxpPe82J4rFn1wFAtfnsDXKG0HV3u9PT81fVHYCgyitL/dGhZmsLgoG1LRpf8j8gI0W/5hq2C+VWYtitI39RFN7OSLa+3hs5LRYH9gIF5XYx+XEAVPwYpSCdh2v4rQvFvAG0eyqZXZXYudv8pX33MgMAjbdnbEJfrhuSJDs+1Sj9ca+0qbsdaXuW1WYBkC28nhs9ajmC2ASiAktiUuTvn91bmqQ19LQNGVEPjAlCtwfvRjV5UJkfigSgyGILoYPFNf/48TbtQDAAS5U24+Jr+XMra5bqSuAgZXX6NXaF4QA7lJ31mK+PJh/1tSHAHGLE5E+pQNE02oa6oBwuVBl+y1nYKMZulbBJvQCXmV2Id5cRCEDfCxEZKu5iulAPPLUYEWu28vuXCD0reehtR60lKIfHQ4tk4Ij4XP1doP2M068KDqngYQc+aItWqR8GsaUS+Jecr0Bncx96lmC3aHg/F3c1+Z5FuFincKRj29Glpy8NzH87liX8gQnYOix9kHQLvMP/PAzaB16m20at6BTHSzdHr8Op8FIUoh4Gtsue8kJZuGkgUJj3OdxE8l7zLr43eZdNE4quVfGk9O6ixPKDHEU7QA+4UQl2svz6nAyc8O8quwTc3N6QcxsAqQY3DM"
`endif