# (C) 2001-2025 Altera Corporation. All rights reserved.
# Your use of Altera Corporation's design tools, logic functions and other 
# software and tools, and its AMPP partner logic functions, and any output 
# files from any of the foregoing (including device programming or simulation 
# files), and any associated documentation or information are expressly subject 
# to the terms and conditions of the Altera Program License Subscription 
# Agreement, Altera IP License Agreement, or other applicable 
# license agreement, including, without limitation, that your use is for the 
# sole purpose of programming logic devices manufactured by Altera and sold by 
# Altera or its authorized distributors.  Please refer to the applicable 
# agreement for further details.


# Check whether a project has already been opened
if {[catch {get_current_project}]} {
    post_message -type error "Please open a project first"
    return
}

proc intel_lvds_get_timequest_name { full_hier_name } {
    set tq_name ""
    set tq_name_list {}
    set lib_inst_pair_list [split $full_hier_name "|"]

    foreach i_lib_inst_pair_str $lib_inst_pair_list {
        set i_lib_inst_pair [split $i_lib_inst_pair_str ":"]

        if {[llength $i_lib_inst_pair] > 1} {
            lappend tq_name_list [join [lrange $i_lib_inst_pair 1 end] :]
        } else {
            lappend tq_name_list [lindex $i_lib_inst_pair end]
        }
    }

    set tq_name [join $tq_name_list "|"]
    return $tq_name
}

proc intel_lvds_get_core_full_instance_list {corename} {
    set instance_list [list]

    if {[is_fitter_in_qhd_mode]} {
        set instance_list_pre [design::get_instances -entity $corename]
    } else {
        set instance_list_pre [get_entity_instances $corename]
    }

    foreach instance $instance_list_pre {
        lappend instance_list [intel_lvds_get_timequest_name $instance]
    }

    if {[ llength $instance_list ] == 0} {
        post_message -type error "The auto-constraining script was not able to detect any instance for core < $corename >"
        post_message -type error "Make sure the core < $corename > is instantiated within another component (wrapper)"
        post_message -type error "and it's not the top-level for your project"
    }

    return $instance_list
}

proc intel_lvds_get_clock_name_from_target { target } {
   set clock_name [list]
   foreach_in_collection i [get_clocks -nowarn] {
      if {![is_clock_defined $i]} {
         continue
      }
      if { [catch { get_node_info -name  [get_clock_info -targets $i] } i_target ] } {
         continue
      }
      if {[string equal $target $i_target]} {
         lappend clock_name [get_clock_info -name $i]
      }
   }
   return $clock_name
}

proc intel_lvds_create_generated_clock {args} {
    array set opts {-invert 0 \
                    -add 0 \
                    -remove_clock 1 \
                    -name "" \
                    -target "" \
                    -source "" \
                    -multiply_by 1 \
                    -divide_by 1 \
                    -phase 0 \
                    -duty_cycle 50.00 \
                    -master_clock ""}

    array set opts $args

    set multiply_by [expr int($opts(-multiply_by))]
    if {[expr $multiply_by - $opts(-multiply_by)] != 0.0} {
        post_message -type error "Specify an integer ranging from 0 to 99999999 for the option -multiply_by"
        return ""
    }
        
    if {$opts(-remove_clock) == 1} {
        set clock_name_to_remove [intel_lvds_get_clock_name_from_target $opts(-target)]
        if {$clock_name_to_remove != ""} {
            foreach i $clock_name_to_remove {
                remove_clock  [get_clock_info -name $i]
            }
        }
    }
        
    set extra_params [list]
    if {$opts(-invert) == 1} {
        lappend extra_params "-invert"
    }
    if {$opts(-add) == 1} {
        lappend extra_params "-add"
    }

    if {$opts(-master_clock) ne ""} {
        lappend extra_params "-master_clock"
        lappend extra_params "$opts(-master_clock)"
    }

    if {$extra_params == ""} {
        eval {create_generated_clock -name $opts(-name) -source $opts(-source) -multiply_by $multiply_by -divide_by $opts(-divide_by) -phase $opts(-phase) -duty_cycle $opts(-duty_cycle) $opts(-target)}
    } else {
        eval {create_generated_clock {*}$extra_params -name $opts(-name) -source $opts(-source) -multiply_by $multiply_by -divide_by $opts(-divide_by) -phase $opts(-phase) -duty_cycle $opts(-duty_cycle) $opts(-target)}
    }
}

proc intel_lvds_get_clock_source {node} {
    set clock_node ""
    catch {get_node_info -name [get_edge_info -src [get_node_info -clock_edges $node]]} clock_node

    if {$clock_node eq ""} {
        post_message -type error "Failed to find clock source of node '$node'"
    }

    return $clock_node
}

proc intel_lvds_byte_used_for_tx {&params byte} {
    upvar 1 ${&params} ip_params
    foreach pin {0001 0203 0405 0607 0809 1011} {
        if {$ip_params(byte_${byte}_pin_${pin}) eq "tx"} {
            return "true"
        }
    }
    return "false"
}

proc intel_lvds_ch_number {&params byte ch} {
    upvar 1 ${&params} ip_params
    set pins [dict create 0 0001 1 0203 2 0405 3 0607 4 0809 5 1011]
    set pin [dict get $pins $ch]

    if {$ip_params(byte_${byte}_pin_${pin}) ne "unused"} {
        return $ip_params(byte_${byte}_pin_${pin}_ch)
    }

    return -1
}

proc intel_lvds_byte_used_for_rx {&params byte} {
    upvar 1 ${&params} ip_params
    foreach pin {0001 0203 0405 0607 0809 1011} {
        if {$ip_params(byte_${byte}_pin_${pin}) eq "non_dpa" || 
            $ip_params(byte_${byte}_pin_${pin}) eq "fifo"    || 
            $ip_params(byte_${byte}_pin_${pin}) eq "cdr"} {
            return "true"
        }
    }
    return "false"
}

proc intel_lvds_ch_used_for_rx {&params byte ch} {
    upvar 1 ${&params} ip_params
    set pins [dict create 0 0001 1 0203 2 0405 3 0607 4 0809 5 1011]
    set pin [dict get $pins $ch]
    if {$ip_params(byte_${byte}_pin_${pin}) eq "non_dpa" || 
        $ip_params(byte_${byte}_pin_${pin}) eq "fifo"    || 
        $ip_params(byte_${byte}_pin_${pin}) eq "cdr"} {
        return "true"
    }
    return "false"
}

proc intel_lvds_ch_used_for_cdr {&params byte ch} {
    upvar 1 ${&params} ip_params
    set pins [dict create 0 0001 1 0203 2 0405 3 0607 4 0809 5 1011]
    set pin [dict get $pins $ch]
    if {$ip_params(byte_${byte}_pin_${pin}) eq "cdr"} {
        return "true"
    }
    return "false"
}

proc intel_lvds_ch_used_for_dpa {&params byte ch} {
    upvar 1 ${&params} ip_params
    set pins [dict create 0 0001 1 0203 2 0405 3 0607 4 0809 5 1011]
    set pin [dict get $pins $ch]
    if {$ip_params(byte_${byte}_pin_${pin}) eq "cdr" || 
        $ip_params(byte_${byte}_pin_${pin}) eq "fifo"} {
        return "true"
    }
    return "false"
}

proc intel_lvds_length_of_collection {collection} {
    set length 0
    foreach_in_collection item $collection {
        incr length
    }
    return $length
}

proc intel_lvds_set_multicycle_or_false_path {start_end from to setup hold} {
    upvar 1 fit_flow l_fit_flow

    if {$l_fit_flow == 1} {
        intel_lvds_set_multicycle $start_end $from $to $setup $hold
    } else {
        set to_regs [get_registers -nowarn "${to}"]
        set from_regs [get_registers -nowarn "${from}"]
        if {[intel_lvds_length_of_collection $to_regs] > 0 && [intel_lvds_length_of_collection $from_regs] > 0} {
            set_false_path -from $from_regs -to $to_regs
        }
    }
}

proc intel_lvds_set_multicycle {start_end from to setup hold} {
    set_multicycle_path "-${start_end}" -from [get_registers "${from}"] -to [get_registers "${to}"] -setup $setup
    set_multicycle_path "-${start_end}" -from [get_registers "${from}"] -to [get_registers "${to}"] -hold  $hold
}

#proc: intel_lvds_get_iopll_atom
#
#  Given a lvds core instance name it finds the IOPLL that connects to the phyclk port
#  
#
#  parameters:
#  lvds_core_instance_name -> string (the one ending in "arch_inst")
#
#  returns:
#  atom id (empty if not found)
#
proc intel_lvds_get_iopll_atom {lvds_core_instance_name} {
    set atom_type IOPLL
    set search_iter 10

    set pll_atom_1 [intel_lvds_get_atom_by_lvds_input $lvds_core_instance_name $atom_type $search_iter "i_phy_clk_fr"]
    set pll_atom_2 [intel_lvds_get_atom_by_lvds_input $lvds_core_instance_name $atom_type $search_iter "i_vco8phs_clk"]
    set pll_atoms [lsort -unique [concat $pll_atom_1 $pll_atom_2]]
    return $pll_atoms
}

proc intel_lvds_get_atom_by_lvds_input { lvds_core_instance_name atom_type search_iter lvds_input} {
    set nodes [intel_lvds_get_lvds_nodes_with_clock_edge $lvds_core_instance_name $lvds_input]
    if {![llength $nodes]} {
        return ""
    }

    set all_atoms {}
    foreach node $nodes {
        set found_atom [intel_lvds_search_back_from_node_to_atom $node $atom_type $search_iter]
        set all_atoms [concat $all_atoms $found_atom]
    }
    return $all_atoms
}

proc intel_lvds_get_lvds_nodes_with_clock_edge { lvds_core_instance_name lvds_input } {
    set nodes {}
    set node ""
    foreach_in_collection node [get_nodes -nowarn "$lvds_core_instance_name*$lvds_input*"] {
        set edges [get_node_info -clock_edges $node]
        if {$edges ne "" } {
            lappend nodes $node
        }
    }
    return $nodes
}

proc intel_lvds_node_is_type { node atom_type } {
    set cell [get_node_info -cell $node]
    set cell_name [get_cell_info -name $cell]                
    set ix [string last _ $cell]
    if {$ix < 0} {
        return ""
    }
    incr ix 1
    set cell_id [string range $cell $ix [string length $cell]]
    set cell_type [get_atom_node_info -key TYPE -node $cell_id]
 
    if {![string match -nocase "*$atom_type*" $cell_type]} {
        return ""
    }
 
    set atom_name [get_atom_node_info -key NAME -node $cell_id]
    if {$atom_name eq $cell_name} {
        return $cell_id
    }
    return ""
}

proc intel_lvds_search_back_from_node_to_atom { node atom_type search_iter} {
    set found_node [intel_lvds_node_is_type $node $atom_type]
    set found_atoms {}
    if {$found_node ne ""} {
        lappend found_atoms $found_node
        return $found_atoms
    }
    if {$search_iter <= 0} {
        return ""
    }
    incr search_iter -1
    foreach edge [get_node_info -clock_edges $node] {
        set src_node [get_edge_info -src $edge]
        set found_node [intel_lvds_search_back_from_node_to_atom $src_node $atom_type $search_iter]

        if {$found_node ne ""} {
            lappend found_atoms $found_node
        }
    }
    return $found_atoms
}

