// (C) 2001-2025 Altera Corporation. All rights reserved.
// Your use of Altera Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Altera Program License Subscription 
// Agreement, Altera IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Altera and sold by 
// Altera or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


//
// Description
//-----------------------------------------------------------------------------
//
//   A general purpose resynchronization module that uses the recommended altera_std_synchronizer 
//   and altera_std_synchronizer_nocut synchronizer
//  
//   Parameters:
//         SYNC_CHAIN_LENGTH
//               Length of the synchronizer chain for metastability retiming.
//         WIDTH
//               Number of bits to synchronize. Controls the width of the d and q ports.
//         INIT_VALUE
//               Initial values of the synchronization registers.
//         NO_CUT
//               0 : Enable embedded set_false path SDC
//               1 : DIsable embedded set_false_path SDC
//        
//-----------------------------------------------------------------------------

`timescale 1ps / 1ps 

module fim_resync #(
   parameter SYNC_CHAIN_LENGTH = 2,  // Number of flip-flops for retiming. Must be >1
   parameter WIDTH             = 1,  // Number of bits to resync
   parameter INIT_VALUE        = 0,
   parameter NO_CUT		= 1	  // See description above
)(
   input  logic              clk,
   input  logic              reset,
   input  logic  [WIDTH-1:0] d,
   output logic  [WIDTH-1:0] q
);

localparam  INT_LEN       = (SYNC_CHAIN_LENGTH > 1) ? SYNC_CHAIN_LENGTH : 2;
localparam  L_INIT_VALUE  = (INIT_VALUE == 1) ? 1'b1 : 1'b0;

genvar ig;

// Generate a synchronizer chain for each bit
generate
   for(ig=0;ig<WIDTH;ig=ig+1) begin : resync_chains
      wire d_in;   // Input to sychronization chain.
      wire sync_d_in;
      wire sync_q_out;
      
      assign d_in = d[ig];
      
      // Adding inverter to the input of first sync register and output of the last sync register to implement power-up high for INIT_VALUE=1
      assign sync_d_in = (INIT_VALUE == 1) ? ~d_in : d_in;
      assign q[ig] = (INIT_VALUE == 1)  ? ~sync_q_out : sync_q_out;
      
      if (NO_CUT == 0) begin
         // Synchronizer with embedded set_false_path SDC
         altera_std_synchronizer #(
            .depth(INT_LEN)				
         ) synchronizer (
            .clk      (clk),
            .reset_n  (~reset),
            .din      (sync_d_in),
            .dout     (sync_q_out)
         );
         
         //synthesis translate_off			
         initial begin
            synchronizer.dreg = {(INT_LEN-1){1'b0}};
            synchronizer.din_s1 = 1'b0;
         end
         //synthesis translate_on

      end else begin
         // Synchronizer WITHOUT embedded set_false_path SDC
         altera_std_synchronizer_nocut #(
            .depth(INT_LEN)
         ) synchronizer_nocut (
            .clk      (clk),
            .reset_n  (~reset),
            .din      (sync_d_in),
            .dout     (sync_q_out)
         );

         //synthesis translate_off
         initial begin
            synchronizer_nocut.dreg = {(INT_LEN-1){1'b0}};
            synchronizer_nocut.din_s1 = 1'b0;
         end
         //synthesis translate_on	
      end
   end // for loop
endgenerate
endmodule

`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "CMmtAmKt5npgltBGSGayMD3S82LbrOyoIRJhaGPVZ9+1rQURt/zpqG45ZW8bu0uYgqdmqml0eEvdD4TRxWnYMa9qVt3DD8wDZFgDyC+4hk8LQe0gOhIT6pyripLwDe8s9OPDkE17E7KIjy+zKcRGPIEWW+S8TS0X4ob1FffXDsCtGX9BGI7Obc9NfOqzu4hBrmZJ8WSoLLvXaI3YwuY52dAiOf8n+195f1Ogp89gbg+MoGz14MkJ1qvUScvL+/FnSDUFl4OVgG43tEj6Tr3ZC8RJEFXolnZ0Ys1nHUUFQ2o+osWA39dCtB0LFWHkfQaWfIv33OX3i41hwBGRnwbO+2uQ2UAb7vDjlaMMD1qOzViuDOxSnuNXhV8/2fflBuGL1/TF/gx7J5LhiOxUobeus5SbKFN2jKLEyRlnBdObh4rsFup20EWLBUDZRY6v8qz88UsjssXCph55XLCl5Tyv6w6oLm4pkAdwUTQkpPZ6GAIviPqiOyZxh1se6/FolWAX+bqHhq0HDZ/35bcLKcWveLGqij9qVLSI4V86D9U19wKOaaZ/Lb1jkYbL/eRaqeDJNDz1ufsCOnuXoH+C+LbKMIXK1At0Al/+FMCzK3Ys3bdFgQ23ityIPDJFDnbWgqhfnHA6h72aElHUzg2EqUldWXNAQd0GjYt217vmX9p06GFlZaSUMrns1b8N5lWTt2gk6181u3Dg1hVzZ5WRd4g2drYehJOUx/0wdU5cfhoJpsWypt82/pPEg5WgAomZzrYjaqSmW0/waG7G+Acc7pqmjrB8TRI9ntH7KimLqG3AFwWDLfYDmfIUP962wPQKrXrHNoX5R6pioVqsfczChOJowkaay+v9/HIKBi/iUCw4d3iBu90Y6AXJ8ONWwsnAG24qJAOmm07srw7iv2TMiKn8sVMSB6mrimtgSaAmX9T16Mn36qjYZKCbPJT0RdzxvwaOfrijWrqaj6NaiBbVrPLPOTRrSlvcfCe6iH5tx7WMk3mLjhZXNXZWVVuFZAmRq2mm"
`endif