
module LOOPBACK_PATTERN(
	clk,
	reset_n,
	
	// start & result
	test_start, // restart a test, edge trigger
	test_done,  // assert when all patterns are test once
	test_error_mask,
	
	// tested bus
	Port_A,
	Port_B
);

parameter PATTERN_WIDTH	= 32;
parameter IO_BIDIRECTION	= 1;

	input											clk;
	input											reset_n;
	
	// start & result
	input											test_start; // restart a test, edge trigger
	output										test_done; // assert when all patterns are test once
	output	[(PATTERN_WIDTH-1):0] 	test_error_mask;
	
	// tested bus
	inout	[(PATTERN_WIDTH-1):0] 		Port_A;
	inout	[(PATTERN_WIDTH-1):0] 		Port_B;

//==============================
// pattern generator
wire [(PATTERN_WIDTH-1):0]  pattern;
wire									 pat_start;

PATTERN_GEN pattern_free_run(
	.clk(clk),
	.reset_n(reset_n & ~start_trigger),
	
	// start & result
	.pat_start(pat_start), // rise edge trigger
	.pattern(pattern)
);

defparam pattern_free_run.PATTERN_WIDTH = PATTERN_WIDTH;


//==============================
// start trigger


//==============================
// write pattern to output port
reg										ff_dir;
reg	[(PATTERN_WIDTH-1):0]	tx_pattern;

// apply in next clock
always @(posedge clk or negedge reset_n)
begin
	if (~reset_n)
		ff_dir	<= 1'b1;
	else if (pat_start & IO_BIDIRECTION)
		ff_dir <= ~ff_dir;
end

// apply in next clock
always @(posedge clk)
begin
	tx_pattern <= pattern;
end

assign Port_A = ff_dir?tx_pattern:{(PATTERN_WIDTH){1'bz}};
assign Port_B = ff_dir?{(PATTERN_WIDTH){1'bz}}:tx_pattern;

//==============================
// verify loopback result

// edge trigger
reg pre_test_start;
wire start_trigger;
assign start_trigger = ~pre_test_start & test_start;
always @(posedge clk or negedge reset_n)
begin
	if (~reset_n)
		pre_test_start <= 1'b0;
	else
		pre_test_start <= test_start;
end

// pattern ready?
reg pattern_ready;
always @(posedge clk or negedge reset_n)
begin
	if (~reset_n)
		pattern_ready <= 1'b0;
	else if (start_trigger)
		pattern_ready <= 1'b0;
	else if (pat_start)
		pattern_ready <= 1'b1;
end

// check result
reg	[(PATTERN_WIDTH-1):0] error;
wire	[(PATTERN_WIDTH-1):0] new_error;
wire	[(PATTERN_WIDTH-1):0] rx_data;

assign rx_data = (ff_dir)?Port_B:Port_A;
assign new_error = tx_pattern ^ rx_data;

assign test_error_mask = error;



always @(posedge clk or negedge reset_n)
begin
	if (~reset_n)
		error <= 0;
	else if (start_trigger)
		error <= 0;
	else if (pattern_ready)
		error <= error | new_error;
end

// done ?
reg one_cycle_done;
reg [15:0] cnt;

always @(posedge clk or negedge reset_n)
begin
	if (~reset_n)
		cnt <= 0;
	else if (start_trigger)
		cnt <= 0;
	else if (pat_start)
	begin
		if (cnt < 16'hFFFF) 
			cnt <= cnt + 1;
	end

end

always @(posedge clk or negedge reset_n)
begin
	if (~reset_n)
		one_cycle_done <= 1'b0;
	else if (start_trigger)
		one_cycle_done <= 1'b0;
	else if (cnt >= (IO_BIDIRECTION?3:2)) 
		one_cycle_done <= 1'b1;
end


assign test_done = one_cycle_done;



endmodule

