module axi4_master #(
	parameter LPDDR4_AXI_ADDR_W			= 32,
	parameter LPDDR4_AXI_ADDR_W_COUNT	= 32'h2000,
	parameter LPDDR4_AXI_ADDR_W_START	= 32'h00_0000,
	parameter LPDDR4_AXI_ADDR_W_END		= 32'h1ff_e000,
	parameter LPDDR4_AXI_DATA_W			= 256,
	parameter LPDDR4_AXI_BURST_LENGHT	= 8'hff,
	parameter LPDDR4_AXI_ID_W				= 4

) (
  input wire             aclk,
  input wire             aresetn,
  input wire				 insert_error_n,
  input wire  [35:0] 	 SEED_p,   

  // AXI4 Master interface
  output wire            					awvalid,
  input wire             					awready,
  output wire [LPDDR4_AXI_ID_W-1:0] 	awid,
  output wire [LPDDR4_AXI_ADDR_W-1:0] awaddr,
  output wire [7:0]       					awlen,
  output wire [2:0]       					awsize,
  output wire [1:0]       					awburst,
  output wire             					awlock,
  output wire [3:0]       					awcache,
  output wire [2:0]       					awprot,

  output wire            					wvalid,
  input wire             					wready,
  output wire [LPDDR4_AXI_DATA_W-1:0] wdata,
  output wire [(LPDDR4_AXI_DATA_W/8)-1:0] wstrb,
  output wire             					wlast,

  input wire             bvalid,
  output wire            bready,
  input wire [LPDDR4_AXI_ID_W-1:0] bid,
  input wire [1:0]       bresp,

  output wire            arvalid,
  input wire             arready,
  output wire [LPDDR4_AXI_ID_W-1:0] arid,
  output wire [LPDDR4_AXI_ADDR_W-1:0] araddr,
  output wire [7:0]       arlen,
  output wire [2:0]       arsize,
  output wire [1:0]       arburst,
  output wire             arlock,
  output wire [3:0]       arcache,
  output wire [2:0]       arprot,

  input wire             rvalid,
  output wire            rready,
  input wire [LPDDR4_AXI_ID_W-1:0] rid,
  input wire [LPDDR4_AXI_DATA_W-1:0] rdata,
  input wire [1:0]       rresp,
  input wire             rlast,

  // result output
  output reg            test_complete,
  output wire           test_pass
);

  // state machine
  localparam STATE_WRITE_IDLE		= 4'b0000,
             STATE_WRITE_ADDR		= 4'b0001,
             STATE_WRITE_DATA		= 4'b0010,
             STATE_WRITE_RESP		= 4'b0011,
				 STATE_WRITE_FINISH 	= 4'b0100,
				 
				 STATE_READ_IDLE 		= 4'b0101,
             STATE_READ_ADDR 		= 4'b0110,
             STATE_READ_DATA		= 4'b0111,
             STATE_READ_WAIT		= 4'b1000,
				 STATE_READ_FINISH 	= 4'b1001;
				 
  reg [3:0] write_current_state 	= STATE_WRITE_IDLE;
  reg [3:0] read_current_state 	= STATE_READ_IDLE;
  

  // AXI4 signal
  reg awvalid_reg									= 1'b0;
  reg [LPDDR4_AXI_ID_W-1:0] awid_reg		= {LPDDR4_AXI_ID_W{1'b0}};
  reg [LPDDR4_AXI_ADDR_W-1:0] awaddr_reg = {LPDDR4_AXI_ADDR_W{1'b0}};
  reg [7:0] awlen_reg							= 8'b0;
  reg [2:0] awsize_reg							= 3'b101; // 256-bit
  reg [1:0] awburst_reg							= 2'b01; // INCR
  reg awlock_reg									= 1'b0;
  reg [3:0] awcache_reg							= 4'b0000;
  reg [2:0] awprot_reg							= 3'b000;
  reg wvalid_reg									= 1'b0;
  reg [LPDDR4_AXI_DATA_W-1:0] wdata_reg	= {LPDDR4_AXI_DATA_W{1'b0}};
  reg [(LPDDR4_AXI_DATA_W/8)-1:0] wstrb_reg = {(LPDDR4_AXI_DATA_W/8){1'b1}};
  reg wlast_reg									= 1'b0;
  reg bready_reg									= 1'b0;
  reg arvalid_reg									= 1'b0;
  reg [LPDDR4_AXI_ID_W-1:0] arid_reg		= {LPDDR4_AXI_ID_W{1'b0}};
  reg [LPDDR4_AXI_ADDR_W-1:0] araddr_reg = {LPDDR4_AXI_ADDR_W{1'b0}};
  reg [7:0] arlen_reg							= 8'b0;
  reg [2:0] arsize_reg							= 3'b101; // 256-bit
  reg [1:0] arburst_reg							= 2'b01; // INCR
  reg arlock_reg									= 1'b0;
  reg [3:0] arcache_reg							= 4'b0000;
  reg [2:0] arprot_reg							= 3'b000;
  reg rready_reg									= 1'b0;
  reg	write_finish								= 1'b0;

  
  // burst count
  reg [7:0] write_burst_count = 8'h0;
  reg [7:0] read_burst_count = 8'h0;
  

wire	[LPDDR4_AXI_DATA_W-1:0] 	axi_writedata;
wire	[LPDDR4_AXI_DATA_W-1:0] 	axi_expected_data;

wire pattern_w_en; 
assign pattern_w_en = (wvalid_reg & wready) || ((write_current_state == STATE_WRITE_DATA) && wready && !write_burst_count);
 
altera_emif_avl_tg_lfsr_wrapper write_data_gen_inst (
	.clk		(aclk),
	.reset_n	(aresetn),
	.SEED		(SEED_p),
	.enable	(pattern_w_en),
	.data		(axi_writedata)
	);
defparam write_data_gen_inst.DATA_WIDTH = LPDDR4_AXI_DATA_W;
	
// Pseudo-random data generator
altera_emif_avl_tg_lfsr_wrapper read_data_gen_inst (
	.clk		(aclk),
	.reset_n	(aresetn & insert_error_n),
	.SEED		(SEED_p),
	.enable	(rvalid & rready),
	.data		(axi_expected_data)
	);
defparam read_data_gen_inst.DATA_WIDTH = LPDDR4_AXI_DATA_W;

	
	
// write
always @(posedge aclk or negedge aresetn) 
begin
	if (!aresetn) begin
		awvalid_reg <= 1'b0;
      wvalid_reg <= 1'b0;
      bready_reg <= 1'b1;
      write_current_state <= STATE_WRITE_IDLE;
      write_burst_count <= LPDDR4_AXI_BURST_LENGHT;
		write_finish <= 1'b0;
   end 
	else begin
		case (write_current_state)
			STATE_WRITE_IDLE: begin
				//init 
				awvalid_reg <= 1'b1;
				awid_reg <= 4'd0;
				awaddr_reg <= LPDDR4_AXI_ADDR_W_START; 
				awlen_reg <= LPDDR4_AXI_BURST_LENGHT; 
				write_current_state <= STATE_WRITE_ADDR;
			end
			
			STATE_WRITE_ADDR: begin 
				if (awready) begin
					awvalid_reg <= 1'b0;
					wvalid_reg <= 1'b1;
					write_burst_count <= awlen_reg;
					write_finish <= 1'b0;
					write_current_state <= STATE_WRITE_DATA;
				end
				else
					write_current_state <= write_current_state;
			end
			
			STATE_WRITE_DATA: begin
				if (wready) begin
					if (write_burst_count == 8'd0) begin
						wlast_reg <= 1'b0;
						wvalid_reg <= 1'b0;
						write_current_state <= STATE_WRITE_RESP;
					end 
					else if(write_burst_count == 8'd1) begin
						write_burst_count <= write_burst_count - 8'd1;
						wlast_reg <= 1'b1;
					end 
					else begin
						write_burst_count <= write_burst_count - 8'd1;
					end
				end
				else begin
					write_current_state <= write_current_state;
				end
			end
		  
			STATE_WRITE_RESP: begin
				wlast_reg <= 1'b0;
				if (bvalid) begin
					if(awaddr_reg == LPDDR4_AXI_ADDR_W_END) begin
						write_finish <= 1'b1;			
						write_current_state <= STATE_WRITE_FINISH;
					end
					else begin
						awvalid_reg <= 1'b1;
						awid_reg <= 4'd0;
						//byte address
						awaddr_reg <= awaddr_reg + LPDDR4_AXI_ADDR_W_COUNT; 
						write_burst_count<= LPDDR4_AXI_BURST_LENGHT;
						write_current_state <= STATE_WRITE_ADDR;	
					end
				end
				else
					write_current_state <= write_current_state;
			end
		  
		  STATE_WRITE_FINISH: begin
				write_current_state <= write_current_state;
		  end
		  
		  default: write_current_state <= STATE_WRITE_IDLE;
      endcase
    end
  end
//
//  read
always @(posedge aclk or negedge aresetn) 
begin
	if (!aresetn) begin
		arvalid_reg <= 1'b0;
      rready_reg <= 1'b0;
      read_burst_count <= LPDDR4_AXI_BURST_LENGHT;
		araddr_reg <= LPDDR4_AXI_ADDR_W_START;
		test_complete <= 1'b0;
		read_current_state <= STATE_READ_IDLE;
    end 
	 else begin
		case (read_current_state)
		
			STATE_READ_IDLE: begin
				if(write_finish)
				begin
					//init
					arvalid_reg <= 1'b1;
					arid_reg <= 4'd0;
					araddr_reg <= LPDDR4_AXI_ADDR_W_START; 
					arlen_reg <= LPDDR4_AXI_BURST_LENGHT; 
					read_current_state <= STATE_READ_ADDR;
				end
				else begin
					read_current_state <= read_current_state;
				end
			end
		  // set read addr
			STATE_READ_ADDR: begin
				if (arready) begin
					arvalid_reg <= 1'b0;
					rready_reg <= 1'b1;
					read_current_state <= STATE_READ_DATA;
				end
			end
			// read data
			STATE_READ_DATA: begin
				if (rvalid && rready) begin
					if (rlast) begin
						rready_reg <= 1'b0;
						if(araddr_reg != LPDDR4_AXI_ADDR_W_END) begin
							araddr_reg <= araddr_reg + LPDDR4_AXI_ADDR_W_COUNT;
							arvalid_reg <= 1'b1;
							arid_reg <= 4'd0;
							read_burst_count <= LPDDR4_AXI_BURST_LENGHT;
							read_current_state <= STATE_READ_ADDR;
						end
						else begin
							read_current_state <= STATE_READ_FINISH;
						end  
					end
					else begin
						read_current_state <= read_current_state;
					end
				end
				else begin
					read_current_state <= read_current_state;
				end
			end
		  		  
		  STATE_READ_FINISH: 
		  begin
			test_complete <= 1'b1;
			read_current_state <= read_current_state;
		  end
		  
        default: read_current_state <= STATE_READ_IDLE;
      endcase
    end
  end
  
 

 //data compare 
wire test_result_reg; 
reg test_ng;  
assign test_result_reg = (rvalid && rready) ? (axi_expected_data == rdata) ? 1'b1:1'b0 : 1'b1;


always@(posedge aclk or negedge aresetn)
begin
	if (!aresetn) begin
		test_ng <= 1'b0;
	end
	
	else begin
		if(test_result_reg) begin
			test_ng <= test_ng;
		end
		else begin
			test_ng <= 1'b1;
		end
	end
  
end
  
  
  //  AXI4 singnal
  assign awvalid = awvalid_reg;
  assign awid = awid_reg;
  assign awaddr = awaddr_reg;
  assign awlen = awlen_reg;
  assign awsize = awsize_reg;
  assign awburst = awburst_reg;
  assign awlock = awlock_reg;
  assign awcache = awcache_reg;
  assign awprot = awprot_reg;
  assign wvalid = wvalid_reg;
  assign wdata = axi_writedata;
  assign wstrb = wstrb_reg;
  assign wlast = wlast_reg;
  assign bready = bready_reg;
  assign arvalid = arvalid_reg;
  assign arid = arid_reg;
  assign araddr = araddr_reg;
  assign arlen = arlen_reg;
  assign arsize = arsize_reg;
  assign arburst = arburst_reg;
  assign arlock = arlock_reg;
  assign arcache = arcache_reg;
  assign arprot = arprot_reg;
  assign rready = rready_reg;

  // test result
  assign test_pass = ~test_ng & test_complete;
  

endmodule