// --------------------------------------------------------------------
// Copyright (c) 2024 by Terasic Technologies Inc.
// --------------------------------------------------------------------
//
// Permission:
//
//   Terasic grants permission to use and modify this code for use
//   in synthesis for all Terasic Development Boards and Altera Development
//   Kits made by Terasic.  Other use of this code, including the selling
//   ,duplication, or modification of any portion is strictly prohibited.
//
// Disclaimer:
//
//   This VHDL/Verilog or C/C++ source code is intended as a design reference
//   which illustrates how these types of functions can be implemented.
//   It is the user's responsibility to verify their design for
//   consistency and functionality through the use of formal
//   verification methods.  Terasic provides no warranty regarding the use
//   or functionality of this code.
//
// --------------------------------------------------------------------
//
//                     Terasic Technologies Inc
//                     No.80, Fenggong Rd., Hukou Township, Hsinchu County 303035. Taiwan
//
//                     web: http://www.terasic.com/
//                     email: support@terasic.com
//
// --------------------------------------------------------------------

#include "terasic_includes.h"
#include "mem_verify.h"

// Mailbox base addresses
#define LPDDR4A_MAILBOX_BASE            (EMIF_LPDDR4A_BASE + 0x5000000)
#define LPDDR4B_MAILBOX_BASE            (EMIF_LPDDR4B_BASE + 0x5000000)

// Calibration status register offset
#define MB_STATUS_CAL_INTF0_OFFSET      0x404

// Function prototypes
bool TEST_DDR4(void);
bool TEST_DDR4_QUICK(void);

//==============================================================================
// Memory Test Functions
//==============================================================================

bool TEST_DDR4(void){
    bool bPassA, bPassB, bLoop;
    bool bPassSubA[32], bPassSubB[32];
    int TestIndex;
    alt_u32 InitValue = 0x01;
    bool bShowMessage = TRUE;
    alt_u32 TimeStart, TimeElapsed;
    alt_u16 ButtonStatus;
    const alt_u8 ButtonMask = 0x01;
    alt_u64 span_base;
    alt_u32 span_base_lo, span_base_hi;
    alt_u32 i;
    const int nSubBankNum_A = 4; // LPDDR4A has 4GB (4 sub-banks)
    const int nSubBankNum_B = 2; // LPDDR4B has 2GB (2 sub-banks)

    printf("===== LPDDR4x2 Test! Size=A: 4GB, B: 2GB =====\r\n");

    printf("\n==========================================================\n");

    printf("Press any BUTTON on the board to start test [BUTTON-0 for continued test] \n");
    ButtonStatus = ButtonMask;
    while((ButtonStatus & ButtonMask) == ButtonMask){
        ButtonStatus = IORD(PIO_KEY_BASE, 0);
    }

    if ((ButtonStatus & 0x01) == 0x00){
        bLoop = TRUE;
    }else{
        bLoop = FALSE;
        usleep(300*1000);
    }

    bPassA = TRUE;
    bPassB = TRUE;
    TestIndex = 0;

    do{
        TestIndex++;
        printf("=====> LPDDR4x2 Testing, Iteration: %d\n", TestIndex);

        TimeStart = alt_nticks();

        ////////////////////////////
        // LPDDR4-A Testing
        TimeStart = alt_nticks();
        printf("== LPDDR4-A Testing...\r\n");

        if (bPassA) {
            for (i = 0; i < nSubBankNum_A; i++){
                printf("LPDDR4A address bank: %luGB ~ %luGB: \r\n", i, i + 1);
                span_base = (alt_u64)i * 0x40000000ULL;

                span_base_lo = span_base & 0xffffffff;
                span_base_hi = (span_base >> 32) & 0xffffffff;
                IOWR(ADDRESS_SPAN_EXTENDER_CNTL_BASE, 0, span_base_lo);
                IOWR(ADDRESS_SPAN_EXTENDER_CNTL_BASE, 1, span_base_hi);

                bPassSubA[i] = TMEM_Verify(ADDRESS_SPAN_EXTENDER_WINDOWED_SLAVE_BASE, ADDRESS_SPAN_EXTENDER_WINDOWED_SLAVE_SPAN, InitValue << i, bShowMessage);
                bPassA = bPassA && bPassSubA[i];
            }
        }

        TimeElapsed = alt_nticks() - TimeStart;
        printf("LPDDR4A test:%s, %d seconds\r\n", bPassA ? "Pass" : "NG", (int)(TimeElapsed/alt_ticks_per_second()));

        ////////////////////////////
        // LPDDR4-B Testing
        TimeStart = alt_nticks();
        printf("== LPDDR4-B Testing...\r\n");

        if (bPassB) {
            for (i = 0; i < nSubBankNum_B; i++){
                printf("LPDDR4B address bank: %luGB ~ %luGB: \r\n", i, i + 1);
                span_base = (alt_u64)0x100000000ULL + (alt_u64)i * 0x40000000ULL;
                span_base_lo = span_base & 0xffffffff;
                span_base_hi = (span_base >> 32) & 0xffffffff;
                IOWR(ADDRESS_SPAN_EXTENDER_CNTL_BASE, 0, span_base_lo);
                IOWR(ADDRESS_SPAN_EXTENDER_CNTL_BASE, 1, span_base_hi);

                bPassSubB[i] = TMEM_Verify(ADDRESS_SPAN_EXTENDER_WINDOWED_SLAVE_BASE, ADDRESS_SPAN_EXTENDER_WINDOWED_SLAVE_SPAN, InitValue << i, bShowMessage);
                bPassB = bPassB && bPassSubB[i];
            }
        }

        TimeElapsed = alt_nticks() - TimeStart;
        printf("LPDDR4B test:%s, %d seconds\r\n", bPassB ? "Pass" : "NG", (int)(TimeElapsed/alt_ticks_per_second()));

        if (bPassA && bPassB && bLoop){
            ButtonStatus = IORD(PIO_KEY_BASE, 0);
            if ((ButtonStatus & ButtonMask) != ButtonMask)
                bLoop = FALSE;
        }

    }while(bLoop && bPassA && bPassB);

    return bPassA && bPassB;
}

bool TEST_DDR4_QUICK(void){
    bool bPassA, bPassB, bLoop;
    bool bPassSubA[32], bPassSubB[32];
    int TestIndex;
    alt_u32 TimeStart, TimeElapsed;
    alt_u16 ButtonStatus;
    const alt_u8 ButtonMask = 0x01;
    alt_u64 span_base;
    alt_u32 span_base_lo, span_base_hi;
    alt_u32 i;
    const int nSubBankNum_A = 4; // LPDDR4A has 4GB (4 sub-banks)
    const int nSubBankNum_B = 2; // LPDDR4B has 2GB (2 sub-banks)

    printf("===== LPDDR4x2 Quick Test! Size=A: 4GB, B: 2GB =====\r\n");

    printf("\n==========================================================\n");

    printf("Press any BUTTON on the board to start test [BUTTON-0 for continued test] \n");
    ButtonStatus = ButtonMask;
    while((ButtonStatus & ButtonMask) == ButtonMask){
        ButtonStatus = IORD(PIO_KEY_BASE, 0);
    }

    if ((ButtonStatus & 0x01) == 0x00){
        bLoop = TRUE;
    }else{
        bLoop = FALSE;
        usleep(300*1000);
    }

    bPassA = TRUE;
    bPassB = TRUE;
    TestIndex = 0;

    do{
        TestIndex++;
        printf("=====> LPDDR4x2 Quick Testing, Iteration: %d\n", TestIndex);

        TimeStart = alt_nticks();

        ////////////////////////////
        // LPDDR4-A Quick Testing
        TimeStart = alt_nticks();
        printf("== LPDDR4-A Quick Testing...\r\n");

        if (bPassA){
            for (i = 0; i < nSubBankNum_A; i++){
                printf("LPDDR4A address bank: %luGB ~ %luGB: ", i, i + 1);
                span_base = (alt_u64)i * 0x40000000ULL;
                span_base_lo = span_base & 0xffffffff;
                span_base_hi = (span_base >> 32) & 0xffffffff;
                IOWR(ADDRESS_SPAN_EXTENDER_CNTL_BASE, 0, span_base_lo);
                IOWR(ADDRESS_SPAN_EXTENDER_CNTL_BASE, 1, span_base_hi);

                bPassSubA[i] = TMEM_QuickVerify(ADDRESS_SPAN_EXTENDER_WINDOWED_SLAVE_BASE, ADDRESS_SPAN_EXTENDER_WINDOWED_SLAVE_SPAN, 32, 28);
                bPassA = bPassA && bPassSubA[i];
            }
        }

        TimeElapsed = alt_nticks() - TimeStart;
        printf("LPDDR4A test:%s, %d seconds\r\n", bPassA ? "Pass" : "NG", (int)(TimeElapsed/alt_ticks_per_second()));

        ////////////////////////////
        // LPDDR4-B Quick Testing
        TimeStart = alt_nticks();
        printf("== LPDDR4-B Quick Testing...\r\n");
        if (bPassB) {
            for (i = 0; i < nSubBankNum_B; i++){
                printf("LPDDR4B address bank: %luGB ~ %luGB: ", i, i + 1);
                span_base = (alt_u64)0x100000000ULL + (alt_u64)i * 0x40000000ULL;
                span_base_lo = span_base & 0xffffffff;
                span_base_hi = (span_base >> 32) & 0xffffffff;
                IOWR(ADDRESS_SPAN_EXTENDER_CNTL_BASE, 0, span_base_lo);
                IOWR(ADDRESS_SPAN_EXTENDER_CNTL_BASE, 1, span_base_hi);

                bPassSubB[i] = TMEM_QuickVerify(ADDRESS_SPAN_EXTENDER_WINDOWED_SLAVE_BASE, ADDRESS_SPAN_EXTENDER_WINDOWED_SLAVE_SPAN, 32, 28);
                bPassB = bPassB && bPassSubB[i];
            }
        }

        TimeElapsed = alt_nticks() - TimeStart;
        printf("LPDDR4B test:%s, %d seconds\r\n", bPassB ? "Pass" : "NG", (int)(TimeElapsed/alt_ticks_per_second()));

        if (bPassA && bPassB && bLoop){
            ButtonStatus = IORD(PIO_KEY_BASE, 0);
            if ((ButtonStatus & ButtonMask) != ButtonMask)
                bLoop = FALSE;
        }

    }while(bLoop && bPassA && bPassB);

    return bPassA && bPassB;
}

//==============================================================================
// Menu and Main
//==============================================================================

void GUI_ShowMenu(void){
    printf("\n======= Agilex 5 NIOS LPDDR4x2 Program =======\r\n");
    printf("[0] LPDDR4x2 Test (Full)\r\n");
    printf("[1] LPDDR4x2 Quick Test\r\n");
    printf("Input your choice:");
}

int GUI_QueryUser(void){
    int nChoice = 0;
    scanf("%d", &nChoice);
    printf("%d\r\n", nChoice);
    return nChoice;
}

int main(void){
    int nChoice;
    bool bPass;
    alt_u32 cal_status_a, cal_status_b;

    printf("\n=== LPDDR4 Calibration Status Check ===\r\n");

    // Check LPDDR4-A calibration
    cal_status_a = IORD_32DIRECT(LPDDR4A_MAILBOX_BASE, MB_STATUS_CAL_INTF0_OFFSET) & 0x07;
    printf("LPDDR4-A Calibration: %s\r\n", (cal_status_a == 0x1) ? "PASS" : "FAIL");

    // Check LPDDR4-B calibration
    cal_status_b = IORD_32DIRECT(LPDDR4B_MAILBOX_BASE, MB_STATUS_CAL_INTF0_OFFSET) & 0x07;
    printf("LPDDR4-B Calibration: %s\r\n", (cal_status_b == 0x1) ? "PASS" : "FAIL");

    if (cal_status_a != 0x1 || cal_status_b != 0x1) {
        printf("\n*** CALIBRATION FAILURE DETECTED ***\r\n");
    }

    while(1){
        GUI_ShowMenu();
        nChoice = GUI_QueryUser();

        switch(nChoice){
            case 0:
                bPass = TEST_DDR4();
                printf("LPDDR4x2 Test:%s\r\n", bPass?"PASS":"NG");
                break;
            case 1:
                bPass = TEST_DDR4_QUICK();
                printf("LPDDR4x2 Quick Test:%s\r\n", bPass?"PASS":"NG");
                break;
            default:
                printf("Invalid selection.\n");
                break;
        }
    }

    return 0;
}
