// (C) 2001-2025 Altera Corporation. All rights reserved.
// Your use of Altera Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Altera Program License Subscription 
// Agreement, Altera IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Altera and sold by 
// Altera or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


// --------------------------------------
// Avalon-MM clock crossing bridge
//
// Clock crosses MM commands and responses with the
// help of asynchronous FIFOs.
//
// This bridge will stop emitting read commands when
// too many read commands are in flight to avoid 
// response FIFO overflow.
// --------------------------------------

`timescale 1 ns / 1 ns
module Qsys_mm_ccb_1_mm_ccb_1940_lr4at6q 
#(
    parameter DATA_WIDTH            = 32,
    parameter SYMBOL_WIDTH          = 8,
    parameter HDL_ADDR_WIDTH        = 10,
    parameter BURSTCOUNT_WIDTH      = 1,

    parameter COMMAND_FIFO_DEPTH    = 4,
    parameter RESPONSE_FIFO_DEPTH   = 4,

    parameter ENABLE_RESPONSE       = 0,

    parameter MASTER_SYNC_DEPTH     = 2,
    parameter SLAVE_SYNC_DEPTH      = 2,
    parameter SYNC_RESET            = 0,
    parameter USE_IN_FILL_LEVEL_CSR      = 1,
    parameter USE_OUT_FILL_LEVEL_CSR     = 1,
    parameter PIPELINE_ENABLE            =1,

    // --------------------------------------
    // Derived parameters
    // --------------------------------------
    parameter BYTEEN_WIDTH = DATA_WIDTH / SYMBOL_WIDTH
)
(
    input                           s0_clk,
    input                           s0_reset,

    input                           m0_clk,
    input                           m0_reset,

    output                          s0_waitrequest,
    output [DATA_WIDTH-1:0]         s0_readdata,
    output                          s0_readdatavalid,
    output [1:0]                    s0_response,
    input  [BURSTCOUNT_WIDTH-1:0]   s0_burstcount,
    input  [DATA_WIDTH-1:0]         s0_writedata,
    input  [HDL_ADDR_WIDTH-1:0]     s0_address, 
    input                           s0_write,  
    input                           s0_read,  
    output                          s0_writeresponsevalid,
    input  [BYTEEN_WIDTH-1:0]       s0_byteenable,  
    input                           s0_debugaccess,

    input                           m0_waitrequest,
    input  [DATA_WIDTH-1:0]         m0_readdata,
    input                           m0_readdatavalid,
    input  [1:0]                    m0_response,
    output [BURSTCOUNT_WIDTH-1:0]   m0_burstcount,
    output [DATA_WIDTH-1:0]         m0_writedata,
    output [HDL_ADDR_WIDTH-1:0]     m0_address, 
    output                          m0_write,  
    input                           m0_writeresponsevalid,
    output                          m0_read,  
    output [BYTEEN_WIDTH-1:0]       m0_byteenable,
    output                          m0_debugaccess,
                                       
    // in csr
    input [5:0]                     in_csr_address,
    input                           in_csr_write,
    input                           in_csr_read,
    output reg [31:0]               in_csr_readdata,
    input  [31:0]                   in_csr_writedata,

    // out csr
    input [5:0]                     out_csr_address,
    input                           out_csr_write,
    input                           out_csr_read,
    output reg [31:0]               out_csr_readdata,
    input  [31:0]                   out_csr_writedata


);

    localparam CMD_WIDTH = BURSTCOUNT_WIDTH + DATA_WIDTH + HDL_ADDR_WIDTH 
                    + BYTEEN_WIDTH 
                    + 3;        // read, write, debugaccess

    localparam NUMSYMBOLS    = DATA_WIDTH / SYMBOL_WIDTH;
    // Pass 2 bits of resp + 1 bit of readdatavalid whne response is enabled
    localparam RSP_WIDTH     = ENABLE_RESPONSE ? (DATA_WIDTH+3) : DATA_WIDTH;
    localparam MAX_BURST     = (1 << (BURSTCOUNT_WIDTH-1));
    localparam COUNTER_WIDTH = log2ceil(RESPONSE_FIFO_DEPTH) + 1;
    localparam COUNTER_WIDTH_CMD = log2ceil(COMMAND_FIFO_DEPTH) + 1;
    localparam NON_BURSTING  = (MAX_BURST == 1);
    localparam BURST_WORDS_W = BURSTCOUNT_WIDTH;

    // --------------------------------------
    // Signals
    // --------------------------------------
    wire [CMD_WIDTH-1:0]     s0_cmd_payload,s0_cmd_payload_reg;
    wire [CMD_WIDTH-1:0]     m0_cmd_payload;
    wire                     s0_cmd_valid;
    wire                     m0_cmd_valid;
    wire                     m0_internal_write;
    wire                     m0_internal_read;
    wire                     s0_cmd_ready;
    wire                     m0_cmd_ready;
    reg  [COUNTER_WIDTH-1:0] pending_read_count;
    wire [COUNTER_WIDTH-1:0] space_avail_resp;
    wire [COUNTER_WIDTH_CMD-1:0] space_avail_cmd;
    wire                     stop_cmd;
    reg                      stop_cmd_r;
    wire                     m0_cmd_accepted;
    wire                     m0_rd_wr_valid;
    wire                     m0_rsp_ready;
    reg                      old_read;
    reg                      old_cmd;
    wire [BURST_WORDS_W-1:0] m0_burstcount_words;
    wire [31:0]              cmd_min_level_in, cmd_max_level_in, cmd_avg_level_in, cmd_min_level_out, cmd_max_level_out, cmd_avg_level_out, cmd_fifo_level_in, cmd_fifo_level_out;
    wire [31:0]              resp_min_level_in,resp_max_level_in,resp_avg_level_in,resp_min_level_out,resp_max_level_out,resp_avg_level_out,resp_fifo_level_in,resp_fifo_level_out;
    
    wire  [RSP_WIDTH-1:0]    m_rresp_payload_reg;   
    reg                      clear_fifo_stats_in, clear_fifo_stats_out;
    // -------------------------------------------------------------------------
    // AVALON-Agent Address Map
    // -------------------------------------------------------------------------
    localparam CSR_RESET                    = 6'h00;
    localparam CSR_CMD_CURRENT_FILL_LEVEL   = 6'h04;
    localparam CSR_CMD_MIN_FILL_LEVEL       = 6'h08;
    localparam CSR_CMD_MAX_FILL_LEVEL       = 6'h0C;
    localparam CSR_CMD_AVG_FILL_LEVEL       = 6'h10;
    localparam CSR_RESP_CURRENT_FILL_LEVEL  = 6'h14;
    localparam CSR_RESP_MIN_FILL_LEVEL      = 6'h18;
    localparam CSR_RESP_MAX_FILL_LEVEL      = 6'h1C;
    localparam CSR_RESP_AVG_FILL_LEVEL      = 6'h20;
    
    localparam RESP_FIFO_DEPTH_BITS    =  log2ceil(RESPONSE_FIFO_DEPTH);
    localparam COMMAND_FIFO_DEPTH_BITS =   log2ceil(COMMAND_FIFO_DEPTH);

    // --------------------------------------
    // Command FIFO
    // --------------------------------------
    (* altera_attribute = "-name ALLOW_ANY_RAM_SIZE_FOR_RECOGNITION ON" *)
    Qsys_mm_ccb_1_mm_ccb_st_dc_fifo_1940_t75gi6a #(
        .SYMBOLS_PER_BEAT (1),
        .BITS_PER_SYMBOL  (CMD_WIDTH),
        .FIFO_DEPTH       (COMMAND_FIFO_DEPTH),
        .WR_SYNC_DEPTH    (MASTER_SYNC_DEPTH),
        .RD_SYNC_DEPTH    (SLAVE_SYNC_DEPTH),
        .BACKPRESSURE_DURING_RESET (1),
        .USE_SPACE_AVAIL_IF (1),
        .USE_IN_FILL_LEVEL (USE_IN_FILL_LEVEL_CSR),
        .USE_OUT_FILL_LEVEL (USE_OUT_FILL_LEVEL_CSR),
        .SYNC_RESET         (SYNC_RESET)
    ) 
    cmd_fifo
    (
        .in_clk          (s0_clk),
        .in_reset_n      (~s0_reset),
        .out_clk         (m0_clk),
        .out_reset_n     (~m0_reset),

        .in_data         (s0_cmd_payload_reg),
        .in_valid        (s0_cmd_valid_reg),
        .in_ready        (s0_cmd_ready_reg),

        .out_data        (m0_cmd_payload),
        .out_valid       (m0_cmd_valid),
        .out_ready       (m0_cmd_ready),
        
        .in_csr_address  (1'b0),
        .in_csr_read     (1'b1),        
        .in_csr_readdata (cmd_fifo_level_in),
        .in_csr_write    (1'b0),
        .in_csr_writedata(32'h0),
        
        .out_csr_address  (1'b0),
        .out_csr_read     (1'b1),        
        .out_csr_readdata (cmd_fifo_level_out),
        .out_csr_write    (1'b0),
        .out_csr_writedata(32'h0),
      
        .space_avail_data (space_avail_cmd)
    );





   
    // Generation of internal reset synchronization
   reg internal_sclr_m;
   generate if (SYNC_RESET == 1) begin // rst_syncronizer
      always @ (posedge m0_clk) begin
         internal_sclr_m <= m0_reset;
      end
   end
   endgenerate
   
    // Generation of internal reset synchronization
   reg internal_sclr_s;
   generate if (SYNC_RESET == 1) begin // rst_syncronizer
      always @ (posedge s0_clk) begin
         internal_sclr_s <= s0_reset;
      end
   end
   endgenerate
    //---------------------------------------
    //PIPELINE_ENABLE =1 for CMD path
    //---------------------------------------
    generate if(PIPELINE_ENABLE==1 && SYNC_RESET == 1) begin
        //Instance of R channel PIPELINE_ENABLE
        input_pipeline_module # (.PAYLOAD_WIDTH(CMD_WIDTH),
                                 .FIFO_DEPTH_BITS(COMMAND_FIFO_DEPTH_BITS))
        CMD_pipeline_module (
            .in_clk(s0_clk),
            .in_reset(internal_sclr_s),
            .in_payload(s0_cmd_payload),
            .in_valid(s0_cmd_valid),
            .in_ready(s0_cmd_ready),        
            .space_avail(space_avail_cmd),
            .out_ready(s0_cmd_ready_reg),
            .out_payload(s0_cmd_payload_reg),
            .out_valid(s0_cmd_valid_reg)
         );
        
    end    
    else begin
        assign  s0_cmd_payload_reg = s0_cmd_payload;
        assign  s0_cmd_valid_reg   = s0_cmd_valid; 
        assign  s0_cmd_ready   = s0_cmd_ready_reg;    
    end
    endgenerate
    // --------------------------------------
    // Command payload
    // --------------------------------------
    assign s0_waitrequest = ~s0_cmd_ready;
    assign s0_cmd_valid   = s0_write | s0_read;

    assign s0_cmd_payload = {s0_address, 
                             s0_burstcount, 
                             s0_read, 
                             s0_write, 
                             s0_writedata, 
                             s0_byteenable,
                             s0_debugaccess};
    assign {m0_address, 
            m0_burstcount, 
            m0_internal_read, 
            m0_internal_write,
            m0_writedata, 
            m0_byteenable,
            m0_debugaccess} = m0_cmd_payload;


    generate 
        if (ENABLE_RESPONSE) begin :RD_WR_ACCEPT
            assign m0_rd_wr_valid = m0_readdatavalid | m0_writeresponsevalid ;
            assign m0_cmd_ready = ~m0_waitrequest & 
                                  ~((m0_internal_read|m0_internal_write) & m0_cmd_valid  & stop_cmd_r & ~old_cmd);
            assign m0_write =  m0_internal_write & m0_cmd_valid & (~stop_cmd_r | old_cmd);
            assign m0_read  =  m0_internal_read  & m0_cmd_valid & (~stop_cmd_r | old_cmd);
            assign m0_cmd_accepted  = (m0_write | m0_read) & ~m0_waitrequest;
        end
        else begin : RD_ACCEPT
            assign m0_rd_wr_valid = m0_readdatavalid;
            assign m0_cmd_ready = ~m0_waitrequest & 
                                  ~(m0_internal_read & stop_cmd_r & ~old_read);
            assign m0_write =  m0_internal_write & m0_cmd_valid ;
            assign m0_read  =  m0_internal_read  & m0_cmd_valid & (~stop_cmd_r | old_read);
            assign m0_cmd_accepted  = m0_read & ~m0_waitrequest;
        end
    endgenerate

    // ---------------------------------------------
    // the non-bursting case
    // ---------------------------------------------
    generate if (NON_BURSTING)
    begin
      if (SYNC_RESET == 0) begin  
    
           always @(posedge m0_clk, posedge m0_reset) begin
               if (m0_reset) begin
                   pending_read_count <= 0;
               end
               else begin
                   if (m0_cmd_accepted & m0_rd_wr_valid)
                       pending_read_count <= pending_read_count;
                   else if (m0_rd_wr_valid)
                       pending_read_count <= pending_read_count - 1'd1;
                   else if (m0_cmd_accepted)
                       pending_read_count <= pending_read_count + 1'd1;
               end
           end
      end // async_rst0

      else begin 
           always @(posedge m0_clk) begin
               if (internal_sclr_m) begin
                   pending_read_count <= 0;
               end
               else begin
                   if (m0_cmd_accepted & m0_rd_wr_valid)
                       pending_read_count <= pending_read_count;
                   else if (m0_rd_wr_valid)
                       pending_read_count <= pending_read_count - 1'd1;
                   else if (m0_cmd_accepted)
                       pending_read_count <= pending_read_count + 1'd1;
               end
           end
      end // sync_rst0
    end // if non_bursting

    // ---------------------------------------------
    // the bursting case
    // ---------------------------------------------
    else begin
        // N responses for single read command
        // single reponse for N write commands
        // detect first  writes, load the wr count on first write and then
        // decrement
        // you know write is through once counter reaches 1
        reg  [BURST_WORDS_W-1:0] m0_wr_count;
        wire [BURST_WORDS_W-1:0] m0_wr_bursrcount;

        assign m0_wr_bursrcount = (m0_wr_count==1) ? 1'd1:1'd0;

        if (ENABLE_RESPONSE) begin : WR_RESP_COUNT
            assign m0_burstcount_words = m0_read ? m0_burstcount : m0_wr_bursrcount;
        end
        else begin : NO_WR_RESP_COUNTS
            assign m0_burstcount_words = m0_burstcount;
        end

      if (SYNC_RESET == 0 ) begin 
           always @ (posedge m0_clk, posedge m0_reset)
               if (m0_reset)
                   m0_wr_count <=1'd1;
               else if (m0_wr_count==1 && m0_write && ~m0_waitrequest)
                   m0_wr_count <= m0_burstcount;
               else if (m0_write && ~m0_waitrequest)
                   m0_wr_count <= m0_wr_count-1'd1;

           always @(posedge m0_clk, posedge m0_reset) begin
               if (m0_reset) begin
                   pending_read_count <= 0;
               end
               else begin
                   if (m0_cmd_accepted & m0_rd_wr_valid)
                       pending_read_count <= pending_read_count +
                                               m0_burstcount_words - 1'd1;
                   else if (m0_rd_wr_valid)
                       pending_read_count <= pending_read_count - 1'd1;
                   else if (m0_cmd_accepted)
                       pending_read_count <= pending_read_count +
                                               m0_burstcount_words;  
               end
           end
      end // async_rst1
      else begin 
           always @ (posedge m0_clk)
               if (internal_sclr_m)
                   m0_wr_count <=1'd1;
               else if (m0_wr_count==1 && m0_write && ~m0_waitrequest)
                   m0_wr_count <= m0_burstcount;
               else if (m0_write && ~m0_waitrequest)
                   m0_wr_count <= m0_wr_count-1;

           always @(posedge m0_clk) begin
               if (internal_sclr_m) begin
                   pending_read_count <= 1'd0;
               end
               else begin
                   if (m0_cmd_accepted & m0_rd_wr_valid)
                       pending_read_count <= pending_read_count +
                                               m0_burstcount_words - 1'd1;
                   else if (m0_rd_wr_valid)
                       pending_read_count <= pending_read_count - 1'd1;
                   else if (m0_cmd_accepted)
                       pending_read_count <= pending_read_count +
                                               m0_burstcount_words;  
               end
           end // @always
      end // sync_rst1


    end // else bursting
    endgenerate
    
    // in case of bursting, pening read count already calculates burst_count 
    // hence no need to add 2*MAX_BURST
    // fix this only for write-response mode for now
    generate 
        if (ENABLE_RESPONSE) begin : WITH_RESP_CALC
            assign stop_cmd = pending_read_count > space_avail_resp;
        end
        else begin : NO_RSP_CALC
            assign stop_cmd = (pending_read_count + 2*MAX_BURST) > space_avail_resp;
        end
    endgenerate
   
    generate
    if (SYNC_RESET == 0) begin // async_rst2
       always @(posedge m0_clk, posedge m0_reset) begin
           if (m0_reset) begin
               stop_cmd_r <= 1'b0;
               old_read   <= 1'b0;
               old_cmd    <= 1'b0;
           end
           else begin
               stop_cmd_r <= stop_cmd;
               old_read   <= m0_read & m0_waitrequest;
               old_cmd    <= (m0_write | m0_read) & m0_waitrequest;
           end
       end
     end // async_rst2

     else begin
       always @(posedge m0_clk) begin
           if (internal_sclr_m) begin
               stop_cmd_r <= 1'b0;
               old_read   <= 1'b0;
               old_cmd    <= 1'b0;
           end
           else begin
               stop_cmd_r <= stop_cmd;
               old_read   <= m0_read & m0_waitrequest;
               old_cmd    <= (m0_write | m0_read) & m0_waitrequest;
           end
       end
     end // sync_rst2
     endgenerate
    // --------------------------------------
    // Response FIFO
    // Contains : reponse, readdatavalid, readdata
    // Why readdatavalid acts as Control and Data here ???? ->
    // Avalon-MM has single response field. So FIFO will be written with
    // - readdatavalid | writeresponsevalid
    // But how do you detect - whether its Rd response or WR response on FIFO
    // Output ?????? We add readdatavalid along data used to detect RD vs WR
    // response
    //
    //
    // --------------------------------------
    wire [RSP_WIDTH-1:0] rsp_fifo_in,rsp_fifo_out;
    wire                 rsp_fifo_in_valid,rsp_fifo_out_valid;
    generate 
        if (ENABLE_RESPONSE) begin : WITH_RESP
            assign rsp_fifo_in = {m0_response,m0_readdatavalid,m0_readdata};
            assign rsp_fifo_in_valid = m0_readdatavalid | m0_writeresponsevalid;

            assign s0_readdata           = rsp_fifo_out[DATA_WIDTH-1:0];
            assign s0_response           = rsp_fifo_out[RSP_WIDTH-1:RSP_WIDTH-2];
            assign s0_readdatavalid      = rsp_fifo_out_valid & rsp_fifo_out[RSP_WIDTH-3];
            assign s0_writeresponsevalid = rsp_fifo_out_valid & (!rsp_fifo_out[RSP_WIDTH-3]);
        end
        else begin : NO_RESP
            assign rsp_fifo_in = m0_readdata;
            assign rsp_fifo_in_valid = m0_readdatavalid ;

            assign s0_readdata           = rsp_fifo_out;
            assign s0_response           = 2'b00;
            assign s0_readdatavalid      = rsp_fifo_out_valid ;
            assign s0_writeresponsevalid = 1'b0;
        end
    endgenerate 
       
    (* altera_attribute = "-name ALLOW_ANY_RAM_SIZE_FOR_RECOGNITION ON" *) 
    Qsys_mm_ccb_1_mm_ccb_st_dc_fifo_1940_yx4ywwy #(
        .SYMBOLS_PER_BEAT   (1),
        .BITS_PER_SYMBOL    (RSP_WIDTH),
        .FIFO_DEPTH         (RESPONSE_FIFO_DEPTH),
        .WR_SYNC_DEPTH      (SLAVE_SYNC_DEPTH),
        .RD_SYNC_DEPTH      (MASTER_SYNC_DEPTH),
        .USE_SPACE_AVAIL_IF (1),
        .USE_IN_FILL_LEVEL  (USE_IN_FILL_LEVEL_CSR),
        .USE_OUT_FILL_LEVEL (USE_OUT_FILL_LEVEL_CSR),  
        .SYNC_RESET         (SYNC_RESET)
    ) 
    rsp_fifo
    (
        .in_clk           (m0_clk),
        .in_reset_n       (~m0_reset),
        .out_clk          (s0_clk),
        .out_reset_n      (~s0_reset),

        .in_data          (m_rresp_payload_reg),
        .in_valid         (m_rresp_valid),

        // ------------------------------------
        // must never overflow, or we're in trouble
        // (we cannot backpressure the response)
        // ------------------------------------
        .in_ready         (m0_rsp_ready_reg),

        .out_data         (rsp_fifo_out),
        .out_valid        (rsp_fifo_out_valid),
        .out_ready        (1'b1),

        .in_csr_address  (1'b0),
        .in_csr_read     (1'b1),        
        .in_csr_readdata (resp_fifo_level_in),
        .in_csr_write    (1'b0),
        .in_csr_writedata(32'h0),
        
        .out_csr_address  (1'b0),
        .out_csr_read     (1'b1),
        .out_csr_readdata (resp_fifo_level_out),
        .out_csr_write    (1'b0),
        .out_csr_writedata(32'h0),

        .space_avail_data (space_avail_resp)

    );
    
    //---------------------------------------
    //PIPELINE_ENABLE =1 for Response path
    //---------------------------------------
    generate if(PIPELINE_ENABLE==1 && SYNC_RESET == 1) begin
        //Instance of R channel PIPELINE_ENABLE
        input_pipeline_module # (.PAYLOAD_WIDTH(RSP_WIDTH),
                                 .FIFO_DEPTH_BITS(RESP_FIFO_DEPTH_BITS))
        resp_pipeline_module (
            .in_clk(m0_clk),
            .in_reset(internal_sclr_m),
            .in_payload(rsp_fifo_in),
            .in_valid(rsp_fifo_in_valid),
            .in_ready(m0_rsp_ready),
            .space_avail(space_avail_resp),
            .out_ready(m0_rsp_ready_reg),
            .out_payload(m_rresp_payload_reg),
            .out_valid(m_rresp_valid));
        
    end
    else begin
        assign  m_rresp_payload_reg = rsp_fifo_in;
        assign  m_rresp_valid       = rsp_fifo_in_valid;
        assign  m0_rsp_ready       =  m0_rsp_ready_reg ;
    end
    endgenerate
    
    
    generate if (USE_IN_FILL_LEVEL_CSR == 1) begin
        // Instance FIFO Status Monitors
        fifo_status_monitor cmd_fifo_status_monitor_in (
            .clk                (s0_clk),
            .current_fifo_level (cmd_fifo_level_in),
            .clear_stats        (clear_fifo_stats_in),
            .min_level          (cmd_min_level_in),
            .max_level          (cmd_max_level_in),
            .avg_level          (cmd_avg_level_in));

        // Instance FIFO Status Monitors
        fifo_status_monitor  resp_fifo_status_monitor_in (
           .clk                (s0_clk),
           .current_fifo_level (resp_fifo_level_in),
           .clear_stats        (clear_fifo_stats_in),
           .min_level          (resp_min_level_in),
           .max_level          (resp_max_level_in),
           .avg_level          (resp_avg_level_in));
    end
    endgenerate

    generate if (SYNC_RESET ==1 && USE_IN_FILL_LEVEL_CSR ==1 ) begin 
        //Register Access reg
        always@(posedge s0_clk ) begin
            if (internal_sclr_s)begin
                in_csr_readdata <= 32'h00000000; // Default to 0
                clear_fifo_stats_in   <= 1'b1;
            end else begin
                in_csr_readdata <= 32'h00000000; // Default to 0   
                clear_fifo_stats_in   <= 1'b0;           
                if (in_csr_read) begin
                    case (in_csr_address)
                        CSR_CMD_CURRENT_FILL_LEVEL:    in_csr_readdata <=  cmd_fifo_level_in;
                        CSR_CMD_MIN_FILL_LEVEL:        in_csr_readdata <=  cmd_min_level_in;
                        CSR_CMD_MAX_FILL_LEVEL:        in_csr_readdata <=  cmd_max_level_in;
                        CSR_CMD_AVG_FILL_LEVEL:        in_csr_readdata <=  cmd_avg_level_in;
                            
                        CSR_RESP_CURRENT_FILL_LEVEL:   in_csr_readdata <=  resp_fifo_level_in;
                        CSR_RESP_MIN_FILL_LEVEL:       in_csr_readdata <=  resp_min_level_in;
                        CSR_RESP_MAX_FILL_LEVEL:       in_csr_readdata <=  resp_max_level_in;
                        CSR_RESP_AVG_FILL_LEVEL:       in_csr_readdata <=  resp_avg_level_in;                           
                           
                        default:                 in_csr_readdata <= 32'h00000000; 
                    endcase
                end else if (in_csr_write) begin
                    case (in_csr_address)
                        CSR_RESET:      clear_fifo_stats_in <= (in_csr_writedata[0] == 1'b1);
                        default:   clear_fifo_stats_in   <= 1'b0; // Read-only registers for status
                    endcase
                end
            end
        end            
    end else if (SYNC_RESET ==0 && USE_IN_FILL_LEVEL_CSR ==1) begin
        always@(posedge s0_clk , posedge s0_reset) begin
            if (s0_reset)begin
                in_csr_readdata <= 32'h00000000; // Default to 0
                clear_fifo_stats_in   <= 1'b1;
            end else begin
                in_csr_readdata <= 32'h00000000; // Default to 0 
                clear_fifo_stats_in   <= 1'b0;               
                if (in_csr_read) begin
                    case (in_csr_address)
                        CSR_CMD_CURRENT_FILL_LEVEL:    in_csr_readdata <=  cmd_fifo_level_in;
                        CSR_CMD_MIN_FILL_LEVEL:        in_csr_readdata <=  cmd_min_level_in;
                        CSR_CMD_MAX_FILL_LEVEL:        in_csr_readdata <=  cmd_max_level_in;
                        CSR_CMD_AVG_FILL_LEVEL:        in_csr_readdata <=  cmd_avg_level_in;
                            
                        CSR_RESP_CURRENT_FILL_LEVEL:   in_csr_readdata <=  resp_fifo_level_in;
                        CSR_RESP_MIN_FILL_LEVEL:       in_csr_readdata <=  resp_min_level_in;
                        CSR_RESP_MAX_FILL_LEVEL:       in_csr_readdata <=  resp_max_level_in;
                        CSR_RESP_AVG_FILL_LEVEL:       in_csr_readdata <=  resp_avg_level_in;                           
                           
                        default:                 in_csr_readdata <= 32'h00000000; 
                    endcase
                end else if (in_csr_write) begin
                    case (in_csr_address)
                        CSR_RESET:      clear_fifo_stats_in <= (in_csr_writedata[0] == 1'b1);
                        default:   clear_fifo_stats_in   <= 1'b0; // Read-only registers for status
                    endcase
                end
            end
        end
    end 
    endgenerate
    
    
    generate if (USE_OUT_FILL_LEVEL_CSR == 1) begin
        // Instance FIFO Status Monitors
        fifo_status_monitor  cmd_fifo_status_monitor_out (
            .clk                (m0_clk),
            .current_fifo_level (cmd_fifo_level_out),
            .clear_stats        (clear_fifo_stats_out),
            .min_level          (cmd_min_level_out),
            .max_level          (cmd_max_level_out),
            .avg_level          (cmd_avg_level_out));
            
        fifo_status_monitor  resp_fifo_status_monitor_out (
            .clk                (m0_clk),
            .current_fifo_level (resp_fifo_level_out),
            .clear_stats        (clear_fifo_stats_out),
            .min_level          (resp_min_level_out),
            .max_level          (resp_max_level_out),
            .avg_level          (resp_avg_level_out));
    end
    endgenerate

    generate if (SYNC_RESET ==1 && USE_OUT_FILL_LEVEL_CSR ==1 ) begin 
        // Register Access 
        always@(posedge m0_clk ) begin
            if (internal_sclr_m)begin
                out_csr_readdata <= 32'h00000000;  // Default to 0
                clear_fifo_stats_out   <= 1'b1;
            end else begin
                out_csr_readdata <= 32'h00000000;  // Default to 0
                clear_fifo_stats_out   <= 1'b0;
                if (out_csr_read) begin
                    case (out_csr_address)
                        CSR_CMD_CURRENT_FILL_LEVEL:    out_csr_readdata <=  cmd_fifo_level_out;
                        CSR_CMD_MIN_FILL_LEVEL:        out_csr_readdata <=  cmd_min_level_out;
                        CSR_CMD_MAX_FILL_LEVEL:        out_csr_readdata <=  cmd_max_level_out;
                        CSR_CMD_AVG_FILL_LEVEL:        out_csr_readdata <=  cmd_avg_level_out;

                        CSR_RESP_CURRENT_FILL_LEVEL:   out_csr_readdata <=  resp_fifo_level_out;
                        CSR_RESP_MIN_FILL_LEVEL:       out_csr_readdata <=  resp_min_level_out;
                        CSR_RESP_MAX_FILL_LEVEL:       out_csr_readdata <=  resp_max_level_out;
                        CSR_RESP_AVG_FILL_LEVEL:       out_csr_readdata <=  resp_avg_level_out;                           

                        default:                 out_csr_readdata <= 32'h00000000; 
                    endcase
                end else if (out_csr_write) begin
                    case (out_csr_address)
                        CSR_RESET:      clear_fifo_stats_out <= (out_csr_writedata[0] == 1'b1);
                        default:   clear_fifo_stats_out   <= 1'b0; // Read-only registers for status
                    endcase
                end
            end
        end
    end else if (SYNC_RESET ==0 && USE_OUT_FILL_LEVEL_CSR ==1) begin
        always@(posedge m0_clk , posedge m0_reset) begin
            if (m0_reset)begin
                out_csr_readdata <= 32'h00000000; // Default to 0
                clear_fifo_stats_out   <= 1'b1;
            end else begin
                out_csr_readdata <= 32'h00000000; // Default to 0
                clear_fifo_stats_out   <= 1'b0;
                if (out_csr_read) begin
                    case (out_csr_address)
                        CSR_CMD_CURRENT_FILL_LEVEL:    out_csr_readdata <=  cmd_fifo_level_out;
                        CSR_CMD_MIN_FILL_LEVEL:        out_csr_readdata <=  cmd_min_level_out;
                        CSR_CMD_MAX_FILL_LEVEL:        out_csr_readdata <=  cmd_max_level_out;
                        CSR_CMD_AVG_FILL_LEVEL:        out_csr_readdata <=  cmd_avg_level_out;
                        
                        CSR_RESP_CURRENT_FILL_LEVEL:   out_csr_readdata <=  resp_fifo_level_out;
                        CSR_RESP_MIN_FILL_LEVEL:       out_csr_readdata <=  resp_min_level_out;
                        CSR_RESP_MAX_FILL_LEVEL:       out_csr_readdata <=  resp_max_level_out;
                        CSR_RESP_AVG_FILL_LEVEL:       out_csr_readdata <=  resp_avg_level_out;                           
                       
                        default:                 out_csr_readdata <= 32'h00000000;
                    endcase
                end else if (out_csr_write) begin
                    case (out_csr_address)
                        CSR_RESET:      clear_fifo_stats_out <= (out_csr_writedata[0] == 1'b1);
                        default:   clear_fifo_stats_out   <= 1'b0; // Read-only registers for status
                    endcase
                end
            end
        end
    end 
    endgenerate

// synthesis translate_off
    always @(posedge m0_clk) begin
        if ((m0_reset === 0) || (m0_reset === 1)) begin //to avoid printing error detetction messages when reset is X or Z.  
            if (~m0_rsp_ready & (m0_readdatavalid|m0_writeresponsevalid) ) begin
                $display("%t %m: internal error, response fifo overflow", $time);
            end
            if(PIPELINE_ENABLE==1) begin
                if (pending_read_count > space_avail_resp+2) begin
                    $display("%t %m: internal error, too many pending reads/writes", $time);
                end
            end else begin
                if (pending_read_count > space_avail_resp) begin
                    $display("%t %m: internal error, too many pending reads/writes", $time);
                end
            end
        end
    end
// synthesis translate_on

    // --------------------------------------------------
    // Calculates the log2ceil of the input value
    // --------------------------------------------------
    function integer log2ceil;
        input integer val;
        integer i;

        begin
            i = 1;
            log2ceil = 0;

            while (i < val) begin
                log2ceil = log2ceil + 1;
                i = i << 1; 
            end
        end
    endfunction

endmodule 




// fifo_status_monitor.sv
// Monitors FIFO level and calculates min, max, and average.
module fifo_status_monitor #(
    parameter LEVEL_WIDTH = 32 // Corresponds to ADDRESS_WIDTH+1 from FIFO
) (
    input                         clk,
    input      [LEVEL_WIDTH-1:0]  current_fifo_level,
    input                         clear_stats, // Signal from AXI-Lite to clear min/max/avg

    output reg [LEVEL_WIDTH-1:0]  min_level,
    output reg [LEVEL_WIDTH-1:0]  max_level,
    output     [LEVEL_WIDTH-1:0]  avg_level
);

    // Min/Max registers
    always @(posedge clk) begin
        if ( clear_stats) begin
            min_level <= {LEVEL_WIDTH{1'b1}}; // Initialize to max value
            max_level <= {LEVEL_WIDTH{1'b0}}; // Initialize to min value
        end else begin
            if (current_fifo_level < min_level) begin
                min_level <= current_fifo_level;
            end
            if (current_fifo_level > max_level) begin
                max_level <= current_fifo_level;
            end
        end
    end

       // Here, we'll implement a sum and count, then divide.
    localparam AVG_WINDOW_BITS = 5; // Number of samples for averaging window
    localparam AVG_WINDOW_SIZE = (1 << AVG_WINDOW_BITS);
    localparam SUM_WIDTH = LEVEL_WIDTH + AVG_WINDOW_BITS; // Width needed for sum

    reg [SUM_WIDTH-1:0]           fifo_level_sum;
    reg                           sample_count;
    reg [LEVEL_WIDTH-1:0]         fifo_level_history [AVG_WINDOW_SIZE-1:0];
    reg [AVG_WINDOW_BITS-1:0]     history_ptr;
    reg [AVG_WINDOW_BITS:0] i;//for for loop
    always @(posedge clk) begin
        if (clear_stats) begin
            fifo_level_sum  <= 37'h000000000;
            sample_count    <= 1'b0;
            history_ptr     <= 5'b00000;
            for (i = 0; i < AVG_WINDOW_SIZE; i=i+1) begin
                fifo_level_history[i] <= 32'h00000000;
            end
        end else begin
            // Update sum: subtract old value, add new value
            fifo_level_sum <= fifo_level_sum - fifo_level_history[history_ptr] + current_fifo_level;
            fifo_level_history[history_ptr] <= current_fifo_level;
            sample_count   <= 1'b1;        
            // Move pointer
            history_ptr <= history_ptr + 1'b1;
        end
    end

    // Calculate average
    assign avg_level = (sample_count == 0) ? 32'h00000000 : fifo_level_sum >> 5;//divide by 32 AVG_WINDOW_SIZE
endmodule




//module input_pipeline_module, it is used to add the register at front to provide better timing to fitter can place logic at better timing locations

module input_pipeline_module #( 
        parameter PAYLOAD_WIDTH   = 32,
        parameter NUM_REG         = 2,
        parameter FIFO_DEPTH_BITS = 4
    ) ( 
      input                              in_clk,
      input                              in_reset,

      input       [PAYLOAD_WIDTH -1:0]   in_payload,
      input                              in_valid,      
      output                             in_ready,
      
      input       [FIFO_DEPTH_BITS :0]   space_avail,
      
      input                              out_ready,//from ST_DC_fifo , ready_backpressure dusring reset
      output      [PAYLOAD_WIDTH -1:0]   out_payload,
      output                             out_valid
      
      
    );
      
    reg [PAYLOAD_WIDTH:0]   in_payload_with_valid_reg  [NUM_REG-1: 0];
    wire [PAYLOAD_WIDTH:0]  out_payload_with_valid,in_payload_with_valid;
    reg                     in_ready_reg [NUM_REG-1: 0];
    wire                    in_valid_with_ready;
    reg                     ready_backpressure;
    integer                 i;
    assign s_ready = (space_avail > (NUM_REG<<1))? 1:0; // ready is actully almost full(num_regx2) based on fifo depth bits to avoid overflow    
    assign in_valid_with_ready   = in_valid & in_ready;
    
    assign in_payload_with_valid = {in_valid_with_ready, in_payload};
    
    always@(posedge in_clk)begin 
        in_ready_reg[0]               <= s_ready;
        in_payload_with_valid_reg[0]  <= in_payload_with_valid;
       
        for (i = 1; i < NUM_REG; i= i+1) begin
           in_ready_reg[i]                <= in_ready_reg[i-1]; 
           in_payload_with_valid_reg[i]   <= in_payload_with_valid_reg[i-1];
        end
    end 
    assign in_ready                 =  in_ready_reg[NUM_REG-1] & ready_backpressure;
    assign out_payload_with_valid   =  in_payload_with_valid_reg[NUM_REG-1];
    assign {out_valid, out_payload} =  out_payload_with_valid;  

    always@(posedge in_clk)begin 
        if (in_reset)
           ready_backpressure<= 1'b0;
        else if (out_ready)
           ready_backpressure<= 1'b1;
    end    
endmodule

