#include <stdbool.h>
#include <stdio.h>

#include "alt_types.h"  // alt_u8, alt_u16, alt_u32
#include "altera_avalon_pio_regs.h"  // IOWR_ALTERA_AVALON_PIO_DATA, IORD_ALTERA_AVALON_PIO_DATA
#include "system.h"  // Platform Designer generated base addresses

typedef bool (*LP_VERIFY_FUNC)(void);

bool TEST_FAN(void);
bool TEST_BOARD_INFO(void);

typedef struct {
  LP_VERIFY_FUNC func;
  char szName[128];
} FUNC_INFO;

FUNC_INFO szFuncList[] = {{TEST_BOARD_INFO, "System Info"},
                          {TEST_FAN, "Fan Control"}};

void GUI_ShowMenu(void) {
  int nNum, i;

  nNum = sizeof(szFuncList) / sizeof(szFuncList[0]);
  printf("======= Board_Info_NiosV Demo Program =======\n");
  for (i = 0; i < nNum; i++) {
    printf("[%d] %s\n", i, szFuncList[i].szName);
  }
  printf("Input your choice:");
}

int GUI_QueryUser(void) {
  int nChoice = 0;
  scanf("%d", &nChoice);
  printf("%d\n", nChoice);
  return nChoice;
}

int main(void) {

  int nChoice;
  int nNum;
  bool bPass;

  nNum = sizeof(szFuncList) / sizeof(szFuncList[0]);
  while (1) {
    GUI_ShowMenu();
    nChoice = GUI_QueryUser();
    if (nChoice >= 0 && nChoice < nNum) {
      bPass = szFuncList[nChoice].func();
      printf("%s Test:%s\n", szFuncList[nChoice].szName, bPass ? "PASS" : "NG");
    }
  }
}

bool TEST_FAN(void) {
  printf("=== FAN CONTROL TEST ===\n");

  // Read current fan speed
  alt_u32 fan_status = IORD_ALTERA_AVALON_PIO_DATA(PIO_FAN_BASE);
  alt_u16 current_speed = fan_status & 0xFFFF;

  // Display only - no validation or control
  printf("Current Fan Speed: %u RPM\n", current_speed);

  printf("\n[FAN CONTROL]\n");
  printf("Result=PASS\n");
  return true;
}

bool TEST_BOARD_INFO(void) {
  printf("=== BOARD INFO TEST ===\n");

  // Read temperatures (signed 16-bit)
  alt_16 fpga_temp = (alt_16)IORD_ALTERA_AVALON_PIO_DATA(FPGA_TEMPERATURE_BASE);
  alt_16 board_temp =
      (alt_16)IORD_ALTERA_AVALON_PIO_DATA(BOARD_TEMPERATURE_BASE);

  // Read power (unsigned 16-bit)
  alt_u16 voltage_mv = IORD_ALTERA_AVALON_PIO_DATA(POWER_VOLTAGE_BASE);
  alt_u16 current_ma = IORD_ALTERA_AVALON_PIO_DATA(POWER_CURRENT_BASE);

  // Calculate power in mW
  alt_u32 power_mw = (alt_u32)voltage_mv * current_ma / 1000;

  // Display results
  printf("FPGA Temperature:  %d C\n", fpga_temp);
  printf("Board Temperature: %d C\n", board_temp);
  printf("Input Voltage:     %u mV (%.2f V)\n", voltage_mv,
         voltage_mv / 1000.0);
  printf("Input Current:     %u mA (%.2f A)\n", current_ma,
         current_ma / 1000.0);
  printf("Power Consumption: %lu mW (%.2f W)\n", power_mw, power_mw / 1000.0);

  // Validation removed - display only
  printf("\n[BOARD INFO]\n");
  printf("Result=PASS\n");
  return true;
}
